"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SampleDir = exports.SampleFile = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const glob = require("glob");
const component_1 = require("./component");
const util_1 = require("./util");
/**
 * Produces a file with the given contents but only once, if the file doesn't already exist.
 * Use this for creating example code files or other resources.
 */
class SampleFile extends component_1.Component {
    /**
     * Creates a new SampleFile object
     * @param project - the project to tie this file to.
     * @param filePath - the relative path in the project to put the file
     * @param options - the options for the file.
     */
    constructor(project, filePath, options) {
        super(project);
        if (options.contents && options.sourcePath) {
            throw new Error("Cannot specify both 'contents' and 'source' fields.");
        }
        if (!options.contents && !options.sourcePath) {
            throw new Error("Must specify at least one of 'contents' or 'source'.");
        }
        this.filePath = filePath;
        this.options = options;
    }
    synthesize() {
        let contents;
        if (this.options.contents) {
            contents = this.options.contents;
        }
        else if (this.options.sourcePath) {
            contents = fs.readFileSync(this.options.sourcePath);
        }
        this.writeOnceFileContents(this.project.outdir, this.filePath, contents ?? "");
    }
    /**
     * A helper function that will write the file once and return if it was written or not.
     * @param dir - the directory for the new file
     * @param filename - the filename for the new file
     * @param contents - the contents of the file to write
     * @return boolean - whether a new file was written or not.
     * @private
     */
    writeOnceFileContents(dir, filename, contents) {
        const fullFilename = path.join(dir, filename);
        if (fs.existsSync(fullFilename)) {
            return;
        }
        (0, util_1.writeFile)(fullFilename, contents, { readonly: false });
    }
}
exports.SampleFile = SampleFile;
_a = JSII_RTTI_SYMBOL_1;
SampleFile[_a] = { fqn: "projen.SampleFile", version: "0.91.13" };
/**
 * Renders the given files into the directory if the directory does not exist. Use this to create sample code files
 */
class SampleDir extends component_1.Component {
    /**
     * Create sample files in the given directory if the given directory does not exist
     * @param project Parent project to add files to.
     * @param dir directory to add files to. If directory already exists, nothing is added.
     * @param options options for which files to create.
     */
    constructor(project, dir, options) {
        super(project);
        if (!options.files && !options.sourceDir) {
            throw new Error("Must specify at least one of 'files' or 'source'.");
        }
        this.dir = dir;
        this.options = options;
    }
    synthesize() {
        const fullOutdir = path.join(this.project.outdir, this.dir);
        if (fs.existsSync(fullOutdir)) {
            return;
        }
        // previously creating the directory to allow empty dirs to be created
        fs.mkdirSync(fullOutdir, { recursive: true });
        if (this.options.sourceDir) {
            const basedir = this.options.sourceDir;
            const files = glob.sync("**", {
                cwd: basedir,
                nodir: true,
                dot: true,
            }); // returns relative file paths with POSIX separators
            for (const file of files) {
                const sourcePath = path.join(basedir, file);
                const targetPath = path.join(fullOutdir, file);
                fs.mkdirSync(path.dirname(targetPath), { recursive: true });
                fs.copyFileSync(sourcePath, targetPath);
                fs.chmodSync(targetPath, (0, util_1.getFilePermissions)({ readonly: false, executable: false }));
            }
        }
        for (const filename in this.options.files) {
            (0, util_1.writeFile)(path.join(fullOutdir, filename), this.options.files[filename]);
        }
    }
}
exports.SampleDir = SampleDir;
_b = JSII_RTTI_SYMBOL_1;
SampleDir[_b] = { fqn: "projen.SampleDir", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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