"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyBuild = void 0;
const component_1 = require("../component");
const json_1 = require("../json");
/**
 * Smithy build configuration options
 * @see https://smithy.io/2.0/guides/building-models/build-config.html
 */
class SmithyBuild extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        this.version = options.version ?? "1.0";
        this.outputDirectory = options.outputDirectory;
        this._imports = options.imports;
        this._projections = options.projections;
        this._plugins = options.plugins;
        this.ignoreMissingPlugins = options.ignoreMissingPlugins;
        this._maven = options.maven;
        this._sources = options.sources;
        this.manifest = {
            version: this.version,
            outputDirectory: this.outputDirectory,
            imports: () => this._imports,
            projections: () => this._projections,
            plugins: () => this._plugins,
            ignoreMissingPlugins: this.ignoreMissingPlugins,
            maven: () => this._maven,
            sources: () => this._sources,
        };
        new json_1.JsonFile(this.project, "smithy-build.json", {
            obj: this.manifest,
            readonly: true, // we want "yarn add" to work and we have anti-tamper
            newline: false, // when file is edited by npm/yarn it doesn't include a newline
        });
    }
    /**
     * Get configured projections
     */
    get projections() {
        return this._projections ? { ...this._projections } : undefined;
    }
    /**
     * Get configured imports
     */
    get imports() {
        return this._imports ? [...this._imports] : undefined;
    }
    /**
     * Get configured plugins
     */
    get plugins() {
        return this._plugins ? { ...this._plugins } : undefined;
    }
    /**
     * Add a smithy build import
     */
    addImport(imp) {
        this._imports ? this._imports.push(imp) : (this._imports = [`${imp}`]);
    }
    /**
     * Add smithy build projections
     */
    addProjections(projections) {
        for (const [k, v] of Object.entries(projections)) {
            this._projections
                ? (this._projections[k] = v)
                : (this._projections = { [k]: v });
        }
    }
    /**
     * Add smithy build plugins
     */
    addPlugins(plugins) {
        for (const [k, v] of Object.entries(plugins)) {
            this._plugins ? (this._plugins[k] = v) : (this._plugins = { [k]: v });
        }
    }
    /**
     * Add maven dependencies to the smithy build for the vs-code plugin
     */
    addMavenDependencies(...dependencies) {
        this._maven = {
            dependencies: [...(this._maven?.dependencies ?? []), ...dependencies],
            repositories: this._maven?.repositories ?? [],
        };
    }
    /**
     * Add maven repositories to the smithy build for the vs-code plugin
     */
    addMavenRepositories(...repositories) {
        this._maven = {
            dependencies: this._maven?.dependencies ?? [],
            repositories: [...(this._maven?.repositories ?? []), ...repositories],
        };
    }
    /**
     * Add relative paths to model source files or directories
     */
    addSources(...sources) {
        this._sources
            ? this._sources.push(...sources)
            : (this._sources = [...sources]);
    }
}
exports.SmithyBuild = SmithyBuild;
//# sourceMappingURL=data:application/json;base64,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