"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaskRuntime = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const path_1 = require("path");
const path = require("path");
const util_1 = require("util");
const chalk_1 = require("chalk");
const common_1 = require("./common");
const logging = require("./logging");
const tasks_1 = require("./util/tasks");
const ENV_TRIM_LEN = 20;
const ARGS_MARKER = "$@";
const QUOTED_ARGS_MARKER = `"${ARGS_MARKER}"`;
/**
 * The runtime component of the tasks engine.
 */
class TaskRuntime {
    constructor(workdir) {
        this.workdir = (0, path_1.resolve)(workdir);
        const manifestPath = (0, path_1.join)(this.workdir, TaskRuntime.MANIFEST_FILE);
        this.manifest = (0, fs_1.existsSync)(manifestPath)
            ? JSON.parse((0, fs_1.readFileSync)(manifestPath, "utf-8"))
            : { tasks: {} };
    }
    /**
     * The tasks in this project.
     */
    get tasks() {
        return Object.values(this.manifest.tasks ?? {});
    }
    /**
     * Find a task by name, or `undefined` if not found.
     */
    tryFindTask(name) {
        if (!this.manifest.tasks) {
            return undefined;
        }
        return this.manifest.tasks[name];
    }
    /**
     * Runs the task.
     * @param name The task name.
     */
    runTask(name, parents = [], args = [], env = {}) {
        const task = this.tryFindTask(name);
        if (!task) {
            throw new Error(`cannot find command ${task}`);
        }
        new RunTask(this, task, parents, args, env);
    }
}
exports.TaskRuntime = TaskRuntime;
_a = JSII_RTTI_SYMBOL_1;
TaskRuntime[_a] = { fqn: "projen.TaskRuntime", version: "0.91.13" };
/**
 * The project-relative path of the tasks manifest file.
 */
TaskRuntime.MANIFEST_FILE = path.posix.join(common_1.PROJEN_DIR, "tasks.json");
class RunTask {
    constructor(runtime, task, parents = [], args = [], envParam = {}) {
        this.runtime = runtime;
        this.task = task;
        this.env = {};
        this.workdir = task.cwd ?? this.runtime.workdir;
        this.parents = parents;
        if (!task.steps || task.steps.length === 0) {
            this.logDebug((0, chalk_1.gray)("No actions have been specified for this task."));
            return;
        }
        this.env = this.resolveEnvironment(envParam, parents);
        const envlogs = [];
        for (const [k, v] of Object.entries(this.env)) {
            const vv = v ?? "";
            const trimmed = vv.length > ENV_TRIM_LEN ? vv.substr(0, ENV_TRIM_LEN) + "..." : vv;
            envlogs.push(`${k}=${trimmed}`);
        }
        if (envlogs.length) {
            this.logDebug((0, chalk_1.gray)(`${(0, chalk_1.underline)("env")}: ${envlogs.join(" ")}`));
        }
        // evaluate condition
        if (!this.evalCondition(task)) {
            this.log("condition exited with non-zero - skipping");
            return;
        }
        // verify we required environment variables are defined
        const merged = { ...process.env, ...this.env };
        const missing = new Array();
        for (const name of task.requiredEnv ?? []) {
            if (!(name in merged)) {
                missing.push(name);
            }
        }
        if (missing.length > 0) {
            throw new Error(`missing required environment variables: ${missing.join(",")}`);
        }
        for (const step of task.steps) {
            // evaluate step condition
            if (!this.evalCondition(step)) {
                this.log("condition exited with non-zero - skipping");
                continue;
            }
            const argsList = [
                ...(step.args || []),
                ...(step.receiveArgs ? args : []),
            ].map((a) => a.toString());
            if (step.say) {
                logging.info(this.fmtLog(step.say));
            }
            if (step.spawn) {
                this.runtime.runTask(step.spawn, [...this.parents, this.task.name], argsList, step.env);
            }
            const execs = step.exec ? [step.exec] : [];
            // Parse step-specific environment variables
            const env = this.evalEnvironment(step.env ?? {});
            if (step.builtin) {
                execs.push(this.renderBuiltin(step.builtin));
            }
            for (const exec of execs) {
                let hasError = false;
                let command = (0, tasks_1.makeCrossPlatform)(exec);
                if (command.includes(QUOTED_ARGS_MARKER)) {
                    // Poorly imitate bash quoted variable expansion. If "$@" is encountered in bash, elements of the arg array
                    // that contain whitespace will be single quoted ('arg'). This preserves whitespace in things like filenames.
                    // Imitate that behavior here by single quoting every element of the arg array when a quoted arg marker ("$@")
                    // is encountered.
                    command = command.replace(QUOTED_ARGS_MARKER, argsList.map((arg) => `'${arg}'`).join(" "));
                }
                else if (command.includes(ARGS_MARKER)) {
                    command = command.replace(ARGS_MARKER, argsList.join(" "));
                }
                else {
                    command = [command, ...argsList].join(" ");
                }
                const cwd = step.cwd;
                try {
                    const result = this.shell({
                        command,
                        cwd,
                        extraEnv: env,
                    });
                    hasError = result.status !== 0;
                }
                catch (e) {
                    // This is the error 'shx' will throw
                    if (e?.message?.startsWith("non-zero exit code:")) {
                        hasError = true;
                    }
                    throw e;
                }
                if (hasError) {
                    throw new Error(`Task "${this.fullname}" failed when executing "${command}" (cwd: ${(0, path_1.resolve)(cwd ?? this.workdir)})`);
                }
            }
        }
    }
    /**
     * Determines if a task should be executed based on "condition".
     *
     * @returns true if the task should be executed or false if the condition
     * evaluates to false (exits with non-zero), indicating that the task should
     * be skipped.
     */
    evalCondition(taskOrStep) {
        // no condition, carry on
        if (!taskOrStep.condition) {
            return true;
        }
        this.log((0, chalk_1.gray)(`${(0, chalk_1.underline)("condition")}: ${taskOrStep.condition}`));
        const result = this.shell({
            command: taskOrStep.condition,
            logprefix: "condition: ",
            quiet: true,
        });
        if (result.status === 0) {
            return true;
        }
        else {
            return false;
        }
    }
    /**
     * Evaluates environment variables from shell commands (e.g. `$(xx)`)
     */
    evalEnvironment(env) {
        const output = {};
        for (const [key, value] of Object.entries(env ?? {})) {
            if (String(value).startsWith("$(") && String(value).endsWith(")")) {
                const query = value.substring(2, value.length - 1);
                const result = this.shellEval({ command: query });
                if (result.status !== 0) {
                    const error = result.error
                        ? result.error.stack
                        : result.stderr?.toString() ?? "unknown error";
                    throw new Error(`unable to evaluate environment variable ${key}=${value}: ${error}`);
                }
                output[key] = result.stdout.toString("utf-8").trim();
            }
            else {
                output[key] = value;
            }
        }
        return output;
    }
    /**
     * Renders the runtime environment for a task. Namely, it supports this syntax
     * `$(xx)` for allowing environment to be evaluated by executing a shell
     * command and obtaining its result.
     */
    resolveEnvironment(envParam, parents) {
        let env = this.runtime.manifest.env ?? {};
        // add env from all parent tasks one by one
        for (const parent of parents) {
            env = {
                ...env,
                ...(this.runtime.tryFindTask(parent)?.env ?? {}),
            };
        }
        // apply task environment, then the specific env last
        env = {
            ...env,
            ...(this.task.env ?? {}),
            ...envParam,
        };
        return this.evalEnvironment(env ?? {});
    }
    /**
     * Returns the "full name" of the task which includes all it's parent task names concatenated by chevrons.
     */
    get fullname() {
        return [...this.parents, this.task.name].join(" » ");
    }
    log(...args) {
        logging.verbose(this.fmtLog(...args));
    }
    logDebug(...args) {
        logging.debug(this.fmtLog(...args));
    }
    fmtLog(...args) {
        return (0, util_1.format)(`${(0, chalk_1.underline)(this.fullname)} |`, ...args);
    }
    shell(options) {
        const quiet = options.quiet ?? false;
        if (!quiet) {
            const log = new Array();
            if (options.logprefix) {
                log.push(options.logprefix);
            }
            log.push(options.command);
            if (options.cwd) {
                log.push(`(cwd: ${options.cwd})`);
            }
            this.log(log.join(" "));
        }
        const cwd = options.cwd ?? this.workdir;
        if (!(0, fs_1.existsSync)(cwd) || !(0, fs_1.statSync)(cwd).isDirectory()) {
            throw new Error(`invalid workdir (cwd): ${cwd} must be an existing directory`);
        }
        return (0, child_process_1.spawnSync)(options.command, {
            ...options,
            cwd,
            shell: true,
            stdio: "inherit",
            env: {
                ...process.env,
                ...this.env,
                ...options.extraEnv,
            },
            ...options.spawnOptions,
        });
    }
    shellEval(options) {
        return this.shell({
            quiet: true,
            ...options,
            spawnOptions: {
                stdio: ["inherit", "pipe", "inherit"],
            },
        });
    }
    renderBuiltin(builtin) {
        const moduleRoot = (0, path_1.dirname)(require.resolve("../package.json"));
        const program = require.resolve((0, path_1.join)(moduleRoot, "lib", `${builtin}.task.js`));
        return `"${process.execPath}" "${program}"`;
    }
}
//# sourceMappingURL=data:application/json;base64,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