"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.exec = exec;
exports.execCapture = execCapture;
exports.execOrUndefined = execOrUndefined;
exports.getFilePermissions = getFilePermissions;
exports.writeFile = writeFile;
exports.decamelizeKeysRecursively = decamelizeKeysRecursively;
exports.isTruthy = isTruthy;
exports.isObject = isObject;
exports.deepMerge = deepMerge;
exports.dedupArray = dedupArray;
exports.sorted = sorted;
exports.formatAsPythonModule = formatAsPythonModule;
exports.getGitVersion = getGitVersion;
exports.kebabCaseKeys = kebabCaseKeys;
exports.snakeCaseKeys = snakeCaseKeys;
exports.tryReadFile = tryReadFile;
exports.tryReadFileSync = tryReadFileSync;
exports.isWritable = isWritable;
exports.assertExecutablePermissions = assertExecutablePermissions;
exports.isExecutable = isExecutable;
exports.getNodeMajorVersion = getNodeMajorVersion;
exports.anySelected = anySelected;
exports.multipleSelected = multipleSelected;
exports.isRoot = isRoot;
exports.findUp = findUp;
exports.normalizePersistedPath = normalizePersistedPath;
const child_process = require("child_process");
const fs_1 = require("fs");
const path = require("path");
const Case = require("case");
const logging = require("./logging");
const MAX_BUFFER = 10 * 1024 * 1024;
/**
 * Executes a command with STDOUT > STDERR.
 */
function exec(command, options) {
    logging.debug(command);
    child_process.execSync(command, {
        stdio: options.stdio || ["inherit", 2, "pipe"], // "pipe" for STDERR means it appears in exceptions
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
        env: options.env,
    });
}
/**
 * Executes command and returns STDOUT. If the command fails (non-zero), throws an error.
 */
function execCapture(command, options) {
    logging.debug(command);
    return child_process.execSync(command, {
        stdio: ["inherit", "pipe", "pipe"], // "pipe" for STDERR means it appears in exceptions
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
        env: {
            ...process.env,
            ...options.modEnv,
        },
    });
}
/**
 * Executes `command` and returns its value or undefined if the command failed.
 */
function execOrUndefined(command, options) {
    try {
        const value = child_process
            .execSync(command, {
            stdio: ["inherit", "pipe", "pipe"], // "pipe" for STDERR means it appears in exceptions
            maxBuffer: MAX_BUFFER,
            cwd: options.cwd,
        })
            .toString("utf-8")
            .trim();
        if (!value) {
            return undefined;
        } // an empty string is the same as undefined
        return value;
    }
    catch {
        return undefined;
    }
}
function getFilePermissions(options) {
    const readonly = options.readonly ?? false;
    const executable = options.executable ?? false;
    if (readonly && executable) {
        return "544";
    }
    else if (readonly) {
        return "444";
    }
    else if (executable) {
        return "755";
    }
    else {
        return "644";
    }
}
function writeFile(filePath, data, options = {}) {
    if ((0, fs_1.existsSync)(filePath)) {
        (0, fs_1.chmodSync)(filePath, "600");
    }
    (0, fs_1.mkdirSync)(path.dirname(filePath), { recursive: true });
    (0, fs_1.writeFileSync)(filePath, data);
    (0, fs_1.chmodSync)(filePath, getFilePermissions(options));
}
function decamelizeKeysRecursively(input, opt) {
    const shouldAlwaysDecamelize = () => true;
    const shouldDecamelize = opt?.shouldDecamelize ?? shouldAlwaysDecamelize;
    const separator = opt?.separator ?? "_";
    const path_ = opt?.path ?? [];
    const maxDepth = opt?.maxDepth ?? 10;
    if (path_.length > maxDepth) {
        throw new Error("Decamelled too deeply - check that the input has no circular references");
    }
    if (Array.isArray(input)) {
        return input.map((k, i) => decamelizeKeysRecursively(k, {
            ...opt,
            path: [...path_, i.toString()],
        }));
    }
    if (typeof input === "object" && input !== null) {
        const mappedObject = {};
        for (const [key, value] of Object.entries(input)) {
            const transformedKey = shouldDecamelize([...path_, key], value)
                ? decamelize(key, separator)
                : key;
            mappedObject[transformedKey] = decamelizeKeysRecursively(value, {
                ...opt,
                path: [...path_, key],
            });
        }
        return mappedObject;
    }
    return input;
}
/**
 * Returns false if value is unset or a falsey value, and true otherwise.
 * @param value an environment variable
 */
function isTruthy(value) {
    return !(value === undefined ||
        ["null", "undefined", "0", "false", ""].includes(value.toLocaleLowerCase()));
}
/**
 * Return whether the given value is an object
 *
 * Even though arrays and instances of classes technically are objects, we
 * usually want to treat them differently, so we return false in those cases.
 */
function isObject(x) {
    return (x !== null &&
        typeof x === "object" &&
        !Array.isArray(x) &&
        x.constructor.name === "Object");
}
/**
 * Recursively merge objects together
 *
 * The leftmost object is mutated and returned. Arrays are not merged
 * but overwritten just like scalars.
 *
 * If an object is merged into a non-object, the non-object is lost.
 *
 * `undefined`s will cause a value to be deleted if destructive is enabled.
 */
function deepMerge(objects, destructive = false) {
    function mergeOne(target, source) {
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (isObject(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof target[key] !== "object") {
                    target[key] = value;
                }
                if ("__$APPEND" in value && Array.isArray(value.__$APPEND)) {
                    if (Array.isArray(target[key])) {
                        target[key].push(...value.__$APPEND);
                    }
                    else {
                        target[key] = value.__$APPEND;
                    }
                }
                mergeOne(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof output === "object" &&
                    Object.keys(output).length === 0 &&
                    destructive) {
                    delete target[key];
                }
            }
            else if (value === undefined && destructive) {
                delete target[key];
            }
            else if (typeof value !== "undefined") {
                target[key] = value;
            }
        }
    }
    const others = objects.filter((x) => x != null);
    if (others.length === 0) {
        return {};
    }
    const into = others.splice(0, 1)[0];
    others.forEach((other) => mergeOne(into, other));
    return into;
}
/*
 * Deduplicate values in a list, returning a new array.
 * @param array list of values
 */
function dedupArray(array) {
    return array.filter((val, idx) => array.indexOf(val) === idx);
}
/**
 * Returns a sorted version of `x` or `undefined` if it is an empty array or object.
 */
function sorted(x) {
    if (x == null) {
        return undefined;
    }
    if (Array.isArray(x)) {
        if (x.length === 0) {
            return undefined;
        }
        return x.sort();
    }
    else if (typeof x === "object") {
        if (Object.keys(x).length === 0) {
            return undefined;
        }
        const result = {};
        for (const [key, value] of Object.entries(x).sort(([l], [r]) => l.localeCompare(r))) {
            result[key] = value;
        }
        return result;
    }
    else {
        return x;
    }
}
function formatAsPythonModule(name) {
    return name.replace(/-/g, "_").replace(/\./g, "_");
}
/**
 * Extract git version number from command line
 *
 * @param gitVersionOutput the output from `git version` CLI
 * @returns the version of git
 */
function getGitVersion(gitVersionOutput) {
    const match = gitVersionOutput.match(/\d+.\d+.\d+/);
    if (!match) {
        throw new Error("Unable to retrieve git version");
    }
    return match[0];
}
function kebabCaseKeys(obj, recursive = true) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => kebabCaseKeys(v, recursive));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = kebabCaseKeys(v, recursive);
        }
        result[decamelize(k).replace(/_/gm, "-")] = v;
    }
    return result;
}
function snakeCaseKeys(obj, recursive = true, exclusiveForRecordKeys = []) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => snakeCaseKeys(v, recursive, exclusiveForRecordKeys));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = snakeCaseKeys(v, recursive, exclusiveForRecordKeys);
        }
        const modifiedKey = exclusiveForRecordKeys.length == 0 || exclusiveForRecordKeys.includes(k)
            ? Case.snake(k)
            : k;
        result[modifiedKey] = v;
    }
    return result;
}
async function tryReadFile(file) {
    if (!(0, fs_1.existsSync)(file)) {
        return "";
    }
    return fs_1.promises.readFile(file, "utf-8");
}
function tryReadFileSync(file) {
    if (!(0, fs_1.existsSync)(file)) {
        return undefined;
    }
    return (0, fs_1.readFileSync)(file, "utf-8");
}
function isWritable(file) {
    try {
        (0, fs_1.accessSync)(file, fs_1.constants.W_OK);
        return true;
    }
    catch {
        return false;
    }
}
/**
 * Asserts that the file should be executable. Always returns true on Windows.
 *
 * In Windows, the executable attribute is stored in the system setting PATHEXT, not in each file. Then, checking for executability is equivalent to checking for existence. To bypass checking for executability, we always return true on Windows.
 *
 * @param filePath The path to the file
 * @param shouldBeExecutable Whether the file should be executable
 * @returns true if `filePath` executable attribute matches `shouldBeExecutable` or if the platform is Windows, false otherwise
 */
function assertExecutablePermissions(filePath, shouldBeExecutable) {
    const isWindows = process.platform === "win32";
    if (isWindows) {
        return true;
    }
    const prevExecutable = isExecutable(filePath);
    return prevExecutable === shouldBeExecutable;
}
function isExecutable(file) {
    try {
        (0, fs_1.accessSync)(file, fs_1.constants.X_OK);
        return true;
    }
    catch (e) {
        return false;
    }
}
function decamelize(s, sep = "_") {
    if (Case.of(s) === "camel") {
        return Case.lower(s, sep);
    }
    else {
        return s;
    }
}
function getNodeMajorVersion() {
    const match = process.version.match(/(\d+)\.(\d+)\.(\d+)/);
    if (match) {
        const [major] = match.slice(1).map((x) => parseInt(x));
        return major;
    }
    return undefined;
}
function anySelected(options) {
    return options.some((opt) => opt);
}
function multipleSelected(options) {
    return options.filter((opt) => opt).length > 1;
}
/**
 * Checks if a path is a FS root
 *
 * Optional uses a provided OS specific path implementation,
 * defaults to use the implementation for the current OS.
 *
 * @internal
 */
function isRoot(dir, osPathLib = path) {
    const parent = osPathLib.dirname(dir);
    return parent === dir;
}
/**
 * Run up project tree to find a file or directory
 *
 * @param lookFor the file or directory to look for
 * @param cwd current working directory, must be an absolute path
 * @returns path to the file or directory we are looking for, undefined if not found
 */
function findUp(lookFor, cwd = process.cwd()) {
    if ((0, fs_1.existsSync)(path.join(cwd, lookFor))) {
        return cwd;
    }
    if (isRoot(cwd)) {
        // This is a root
        return undefined;
    }
    return findUp(lookFor, path.dirname(cwd));
}
/**
 * Normalizes a path that is going to be persisted to have a cross platform representation.
 *
 * Normalized paths can be persisted and doesn't need to be modified when the platform changes.
 * `normalizePersistedPath` takes care of platform-specific properties like directory separator.
 * It uses `path.posix.sep` that is supported both in Windows and Unix platforms.
 *
 *
 * @param p the path to be normalized
 * @returns the normalized path
 */
function normalizePersistedPath(p) {
    return p.replace(/\\/g, path.posix.sep);
}
//# sourceMappingURL=data:application/json;base64,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