"""
Key definitions for htty
Based on the key parsing logic in ht's stdio.rs file.
"""

from enum import Enum
from typing import Union


class Press(Enum):
    """@private"""

    ENTER = "Enter"
    TAB = "Tab"
    SPACE = "Space"
    ESCAPE = "Escape"
    LEFT = "Left"
    RIGHT = "Right"
    UP = "Up"
    DOWN = "Down"
    HOME = "Home"
    END = "End"
    PAGE_UP = "PageUp"
    PAGE_DOWN = "PageDown"
    BACKSPACE = "Backspace"
    F1 = "F1"
    F2 = "F2"
    F3 = "F3"
    F4 = "F4"
    F5 = "F5"
    F6 = "F6"
    F7 = "F7"
    F8 = "F8"
    F9 = "F9"
    F10 = "F10"
    F11 = "F11"
    F12 = "F12"
    CTRL_A = "C-a"
    CTRL_B = "C-b"
    CTRL_C = "C-c"
    CTRL_D = "C-d"
    CTRL_E = "C-e"
    CTRL_F = "C-f"
    CTRL_G = "C-g"
    CTRL_H = "C-h"
    CTRL_I = "C-i"
    CTRL_J = "C-j"
    CTRL_K = "C-k"
    CTRL_L = "C-l"
    CTRL_M = "C-m"
    CTRL_N = "C-n"
    CTRL_O = "C-o"
    CTRL_P = "C-p"
    CTRL_Q = "C-q"
    CTRL_R = "C-r"
    CTRL_S = "C-s"
    CTRL_T = "C-t"
    CTRL_U = "C-u"
    CTRL_V = "C-v"
    CTRL_W = "C-w"
    CTRL_X = "C-x"
    CTRL_Y = "C-y"
    CTRL_Z = "C-z"
    CTRL_SHIFT_A = "C-A"
    CTRL_SHIFT_B = "C-B"
    CTRL_SHIFT_C = "C-C"
    CTRL_SHIFT_D = "C-D"
    CTRL_SHIFT_E = "C-E"
    CTRL_SHIFT_F = "C-F"
    CTRL_SHIFT_G = "C-G"
    CTRL_SHIFT_H = "C-H"
    CTRL_SHIFT_I = "C-I"
    CTRL_SHIFT_J = "C-J"
    CTRL_SHIFT_K = "C-K"
    CTRL_SHIFT_L = "C-L"
    CTRL_SHIFT_M = "C-M"
    CTRL_SHIFT_N = "C-N"
    CTRL_SHIFT_O = "C-O"
    CTRL_SHIFT_P = "C-P"
    CTRL_SHIFT_Q = "C-Q"
    CTRL_SHIFT_R = "C-R"
    CTRL_SHIFT_S = "C-S"
    CTRL_SHIFT_T = "C-T"
    CTRL_SHIFT_U = "C-U"
    CTRL_SHIFT_V = "C-V"
    CTRL_SHIFT_W = "C-W"
    CTRL_SHIFT_X = "C-X"
    CTRL_SHIFT_Y = "C-Y"
    CTRL_SHIFT_Z = "C-Z"
    CTRL_AT = "C-@"  # Also equivalent to C-Space
    CTRL_SPACE = "C-Space"  # Also equivalent to C-@
    CTRL_LBRACKET = "C-["  # Also equivalent to Escape
    CTRL_BACKSLASH = "C-\\"
    CTRL_RBRACKET = "C-]"
    CTRL_CARET = "C-^"  # Also equivalent to C-/
    CTRL_SLASH = "C-/"  # Also equivalent to C-^
    CTRL_MINUS = "C--"  # Also equivalent to C-_
    CTRL_UNDERSCORE = "C-_"  # Also equivalent to C--
    CTRL_LEFT = "C-Left"
    CTRL_RIGHT = "C-Right"
    CTRL_UP = "C-Up"
    CTRL_DOWN = "C-Down"
    CTRL_HOME = "C-Home"
    CTRL_END = "C-End"
    CTRL_PAGE_UP = "C-PageUp"
    CTRL_PAGE_DOWN = "C-PageDown"
    CTRL_F1 = "C-F1"
    CTRL_F2 = "C-F2"
    CTRL_F3 = "C-F3"
    CTRL_F4 = "C-F4"
    CTRL_F5 = "C-F5"
    CTRL_F6 = "C-F6"
    CTRL_F7 = "C-F7"
    CTRL_F8 = "C-F8"
    CTRL_F9 = "C-F9"
    CTRL_F10 = "C-F10"
    CTRL_F11 = "C-F11"
    CTRL_F12 = "C-F12"
    SHIFT_LEFT = "S-Left"
    SHIFT_RIGHT = "S-Right"
    SHIFT_UP = "S-Up"
    SHIFT_DOWN = "S-Down"
    SHIFT_HOME = "S-Home"
    SHIFT_END = "S-End"
    SHIFT_PAGE_UP = "S-PageUp"
    SHIFT_PAGE_DOWN = "S-PageDown"
    SHIFT_F1 = "S-F1"
    SHIFT_F2 = "S-F2"
    SHIFT_F3 = "S-F3"
    SHIFT_F4 = "S-F4"
    SHIFT_F5 = "S-F5"
    SHIFT_F6 = "S-F6"
    SHIFT_F7 = "S-F7"
    SHIFT_F8 = "S-F8"
    SHIFT_F9 = "S-F9"
    SHIFT_F10 = "S-F10"
    SHIFT_F11 = "S-F11"
    SHIFT_F12 = "S-F12"
    ALT_A = "A-a"
    ALT_B = "A-b"
    ALT_C = "A-c"
    ALT_D = "A-d"
    ALT_E = "A-e"
    ALT_F = "A-f"
    ALT_G = "A-g"
    ALT_H = "A-h"
    ALT_I = "A-i"
    ALT_J = "A-j"
    ALT_K = "A-k"
    ALT_L = "A-l"
    ALT_M = "A-m"
    ALT_N = "A-n"
    ALT_O = "A-o"
    ALT_P = "A-p"
    ALT_Q = "A-q"
    ALT_R = "A-r"
    ALT_S = "A-s"
    ALT_T = "A-t"
    ALT_U = "A-u"
    ALT_V = "A-v"
    ALT_W = "A-w"
    ALT_X = "A-x"
    ALT_Y = "A-y"
    ALT_Z = "A-z"
    ALT_SHIFT_A = "A-A"
    ALT_SHIFT_B = "A-B"
    ALT_SHIFT_C = "A-C"
    ALT_SHIFT_D = "A-D"
    ALT_SHIFT_E = "A-E"
    ALT_SHIFT_F = "A-F"
    ALT_SHIFT_G = "A-G"
    ALT_SHIFT_H = "A-H"
    ALT_SHIFT_I = "A-I"
    ALT_SHIFT_J = "A-J"
    ALT_SHIFT_K = "A-K"
    ALT_SHIFT_L = "A-L"
    ALT_SHIFT_M = "A-M"
    ALT_SHIFT_N = "A-N"
    ALT_SHIFT_O = "A-O"
    ALT_SHIFT_P = "A-P"
    ALT_SHIFT_Q = "A-Q"
    ALT_SHIFT_R = "A-R"
    ALT_SHIFT_S = "A-S"
    ALT_SHIFT_T = "A-T"
    ALT_SHIFT_U = "A-U"
    ALT_SHIFT_V = "A-V"
    ALT_SHIFT_W = "A-W"
    ALT_SHIFT_X = "A-X"
    ALT_SHIFT_Y = "A-Y"
    ALT_SHIFT_Z = "A-Z"
    ALT_0 = "A-0"
    ALT_1 = "A-1"
    ALT_2 = "A-2"
    ALT_3 = "A-3"
    ALT_4 = "A-4"
    ALT_5 = "A-5"
    ALT_6 = "A-6"
    ALT_7 = "A-7"
    ALT_8 = "A-8"
    ALT_9 = "A-9"
    ALT_SPACE = "A- "
    ALT_EXCLAMATION = "A-!"
    ALT_AT = "A-@"
    ALT_HASH = "A-#"
    ALT_DOLLAR = "A-$"
    ALT_PERCENT = "A-%"
    ALT_CARET = "A-^"
    ALT_AMPERSAND = "A-&"
    ALT_ASTERISK = "A-*"
    ALT_LPAREN = "A-("
    ALT_RPAREN = "A-)"
    ALT_MINUS = "A--"
    ALT_UNDERSCORE = "A-_"
    ALT_EQUALS = "A-="
    ALT_PLUS = "A-+"
    ALT_LBRACKET = "A-["
    ALT_RBRACKET = "A-]"
    ALT_BACKSLASH = "A-\\"
    ALT_PIPE = "A-|"
    ALT_SEMICOLON = "A-;"
    ALT_COLON = "A-:"
    ALT_QUOTE = "A-'"
    ALT_DOUBLEQUOTE = 'A-"'
    ALT_COMMA = "A-,"
    ALT_DOT = "A-."
    ALT_SLASH = "A-/"
    ALT_QUESTION = "A-?"
    ALT_BACKTICK = "A-`"
    ALT_TILDE = "A-~"
    ALT_LEFT = "A-Left"
    ALT_RIGHT = "A-Right"
    ALT_UP = "A-Up"
    ALT_DOWN = "A-Down"
    ALT_HOME = "A-Home"
    ALT_END = "A-End"
    ALT_PAGE_UP = "A-PageUp"
    ALT_PAGE_DOWN = "A-PageDown"
    ALT_F1 = "A-F1"
    ALT_F2 = "A-F2"
    ALT_F3 = "A-F3"
    ALT_F4 = "A-F4"
    ALT_F5 = "A-F5"
    ALT_F6 = "A-F6"
    ALT_F7 = "A-F7"
    ALT_F8 = "A-F8"
    ALT_F9 = "A-F9"
    ALT_F10 = "A-F10"
    ALT_F11 = "A-F11"
    ALT_F12 = "A-F12"
    CTRL_SHIFT_LEFT = "C-S-Left"
    CTRL_SHIFT_RIGHT = "C-S-Right"
    CTRL_SHIFT_UP = "C-S-Up"
    CTRL_SHIFT_DOWN = "C-S-Down"
    CTRL_ALT_LEFT = "C-A-Left"
    CTRL_ALT_RIGHT = "C-A-Right"
    CTRL_ALT_UP = "C-A-Up"
    CTRL_ALT_DOWN = "C-A-Down"
    ALT_SHIFT_LEFT = "A-S-Left"
    ALT_SHIFT_RIGHT = "A-S-Right"
    ALT_SHIFT_UP = "A-S-Up"
    ALT_SHIFT_DOWN = "A-S-Down"
    CTRL_ALT_SHIFT_LEFT = "C-A-S-Left"
    CTRL_ALT_SHIFT_RIGHT = "C-A-S-Right"
    CTRL_ALT_SHIFT_UP = "C-A-S-Up"
    CTRL_ALT_SHIFT_DOWN = "C-A-S-Down"

    # ===== CARET NOTATION (alternative syntax for Control keys) =====
    # ht supports both C-a and ^a for the same keys
    CARET_A = "^a"
    CARET_B = "^b"
    CARET_C = "^c"
    CARET_D = "^d"
    CARET_E = "^e"
    CARET_F = "^f"
    CARET_G = "^g"
    CARET_H = "^h"
    CARET_I = "^i"
    CARET_J = "^j"
    CARET_K = "^k"
    CARET_L = "^l"
    CARET_M = "^m"
    CARET_N = "^n"
    CARET_O = "^o"
    CARET_P = "^p"
    CARET_Q = "^q"
    CARET_R = "^r"
    CARET_S = "^s"
    CARET_T = "^t"
    CARET_U = "^u"
    CARET_V = "^v"
    CARET_W = "^w"
    CARET_X = "^x"
    CARET_Y = "^y"
    CARET_Z = "^z"
    CARET_SHIFT_A = "^A"
    CARET_SHIFT_B = "^B"
    CARET_SHIFT_C = "^C"
    CARET_SHIFT_D = "^D"
    CARET_SHIFT_E = "^E"
    CARET_SHIFT_F = "^F"
    CARET_SHIFT_G = "^G"
    CARET_SHIFT_H = "^H"
    CARET_SHIFT_I = "^I"
    CARET_SHIFT_J = "^J"
    CARET_SHIFT_K = "^K"
    CARET_SHIFT_L = "^L"
    CARET_SHIFT_M = "^M"
    CARET_SHIFT_N = "^N"
    CARET_SHIFT_O = "^O"
    CARET_SHIFT_P = "^P"
    CARET_SHIFT_Q = "^Q"
    CARET_SHIFT_R = "^R"
    CARET_SHIFT_S = "^S"
    CARET_SHIFT_T = "^T"
    CARET_SHIFT_U = "^U"
    CARET_SHIFT_V = "^V"
    CARET_SHIFT_W = "^W"
    CARET_SHIFT_X = "^X"
    CARET_SHIFT_Y = "^Y"
    CARET_SHIFT_Z = "^Z"
    CARET_AT = "^@"
    CARET_LBRACKET = "^["
    CARET_BACKSLASH = "^\\"
    CARET_RBRACKET = "^]"


KeyInput = Union[Press, str]


def key_to_string(key: KeyInput) -> str:
    if isinstance(key, str):
        return key
    elif hasattr(key, "value"):
        return key.value
    else:
        return str(key)


def keys_to_strings(keys: Union[KeyInput, list[KeyInput]]) -> list[str]:
    if not isinstance(keys, list):
        keys = [keys]

    return [key_to_string(key) for key in keys]
