"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createOrUpdateRepoSecretParameter = exports.createOrUpdateOrgSecretParameter = exports.createOrUpdateEnvironmentSecretParameter = exports.executeGithubApiCall = void 0;
const encrypt_1 = require("./encrypt");
const get_secret_value_1 = require("./get-secret-value");
exports.executeGithubApiCall = async (octokit, call) => {
    switch (`${call.endpoint}.${call.method}`) {
        // Handles the create or update of actions secrets. Retrieves and encrypts the SecretsManager Secret.
        case "actions.createOrUpdateEnvironmentSecret":
            return exports.createOrUpdateEnvironmentSecretParameter(octokit, call.parameters);
        case "actions.createOrUpdateOrgSecret":
            return exports.createOrUpdateOrgSecretParameter(octokit, call.parameters);
        case "actions.createOrUpdateRepoSecret":
            return exports.createOrUpdateRepoSecretParameter(octokit, call.parameters);
        default:
            // https://github.com/octokit/plugin-rest-endpoint-methods.js/#usage
            // @ts-ignore
            return octokit.rest[call.endpoint][call.method](parameters);
    }
};
/**
 * Executes the environment secret encryption.
 */
exports.createOrUpdateEnvironmentSecretParameter = async (octokit, parameters) => {
    const { data: { key_id, key }, } = await octokit.rest.actions.getEnvironmentPublicKey({
        repository_id: parameters.repository_id,
        environment_name: parameters.environment_name,
    });
    const { arn, field } = parameters.value;
    const secretString = await get_secret_value_1.getSecretValue(arn, field);
    const encryptedValue = await encrypt_1.encrypt(secretString, key);
    return octokit.rest.actions.createOrUpdateEnvironmentSecret({
        repository_id: parameters.repository_id,
        environment_name: parameters.environment_name,
        secret_name: parameters.secret_name,
        encrypted_value: encryptedValue,
        key_id: key_id,
    });
};
/**
 * Executes the org secret encryption.
 */
exports.createOrUpdateOrgSecretParameter = async (octokit, parameters) => {
    const { data: { key_id, key }, } = await octokit.rest.actions.getOrgPublicKey({
        org: parameters.org,
    });
    const { arn, field } = parameters.value;
    const secretString = await get_secret_value_1.getSecretValue(arn, field);
    const encryptedValue = await encrypt_1.encrypt(secretString, key);
    return octokit.rest.actions.createOrUpdateOrgSecret({
        org: parameters.org,
        visibility: parameters.visibility,
        secret_name: parameters.secret_name,
        encrypted_value: encryptedValue,
        key_id: key_id,
    });
};
/**
 * Executes the environment secret encryption.
 */
exports.createOrUpdateRepoSecretParameter = async (octokit, parameters) => {
    const { data: { key_id, key }, } = await octokit.rest.actions.getRepoPublicKey({
        owner: parameters.owner,
        repo: parameters.repo,
    });
    const { arn, field } = parameters.value;
    const secretString = await get_secret_value_1.getSecretValue(arn, field);
    const encryptedValue = await encrypt_1.encrypt(secretString, key);
    return octokit.rest.actions.createOrUpdateRepoSecret({
        owner: parameters.owner,
        repo: parameters.repo,
        secret_name: parameters.secret_name,
        encrypted_value: encryptedValue,
        key_id: key_id,
    });
};
//# sourceMappingURL=data:application/json;base64,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