"""
Maps all the MQTT topics to either attributes or metrics.
"""

from typing import List
from .constants import MetricKind, MetricNature, MetricType, ValueType, RangeType
from .data_classes import TopicDescriptor
from ._victron_enums import DeviceType, InverterMode, GenericOnOff, EvChargerMode, InverterState, TemperatureStatus, TemperatureType

topics: List[TopicDescriptor] = [
    # generic device attributes
    TopicDescriptor(
        topic = "N/+/+/+/ProductName",
        message_type=MetricKind.ATTRIBUTE,
        short_id="model",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/+/+/+/Serial",
        message_type=MetricKind.ATTRIBUTE,
        short_id="serial_number",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        value_type=ValueType.STRING,
    ),
    # inverter hides its serial number away in a different topic
    TopicDescriptor(
        topic="N/+/vebus/+/Devices/0/SerialNumber",
        message_type=MetricKind.ATTRIBUTE,
        short_id="serial_number",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/+/+/+/Manufacturer",
        message_type=MetricKind.ATTRIBUTE,
        short_id="manufacturer",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/+/+/+/ProductId",
        message_type=MetricKind.ATTRIBUTE,
        short_id="victron_productid",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        value_type=ValueType.INT,
        precision=0,
    ),
    TopicDescriptor(
        topic="N/+/+/+/FirmwareVersion",
        message_type=MetricKind.ATTRIBUTE,
        short_id="firmware_version",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/+/+/+/CustomName",
        message_type=MetricKind.ATTRIBUTE,
        short_id="custom_name",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        value_type=ValueType.STRING,
    ),
    # grid
    TopicDescriptor(
        topic="N/+/system/+/Ac/Grid/NumberOfPhases",
        message_type=MetricKind.SENSOR,
        short_id="system_grid_phases",  # system attribute
        name="Grid phases",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.INT_DEFAULT_0,
        precision=0,
    ),
    # individual grid phases
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage_{phase}",
        name="Grid voltage on {phase}",
        unit_of_measurement="V",
        metric_type=MetricType.VOLTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="grid_current_{phase}",
        name="Grid current on {phase}",
        unit_of_measurement="A",
        metric_type=MetricType.CURRENT,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="grid_power_{phase}",
        name="Grid power on {phase}",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    # VM-3P75CT
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_forward_{phase}",
        name="Grid consumption on {phase}",
        unit_of_measurement="kWh",
        metric_type=MetricType.ENERGY,
        metric_nature=MetricNature.CUMULATIVE,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/Energy/Reverse",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_reverse_{phase}",
        name="Grid feed-in on {phase}",
        unit_of_measurement="kWh",
        metric_type=MetricType.ENERGY,
        metric_nature=MetricNature.CUMULATIVE,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/PENVoltage", 
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage_pen",
        name="Grid voltage on PEN",
        unit_of_measurement="V",
        metric_type=MetricType.VOLTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/N/Current",
        message_type=MetricKind.SENSOR,
        short_id="grid_current_n",
        name="Grid current on N",
        unit_of_measurement="A",
        metric_type=MetricType.CURRENT,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Frequency",
        message_type=MetricKind.SENSOR,
        short_id="grid_frequency",
        name="Grid frequency",
        unit_of_measurement="Hz",
        metric_type=MetricType.FREQUENCY,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=2,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/VoltageLineToLine",
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage_{phase}_{next_phase}",
        name="Grid voltage {phase} to {next_phase}",
        unit_of_measurement="V",
        metric_type=MetricType.VOLTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    # total grid
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage",
        name="Grid voltage",
        unit_of_measurement="V",
        metric_type=MetricType.VOLTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Current",
        message_type=MetricKind.SENSOR,
        short_id="grid_current",
        name="Grid current",
        unit_of_measurement="A",
        metric_type=MetricType.CURRENT,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Power",
        message_type=MetricKind.SENSOR,
        short_id="grid_power",
        name="Grid power",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_forward",
        name="Grid consumption",
        unit_of_measurement="kWh",
        metric_type=MetricType.ENERGY,
        metric_nature=MetricNature.CUMULATIVE,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Energy/Reverse",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_reverse",
        name="Grid feed-in",
        unit_of_measurement="kWh",
        metric_type=MetricType.ENERGY,
        metric_nature=MetricNature.CUMULATIVE,
        device_type=DeviceType.GRID,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    # solar / MPPT
    TopicDescriptor(
        topic="N/+/solarcharger/+/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="solar_voltage",
        name="DC Bus voltage",
        unit_of_measurement="V",
        metric_type=MetricType.VOLTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.SOLAR_CHARGER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="solar_current",
        name="DC current",
        unit_of_measurement="A",
        metric_type=MetricType.CURRENT,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.SOLAR_CHARGER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Yield/Power",
        message_type=MetricKind.SENSOR,
        short_id="solar_power",
        name="Power",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.SOLAR_CHARGER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Yield/User",
        message_type=MetricKind.SENSOR,
        short_id="solar_yield",
        name="Yield",
        unit_of_measurement="kWh",
        metric_type=MetricType.ENERGY,
        metric_nature=MetricNature.CUMULATIVE,
        device_type=DeviceType.SOLAR_CHARGER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/History/Daily/0/MaxPower",
        message_type=MetricKind.SENSOR,
        short_id="solar_max_power",
        name="Max Power Today",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.SOLAR_CHARGER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    # batteries
    TopicDescriptor(
        topic="N/+/battery/+/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_voltage",
        name="Battery voltage",
        unit_of_measurement="V",
        metric_type=MetricType.VOLTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="battery_current",
        name="Battery current",
        unit_of_measurement="A",
        metric_type=MetricType.CURRENT,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Dc/0/Power",
        message_type=MetricKind.SENSOR,
        short_id="battery_power",
        name="Battery power",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Dc/0/Temperature",
        message_type=MetricKind.SENSOR,
        short_id="battery_temperature",
        name="Battery temperature",
        unit_of_measurement="°C",
        metric_type=MetricType.TEMPERATURE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/History/DischargedEnergy",
        message_type=MetricKind.SENSOR,
        short_id="battery_discharged_energy",
        name="Discharged energy",
        unit_of_measurement="kWh",
        metric_type=MetricType.ENERGY,
        metric_nature=MetricNature.CUMULATIVE,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/History/ChargedEnergy",
        message_type=MetricKind.SENSOR,
        short_id="battery_charged_energy",
        name="Charged energy",
        unit_of_measurement="kWh",
        metric_type=MetricType.ENERGY,
        metric_nature=MetricNature.CUMULATIVE,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Capacity",
        message_type=MetricKind.SENSOR,
        short_id="battery_capacity",
        name="Battery capacity",
        unit_of_measurement="Ah",
        metric_type=MetricType.ELECTRIC_STORAGE_CAPACITY,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/InstalledCapacity",
        message_type=MetricKind.SENSOR,
        short_id="battery_installed_capacity",
        name="Battery installed capacity",
        unit_of_measurement="Ah",
        metric_type=MetricType.ELECTRIC_STORAGE_CAPACITY,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Soc",
        message_type=MetricKind.SENSOR,
        short_id="battery_soc",
        name="Battery charge",
        unit_of_measurement="%",
        metric_type=MetricType.PERCENTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/System/MinCellVoltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_min_cell_voltage",
        name="Battery minimum cell voltage",
        unit_of_measurement="V",
        metric_type=MetricType.VOLTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=3,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/System/MaxCellVoltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_max_cell_voltage",
        name="Battery maximum cell voltage",
        unit_of_measurement="V",
        metric_type=MetricType.VOLTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.BATTERY,
        value_type=ValueType.FLOAT,
        precision=3,
    ),
    # inverter
    TopicDescriptor(
        topic="N/+/vebus/+/Mode",
        message_type=MetricKind.SELECT,
        short_id="inverter_mode",
        name="Inverter mode",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        device_type=DeviceType.INVERTER,
        value_type=ValueType.ENUM,
        enum=InverterMode,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/State",
        message_type=MetricKind.SENSOR,
        short_id="inverter_state",
        name="Inverter state",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        device_type=DeviceType.INVERTER,
        value_type=ValueType.ENUM,
        enum=InverterState,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/ActiveIn/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="inverter_input_power_{phase}",
        name="Inverter input power {phase}",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.INVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/ActiveIn/{phase}/F",
        message_type=MetricKind.SENSOR,
        short_id="inverter_input_frequency_{phase}",
        name="Inverter input frequency {phase}",
        unit_of_measurement="Hz",
        metric_type=MetricType.FREQUENCY,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.INVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/ActiveIn/{phase}/S",
        message_type=MetricKind.SENSOR,
        short_id="inverter_input_apparent_power_{phase}",
        name="Inverter input apparent power {phase}",
        unit_of_measurement="VA",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.INVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/Out/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="inverter_output_power_{phase}",
        name="Inverter output power {phase}",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.INVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/Out/{phase}/F",
        message_type=MetricKind.SENSOR,
        short_id="inverter_output_frequency_{phase}",
        name="Inverter output frequency {phase}",
        unit_of_measurement="Hz",
        metric_type=MetricType.FREQUENCY,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.INVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/Out/{phase}/S",
        message_type=MetricKind.SENSOR,
        short_id="inverter_output_apparent_power_{phase}",
        name="Inverter output apparent power {phase}",
        unit_of_measurement="VA",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.INVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Connected",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="inverter_connected",
        name="Inverter connected",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        device_type=DeviceType.INVERTER,
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/ActiveIn/CurrentLimit",
        message_type=MetricKind.NUMBER,
        short_id="inverter_current_limit",
        name="Inverter current limit",
        unit_of_measurement="A",
        metric_type=MetricType.CURRENT,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.INVERTER,
        value_type=ValueType.INT,
        precision=0,
        min=0,
        max=RangeType.DYNAMIC,  # Dynamic range, depends on device model
        is_adjustable_suffix = "CurrentLimitIsAdjustable"
    ),
    # integrated system. Note that this might not be currently accurate for all systems
    #  as there are different physical configurations
    # and don't have access to any other for testing or feedback.
    TopicDescriptor(
        topic="N/+/system/+/Ac/ConsumptionOnOutput/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_critical_loads_{phase}",
        name="Critical loads on {phase}",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.SYSTEM,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Ac/ConsumptionOnInput/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_ac_loads_{phase}",
        name="AC loads on {phase}",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.SYSTEM,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Dc/System/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_consumption",
        name="DC Consumption",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.SYSTEM,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Relay/0/State",
        message_type=MetricKind.SWITCH,
        short_id="system_relay_1",
        name="Relay 1 state",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        device_type=DeviceType.SYSTEM,
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Relay/1/State",
        message_type=MetricKind.SWITCH,
        short_id="system_relay_2",
        name="Relay 2 state",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        device_type=DeviceType.SYSTEM,
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    # evcharger
    TopicDescriptor(
        topic="N/+/evcharger/+/Mode",
        message_type=MetricKind.SELECT,
        short_id="evcharger_mode",
        name="EV charger mode",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        device_type=DeviceType.EVCHARGER,
        value_type=ValueType.ENUM,
        enum=EvChargerMode,
    ),
    TopicDescriptor(
        topic="N/+/evcharger/+/StartStop",
        message_type=MetricKind.SWITCH,
        short_id="evcharger_charge",
        name="EV charger charge",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        device_type=DeviceType.EVCHARGER,
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/evcharger/+/Connected",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="evcharger_connected",
        name="EV charger connected",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        device_type=DeviceType.EVCHARGER,
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/evcharger/+/Current",
        message_type=MetricKind.SENSOR,
        short_id="evcharger_current",
        name="EV charger current",
        unit_of_measurement="A",
        metric_type=MetricType.CURRENT,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.EVCHARGER,
        value_type=ValueType.FLOAT,
        precision=1
    ),
    TopicDescriptor(
        topic="N/+/evcharger/+/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="evcharger_power_{phase}",
        name="EV charger power {phase}",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.EVCHARGER,
        value_type=ValueType.FLOAT,
        precision=1
    ),
    TopicDescriptor(
        topic="N/+/evcharger/+/SetCurrent",
        message_type=MetricKind.NUMBER,
        short_id="evcharger_set_current",
        name="EV charger set current",
        unit_of_measurement="A",
        metric_type=MetricType.CURRENT,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.EVCHARGER,
        value_type=ValueType.INT,
        precision=0,
        min=0,
        max=16,
    ),
    #pvinverter
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/{phase}/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="solar_voltage_{phase}",
        name="Voltage {phase}",
        unit_of_measurement="V",
        metric_type=MetricType.VOLTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.PVINVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="solar_current_{phase}",
        name="Current {phase}",
        unit_of_measurement="A",
        metric_type=MetricType.CURRENT,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.PVINVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/Power",
        message_type=MetricKind.SENSOR,
        short_id="solar_power_total",
        name="Power Total",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.PVINVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="solar_power_{phase}",
        name="Power {phase}",
        unit_of_measurement="W",
        metric_type=MetricType.POWER,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.PVINVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="solar_yield_total",
        name="Total Yield",
        unit_of_measurement="kWh",
        metric_type=MetricType.ENERGY,
        metric_nature=MetricNature.CUMULATIVE,
        device_type=DeviceType.PVINVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/{phase}/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="solar_yield_{phase}",
        name="Yield {phase}",
        unit_of_measurement="kWh",
        metric_type=MetricType.ENERGY,
        metric_nature=MetricNature.CUMULATIVE,
        device_type=DeviceType.PVINVERTER,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    # temperature devices
    TopicDescriptor(
        topic="N/+/temperature/+/Temperature",
        message_type=MetricKind.SENSOR,
        short_id="temperature_temperature",
        name="Temperature",
        unit_of_measurement="°C",
        metric_type=MetricType.TEMPERATURE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.TEMPERATURE,
        value_type=ValueType.INT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/temperature/+/Status",
        message_type=MetricKind.SENSOR,
        short_id="temperature_status",
        name="Temperature sensor status",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        device_type=DeviceType.TEMPERATURE,
        value_type=ValueType.ENUM,
        enum=TemperatureStatus,
    ),
    TopicDescriptor(
        topic="N/+/temperature/+/TemperatureType",
        message_type=MetricKind.SENSOR,
        short_id="temperature_type",
        name="Temperature sensor type",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.NONE,
        device_type=DeviceType.TEMPERATURE,
        value_type=ValueType.ENUM,
        enum=TemperatureType,
    ),
    TopicDescriptor(
        topic="N/+/temperature/+/Offset",
        message_type=MetricKind.NUMBER,
        short_id="temperature_offset",
        name="Temperature offset",
        unit_of_measurement="°C",
        metric_type=MetricType.TEMPERATURE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.TEMPERATURE,
        value_type=ValueType.FLOAT,
        precision=1,
    ),
    TopicDescriptor(
        topic="N/+/temperature/+/Scale",
        message_type=MetricKind.NUMBER,
        short_id="temperature_scale",
        name="Temperature scale factor",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.INSTANTANEOUS,
        device_type=DeviceType.TEMPERATURE,
        value_type=ValueType.FLOAT,
        precision=2,
    ),
]
