"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchDynamoTable = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const DEFAULT_PERCENT = 80;
class WatchDynamoTable extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const table = props.table;
        this.watchful = props.watchful;
        const cfnTable = table.node.defaultChild;
        const throughput = cfnTable.provisionedThroughput;
        const readCapacityMetric = metricForDynamoTable(table, 'ConsumedReadCapacityUnits', {
            label: 'Consumed',
            period: core_1.Duration.minutes(1),
            statistic: 'sum',
        });
        const writeCapacityMetric = metricForDynamoTable(table, 'ConsumedWriteCapacityUnits', {
            label: 'Consumed',
            period: core_1.Duration.minutes(1),
            statistic: 'sum',
        });
        this.watchful.addAlarm(this.createDynamoCapacityAlarm('read', readCapacityMetric, throughput.readCapacityUnits, props.readCapacityThresholdPercent));
        this.watchful.addAlarm(this.createDynamoCapacityAlarm('write', writeCapacityMetric, throughput.writeCapacityUnits, props.writeCapacityThresholdPercent));
        this.watchful.addSection(props.title, {
            links: [{ title: 'Amazon DynamoDB Console', url: linkForDynamoTable(table) }],
        });
        this.watchful.addWidgets(this.createDynamoCapacityGraph('Read', readCapacityMetric, throughput.readCapacityUnits, props.readCapacityThresholdPercent), this.createDynamoCapacityGraph('Write', writeCapacityMetric, throughput.writeCapacityUnits, props.writeCapacityThresholdPercent));
    }
    createDynamoCapacityGraph(type, metric, provisioned, percent = DEFAULT_PERCENT) {
        return new cloudwatch.GraphWidget({
            title: `${type} Capacity Units/${metric.period.toMinutes()}min`,
            width: 12,
            stacked: true,
            left: [metric],
            leftAnnotations: [
                {
                    label: 'Provisioned',
                    value: provisioned * metric.period.toSeconds(),
                    color: '#58D68D',
                },
                {
                    color: '#FF3333',
                    label: `Alarm on ${percent}%`,
                    value: calculateUnits(provisioned, percent, metric.period),
                },
            ],
        });
    }
    createDynamoCapacityAlarm(type, metric, provisioned, percent = DEFAULT_PERCENT) {
        const periodMinutes = 5;
        const threshold = calculateUnits(provisioned, percent, core_1.Duration.minutes(periodMinutes));
        const alarm = metric.createAlarm(this, `CapacityAlarm:${type}`, {
            alarmDescription: `at ${threshold}% of ${type} capacity`,
            threshold,
            period: core_1.Duration.minutes(periodMinutes),
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            statistic: 'sum',
        });
        return alarm;
    }
}
exports.WatchDynamoTable = WatchDynamoTable;
function linkForDynamoTable(table, tab = 'overview') {
    return `https://console.aws.amazon.com/dynamodb/home?region=${table.stack.region}#tables:selected=${table.tableName};tab=${tab}`;
}
function calculateUnits(provisioned, percent, period) {
    return provisioned * ((percent === undefined ? 80 : percent) / 100) * period.toSeconds();
}
function metricForDynamoTable(table, metricName, options = {}) {
    return new cloudwatch.Metric({
        metricName,
        namespace: 'AWS/DynamoDB',
        dimensions: {
            TableName: table.tableName,
        },
        unit: cloudwatch.Unit.COUNT,
        label: metricName,
        ...options,
    });
}
//# sourceMappingURL=data:application/json;base64,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