"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchEcsService = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const cdk = require("@aws-cdk/core");
class WatchEcsService extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.watchful = props.watchful;
        if (props.ec2Service) {
            this.ecsService = props.ec2Service;
        }
        if (props.fargateService) {
            this.ecsService = props.fargateService;
        }
        this.targetGroup = props.targetGroup;
        this.watchful.addSection(props.title, {
            links: [
                { title: 'ECS Service', url: linkForEcsService(this.ecsService) },
            ],
        });
        const { cpuUtilizationMetric, cpuUtilizationAlarm } = this.createCpuUtilizationMonitor(props.cpuMaximumThresholdPercent);
        const { memoryUtilizationMetric, memoryUtilizationAlarm } = this.createMemoryUtilizationMonitor(props.memoryMaximumThresholdPercent);
        const { targetResponseTimeMetric, targetResponseTimeAlarm } = this.createTargetResponseTimeMonitor(props.targetResponseTimeThreshold);
        const { healthyHostsMetric, unhealthyHostsMetric } = this.createHostCountMetrics();
        const { requestsMetric, requestsAlarm } = this.createRequestsMonitor(props.requestsThreshold);
        const { http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric } = this.createHttpRequestsMetrics();
        const { requestsErrorRateMetric, requestsErrorRateAlarm } = this.requestsErrorRate(http4xxMetric, http5xxMetric, requestsMetric);
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `CPUUtilization/${cpuUtilizationMetric.period.toMinutes()}min`,
            width: 12,
            left: [cpuUtilizationMetric],
            leftAnnotations: [cpuUtilizationAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `MemoryUtilization/${memoryUtilizationMetric.period.toMinutes()}min`,
            width: 12,
            left: [memoryUtilizationMetric],
            leftAnnotations: [memoryUtilizationAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.SingleValueWidget({
            title: 'Healthy Hosts',
            height: 6,
            width: 6,
            metrics: [healthyHostsMetric],
        }), new cloudwatch.SingleValueWidget({
            title: 'UnHealthy Hosts',
            height: 6,
            width: 6,
            metrics: [unhealthyHostsMetric],
        }), new cloudwatch.GraphWidget({
            title: `TargetResponseTime/${targetResponseTimeMetric.period.toMinutes()}min`,
            width: 6,
            left: [targetResponseTimeMetric],
            leftAnnotations: [targetResponseTimeAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Requests/${requestsMetric.period.toMinutes()}min`,
            width: 6,
            left: [requestsMetric],
            leftAnnotations: [requestsAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: 'HTTP Requests Overview',
            width: 12,
            left: [http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric],
        }), new cloudwatch.GraphWidget({
            title: `HTTP Requests Error rate/${requestsErrorRateMetric.period.toMinutes()}min`,
            width: 12,
            left: [requestsErrorRateMetric],
            leftAnnotations: [requestsErrorRateAlarm.toAnnotation()],
        }));
    }
    createCpuUtilizationMonitor(cpuMaximumThresholdPercent = 0) {
        const ecsService = this.ecsService;
        const cpuUtilizationMetric = ecsService.metricCpuUtilization();
        const cpuUtilizationAlarm = cpuUtilizationMetric.createAlarm(this, 'cpuUtilizationAlarm', {
            alarmDescription: 'cpuUtilizationAlarm',
            threshold: cpuMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(cpuUtilizationAlarm);
        return { cpuUtilizationMetric, cpuUtilizationAlarm };
    }
    createMemoryUtilizationMonitor(memoryMaximumThresholdPercent = 0) {
        const ecsService = this.ecsService;
        const memoryUtilizationMetric = ecsService.metricMemoryUtilization();
        const memoryUtilizationAlarm = memoryUtilizationMetric.createAlarm(this, 'memoryUtilizationAlarm', {
            alarmDescription: 'memoryUtilizationAlarm',
            threshold: memoryMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(memoryUtilizationAlarm);
        return { memoryUtilizationMetric, memoryUtilizationAlarm };
    }
    createTargetResponseTimeMonitor(targetResponseTimeThreshold = 0) {
        const targetGroup = this.targetGroup;
        const targetResponseTimeMetric = targetGroup.metricTargetResponseTime();
        const targetResponseTimeAlarm = targetResponseTimeMetric.createAlarm(this, 'targetResponseTimeAlarm', {
            alarmDescription: 'targetResponseTimeAlarm',
            threshold: targetResponseTimeThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(targetResponseTimeAlarm);
        return { targetResponseTimeMetric, targetResponseTimeAlarm };
    }
    createRequestsMonitor(requestsThreshold = 0) {
        const targetGroup = this.targetGroup;
        const requestsMetric = targetGroup.metricRequestCount();
        const requestsAlarm = requestsMetric.createAlarm(this, 'requestsAlarm', {
            alarmDescription: 'requestsAlarm',
            threshold: requestsThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(requestsAlarm);
        return { requestsMetric, requestsAlarm };
    }
    createHttpRequestsMetrics() {
        const targetGroup = this.targetGroup;
        const http2xxMetric = targetGroup.metricHttpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_2XX_COUNT);
        const http3xxMetric = targetGroup.metricHttpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_3XX_COUNT);
        const http4xxMetric = targetGroup.metricHttpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_4XX_COUNT);
        const http5xxMetric = targetGroup.metricHttpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_5XX_COUNT);
        return { http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric };
    }
    createHostCountMetrics() {
        const targetGroup = this.targetGroup;
        const healthyHostsMetric = targetGroup.metricHealthyHostCount();
        const unhealthyHostsMetric = targetGroup.metricUnhealthyHostCount();
        return { healthyHostsMetric, unhealthyHostsMetric };
    }
    requestsErrorRate(http4xxMetric, http5xxMetric, requestsMetric, requestsErrorRateThreshold = 0) {
        const requestsErrorRateMetric = new cloudwatch.MathExpression({
            expression: 'http4xx + http5xx / requests',
            usingMetrics: {
                http4xx: http4xxMetric,
                http5xx: http5xxMetric,
                requests: requestsMetric,
            },
        });
        const requestsErrorRateAlarm = requestsErrorRateMetric.createAlarm(this, 'requestsErrorRateAlarm', {
            alarmDescription: 'requestsErrorRateAlarm',
            threshold: requestsErrorRateThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(requestsErrorRateAlarm);
        return { requestsErrorRateMetric, requestsErrorRateAlarm };
    }
}
exports.WatchEcsService = WatchEcsService;
function linkForEcsService(ecsService) {
    return `https://console.aws.amazon.com/ecs/home?region=${ecsService.stack.region}#/clusters/${ecsService.cluster.clusterName}/services/${ecsService.serviceName}/details`;
}
//# sourceMappingURL=data:application/json;base64,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