"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchLambdaFunction = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const DEFAULT_DURATION_THRESHOLD_PERCENT = 80;
class WatchLambdaFunction extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const cfnFunction = props.fn.node.defaultChild;
        const timeoutSec = cfnFunction.timeout || 3;
        this.watchful = props.watchful;
        this.fn = props.fn;
        this.watchful.addSection(props.title, {
            links: [
                { title: 'AWS Lambda Console', url: linkForLambdaFunction(this.fn) },
                { title: 'CloudWatch Logs', url: linkForLambdaLogs(this.fn) },
            ],
        });
        const { errorsMetric, errorsAlarm } = this.createErrorsMonitor(props.errorsPerMinuteThreshold);
        const { throttlesMetric, throttlesAlarm } = this.createThrottlesMonitor(props.throttlesPerMinuteThreshold);
        const { durationMetric, durationAlarm } = this.createDurationMonitor(timeoutSec, props.durationThresholdPercent);
        const invocationsMetric = this.fn.metricInvocations();
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `Invocations/${invocationsMetric.period.toMinutes()}min`,
            width: 6,
            left: [invocationsMetric],
        }), new cloudwatch.GraphWidget({
            title: `Errors/${errorsMetric.period.toMinutes()}min`,
            width: 6,
            left: [errorsMetric],
            leftAnnotations: [errorsAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Throttles/${throttlesMetric.period.toMinutes()}min`,
            width: 6,
            left: [throttlesMetric],
            leftAnnotations: [throttlesAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Duration/${durationMetric.period.toMinutes()}min`,
            width: 6,
            left: [durationMetric],
            leftAnnotations: [durationAlarm.toAnnotation()],
        }));
    }
    createErrorsMonitor(errorsPerMinuteThreshold = 0) {
        const fn = this.fn;
        const errorsMetric = fn.metricErrors();
        const errorsAlarm = errorsMetric.createAlarm(this, 'ErrorsAlarm', {
            alarmDescription: `Over ${errorsPerMinuteThreshold} errors per minute`,
            threshold: errorsPerMinuteThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(errorsAlarm);
        return { errorsMetric, errorsAlarm };
    }
    createThrottlesMonitor(throttlesPerMinuteThreshold = 0) {
        const fn = this.fn;
        const throttlesMetric = fn.metricThrottles();
        const throttlesAlarm = throttlesMetric.createAlarm(this, 'ThrottlesAlarm', {
            alarmDescription: `Over ${throttlesPerMinuteThreshold} throttles per minute`,
            threshold: throttlesPerMinuteThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(throttlesAlarm);
        return { throttlesMetric, throttlesAlarm };
    }
    createDurationMonitor(timeoutSec, durationPercentThreshold = DEFAULT_DURATION_THRESHOLD_PERCENT) {
        const fn = this.fn;
        const durationMetric = fn.metricDuration();
        const durationThresholdSec = Math.floor(durationPercentThreshold / 100 * timeoutSec);
        const durationAlarm = durationMetric.createAlarm(this, 'DurationAlarm', {
            alarmDescription: `p99 latency >= ${durationThresholdSec}s (${durationPercentThreshold}%)`,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            threshold: durationThresholdSec * 1000,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(durationAlarm);
        return { durationMetric, durationAlarm };
    }
}
exports.WatchLambdaFunction = WatchLambdaFunction;
function linkForLambdaFunction(fn, tab = 'graph') {
    return `https://console.aws.amazon.com/lambda/home?region=${fn.stack.region}#/functions/${fn.functionName}?tab=${tab}`;
}
function linkForLambdaLogs(fn) {
    return `https://console.aws.amazon.com/cloudwatch/home?region=${fn.stack.region}#logEventViewer:group=/aws/lambda/${fn.functionName}`;
}
//# sourceMappingURL=data:application/json;base64,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