"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchRdsAurora = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
class WatchRdsAurora extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.watchful = props.watchful;
        this.cluster = props.cluster;
        this.watchful.addSection(props.title, {
            links: [
                { title: 'AWS RDS Cluster', url: linkForRDS(this.cluster) },
            ],
        });
        const { cpuUtilizationMetric, cpuUtilizationAlarm } = this.createCpuUtilizationMonitor(props.cpuMaximumThresholdPercent);
        const { dbConnectionsMetric, dbConnectionsAlarm } = this.createDbConnectionsMonitor(props.dbConnectionsMaximumThreshold);
        const { dbReplicaLagMetric, dbReplicaLagAlarm } = this.createDbReplicaLagMonitor(props.dbReplicaLagMaximumThreshold);
        const { dbBufferCacheHitRatioMetric, dbBufferCacheHitRatioAlarm } = this.createDbBufferCacheMonitor(props.dbBufferCacheMinimumThreshold);
        const { dbInsertThroughputMetric, dbUpdateThroughputMetric, dbSelectThroughputMetric, dbDeleteThroughputMetric } = this.createDbDmlThroughputMonitor(props.dbThroughputMaximumThreshold);
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `CPUUtilization/${cpuUtilizationMetric.period.toMinutes()}min`,
            width: 6,
            left: [cpuUtilizationMetric],
            leftAnnotations: [cpuUtilizationAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB Connections/${dbConnectionsMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbConnectionsMetric],
            leftAnnotations: [dbConnectionsAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB Replica Lag/${dbReplicaLagMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbReplicaLagMetric],
            leftAnnotations: [dbReplicaLagAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB BufferCache Hit Ratio/${dbBufferCacheHitRatioMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbBufferCacheHitRatioMetric],
            leftAnnotations: [dbBufferCacheHitRatioAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: 'RDS DML Overview',
            width: 24,
            left: [dbInsertThroughputMetric, dbUpdateThroughputMetric, dbSelectThroughputMetric, dbDeleteThroughputMetric],
        }));
    }
    createCpuUtilizationMonitor(cpuMaximumThresholdPercent = 80) {
        const cpuUtilizationMetric = new cloudwatch.Metric({
            metricName: 'CPUUtilization',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const cpuUtilizationAlarm = cpuUtilizationMetric.createAlarm(this, 'CpuUtilizationAlarm', {
            alarmDescription: 'cpuUtilizationAlarm',
            threshold: cpuMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { cpuUtilizationMetric, cpuUtilizationAlarm };
    }
    createDbConnectionsMonitor(dbConnectionsMaximumThreshold = 0) {
        const dbConnectionsMetric = new cloudwatch.Metric({
            metricName: 'DatabaseConnections',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbConnectionsAlarm = dbConnectionsMetric.createAlarm(this, 'DbConnectionsAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbConnectionsMaximumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbConnectionsMetric, dbConnectionsAlarm };
    }
    createDbReplicaLagMonitor(dbReplicaLagMaximumThreshold = 0) {
        const dbReplicaLagMetric = new cloudwatch.Metric({
            metricName: 'AuroraReplicaLag',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbReplicaLagAlarm = dbReplicaLagMetric.createAlarm(this, 'DbReplicaLagAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbReplicaLagMaximumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbReplicaLagMetric, dbReplicaLagAlarm };
    }
    createDbBufferCacheMonitor(dbBufferCacheMinimumThreshold = 0) {
        const dbBufferCacheHitRatioMetric = new cloudwatch.Metric({
            metricName: 'BufferCacheHitRatio',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbBufferCacheHitRatioAlarm = dbBufferCacheHitRatioMetric.createAlarm(this, 'DbBufferCacheHitRatioAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbBufferCacheMinimumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbBufferCacheHitRatioMetric, dbBufferCacheHitRatioAlarm };
    }
    createDbDmlThroughputMonitor(dbThroughputMaximumThreshold = 0) {
        // @ts-ignore
        const AlarmThreshold = dbThroughputMaximumThreshold;
        const dbInsertThroughputMetric = new cloudwatch.Metric({
            metricName: 'InsertThroughput',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbUpdateThroughputMetric = new cloudwatch.Metric({
            metricName: 'UpdateThroughput',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbSelectThroughputMetric = new cloudwatch.Metric({
            metricName: 'SelectThroughput',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        const dbDeleteThroughputMetric = new cloudwatch.Metric({
            metricName: 'DeleteThroughput',
            namespace: 'AWS/RDS',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
            dimensions: {
                DBClusterIdentifier: this.cluster.clusterIdentifier,
            },
        });
        return {
            dbInsertThroughputMetric,
            dbUpdateThroughputMetric,
            dbSelectThroughputMetric,
            dbDeleteThroughputMetric,
        };
    }
}
exports.WatchRdsAurora = WatchRdsAurora;
function linkForRDS(cluster) {
    return `https://console.aws.amazon.com/rds/home?region=${cluster.stack.region}#database:id=${cluster.clusterIdentifier};is-cluster=true`;
}
//# sourceMappingURL=data:application/json;base64,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