"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Watchful = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cloudwatch_actions = require("@aws-cdk/aws-cloudwatch-actions");
const sns = require("@aws-cdk/aws-sns");
const sns_subscriptions = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
const api_gateway_1 = require("./api-gateway");
const aspect_1 = require("./aspect");
const dynamodb_1 = require("./dynamodb");
const ecs_1 = require("./ecs");
const lambda_1 = require("./lambda");
const rds_aurora_1 = require("./rds-aurora");
class Watchful extends core_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        if ((props.alarmEmail || props.alarmSqs) && !props.alarmSns) {
            this.alarmTopic = new sns.Topic(this, 'AlarmTopic', { displayName: 'Watchful Alarms' });
        }
        if (props.alarmSns) {
            this.alarmTopic = props.alarmSns;
        }
        if (props.alarmEmail && this.alarmTopic) {
            this.alarmTopic.addSubscription(new sns_subscriptions.EmailSubscription(props.alarmEmail));
        }
        if (props.alarmSqs && this.alarmTopic) {
            this.alarmTopic.addSubscription(new sns_subscriptions.SqsSubscription(
            // sqs.Queue.fromQueueArn(this, 'AlarmQueue', props.alarmSqs)
            props.alarmSqs));
        }
        this.dash = new cloudwatch.Dashboard(this, 'Dashboard');
        new core_1.CfnOutput(this, 'WatchfulDashboard', {
            value: linkForDashboard(this.dash),
        });
    }
    addWidgets(...widgets) {
        this.dash.addWidgets(...widgets);
    }
    addAlarm(alarm) {
        if (this.alarmTopic) {
            alarm.addAlarmAction(new cloudwatch_actions.SnsAction(this.alarmTopic));
        }
    }
    addSection(title, options = {}) {
        const markdown = [
            `# ${title}`,
            (options.links || []).map(link => `[button:${link.title}](${link.url})`).join(' | '),
        ];
        this.addWidgets(new cloudwatch.TextWidget({ width: 24, markdown: markdown.join('\n') }));
    }
    watchScope(scope, options) {
        const aspect = new aspect_1.WatchfulAspect(this, options);
        scope.node.applyAspect(aspect);
    }
    watchDynamoTable(title, table, options = {}) {
        return new dynamodb_1.WatchDynamoTable(this, table.node.uniqueId, {
            title,
            watchful: this,
            table,
            ...options,
        });
    }
    watchApiGateway(title, restApi, options = {}) {
        return new api_gateway_1.WatchApiGateway(this, restApi.node.uniqueId, {
            title, watchful: this, restApi, ...options,
        });
    }
    watchLambdaFunction(title, fn, options = {}) {
        return new lambda_1.WatchLambdaFunction(this, fn.node.uniqueId, {
            title, watchful: this, fn, ...options,
        });
    }
    watchRdsAuroraCluster(title, cluster, options = {}) {
        return new rds_aurora_1.WatchRdsAurora(this, cluster.node.uniqueId, {
            title, watchful: this, cluster, ...options,
        });
    }
    watchFargateEcs(title, fargateService, targetGroup, options = {}) {
        return new ecs_1.WatchEcsService(this, fargateService.node.uniqueId, {
            title, watchful: this, fargateService, targetGroup, ...options,
        });
    }
    watchEc2Ecs(title, ec2Service, targetGroup, options = {}) {
        return new ecs_1.WatchEcsService(this, ec2Service.node.uniqueId, {
            title, watchful: this, ec2Service, targetGroup, ...options,
        });
    }
}
exports.Watchful = Watchful;
function linkForDashboard(dashboard) {
    const cfnDashboard = dashboard.node.defaultChild;
    return `https://console.aws.amazon.com/cloudwatch/home?region=${dashboard.stack.region}#dashboards:name=${cfnDashboard.ref}`;
}
//# sourceMappingURL=data:application/json;base64,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