import logging
import json
from typing import Optional, Dict, Any, Union
from jinja2 import Template
from pathlib import Path

from healthchain.pipeline.components.base import BaseComponent
from healthchain.io.containers import Document
from healthchain.models.responses.cdsresponse import Card, Source, IndicatorEnum


logger = logging.getLogger(__name__)


class CdsCardCreator(BaseComponent[str]):
    """
    Component that creates CDS Hooks cards from model outputs or static content.

    This component formats text into CDS Hooks cards that can be displayed in an EHR system.
    It can create cards from either:
    1. Model-generated text stored in a document's model outputs container
    2. Static content provided during initialization

    The component uses Jinja2 templates to format the text into valid CDS Hooks card JSON.
    The generated cards are added to the document's CDS container.

    Args:
        template (str, optional): Jinja2 template string for card creation. If not provided,
            uses a default template that creates an info card.
        template_path (Union[str, Path], optional): Path to a Jinja2 template file.
        static_content (str, optional): Static text to use instead of model output.
        source (str, optional): Source framework to get model output from (e.g. "huggingface").
        task (str, optional): Task name to get model output from (e.g. "summarization").
        delimiter (str, optional): String to split model output into multiple cards.
        default_source (Dict[str, Any], optional): Default source info for cards.
            Defaults to {"label": "Card Generated by HealthChain"}.

    Example:
        >>> # Create cards from model output
        >>> creator = CdsCardCreator(source="huggingface", task="summarization")
        >>> doc = creator(doc)  # Creates cards from model output
        >>>
        >>> # Create cards with static content
        >>> creator = CdsCardCreator(static_content="Static card message")
        >>> doc = creator(doc)  # Creates card with static content
        >>>
        >>> # Create cards with custom template
        >>> template = '''
        ... {
        ...     "summary": "{{ model_output[:140] }}",
        ...     "indicator": "info",
        ...     "source": {{ default_source | tojson }},
        ...     "detail": "{{ model_output }}"
        ... }
        ... '''
        >>> creator = CdsCardCreator(
        ...     template=template,
        ...     source="langchain",
        ...     task="chat",
        ...     delimiter="\n"
        ... )
        >>> doc = creator(doc)  # Creates cards split by newlines
    """

    # TODO: make source and other fields configurable from model too
    DEFAULT_TEMPLATE = """
    {
        "summary": "{{ model_output[:140] }}",
        "indicator": "info",
        "source": {{ default_source | tojson }},
        "detail": "{{ model_output }}"
    }
    """

    def __init__(
        self,
        template: Optional[str] = None,
        template_path: Optional[Union[str, Path]] = None,
        static_content: Optional[str] = None,
        source: Optional[str] = None,
        task: Optional[str] = None,
        delimiter: Optional[str] = None,
        default_source: Optional[Dict[str, Any]] = None,
    ):
        # Load template from file or use string template
        if template_path:
            try:
                template_path = Path(template_path)
                if not template_path.exists():
                    raise FileNotFoundError(f"Template file not found: {template_path}")
                with open(template_path) as f:
                    template = f.read()
            except Exception as e:
                logger.error(f"Error loading template from {template_path}: {str(e)}")
                template = self.DEFAULT_TEMPLATE

        self.template = Template(
            template if template is not None else self.DEFAULT_TEMPLATE
        )
        self.static_content = static_content
        self.source = source
        self.task = task
        self.delimiter = delimiter
        self.default_source = default_source or {
            "label": "Card Generated by HealthChain"
        }

    def create_card(self, content: str) -> Card:
        """Creates a CDS Card using the template and model output."""
        try:
            # Clean and escape the content
            # TODO: format to html that can be rendered in card
            content = content.replace("\n", " ").replace("\r", " ").strip()
            content = content.replace('"', '\\"')  # Escape double quotes

            try:
                card_json = self.template.render(
                    model_output=content, default_source=self.default_source
                )
            except Exception as e:
                raise ValueError(f"Error rendering template: {str(e)}")

            # Parse the rendered JSON into card fields
            card_fields = json.loads(card_json)

            return Card(
                summary=card_fields["summary"][:140],  # Enforce max length
                indicator=IndicatorEnum(card_fields["indicator"]),
                source=Source(**card_fields["source"]),
                detail=card_fields.get("detail"),
                suggestions=card_fields.get("suggestions"),
                selectionBehavior=card_fields.get("selectionBehavior"),
                overrideReasons=card_fields.get("overrideReasons"),
                links=card_fields.get("links"),
            )
        except Exception as e:
            raise ValueError(
                f"Error creating CDS card: Failed to render template or parse card fields: {str(e)}"
            )

    def __call__(self, doc: Document) -> Document:
        """
        Process a document and create CDS Hooks cards from model outputs or static content.

        Creates cards in one of two ways:
        1. From model-generated text stored in the document's model outputs container,
           accessed using the configured source and task
        2. From static content provided during initialization

        The generated text can optionally be split into multiple cards using a delimiter.
        Each piece of text is formatted using the configured template into a CDS Hooks card
        and added to the document's CDS container.

        Args:
            doc (Document): Document containing model outputs and CDS container

        Returns:
            Document: The input document with generated CDS cards added to its CDS container

        Raises:
            ValueError: If neither model configuration (source and task) nor static content
                is provided for card creation
        """
        if self.source and self.task:
            generated_text = doc.models.get_generated_text(self.source, self.task)
            if not generated_text:
                logger.warning(
                    f"No generated text for {self.source}/{self.task} found for CDS card creation!"
                )
                return doc
        elif self.static_content:
            generated_text = [self.static_content]
        else:
            raise ValueError(
                "Either model output (source and task) or content need to be provided for CDS card creation!"
            )

        # Create card from model output
        cards = []
        for text in generated_text:
            texts = [text] if not self.delimiter else text.split(self.delimiter)
            for t in texts:
                try:
                    cards.append(self.create_card(t))
                except Exception as e:
                    logger.warning(f"Error creating card: {str(e)}")

        if cards:
            doc.cds.cards = cards

        return doc
