"""
Synthea dataset loader.

Loads synthetic patient data generated by Synthea.
"""

import json
import logging
import random
from pathlib import Path
from typing import Dict, List, Optional

from fhir.resources.R4B.bundle import Bundle

from healthchain.sandbox.datasets import DatasetLoader

log = logging.getLogger(__name__)


class SyntheaFHIRPatientLoader(DatasetLoader):
    """
    Loader for Synthea-generated FHIR patient data.

    Synthea is an open-source synthetic patient generator that produces
    realistic patient records in FHIR format. This loader loads a single
    patient's Bundle (typically containing 100-500 resources), which is
    sufficient for quick demos and testing.

    The loader supports multiple ways to specify which patient file to load:
    - By patient_id (UUID portion of filename)
    - By patient_file (exact filename)
    - Default: first .json file found

    Examples:
        Load by patient ID:
        >>> client = SandboxClient(...)
        >>> client.load_from_registry(
        ...     "synthea-patient",
        ...     data_dir="./synthea_sample_data_fhir_latest",
        ...     patient_id="a969c177-a995-7b89-7b6d-885214dfa253",
        ...     resource_types=["Condition", "MedicationRequest"],
        ...     sample_size=5
        ... )

        Load by filename:
        >>> client.load_from_registry(
        ...     "synthea-patient",
        ...     data_dir="./synthea_sample_data_fhir_latest",
        ...     patient_file="Alton320_Gutkowski940_a969c177-a995-7b89-7b6d-885214dfa253.json"
        ... )

        Load first patient (quick demo):
        >>> client.load_from_registry(
        ...     "synthea-patient",
        ...     data_dir="./synthea_sample_data_fhir_latest"
        ... )
    """

    @property
    def name(self) -> str:
        """Dataset name for registration."""
        return "synthea-patient"

    @property
    def description(self) -> str:
        """Dataset description."""
        return "Synthea: Synthetic FHIR patient data generated by SyntheaTM (single patient per load)"

    def load(
        self,
        data_dir: str,
        patient_id: Optional[str] = None,
        patient_file: Optional[str] = None,
        resource_types: Optional[List[str]] = None,
        sample_size: Optional[int] = None,
        random_seed: Optional[int] = None,
        **kwargs,
    ) -> Dict[str, Bundle]:
        """
        Load a single Synthea FHIR patient Bundle.

        Args:
            data_dir: Path to Synthea FHIR output directory
            patient_id: Patient UUID (the ID portion of the filename after the name)
                       e.g., "a969c177-a995-7b89-7b6d-885214dfa253"
            patient_file: Exact filename to load
                         e.g., "Alton320_Gutkowski940_a969c177-a995-7b89-7b6d-885214dfa253.json"
            resource_types: FHIR resource types to include (e.g., ["Condition", "MedicationRequest"])
                          If None, all resource types are included
            sample_size: Number of resources to randomly sample per resource type
                        If None, all resources of each type are included
            random_seed: Random seed for reproducible sampling
            **kwargs: Additional parameters (reserved for future use)

        Returns:
            Dict mapping resource type to FHIR R4B Bundle in prefetch format
            e.g., {"Condition": Bundle(...), "MedicationStatement": Bundle(...)}

        Raises:
            FileNotFoundError: If data directory or patient file not found
            ValueError: If patient file is not a valid FHIR Bundle or no resources found

        Example:
            >>> loader = SyntheaFHIRPatientLoader()
            >>> data = loader.load(
            ...     data_dir="./synthea_sample_data_fhir_latest",
            ...     patient_id="a969c177-a995-7b89-7b6d-885214dfa253",
            ...     resource_types=["Condition", "MedicationRequest"],
            ...     sample_size=3
            ... )
            >>> # Returns: {"Condition": Bundle(...), "MedicationRequest": Bundle(...)}
        """
        data_dir = Path(data_dir)
        if not data_dir.exists():
            raise FileNotFoundError(
                f"Synthea data directory not found at: {data_dir}\n"
                "Please provide a valid data_dir containing Synthea FHIR patient files."
            )

        # Find the patient file
        patient_file_path = self._find_patient_file(data_dir, patient_id, patient_file)

        # Load and validate the Bundle
        bundle_dict = self._load_bundle(patient_file_path)

        # Log patient information
        self._log_patient_info(bundle_dict, patient_file_path.name)

        # Set random seed if provided
        if random_seed is not None:
            random.seed(random_seed)

        # Extract and group resources by type
        resources_by_type = self._extract_resources(bundle_dict, resource_types)

        if not resources_by_type:
            available_types = self._get_available_resource_types(bundle_dict)
            if resource_types:
                raise ValueError(
                    f"No resources found for requested types: {resource_types}\n"
                    f"Available resource types in this patient file: {available_types}"
                )
            else:
                raise ValueError(
                    f"No valid resources found in patient file: {patient_file_path.name}"
                )

        # Apply sampling if requested
        if sample_size is not None:
            resources_by_type = self._sample_resources(resources_by_type, sample_size)

        # Convert to Bundle objects
        bundles = {}
        for resource_type, resources in resources_by_type.items():
            bundles[resource_type.lower()] = Bundle(
                type="collection",
                entry=[{"resource": resource} for resource in resources],
            )
            log.info(
                f"Loaded {len(resources)} {resource_type} resource(s) from {patient_file_path.name}"
            )

        return bundles

    def _find_patient_file(
        self,
        data_dir: Path,
        patient_id: Optional[str] = None,
        patient_file: Optional[str] = None,
    ) -> Path:
        """
        Find the patient file to load based on provided parameters.

        Args:
            data_dir: Directory containing patient files
            patient_id: Patient UUID to search for
            patient_file: Exact filename

        Returns:
            Path to the patient file

        Raises:
            FileNotFoundError: If no matching file is found
            ValueError: If multiple files match the patient_id
        """
        # Option 1: Exact filename provided
        if patient_file:
            file_path = data_dir / patient_file
            if not file_path.exists():
                raise FileNotFoundError(
                    f"Patient file not found: {file_path}\n"
                    f"Please check that the file exists in {data_dir}"
                )
            return file_path

        # Option 2: Patient ID provided - search for matching file
        if patient_id:
            matching_files = list(data_dir.glob(f"*{patient_id}*.json"))
            if not matching_files:
                # List available files for helpful error message
                available_files = list(data_dir.glob("*.json"))
                error_msg = f"No patient file found with ID: {patient_id}\n"
                if available_files:
                    error_msg += f"\nAvailable patient files in {data_dir}:\n"
                    error_msg += "\n".join(f"  - {f.name}" for f in available_files[:5])
                    if len(available_files) > 5:
                        error_msg += f"\n  ... and {len(available_files) - 5} more"
                else:
                    error_msg += f"\nNo .json files found in {data_dir}"
                raise FileNotFoundError(error_msg)

            if len(matching_files) > 1:
                raise ValueError(
                    f"Multiple patient files found with ID '{patient_id}':\n"
                    + "\n".join(f"  - {f.name}" for f in matching_files)
                    + "\nPlease use patient_file parameter to specify the exact file."
                )
            return matching_files[0]

        # Option 3: Default - use first .json file
        json_files = list(data_dir.glob("*.json"))
        if not json_files:
            raise FileNotFoundError(
                f"No patient files (.json) found in {data_dir}\n"
                "Please ensure the directory contains Synthea FHIR patient files."
            )

        log.info(
            f"No patient_id or patient_file specified, using first file: {json_files[0].name}"
        )
        return json_files[0]

    def _load_bundle(self, file_path: Path) -> Dict:
        """
        Load and validate a Synthea FHIR Bundle from JSON file.

        Args:
            file_path: Path to the patient Bundle JSON file

        Returns:
            Bundle as dict

        Raises:
            ValueError: If file is not a valid FHIR Bundle
        """
        try:
            with open(file_path, "r") as f:
                bundle_dict = json.load(f)
        except json.JSONDecodeError as e:
            raise ValueError(f"Invalid JSON in file {file_path.name}: {e}")

        # Validate it's a FHIR Bundle
        if not isinstance(bundle_dict, dict):
            raise ValueError(
                f"File {file_path.name} does not contain a valid JSON object"
            )

        if bundle_dict.get("resourceType") != "Bundle":
            raise ValueError(
                f"File {file_path.name} is not a FHIR Bundle. "
                f"Found resourceType: {bundle_dict.get('resourceType')}"
            )

        if "entry" not in bundle_dict:
            raise ValueError(f"Bundle in {file_path.name} has no 'entry' field")

        return bundle_dict

    def _log_patient_info(self, bundle_dict: Dict, filename: str) -> None:
        """
        Log information about the loaded patient.

        Args:
            bundle_dict: Bundle dictionary
            filename: Name of the patient file
        """
        entries = bundle_dict.get("entry", [])
        total_resources = len(entries)

        # Try to find Patient resource for additional info
        patient_info = None
        for entry in entries:
            resource = entry.get("resource", {})
            if resource.get("resourceType") == "Patient":
                patient_id = resource.get("id", "unknown")
                name_list = resource.get("name", [])
                if name_list:
                    name = name_list[0]
                    given = " ".join(name.get("given", []))
                    family = name.get("family", "")
                    patient_info = f"{given} {family} (ID: {patient_id})"
                break

        if patient_info:
            log.info(
                f"Loaded patient: {patient_info} from {filename} ({total_resources} resources)"
            )
        else:
            log.info(f"Loaded patient file {filename} ({total_resources} resources)")

    def _get_available_resource_types(self, bundle_dict: Dict) -> List[str]:
        """
        Get list of available resource types in the bundle.

        Args:
            bundle_dict: Bundle dictionary

        Returns:
            List of unique resource types
        """
        resource_types = set()
        for entry in bundle_dict.get("entry", []):
            resource = entry.get("resource", {})
            resource_type = resource.get("resourceType")
            if resource_type:
                resource_types.add(resource_type)
        return sorted(resource_types)

    def _extract_resources(
        self, bundle_dict: Dict, resource_types: Optional[List[str]] = None
    ) -> Dict[str, List[Dict]]:
        """
        Extract and group resources by type from the bundle.

        Args:
            bundle_dict: Bundle dictionary
            resource_types: Optional list of resource types to filter by

        Returns:
            Dict mapping resource type to list of resource dicts
        """
        resources_by_type: Dict[str, List[Dict]] = {}

        for entry in bundle_dict.get("entry", []):
            resource = entry.get("resource", {})
            resource_type = resource.get("resourceType")

            if not resource_type:
                log.warning("Skipping entry with no resourceType")
                continue

            # Filter by resource types if specified
            if resource_types and resource_type not in resource_types:
                continue

            if resource_type not in resources_by_type:
                resources_by_type[resource_type] = []
            resources_by_type[resource_type].append(resource)

        return resources_by_type

    def _sample_resources(
        self, resources_by_type: Dict[str, List[Dict]], sample_size: int
    ) -> Dict[str, List[Dict]]:
        """
        Randomly sample resources from each type.

        Args:
            resources_by_type: Dict mapping resource type to list of resources
            sample_size: Number of resources to sample per type

        Returns:
            Dict with sampled resources
        """
        sampled = {}
        for resource_type, resources in resources_by_type.items():
            if len(resources) <= sample_size:
                sampled[resource_type] = resources
            else:
                sampled[resource_type] = random.sample(resources, sample_size)
                log.info(
                    f"Sampled {sample_size} of {len(resources)} {resource_type} resources"
                )
        return sampled
