# Matplotlab

Extended plotting and machine learning utilities library for educational purposes.

A comprehensive Python library providing:
- **Reinforcement Learning (RL)** - Monte Carlo, TD Learning, Policy/Value Iteration, Dynamic Programming
- **Artificial Neural Networks (ANN)** - Deep Learning implementations with PyTorch and TensorFlow
- **Speech Processing (SP)** - Audio analysis, MFCC extraction, vowel synthesis, formant analysis, dataset loading
- **Visualization Tools** - Enhanced plotting capabilities for ML workflows

## Installation

```bash
# Install from PyPI
pip install matplotlab

# Or install from source
git clone https://github.com/Sohail-Creates/matplotlab.git
cd matplotlab
pip install -e .
```

## Quick Start

### Reinforcement Learning
```python
from matplotlab import rl

# Create environment and find optimal policy
env = rl.create_frozenlake_env()
policy, V, iterations = rl.policy_iteration(env, gamma=0.99)
print(f"Converged in {iterations} iterations")

# Visualize results
rl.plot_value_heatmap(V)
rl.plot_grid_policy(policy)

# NEW: See complete lab workflow (Lab 1-6 + OEL)
rl.flowlab3()  # Shows complete Lab 3 code from import to visualization
rl.flowlab5()  # Shows complete Lab 5 (Policy Iteration) workflow
```

### Artificial Neural Networks
```python
from matplotlab import ann
import torch.nn as nn

# Create simple MLP model
model = ann.create_mlp_model(input_size=10, hidden_sizes=[16, 8], output_size=1)

# Or create CNN
cnn_model = ann.create_fashion_cnn()

# Training is straightforward
for epoch in range(50):
    y_pred = model(X_train)
    loss = loss_fn(y_pred, y_train)
    optimizer.zero_grad()
    loss.backward()
    optimizer.step()
```

### Speech Processing (NEW in v0.1.6!)
```python
from matplotlab import sp

# Synthesize vowel sounds
audio, sr = sp.synthesize_vowel('A', f0=150, duration=1.0)
sp.play_audio(audio, sr)

# Analyze formants (OEL - important for teacher questions!)
formants = sp.identify_formants(audio, sr)
# Shows: Waveform + Spectrum + Spectrogram with F1, F2, F3 marked

# Extract MFCC features
mfcc = sp.extract_mfcc(audio, sr, n_mfcc=13)
# Or show manual implementation
mfcc_manual = sp.extract_mfcc_from_scratch(audio, sr)

# Load datasets with automatic preview
dataset = sp.load_audio_dataset('vowels.zip', sr=16000)
# Automatically shows: structure, waveforms, sample info, audio players

# Generate synthetic datasets
dataset = sp.generate_vowel_dataset(n_samples_per_vowel=100)

# Plot spectrograms and analyze pitch
sp.plot_mel_spectrogram(audio, sr)
sp.plot_pitch_histogram(audio, sr)

# Complete OEL workflow
sp.flowoel()  # Shows complete vowel synthesis code
```

## Features

### ✅ Reinforcement Learning Module (44 functions)
- **Environments**: FrozenLake, Custom GridWorld
- **Algorithms**: Monte Carlo, TD Learning, Policy Iteration, Value Iteration
- **MDP Utilities**: State transitions, reward functions, probability computations
- **Visualization**: Heatmaps, policy arrows, convergence plots
- **Lab Workflows** (NEW): Complete code references for Labs 1-6 and OEL
  - `flowlab1()` through `flowlab6()` - Show full lab code workflows
  - `flowoel()` - Complete OEL implementation reference
  - Perfect for when you forget the sequence of steps!

### ✅ Artificial Neural Networks Module (85 functions)
- **Tensor Operations**: PyTorch basics, autograd
- **Perceptron**: sklearn implementation
- **ADALINE**: Manual and PyTorch versions
- **MLP**: Multi-layer perceptron for classification and regression
- **CNN**: Convolutional neural networks (simple nn.Sequential style)
- **Filters**: Custom CNN filters with TensorFlow
- **Transfer Learning**: Pre-trained model fine-tuning

### ✅ Speech Processing Module (42 functions) - NEW in v0.1.6!
- **Audio Loading**: Load, play, save, resample audio files
- **Spectrograms**: Linear, Mel, Narrowband, Wideband spectrograms
- **MFCC**: Extract MFCC features (librosa + manual implementation)
- **Vowel Synthesis**: Generate vowel sounds with formant frequencies (OEL)
- **Formant Analysis**: Identify F1, F2, F3 formants automatically
- **Pitch Analysis**: Extract and visualize pitch histograms
- **Dataset Utilities**: Load datasets from ZIP/Drive with auto-preview
- **Dataset Generation**: Create synthetic vowel datasets
- **Lab Workflows**: Complete code for Labs 1, 2, 4, 6, OEL, Quiz
- **OEL Functions**: Perfect for teacher questions about vowel synthesis and analysis!

## Requirements

### Core (always installed)
- Python >= 3.7
- NumPy >= 1.21.0

### Optional (install what you need)
```bash
# For Reinforcement Learning
pip install matplotlab[rl]

# For Artificial Neural Networks
pip install matplotlab[ann]

# For Speech Processing (NEW!)
pip install matplotlab[sp]

# Install everything
pip install matplotlab[all]
```

**Individual module dependencies:**
- **RL**: matplotlib, gymnasium, google-generativeai
- **ANN**: matplotlib, pytorch, tensorflow, scikit-learn
- **SP**: matplotlib, librosa, soundfile, scipy, ipython, google-generativeai

## Key Design Philosophy

**Simple, Beginner-Friendly Code:**
- Uses `nn.Sequential()` for neural networks (no complex classes)
- Clear variable names: `X_train`, `y_train`, `model`, `loss_fn`
- Simple for loops and if-else statements
- No lambda functions or advanced Python features
- Easy to understand and modify

## Documentation

- **171 total functions** (44 RL + 85 ANN + 42 SP)
- Complete docstrings for every function
- Usage examples included
- All functions have `.show()` method to view source code
- See `SP_MODULE_FUNCTIONS.md` for complete SP function guide
- See `DATASET_LOADING_GUIDE.md` for dataset utilities

### Quick Tips
```python
# View source code of any function
sp.synthesize_vowel.show()
rl.policy_iteration.show()
ann.create_mlp_model.show()

# Get AI help
sp.query("How do I extract MFCC features?")
rl.query("Explain policy iteration")
```

## License

MIT License - Free for educational use

## Links

- PyPI: https://pypi.org/project/matplotlab/
- GitHub: https://github.com/Sohail-Creates/matplotlab/

---

**For educational purposes** | ML/RL implementations made simple
