from __future__ import annotations
import os
from pathlib import Path
from typing import Optional, Union

from .model import ProjectPaths


def generate_static_model_text() -> str:
    """
    Generates the Python code for the static dataclass model.
    """
    lines = [
        "from pathlib import Path",
        "from dataclasses import dataclass, field",
        "",
        "# This file is auto-generated by ptool-serena. Do not edit manually.",
        "# Run `ptool gen-static` or `just regen` to regenerate.",
        "",
        "@dataclass(frozen=True)",
        "class ProjectPathsStatic:",
        '    """A static, auto-generated dataclass for project paths."""',
        "",
    ]

    # Instantiate from pyproject.toml to get the default fields
    model = ProjectPaths.from_pyproject()

    # Use Pydantic's model_fields to get the defined fields and their defaults
    for field_name, field_info in model.model_fields.items():
        # The default_factory is a lambda, we need to call it to get the path
        if field_info.default_factory:
            default_path = field_info.default_factory()  # type: ignore[call-arg,misc]

            # Special case: base_dir should use Path.cwd() for portability
            if field_name == "base_dir":
                line = f"    {field_name}: Path = field(default_factory=Path.cwd)"
            # Special case for Path.home() to use the factory
            elif default_path == Path.home():
                line = f"    {field_name}: Path = field(default_factory=Path.home)"
            else:
                # We want to store the path as a string relative to the base_dir for portability
                # Determine if the path is truly inside the base directory.
                # Using os.path.relpath can be tricky across OSes and for non-subpaths.
                # A more reliable check is to see if base_dir is one of the parents of the path.
                base_dir = getattr(model, "base_dir", Path.cwd())
                is_subpath = base_dir.resolve() in default_path.resolve().parents

                if is_subpath:
                    # For paths inside the project, make them relative for portability.
                    rel_path = os.path.relpath(
                        default_path.resolve(), base_dir.resolve()
                    )
                    rel_path_posix = str(Path(rel_path).as_posix())
                    line = f'    {field_name}: Path = field(default_factory=lambda: Path.cwd() / "{rel_path_posix}")'
                else:
                    # For external absolute paths, store them directly.
                    line = f'    {field_name}: Path = field(default=Path("{default_path}"))'

        else:
            # Handle fields with a simple default value
            line = (
                f'    {field_name}: Path = field(default=Path("{field_info.default}"))'
            )

        lines.append(line)

    return "\n".join(lines)


def write_dataclass_file(
    output_path: Optional[Union[str, Path]] = None,
    class_name: str = "ProjectPathsStatic",  # class_name is not used yet, but kept for future
) -> None:
    """
    Generates and writes the static dataclass file.
    """
    if output_path is None:
        # Default path within the src directory
        output_path = Path(__file__).parent / "project_paths_static.py"

    resolved_path = Path(output_path).resolve()
    resolved_path.parent.mkdir(parents=True, exist_ok=True)

    # Generate the code
    generated_code = generate_static_model_text()

    # Write to a temporary file first to avoid race conditions (TOCTOU)
    temp_path = resolved_path.with_suffix(".tmp")

    # Write and sync to disk before replace (prevents TOCTOU vulnerability)
    with temp_path.open("w", encoding="utf-8") as f:
        f.write(generated_code)
        f.flush()
        os.fsync(f.fileno())  # Ensure data is written to disk

    # Atomically replace the old file with the new one (atomic on POSIX)
    temp_path.replace(resolved_path)

    # Verify the file exists (after atomic replace, no TOCTOU risk)
    if resolved_path.exists():
        print(f"✅ Static model written to {resolved_path}")
    else:
        raise FileNotFoundError(f"❌ Failed to write static model to {resolved_path}")
