#!/usr/bin/env python3
"""NIST strong lines, autogenerated by helpers/strong_lines_gen.py."""

# pylint: disable=too-many-lines

from dataclasses import dataclass
from enum import Enum

class Flag(Enum):
    """Flags for strong line."""
    BAND_HEAD = ('b', 'band head')
    COMPLEX = ('c', 'complex')
    TWO_UNRES_LINES = ('d', 'line consists of two unresolved lines')
    HAZY = ('h', 'hazy')
    SHADED_LONGER = ('l', 'shaded to longer wavelengths')
    PERSISTENT = ('P', 'a persistent line')
    EASILY_REVERSED = ('r', 'easily reversed')
    SHADED_SHORTER = ('s', 'shaded to shorter wavelengths')
    UNRESOLVED = ('u', 'unresolved shoulder on strong line')
    WIDE = ('w', 'wide')
    TOO_CLOSE = ('*', 'peaks so close that separate intensity measurement was infeasible')

    def __init__(self, code: str, description: str):
        self.code = code
        self.description = description

    def __str__(self):
        """Convert to readable string"""
        return f'{self.code} ({self.description})'

_FLAG_MAP = {flag.code: flag for flag in Flag}

@dataclass(frozen=True)
class StrongLine:
    """Represents one strong line for given element."""
    element: str
    wavelength: float
    intensity: int
    raw_flags: str

    def flags(self):
        """Get humanized flags for raw_flags."""
        return [_FLAG_MAP[f] for f in list(self.raw_flags)]

@dataclass(frozen=True)
class StrongLines:
    """Represents collection of strong lines for given element."""
    element: str
    lines: list[StrongLine]
    persistent_lines: list[StrongLine]  # lines with 'P' flag

    def for_wavelength_range(self, wave_rng, only_persistent=False):
        """Return list of lines for given wavelength range."""
        src = self.persistent_lines if only_persistent else self.lines
        return [x for x in src if wave_rng.start <= x.wavelength <= wave_rng.stop]

    def for_intensity_range(self, intensity_rng, only_persistent=False):
        """Return list of lines for given intensity range."""
        src = self.persistent_lines if only_persistent else self.lines
        return [x for x in src if intensity_rng.start <= x.intensity <= intensity_rng.stop]

    def for_wavelength_and_intensity_range(self, wave_rng, int_rng, only_persistent=False):
        """Return list of lines for given intensity range."""
        src = self.persistent_lines if only_persistent else self.lines
        return [x for x in src if (
            wave_rng.start <= x.wavelength <= wave_rng.stop and
            int_rng.start <= x.intensity <= int_rng.stop)]

STRONG_LINES = {

    "Ar": StrongLines(
        element="Ar",
        lines=[
            StrongLine(element="Ar", wavelength=48.72272, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=49.06495, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=49.07013, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=51.93270, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=54.29124, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=54.32033, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=54.74606, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=55.68170, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=57.33619, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=57.67364, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=58.02632, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=58.34371, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=59.77001, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=60.28584, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=61.23716, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=66.18690, intensity=150, raw_flags="P"),
            StrongLine(element="Ar", wavelength=66.45623, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=66.60109, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=67.09455, intensity=300, raw_flags="P"),
            StrongLine(element="Ar", wavelength=67.18513, intensity=1000, raw_flags="P"),
            StrongLine(element="Ar", wavelength=67.62425, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=67.79518, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=67.92184, intensity=10, raw_flags=""),
            StrongLine(element="Ar", wavelength=67.94006, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=71.80899, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=72.33606, intensity=1000, raw_flags="P"),
            StrongLine(element="Ar", wavelength=72.55485, intensity=150, raw_flags=""),
            StrongLine(element="Ar", wavelength=73.09297, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=74.02692, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=74.49248, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=74.53223, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=80.285896, intensity=20, raw_flags=""),
            StrongLine(element="Ar", wavelength=80.64710, intensity=100, raw_flags=""),
            StrongLine(element="Ar", wavelength=80.686887, intensity=60, raw_flags=""),
            StrongLine(element="Ar", wavelength=80.721842, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=80.76529, intensity=40, raw_flags=""),
            StrongLine(element="Ar", wavelength=80.992660, intensity=50, raw_flags=""),
            StrongLine(element="Ar", wavelength=81.623193, intensity=120, raw_flags=""),
            StrongLine(element="Ar", wavelength=81.646391, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=82.012352, intensity=80, raw_flags=""),
            StrongLine(element="Ar", wavelength=82.534592, intensity=120, raw_flags=""),
            StrongLine(element="Ar", wavelength=82.636484, intensity=120, raw_flags=""),
            StrongLine(element="Ar", wavelength=83.43918, intensity=150, raw_flags=""),
            StrongLine(element="Ar", wavelength=83.500210, intensity=100, raw_flags=""),
            StrongLine(element="Ar", wavelength=84.280506, intensity=100, raw_flags=""),
            StrongLine(element="Ar", wavelength=86.679997, intensity=180, raw_flags="P"),
            StrongLine(element="Ar", wavelength=86.975411, intensity=150, raw_flags="P"),
            StrongLine(element="Ar", wavelength=87.605767, intensity=180, raw_flags="P"),
            StrongLine(element="Ar", wavelength=87.994656, intensity=180, raw_flags="P"),
            StrongLine(element="Ar", wavelength=89.431013, intensity=150, raw_flags="P"),
            StrongLine(element="Ar", wavelength=91.97810, intensity=300, raw_flags="P"),
            StrongLine(element="Ar", wavelength=93.20537, intensity=300, raw_flags="P"),
            StrongLine(element="Ar", wavelength=104.821987, intensity=1000, raw_flags="P"),
            StrongLine(element="Ar", wavelength=106.665980, intensity=500, raw_flags="P"),
            StrongLine(element="Ar", wavelength=242.0456, intensity=2, raw_flags=""),
            StrongLine(element="Ar", wavelength=251.6789, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=253.4709, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=256.2087, intensity=5, raw_flags=""),
            StrongLine(element="Ar", wavelength=289.1612, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=294.2893, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=297.9050, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=303.3508, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=309.3402, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=324.3689, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=329.3640, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=330.7228, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=335.0924, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=337.6436, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=338.8531, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=347.6747, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=347.8232, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=349.1244, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=349.1536, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=350.9778, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=351.4388, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=354.5596, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=354.5845, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=355.9508, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=356.1030, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=357.6616, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=358.1608, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=358.2355, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=358.8441, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=362.2138, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=363.9833, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=371.8206, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=372.9309, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=373.7889, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=376.5270, intensity=50, raw_flags=""),
            StrongLine(element="Ar", wavelength=376.6119, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=377.0369, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=377.0520, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=378.0840, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=380.3172, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=380.9456, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=385.0581, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=386.8528, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=392.5719, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=392.8623, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=393.2547, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=394.6097, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=394.8979, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=397.9356, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=399.4792, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=401.3857, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=403.3809, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=403.5460, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=404.2894, intensity=50, raw_flags=""),
            StrongLine(element="Ar", wavelength=404.4418, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=405.2921, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=407.2005, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=407.2385, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=407.6628, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=407.9574, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=408.2387, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=410.3912, intensity=50, raw_flags=""),
            StrongLine(element="Ar", wavelength=413.1724, intensity=100, raw_flags=""),
            StrongLine(element="Ar", wavelength=415.6086, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=415.8590, intensity=11, raw_flags=""),
            StrongLine(element="Ar", wavelength=416.4180, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=417.9297, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=418.1884, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=419.0713, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=419.1029, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=419.8317, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=420.0674, intensity=11, raw_flags=""),
            StrongLine(element="Ar", wavelength=421.8665, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=422.2637, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=422.6988, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=422.8158, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=423.7220, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=425.1185, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=425.9362, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=426.6286, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=426.6527, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=427.2169, intensity=4, raw_flags=""),
            StrongLine(element="Ar", wavelength=427.7528, intensity=200, raw_flags="P"),
            StrongLine(element="Ar", wavelength=428.2898, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=430.0101, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=430.0650, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=430.9239, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=433.1200, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=433.2030, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=433.3561, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=433.5338, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=434.5168, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=434.8064, intensity=250, raw_flags="P"),
            StrongLine(element="Ar", wavelength=435.2205, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=436.2066, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=436.7832, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=437.0753, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=437.1329, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=437.5954, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=437.9667, intensity=50, raw_flags=""),
            StrongLine(element="Ar", wavelength=438.5057, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=440.0097, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=440.0986, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=442.6001, intensity=130, raw_flags=""),
            StrongLine(element="Ar", wavelength=443.0189, intensity=50, raw_flags=""),
            StrongLine(element="Ar", wavelength=443.0996, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=443.3838, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=443.9461, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=444.8879, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=447.4759, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=448.1811, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=451.0733, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=452.2323, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=453.0552, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=454.5052, intensity=130, raw_flags=""),
            StrongLine(element="Ar", wavelength=456.4405, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=457.9350, intensity=130, raw_flags=""),
            StrongLine(element="Ar", wavelength=458.9898, intensity=130, raw_flags=""),
            StrongLine(element="Ar", wavelength=460.9567, intensity=200, raw_flags="P"),
            StrongLine(element="Ar", wavelength=463.7233, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=465.7901, intensity=130, raw_flags=""),
            StrongLine(element="Ar", wavelength=472.1591, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=472.6868, intensity=200, raw_flags="P"),
            StrongLine(element="Ar", wavelength=473.2053, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=473.5906, intensity=100, raw_flags=""),
            StrongLine(element="Ar", wavelength=476.4865, intensity=250, raw_flags="P"),
            StrongLine(element="Ar", wavelength=480.6020, intensity=200, raw_flags="P"),
            StrongLine(element="Ar", wavelength=484.7810, intensity=50, raw_flags=""),
            StrongLine(element="Ar", wavelength=486.5910, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=487.9864, intensity=250, raw_flags="P"),
            StrongLine(element="Ar", wavelength=488.9042, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=490.4752, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=493.3209, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=496.5080, intensity=70, raw_flags=""),
            StrongLine(element="Ar", wavelength=500.9334, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=501.7163, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=506.2037, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=509.0495, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=514.1783, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=514.5308, intensity=25, raw_flags=""),
            StrongLine(element="Ar", wavelength=516.5773, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=518.7746, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=521.6814, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=549.5874, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=555.8702, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=560.6733, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=565.0704, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=588.8584, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=591.2085, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=603.2127, intensity=2, raw_flags=""),
            StrongLine(element="Ar", wavelength=604.3223, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=605.9372, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=611.4923, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=617.2278, intensity=50, raw_flags=""),
            StrongLine(element="Ar", wavelength=624.3120, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=638.4717, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=641.6307, intensity=2, raw_flags=""),
            StrongLine(element="Ar", wavelength=648.3082, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=663.8221, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=663.9740, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=664.3698, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=666.6359, intensity=8, raw_flags=""),
            StrongLine(element="Ar", wavelength=667.7282, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=668.4293, intensity=12, raw_flags=""),
            StrongLine(element="Ar", wavelength=675.2834, intensity=4, raw_flags=""),
            StrongLine(element="Ar", wavelength=686.1269, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=687.1289, intensity=4, raw_flags=""),
            StrongLine(element="Ar", wavelength=693.7664, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=696.5431, intensity=300, raw_flags="P"),
            StrongLine(element="Ar", wavelength=703.0251, intensity=4, raw_flags=""),
            StrongLine(element="Ar", wavelength=706.7218, intensity=300, raw_flags="P"),
            StrongLine(element="Ar", wavelength=706.8736, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=710.7478, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=712.5820, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=714.7042, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=720.6980, intensity=2, raw_flags=""),
            StrongLine(element="Ar", wavelength=727.2936, intensity=60, raw_flags=""),
            StrongLine(element="Ar", wavelength=731.1716, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=731.6005, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=735.3293, intensity=2, raw_flags=""),
            StrongLine(element="Ar", wavelength=737.2118, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=738.0426, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=738.3980, intensity=300, raw_flags=""),
            StrongLine(element="Ar", wavelength=739.2980, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=743.5368, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=750.3869, intensity=600, raw_flags="P"),
            StrongLine(element="Ar", wavelength=751.4652, intensity=400, raw_flags=""),
            StrongLine(element="Ar", wavelength=763.5106, intensity=700, raw_flags="P"),
            StrongLine(element="Ar", wavelength=772.3761, intensity=400, raw_flags=""),
            StrongLine(element="Ar", wavelength=772.4207, intensity=300, raw_flags=""),
            StrongLine(element="Ar", wavelength=794.8176, intensity=600, raw_flags="P"),
            StrongLine(element="Ar", wavelength=800.6157, intensity=600, raw_flags="P"),
            StrongLine(element="Ar", wavelength=801.4786, intensity=700, raw_flags="P"),
            StrongLine(element="Ar", wavelength=810.3693, intensity=600, raw_flags="P"),
            StrongLine(element="Ar", wavelength=811.5311, intensity=1000, raw_flags="P"),
            StrongLine(element="Ar", wavelength=826.4522, intensity=300, raw_flags=""),
            StrongLine(element="Ar", wavelength=839.227, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=840.8210, intensity=400, raw_flags=""),
            StrongLine(element="Ar", wavelength=842.4648, intensity=600, raw_flags=""),
            StrongLine(element="Ar", wavelength=852.1442, intensity=400, raw_flags=""),
            StrongLine(element="Ar", wavelength=866.7944, intensity=130, raw_flags=""),
            StrongLine(element="Ar", wavelength=877.1860, intensity=7, raw_flags=""),
            StrongLine(element="Ar", wavelength=884.991, intensity=5, raw_flags=""),
            StrongLine(element="Ar", wavelength=907.5394, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=912.2967, intensity=1000, raw_flags="P"),
            StrongLine(element="Ar", wavelength=919.4638, intensity=15, raw_flags=""),
            StrongLine(element="Ar", wavelength=922.4499, intensity=400, raw_flags=""),
            StrongLine(element="Ar", wavelength=929.1531, intensity=11, raw_flags=""),
            StrongLine(element="Ar", wavelength=935.4220, intensity=50, raw_flags=""),
            StrongLine(element="Ar", wavelength=965.7786, intensity=700, raw_flags="P"),
            StrongLine(element="Ar", wavelength=978.4503, intensity=130, raw_flags=""),
            StrongLine(element="Ar", wavelength=1005.206, intensity=5, raw_flags=""),
            StrongLine(element="Ar", wavelength=1033.272, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1046.7177, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=1047.0054, intensity=50, raw_flags=""),
            StrongLine(element="Ar", wavelength=1050.650, intensity=5, raw_flags=""),
            StrongLine(element="Ar", wavelength=1067.3565, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1068.3034, intensity=2, raw_flags=""),
            StrongLine(element="Ar", wavelength=1073.387, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1075.916, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1081.2896, intensity=2, raw_flags=""),
            StrongLine(element="Ar", wavelength=1110.646, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1148.8109, intensity=11, raw_flags=""),
            StrongLine(element="Ar", wavelength=1166.8710, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1211.2326, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1213.9738, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1234.3393, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1240.2827, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1243.9321, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1245.612, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=1248.7663, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1270.2281, intensity=4, raw_flags=""),
            StrongLine(element="Ar", wavelength=1273.3418, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1280.2739, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1293.3195, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1295.6659, intensity=14, raw_flags=""),
            StrongLine(element="Ar", wavelength=1300.8264, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1321.399, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1322.8107, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1323.090, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=1327.264, intensity=14, raw_flags=""),
            StrongLine(element="Ar", wavelength=1331.3210, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=1336.7111, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=1349.941, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1350.4191, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=1359.9333, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1362.2659, intensity=11, raw_flags=""),
            StrongLine(element="Ar", wavelength=1367.8550, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1371.8577, intensity=30, raw_flags=""),
            StrongLine(element="Ar", wavelength=1409.3640, intensity=6, raw_flags=""),
            StrongLine(element="Ar", wavelength=1504.650, intensity=3, raw_flags=""),
            StrongLine(element="Ar", wavelength=1517.269, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1598.949, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1651.986, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=1694.058, intensity=14, raw_flags=""),
            StrongLine(element="Ar", wavelength=2061.623, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=2098.611, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=2313.320, intensity=1, raw_flags=""),
            StrongLine(element="Ar", wavelength=2396.652, intensity=1, raw_flags=""),
        ],
        persistent_lines=[
            StrongLine(element="Ar", wavelength=66.18690, intensity=150, raw_flags="P"),
            StrongLine(element="Ar", wavelength=67.09455, intensity=300, raw_flags="P"),
            StrongLine(element="Ar", wavelength=67.18513, intensity=1000, raw_flags="P"),
            StrongLine(element="Ar", wavelength=72.33606, intensity=1000, raw_flags="P"),
            StrongLine(element="Ar", wavelength=86.679997, intensity=180, raw_flags="P"),
            StrongLine(element="Ar", wavelength=86.975411, intensity=150, raw_flags="P"),
            StrongLine(element="Ar", wavelength=87.605767, intensity=180, raw_flags="P"),
            StrongLine(element="Ar", wavelength=87.994656, intensity=180, raw_flags="P"),
            StrongLine(element="Ar", wavelength=89.431013, intensity=150, raw_flags="P"),
            StrongLine(element="Ar", wavelength=91.97810, intensity=300, raw_flags="P"),
            StrongLine(element="Ar", wavelength=93.20537, intensity=300, raw_flags="P"),
            StrongLine(element="Ar", wavelength=104.821987, intensity=1000, raw_flags="P"),
            StrongLine(element="Ar", wavelength=106.665980, intensity=500, raw_flags="P"),
            StrongLine(element="Ar", wavelength=427.7528, intensity=200, raw_flags="P"),
            StrongLine(element="Ar", wavelength=434.8064, intensity=250, raw_flags="P"),
            StrongLine(element="Ar", wavelength=460.9567, intensity=200, raw_flags="P"),
            StrongLine(element="Ar", wavelength=472.6868, intensity=200, raw_flags="P"),
            StrongLine(element="Ar", wavelength=476.4865, intensity=250, raw_flags="P"),
            StrongLine(element="Ar", wavelength=480.6020, intensity=200, raw_flags="P"),
            StrongLine(element="Ar", wavelength=487.9864, intensity=250, raw_flags="P"),
            StrongLine(element="Ar", wavelength=696.5431, intensity=300, raw_flags="P"),
            StrongLine(element="Ar", wavelength=706.7218, intensity=300, raw_flags="P"),
            StrongLine(element="Ar", wavelength=750.3869, intensity=600, raw_flags="P"),
            StrongLine(element="Ar", wavelength=763.5106, intensity=700, raw_flags="P"),
            StrongLine(element="Ar", wavelength=794.8176, intensity=600, raw_flags="P"),
            StrongLine(element="Ar", wavelength=800.6157, intensity=600, raw_flags="P"),
            StrongLine(element="Ar", wavelength=801.4786, intensity=700, raw_flags="P"),
            StrongLine(element="Ar", wavelength=810.3693, intensity=600, raw_flags="P"),
            StrongLine(element="Ar", wavelength=811.5311, intensity=1000, raw_flags="P"),
            StrongLine(element="Ar", wavelength=912.2967, intensity=1000, raw_flags="P"),
            StrongLine(element="Ar", wavelength=965.7786, intensity=700, raw_flags="P"),
        ]
    ),
    "C": StrongLines(
        element="C",
        lines=[
            StrongLine(element="C", wavelength=68.70526, intensity=600, raw_flags="P"),
            StrongLine(element="C", wavelength=68.7346, intensity=1000, raw_flags="P"),
            StrongLine(element="C", wavelength=90.36235, intensity=150, raw_flags=""),
            StrongLine(element="C", wavelength=90.39616, intensity=300, raw_flags=""),
            StrongLine(element="C", wavelength=90.41416, intensity=750, raw_flags="P"),
            StrongLine(element="C", wavelength=90.44801, intensity=150, raw_flags=""),
            StrongLine(element="C", wavelength=103.63367, intensity=250, raw_flags="P"),
            StrongLine(element="C", wavelength=103.70182, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=119.30088, intensity=300, raw_flags="*"),
            StrongLine(element="C", wavelength=119.30308, intensity=300, raw_flags="*"),
            StrongLine(element="C", wavelength=119.32402, intensity=400, raw_flags=""),
            StrongLine(element="C", wavelength=119.32644, intensity=100, raw_flags=""),
            StrongLine(element="C", wavelength=126.15520, intensity=200, raw_flags=""),
            StrongLine(element="C", wavelength=127.72452, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=127.72824, intensity=800, raw_flags="P"),
            StrongLine(element="C", wavelength=127.75131, intensity=200, raw_flags="P"),
            StrongLine(element="C", wavelength=127.75497, intensity=1000, raw_flags="P"),
            StrongLine(element="C", wavelength=128.03330, intensity=800, raw_flags="P"),
            StrongLine(element="C", wavelength=132.95775, intensity=200, raw_flags=""),
            StrongLine(element="C", wavelength=132.96005, intensity=100, raw_flags=""),
            StrongLine(element="C", wavelength=133.45323, intensity=80, raw_flags="P"),
            StrongLine(element="C", wavelength=133.57077, intensity=150, raw_flags="P"),
            StrongLine(element="C", wavelength=156.0309, intensity=200, raw_flags=""),
            StrongLine(element="C", wavelength=156.0682, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=156.0709, intensity=200, raw_flags="P"),
            StrongLine(element="C", wavelength=156.1438, intensity=600, raw_flags="P"),
            StrongLine(element="C", wavelength=165.6267, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=165.6928, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=165.7008, intensity=1000, raw_flags="P"),
            StrongLine(element="C", wavelength=165.7379, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=165.7907, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=165.8121, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=175.1827, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=193.0906, intensity=1000, raw_flags="P"),
            StrongLine(element="C", wavelength=247.8561, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=283.6710, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=283.7603, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=299.2618, intensity=400, raw_flags="h"),
            StrongLine(element="C", wavelength=391.8978, intensity=300, raw_flags="P"),
            StrongLine(element="C", wavelength=392.0693, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=426.7003, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=426.7258, intensity=500, raw_flags="Pc"),
            StrongLine(element="C", wavelength=514.516, intensity=300, raw_flags=""),
            StrongLine(element="C", wavelength=515.109, intensity=200, raw_flags=""),
            StrongLine(element="C", wavelength=538.034, intensity=150, raw_flags=""),
            StrongLine(element="C", wavelength=588.977, intensity=300, raw_flags=""),
            StrongLine(element="C", wavelength=601.322, intensity=150, raw_flags=""),
            StrongLine(element="C", wavelength=657.805, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=658.288, intensity=300, raw_flags=""),
            StrongLine(element="C", wavelength=723.132, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=723.642, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=833.515, intensity=250, raw_flags="P"),
            StrongLine(element="C", wavelength=909.483, intensity=220, raw_flags=""),
            StrongLine(element="C", wavelength=911.180, intensity=150, raw_flags=""),
            StrongLine(element="C", wavelength=940.573, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=965.844, intensity=150, raw_flags=""),
            StrongLine(element="C", wavelength=1069.125, intensity=150, raw_flags=""),
            StrongLine(element="C", wavelength=1174.822, intensity=90, raw_flags=""),
            StrongLine(element="C", wavelength=1175.332, intensity=140, raw_flags="P"),
            StrongLine(element="C", wavelength=1175.476, intensity=110, raw_flags="P"),
            StrongLine(element="C", wavelength=1689.038, intensity=50, raw_flags=""),
        ],
        persistent_lines=[
            StrongLine(element="C", wavelength=68.70526, intensity=600, raw_flags="P"),
            StrongLine(element="C", wavelength=68.7346, intensity=1000, raw_flags="P"),
            StrongLine(element="C", wavelength=90.41416, intensity=750, raw_flags="P"),
            StrongLine(element="C", wavelength=103.63367, intensity=250, raw_flags="P"),
            StrongLine(element="C", wavelength=103.70182, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=127.72452, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=127.72824, intensity=800, raw_flags="P"),
            StrongLine(element="C", wavelength=127.75131, intensity=200, raw_flags="P"),
            StrongLine(element="C", wavelength=127.75497, intensity=1000, raw_flags="P"),
            StrongLine(element="C", wavelength=128.03330, intensity=800, raw_flags="P"),
            StrongLine(element="C", wavelength=133.45323, intensity=80, raw_flags="P"),
            StrongLine(element="C", wavelength=133.57077, intensity=150, raw_flags="P"),
            StrongLine(element="C", wavelength=156.0682, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=156.0709, intensity=200, raw_flags="P"),
            StrongLine(element="C", wavelength=156.1438, intensity=600, raw_flags="P"),
            StrongLine(element="C", wavelength=165.6267, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=165.6928, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=165.7008, intensity=1000, raw_flags="P"),
            StrongLine(element="C", wavelength=165.7379, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=165.7907, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=165.8121, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=175.1827, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=193.0906, intensity=1000, raw_flags="P"),
            StrongLine(element="C", wavelength=247.8561, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=283.6710, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=283.7603, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=391.8978, intensity=300, raw_flags="P"),
            StrongLine(element="C", wavelength=392.0693, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=426.7003, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=426.7258, intensity=500, raw_flags="Pc"),
            StrongLine(element="C", wavelength=657.805, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=723.132, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=723.642, intensity=500, raw_flags="P"),
            StrongLine(element="C", wavelength=833.515, intensity=250, raw_flags="P"),
            StrongLine(element="C", wavelength=940.573, intensity=400, raw_flags="P"),
            StrongLine(element="C", wavelength=1175.332, intensity=140, raw_flags="P"),
            StrongLine(element="C", wavelength=1175.476, intensity=110, raw_flags="P"),
        ]
    ),
    "H": StrongLines(
        element="H",
        lines=[
            StrongLine(element="H", wavelength=92.62256, intensity=15, raw_flags=""),
            StrongLine(element="H", wavelength=93.07482, intensity=20, raw_flags=""),
            StrongLine(element="H", wavelength=93.78034, intensity=30, raw_flags=""),
            StrongLine(element="H", wavelength=94.97430, intensity=50, raw_flags="P"),
            StrongLine(element="H", wavelength=97.25367, intensity=100, raw_flags="P"),
            StrongLine(element="H", wavelength=102.57222, intensity=300, raw_flags="P"),
            StrongLine(element="H", wavelength=121.566824, intensity=1000, raw_flags="P"),
            StrongLine(element="H", wavelength=121.567364, intensity=500, raw_flags="P"),
            StrongLine(element="H", wavelength=383.5384, intensity=5, raw_flags=""),
            StrongLine(element="H", wavelength=388.9049, intensity=6, raw_flags=""),
            StrongLine(element="H", wavelength=397.0072, intensity=8, raw_flags=""),
            StrongLine(element="H", wavelength=410.174, intensity=15, raw_flags=""),
            StrongLine(element="H", wavelength=434.0462, intensity=30, raw_flags="P"),
            StrongLine(element="H", wavelength=486.12786, intensity=30, raw_flags="P"),
            StrongLine(element="H", wavelength=486.12870, intensity=10, raw_flags="P"),
            StrongLine(element="H", wavelength=486.13615, intensity=60, raw_flags="P"),
            StrongLine(element="H", wavelength=656.27110, intensity=90, raw_flags="P"),
            StrongLine(element="H", wavelength=656.27248, intensity=30, raw_flags="P"),
            StrongLine(element="H", wavelength=656.28518, intensity=180, raw_flags="P"),
            StrongLine(element="H", wavelength=954.597, intensity=5, raw_flags=""),
            StrongLine(element="H", wavelength=1004.94, intensity=7, raw_flags=""),
            StrongLine(element="H", wavelength=1093.81, intensity=12, raw_flags=""),
            StrongLine(element="H", wavelength=1281.807, intensity=20, raw_flags="P"),
            StrongLine(element="H", wavelength=1875.101, intensity=40, raw_flags="Pc"),
            StrongLine(element="H", wavelength=2165.53, intensity=5, raw_flags=""),
            StrongLine(element="H", wavelength=2625.15, intensity=8, raw_flags=""),
            StrongLine(element="H", wavelength=4051.16, intensity=15, raw_flags=""),
            StrongLine(element="H", wavelength=4652.51, intensity=4, raw_flags=""),
            StrongLine(element="H", wavelength=74578, intensity=6, raw_flags=""),
            StrongLine(element="H", wavelength=123685, intensity=3, raw_flags=""),
        ],
        persistent_lines=[
            StrongLine(element="H", wavelength=94.97430, intensity=50, raw_flags="P"),
            StrongLine(element="H", wavelength=97.25367, intensity=100, raw_flags="P"),
            StrongLine(element="H", wavelength=102.57222, intensity=300, raw_flags="P"),
            StrongLine(element="H", wavelength=121.566824, intensity=1000, raw_flags="P"),
            StrongLine(element="H", wavelength=121.567364, intensity=500, raw_flags="P"),
            StrongLine(element="H", wavelength=434.0462, intensity=30, raw_flags="P"),
            StrongLine(element="H", wavelength=486.12786, intensity=30, raw_flags="P"),
            StrongLine(element="H", wavelength=486.12870, intensity=10, raw_flags="P"),
            StrongLine(element="H", wavelength=486.13615, intensity=60, raw_flags="P"),
            StrongLine(element="H", wavelength=656.27110, intensity=90, raw_flags="P"),
            StrongLine(element="H", wavelength=656.27248, intensity=30, raw_flags="P"),
            StrongLine(element="H", wavelength=656.28518, intensity=180, raw_flags="P"),
            StrongLine(element="H", wavelength=1281.807, intensity=20, raw_flags="P"),
            StrongLine(element="H", wavelength=1875.101, intensity=40, raw_flags="Pc"),
        ]
    ),
    "He": StrongLines(
        element="He",
        lines=[
            StrongLine(element="He", wavelength=23.14541, intensity=15, raw_flags="c"),
            StrongLine(element="He", wavelength=23.25842, intensity=20, raw_flags="c"),
            StrongLine(element="He", wavelength=23.43472, intensity=30, raw_flags="c"),
            StrongLine(element="He", wavelength=23.73307, intensity=50, raw_flags="c"),
            StrongLine(element="He", wavelength=24.30266, intensity=100, raw_flags="c"),
            StrongLine(element="He", wavelength=25.63166, intensity=300, raw_flags="P"),
            StrongLine(element="He", wavelength=25.63177, intensity=150, raw_flags="P"),
            StrongLine(element="He", wavelength=30.37804, intensity=1000, raw_flags="P"),
            StrongLine(element="He", wavelength=30.37858, intensity=500, raw_flags="P"),
            StrongLine(element="He", wavelength=32.0293, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=50.550035, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=50.568433, intensity=3, raw_flags=""),
            StrongLine(element="He", wavelength=50.591252, intensity=4, raw_flags=""),
            StrongLine(element="He", wavelength=50.620034, intensity=5, raw_flags=""),
            StrongLine(element="He", wavelength=50.657057, intensity=7, raw_flags=""),
            StrongLine(element="He", wavelength=50.705802, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=50.771809, intensity=15, raw_flags=""),
            StrongLine(element="He", wavelength=50.864338, intensity=20, raw_flags=""),
            StrongLine(element="He", wavelength=50.999829, intensity=25, raw_flags=""),
            StrongLine(element="He", wavelength=51.209856, intensity=35, raw_flags=""),
            StrongLine(element="He", wavelength=51.561684, intensity=50, raw_flags=""),
            StrongLine(element="He", wavelength=52.221309, intensity=100, raw_flags="P"),
            StrongLine(element="He", wavelength=53.702992, intensity=400, raw_flags="P"),
            StrongLine(element="He", wavelength=58.433436, intensity=1000, raw_flags="P"),
            StrongLine(element="He", wavelength=59.141207, intensity=50, raw_flags=""),
            StrongLine(element="He", wavelength=95.870, intensity=5, raw_flags="c"),
            StrongLine(element="He", wavelength=97.211, intensity=6, raw_flags="c"),
            StrongLine(element="He", wavelength=99.236, intensity=8, raw_flags="c"),
            StrongLine(element="He", wavelength=102.527, intensity=15, raw_flags="c"),
            StrongLine(element="He", wavelength=108.494, intensity=30, raw_flags="c"),
            StrongLine(element="He", wavelength=121.509, intensity=35, raw_flags=""),
            StrongLine(element="He", wavelength=121.517, intensity=50, raw_flags="c"),
            StrongLine(element="He", wavelength=164.03321, intensity=120, raw_flags="P"),
            StrongLine(element="He", wavelength=164.03447, intensity=50, raw_flags="P"),
            StrongLine(element="He", wavelength=164.03750, intensity=7, raw_flags="P"),
            StrongLine(element="He", wavelength=164.03914, intensity=25, raw_flags="P"),
            StrongLine(element="He", wavelength=164.04742, intensity=180, raw_flags="P"),
            StrongLine(element="He", wavelength=164.04897, intensity=25, raw_flags="P"),
            StrongLine(element="He", wavelength=164.05326, intensity=15, raw_flags="P"),
            StrongLine(element="He", wavelength=238.540, intensity=7, raw_flags="c"),
            StrongLine(element="He", wavelength=251.120, intensity=9, raw_flags="c"),
            StrongLine(element="He", wavelength=272.319, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=273.330, intensity=12, raw_flags="c"),
            StrongLine(element="He", wavelength=276.380, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=281.82, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=282.908, intensity=4, raw_flags=""),
            StrongLine(element="He", wavelength=294.511, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=301.37, intensity=40, raw_flags=""),
            StrongLine(element="He", wavelength=318.774, intensity=20, raw_flags=""),
            StrongLine(element="He", wavelength=320.310, intensity=15, raw_flags="c"),
            StrongLine(element="He", wavelength=335.455, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=344.759, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=358.727, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=361.364, intensity=3, raw_flags=""),
            StrongLine(element="He", wavelength=363.423, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=370.500, intensity=3, raw_flags=""),
            StrongLine(element="He", wavelength=373.286, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=381.9607, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=381.976, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=386.748, intensity=3, raw_flags=""),
            StrongLine(element="He", wavelength=388.86046, intensity=60, raw_flags="P"),
            StrongLine(element="He", wavelength=388.86456, intensity=200, raw_flags="P"),
            StrongLine(element="He", wavelength=388.86489, intensity=300, raw_flags="P"),
            StrongLine(element="He", wavelength=396.4729, intensity=20, raw_flags=""),
            StrongLine(element="He", wavelength=400.927, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=402.6191, intensity=50, raw_flags=""),
            StrongLine(element="He", wavelength=402.636, intensity=5, raw_flags=""),
            StrongLine(element="He", wavelength=412.082, intensity=12, raw_flags=""),
            StrongLine(element="He", wavelength=412.099, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=414.376, intensity=3, raw_flags=""),
            StrongLine(element="He", wavelength=438.7929, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=443.755, intensity=3, raw_flags=""),
            StrongLine(element="He", wavelength=447.1479, intensity=200, raw_flags=""),
            StrongLine(element="He", wavelength=447.168, intensity=25, raw_flags=""),
            StrongLine(element="He", wavelength=468.53769, intensity=4, raw_flags="Pc"),
            StrongLine(element="He", wavelength=468.54072, intensity=3, raw_flags="Pc"),
            StrongLine(element="He", wavelength=468.58041, intensity=12, raw_flags="Pc"),
            StrongLine(element="He", wavelength=471.3146, intensity=30, raw_flags=""),
            StrongLine(element="He", wavelength=471.338, intensity=4, raw_flags=""),
            StrongLine(element="He", wavelength=492.1931, intensity=20, raw_flags=""),
            StrongLine(element="He", wavelength=501.5678, intensity=100, raw_flags=""),
            StrongLine(element="He", wavelength=504.774, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=541.152, intensity=5, raw_flags="c"),
            StrongLine(element="He", wavelength=587.56148, intensity=500, raw_flags="P"),
            StrongLine(element="He", wavelength=587.56404, intensity=250, raw_flags="P"),
            StrongLine(element="He", wavelength=587.59663, intensity=120, raw_flags="P"),
            StrongLine(element="He", wavelength=656.010, intensity=8, raw_flags="c"),
            StrongLine(element="He", wavelength=667.81517, intensity=200, raw_flags="P"),
            StrongLine(element="He", wavelength=706.51771, intensity=100, raw_flags="P"),
            StrongLine(element="He", wavelength=706.52153, intensity=60, raw_flags="P"),
            StrongLine(element="He", wavelength=706.57086, intensity=20, raw_flags="P"),
            StrongLine(element="He", wavelength=728.135, intensity=50, raw_flags=""),
            StrongLine(element="He", wavelength=781.615, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=836.169, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=906.327, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=921.034, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=946.361, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=951.660, intensity=4, raw_flags=""),
            StrongLine(element="He", wavelength=952.617, intensity=3, raw_flags=""),
            StrongLine(element="He", wavelength=952.927, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=960.342, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=970.260, intensity=3, raw_flags=""),
            StrongLine(element="He", wavelength=1002.773, intensity=6, raw_flags=""),
            StrongLine(element="He", wavelength=1003.116, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=1012.36, intensity=15, raw_flags="c"),
            StrongLine(element="He", wavelength=1013.850, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=1031.123, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=1031.154, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=1066.765, intensity=3, raw_flags=""),
            StrongLine(element="He", wavelength=1082.90911, intensity=150, raw_flags="P"),
            StrongLine(element="He", wavelength=1083.02501, intensity=500, raw_flags="P"),
            StrongLine(element="He", wavelength=1083.03398, intensity=1000, raw_flags="P"),
            StrongLine(element="He", wavelength=1091.305, intensity=9, raw_flags=""),
            StrongLine(element="He", wavelength=1091.710, intensity=3, raw_flags=""),
            StrongLine(element="He", wavelength=1162.64, intensity=4, raw_flags="c"),
            StrongLine(element="He", wavelength=1196.912, intensity=30, raw_flags=""),
            StrongLine(element="He", wavelength=1252.752, intensity=20, raw_flags=""),
            StrongLine(element="He", wavelength=1278.499, intensity=50, raw_flags=""),
            StrongLine(element="He", wavelength=1279.057, intensity=20, raw_flags=""),
            StrongLine(element="He", wavelength=1284.596, intensity=7, raw_flags=""),
            StrongLine(element="He", wavelength=1296.845, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=1298.489, intensity=2, raw_flags=""),
            StrongLine(element="He", wavelength=1508.364, intensity=12, raw_flags=""),
            StrongLine(element="He", wavelength=1700.247, intensity=200, raw_flags=""),
            StrongLine(element="He", wavelength=1855.555, intensity=1, raw_flags=""),
            StrongLine(element="He", wavelength=1863.68, intensity=6, raw_flags="c"),
            StrongLine(element="He", wavelength=1868.534, intensity=500, raw_flags=""),
            StrongLine(element="He", wavelength=1869.723, intensity=200, raw_flags=""),
            StrongLine(element="He", wavelength=1908.938, intensity=100, raw_flags=""),
            StrongLine(element="He", wavelength=1954.308, intensity=20, raw_flags=""),
            StrongLine(element="He", wavelength=2058.1287, intensity=500, raw_flags="P"),
            StrongLine(element="He", wavelength=2112.007, intensity=80, raw_flags=""),
            StrongLine(element="He", wavelength=2112.143, intensity=10, raw_flags=""),
            StrongLine(element="He", wavelength=2113.203, intensity=20, raw_flags=""),
            StrongLine(element="He", wavelength=3090.85, intensity=3, raw_flags="c"),
            StrongLine(element="He", wavelength=4047.890, intensity=4, raw_flags=""),
        ],
        persistent_lines=[
            StrongLine(element="He", wavelength=25.63166, intensity=300, raw_flags="P"),
            StrongLine(element="He", wavelength=25.63177, intensity=150, raw_flags="P"),
            StrongLine(element="He", wavelength=30.37804, intensity=1000, raw_flags="P"),
            StrongLine(element="He", wavelength=30.37858, intensity=500, raw_flags="P"),
            StrongLine(element="He", wavelength=52.221309, intensity=100, raw_flags="P"),
            StrongLine(element="He", wavelength=53.702992, intensity=400, raw_flags="P"),
            StrongLine(element="He", wavelength=58.433436, intensity=1000, raw_flags="P"),
            StrongLine(element="He", wavelength=164.03321, intensity=120, raw_flags="P"),
            StrongLine(element="He", wavelength=164.03447, intensity=50, raw_flags="P"),
            StrongLine(element="He", wavelength=164.03750, intensity=7, raw_flags="P"),
            StrongLine(element="He", wavelength=164.03914, intensity=25, raw_flags="P"),
            StrongLine(element="He", wavelength=164.04742, intensity=180, raw_flags="P"),
            StrongLine(element="He", wavelength=164.04897, intensity=25, raw_flags="P"),
            StrongLine(element="He", wavelength=164.05326, intensity=15, raw_flags="P"),
            StrongLine(element="He", wavelength=388.86046, intensity=60, raw_flags="P"),
            StrongLine(element="He", wavelength=388.86456, intensity=200, raw_flags="P"),
            StrongLine(element="He", wavelength=388.86489, intensity=300, raw_flags="P"),
            StrongLine(element="He", wavelength=468.53769, intensity=4, raw_flags="Pc"),
            StrongLine(element="He", wavelength=468.54072, intensity=3, raw_flags="Pc"),
            StrongLine(element="He", wavelength=468.58041, intensity=12, raw_flags="Pc"),
            StrongLine(element="He", wavelength=587.56148, intensity=500, raw_flags="P"),
            StrongLine(element="He", wavelength=587.56404, intensity=250, raw_flags="P"),
            StrongLine(element="He", wavelength=587.59663, intensity=120, raw_flags="P"),
            StrongLine(element="He", wavelength=667.81517, intensity=200, raw_flags="P"),
            StrongLine(element="He", wavelength=706.51771, intensity=100, raw_flags="P"),
            StrongLine(element="He", wavelength=706.52153, intensity=60, raw_flags="P"),
            StrongLine(element="He", wavelength=706.57086, intensity=20, raw_flags="P"),
            StrongLine(element="He", wavelength=1082.90911, intensity=150, raw_flags="P"),
            StrongLine(element="He", wavelength=1083.02501, intensity=500, raw_flags="P"),
            StrongLine(element="He", wavelength=1083.03398, intensity=1000, raw_flags="P"),
            StrongLine(element="He", wavelength=2058.1287, intensity=500, raw_flags="P"),
        ]
    ),
    "Hg": StrongLines(
        element="Hg",
        lines=[
            StrongLine(element="Hg", wavelength=89.30847, intensity=20, raw_flags=""),
            StrongLine(element="Hg", wavelength=91.5819, intensity=12, raw_flags=""),
            StrongLine(element="Hg", wavelength=94.2630, intensity=20, raw_flags=""),
            StrongLine(element="Hg", wavelength=96.2711, intensity=25, raw_flags=""),
            StrongLine(element="Hg", wavelength=96.9142, intensity=25, raw_flags=""),
            StrongLine(element="Hg", wavelength=103.96315, intensity=20, raw_flags=""),
            StrongLine(element="Hg", wavelength=106.27802, intensity=20, raw_flags=""),
            StrongLine(element="Hg", wavelength=164.99373, intensity=1000, raw_flags="P"),
            StrongLine(element="Hg", wavelength=184.9499, intensity=1000, raw_flags="P"),
            StrongLine(element="Hg", wavelength=194.2273, intensity=1000, raw_flags="P"),
            StrongLine(element="Hg", wavelength=197.3794, intensity=15, raw_flags=""),
            StrongLine(element="Hg", wavelength=198.7841, intensity=10, raw_flags=""),
            StrongLine(element="Hg", wavelength=202.6860, intensity=20, raw_flags=""),
            StrongLine(element="Hg", wavelength=205.2828, intensity=400, raw_flags="P"),
            StrongLine(element="Hg", wavelength=222.4711, intensity=20, raw_flags=""),
            StrongLine(element="Hg", wavelength=225.2786, intensity=10, raw_flags=""),
            StrongLine(element="Hg", wavelength=226.0294, intensity=60, raw_flags=""),
            StrongLine(element="Hg", wavelength=226.2223, intensity=400, raw_flags="P"),
            StrongLine(element="Hg", wavelength=226.3634, intensity=10, raw_flags=""),
            StrongLine(element="Hg", wavelength=253.6517, intensity=1000, raw_flags="Pc"),
            StrongLine(element="Hg", wavelength=265.2039, intensity=25, raw_flags=""),
            StrongLine(element="Hg", wavelength=265.3679, intensity=40, raw_flags=""),
            StrongLine(element="Hg", wavelength=284.7675, intensity=400, raw_flags="P"),
            StrongLine(element="Hg", wavelength=291.6250, intensity=30, raw_flags=""),
            StrongLine(element="Hg", wavelength=294.7074, intensity=25, raw_flags=""),
            StrongLine(element="Hg", wavelength=296.7280, intensity=250, raw_flags="P"),
            StrongLine(element="Hg", wavelength=302.1498, intensity=70, raw_flags=""),
            StrongLine(element="Hg", wavelength=312.5668, intensity=90, raw_flags=""),
            StrongLine(element="Hg", wavelength=313.1548, intensity=80, raw_flags=""),
            StrongLine(element="Hg", wavelength=313.1839, intensity=80, raw_flags=""),
            StrongLine(element="Hg", wavelength=320.8169, intensity=12, raw_flags=""),
            StrongLine(element="Hg", wavelength=353.2594, intensity=10, raw_flags=""),
            StrongLine(element="Hg", wavelength=360.5762, intensity=10, raw_flags=""),
            StrongLine(element="Hg", wavelength=365.0153, intensity=600, raw_flags="P"),
            StrongLine(element="Hg", wavelength=365.4836, intensity=70, raw_flags=""),
            StrongLine(element="Hg", wavelength=366.3279, intensity=50, raw_flags=""),
            StrongLine(element="Hg", wavelength=398.3931, intensity=1000, raw_flags="Pc"),
            StrongLine(element="Hg", wavelength=404.6563, intensity=400, raw_flags="P"),
            StrongLine(element="Hg", wavelength=433.9223, intensity=60, raw_flags=""),
            StrongLine(element="Hg", wavelength=434.7494, intensity=100, raw_flags=""),
            StrongLine(element="Hg", wavelength=435.8328, intensity=1000, raw_flags="P"),
            StrongLine(element="Hg", wavelength=512.8442, intensity=12, raw_flags="c"),
            StrongLine(element="Hg", wavelength=520.4768, intensity=15, raw_flags=""),
            StrongLine(element="Hg", wavelength=542.5253, intensity=80, raw_flags="P"),
            StrongLine(element="Hg", wavelength=546.0735, intensity=500, raw_flags="P"),
            StrongLine(element="Hg", wavelength=567.7105, intensity=200, raw_flags="P"),
            StrongLine(element="Hg", wavelength=576.9598, intensity=50, raw_flags=""),
            StrongLine(element="Hg", wavelength=579.0663, intensity=60, raw_flags=""),
            StrongLine(element="Hg", wavelength=587.1279, intensity=12, raw_flags=""),
            StrongLine(element="Hg", wavelength=588.8939, intensity=20, raw_flags="c"),
            StrongLine(element="Hg", wavelength=614.6435, intensity=15, raw_flags=""),
            StrongLine(element="Hg", wavelength=614.9475, intensity=250, raw_flags="Pc"),
            StrongLine(element="Hg", wavelength=708.190, intensity=25, raw_flags=""),
            StrongLine(element="Hg", wavelength=734.6508, intensity=6, raw_flags=""),
            StrongLine(element="Hg", wavelength=794.4555, intensity=250, raw_flags="P"),
            StrongLine(element="Hg", wavelength=952.0198, intensity=6, raw_flags="h"),
            StrongLine(element="Hg", wavelength=1013.976, intensity=200, raw_flags="P"),
            StrongLine(element="Hg", wavelength=1357.021, intensity=50, raw_flags=""),
            StrongLine(element="Hg", wavelength=1367.351, intensity=40, raw_flags=""),
            StrongLine(element="Hg", wavelength=1529.582, intensity=50, raw_flags=""),
            StrongLine(element="Hg", wavelength=1707.279, intensity=50, raw_flags=""),
            StrongLine(element="Hg", wavelength=2325.307, intensity=25, raw_flags=""),
        ],
        persistent_lines=[
            StrongLine(element="Hg", wavelength=164.99373, intensity=1000, raw_flags="P"),
            StrongLine(element="Hg", wavelength=184.9499, intensity=1000, raw_flags="P"),
            StrongLine(element="Hg", wavelength=194.2273, intensity=1000, raw_flags="P"),
            StrongLine(element="Hg", wavelength=205.2828, intensity=400, raw_flags="P"),
            StrongLine(element="Hg", wavelength=226.2223, intensity=400, raw_flags="P"),
            StrongLine(element="Hg", wavelength=253.6517, intensity=1000, raw_flags="Pc"),
            StrongLine(element="Hg", wavelength=284.7675, intensity=400, raw_flags="P"),
            StrongLine(element="Hg", wavelength=296.7280, intensity=250, raw_flags="P"),
            StrongLine(element="Hg", wavelength=365.0153, intensity=600, raw_flags="P"),
            StrongLine(element="Hg", wavelength=398.3931, intensity=1000, raw_flags="Pc"),
            StrongLine(element="Hg", wavelength=404.6563, intensity=400, raw_flags="P"),
            StrongLine(element="Hg", wavelength=435.8328, intensity=1000, raw_flags="P"),
            StrongLine(element="Hg", wavelength=542.5253, intensity=80, raw_flags="P"),
            StrongLine(element="Hg", wavelength=546.0735, intensity=500, raw_flags="P"),
            StrongLine(element="Hg", wavelength=567.7105, intensity=200, raw_flags="P"),
            StrongLine(element="Hg", wavelength=614.9475, intensity=250, raw_flags="Pc"),
            StrongLine(element="Hg", wavelength=794.4555, intensity=250, raw_flags="P"),
            StrongLine(element="Hg", wavelength=1013.976, intensity=200, raw_flags="P"),
        ]
    ),
    "Kr": StrongLines(
        element="Kr",
        lines=[
            StrongLine(element="Kr", wavelength=72.940, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=76.118, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=76.398, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=76.620, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=77.103, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=77.369, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=78.210, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=78.372, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=81.815, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=83.038, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=84.406, intensity=30, raw_flags="P"),
            StrongLine(element="Kr", wavelength=86.482, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=86.887, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=88.414, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=88.630, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=89.101, intensity=130, raw_flags=""),
            StrongLine(element="Kr", wavelength=91.139, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=91.743, intensity=700, raw_flags="P"),
            StrongLine(element="Kr", wavelength=94.54414, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=94.65443, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=95.1056, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=95.34041, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=96.33745, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=96.497, intensity=700, raw_flags="P"),
            StrongLine(element="Kr", wavelength=100.10606, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=100.35504, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=103.00232, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=116.48671, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=123.58378, intensity=1000, raw_flags="P"),
            StrongLine(element="Kr", wavelength=246.477, intensity=30, raw_flags="h"),
            StrongLine(element="Kr", wavelength=249.248, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=271.240, intensity=25, raw_flags="h"),
            StrongLine(element="Kr", wavelength=283.300, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=360.788, intensity=30, raw_flags="h"),
            StrongLine(element="Kr", wavelength=363.1889, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=365.3928, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=366.5324, intensity=13, raw_flags=""),
            StrongLine(element="Kr", wavelength=366.901, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=367.958, intensity=15, raw_flags="d"),
            StrongLine(element="Kr", wavelength=368.6182, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=371.802, intensity=100, raw_flags="h"),
            StrongLine(element="Kr", wavelength=371.8595, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=372.1350, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=374.1638, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=374.480, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=375.4245, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=377.8046, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=378.3095, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=387.544, intensity=50, raw_flags="h"),
            StrongLine(element="Kr", wavelength=390.6177, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=392.0081, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=399.4840, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=399.7793, intensity=30, raw_flags="h"),
            StrongLine(element="Kr", wavelength=405.7037, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=406.5128, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=408.8337, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=409.8729, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=410.9248, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=414.5122, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=425.0580, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=427.39694, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=428.29674, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=429.2923, intensity=200, raw_flags=""),
            StrongLine(element="Kr", wavelength=430.049, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=431.781, intensity=150, raw_flags="h"),
            StrongLine(element="Kr", wavelength=431.85513, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=431.95794, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=432.298, intensity=50, raw_flags="h"),
            StrongLine(element="Kr", wavelength=435.13597, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=435.5477, intensity=1000, raw_flags="P"),
            StrongLine(element="Kr", wavelength=436.26416, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=436.969, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=437.61216, intensity=130, raw_flags=""),
            StrongLine(element="Kr", wavelength=438.654, intensity=100, raw_flags="h"),
            StrongLine(element="Kr", wavelength=439.99663, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=442.51901, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=443.1685, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=443.6812, intensity=200, raw_flags=""),
            StrongLine(element="Kr", wavelength=445.39175, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=446.36900, intensity=130, raw_flags=""),
            StrongLine(element="Kr", wavelength=447.5014, intensity=250, raw_flags=""),
            StrongLine(element="Kr", wavelength=448.988, intensity=130, raw_flags="h"),
            StrongLine(element="Kr", wavelength=450.23543, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=452.314, intensity=130, raw_flags="h"),
            StrongLine(element="Kr", wavelength=455.661, intensity=70, raw_flags="h"),
            StrongLine(element="Kr", wavelength=457.7209, intensity=250, raw_flags=""),
            StrongLine(element="Kr", wavelength=458.2978, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=459.280, intensity=50, raw_flags="h"),
            StrongLine(element="Kr", wavelength=461.5292, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=461.9166, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=463.3885, intensity=250, raw_flags=""),
            StrongLine(element="Kr", wavelength=465.8876, intensity=700, raw_flags="P"),
            StrongLine(element="Kr", wavelength=468.0406, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=469.1301, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=469.4360, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=473.9002, intensity=1000, raw_flags="P"),
            StrongLine(element="Kr", wavelength=476.2435, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=476.5744, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=481.176, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=482.518, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=483.2077, intensity=250, raw_flags="P"),
            StrongLine(element="Kr", wavelength=484.6612, intensity=250, raw_flags="P"),
            StrongLine(element="Kr", wavelength=485.720, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=494.559, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=502.240, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=508.652, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=512.573, intensity=130, raw_flags="h"),
            StrongLine(element="Kr", wavelength=520.832, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=530.866, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=533.341, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=546.817, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=556.22253, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=557.02894, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=558.03873, intensity=13, raw_flags=""),
            StrongLine(element="Kr", wavelength=564.95618, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=568.189, intensity=130, raw_flags=""),
            StrongLine(element="Kr", wavelength=569.035, intensity=70, raw_flags="h"),
            StrongLine(element="Kr", wavelength=583.28566, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=587.09160, intensity=500, raw_flags="P"),
            StrongLine(element="Kr", wavelength=599.222, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=599.38502, intensity=10, raw_flags=""),
            StrongLine(element="Kr", wavelength=605.61263, intensity=10, raw_flags=""),
            StrongLine(element="Kr", wavelength=642.018, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=642.10270, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=645.62889, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=657.007, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=669.92296, intensity=10, raw_flags=""),
            StrongLine(element="Kr", wavelength=690.46788, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=721.313, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=722.4104, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=728.7258, intensity=13, raw_flags=""),
            StrongLine(element="Kr", wavelength=728.978, intensity=130, raw_flags=""),
            StrongLine(element="Kr", wavelength=740.702, intensity=130, raw_flags=""),
            StrongLine(element="Kr", wavelength=742.5541, intensity=10, raw_flags=""),
            StrongLine(element="Kr", wavelength=743.578, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=748.6862, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=752.446, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=758.74136, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=760.15457, intensity=300, raw_flags=""),
            StrongLine(element="Kr", wavelength=764.116, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=768.52459, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=769.45401, intensity=200, raw_flags=""),
            StrongLine(element="Kr", wavelength=773.569, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=774.6827, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=785.48234, intensity=130, raw_flags=""),
            StrongLine(element="Kr", wavelength=791.34251, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=792.85988, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=793.322, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=797.362, intensity=40, raw_flags=""),
            StrongLine(element="Kr", wavelength=798.2401, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=805.95048, intensity=250, raw_flags=""),
            StrongLine(element="Kr", wavelength=810.43655, intensity=700, raw_flags="P"),
            StrongLine(element="Kr", wavelength=811.29012, intensity=1000, raw_flags="P"),
            StrongLine(element="Kr", wavelength=813.2967, intensity=10, raw_flags=""),
            StrongLine(element="Kr", wavelength=819.00566, intensity=500, raw_flags="P"),
            StrongLine(element="Kr", wavelength=820.272, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=821.8365, intensity=13, raw_flags=""),
            StrongLine(element="Kr", wavelength=826.32426, intensity=500, raw_flags="P"),
            StrongLine(element="Kr", wavelength=827.2353, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=828.10522, intensity=250, raw_flags=""),
            StrongLine(element="Kr", wavelength=829.81099, intensity=800, raw_flags="P"),
            StrongLine(element="Kr", wavelength=841.2430, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=850.88728, intensity=500, raw_flags="P"),
            StrongLine(element="Kr", wavelength=876.4110, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=877.67505, intensity=1000, raw_flags="P"),
            StrongLine(element="Kr", wavelength=892.86934, intensity=300, raw_flags=""),
            StrongLine(element="Kr", wavelength=923.848, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=929.382, intensity=150, raw_flags="hl"),
            StrongLine(element="Kr", wavelength=932.099, intensity=70, raw_flags="h"),
            StrongLine(element="Kr", wavelength=936.195, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=936.2082, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=940.282, intensity=70, raw_flags="h"),
            StrongLine(element="Kr", wavelength=947.093, intensity=70, raw_flags="h"),
            StrongLine(element="Kr", wavelength=957.752, intensity=150, raw_flags="P"),
            StrongLine(element="Kr", wavelength=960.580, intensity=150, raw_flags="h"),
            StrongLine(element="Kr", wavelength=961.961, intensity=130, raw_flags="h"),
            StrongLine(element="Kr", wavelength=966.334, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=971.160, intensity=70, raw_flags="h"),
            StrongLine(element="Kr", wavelength=975.17610, intensity=300, raw_flags=""),
            StrongLine(element="Kr", wavelength=980.314, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=985.6314, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=1022.146, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=1118.7108, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=1125.7711, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=1125.9126, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=1145.7481, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=1179.2425, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=1181.9377, intensity=250, raw_flags=""),
            StrongLine(element="Kr", wavelength=1199.7105, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=1207.7224, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=1286.1892, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=1317.7412, intensity=200, raw_flags=""),
            StrongLine(element="Kr", wavelength=1362.2415, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=1363.4220, intensity=400, raw_flags="P"),
            StrongLine(element="Kr", wavelength=1365.8394, intensity=130, raw_flags=""),
            StrongLine(element="Kr", wavelength=1371.1036, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=1373.8851, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=1397.4027, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=1404.5657, intensity=90, raw_flags=""),
            StrongLine(element="Kr", wavelength=1410.4298, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=1440.222, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=1442.6793, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=1451.784, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=1473.4436, intensity=250, raw_flags=""),
            StrongLine(element="Kr", wavelength=1476.2672, intensity=90, raw_flags=""),
            StrongLine(element="Kr", wavelength=1476.5472, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=1496.1894, intensity=70, raw_flags=""),
            StrongLine(element="Kr", wavelength=1500.5307, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=1520.9526, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=1523.9615, intensity=300, raw_flags=""),
            StrongLine(element="Kr", wavelength=1532.6480, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=1533.4958, intensity=250, raw_flags=""),
            StrongLine(element="Kr", wavelength=1537.2037, intensity=120, raw_flags=""),
            StrongLine(element="Kr", wavelength=1547.4026, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=1568.102, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=1582.009, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=1672.6513, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=1678.5128, intensity=300, raw_flags=""),
            StrongLine(element="Kr", wavelength=1685.3488, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=1689.0441, intensity=400, raw_flags="P"),
            StrongLine(element="Kr", wavelength=1689.6753, intensity=250, raw_flags=""),
            StrongLine(element="Kr", wavelength=1693.5806, intensity=300, raw_flags=""),
            StrongLine(element="Kr", wavelength=1709.8771, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=1736.7606, intensity=120, raw_flags=""),
            StrongLine(element="Kr", wavelength=1740.4443, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=1761.6854, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=1784.2737, intensity=110, raw_flags=""),
            StrongLine(element="Kr", wavelength=1800.2229, intensity=120, raw_flags=""),
            StrongLine(element="Kr", wavelength=1816.7315, intensity=400, raw_flags="P"),
            StrongLine(element="Kr", wavelength=1839.9786, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=1858.0896, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=1869.6294, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=1878.5460, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=1879.7703, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=2020.9878, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=2042.3964, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=2044.6971, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=2116.5471, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=2190.2513, intensity=300, raw_flags=""),
            StrongLine(element="Kr", wavelength=2248.5775, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=2334.0416, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=2426.0506, intensity=20, raw_flags=""),
            StrongLine(element="Kr", wavelength=2429.2221, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=2523.3820, intensity=100, raw_flags=""),
            StrongLine(element="Kr", wavelength=2861.055, intensity=30, raw_flags=""),
            StrongLine(element="Kr", wavelength=2865.572, intensity=150, raw_flags=""),
            StrongLine(element="Kr", wavelength=2876.971, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=2882.249, intensity=25, raw_flags=""),
            StrongLine(element="Kr", wavelength=2923.669, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=3066.354, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=3097.916, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=3930.06, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=3948.652, intensity=200, raw_flags=""),
            StrongLine(element="Kr", wavelength=3955.725, intensity=40, raw_flags=""),
            StrongLine(element="Kr", wavelength=3957.260, intensity=15, raw_flags=""),
            StrongLine(element="Kr", wavelength=3958.84, intensity=250, raw_flags=""),
            StrongLine(element="Kr", wavelength=3958.96, intensity=200, raw_flags=""),
            StrongLine(element="Kr", wavelength=3995.48, intensity=80, raw_flags=""),
            StrongLine(element="Kr", wavelength=3996.66, intensity=50, raw_flags=""),
            StrongLine(element="Kr", wavelength=4030.61, intensity=200, raw_flags=""),
            StrongLine(element="Kr", wavelength=4068.516, intensity=40, raw_flags=""),
        ],
        persistent_lines=[
            StrongLine(element="Kr", wavelength=84.406, intensity=30, raw_flags="P"),
            StrongLine(element="Kr", wavelength=88.630, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=91.743, intensity=700, raw_flags="P"),
            StrongLine(element="Kr", wavelength=96.497, intensity=700, raw_flags="P"),
            StrongLine(element="Kr", wavelength=116.48671, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=123.58378, intensity=1000, raw_flags="P"),
            StrongLine(element="Kr", wavelength=435.5477, intensity=1000, raw_flags="P"),
            StrongLine(element="Kr", wavelength=461.9166, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=465.8876, intensity=700, raw_flags="P"),
            StrongLine(element="Kr", wavelength=473.9002, intensity=1000, raw_flags="P"),
            StrongLine(element="Kr", wavelength=476.5744, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=483.2077, intensity=250, raw_flags="P"),
            StrongLine(element="Kr", wavelength=484.6612, intensity=250, raw_flags="P"),
            StrongLine(element="Kr", wavelength=557.02894, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=587.09160, intensity=500, raw_flags="P"),
            StrongLine(element="Kr", wavelength=810.43655, intensity=700, raw_flags="P"),
            StrongLine(element="Kr", wavelength=811.29012, intensity=1000, raw_flags="P"),
            StrongLine(element="Kr", wavelength=819.00566, intensity=500, raw_flags="P"),
            StrongLine(element="Kr", wavelength=826.32426, intensity=500, raw_flags="P"),
            StrongLine(element="Kr", wavelength=829.81099, intensity=800, raw_flags="P"),
            StrongLine(element="Kr", wavelength=850.88728, intensity=500, raw_flags="P"),
            StrongLine(element="Kr", wavelength=877.67505, intensity=1000, raw_flags="P"),
            StrongLine(element="Kr", wavelength=957.752, intensity=150, raw_flags="P"),
            StrongLine(element="Kr", wavelength=1022.146, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=1363.4220, intensity=400, raw_flags="P"),
            StrongLine(element="Kr", wavelength=1442.6793, intensity=300, raw_flags="P"),
            StrongLine(element="Kr", wavelength=1689.0441, intensity=400, raw_flags="P"),
            StrongLine(element="Kr", wavelength=1816.7315, intensity=400, raw_flags="P"),
        ]
    ),
    "N": StrongLines(
        element="N",
        lines=[
            StrongLine(element="N", wavelength=64.4635, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=64.4837, intensity=600, raw_flags="P"),
            StrongLine(element="N", wavelength=64.5179, intensity=1000, raw_flags="P"),
            StrongLine(element="N", wavelength=69.270, intensity=80, raw_flags=""),
            StrongLine(element="N", wavelength=77.5967, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=91.5613, intensity=50, raw_flags="P"),
            StrongLine(element="N", wavelength=91.5963, intensity=50, raw_flags="P"),
            StrongLine(element="N", wavelength=91.6019, intensity=60, raw_flags="Pd"),
            StrongLine(element="N", wavelength=91.6708, intensity=200, raw_flags="Pd"),
            StrongLine(element="N", wavelength=108.3994, intensity=25, raw_flags="P"),
            StrongLine(element="N", wavelength=108.4580, intensity=60, raw_flags="Pd"),
            StrongLine(element="N", wavelength=108.5550, intensity=15, raw_flags="P"),
            StrongLine(element="N", wavelength=108.5710, intensity=100, raw_flags="P"),
            StrongLine(element="N", wavelength=113.4165, intensity=90, raw_flags=""),
            StrongLine(element="N", wavelength=113.4415, intensity=90, raw_flags=""),
            StrongLine(element="N", wavelength=119.9550, intensity=1000, raw_flags="P"),
            StrongLine(element="N", wavelength=120.0223, intensity=700, raw_flags="P"),
            StrongLine(element="N", wavelength=120.0710, intensity=300, raw_flags="P"),
            StrongLine(element="N", wavelength=124.3179, intensity=250, raw_flags="P"),
            StrongLine(element="N", wavelength=141.194, intensity=90, raw_flags=""),
            StrongLine(element="N", wavelength=149.2625, intensity=250, raw_flags="P"),
            StrongLine(element="N", wavelength=149.2820, intensity=120, raw_flags=""),
            StrongLine(element="N", wavelength=149.4675, intensity=150, raw_flags=""),
            StrongLine(element="N", wavelength=174.2729, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=174.5252, intensity=150, raw_flags=""),
            StrongLine(element="N", wavelength=395.585, intensity=20, raw_flags=""),
            StrongLine(element="N", wavelength=399.500, intensity=50, raw_flags="P"),
            StrongLine(element="N", wavelength=404.131, intensity=30, raw_flags=""),
            StrongLine(element="N", wavelength=424.178, intensity=20, raw_flags=""),
            StrongLine(element="N", wavelength=444.703, intensity=30, raw_flags="P"),
            StrongLine(element="N", wavelength=460.148, intensity=30, raw_flags=""),
            StrongLine(element="N", wavelength=460.716, intensity=20, raw_flags=""),
            StrongLine(element="N", wavelength=462.139, intensity=20, raw_flags=""),
            StrongLine(element="N", wavelength=463.054, intensity=40, raw_flags="P"),
            StrongLine(element="N", wavelength=464.308, intensity=30, raw_flags=""),
            StrongLine(element="N", wavelength=480.329, intensity=20, raw_flags=""),
            StrongLine(element="N", wavelength=496.398, intensity=80, raw_flags=""),
            StrongLine(element="N", wavelength=499.436, intensity=20, raw_flags=""),
            StrongLine(element="N", wavelength=500.148, intensity=30, raw_flags=""),
            StrongLine(element="N", wavelength=500.515, intensity=40, raw_flags="P"),
            StrongLine(element="N", wavelength=500.732, intensity=30, raw_flags=""),
            StrongLine(element="N", wavelength=501.062, intensity=20, raw_flags=""),
            StrongLine(element="N", wavelength=504.510, intensity=30, raw_flags=""),
            StrongLine(element="N", wavelength=566.663, intensity=30, raw_flags=""),
            StrongLine(element="N", wavelength=567.602, intensity=30, raw_flags=""),
            StrongLine(element="N", wavelength=567.956, intensity=40, raw_flags="P"),
            StrongLine(element="N", wavelength=568.621, intensity=20, raw_flags=""),
            StrongLine(element="N", wavelength=571.077, intensity=20, raw_flags=""),
            StrongLine(element="N", wavelength=575.250, intensity=150, raw_flags=""),
            StrongLine(element="N", wavelength=593.178, intensity=30, raw_flags=""),
            StrongLine(element="N", wavelength=594.165, intensity=30, raw_flags="P"),
            StrongLine(element="N", wavelength=648.205, intensity=40, raw_flags="P"),
            StrongLine(element="N", wavelength=648.270, intensity=90, raw_flags=""),
            StrongLine(element="N", wavelength=661.056, intensity=40, raw_flags="P"),
            StrongLine(element="N", wavelength=742.364, intensity=150, raw_flags=""),
            StrongLine(element="N", wavelength=744.229, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=746.831, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=818.487, intensity=90, raw_flags=""),
            StrongLine(element="N", wavelength=818.802, intensity=90, raw_flags=""),
            StrongLine(element="N", wavelength=821.634, intensity=140, raw_flags=""),
            StrongLine(element="N", wavelength=822.314, intensity=90, raw_flags=""),
            StrongLine(element="N", wavelength=824.239, intensity=90, raw_flags=""),
            StrongLine(element="N", wavelength=843.874, intensity=30, raw_flags=""),
            StrongLine(element="N", wavelength=856.774, intensity=120, raw_flags=""),
            StrongLine(element="N", wavelength=859.400, intensity=140, raw_flags=""),
            StrongLine(element="N", wavelength=862.924, intensity=150, raw_flags=""),
            StrongLine(element="N", wavelength=865.589, intensity=120, raw_flags=""),
            StrongLine(element="N", wavelength=868.028, intensity=150, raw_flags="P"),
            StrongLine(element="N", wavelength=868.340, intensity=150, raw_flags="P"),
            StrongLine(element="N", wavelength=868.615, intensity=120, raw_flags=""),
            StrongLine(element="N", wavelength=870.325, intensity=120, raw_flags=""),
            StrongLine(element="N", wavelength=871.170, intensity=140, raw_flags=""),
            StrongLine(element="N", wavelength=871.883, intensity=120, raw_flags=""),
            StrongLine(element="N", wavelength=938.680, intensity=120, raw_flags=""),
            StrongLine(element="N", wavelength=939.279, intensity=140, raw_flags=""),
            StrongLine(element="N", wavelength=1011.248, intensity=80, raw_flags=""),
            StrongLine(element="N", wavelength=1011.464, intensity=90, raw_flags=""),
            StrongLine(element="N", wavelength=1218.682, intensity=90, raw_flags=""),
            StrongLine(element="N", wavelength=1246.962, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=1342.961, intensity=120, raw_flags=""),
            StrongLine(element="N", wavelength=1358.133, intensity=200, raw_flags="P"),
        ],
        persistent_lines=[
            StrongLine(element="N", wavelength=64.4635, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=64.4837, intensity=600, raw_flags="P"),
            StrongLine(element="N", wavelength=64.5179, intensity=1000, raw_flags="P"),
            StrongLine(element="N", wavelength=77.5967, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=91.5613, intensity=50, raw_flags="P"),
            StrongLine(element="N", wavelength=91.5963, intensity=50, raw_flags="P"),
            StrongLine(element="N", wavelength=91.6019, intensity=60, raw_flags="Pd"),
            StrongLine(element="N", wavelength=91.6708, intensity=200, raw_flags="Pd"),
            StrongLine(element="N", wavelength=108.3994, intensity=25, raw_flags="P"),
            StrongLine(element="N", wavelength=108.4580, intensity=60, raw_flags="Pd"),
            StrongLine(element="N", wavelength=108.5550, intensity=15, raw_flags="P"),
            StrongLine(element="N", wavelength=108.5710, intensity=100, raw_flags="P"),
            StrongLine(element="N", wavelength=119.9550, intensity=1000, raw_flags="P"),
            StrongLine(element="N", wavelength=120.0223, intensity=700, raw_flags="P"),
            StrongLine(element="N", wavelength=120.0710, intensity=300, raw_flags="P"),
            StrongLine(element="N", wavelength=124.3179, intensity=250, raw_flags="P"),
            StrongLine(element="N", wavelength=149.2625, intensity=250, raw_flags="P"),
            StrongLine(element="N", wavelength=174.2729, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=399.500, intensity=50, raw_flags="P"),
            StrongLine(element="N", wavelength=444.703, intensity=30, raw_flags="P"),
            StrongLine(element="N", wavelength=463.054, intensity=40, raw_flags="P"),
            StrongLine(element="N", wavelength=500.515, intensity=40, raw_flags="P"),
            StrongLine(element="N", wavelength=567.956, intensity=40, raw_flags="P"),
            StrongLine(element="N", wavelength=594.165, intensity=30, raw_flags="P"),
            StrongLine(element="N", wavelength=648.205, intensity=40, raw_flags="P"),
            StrongLine(element="N", wavelength=661.056, intensity=40, raw_flags="P"),
            StrongLine(element="N", wavelength=744.229, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=746.831, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=868.028, intensity=150, raw_flags="P"),
            StrongLine(element="N", wavelength=868.340, intensity=150, raw_flags="P"),
            StrongLine(element="N", wavelength=1246.962, intensity=200, raw_flags="P"),
            StrongLine(element="N", wavelength=1358.133, intensity=200, raw_flags="P"),
        ]
    ),
    "Ne": StrongLines(
        element="Ne",
        lines=[
            StrongLine(element="Ne", wavelength=35.29549, intensity=90, raw_flags="P"),
            StrongLine(element="Ne", wavelength=35.49620, intensity=60, raw_flags="P"),
            StrongLine(element="Ne", wavelength=36.14321, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=36.24544, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=40.58538, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=40.71377, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=44.50393, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=44.62552, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=44.65902, intensity=250, raw_flags="P"),
            StrongLine(element="Ne", wavelength=44.78146, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=45.46540, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=45.52730, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=45.62728, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=45.63485, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=45.68962, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=46.07284, intensity=1000, raw_flags="P"),
            StrongLine(element="Ne", wavelength=46.23908, intensity=500, raw_flags="P"),
            StrongLine(element="Ne", wavelength=58.72127, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=58.71792, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=58.99114, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=59.18306, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=59.59200, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=59.87056, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=59.88897, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=60.00365, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=60.27263, intensity=130, raw_flags=""),
            StrongLine(element="Ne", wavelength=61.56283, intensity=170, raw_flags="P"),
            StrongLine(element="Ne", wavelength=61.86716, intensity=170, raw_flags="P"),
            StrongLine(element="Ne", wavelength=61.91023, intensity=130, raw_flags="P"),
            StrongLine(element="Ne", wavelength=62.68232, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=62.97388, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=73.58962, intensity=1000, raw_flags="P"),
            StrongLine(element="Ne", wavelength=74.37195, intensity=400, raw_flags="P"),
            StrongLine(element="Ne", wavelength=99.38825, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=106.86488, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=113.17224, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=113.18490, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=122.98367, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=141.83779, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=142.85822, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=143.60813, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=168.16840, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=168.83553, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=188.81064, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=188.97120, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=190.74940, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=191.60818, intensity=500, raw_flags="P"),
            StrongLine(element="Ne", wavelength=193.00345, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=193.88269, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=194.54521, intensity=100, raw_flags="c"),
            StrongLine(element="Ne", wavelength=200.7009, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=202.5560, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=208.5466, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=209.6106, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=209.6248, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=256.2123, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=256.7121, intensity=90, raw_flags="w"),
            StrongLine(element="Ne", wavelength=262.3107, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=262.9885, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=263.6069, intensity=90, raw_flags="w"),
            StrongLine(element="Ne", wavelength=263.8289, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=264.4097, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=276.2921, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=279.2019, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=279.4221, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=280.9485, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=290.6592, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=290.6816, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=291.0061, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=291.0408, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=291.1138, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=291.5122, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=292.5618, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=293.2103, intensity=80, raw_flags="w"),
            StrongLine(element="Ne", wavelength=294.0653, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=294.6044, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=295.5725, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=296.3236, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=296.7184, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=297.2997, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=297.47189, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=297.9461, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=298.26696, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=300.1668, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=301.7311, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=302.7016, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=302.8864, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=303.0787, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=303.4461, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=303.5923, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=303.7720, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=303.9586, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=304.4088, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=304.5556, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=304.7556, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=305.4345, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=305.4677, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=305.73907, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=305.9106, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=306.2491, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=306.3301, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=307.0887, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=307.1529, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=307.5731, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=308.8166, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=309.2092, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=309.2901, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=309.4006, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=309.5103, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=309.7131, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=311.7980, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=311.8160, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=314.1332, intensity=300, raw_flags=""),
            StrongLine(element="Ne", wavelength=314.3721, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=314.8681, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=316.4429, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=316.5648, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=318.8743, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=319.4579, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=319.8586, intensity=500, raw_flags="P"),
            StrongLine(element="Ne", wavelength=320.8965, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=320.9356, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=321.3735, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=321.4329, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=321.8193, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=322.4818, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=322.9573, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=323.0070, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=323.0419, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=323.2022, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=323.2372, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=324.3396, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=324.4095, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=324.8345, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=325.0355, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=329.7726, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=330.9740, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=331.9722, intensity=300, raw_flags=""),
            StrongLine(element="Ne", wavelength=332.3745, intensity=1000, raw_flags="P"),
            StrongLine(element="Ne", wavelength=332.7153, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=332.9158, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=333.4836, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=334.4395, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=334.5453, intensity=300, raw_flags=""),
            StrongLine(element="Ne", wavelength=334.5829, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=335.5016, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=335.7820, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=336.0597, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=336.2161, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=336.2707, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=336.7218, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=336.98076, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=336.99072, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=337.1799, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=337.8216, intensity=500, raw_flags="P"),
            StrongLine(element="Ne", wavelength=338.8417, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=338.8945, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=339.2798, intensity=300, raw_flags=""),
            StrongLine(element="Ne", wavelength=340.4822, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=340.6947, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=341.3148, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=341.6914, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=341.7688, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=341.79031, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=341.80055, intensity=5, raw_flags=""),
            StrongLine(element="Ne", wavelength=342.8687, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=344.77024, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=345.41944, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=345.6610, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=345.9321, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=346.05237, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=346.43382, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=346.65781, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=347.25706, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=347.9519, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=348.0718, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=348.1933, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=349.80636, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=350.12159, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=351.51902, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=352.04711, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=354.2847, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=355.7805, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=356.1198, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=356.8502, intensity=250, raw_flags=""),
            StrongLine(element="Ne", wavelength=357.4181, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=357.4612, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=359.35257, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=359.36389, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=360.01685, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=363.36640, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=364.3927, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=366.4073, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=368.22421, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=368.57352, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=369.4213, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=370.12244, intensity=4, raw_flags=""),
            StrongLine(element="Ne", wavelength=370.9622, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=371.3079, intensity=250, raw_flags=""),
            StrongLine(element="Ne", wavelength=372.7107, intensity=250, raw_flags=""),
            StrongLine(element="Ne", wavelength=376.6259, intensity=800, raw_flags="P"),
            StrongLine(element="Ne", wavelength=377.7133, intensity=1000, raw_flags="P"),
            StrongLine(element="Ne", wavelength=381.8427, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=382.9749, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=421.9745, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=423.3850, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=425.0649, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=436.9862, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=437.9400, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=437.9550, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=438.5059, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=439.1991, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=439.7990, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=440.9299, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=441.3215, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=442.1389, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=442.8516, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=442.8634, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=443.0904, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=443.0942, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=445.7049, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=452.2720, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=453.77545, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=456.9057, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=470.43949, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=470.88594, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=471.00650, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=471.20633, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=471.5344, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=475.27320, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=478.89258, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=479.02195, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=482.7338, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=488.49170, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=500.51587, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=503.77512, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=514.49384, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=533.07775, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=534.10938, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=534.32834, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=540.05618, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=556.27662, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=565.66588, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=571.92248, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=574.82985, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=576.44188, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=580.44496, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=582.01558, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=585.24879, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=587.28275, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=588.18952, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=590.24623, intensity=5, raw_flags=""),
            StrongLine(element="Ne", wavelength=590.64294, intensity=5, raw_flags=""),
            StrongLine(element="Ne", wavelength=594.48342, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=596.54710, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=597.46273, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=597.55340, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=598.79074, intensity=15, raw_flags=""),
            StrongLine(element="Ne", wavelength=602.99969, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=607.43377, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=609.61631, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=612.84499, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=614.30626, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=616.35939, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=618.21460, intensity=15, raw_flags=""),
            StrongLine(element="Ne", wavelength=621.72812, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=626.64950, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=630.47889, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=632.81646, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=633.44278, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=638.29917, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=640.2248, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=650.65281, intensity=150, raw_flags="P"),
            StrongLine(element="Ne", wavelength=653.28822, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=659.89529, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=665.20927, intensity=15, raw_flags=""),
            StrongLine(element="Ne", wavelength=667.82762, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=671.70430, intensity=7, raw_flags=""),
            StrongLine(element="Ne", wavelength=692.94673, intensity=1000, raw_flags="P"),
            StrongLine(element="Ne", wavelength=702.40504, intensity=300, raw_flags=""),
            StrongLine(element="Ne", wavelength=703.24131, intensity=800, raw_flags="P"),
            StrongLine(element="Ne", wavelength=705.12923, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=705.91074, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=717.39381, intensity=800, raw_flags="P"),
            StrongLine(element="Ne", wavelength=721.3200, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=723.5188, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=724.51666, intensity=800, raw_flags="P"),
            StrongLine(element="Ne", wavelength=734.3945, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=747.24386, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=748.88712, intensity=300, raw_flags=""),
            StrongLine(element="Ne", wavelength=749.2102, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=752.2818, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=753.57741, intensity=300, raw_flags=""),
            StrongLine(element="Ne", wavelength=754.40443, intensity=130, raw_flags=""),
            StrongLine(element="Ne", wavelength=772.46233, intensity=1, raw_flags=""),
            StrongLine(element="Ne", wavelength=774.0738, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=783.90529, intensity=2, raw_flags=""),
            StrongLine(element="Ne", wavelength=792.6201, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=792.71177, intensity=3, raw_flags=""),
            StrongLine(element="Ne", wavelength=793.69961, intensity=13, raw_flags=""),
            StrongLine(element="Ne", wavelength=794.31814, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=808.24580, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=808.4345, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=811.85492, intensity=40, raw_flags=""),
            StrongLine(element="Ne", wavelength=812.89108, intensity=12, raw_flags=""),
            StrongLine(element="Ne", wavelength=813.64054, intensity=170, raw_flags=""),
            StrongLine(element="Ne", wavelength=825.93790, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=826.4807, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=826.60772, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=826.71162, intensity=10, raw_flags=""),
            StrongLine(element="Ne", wavelength=830.03258, intensity=300, raw_flags=""),
            StrongLine(element="Ne", wavelength=831.4995, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=836.57466, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=837.2106, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=837.76080, intensity=800, raw_flags="P"),
            StrongLine(element="Ne", wavelength=841.71606, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=841.84274, intensity=250, raw_flags=""),
            StrongLine(element="Ne", wavelength=846.33575, intensity=40, raw_flags=""),
            StrongLine(element="Ne", wavelength=848.44435, intensity=13, raw_flags=""),
            StrongLine(element="Ne", wavelength=849.53598, intensity=700, raw_flags=""),
            StrongLine(element="Ne", wavelength=854.46958, intensity=15, raw_flags=""),
            StrongLine(element="Ne", wavelength=857.13524, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=859.12584, intensity=400, raw_flags=""),
            StrongLine(element="Ne", wavelength=863.46470, intensity=350, raw_flags=""),
            StrongLine(element="Ne", wavelength=864.70411, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=865.43831, intensity=600, raw_flags="P"),
            StrongLine(element="Ne", wavelength=865.55220, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=866.8256, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=867.94925, intensity=130, raw_flags=""),
            StrongLine(element="Ne", wavelength=868.19211, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=870.41116, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=877.16563, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=878.06226, intensity=600, raw_flags="P"),
            StrongLine(element="Ne", wavelength=878.37533, intensity=400, raw_flags="P"),
            StrongLine(element="Ne", wavelength=883.09072, intensity=6, raw_flags=""),
            StrongLine(element="Ne", wavelength=885.38668, intensity=300, raw_flags=""),
            StrongLine(element="Ne", wavelength=886.53063, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=886.57552, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=891.95006, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=898.85564, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=907.9462, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=914.86716, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=920.17591, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=922.00601, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=922.15801, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=922.66903, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=927.55196, intensity=9, raw_flags=""),
            StrongLine(element="Ne", wavelength=928.7563, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=930.08527, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=931.05839, intensity=8, raw_flags=""),
            StrongLine(element="Ne", wavelength=931.39726, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=932.65068, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=937.33078, intensity=15, raw_flags=""),
            StrongLine(element="Ne", wavelength=942.53788, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=945.92095, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=948.66818, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=953.41629, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=954.74049, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=957.7013, intensity=120, raw_flags=""),
            StrongLine(element="Ne", wavelength=966.54197, intensity=180, raw_flags=""),
            StrongLine(element="Ne", wavelength=980.8860, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=1029.54174, intensity=4, raw_flags=""),
            StrongLine(element="Ne", wavelength=1056.24075, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=1079.80429, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=1084.44772, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=1114.30200, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=1117.75240, intensity=500, raw_flags="P"),
            StrongLine(element="Ne", wavelength=1139.04339, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=1140.91343, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=1152.27459, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=1152.50194, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=1153.63445, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=1160.15366, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=1161.40807, intensity=130, raw_flags=""),
            StrongLine(element="Ne", wavelength=1168.80017, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=1176.67924, intensity=150, raw_flags=""),
            StrongLine(element="Ne", wavelength=1178.90435, intensity=130, raw_flags=""),
            StrongLine(element="Ne", wavelength=1178.98891, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=1198.4912, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=1206.6334, intensity=200, raw_flags=""),
            StrongLine(element="Ne", wavelength=1245.9389, intensity=40, raw_flags=""),
            StrongLine(element="Ne", wavelength=1268.9201, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=1291.2014, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=1321.9241, intensity=40, raw_flags=""),
            StrongLine(element="Ne", wavelength=1523.0714, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=1716.1929, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=1803.5812, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=1808.3181, intensity=40, raw_flags=""),
            StrongLine(element="Ne", wavelength=1808.3263, intensity=9, raw_flags=""),
            StrongLine(element="Ne", wavelength=1822.1087, intensity=15, raw_flags=""),
            StrongLine(element="Ne", wavelength=1822.7016, intensity=13, raw_flags=""),
            StrongLine(element="Ne", wavelength=1827.6642, intensity=140, raw_flags=""),
            StrongLine(element="Ne", wavelength=1828.2614, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=1830.3967, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=1835.9094, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=1838.4826, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=1838.9937, intensity=90, raw_flags=""),
            StrongLine(element="Ne", wavelength=1840.2836, intensity=40, raw_flags=""),
            StrongLine(element="Ne", wavelength=1842.2402, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=1845.8640, intensity=13, raw_flags=""),
            StrongLine(element="Ne", wavelength=1847.5800, intensity=40, raw_flags=""),
            StrongLine(element="Ne", wavelength=1859.1541, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=1859.7698, intensity=100, raw_flags=""),
            StrongLine(element="Ne", wavelength=1861.8908, intensity=16, raw_flags=""),
            StrongLine(element="Ne", wavelength=1862.5159, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=2104.127, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=2170.811, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=2224.736, intensity=13, raw_flags=""),
            StrongLine(element="Ne", wavelength=2242.814, intensity=13, raw_flags=""),
            StrongLine(element="Ne", wavelength=2253.038, intensity=80, raw_flags=""),
            StrongLine(element="Ne", wavelength=2266.179, intensity=13, raw_flags=""),
            StrongLine(element="Ne", wavelength=2310.048, intensity=25, raw_flags=""),
            StrongLine(element="Ne", wavelength=2326.027, intensity=40, raw_flags=""),
            StrongLine(element="Ne", wavelength=2337.296, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=2356.533, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=2363.648, intensity=170, raw_flags=""),
            StrongLine(element="Ne", wavelength=2370.166, intensity=12, raw_flags=""),
            StrongLine(element="Ne", wavelength=2370.913, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=2395.140, intensity=110, raw_flags=""),
            StrongLine(element="Ne", wavelength=2395.643, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=2397.816, intensity=60, raw_flags=""),
            StrongLine(element="Ne", wavelength=2409.857, intensity=11, raw_flags=""),
            StrongLine(element="Ne", wavelength=2416.143, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=2424.961, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=2436.501, intensity=70, raw_flags=""),
            StrongLine(element="Ne", wavelength=2437.161, intensity=40, raw_flags=""),
            StrongLine(element="Ne", wavelength=2444.786, intensity=20, raw_flags=""),
            StrongLine(element="Ne", wavelength=2445.939, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=2477.649, intensity=17, raw_flags=""),
            StrongLine(element="Ne", wavelength=2492.889, intensity=30, raw_flags=""),
            StrongLine(element="Ne", wavelength=2516.170, intensity=13, raw_flags=""),
            StrongLine(element="Ne", wavelength=2552.433, intensity=50, raw_flags=""),
            StrongLine(element="Ne", wavelength=2838.620, intensity=6, raw_flags=""),
            StrongLine(element="Ne", wavelength=3020.049, intensity=6, raw_flags=""),
            StrongLine(element="Ne", wavelength=3317.309, intensity=8, raw_flags=""),
            StrongLine(element="Ne", wavelength=3335.238, intensity=17, raw_flags=""),
            StrongLine(element="Ne", wavelength=3389.981, intensity=5, raw_flags=""),
            StrongLine(element="Ne", wavelength=3390.302, intensity=4, raw_flags=""),
            StrongLine(element="Ne", wavelength=3391.310, intensity=12, raw_flags=""),
            StrongLine(element="Ne", wavelength=3413.134, intensity=4, raw_flags=""),
            StrongLine(element="Ne", wavelength=3447.143, intensity=6, raw_flags=""),
            StrongLine(element="Ne", wavelength=3583.481, intensity=8, raw_flags=""),
        ],
        persistent_lines=[
            StrongLine(element="Ne", wavelength=35.29549, intensity=90, raw_flags="P"),
            StrongLine(element="Ne", wavelength=35.49620, intensity=60, raw_flags="P"),
            StrongLine(element="Ne", wavelength=44.50393, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=44.62552, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=44.65902, intensity=250, raw_flags="P"),
            StrongLine(element="Ne", wavelength=46.07284, intensity=1000, raw_flags="P"),
            StrongLine(element="Ne", wavelength=46.23908, intensity=500, raw_flags="P"),
            StrongLine(element="Ne", wavelength=61.56283, intensity=170, raw_flags="P"),
            StrongLine(element="Ne", wavelength=61.86716, intensity=170, raw_flags="P"),
            StrongLine(element="Ne", wavelength=61.91023, intensity=130, raw_flags="P"),
            StrongLine(element="Ne", wavelength=62.68232, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=62.97388, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=73.58962, intensity=1000, raw_flags="P"),
            StrongLine(element="Ne", wavelength=74.37195, intensity=400, raw_flags="P"),
            StrongLine(element="Ne", wavelength=191.60818, intensity=500, raw_flags="P"),
            StrongLine(element="Ne", wavelength=193.00345, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=302.7016, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=302.8864, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=319.8586, intensity=500, raw_flags="P"),
            StrongLine(element="Ne", wavelength=332.3745, intensity=1000, raw_flags="P"),
            StrongLine(element="Ne", wavelength=337.8216, intensity=500, raw_flags="P"),
            StrongLine(element="Ne", wavelength=352.04711, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=376.6259, intensity=800, raw_flags="P"),
            StrongLine(element="Ne", wavelength=377.7133, intensity=1000, raw_flags="P"),
            StrongLine(element="Ne", wavelength=540.05618, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=585.24879, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=602.99969, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=607.43377, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=614.30626, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=616.35939, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=621.72812, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=626.64950, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=638.29917, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=640.2248, intensity=200, raw_flags="P"),
            StrongLine(element="Ne", wavelength=650.65281, intensity=150, raw_flags="P"),
            StrongLine(element="Ne", wavelength=659.89529, intensity=100, raw_flags="P"),
            StrongLine(element="Ne", wavelength=692.94673, intensity=1000, raw_flags="P"),
            StrongLine(element="Ne", wavelength=703.24131, intensity=800, raw_flags="P"),
            StrongLine(element="Ne", wavelength=717.39381, intensity=800, raw_flags="P"),
            StrongLine(element="Ne", wavelength=724.51666, intensity=800, raw_flags="P"),
            StrongLine(element="Ne", wavelength=837.76080, intensity=800, raw_flags="P"),
            StrongLine(element="Ne", wavelength=865.43831, intensity=600, raw_flags="P"),
            StrongLine(element="Ne", wavelength=878.06226, intensity=600, raw_flags="P"),
            StrongLine(element="Ne", wavelength=878.37533, intensity=400, raw_flags="P"),
            StrongLine(element="Ne", wavelength=1114.30200, intensity=300, raw_flags="P"),
            StrongLine(element="Ne", wavelength=1117.75240, intensity=500, raw_flags="P"),
            StrongLine(element="Ne", wavelength=1152.27459, intensity=300, raw_flags="P"),
        ]
    ),
    "O": StrongLines(
        element="O",
        lines=[
            StrongLine(element="O", wavelength=42.9918, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=43.0041, intensity=700, raw_flags="P"),
            StrongLine(element="O", wavelength=43.0177, intensity=1000, raw_flags="P"),
            StrongLine(element="O", wavelength=53.78319, intensity=800, raw_flags="P"),
            StrongLine(element="O", wavelength=53.82636, intensity=1000, raw_flags="P"),
            StrongLine(element="O", wavelength=53.8318, intensity=600, raw_flags="P"),
            StrongLine(element="O", wavelength=53.90855, intensity=800, raw_flags="P"),
            StrongLine(element="O", wavelength=53.95489, intensity=700, raw_flags="P"),
            StrongLine(element="O", wavelength=53.98544, intensity=500, raw_flags="P"),
            StrongLine(element="O", wavelength=71.85036, intensity=800, raw_flags="P"),
            StrongLine(element="O", wavelength=71.85663, intensity=500, raw_flags="P"),
            StrongLine(element="O", wavelength=79.6664, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=79.6682, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=83.27587, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=83.33302, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=83.44655, intensity=500, raw_flags="P"),
            StrongLine(element="O", wavelength=98.8773, intensity=50, raw_flags=""),
            StrongLine(element="O", wavelength=102.5762, intensity=80, raw_flags=""),
            StrongLine(element="O", wavelength=103.9230, intensity=50, raw_flags=""),
            StrongLine(element="O", wavelength=130.2168, intensity=1000, raw_flags="P"),
            StrongLine(element="O", wavelength=130.4858, intensity=700, raw_flags="P"),
            StrongLine(element="O", wavelength=130.6029, intensity=300, raw_flags="P"),
            StrongLine(element="O", wavelength=244.5538, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=273.3289, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=297.229, intensity=90, raw_flags=""),
            StrongLine(element="O", wavelength=313.4720, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=313.8335, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=328.7472, intensity=300, raw_flags=""),
            StrongLine(element="O", wavelength=339.0209, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=347.0676, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=372.7320, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=374.9486, intensity=300, raw_flags=""),
            StrongLine(element="O", wavelength=391.1957, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=394.729, intensity=60, raw_flags=""),
            StrongLine(element="O", wavelength=394.748, intensity=50, raw_flags=""),
            StrongLine(element="O", wavelength=394.759, intensity=50, raw_flags=""),
            StrongLine(element="O", wavelength=397.32562, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=398.27140, intensity=200, raw_flags=""),
            StrongLine(element="O", wavelength=407.2157, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=407.5862, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=411.9215, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=418.5449, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=418.9789, intensity=400, raw_flags=""),
            StrongLine(element="O", wavelength=431.7138, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=434.9426, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=441.4905, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=441.6974, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=459.0972, intensity=300, raw_flags=""),
            StrongLine(element="O", wavelength=459.6175, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=464.1810, intensity=300, raw_flags="P"),
            StrongLine(element="O", wavelength=464.9135, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=466.1633, intensity=300, raw_flags=""),
            StrongLine(element="O", wavelength=467.6235, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=470.5352, intensity=250, raw_flags=""),
            StrongLine(element="O", wavelength=615.598, intensity=130, raw_flags=""),
            StrongLine(element="O", wavelength=615.677, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=615.818, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=645.598, intensity=130, raw_flags=""),
            StrongLine(element="O", wavelength=700.223, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=725.415, intensity=130, raw_flags=""),
            StrongLine(element="O", wavelength=725.445, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=777.194, intensity=300, raw_flags="P"),
            StrongLine(element="O", wavelength=777.417, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=777.539, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=822.182, intensity=130, raw_flags=""),
            StrongLine(element="O", wavelength=844.625, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=844.636, intensity=300, raw_flags="P"),
            StrongLine(element="O", wavelength=844.676, intensity=300, raw_flags="P"),
            StrongLine(element="O", wavelength=926.081, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=926.084, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=926.094, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=926.258, intensity=130, raw_flags=""),
            StrongLine(element="O", wavelength=926.267, intensity=200, raw_flags=""),
            StrongLine(element="O", wavelength=926.277, intensity=200, raw_flags=""),
            StrongLine(element="O", wavelength=926.594, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=926.601, intensity=200, raw_flags="P"),
            StrongLine(element="O", wavelength=1128.634, intensity=200, raw_flags=""),
            StrongLine(element="O", wavelength=1128.691, intensity=200, raw_flags=""),
            StrongLine(element="O", wavelength=1128.702, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=1128.732, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=1129.510, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=1129.768, intensity=200, raw_flags=""),
            StrongLine(element="O", wavelength=1130.238, intensity=200, raw_flags=""),
            StrongLine(element="O", wavelength=1246.402, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=1257.004, intensity=150, raw_flags=""),
            StrongLine(element="O", wavelength=1316.389, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=1316.485, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=1316.511, intensity=200, raw_flags=""),
            StrongLine(element="O", wavelength=1802.121, intensity=200, raw_flags=""),
            StrongLine(element="O", wavelength=1824.363, intensity=200, raw_flags=""),
        ],
        persistent_lines=[
            StrongLine(element="O", wavelength=42.9918, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=43.0041, intensity=700, raw_flags="P"),
            StrongLine(element="O", wavelength=43.0177, intensity=1000, raw_flags="P"),
            StrongLine(element="O", wavelength=53.78319, intensity=800, raw_flags="P"),
            StrongLine(element="O", wavelength=53.82636, intensity=1000, raw_flags="P"),
            StrongLine(element="O", wavelength=53.8318, intensity=600, raw_flags="P"),
            StrongLine(element="O", wavelength=53.90855, intensity=800, raw_flags="P"),
            StrongLine(element="O", wavelength=53.95489, intensity=700, raw_flags="P"),
            StrongLine(element="O", wavelength=53.98544, intensity=500, raw_flags="P"),
            StrongLine(element="O", wavelength=71.85036, intensity=800, raw_flags="P"),
            StrongLine(element="O", wavelength=71.85663, intensity=500, raw_flags="P"),
            StrongLine(element="O", wavelength=83.27587, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=83.33302, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=83.44655, intensity=500, raw_flags="P"),
            StrongLine(element="O", wavelength=130.2168, intensity=1000, raw_flags="P"),
            StrongLine(element="O", wavelength=130.4858, intensity=700, raw_flags="P"),
            StrongLine(element="O", wavelength=130.6029, intensity=300, raw_flags="P"),
            StrongLine(element="O", wavelength=313.4720, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=391.1957, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=397.32562, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=407.5862, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=441.4905, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=464.1810, intensity=300, raw_flags="P"),
            StrongLine(element="O", wavelength=464.9135, intensity=400, raw_flags="P"),
            StrongLine(element="O", wavelength=777.194, intensity=300, raw_flags="P"),
            StrongLine(element="O", wavelength=777.417, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=777.539, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=844.625, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=844.636, intensity=300, raw_flags="P"),
            StrongLine(element="O", wavelength=844.676, intensity=300, raw_flags="P"),
            StrongLine(element="O", wavelength=926.601, intensity=200, raw_flags="P"),
            StrongLine(element="O", wavelength=1316.389, intensity=250, raw_flags="P"),
            StrongLine(element="O", wavelength=1316.485, intensity=250, raw_flags="P"),
        ]
    ),
    "Xe": StrongLines(
        element="Xe",
        lines=[
            StrongLine(element="Xe", wavelength=74.041, intensity=120, raw_flags=""),
            StrongLine(element="Xe", wavelength=80.307, intensity=120, raw_flags=""),
            StrongLine(element="Xe", wavelength=88.080, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=88.554, intensity=120, raw_flags=""),
            StrongLine(element="Xe", wavelength=92.587, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=93.540, intensity=80, raw_flags=""),
            StrongLine(element="Xe", wavelength=97.277, intensity=250, raw_flags="P"),
            StrongLine(element="Xe", wavelength=97.668, intensity=250, raw_flags=""),
            StrongLine(element="Xe", wavelength=103.244, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=103.768, intensity=250, raw_flags=""),
            StrongLine(element="Xe", wavelength=104.131, intensity=400, raw_flags="P"),
            StrongLine(element="Xe", wavelength=104.827, intensity=300, raw_flags="P"),
            StrongLine(element="Xe", wavelength=105.192, intensity=400, raw_flags="P"),
            StrongLine(element="Xe", wavelength=107.448, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=108.386, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=110.043, intensity=400, raw_flags="P"),
            StrongLine(element="Xe", wavelength=115.847, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=116.963, intensity=80, raw_flags=""),
            StrongLine(element="Xe", wavelength=118.305, intensity=250, raw_flags="P"),
            StrongLine(element="Xe", wavelength=119.20376, intensity=25, raw_flags="P"),
            StrongLine(element="Xe", wavelength=124.476, intensity=200, raw_flags="P"),
            StrongLine(element="Xe", wavelength=125.02091, intensity=25, raw_flags="P"),
            StrongLine(element="Xe", wavelength=129.55878, intensity=100, raw_flags="P"),
            StrongLine(element="Xe", wavelength=146.96123, intensity=60, raw_flags="P"),
            StrongLine(element="Xe", wavelength=286.473, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=289.522, intensity=50, raw_flags="h"),
            StrongLine(element="Xe", wavelength=297.932, intensity=130, raw_flags=""),
            StrongLine(element="Xe", wavelength=301.743, intensity=30, raw_flags="h"),
            StrongLine(element="Xe", wavelength=312.187, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=336.672, intensity=70, raw_flags="h"),
            StrongLine(element="Xe", wavelength=346.126, intensity=30, raw_flags="h"),
            StrongLine(element="Xe", wavelength=350.674, intensity=1, raw_flags=""),
            StrongLine(element="Xe", wavelength=354.986, intensity=1, raw_flags=""),
            StrongLine(element="Xe", wavelength=355.404, intensity=1, raw_flags=""),
            StrongLine(element="Xe", wavelength=361.032, intensity=2, raw_flags=""),
            StrongLine(element="Xe", wavelength=361.306, intensity=1, raw_flags=""),
            StrongLine(element="Xe", wavelength=363.306, intensity=1, raw_flags=""),
            StrongLine(element="Xe", wavelength=366.991, intensity=1, raw_flags=""),
            StrongLine(element="Xe", wavelength=368.590, intensity=4, raw_flags=""),
            StrongLine(element="Xe", wavelength=369.349, intensity=4, raw_flags=""),
            StrongLine(element="Xe", wavelength=390.791, intensity=30, raw_flags="l"),
            StrongLine(element="Xe", wavelength=403.759, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=405.746, intensity=70, raw_flags="l"),
            StrongLine(element="Xe", wavelength=409.889, intensity=30, raw_flags="h"),
            StrongLine(element="Xe", wavelength=415.804, intensity=70, raw_flags="l"),
            StrongLine(element="Xe", wavelength=418.010, intensity=300, raw_flags="h"),
            StrongLine(element="Xe", wavelength=419.315, intensity=150, raw_flags="h"),
            StrongLine(element="Xe", wavelength=420.848, intensity=100, raw_flags="h"),
            StrongLine(element="Xe", wavelength=420.947, intensity=30, raw_flags="h"),
            StrongLine(element="Xe", wavelength=421.372, intensity=100, raw_flags="h"),
            StrongLine(element="Xe", wavelength=421.560, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=422.300, intensity=100, raw_flags="h"),
            StrongLine(element="Xe", wavelength=423.825, intensity=130, raw_flags="h"),
            StrongLine(element="Xe", wavelength=424.538, intensity=150, raw_flags="h"),
            StrongLine(element="Xe", wavelength=425.157, intensity=30, raw_flags="l"),
            StrongLine(element="Xe", wavelength=429.640, intensity=150, raw_flags="h"),
            StrongLine(element="Xe", wavelength=431.051, intensity=150, raw_flags="h"),
            StrongLine(element="Xe", wavelength=433.052, intensity=300, raw_flags="l"),
            StrongLine(element="Xe", wavelength=436.920, intensity=70, raw_flags="h"),
            StrongLine(element="Xe", wavelength=437.378, intensity=30, raw_flags="l"),
            StrongLine(element="Xe", wavelength=439.320, intensity=150, raw_flags="h"),
            StrongLine(element="Xe", wavelength=439.577, intensity=150, raw_flags="l"),
            StrongLine(element="Xe", wavelength=440.688, intensity=70, raw_flags="l"),
            StrongLine(element="Xe", wavelength=441.607, intensity=50, raw_flags="l"),
            StrongLine(element="Xe", wavelength=444.813, intensity=150, raw_flags="h"),
            StrongLine(element="Xe", wavelength=446.219, intensity=300, raw_flags="h"),
            StrongLine(element="Xe", wavelength=448.086, intensity=150, raw_flags="l"),
            StrongLine(element="Xe", wavelength=452.186, intensity=30, raw_flags="l"),
            StrongLine(element="Xe", wavelength=473.4152, intensity=60, raw_flags=""),
            StrongLine(element="Xe", wavelength=479.2619, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=480.702, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=482.971, intensity=40, raw_flags=""),
            StrongLine(element="Xe", wavelength=484.329, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=484.433, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=491.651, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=492.3152, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=497.171, intensity=70, raw_flags="l"),
            StrongLine(element="Xe", wavelength=497.271, intensity=130, raw_flags=""),
            StrongLine(element="Xe", wavelength=498.877, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=499.117, intensity=30, raw_flags="l"),
            StrongLine(element="Xe", wavelength=502.8280, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=504.492, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=508.062, intensity=300, raw_flags=""),
            StrongLine(element="Xe", wavelength=512.242, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=512.570, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=517.882, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=518.804, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=519.137, intensity=130, raw_flags=""),
            StrongLine(element="Xe", wavelength=519.210, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=526.044, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=526.195, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=529.222, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=530.927, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=531.387, intensity=300, raw_flags=""),
            StrongLine(element="Xe", wavelength=533.933, intensity=700, raw_flags=""),
            StrongLine(element="Xe", wavelength=536.320, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=536.807, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=537.239, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=539.280, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=541.915, intensity=1000, raw_flags="P"),
            StrongLine(element="Xe", wavelength=543.896, intensity=250, raw_flags=""),
            StrongLine(element="Xe", wavelength=544.545, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=545.045, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=546.039, intensity=130, raw_flags=""),
            StrongLine(element="Xe", wavelength=547.261, intensity=300, raw_flags=""),
            StrongLine(element="Xe", wavelength=549.486, intensity=30, raw_flags="l"),
            StrongLine(element="Xe", wavelength=552.553, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=553.107, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=556.662, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=561.667, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=565.938, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=566.756, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=567.091, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=569.575, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=569.961, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=571.610, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=572.691, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=575.103, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=575.865, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=577.639, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=581.596, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=582.389, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=582.480, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=587.502, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=589.329, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=589.499, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=590.513, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=593.417, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=594.553, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=597.113, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=597.646, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=600.892, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=603.620, intensity=300, raw_flags=""),
            StrongLine(element="Xe", wavelength=605.115, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=609.350, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=609.759, intensity=500, raw_flags="P"),
            StrongLine(element="Xe", wavelength=610.143, intensity=130, raw_flags=""),
            StrongLine(element="Xe", wavelength=611.508, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=614.645, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=617.830, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=617.966, intensity=12, raw_flags=""),
            StrongLine(element="Xe", wavelength=618.242, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=619.407, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=619.826, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=627.082, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=627.754, intensity=130, raw_flags=""),
            StrongLine(element="Xe", wavelength=628.441, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=628.601, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=630.086, intensity=80, raw_flags=""),
            StrongLine(element="Xe", wavelength=631.806, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=634.396, intensity=130, raw_flags=""),
            StrongLine(element="Xe", wavelength=635.635, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=637.528, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=639.799, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=646.970, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=647.284, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=648.776, intensity=12, raw_flags=""),
            StrongLine(element="Xe", wavelength=649.872, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=650.418, intensity=20, raw_flags="h"),
            StrongLine(element="Xe", wavelength=651.283, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=652.865, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=653.316, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=659.501, intensity=300, raw_flags=""),
            StrongLine(element="Xe", wavelength=659.556, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=659.725, intensity=130, raw_flags=""),
            StrongLine(element="Xe", wavelength=659.884, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=666.892, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=669.432, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=672.801, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=678.871, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=679.037, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=680.574, intensity=300, raw_flags=""),
            StrongLine(element="Xe", wavelength=682.732, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=687.211, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=688.216, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=691.022, intensity=25, raw_flags=""),
            StrongLine(element="Xe", wavelength=692.553, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=694.211, intensity=250, raw_flags="h"),
            StrongLine(element="Xe", wavelength=697.618, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=699.088, intensity=700, raw_flags=""),
            StrongLine(element="Xe", wavelength=708.215, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=711.960, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=714.750, intensity=15, raw_flags="s"),
            StrongLine(element="Xe", wavelength=714.903, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=716.483, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=728.434, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=730.180, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=733.930, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=738.600, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=739.379, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=754.845, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=758.468, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=761.857, intensity=25, raw_flags=""),
            StrongLine(element="Xe", wavelength=764.202, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=764.391, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=767.066, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=778.704, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=780.265, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=788.132, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=788.740, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=796.734, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=802.967, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=805.726, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=806.134, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=810.198, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=815.180, intensity=50, raw_flags="h"),
            StrongLine(element="Xe", wavelength=817.102, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=820.634, intensity=70, raw_flags=""),
            StrongLine(element="Xe", wavelength=823.1635, intensity=1000, raw_flags="P"),
            StrongLine(element="Xe", wavelength=826.652, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=828.0117, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=834.6823, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=834.724, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=840.9191, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=851.519, intensity=15, raw_flags="h"),
            StrongLine(element="Xe", wavelength=857.601, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=860.423, intensity=15, raw_flags="h"),
            StrongLine(element="Xe", wavelength=864.854, intensity=25, raw_flags=""),
            StrongLine(element="Xe", wavelength=869.220, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=869.686, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=871.619, intensity=15, raw_flags="h"),
            StrongLine(element="Xe", wavelength=873.939, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=875.820, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=881.9411, intensity=500, raw_flags="P"),
            StrongLine(element="Xe", wavelength=886.232, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=890.873, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=893.083, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=895.2252, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=898.105, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=898.757, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=904.545, intensity=40, raw_flags=""),
            StrongLine(element="Xe", wavelength=916.265, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=916.752, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=937.476, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=951.338, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=959.135, intensity=15, raw_flags="h"),
            StrongLine(element="Xe", wavelength=968.532, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=969.868, intensity=15, raw_flags="l"),
            StrongLine(element="Xe", wavelength=971.816, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=979.9700, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=992.3190, intensity=300, raw_flags="P"),
            StrongLine(element="Xe", wavelength=1083.837, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=1174.201, intensity=9, raw_flags=""),
            StrongLine(element="Xe", wavelength=1223.524, intensity=40, raw_flags=""),
            StrongLine(element="Xe", wavelength=1225.776, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=1259.020, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=1262.3399, intensity=250, raw_flags="P"),
            StrongLine(element="Xe", wavelength=1354.415, intensity=25, raw_flags=""),
            StrongLine(element="Xe", wavelength=1365.706, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=1414.244, intensity=120, raw_flags=""),
            StrongLine(element="Xe", wavelength=1424.096, intensity=80, raw_flags=""),
            StrongLine(element="Xe", wavelength=1436.499, intensity=40, raw_flags=""),
            StrongLine(element="Xe", wavelength=1466.081, intensity=14, raw_flags=""),
            StrongLine(element="Xe", wavelength=1473.2816, intensity=300, raw_flags="P"),
            StrongLine(element="Xe", wavelength=1509.972, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=1541.839, intensity=250, raw_flags=""),
            StrongLine(element="Xe", wavelength=1555.713, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=1597.954, intensity=25, raw_flags=""),
            StrongLine(element="Xe", wavelength=1603.990, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=1605.328, intensity=100, raw_flags=""),
            StrongLine(element="Xe", wavelength=1655.449, intensity=12, raw_flags=""),
            StrongLine(element="Xe", wavelength=1672.815, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=1732.577, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=1878.813, intensity=40, raw_flags=""),
            StrongLine(element="Xe", wavelength=2018.719, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=2026.224, intensity=300, raw_flags=""),
            StrongLine(element="Xe", wavelength=2147.009, intensity=25, raw_flags=""),
            StrongLine(element="Xe", wavelength=2319.333, intensity=120, raw_flags=""),
            StrongLine(element="Xe", wavelength=2327.954, intensity=11, raw_flags=""),
            StrongLine(element="Xe", wavelength=2482.471, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=2514.584, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=2626.908, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=2651.086, intensity=250, raw_flags=""),
            StrongLine(element="Xe", wavelength=2838.154, intensity=25, raw_flags=""),
            StrongLine(element="Xe", wavelength=2858.225, intensity=80, raw_flags=""),
            StrongLine(element="Xe", wavelength=2938.441, intensity=30, raw_flags=""),
            StrongLine(element="Xe", wavelength=2944.806, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=2964.958, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=2981.362, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=3025.314, intensity=60, raw_flags=""),
            StrongLine(element="Xe", wavelength=3047.546, intensity=150, raw_flags=""),
            StrongLine(element="Xe", wavelength=3050.412, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=3079.418, intensity=50, raw_flags=""),
            StrongLine(element="Xe", wavelength=3106.923, intensity=600, raw_flags="P"),
            StrongLine(element="Xe", wavelength=3133.601, intensity=12, raw_flags=""),
            StrongLine(element="Xe", wavelength=3160.791, intensity=60, raw_flags=""),
            StrongLine(element="Xe", wavelength=3229.308, intensity=10, raw_flags=""),
            StrongLine(element="Xe", wavelength=3273.926, intensity=200, raw_flags=""),
            StrongLine(element="Xe", wavelength=3366.669, intensity=400, raw_flags=""),
            StrongLine(element="Xe", wavelength=3401.467, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=3433.527, intensity=40, raw_flags=""),
            StrongLine(element="Xe", wavelength=3474.400, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=3507.026, intensity=500, raw_flags="P"),
            StrongLine(element="Xe", wavelength=3524.692, intensity=11, raw_flags=""),
            StrongLine(element="Xe", wavelength=3620.921, intensity=25, raw_flags=""),
            StrongLine(element="Xe", wavelength=3623.174, intensity=15, raw_flags=""),
            StrongLine(element="Xe", wavelength=3650.836, intensity=40, raw_flags=""),
            StrongLine(element="Xe", wavelength=3678.883, intensity=80, raw_flags=""),
            StrongLine(element="Xe", wavelength=3868.598, intensity=14, raw_flags=""),
            StrongLine(element="Xe", wavelength=3873.782, intensity=20, raw_flags=""),
            StrongLine(element="Xe", wavelength=3893.960, intensity=25, raw_flags=""),
            StrongLine(element="Xe", wavelength=3995.514, intensity=12, raw_flags=""),
        ],
        persistent_lines=[
            StrongLine(element="Xe", wavelength=97.277, intensity=250, raw_flags="P"),
            StrongLine(element="Xe", wavelength=104.131, intensity=400, raw_flags="P"),
            StrongLine(element="Xe", wavelength=104.827, intensity=300, raw_flags="P"),
            StrongLine(element="Xe", wavelength=105.192, intensity=400, raw_flags="P"),
            StrongLine(element="Xe", wavelength=107.448, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=110.043, intensity=400, raw_flags="P"),
            StrongLine(element="Xe", wavelength=118.305, intensity=250, raw_flags="P"),
            StrongLine(element="Xe", wavelength=119.20376, intensity=25, raw_flags="P"),
            StrongLine(element="Xe", wavelength=124.476, intensity=200, raw_flags="P"),
            StrongLine(element="Xe", wavelength=125.02091, intensity=25, raw_flags="P"),
            StrongLine(element="Xe", wavelength=129.55878, intensity=100, raw_flags="P"),
            StrongLine(element="Xe", wavelength=146.96123, intensity=60, raw_flags="P"),
            StrongLine(element="Xe", wavelength=484.433, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=529.222, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=541.915, intensity=1000, raw_flags="P"),
            StrongLine(element="Xe", wavelength=597.646, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=605.115, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=609.759, intensity=500, raw_flags="P"),
            StrongLine(element="Xe", wavelength=823.1635, intensity=1000, raw_flags="P"),
            StrongLine(element="Xe", wavelength=828.0117, intensity=700, raw_flags="P"),
            StrongLine(element="Xe", wavelength=881.9411, intensity=500, raw_flags="P"),
            StrongLine(element="Xe", wavelength=992.3190, intensity=300, raw_flags="P"),
            StrongLine(element="Xe", wavelength=1262.3399, intensity=250, raw_flags="P"),
            StrongLine(element="Xe", wavelength=1473.2816, intensity=300, raw_flags="P"),
            StrongLine(element="Xe", wavelength=3106.923, intensity=600, raw_flags="P"),
            StrongLine(element="Xe", wavelength=3507.026, intensity=500, raw_flags="P"),
        ]
    ),
}
