> Python SDK for AdsOverAI - Native advertising for AI chat interfaces

[![PyPI version](https://badge.fury.io/py/adsoverai.svg)](https://badge.fury.io/py/adsoverai)
[![Python Support](https://img.shields.io/pypi/pyversions/adsoverai.svg)](https://pypi.org/project/adsoverai/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

## Features

- 🎯 **Context-Aware Ads** - Display relevant ads based on AI conversation context
- ⚡ **Fast & Lightweight** - Optimized for performance with built-in caching
- 🔒 **Type Safe** - Full type hints and validation
- 🛡️ **Error Handling** - Comprehensive exception handling
- 🌙 **Theme Support** - Light, dark, and auto theme modes
- 📝 **Logging** - Debug mode with detailed logging

## Installation

```bash
pip install adsoverai
```

## Quick Start

```python
from adsoverai import AdsOverAI

# Initialize the client
client = AdsOverAI(
    api_key="your-api-key",
    max_ads=3,
    theme="auto"
)

# Fetch ads based on conversation context
ads = client.get_ads(
    query="What are the best running shoes?",
    response="Here are some great running shoe options..."
)

# Display ads
for ad in ads:
    print(f"Title: {ad.title}")
    print(f"Description: {ad.description}")
    print(f"URL: {ad.url}")
    print("---")
```

## Convenience Function

```python
from adsoverai import get_ads

ads = get_ads(
    api_key="your-api-key",
    query="best headphones",
    response="Here are top-rated headphones...",
    max_ads=2
)
```

## API Reference

### `AdsOverAI`

Main client class for the SDK.

#### Parameters

- `api_key` (str, required): Your AdsOverAI API key
- `max_ads` (int, default=3): Maximum number of ads to return (1-10)
- `theme` (str, default="auto"): Theme preference ("light", "dark", "auto")
- `api_url` (str): API endpoint URL
- `debug_mode` (bool, default=False): Enable debug logging
- `timeout` (int, default=10): Request timeout in seconds
- `cache_ttl` (int, default=300): Cache time-to-live in seconds

#### Methods

**`get_ads(query, response, max_ads=None, use_cache=True)`**

Fetch contextual ads based on conversation.

**`clear_cache()`**

Clear all cached results.

**`update_config(max_ads=None, theme=None, debug_mode=None)`**

Update client configuration.

**`close()`**

Close the session and cleanup resources.

### `AdResult`

Data class representing an ad result.

#### Attributes

- `title` (str): Ad title
- `description` (str): Ad description
- `url` (str): Ad URL
- `image_url` (str, optional): Ad image URL
- `advertiser` (str, optional): Advertiser name
- `cta_text` (str): Call-to-action text
- `metadata` (dict): Additional metadata

## Advanced Usage

### Context Manager

```python
with AdsOverAI(api_key="your-key") as client:
    ads = client.get_ads(query="...", response="...")
    for ad in ads:
        print(ad.title)
# Session automatically closed
```

### Error Handling

```python
from adsoverai import AdsOverAI, AuthenticationError, APIError

try:
    client = AdsOverAI(api_key="your-key")
    ads = client.get_ads(query="...", response="...")
except AuthenticationError:
    print("Invalid API key")
except APIError as e:
    print(f"API error: {e}")
```

### Debug Mode

```python
client = AdsOverAI(
    api_key="your-key",
    debug_mode=True  # Enable detailed logging
)
```

### Cache Control

```python
# Disable cache for a specific request
ads = client.get_ads(
    query="...",
    response="...",
    use_cache=False
)

# Clear all cached results
client.clear_cache()
```

## Examples

See the `examples/` directory for more usage examples:

- `basic_usage.py` - Simple usage example
- `advanced_usage.py` - Advanced features
- `django_integration.py` - Django integration example

## Development

### Setup

```bash
git clone https://github.com/adsoverai/python-sdk.git
cd python-sdk
pip install -e ".[dev]"
```

### Run Tests

```bash
pytest
```

### Code Formatting

```bash
black adsoverai/ tests/
flake8 adsoverai/ tests/
mypy adsoverai/
```

## Support

- 📧 Email: hello@adsoverai.com
- 🌐 Website: https://www.adsoverai.com
- 📚 Documentation: https://docs.adsoverai.com
- 🐛 Issues: https://github.com/adsoverai/python-sdk/issues

## License

MIT © AdsOverAI Team