"""Main client for AdsOverAI SDK"""

from typing import Optional, List, Dict, Literal
import requests
import logging
import time

from .models import AdResult
from .exceptions import (
    AuthenticationError,
    ValidationError,
    APIError,
    RateLimitError,
    NetworkError
)
from .utils import generate_cache_key

__version__ = "1.0.0"

ThemeType = Literal["light", "dark", "auto"]


class AdsOverAI:
    """
    AdsOverAI Python SDK Client
    
    Provides context-aware advertising for AI chat interfaces.
    
    Args:
        api_key: Your AdsOverAI API key (required)
        max_ads: Maximum number of ads to return (default: 3)
        theme: Theme preference - 'light', 'dark', or 'auto' (default: 'auto')
        api_url: API endpoint URL (default: 'https://www.adsoverai.com')
        debug_mode: Enable debug logging (default: False)
        timeout: Request timeout in seconds (default: 10)
        cache_ttl: Cache time-to-live in seconds (default: 300)
    
    Example:
        >>> client = AdsOverAI(api_key="your-key", max_ads=2)
        >>> ads = client.get_ads(
        ...     query="best laptops for programming",
        ...     response="Here are the top laptops..."
        ... )
        >>> for ad in ads:
        ...     print(ad.title)
    """
    
    def __init__(
        self,
        api_key: str,
        max_ads: int = 3,
        theme: ThemeType = "auto",
        api_url: str = "https://www.adsoverai.com",
        debug_mode: bool = False,
        timeout: int = 10,
        cache_ttl: int = 300
    ):
        if not api_key:
            raise AuthenticationError("API key is required")
        
        if max_ads < 1 or max_ads > 10:
            raise ValidationError("max_ads must be between 1 and 10")
        
        self.api_key = api_key
        self.max_ads = max_ads
        self.theme = theme
        self.api_url = api_url.rstrip('/')
        self.debug_mode = debug_mode
        self.timeout = timeout
        self.cache_ttl = cache_ttl
        
        # Setup logging
        self.logger = logging.getLogger(__name__)
        if debug_mode:
            logging.basicConfig(
                level=logging.DEBUG,
                format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
            )
            self.logger.setLevel(logging.DEBUG)
        
        # Session for connection pooling
        self.session = requests.Session()
        self.session.headers.update({
            'Authorization': f'Bearer {api_key}',
            'Content-Type': 'application/json',
            'User-Agent': f'adsoverai-python/{__version__}'
        })
        
        # Simple cache
        self._cache: Dict[str, tuple[List[AdResult], float]] = {}
        
        self._log_debug(f"AdsOverAI client initialized with max_ads={max_ads}, theme={theme}")
    
    def _log_debug(self, message: str):
        """Log debug message if debug mode is enabled"""
        if self.debug_mode:
            self.logger.debug(message)
    
    def _get_from_cache(self, cache_key: str) -> Optional[List[AdResult]]:
        """Get cached results if still valid"""
        if cache_key in self._cache:
            results, timestamp = self._cache[cache_key]
            if time.time() - timestamp < self.cache_ttl:
                self._log_debug(f"Cache hit for key: {cache_key}")
                return results
            else:
                del self._cache[cache_key]
        return None
    
    def _set_cache(self, cache_key: str, results: List[AdResult]):
        """Store results in cache"""
        self._cache[cache_key] = (results, time.time())
        self._log_debug(f"Cached {len(results)} results")
    
    def get_ads(
        self,
        query: str,
        response: str,
        max_ads: Optional[int] = None,
        use_cache: bool = True
    ) -> List[AdResult]:
        """
        Fetch contextual ads based on AI conversation
        
        Args:
            query: The user's query/question
            response: The AI's response
            max_ads: Override default max_ads for this request
            use_cache: Whether to use cached results (default: True)
        
        Returns:
            List of AdResult objects
        
        Raises:
            ValidationError: If query or response is empty
            AuthenticationError: If API key is invalid
            APIError: If API request fails
        """
        if not query or not query.strip():
            raise ValidationError("query cannot be empty")
        
        if not response or not response.strip():
            raise ValidationError("response cannot be empty")
        
        max_ads = max_ads or self.max_ads
        
        # Check cache
        if use_cache:
            cache_key = generate_cache_key(query, response, max_ads)
            cached_results = self._get_from_cache(cache_key)
            if cached_results is not None:
                return cached_results[:max_ads]
        
        # Make API request
        endpoint = f"{self.api_url}/api/ads"
        payload = {
            'query': query,
            'response': response,
            'max_ads': max_ads,
            'theme': self.theme
        }
        
        self._log_debug(f"Fetching ads from {endpoint}")
        
        try:
            response_obj = self.session.post(
                endpoint,
                json=payload,
                timeout=self.timeout
            )
            
            if response_obj.status_code == 401:
                raise AuthenticationError("Invalid API key")
            
            if response_obj.status_code == 429:
                raise RateLimitError("Rate limit exceeded")
            
            response_obj.raise_for_status()
            
            data = response_obj.json()
            ads_data = data.get('ads', [])
            
            results = [AdResult.from_dict(ad) for ad in ads_data]
            
            self._log_debug(f"Received {len(results)} ads from API")
            
            if use_cache:
                cache_key = generate_cache_key(query, response, max_ads)
                self._set_cache(cache_key, results)
            
            return results[:max_ads]
            
        except requests.exceptions.Timeout:
            raise NetworkError(f"Request timed out after {self.timeout} seconds")
        except requests.exceptions.ConnectionError:
            raise NetworkError("Failed to connect to API")
        except requests.exceptions.RequestException as e:
            raise APIError(f"API request failed: {str(e)}")
    
    def clear_cache(self):
        """Clear all cached results"""
        self._cache.clear()
        self._log_debug("Cache cleared")
    
    def update_config(
        self,
        max_ads: Optional[int] = None,
        theme: Optional[ThemeType] = None,
        debug_mode: Optional[bool] = None
    ):
        """Update client configuration"""
        if max_ads is not None:
            if max_ads < 1 or max_ads > 10:
                raise ValidationError("max_ads must be between 1 and 10")
            self.max_ads = max_ads
        
        if theme is not None:
            self.theme = theme
        
        if debug_mode is not None:
            self.debug_mode = debug_mode
            if debug_mode:
                self.logger.setLevel(logging.DEBUG)
            else:
                self.logger.setLevel(logging.INFO)
    
    def close(self):
        """Close the session and cleanup resources"""
        self.session.close()
        self._log_debug("Session closed")
    
    def __enter__(self):
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()


def get_ads(
    api_key: str,
    query: str,
    response: str,
    max_ads: int = 3,
    theme: ThemeType = "auto"
) -> List[AdResult]:
    """
    Convenience function to fetch ads without creating a client instance
    """
    with AdsOverAI(api_key=api_key, max_ads=max_ads, theme=theme) as client:
        return client.get_ads(query, response)