"""Data models for AdsOverAI SDK"""

from typing import Optional, Dict, Any
from dataclasses import dataclass, field


@dataclass
class AdResult:
    """Represents a single ad result"""
    title: str
    description: str
    url: str
    image_url: Optional[str] = None
    advertiser: Optional[str] = None
    cta_text: Optional[str] = "Learn More"
    metadata: Dict[str, Any] = field(default_factory=dict)

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'AdResult':
        """Create AdResult from API response dictionary"""
        return cls(
            title=data.get('title', ''),
            description=data.get('description', ''),
            url=data.get('url', ''),
            image_url=data.get('image_url'),
            advertiser=data.get('advertiser'),
            cta_text=data.get('cta_text', 'Learn More'),
            metadata=data.get('metadata', {})
        )

    def to_dict(self) -> Dict[str, Any]:
        """Convert AdResult to dictionary"""
        return {
            'title': self.title,
            'description': self.description,
            'url': self.url,
            'image_url': self.image_url,
            'advertiser': self.advertiser,
            'cta_text': self.cta_text,
            'metadata': self.metadata
        }

    def __repr__(self) -> str:
        return f"AdResult(title='{self.title}', url='{self.url}')"