"""Advanced usage example with error handling and caching"""

from adsoverai import (
    AdsOverAI,
    AuthenticationError,
    ValidationError,
    APIError
)

def main():
    try:
        # Use context manager
        with AdsOverAI(
            api_key="your-api-key-here",
            max_ads=5,
            theme="dark",
            debug_mode=True,
            cache_ttl=600  # 10 minutes cache
        ) as client:
            
            # First request (cached)
            print("First request (will be cached):")
            ads1 = client.get_ads(
                query="best laptops for programming",
                response="Top laptops for developers include..."
            )
            print(f"Received {len(ads1)} ads\n")
            
            # Second request (from cache)
            print("Second request (from cache):")
            ads2 = client.get_ads(
                query="best laptops for programming",
                response="Top laptops for developers include..."
            )
            print(f"Received {len(ads2)} ads (cached)\n")
            
            # Request without cache
            print("Third request (bypassing cache):")
            ads3 = client.get_ads(
                query="best laptops for programming",
                response="Top laptops for developers include...",
                use_cache=False
            )
            print(f"Received {len(ads3)} ads (fresh)\n")
            
            # Update configuration
            client.update_config(max_ads=2, theme="light")
            print("Configuration updated\n")
            
            # Clear cache
            client.clear_cache()
            print("Cache cleared\n")
            
    except AuthenticationError:
        print("Error: Invalid API key")
    except ValidationError as e:
        print(f"Validation error: {e}")
    except APIError as e:
        print(f"API error: {e}")
    except Exception as e:
        print(f"Unexpected error: {e}")

if __name__ == "__main__":
    main()