"""Tests for the AdsOverAI client"""

import pytest
from unittest.mock import Mock, patch
from adsoverai import AdsOverAI
from adsoverai.exceptions import (
    AuthenticationError,
    ValidationError,
    APIError,
    NetworkError
)


def test_client_initialization(api_key):
    """Test client initialization"""
    client = AdsOverAI(api_key=api_key, max_ads=5)
    assert client.api_key == api_key
    assert client.max_ads == 5
    assert client.theme == "auto"


def test_client_requires_api_key():
    """Test that API key is required"""
    with pytest.raises(AuthenticationError):
        AdsOverAI(api_key="")


def test_client_validates_max_ads(api_key):
    """Test max_ads validation"""
    with pytest.raises(ValidationError):
        AdsOverAI(api_key=api_key, max_ads=0)
    
    with pytest.raises(ValidationError):
        AdsOverAI(api_key=api_key, max_ads=11)


def test_get_ads_validates_input(client):
    """Test input validation for get_ads"""
    with pytest.raises(ValidationError):
        client.get_ads(query="", response="test")
    
    with pytest.raises(ValidationError):
        client.get_ads(query="test", response="")


@patch('adsoverai.client.requests.Session.post')
def test_get_ads_success(mock_post, client, mock_api_response):
    """Test successful ad fetching"""
    mock_response = Mock()
    mock_response.status_code = 200
    mock_response.json.return_value = mock_api_response
    mock_post.return_value = mock_response
    
    ads = client.get_ads(
        query="test query",
        response="test response"
    )
    
    assert len(ads) == 2
    assert ads[0].title == "Product 1"
    assert ads[1].title == "Product 2"


@patch('adsoverai.client.requests.Session.post')
def test_get_ads_authentication_error(mock_post, client):
    """Test authentication error handling"""
    mock_response = Mock()
    mock_response.status_code = 401
    mock_post.return_value = mock_response
    
    with pytest.raises(AuthenticationError):
        client.get_ads(query="test", response="test")


@patch('adsoverai.client.requests.Session.post')
def test_get_ads_uses_cache(mock_post, client, mock_api_response):
    """Test that caching works"""
    mock_response = Mock()
    mock_response.status_code = 200
    mock_response.json.return_value = mock_api_response
    mock_post.return_value = mock_response
    
    # First call
    ads1 = client.get_ads(query="test", response="test")
    
    # Second call (should use cache)
    ads2 = client.get_ads(query="test", response="test")
    
    # API should only be called once
    assert mock_post.call_count == 1
    assert len(ads1) == len(ads2)


def test_cache_clear(client):
    """Test cache clearing"""
    client._cache = {"test_key": ([], 0)}
    client.clear_cache()
    assert len(client._cache) == 0


def test_context_manager(api_key):
    """Test context manager usage"""
    with AdsOverAI(api_key=api_key) as client:
        assert client.api_key == api_key
    
    # Session should be closed after context
    assert client.session is not None


def test_update_config(client):
    """Test configuration updates"""
    client.update_config(max_ads=5, theme="dark")
    assert client.max_ads == 5
    assert client.theme == "dark"


def test_update_config_validates(client):
    """Test configuration validation"""
    with pytest.raises(ValidationError):
        client.update_config(max_ads=15)