This file is a merged representation of the entire codebase, combined into a single document by Repomix.

# File Summary

## Purpose
This file contains a packed representation of the entire repository's contents.
It is designed to be easily consumable by AI systems for analysis, code review,
or other automated processes.

## File Format
The content is organized as follows:
1. This summary section
2. Repository information
3. Directory structure
4. Repository files (if enabled)
5. Multiple file entries, each consisting of:
  a. A header with the file path (## File: path/to/file)
  b. The full contents of the file in a code block

## Usage Guidelines
- This file should be treated as read-only. Any changes should be made to the
  original repository files, not this packed version.
- When processing this file, use the file path to distinguish
  between different files in the repository.
- Be aware that this file may contain sensitive information. Handle it with
  the same level of security as you would the original repository.

## Notes
- Some files may have been excluded based on .gitignore rules and Repomix's configuration
- Binary files are not included in this packed representation. Please refer to the Repository Structure section for a complete list of file paths, including binary files
- Files matching patterns in .gitignore are excluded
- Files matching default ignore patterns are excluded
- Files are sorted by Git change count (files with more changes are at the bottom)

# Directory Structure
```
.github/
  workflows/
    publish.yml
docs/
  source/
    client.rst
    conf.py
    getting_started.rst
    index.rst
    parsers.rst
  requirements.txt
tests/
  client/
    test_cellar.py
    test_client.py
    test_normattiva.py
  metadata/
    query_results/
      ai_act.json
      query_results.json
      results_formex.json
    schemas/
      akomantoso30.xsd
      xml.xsd
  parsers/
    html/
      test_cellar_html.py
      test_html.py
    xml/
      test_akomantoso.py
      test_formex.py
    test_parser.py
tulit/
  client/
    queries/
      formex_query.rq
      html_query.rq
    __init__.py
    boe.py
    cellar.py
    client.py
    legifrance.py
    legilux.py
    normattiva.py
    veneto.py
  parsers/
    html/
      cellar.py
      veneto.py
      xhtml.py
    xml/
      assets/
        import/
          xml.xsd
        akomantoso30.xsd
        eli.owl
        formex4.xsd
        subdivisions.xml
        xml.xsd
      akomantoso.py
      formex.py
      xml.py
    __init__.py
    parser.py
  queries/
    formex_query.rq
    html_query.rq
  __init__.py
  main.py
  rdf.py
.gitignore
.readthedocs.yaml
LICENSE
MANIFEST.in
pyproject.toml
README.md
```

# Files

## File: .github/workflows/publish.yml
````yaml
name: Publish Package to PyPI
on: push

jobs:
  build:
    name: Build distribution 📦
    if: startsWith(github.ref, 'refs/tags/')  # only build on tag pushes
    runs-on: ubuntu-latest

    steps:
    - uses: actions/checkout@v4
    - name: Set up Python
      uses: actions/setup-python@v5
      with:
        python-version: "3.x"
    - name: Install Poetry
      run: curl -sSL https://install.python-poetry.org | python3 -
    - name: Configure Poetry
      run: poetry config virtualenvs.create false
    - name: Install dependencies
      run: poetry install --no-interaction --no-ansi
    - name: Build the package
      run: poetry build
    - name: Store the distribution packages
      uses: actions/upload-artifact@v4
      with:
        name: python-package-distributions
        path: dist/
  publish-to-pypi:
    name: Publish package to PyPI
    if: startsWith(github.ref, 'refs/tags/')  # only publish to PyPI on tag pushes
    needs:
    - build
    runs-on: ubuntu-latest
    environment:
      name: pypi
      url: https://pypi.org/p/ulit-python  # Replace <package-name> with your PyPI project name
    permissions:
      id-token: write  # IMPORTANT: mandatory for trusted publishing
    steps:
    - name: Download all the dists
      uses: actions/download-artifact@v4
      with:
        name: python-package-distributions
        path: dist/
    - name: Publish package to PyPI
      env:
        POETRY_PYPI_TOKEN_PYPI: ${{ secrets.PYPI_API_TOKEN }}
      run: poetry publish --username __token__ --password $POETRY_PYPI_TOKEN_PYPI
````

## File: docs/source/client.rst
````
Client
===============

The tulit Client library currently supports only the Cellar endpoint.

Some draft implementation were provided for Normattiva, LegiLux, and the Veneto Region, but as machine-readable endpoint are not sufficiently supported, they are to be considered only as provisional solutions, until such endpoints can be provided.

.. toctree::
    :maxdepth: 2

.. automodule:: tulit.client.client
     :members:
     :undoc-members:
     :show-inheritance:

.. automodule:: tulit.client.cellar
     :members:
     :undoc-members:
     :show-inheritance:

.. automodule:: tulit.client.normattiva
     :members:
     :undoc-members:
     :show-inheritance:

.. automodule:: tulit.client.legilux
     :members:
     :undoc-members:
     :show-inheritance:

.. automodule:: tulit.client.veneto
     :members:
     :undoc-members:
     :show-inheritance:
````

## File: docs/source/conf.py
````python
# Configuration file for the Sphinx documentation builder.

import os
import sys

# -- Project information

project = 'tulit'
author = 'AlessioNar'

release = '0.2.11'
version = '0.2.11'

# -- General configuration
sys.path.insert(0, os.path.abspath('../../tulit'))

extensions = [
    'sphinx.ext.duration',
    'sphinx.ext.doctest',
    'sphinx.ext.autodoc',
    'sphinx.ext.autosummary',
    'sphinx.ext.intersphinx',
    'sphinx.ext.napoleon',  # Ensure napoleon extension is included for Google style docstrings
    'sphinx.ext.inheritance_diagram',

]

intersphinx_mapping = {
    'python': ('https://docs.python.org/3/', None),
    'sphinx': ('https://www.sphinx-doc.org/en/master/', None),
}
intersphinx_disabled_domains = ['std']

templates_path = ['_templates']

# -- Options for HTML output
autodocs_mock_imports = ['tulit']
autodoc_member_order = 'bysource'

html_theme = 'sphinx_rtd_theme'

# -- Options for EPUB output
epub_show_urls = 'footnote'

# -- Napoleon settings
napoleon_google_docstring = True
napoleon_numpy_docstring = True
napoleon_include_init_with_doc = True
napoleon_include_private_with_doc = False
napoleon_include_special_with_doc = True
napoleon_use_admonition_for_examples = False
napoleon_use_admonition_for_notes = False
napoleon_use_admonition_for_references = False
napoleon_use_ivar = False
napoleon_use_param = True
napoleon_use_rtype = True
````

## File: docs/source/getting_started.rst
````
Getting Started
===============

tulit is hosted on PyPi, and documentation is published on readthedocs.io.

Installation
------------

To use tulit, first install it using poetry:

.. code-block:: console

    $ poetry shell
    $ poetry install

This will install the package and its dependencies.

Alternatively, you can install the package using pip:

.. code-block:: console

    $ pip install tulit


Basic usage
-----------

The `tulit` package main components are:
* a client to query and retrieve data from a variety of legal data sources. Currently the package supports the Cellar, LegiLux and Normattiva.
* a parser to convert legal documents from a variety of formats to a json representation.

Retrieving legal documents
---------------------------

The `tulit` package provides a client to query and retrieve data from a variety of legal data sources. The following code snippet shows how to use the `tulit` package to retrieve a legal document from Cellar, given its CELEX number:

.. code-block:: python

    from tulit.client.cellar import CellarClient
    
    client = CellarClient(download_dir='./database', log_dir='./logs')

    file_format = 'fmx4' # Or xhtml
    celex = "32024R0903"

    documents = client.download(celex=celex, format=file_format)

    # Location of the documents
    print(documents)

Parsing legal documents
-----------------------

The `tulit` parsers support exclusively legislative documents which were adopted in the following formats:
* Akoma Ntoso 3.0
* FORMEX 4
* XHTML originated from Cellar

The following code snippet shows how to use the `tulit` package to parse a legal document in Akoma Ntoso format:

.. code-block:: python

    from tulit.parsers.xml.akomantoso import AkomaNtosoParser

    parser = AkomaNtosoParser()
    
    file_to_parse = 'tests/data/akn/eu/32014L0092.akn'
    parser.parse(file_to_parse)

A similar approach can be used to parse a legal document in FORMEX and XHTML format:

.. code-block:: python

    from tulit.parsers.xml.formex import FormexParser

    formex_file = 'tests/data/formex/c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006.02/DOC_1/L_202400903EN.000101.fmx.xml'
    parser = FormexParser()

    parser.parse(formex_file)

    from tulit.parsers.html.xhtml import HTMLParser

    html_file = 'tests/data/html/c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006.03/DOC_1.html'

    parser = HTMLParser()
    parser.parse(html_file)


After parsing the document, the various attributes of the parser can be accessed as follows:

.. code-block:: python
    
    print(parser.preface)
    print(parser.citations)
    print(parser.recitals)
    print(parser.chapters)
    print(parser.articles)


`tulit` clients and parsers can also be called via command line by providing input and output paths of the file to be parsed, for example:

.. code-block:: bash

    python ./tulit/parsers/xml/formex.py --input input/file/path.xml --output output/file/path.json
````

## File: docs/source/index.rst
````
Welcome to tulit's documentation!
===================================

**tulit** is a Python library for processing legal documents in various formats,
including Akoma Ntoso and Formex.

Check out the :doc:`getting_started` section for further information.

.. note::

   This project is under active development.

Contents
--------

.. toctree::
   :maxdepth: 3

   getting_started

.. toctree::
   :maxdepth: 3

   client

.. toctree::
   :maxdepth: 3

   parsers
````

## File: docs/source/parsers.rst
````
Parsers
===============

This subpackage contains modules for parsing documents of type act in various formats into a JSON representation.

The formats currently supported are:

* Formex 4
* Akoma Ntoso
* XHTML as published in Cellar Semantic Repository of the Publications Office

.. automodule:: tulit.parsers.parser
    :members:
    :undoc-members:
    :show-inheritance:

.. automodule:: tulit.parsers.xml.xml
    :members:
    :undoc-members:
    :show-inheritance:

.. automodule:: tulit.parsers.xml.formex
    :members:
    :undoc-members:
    :show-inheritance:
    :exclude-members: main

.. automodule:: tulit.parsers.xml.akomantoso
    :members:
    :undoc-members:
    :show-inheritance:
    :exclude-members: main


.. automodule:: tulit.parsers.html.xhtml
    :members:
    :undoc-members:
    :show-inheritance:
    :exclude-members: main

.. automodule:: tulit.parsers.html.cellar
    :members:
    :undoc-members:
    :show-inheritance:
    :exclude-members: main
````

## File: docs/requirements.txt
````
sphinx==7.1.2
sphinx-rtd-theme==1.3.0rc1
tulit>=0.0.9
````

## File: tests/client/test_cellar.py
````python
import unittest
import json
from tulit.client.cellar import CellarClient
import os
from unittest.mock import patch, Mock
import requests
import io

class TestCellarClient(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None
        self.downloader = CellarClient(download_dir='./tests/data', log_dir='./tests/logs', proxies=None)
                
    def test_download(self):
        celex = "32008R1137"
        
        # Download the documents                           
        document_paths = self.downloader.download(celex, format='fmx4')

        expected = ['tests\\data\\e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04\\DOC_1.xml', 'tests\\data\\e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04\\DOC_2.xml', 'tests\\data\\e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04\\DOC_3.xml', 'tests\\data\\e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04\\DOC_4.xml']
        
        self.assertEqual(document_paths, expected)
    
    def test_get_cellar_ids_from_json_results(self):
        
        with open('./tests/metadata/query_results/query_results.json', 'r') as f:
            cellar_results = json.loads(f.read())
        
        self.downloader = CellarClient(download_dir='./tests/data', log_dir='./tests/logs')
        
        # Test for formex format
        extracted_ids = self.downloader.get_cellar_ids_from_json_results(cellar_results, 'fmx4')
        expected = [
            'e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_1', 'e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_2', 'e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_3', 'e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_4'
            ]
        
        self.assertEqual(extracted_ids, expected)

    def test_build_request_url(self):

        params = {'cellar': 'e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_1'}
        expected_url = 'http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_1'
        actual_url = self.downloader.build_request_url(params)
        self.assertEqual(actual_url, expected_url)
    
    @patch('tulit.client.client.requests.request')
    def test_fetch_content(self, mock_request):
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.raise_for_status.return_value = None
        mock_request.return_value = mock_response

        url = 'http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_1'
        response = self.downloader.fetch_content(url)

        # Check that the request was made with the correct URL and headers
        headers = {
            'Accept': "*, application/zip, application/zip;mtype=fmx4, application/xml;mtype=fmx4, application/xhtml+xml, text/html, text/html;type=simplified, application/msword, text/plain, application/xml, application/xml;notice=object",
            'Accept-Language': "eng",
            'Content-Type': "application/x-www-form-urlencoded",
            'Host': "publications.europa.eu"
        }
        mock_request.assert_called_once_with("GET", url, headers=headers)

        # Check that the response is as expected
        self.assertEqual(response, mock_response)

    @patch('tulit.client.client.requests.request')
    def test_fetch_content_request_exception(self, mock_request):
        # Mock request to raise a RequestException
        mock_request.side_effect = requests.RequestException("Error sending GET request")

        url = 'http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_1'
        response = self.downloader.fetch_content(url)

        # Check that the response is None when an exception is raised
        self.assertIsNone(response)

    def test_send_sparql_query(self):    
        sparql_query = """
        PREFIX cdm: <http://publications.europa.eu/ontology/cdm#>
        PREFIX purl: <http://purl.org/dc/elements/1.1/>

        SELECT DISTINCT ?cellarURIs, ?manif, ?format, ?expr
        WHERE {
                ?work owl:sameAs <http://publications.europa.eu/resource/celex/{CELEX}> .
                ?expr cdm:expression_belongs_to_work ?work ;
                cdm:expression_uses_language ?lang .
                ?lang purl:identifier ?langCode .
                ?manif cdm:manifestation_manifests_expression ?expr;
                cdm:manifestation_type ?format.
                ?cellarURIs cdm:item_belongs_to_manifestation ?manif.
        FILTER(str(?format)="fmx4" && str(?langCode)="ENG")
        }
        ORDER BY ?cellarURIs
        LIMIT 10
        """    
        celex = "32024R0903"
        # Send query
        response = self.downloader.send_sparql_query(sparql_query=sparql_query, celex=celex)        
        expected_results = json.loads('''{"head": {"link": [], "vars": ["cellarURIs", "manif", "format", "expr"]}, "results": {"distinct": false, "ordered": true, "bindings": [{"cellarURIs": {"type": "uri", "value": "http://publications.europa.eu/resource/cellar/c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006.02/DOC_1"}, "manif": {"type": "uri", "value": "http://publications.europa.eu/resource/cellar/c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006.02"}, "format": {"type": "typed-literal", "datatype": "http://www.w3.org/2001/XMLSchema#string", "value": "fmx4"}, "expr": {"type": "uri", "value": "http://publications.europa.eu/resource/cellar/c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006"}}]}}''')        
        self.assertEqual(response, expected_results)

if __name__ == "__main__":
    unittest.main()
````

## File: tests/client/test_client.py
````python
import unittest
from unittest.mock import patch, Mock
import os
from tulit.client.client import Client

class TestClient(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None
        self.downloader = Client(download_dir='./tests/data', log_dir='./tests/logs')
        
    def test_get_extension_from_content_type(self):
        content_type_mapping = {
            'text/html': 'html',
            'application/json': 'json',
            'application/xml': 'xml',
            'text/plain': 'txt',
            'application/zip': 'zip'
        }
        for content_type, expected_extension in content_type_mapping.items():
            actual_extension = self.downloader.get_extension_from_content_type(content_type)
            self.assertEqual(actual_extension, expected_extension)

    @patch('tulit.client.client.zipfile.ZipFile')
    def test_extract_zip(self, mock_zipfile):
        # Mock zipfile object
        mock_zip = Mock()
        mock_zipfile.return_value = mock_zip

        response = Mock()
        response.content = b'fake zip content'

        folder_path = './tests/data'
        self.downloader.extract_zip(response, folder_path)

        # Check that the zipfile was opened and extracted
        args, kwargs = mock_zipfile.call_args
        self.assertEqual(args[0].getvalue(), response.content)
        mock_zip.extractall.assert_called_once_with(folder_path)
    
    @patch('tulit.client.client.Client.extract_zip')
    @patch('tulit.client.client.os.makedirs')
    def test_handle_response(self, mock_makedirs, mock_extract_zip):
        # Mock response object
        response = Mock()
        response.headers = {'Content-Type': 'application/zip'}
        response.content = b'fake zip content'

        cellar_id = 'e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04\\DOC_1'
        target_path = os.path.join(self.downloader.download_dir, cellar_id)

        # Test handling zip content
        result = self.downloader.handle_response(response, cellar_id)
        mock_extract_zip.assert_called_once_with(response, target_path)
        self.assertEqual(result, target_path)

        # Test handling non-zip content
        response.headers = {'Content-Type': 'application/xml'}
        mock_extract_zip.reset_mock()
        result = self.downloader.handle_response(response, cellar_id)
        expected_file_path = os.path.normpath(f"{target_path}.xml")
        self.assertEqual(os.path.normpath(result), expected_file_path)



if __name__ == "__main__":
    unittest.main()
````

## File: tests/client/test_normattiva.py
````python
import unittest
from unittest.mock import patch, Mock
from tulit.client.normattiva import NormattivaClient

class TestNormattivaClient(unittest.TestCase):
    def setUp(self):
        self.downloader = NormattivaClient(download_dir='./tests/data/akn/italy', log_dir='./tests/logs')
    
    @patch('tulit.client.normattiva.requests.get')
    def test_build_request_url(self, mock_get):
        params = {
            'dataGU': '20210101',
            'codiceRedaz': '12345',
            'dataVigenza': '20211231',
            'date': '2021/01/01'
        }
        uri, url = self.downloader.build_request_url(params)
        expected_uri = "https://www.normattiva.it/eli/id/2021/01/01//12345/CONSOLIDATED"
        expected_url = "https://www.normattiva.it/do/atto/caricaAKN?dataGU=20210101&codiceRedaz=12345&dataVigenza=20211231"
        self.assertEqual(uri, expected_uri)
        self.assertEqual(url, expected_url)
    
    @patch('tulit.client.normattiva.requests.get')
    def test_fetch_content(self, mock_get):
        mock_response = Mock()
        mock_response.raise_for_status = Mock()
        mock_response.cookies = {'cookie_key': 'cookie_value'}
        mock_get.return_value = mock_response
        
        uri = "https://www.normattiva.it/eli/id/2021/01/01//12345/CONSOLIDATED"
        url = "https://www.normattiva.it/do/atto/caricaAKN?dataGU=20210101&codiceRedaz=12345&dataVigenza=20211231"
        
        response = self.downloader.fetch_content(uri, url)
        self.assertEqual(response, mock_response)
    
    @patch('tulit.client.normattiva.requests.get')
    @patch('tulit.client.normattiva.NormattivaClient.handle_response')
    def test_download(self, mock_handle_response, mock_get):
        mock_response = Mock()
        mock_response.raise_for_status = Mock()
        mock_response.headers = {'Content-Type': 'application/xml'}
        mock_get.return_value = mock_response
        mock_handle_response.return_value = './tests/data/akn/italy/20210101_12345_VIGENZA_20211231.xml'
        
        document_paths = self.downloader.download(dataGU='20210101', codiceRedaz='12345', dataVigenza='20211231')
        expected_paths = ['./tests/data/akn/italy/20210101_12345_VIGENZA_20211231.xml']
        self.assertEqual(document_paths, expected_paths)

if __name__ == "__main__":
    unittest.main()
````

## File: tests/metadata/query_results/ai_act.json
````json
{
    "head": {
        "link": [],
        "vars": [
            "cellarURIs",
            "manif",
            "format",
            "expr"
        ]
    },
    "results": {
        "distinct": false,
        "ordered": true,
        "bindings": [
            {
                "cellarURIs": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/dc8116a1-3fe6-11ef-865a-01aa75ed71a1.0006.02/DOC_1"
                },
                "manif": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/dc8116a1-3fe6-11ef-865a-01aa75ed71a1.0006.02"
                },
                "format": {
                    "type": "typed-literal",
                    "datatype": "http://www.w3.org/2001/XMLSchema#string",
                    "value": "fmx4"
                },
                "expr": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/dc8116a1-3fe6-11ef-865a-01aa75ed71a1.0006"
                }
            }
        ]
    }
}
````

## File: tests/metadata/query_results/query_results.json
````json
{
    "head": {
        "link": [],
        "vars": [
            "cellarURIs",
            "manif",
            "format",
            "expr"
        ]
    },
    "results": {
        "distinct": false,
        "ordered": true,
        "bindings": [
            {
                "cellarURIs": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_1"
                },
                "manif": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04"
                },
                "format": {
                    "type": "typed-literal",
                    "datatype": "http://www.w3.org/2001/XMLSchema#string",
                    "value": "fmx4"
                },
                "expr": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006"
                }
            },
            {
                "cellarURIs": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_2"
                },
                "manif": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04"
                },
                "format": {
                    "type": "typed-literal",
                    "datatype": "http://www.w3.org/2001/XMLSchema#string",
                    "value": "fmx4"
                },
                "expr": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006"
                }
            },
            {
                "cellarURIs": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_3"
                },
                "manif": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04"
                },
                "format": {
                    "type": "typed-literal",
                    "datatype": "http://www.w3.org/2001/XMLSchema#string",
                    "value": "fmx4"
                },
                "expr": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006"
                }
            },
            {
                "cellarURIs": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04/DOC_4"
                },
                "manif": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006.04"
                },
                "format": {
                    "type": "typed-literal",
                    "datatype": "http://www.w3.org/2001/XMLSchema#string",
                    "value": "fmx4"
                },
                "expr": {
                    "type": "uri",
                    "value": "http://publications.europa.eu/resource/cellar/e115172d-3ab3-4b14-b0a4-dfdcc9871793.0006"
                }
            }
        ]
    }
}
````

## File: tests/metadata/query_results/results_formex.json
````json
{
  "head": { "link": [], "vars": ["cellarURIs", "manif", "format", "expr"] },
  "results": {
    "distinct": false,
    "ordered": true,
    "bindings": [
      {
        "cellarURIs": {
          "type": "uri",
          "value": "http://publications.europa.eu/resource/cellar/c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006.02/DOC_1"
        },
        "manif": {
          "type": "uri",
          "value": "http://publications.europa.eu/resource/cellar/c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006.02"
        },
        "format": {
          "type": "typed-literal",
          "datatype": "http://www.w3.org/2001/XMLSchema#string",
          "value": "fmx4"
        },
        "expr": {
          "type": "uri",
          "value": "http://publications.europa.eu/resource/cellar/c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006"
        }
      }
    ]
  }
}
````

## File: tests/metadata/schemas/akomantoso30.xsd
````
<?xml version="1.0" encoding="UTF-8"?>
<!--
     Akoma Ntoso Version 1.0. Part 2: Specifications
     Committee Specification Draft 02 / Public Review Draft 02
     04 May 2016
     Copyright (c) OASIS Open 2016. All Rights Reserved.
     Source: http://docs.oasis-open.org/legaldocml/akn-core/v1.0/csprd02/part2-specs/schemas/
     Latest version of the specification: http://docs.oasis-open.org.legaldocml/akn-core/v1.0/akn-core-v1.0-part2-specs.html
     TC IPR Statement: https://www.oasis-open.org/committees/legaldocml/ipr.php
-->
<xsd:schema xmlns="http://docs.oasis-open.org/legaldocml/ns/akn/3.0"
            xmlns:xsd="http://www.w3.org/2001/XMLSchema"
            xmlns:xml="http://www.w3.org/XML/1998/namespace"
            targetNamespace="http://docs.oasis-open.org/legaldocml/ns/akn/3.0"
            elementFormDefault="qualified">

	<xsd:import namespace="http://www.w3.org/XML/1998/namespace"
				schemaLocation="http://www.w3.org/2001/03/xml.xsd"/>


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		Akoma Ntoso main schema
		
		Release 01/04/2016 - Akoma Ntoso 3.0 WD17 with errata
		Complete version.
		
		
		technical supervision Fabio Vitali - University of Bologna
		legal domain supervision Monica Palmirani - University of Bologna
		
		The present schema uses systematically XML design patterns as
		described in http://www.xmlpatterns.com/
		A few of them are explicitly mentioned, when their use or approach
		was not felt clear or natural enough. Others are just used without
		mentions.
				
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Main element groups
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:group name="ANhier">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANhier</name>
            <comment>
The group ANhier lists the elements that belong to the Akoma Ntoso legislative hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="clause"/>
			      <xsd:element ref="section"/>
			      <xsd:element ref="part"/>
			      <xsd:element ref="paragraph"/>
			      <xsd:element ref="chapter"/>
			      <xsd:element ref="title"/>
			      <xsd:element ref="article"/>
			      <xsd:element ref="book"/>
			      <xsd:element ref="tome"/>
			      <xsd:element ref="division"/>
			      <xsd:element ref="list"/>
			      <xsd:element ref="point"/>
			      <xsd:element ref="indent"/>
			      <xsd:element ref="alinea"/>
			      <xsd:element ref="rule"/>
			      <xsd:element ref="subrule"/>
			      <xsd:element ref="proviso"/>
			      <xsd:element ref="subsection"/>
			      <xsd:element ref="subpart"/>
			      <xsd:element ref="subparagraph"/>
			      <xsd:element ref="subchapter"/>
			      <xsd:element ref="subtitle"/>
			      <xsd:element ref="subdivision"/>
			      <xsd:element ref="subclause"/>
			      <xsd:element ref="sublist"/>
			      <xsd:element ref="level"/>
			      <xsd:element ref="transitional"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANcontainers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANcontainers</name>
            <comment>
The group ANcontainers lists the elements that are containers and are specific to the Akoma Ntoso debate vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="speechGroup"/>
			      <xsd:element ref="speech"/>
			      <xsd:element ref="question"/>
			      <xsd:element ref="answer"/>
			      <xsd:element ref="other"/>
			      <xsd:element ref="scene"/>
			      <xsd:element ref="narrative"/>
			      <xsd:element ref="summary"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="basicContainers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>basicContainers</name>
            <comment>
The group basicContainers lists the elements that are containers and are specific to vocabulary of preambles, prefaces, conclusions and coverPages</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="longTitle"/>
			      <xsd:element ref="formula"/>
			      <xsd:element ref="container"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="preambleContainers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>preambleContainers</name>
            <comment>
The group preambleContainers lists the elements that are containers and are specific to the Akoma Ntoso preamble vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="recitals"/>
			      <xsd:element ref="citations"/>
			      <xsd:element ref="formula"/>
			      <xsd:element ref="container"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="prefaceContainers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>prefaceContainers</name>
            <comment>
The group prefaceContainers lists the elements that are containers and are specific to the Akoma Ntoso preface vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="longTitle"/>
			      <xsd:element ref="formula"/>
			      <xsd:element ref="container"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANblock">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANblock</name>
            <comment>
The group ANblock lists the elements that are blocks and are specific to the Akoma Ntoso vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="blockList"/>
			      <xsd:element ref="blockContainer"/>
			      <xsd:element ref="tblock"/>
			      <xsd:element ref="toc"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANinline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANinline</name>
            <comment>
The group ANinline lists the elements that are inline and are specific to the Akoma Ntoso vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="ref"/>
			      <xsd:element ref="mref"/>
			      <xsd:element ref="rref"/>
			      <xsd:element ref="mod"/>
			      <xsd:element ref="mmod"/>
			      <xsd:element ref="rmod"/>
			      <xsd:element ref="remark"/>
			      <xsd:element ref="recordedTime"/>
			      <xsd:element ref="vote"/>
			      <xsd:element ref="outcome"/>
			      <xsd:element ref="ins"/>
			      <xsd:element ref="del"/>
			      <xsd:element ref="omissis"/>
			      <xsd:element ref="embeddedText"/>
			      <xsd:element ref="embeddedStructure"/>
			      <xsd:element ref="placeholder"/>
			      <xsd:element ref="fillIn"/>
			      <xsd:element ref="decoration"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANtitleInline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANtitleInline</name>
            <comment>
The group ANtitleInline lists the elements that are inline, are specific to the Akoma Ntoso vocabulary, and should only be used within the title element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="docType"/>
			      <xsd:element ref="docTitle"/>
			      <xsd:element ref="docNumber"/>
			      <xsd:element ref="docProponent"/>
			      <xsd:element ref="docDate"/>
			      <xsd:element ref="legislature"/>
			      <xsd:element ref="session"/>
			      <xsd:element ref="shortTitle"/>
			      <xsd:element ref="docAuthority"/>
			      <xsd:element ref="docPurpose"/>
			      <xsd:element ref="docCommittee"/>
			      <xsd:element ref="docIntroducer"/>
			      <xsd:element ref="docStage"/>
			      <xsd:element ref="docStatus"/>
			      <xsd:element ref="docJurisdiction"/>
			      <xsd:element ref="docketNumber"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANheaderInline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANheaderInline</name>
            <comment>
The group ANheaderInline lists the elements that are inline, are specific to the Akoma Ntoso vocabulary, and should only be used within the header element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="courtType"/>
			      <xsd:element ref="neutralCitation"/>
			      <xsd:element ref="party"/>
			      <xsd:element ref="judge"/>
			      <xsd:element ref="lawyer"/>
			      <xsd:element ref="signature"/>
			      <xsd:element ref="opinion"/>
			      <xsd:element ref="argument"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANsemanticInline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANsemanticInline</name>
            <comment>
The group ANsemanticInline lists additional elements that are inline, and are specific to the Akoma Ntoso vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="date"/>
			      <xsd:element ref="time"/>
			      <xsd:element ref="person"/>
			      <xsd:element ref="organization"/>
			      <xsd:element ref="concept"/>
			      <xsd:element ref="object"/>
			      <xsd:element ref="event"/>
			      <xsd:element ref="location"/>
			      <xsd:element ref="process"/>
			      <xsd:element ref="role"/>
			      <xsd:element ref="term"/>
			      <xsd:element ref="quantity"/>
			      <xsd:element ref="def"/>
			      <xsd:element ref="entity"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANmarker">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANmarker</name>
            <comment>
The group ANmarker lists the elements that are markers and are specific to the Akoma Ntoso vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="noteRef"/>
			      <xsd:element ref="eol"/>
			      <xsd:element ref="eop"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANsubFlow">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANsubFlow</name>
            <comment>
The group ANsubFlow lists the elements that are subFlows and are specific to the Akoma Ntoso vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="authorialNote"/>
		    </xsd:sequence>
	  </xsd:group>

	  <xsd:group name="HTMLcontainers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>HTMLcontainers</name>
            <comment>
The group HTMLcontainers lists the elements that are containers and were inherited from the HTML vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="div"/>
		    </xsd:sequence>
	  </xsd:group>

	  <xsd:group name="HTMLblock">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>HTMLblock</name>
            <comment>
The group HTMLblock lists the elements that are blocks and were inherited from the HTML vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="ul"/>
			      <xsd:element ref="ol"/>
			      <xsd:element ref="table"/>
			      <xsd:element ref="p"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="HTMLinline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>HTMLinline</name>
            <comment>
The group HTMLinline lists the elements that are inline and were inherited from the HTML vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="b"/>
			      <xsd:element ref="i"/>
			      <xsd:element ref="a"/>
			      <xsd:element ref="u"/>
			      <xsd:element ref="sub"/>
			      <xsd:element ref="sup"/>
			      <xsd:element ref="abbr"/>
			      <xsd:element ref="span"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="HTMLmarker">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>HTMLmarker</name>
            <comment>
The group HTMLmarker lists the elements that are marker and were inherited from the HTML vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="img"/>
			      <xsd:element ref="br"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="judgmentBlock">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>judgmentBlock</name>
            <comment>
The group judgmentBlock lists the structures that should be found in a judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="introduction"/>
			      <xsd:element ref="background"/>
			      <xsd:element ref="arguments"/>
			      <xsd:element ref="remedies"/>
			      <xsd:element ref="motivation"/>
			      <xsd:element ref="decision"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="amendmentBlock">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>amendmentBlock</name>
            <comment>
The group amendmentBlock lists the structures that should be found in an amendment</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="amendmentHeading"/>
			      <xsd:element ref="amendmentContent"/>
			      <xsd:element ref="amendmentReference"/>
			      <xsd:element ref="amendmentJustification"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="amendmentInline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>amendmentInline</name>
            <comment>
The group amendmentInline lists the inline elements that should be found in the preface of an amendment</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="affectedDocument"/>
			      <xsd:element ref="relatedDocument"/>
			      <xsd:element ref="change"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="speechSection">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>speechSection</name>
            <comment>
The group speechSection lists the structures that should contain speeches</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="administrationOfOath"/>
			      <xsd:element ref="rollCall"/>
			      <xsd:element ref="prayers"/>
			      <xsd:element ref="oralStatements"/>
			      <xsd:element ref="writtenStatements"/>
			      <xsd:element ref="personalStatements"/>
			      <xsd:element ref="ministerialStatements"/>
			      <xsd:element ref="resolutions"/>
			      <xsd:element ref="nationalInterest"/>
			      <xsd:element ref="declarationOfVote"/>
			      <xsd:element ref="communication"/>
			      <xsd:element ref="petitions"/>
			      <xsd:element ref="papers"/>
			      <xsd:element ref="noticesOfMotion"/>
			      <xsd:element ref="questions"/>
			      <xsd:element ref="address"/>
			      <xsd:element ref="proceduralMotions"/>
			      <xsd:element ref="pointOfOrder"/>
			      <xsd:element ref="adjournment"/>
			      <xsd:element ref="debateSection"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="hierElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>hierElements</name>
            <comment>
The group hierElements lists all the elements that are hierarchical</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="ANhier"/>
			      <xsd:element ref="hcontainer"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="containerElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>containerElements</name>
            <comment>
The group containerElements lists all the elements that are containers</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="speechSection"/>
			      <xsd:group ref="HTMLcontainers"/>
			      <xsd:element ref="container"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="blockElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>blockElements</name>
            <comment>
The group blockElements lists all the elements that are blocks</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="ANblock"/>
			      <xsd:group ref="HTMLblock"/>
			      <xsd:element ref="foreign"/>
			      <xsd:element ref="block"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="inlineElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>inlineElements</name>
            <comment>
The group inlineElements lists all the elements that are inline</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="ANinline"/>
			      <xsd:group ref="HTMLinline"/>
			      <xsd:group ref="ANtitleInline"/>
			      <xsd:group ref="ANsemanticInline"/>
			      <xsd:group ref="ANheaderInline"/>
			      <xsd:group ref="amendmentInline"/>
			      <xsd:element ref="inline"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="subFlowElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>subFlowElements</name>
            <comment>
The group subFlowElements lists all the elements that are subFlows</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="ANsubFlow"/>
			      <xsd:element ref="subFlow"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="markerElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>markerElements</name>
            <comment>
The group markerElements lists all the elements that are markers</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="ANmarker"/>
			      <xsd:group ref="HTMLmarker"/>
			      <xsd:element ref="marker"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="inlineCM">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>inlineCM</name>
            <comment>
The group inlineCM is the content model of blocks and inlines, and is composed of all the inlines and all the markers</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="inlineElements"/>
			      <xsd:group ref="markerElements"/>
			      <xsd:group ref="subFlowElements"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Attribute groups
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:attributeGroup name="contains">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>contains</name>
            <comment>
The attribute contains is used in the root of document types to indicate whether the document is original or modified wrt the orignal version</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="contains" type="versionType" default="originalVersion"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="alt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>alt</name>
            <comment>
The attribute alternativeTo is used to specify, when the element contains an alternative version of some content, the eId of the main element which this element is an alternative copy of</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="alternativeTo" type="eIdRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="name">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>name</name>
            <comment>
The attribute name is used to give a name to all generic elements</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="name" type="xsd:string" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="number">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>number</name>
            <comment>
The attribute number is used to specify a number in the publication of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="number" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="source">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>source</name>
            <comment>
The attribute source links to the agent (person, organization) providing the specific annotation or markup</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="source" type="agentRef" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="date">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>date</name>
            <comment>
The attribute date is used to give a normalized value for a date according to the XSD syntax YYYY-MM-DD or a normalized value for a dateTime according to the XSD syntax YYYY-MM-DDThh:mm:ss(zzzz)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="date" use="required">
			      <xsd:simpleType>
				        <xsd:union>
					          <xsd:simpleType>
						            <xsd:restriction base="xsd:date"/>
					          </xsd:simpleType>
					          <xsd:simpleType>
						            <xsd:restriction base="xsd:dateTime"/>
					          </xsd:simpleType>
				        </xsd:union>
			      </xsd:simpleType>
		    </xsd:attribute>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="time">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>time</name>
            <comment>
The attribute time is used to give a normalized value for a time according to the XSD syntax HH:MM:SS</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="time" use="required">
			      <xsd:simpleType>
				        <xsd:union>
					          <xsd:simpleType>
						            <xsd:restriction base="xsd:time"/>
					          </xsd:simpleType>
					          <xsd:simpleType>
						            <xsd:restriction base="xsd:dateTime"/>
					          </xsd:simpleType>
				        </xsd:union>
			      </xsd:simpleType>
		    </xsd:attribute>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="link">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>link</name>
            <comment>
The attribute href is used to specify an internal or external destination for a reference, a citation, an access to the ontology or a hypertext link. In elements using this attribute definition the href attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="href" type="xsd:anyURI" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="linkopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>linkopt</name>
            <comment>
The attribute href is used to specify an internal or external destination for a reference, a citation, an access to the ontology or a hypertext link. In elements using this attribute definition the href attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="href" type="xsd:anyURI"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="value">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>value</name>
            <comment>
The attribute value is used to specify a value for metadata elements. In elements using this attribute definition the value attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="value" type="xsd:string" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="optvalue">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>optvalue</name>
            <comment>
The attribute value is used to specify a value for metadata elements. In elements using this attribute definition the value attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="value" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="booleanvalue">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>booleanvalue</name>
            <comment>
The attribute value is used here to specify a boolean value for metadata elements. In elements using this attribute definition the value attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="value" type="xsd:boolean" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="interval">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>interval</name>
            <comment>
The start and end attributes are used to specify events that mark the beginning and the end of a time interval. Both are references to eventRef elements in the lifecycle element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="start" type="eventRefRef"/>
		    <xsd:attribute name="end" type="eventRefRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="duration">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>duration</name>
            <comment>
The duration attribute is used to specify a duration when either the start or the end event of a time interval is not known.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="duration" type="xsd:duration"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="speechAtts">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>speechAtts</name>
            <comment>
The attributes in speechAtts are used in speeches to identify agents and roles of speeches. In particular, attribute 'by' identifies the speaker, optional attribute 'as' identifies the role under which the speaker is speaking, optional attribute startTime specifies the absolute date and time where the individual speech item started, optional attribute endTime specifies the absolute date and time where the individual speech item ended, and  optional attribute to identifies the addressee of the speech. All of them are references to person or organization elements in the references section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="agent"/>
		    <xsd:attributeGroup ref="role"/>
		    <xsd:attribute name="startTime" type="xsd:dateTime"/>
		    <xsd:attribute name="endTime" type="xsd:dateTime"/>
		    <xsd:attribute name="to" type="agentRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="voteAtts">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>voteAtts</name>
            <comment>
The attributes in voteAtts are used in votes to identify agents and choices in votes. In particular, attribute 'by' identifies the voter, optional attribute 'as' identifies the role under which the voter is acting, optional attribute 'choice' specifies the vote that was casted.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="agent"/>
		    <xsd:attributeGroup ref="role"/>
		    <xsd:attribute name="choice" type="voteRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="frozen">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>frozen</name>
            <comment>
The attribute frozen for the condition element specifies whether the condition is frozen or not in the time period of this expression.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="frozen" type="xsd:boolean"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="lawyerAtts">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>lawyerAtts</name>
            <comment>
The attributes in lawyerAtts are used in the inline element lawyer to identify representation aspects of lawyers. In particular, attribute 'as' identifies his/her role, attribute 'for' identifies which party it represents, and attribute 'empoweredBy' identifies another lawyer, if any, this lawyer received the power delegation of power.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="role"/>
		    <xsd:attribute name="for" type="agentRef"/>
		    <xsd:attribute name="empoweredBy" type="agentRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="normalizedAtt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>normalizedAtt</name>
            <comment>
The attribute normalized is used in the inline element quantity to provide a normalized value of the number, if appropriate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="normalized" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="for">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>for</name>
            <comment>
The attribute for is used in modification elements to specify the element this is a modification of.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="for" type="eIdRef"/>
	  </xsd:attributeGroup>

	  <xsd:annotation>
		    <xsd:documentation>
		 The mappingAtts attributes represent the reference to the permanent wId (attribute original) of a structure, and to the eId (attribute current) such structure had during the time interval included between an initial temporalGroup (attribute start) and a final temporalGroup (attribute end). 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:attributeGroup name="mappingAtts">
		

		    <xsd:attribute name="original" type="wIdRef"/>
		    <xsd:attribute name="current" type="eIdRef"/>
		    <xsd:attribute name="start" type="eventRefRef"/>
		    <xsd:attribute name="end" type="eventRefRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="originating">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>originating</name>
            <comment>
The attribute originating is used to identify whether the event is the one originating this expression</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="originatingExpression" type="xsd:boolean"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="outcome">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>outcome</name>
            <comment>
The attribute outcome is used to identify the outcome of a step in a workflow. It is a reference to a TLCConcept element in the references section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="outcome" type="conceptRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="portionAtt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>portionAtt</name>
            <comment>
the attribute includedIn specifies the eId of the element of the references section identifying the document that this portion is a portion of.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="includedIn" type="referenceRef" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="level">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>level</name>
            <comment>
the attribute level specifies the level of the toc element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="level" type="xsd:integer" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="target">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>target</name>
            <comment>
the attribute target specifies the target of the a element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="target" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="imgAtts">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>imgAtts</name>
            <comment>
these attributes specify the dimension of the image.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="width" type="xsd:integer"/>
		    <xsd:attribute name="height" type="xsd:integer"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="tableAtts">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>tableAtts</name>
            <comment>
these attributes specify the characteristics of the table.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="width" type="xsd:integer"/>
		    <xsd:attribute name="border" type="xsd:integer"/>
		    <xsd:attribute name="cellspacing" type="xsd:integer"/>
		    <xsd:attribute name="cellpadding" type="xsd:integer"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="originalText">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>originalText</name>
            <comment>
the attribute originalText contains the original text of a placeholder</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="originalText" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="fillInWidth">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>fillInWidth</name>
            <comment>
the attribute width of the element fillIn specifies the width of the fillIn element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="width" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="upTo">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>upTo</name>
            <comment>
The attribute upTo is used in range-based elements to specify the eId of the ending element of the range.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="upTo" type="eIdRef" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="upToOpt">
		

		    <xsd:attribute name="upTo" type="eIdRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="range">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>range</name>
            <comment>
These attributes are used in range-based elements to specify the eId of the beginning and ending element of the range.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="from" type="eIdRef" use="required"/>
		    <xsd:attributeGroup ref="upTo"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="rangeOpt">
		

		    <xsd:attribute name="from" type="eIdRef" use="required"/>
		    <xsd:attributeGroup ref="upToOpt"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="pos">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>pos</name>
            <comment>
The attribute pos is used to identify the specific position of the reference (e.g., in source or destination)
with respect to the element being identified with the relative eId.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="pos" type="posType"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="language">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>language</name>
            <comment>
This attribute specifies the human language in which the document the element refers to is expressed. Values are taken from the RFC 4646.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="language" type="language" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="fromLanguage">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>fromLanguage</name>
            <comment>
This attribute specifies the human language of the expression this is a translation of. Values are taken from the RFC 4646.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="fromLanguage" type="language" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="pivot">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>pivot</name>
            <comment>
This attribute specifies the human language used as a pivot in the translation. Values are taken from the RFC 4646.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="pivot" type="language"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="authoritative">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>authoritative</name>
            <comment>
This attribute specifies whether the translation is authoritative or not</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="authoritative" type="xsd:boolean"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="dictionary">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>dictionary</name>
            <comment>
This attribute specifies the eId of the element in the references section detailing the dictionary used for the keyword.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="dictionary" type="referenceRef" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Attribute groups: attribute "type"
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:attributeGroup name="type">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>type</name>
            <comment>
The type attribute is used in various element to identify a type without restrictions as to its values.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="type" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="eventType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>eventType</name>
            <comment>
This is the list of allowed values for the type attribute of the event and action elements</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="generation"/>
			      <xsd:enumeration value="amendment"/>
			      <xsd:enumeration value="repeal"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="eventType">
		

		    <xsd:attribute name="type" type="eventType"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="opinionType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>opinionType</name>
            <comment>
This is the list of allowed values for the type attribute of the opinion element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="dissenting"/>
			      <xsd:enumeration value="agreeing"/>
			      <xsd:enumeration value="null"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="opinionType">
		

		    <xsd:attribute name="type" type="opinionType"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="resultType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>resultType</name>
            <comment>
This is the list of allowed values for the type attribute of the result element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="deny"/>
			      <xsd:enumeration value="dismiss"/>
			      <xsd:enumeration value="uphold"/>
			      <xsd:enumeration value="revert"/>
			      <xsd:enumeration value="replaceOrder"/>
			      <xsd:enumeration value="remit"/>
			      <xsd:enumeration value="decide"/>
			      <xsd:enumeration value="approve"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="resultType">
		

		    <xsd:attribute name="type" type="resultType" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="restrictionType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>restrictionType</name>
            <comment>
This is the list of allowed values for the restriction type attribute</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="jurisdiction"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="restrictionType">
		

		    <xsd:attribute name="type" type="restrictionType" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="remarkType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>remarkType</name>
            <comment>
This is the list of allowed values for the type attribute of the remark element for the specification of editorial remarks (e.g., applauses, laughters, etc.) especially within debate records</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="sceneDescription"/>
			      <xsd:enumeration value="phenomenon"/>
			      <xsd:enumeration value="caption"/>
			      <xsd:enumeration value="translation"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="remarkType">
		

		    <xsd:attribute name="type" type="remarkType"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="timeType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>timeType</name>
            <comment>
This is the list of allowed values for the type attribute of the recordedTime element for the specification of an explicit mention of a time (e.g., in a debate)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="startEvent"/>
			      <xsd:enumeration value="endEvent"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="recordedTimeType">
		

		    <xsd:attribute name="type" type="timeType"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="TextualMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>TextualMods</name>
            <comment>
The simple type TextualMods lists all the types of textual modifications as values for the type attribute of the textualMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="repeal"/>
			      <xsd:enumeration value="substitution"/>
			      <xsd:enumeration value="insertion"/>
			      <xsd:enumeration value="replacement"/>
			      <xsd:enumeration value="renumbering"/>
			      <xsd:enumeration value="split"/>
			      <xsd:enumeration value="join"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="textualModType">
		

		    <xsd:attribute name="type" type="TextualMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="MeaningMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>MeaningMods</name>
            <comment>
The simple type MeaningMods lists all the types of modifications in meaning as values for the type attribute of the meaningMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="variation"/>
			      <xsd:enumeration value="termModification"/>
			      <xsd:enumeration value="authenticInterpretation"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="meaningModType">
		

		    <xsd:attribute name="type" type="MeaningMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="ScopeMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>ScopeMods</name>
            <comment>
The simple type ScopeMods lists all the types of modifications in scope as values for the type attribute of the scopeMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="exceptionOfScope"/>
			      <xsd:enumeration value="extensionOfScope"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="scopeModType">
		

		    <xsd:attribute name="type" type="ScopeMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="ForceMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>ForceMods</name>
            <comment>
The simple type ForceMods lists all the types of modifications in force as values for the type attribute of the forceMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="entryIntoForce"/>
			      <xsd:enumeration value="endOfEnactment"/>
			      <xsd:enumeration value="postponementOfEntryIntoForce"/>
			      <xsd:enumeration value="prorogationOfForce"/>
			      <xsd:enumeration value="reEnactment"/>
			      <xsd:enumeration value="unconstitutionality"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="forceModType">
		

		    <xsd:attribute name="type" type="ForceMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="EfficacyMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>EfficacyMods</name>
            <comment>
The simple type EfficacyMods lists all the types of modifications in efficacy as values for the type attribute of the efficacyMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="entryIntoEfficacy"/>
			      <xsd:enumeration value="endOfEfficacy"/>
			      <xsd:enumeration value="inapplication"/>
			      <xsd:enumeration value="retroactivity"/>
			      <xsd:enumeration value="extraefficacy"/>
			      <xsd:enumeration value="postponementOfEfficacy"/>
			      <xsd:enumeration value="prorogationOfEfficacy"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="efficacyModType">
		

		    <xsd:attribute name="type" type="EfficacyMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="LegalSystemMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>LegalSystemMods</name>
            <comment>
The simple type LegalSystemMods lists all the types of modifications in the legal system as values for the type attribute of the LegalSystemMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="staticReference"/>
			      <xsd:enumeration value="implementation"/>
			      <xsd:enumeration value="ratification"/>
			      <xsd:enumeration value="application"/>
			      <xsd:enumeration value="legislativeDelegation"/>
			      <xsd:enumeration value="deregulation"/>
			      <xsd:enumeration value="conversion"/>
			      <xsd:enumeration value="expiration"/>
			      <xsd:enumeration value="reiteration"/>
			      <xsd:enumeration value="remaking"/>
			      <xsd:enumeration value="republication"/>
			      <xsd:enumeration value="coordination"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="legalSystemModType">
		

		    <xsd:attribute name="type" type="LegalSystemMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="show">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>show</name>
            <comment>
These attributes are used in metadata to propose visible representations of the metadata itself. Both a full visualization (attribute showAs) and an abbreviated one (attribute shortForm) can be specified</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="showAs" type="xsd:string" use="required"/>
		    <xsd:attribute name="shortForm" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="showopt">
		

		    <xsd:attribute name="showAs" type="xsd:string"/>
		    <xsd:attribute name="shortForm" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="src">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>src</name>
            <comment>
These attributes are used in manifestation-level references to specify external manifestation-level resources to be loaded in place. The src attribute holds the manifestation-level IRI of the resource, while the alt attribute holds the text to be displayed in case the loading of the external resource is not possible for any reason.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="src" type="manifestationURI" use="required"/>
		    <xsd:attribute name="alt" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="period">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>period</name>
            <comment>
The period attribute is used in versioned content and metadata elements to indicate a time interval in which they were in force, in efficacy, or in any other type of interval as specified in the corresponding temporalGroup.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="period" type="temporalGroupRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="enactment">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>enactment</name>
            <comment>
These attributes are those already defined in attribute list "period", plus the attribute status, that allows to specify the status of the piece of text it wraps.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="period"/>
		    <xsd:attribute name="status" type="statusType"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="notes">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>notes</name>
            <comment>
These attributes are used by notes, both authorial and editorial. The marker is the string to be used as a marker of the note, the placement is the position where the note should be placed, and placementBase is the eId of the element respect to which the note should be placed near.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="marker" type="xsd:string"/>
		    <xsd:attribute name="placement" type="placementType"/>
		    <xsd:attribute name="placementBase" type="eIdRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="modifiers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>modifiers</name>
            <comment>
These attributes are used in the analysis to allow manifestation editors to specify whether the analysis is complete and/or ignored in the text</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="exclusion" type="xsd:boolean"/>
		    <xsd:attribute name="incomplete" type="xsd:boolean"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="role">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>role</name>
            <comment>
The attribute role is used to identify the role of an individual mentioned in the text. It is a reference to a TLCRole element in the references section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="as" type="roleRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="agent">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>agent</name>
            <comment>
The attribute by is used to identify the agent of an action. It is a reference to a TLCPerson or TLCOrganization element in the references section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="by" type="agentRef" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="quote">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>quote</name>
            <comment>
The attributes startQuote and endQuote are used in quotedText, quotedStructure, embeddedText and embeddedStructure to specify the start and quote character delimiting the quoted or embedded part. If the characters are the same, one can use only startQuote.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="startQuote" type="xsd:string"/>
		    <xsd:attribute name="endQuote" type="xsd:string"/>
		    <xsd:attribute name="inlineQuote" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="cellattrs">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>cellattrs</name>
            <comment>
These attributes are used to specify that table cells span more than one row or one column, exactly as in HTML</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="rowspan" type="xsd:integer" default="1"/>
		    <xsd:attribute name="colspan" type="xsd:integer" default="1"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="HTMLattrs">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>HTMLattrs</name>
            <comment>
These attributes are used to specify class, style and title of the element, exactly as in HTML</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="class" type="xsd:string"/>
		    <xsd:attribute name="style" type="xsd:string"/>
		    <xsd:attribute name="title" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="core">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>core</name>
            <comment>
This attribute list are used to specify the fact that any attribute can be specified for this element if it belongs to a different namespace.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:anyAttribute namespace="##other" processContents="lax"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="idreq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>idreq</name>
            <comment>
These attributes identify the element in an absolute manner. In elements using this attribute definition the eId attribute is required whenever the Akoma Ntoso naming Convention is used. The wId is used to mark the identifier that the structure used to have in the original version, and is only needed when a renumbering occurred.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="eId" type="noWhiteSpace"/>
		    <xsd:attribute name="wId" type="noWhiteSpace"/>
		    <xsd:attribute name="GUID" type="noWhiteSpace"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="idopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>idopt</name>
            <comment>
These attributes identify the element in an absolute manner. In elements using this attribute definition the eId attribute is optional. The wId is used to mark the identifier that the structure used to have in the original version, and is only needed when a renumbering occurred.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="eId" type="noWhiteSpace"/>
		    <xsd:attribute name="wId" type="noWhiteSpace"/>
		    <xsd:attribute name="GUID" type="noWhiteSpace"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="refersreq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>refersreq</name>
            <comment>
This attribute creates a connection between the element and an element of the Akoma Ntoso ontology to which it refers. In elements using this attribute definition the refersTo attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="refersTo" use="required">
			      <xsd:simpleType>
				        <xsd:list>
					          <xsd:simpleType>
						            <xsd:restriction base="referenceRef"/>
					          </xsd:simpleType>
				        </xsd:list>
			      </xsd:simpleType>
		    </xsd:attribute>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="refers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>refers</name>
            <comment>
This attribute creates a connection between the element and an element of the Akoma Ntoso ontology to which it refers. In elements using this attribute definition the refersTo attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="refersTo">
			      <xsd:simpleType>
				        <xsd:list>
					          <xsd:simpleType>
						            <xsd:restriction base="referenceRef"/>
					          </xsd:simpleType>
				        </xsd:list>
			      </xsd:simpleType>
		    </xsd:attribute>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="xmllang">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>xmllang</name>
            <comment>
These attribute specify the human language in which the content of the element is expressed as well as the rules for whitespace management in the element. Values for xml:lang are taken from the RFC 4646. Both xml:lang and xml:space are reserved attributes of the XML language, and cannot be used for any other purpose than these ones.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute ref="xml:lang"/>
		    <xsd:attribute ref="xml:space"/>
		    <!--<xsd:attribute ref="xml:id"/>-->
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="corereq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>corereq</name>
            <comment>
This is the list of the core attributes that all elements in the content part of the document must have. In elements using this attribute definition the refersTo attribute is optional but the eId attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="HTMLattrs"/>
		    <xsd:attributeGroup ref="enactment"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="refers"/>
		    <xsd:attributeGroup ref="xmllang"/>
		    <xsd:attributeGroup ref="alt"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="corereqreq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>corereqreq</name>
            <comment>
This is the list of the core attributes that all elements in the content part of the document must have. In elements using this attribute definition both the refersTo attribute and the eId attribute are required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="HTMLattrs"/>
		    <xsd:attributeGroup ref="enactment"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="refersreq"/>
		    <xsd:attributeGroup ref="xmllang"/>
		    <xsd:attributeGroup ref="alt"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="coreopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>coreopt</name>
            <comment>
This is the list of the core attributes that all elements in the content part of the document must have. In elements using this attribute definition both the refersTo attribute and the eId attribute are optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="HTMLattrs"/>
		    <xsd:attributeGroup ref="enactment"/>
		    <xsd:attributeGroup ref="idopt"/>
		    <xsd:attributeGroup ref="refers"/>
		    <xsd:attributeGroup ref="xmllang"/>
		    <xsd:attributeGroup ref="alt"/>
	  </xsd:attributeGroup>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Simple types (data values)
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:simpleType name="noWhiteSpace">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>noWhiteSpace</name>
            <comment>
These values are those of ids such as eId, wId and GUID as a collection of any printable character except whitespaces.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:pattern value="[^\s]+"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:simpleType name="eIdRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>eIdRef</name>
            <comment>
These values are references to existing eIds only, i.e., given an existing eId of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="wIdRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>wIdRef</name>
            <comment>
These values are references to existing wIds only, i.e., given an existing wId of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="roleRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>roleRef</name>
            <comment>
These values are references to existing roles (i.e., TLCRole) only, i.e., given an existing eId of a role of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="agentRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>agentRef</name>
            <comment>
These values are references to existing agents (i.e., TLCPerson or TLCOrganization) only, i.e., given an existing eId of an agent of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="conceptRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>conceptRef</name>
            <comment>
These values are references to existing concepts (i.e., TLCConcept) only, i.e., given an existing eId of a concept of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="referenceRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>referenceRef</name>
            <comment>
These values are references to existing References elements only, i.e., given an existing eId of an element of the References section of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="voteRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>voteRef</name>
            <comment>
These values are references to existing votes (i.e., quorum or count) only, i.e., given an existing eId of a vote element of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="eventRefRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>eventRefRef</name>
            <comment>
These values are references to existing event references (i.e., eventRef) only, i.e., given an existing eId of an eventRef of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="temporalGroupRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>temporalGroupRef</name>
            <comment>
These values are references to existing temporal groups only, i.e., given an existing eId of a temporal group of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="manifestationURI">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>manifestationURI</name>
            <comment>
These values are references of a manifestation-Level document only</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="language">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>language</name>
            <comment>
This type specifies the human language in which the document the element refers to is expressed. Values are taken from the RFC 4646.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="versionType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>versionType</name>
            <comment>
This is the list of allowed values for the contains attribute</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="originalVersion"/>
			      <xsd:enumeration value="singleVersion"/>
			      <xsd:enumeration value="multipleVersions"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:simpleType name="placementType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>placementType</name>
            <comment>
This is the list of allowed values for the placement attribute of notes</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="side"/>
			      <xsd:enumeration value="left"/>
			      <xsd:enumeration value="right"/>
			      <xsd:enumeration value="bottom"/>
			      <xsd:enumeration value="inline"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:simpleType name="statusType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>statusType</name>
            <comment>
This is the list of allowed values for the status attribute. This is the list of possible reasons for a dscrepancy between the manifestation as it should be (e.g., a faithful representation of the content of an expression), and the manifestation as it actually is. Values should be interpreted as follows:
- removed: the content of the element is present in the markup (manifestation) but is not present in the real content of the document (expression level) because it has been definitely removed (either ex tunc, as in annullments, or ex nunc, as in abrogations).
- temporarily removed: the content of the element is present in the markup (manifestation) but is not present in the real content of the document (expression level) because it has been temporarily removed (e.g., for a temporary suspension or limitation of efficacy).
- translated: the content of the element is present in the markup (manifestation) in a different form than in the real content of the document (expression level) because it has been translated into a different language (e.g., to match the rest of the document or because of other editorial decisions).
- editorial: the content of the element is present in the markup (manifestation) but is not present in the real content of the document (expression level) because it has been inserted as an editorial process when creating the XML markup.
- edited: the content of the element is different in the markup (manifestation) than in the real content of the document (expression level) because it has been amended (e.g., to remove scurrilous or offensive remarks).
- verbatim: the content of the element is present in the markup (manifestation) is EXACTLY as it was in the real content of the document (expression level) because usual silent fixes and edits were NOT performed (e.g. to punctuation, grammatical errors or other usually non-debatable problems).
- incomplete: the content of the element or the value of a required attribute is NOT present in the markup (manifestation), although it should, because the missing data is not known at the moment, but in the future it might become known. This is especially appropriate for documents in drafting phase (e.g., the publication date of the act while drafting the bill)
- unknown: the content of the element or the value of a required attribute is NOT present in the markup (manifestation), although it should, because the author of the manifestation does not know it.
- undefined: the content of the element or the value of a required attribute is NOT present in the markup (manifestation), because the information is not defined in the original document, or it doesn't exist in some legal tradition (e.g. an anonymous speech cannot specify the attribute by, or some publications do not record the numbering of the items, etc.)
- ignored: the content of the element or the value of a required attribute is NOT present in the markup (manifestation) because the information exists but the author of the manifestation is not interested in reporting it (e.g., omitted parts of the document due to editorial reasons, etc.)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="removed"/>
			      <xsd:enumeration value="temporarilyRemoved"/>
			      <xsd:enumeration value="translated"/>
			      <xsd:enumeration value="editorial"/>
			      <xsd:enumeration value="edited"/>
			      <xsd:enumeration value="verbatim"/>
			      <xsd:enumeration value="incomplete"/>
			      <xsd:enumeration value="unknown"/>
			      <xsd:enumeration value="undefined"/>
			      <xsd:enumeration value="ignored"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:simpleType name="posType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>posType</name>
            <comment>
This is the list of possible positions of the text being analyzed by the element in the analysis section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="start"/>
			      <xsd:enumeration value="before"/>
			      <xsd:enumeration value="inside"/>
			      <xsd:enumeration value="after"/>
			      <xsd:enumeration value="end"/>
			      <xsd:enumeration value="unspecified"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Complex types (markup structures)
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:complexType name="basehierarchy">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>basehierarchy</name>
            <comment>
The complex type basehierarchy is not used by any element, but is derived by other types to contain the basic structure of hierarchical elements</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="0" maxOccurs="unbounded">
			      <xsd:element ref="num" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="heading" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="subheading" minOccurs="0" maxOccurs="1"/>
		    </xsd:choice>
	  </xsd:complexType>

	  <xsd:complexType name="hierarchy">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>hierarchy</name>
            <comment>
The complex type hierarchy is used by most or all the hierarchical elements of act-like documents.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:choice>
					          <xsd:sequence>
						            <xsd:element ref="intro" minOccurs="0" maxOccurs="1"/>
						            <xsd:choice minOccurs="0" maxOccurs="unbounded">
							              <xsd:element ref="componentRef"/>
							              <xsd:group ref="hierElements"/>
							              <xsd:element ref="crossHeading"/>
						            </xsd:choice>
						            <xsd:element ref="wrapUp" minOccurs="0" maxOccurs="1"/>
					          </xsd:sequence>
					          <xsd:element ref="content"/>
				        </xsd:choice>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="althierarchy">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>althierarchy</name>
            <comment>
The complex type althierarchy is used by most or all the hierarchical elements of documents that are not act-like.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:choice minOccurs="0" maxOccurs="unbounded">
					          <xsd:group ref="containerElements"/>
					          <xsd:element ref="componentRef"/>
					          <xsd:group ref="ANcontainers"/>
					          <xsd:group ref="blockElements"/>
				        </xsd:choice>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="blocksreq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>blocksreq</name>
            <comment>
the complex type blocksreq defines the content model and attributes shared by all containers. Here the eId attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="corereq"/>
	  </xsd:complexType>

	  <xsd:complexType name="blocksopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>blocksopt</name>
            <comment>
the complex type blocksopt defines the content model and attributes shared by all containers. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="inline" mixed="true">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>inline</name>
            <comment>
the complex type inline defines the content model and attributes shared by all blocks and inlines. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="inlineCM"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="inlinereq" mixed="true">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>inlinereq</name>
            <comment>
the complex type inlinereq defines the content model and attributes shared by all blocks and inlines. Here the eId attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="inlineCM"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="corereq"/>
	  </xsd:complexType>

	  <xsd:complexType name="inlinereqreq" mixed="true">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>inlinereqreq</name>
            <comment>
the complex type inlinereq defines the content model and attributes shared by all blocks and inlines. Here the eId attribute is required and also the refersTo is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="inlineCM"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="corereqreq"/>
	  </xsd:complexType>

	  <xsd:complexType name="markerreq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>markerreq</name>
            <comment>
the complex type markerreq defines the content model and attributes shared by all marker elements. Here the eId attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="corereq"/>
	  </xsd:complexType>

	  <xsd:complexType name="markeropt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>markeropt</name>
            <comment>
the complex type markeropt defines the content model and attributes shared by all marker elements. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="metareq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>metareq</name>
            <comment>
the complex type metareq defines the content model and attributes shared by all metadata elements. Here the eId attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
	  </xsd:complexType>

	  <xsd:complexType name="metaopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>metaopt</name>
            <comment>
the complex type metaopt defines the content model and attributes shared by all metadata elements. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="maincontent">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>maincontent</name>
            <comment>
the complex type maincontent is used by container elements that can contain basically any other Akoma Ntoso structure</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:element ref="componentRef"/>
			      <xsd:group ref="hierElements"/>
			      <xsd:group ref="blockElements"/>
			      <xsd:group ref="containerElements"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="speechType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>speechType</name>
            <comment>
the complex type speechType defines the content model and attributes shared by all speech elements. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:sequence>
					          <xsd:element ref="from" minOccurs="0" maxOccurs="1"/>
					          <xsd:sequence minOccurs="1" maxOccurs="unbounded">
						            <xsd:group ref="blockElements"/>
					          </xsd:sequence>
				        </xsd:sequence>
				        <xsd:attributeGroup ref="coreopt"/>
				        <xsd:attributeGroup ref="speechAtts"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="itemType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>itemType</name>
            <comment>
The complex type itemType is similar to a hierarchical element, but is isolated and does not belong to a full hierarchy.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:sequence minOccurs="1" maxOccurs="unbounded">
					          <xsd:group ref="blockElements"/>
				        </xsd:sequence>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="referenceType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>referenceType</name>
            <comment>
the complex type referenceType defines the empty content model and the list of attributes for metadata elements in the references section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="refers"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="link"/>
		    <xsd:attributeGroup ref="show"/>
	  </xsd:complexType>

	  <xsd:complexType name="srcType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>srcType</name>
            <comment>
the complex type srcType defines the empty content model and the list of attributes for manifestation-level references to external resources</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idopt"/>
		    <xsd:attributeGroup ref="src"/>
		    <xsd:attributeGroup ref="show"/>
	  </xsd:complexType>

	  <xsd:complexType name="linkType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>linkType</name>
            <comment>
the complex type linkType defines the empty content model and the list of attributes for Work- or Expression-level references to external resources</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="link"/>
		    <xsd:attributeGroup ref="show"/>
	  </xsd:complexType>

	  <xsd:complexType name="anyOtherType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>anyOtherType</name>
            <comment>
the complex type anyOtherType defines an open content model for elements that may use elements from other namespaces.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:any namespace="##other"
                  processContents="lax"
                  minOccurs="0"
                  maxOccurs="unbounded"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idopt"/>
		    <xsd:attributeGroup ref="linkopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="docContainerType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>docContainerType</name>
            <comment>
the complex type docContainerType defines a shared content model for elements that contain whole documents, namely attachment, collectionItem, component.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:choice>
					          <xsd:group ref="documentType"/>
					          <xsd:element ref="interstitial"/>
					          <xsd:element ref="toc"/>
					          <xsd:element ref="documentRef"/>
				        </xsd:choice>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Root element
		Comment: One generic root element contains all document elements
		Pattern: Universal Root
		Reference: http://www.xmlpatterns.com/UniversalRootMain.shtml
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:group name="documentType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>documentType</name>
            <comment>
the type documentType lists all the document types that are managed by Akoma Ntoso</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="amendmentList"/>
			      <xsd:element ref="officialGazette"/>
			      <xsd:element ref="documentCollection"/>
			      <xsd:element ref="act"/>
			      <xsd:element ref="bill"/>
			      <xsd:element ref="debateReport"/>
			      <xsd:element ref="debate"/>
			      <xsd:element ref="statement"/>
			      <xsd:element ref="amendment"/>
			      <xsd:element ref="judgment"/>
			      <xsd:element ref="portion"/>
			      <xsd:element ref="doc"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Document elements
		Comment: All document elements share just two content models
		Pattern: Consistent Element Set
		Reference: http://www.xmlpatterns.com/ConsistentElementSetMain.shtml
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:complexType name="akomaNtosoType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>akomaNtosoType</name>
            <comment>
the complex type akomaNtosoType is the type for the root element in Akoma Ntoso.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:sequence>
				        <xsd:group ref="documentType"/>
			      </xsd:sequence>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
	  </xsd:complexType>

	  <xsd:element name="akomaNtoso" type="akomaNtosoType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element (root)</type>
            <name>akomaNtoso</name>
            <comment>
NAME
akomaNtoso
the element akomaNtoso is the root element of all document types in Akoma Ntoso. It follows the pattern Universal Root (http://www.xmlpatterns.com/UniversalRootMain.shtml)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="openStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>openStructure</name>
            <comment>
the type openStructure specifies the overall content model of all the document types that do not have a specific and peculiar structure</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preface" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preamble" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="mainBody"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="doc" type="openStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>doc</name>
            <comment>
Element doc is used for describing the structure and content of any other document that is not included in the list of document explicitly managed by Akoma Ntoso</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-doc">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-doc">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="mainBody" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>mainBody</name>
            <comment>
the element mainBody is the container of the main part of all other document types</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="statement" type="openStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>statement</name>
            <comment>
Element statement is used for describing the structure and content of a official documents of a legislative assembly that are not normative in structure (e.g., statements, non-normative resolutions, etc.).</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-statement">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-statement">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="components">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="component"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="idreq"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="component" type="docContainerType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>component</name>
            <comment>
The element component is a container of a subdocument specified in a composite document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="collectionStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>collectionStructure</name>
            <comment>
the type collectionStructure specifies the overall content model of the document types that are collections of other documents</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preface" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preamble" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="collectionBody"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="amendmentList" type="collectionStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentList</name>
            <comment>
Element amendmentList is used for describing the structure and content of a collection of amendments</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-amendmentList">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-amendmentList">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="officialGazette" type="collectionStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>officialGazette</name>
            <comment>
Element officialGazette is used for describing the structure and content of an issue of an official gazette</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-officialGazette">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-officialGazette">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="documentCollection" type="collectionStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>documentCollection</name>
            <comment>
Element documentCollection is used for describing the structure and content of a collection of other documents chosen and combined for any reason whatsoever</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-documentCollection">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-documentCollection">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="collectionBodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>collectionBodyType</name>
            <comment>
the type collectionBodyType specifies a content model of a container of a list of other documents (e.g, acts, bills, amendments, etc.) possibly  interspersed with interstitial elements with content that does not form an individual document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="component" minOccurs="1" maxOccurs="unbounded"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="collectionBody" type="collectionBodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>collectionBody</name>
            <comment>
the element collectionBody is the container of a list of other documents (e.g, acts, bills, amendments, etc.) possibly  interspersed with interstitial elements with content that does not form an individual document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="portionStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>portionStructure</name>
            <comment>
the type portionStructure specifies the overall content model of the document type that is a portion of another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="portionBody"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="portionAtt"/>
	  </xsd:complexType>

	  <xsd:element name="portion" type="portionStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>portion</name>
            <comment>
Element portion is used for describing the structure and content of an independent portion of a document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-portion">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-portion">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="portionBodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>portionBodyType</name>
            <comment>
the type portionBodyType specifies a content model of a container of a portion of another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="containerElements"/>
			      <xsd:group ref="hierElements"/>
			      <xsd:group ref="ANcontainers"/>
			      <xsd:element ref="recitals"/>
			      <xsd:element ref="recital"/>
			      <xsd:element ref="citations"/>
			      <xsd:element ref="citation"/>
			      <xsd:element ref="longTitle"/>
			      <xsd:element ref="formula"/>
			      <xsd:element ref="coverPage"/>
			      <xsd:element ref="preface"/>
			      <xsd:element ref="preamble"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="portionBody" type="portionBodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>portionBody</name>
            <comment>
the element portionBody is the container of a portion of another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="hierarchicalStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>hierarchicalStructure</name>
            <comment>
the type hierarchicalStructure specifies the overall content model of the document types that are hierarchical in nature, especially acts and bills</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preface" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preamble" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="body"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="act" type="hierarchicalStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>act</name>
            <comment>
Element act is used for describing the structure and content of an act</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-act">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-act">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="bill" type="hierarchicalStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>bill</name>
            <comment>
Element bill is used for describing the structure and content of a bill</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-bill">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-bill">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="bodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>bodyType</name>
            <comment>
the type bodyType specifies a content model of the main hierarchy of a hierarchical document (e.g, an act or a bill)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:element ref="componentRef"/>
			      <xsd:group ref="hierElements"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="body" type="bodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>body</name>
            <comment>
the element body is the container of the main hierarchy of a hierarchical document (e.g, an act or a bill)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="debateStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>debateStructure</name>
            <comment>
the type debateStructure specifies the overall content model of the document types that describe debates</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preface" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="debateBody"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="debateReport" type="openStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>debateReport</name>
            <comment>
Element debateReport is used for describing the structure and content of a report</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-debateReport">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-debateReport">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="debate" type="debateStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>debate</name>
            <comment>
Element debate is used for describing the structure and content of a debate record</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-debate">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-debate">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="debateBodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>debateBodyType</name>
            <comment>
the type debateBodyType specifies a content model of the main hierarchy of a debate</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="speechSection"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="debateBody" type="debateBodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>debateBody</name>
            <comment>
the element debateBody is the container of the main hierarchy of a debate</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="judgmentStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>judgmentStructure</name>
            <comment>
the type judgmentStructure specifies the overall content model of the document types that describe judgments</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="header"/>
			      <xsd:element ref="judgmentBody"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="judgment" type="judgmentStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>judgment</name>
            <comment>
Element judgment is used for describing the structure and content of a judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-judgment">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-judgment">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="judgmentBodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>judgmentBodyType</name>
            <comment>
the type judgmentBodyType specifies a content model of the main hierarchy of a judgment document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="judgmentBlock"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="judgmentBody" type="judgmentBodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>judgmentBody</name>
            <comment>
the element judgmentBody is the container of the main hierarchy of a judgment document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="amendmentStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>amendmentStructure</name>
            <comment>
the type amendmentStructure specifies the overall content model of the document types that describe amendments</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preface" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="amendmentBody"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="amendment" type="amendmentStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendment</name>
            <comment>
Element amendment is used for describing the structure and content of an amendment</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-amendment">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-amendment">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="amendmentBodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>amendmentBodyType</name>
            <comment>
the type amendmentBodyType specifies a content model of the main hierarchy of a amendment document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="amendmentBlock"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="amendmentBody" type="amendmentBodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentBody</name>
            <comment>
the element amendmentBody is the container of the main hierarchy of a amendment document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="recitalHierarchy">
		

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:sequence>
					          <xsd:element ref="intro" minOccurs="0" maxOccurs="1"/>
					          <xsd:choice minOccurs="1" maxOccurs="unbounded">
						            <xsd:element ref="componentRef"/>
						            <xsd:element ref="recital"/>
					          </xsd:choice>
					          <xsd:element ref="wrapUp" minOccurs="0" maxOccurs="1"/>
				        </xsd:sequence>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="recitals" type="recitalHierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>recitals</name>
            <comment>
the element recitals is the section of the preamble that contains recitals</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="recital" type="itemType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>recital</name>
            <comment>
the element recital is the individual element of the preface that is called recital</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="citationHierarchy">
		

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:sequence>
					          <xsd:element ref="intro" minOccurs="0" maxOccurs="1"/>
					          <xsd:choice minOccurs="1" maxOccurs="unbounded">
						            <xsd:element ref="componentRef"/>
						            <xsd:element ref="citation"/>
					          </xsd:choice>
					          <xsd:element ref="wrapUp" minOccurs="0" maxOccurs="1"/>
				        </xsd:sequence>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="citations" type="citationHierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>citations</name>
            <comment>
the element citations is the section of the preamble that contains citations</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="citation" type="itemType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>citation</name>
            <comment>
the element citation is the individual element of the preface that is called citation</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="longTitle" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>longTitle</name>
            <comment>
the element longTitle is the section of the preface or preamble that is called long title</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="formula">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>formula</name>
            <comment>
the element formula is a section of the preface or preamble that contains a formulaic expression that is systematically or frequently present in a preface or a preamble and has e precise legal meaning (e.g. an enacting formula). Use the refersTo attribute for the specification of the actual type of formula.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="blocksreq">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="basicopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>basicopt</name>
            <comment>
the complex type basicopt defines the content model and attributes used by basic containers such as coverPage and conclusions. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
			      <xsd:group ref="basicContainers"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="coverPage" type="basicopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>coverPage</name>
            <comment>
the element coverPage is used as a container of the text that acts as a cover page</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="preambleopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>preambleopt</name>
            <comment>
the complex type preambleopt defines the content model and attributes used by preambles. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
			      <xsd:group ref="preambleContainers"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="preamble" type="preambleopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>preamble</name>
            <comment>
the element preamble is used as a container of the text that opens the main body of the document as a preamble</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="prefaceopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>prefaceopt</name>
            <comment>
the complex type prefaceopt defines the content model and attributes used by preface. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
			      <xsd:group ref="prefaceContainers"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="preface" type="prefaceopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>preface</name>
            <comment>
the element preface is used as a container of all prefacing material (e.g. headers, formulas, etc.)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="conclusions" type="basicopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>conclusions</name>
            <comment>
the element conclusion is used as a container of all concluding material (e.g. dates, signatures, formulas, etc.)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="header" type="blocksopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>header</name>
            <comment>
the element header is used as a container of all prefacing material of judgments (e.g. headers, formulas, etc.)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="attachments">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="attachment"/>
			      </xsd:sequence>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="attachment" type="docContainerType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>attachment</name>
            <comment>
the element attachment is used as a container of individual attachment elements</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="interstitial" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>interstitial</name>
            <comment>
the element interstitial is used as a container of text elements and blocks that are placed for any reason between individual documents in a collection of documents</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="componentRef" type="srcType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>componentRef</name>
            <comment>
the element componentRef is a reference to a separate manifestation-level resource that holds the content of the component of the document not physically placed at the position specified. Actual resources can either be external (e.g. in the package or even in a different position) or internal (within the components element)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="documentRef" type="linkType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>documentRef</name>
            <comment>
the element documentRef is a reference to a separate work- or expression-level resource that should be placed in this position. Actual resources are external (e.g. in the package or even in a different position) and are (an expression or any expression of) a separate Work.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Elements for hierarchical documents
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:element name="clause" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>clause</name>
            <comment>
this element is a hierarchical container called "clause" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="section" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>section</name>
            <comment>
this element is a hierarchical container called "section" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="part" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>part</name>
            <comment>
this element is a hierarchical container called "part" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="paragraph" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>paragraph</name>
            <comment>
this element is a hierarchical container called "paragraph" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="chapter" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>chapter</name>
            <comment>
this element is a hierarchical container called "chapter" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="title" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>title</name>
            <comment>
this element is a hierarchical container called "title" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="book" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>book</name>
            <comment>
this element is a hierarchical container called "book" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="tome" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>tome</name>
            <comment>
this element is a hierarchical container called "tome" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="article" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>article</name>
            <comment>
this element is a hierarchical container called "article" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="division" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>division</name>
            <comment>
this element is a hierarchical container called "division" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="list" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>list</name>
            <comment>
this element is a hierarchical container called "list" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="point" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>point</name>
            <comment>
this element is a hierarchical container called "point" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="indent" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>indent</name>
            <comment>
this element is a hierarchical container called "indent" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="alinea" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>alinea</name>
            <comment>
this element is a hierarchical container called "alinea" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="rule" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>rule</name>
            <comment>
this element is a hierarchical container called "rule" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subrule" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subrule</name>
            <comment>
this element is a hierarchical container called "subrule" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="proviso" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>proviso</name>
            <comment>
this element is a hierarchical container called "proviso" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subsection" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subsection</name>
            <comment>
this element is a hierarchical container called "subsection" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subpart" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subpart</name>
            <comment>
this element is a hierarchical container called "subpart" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subparagraph" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subparagraph</name>
            <comment>
this element is a hierarchical container called "subparagraph" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subchapter" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subchapter</name>
            <comment>
this element is a hierarchical container called "subchapter" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subtitle" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subtitle</name>
            <comment>
this element is a hierarchical container called "subtitle" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subdivision" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subdivision</name>
            <comment>
this element is a hierarchical container called "subdivision" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subclause" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subclause</name>
            <comment>
this element is a hierarchical container called "subclause" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="sublist" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>sublist</name>
            <comment>
this element is a hierarchical container called "sublist" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="level" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>level</name>
            <comment>
this element is a hierarchical container called "level" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="transitional" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>transitional</name>
            <comment>
this element is a hierarchical container called "transitional" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="content" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>content</name>
            <comment>
the element content is the final container in a hierarchy, and is where the blocks of text of the content of the structure are finally specified</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="num" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>num</name>
            <comment>
the element num is a heading element in a hierarchy that contains a number or any other ordered mechanism to identify the structure.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="heading" type="inlinereq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>heading</name>
            <comment>
the element heading is a heading element in a hierarchy that contains a title or any other textual content to describe the structure.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subheading" type="inlinereq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subheading</name>
            <comment>
the element subheading is a heading element in a hierarchy that contains a subtitle or any other textual content to further describe the structure.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="crossHeading" type="inlinereq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>crossHeading</name>
            <comment>
the element crossHeading is a heading element that is placed side by side with hierarchical containers .</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="intro" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>intro</name>
            <comment>
the element intro is a heading element in a hierarchy that contains paragraphs introducing one or more lower hierarchical elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="wrapUp" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>wrapUp</name>
            <comment>
the element wrapUp is a concluding element in a hierarchy that contains paragraphs wrapping up the preceding lower hierarchical elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Elements for debates documents
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:element name="administrationOfOath" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>administrationOfOath</name>
            <comment>
this element is a structural container for parts of a debates that contain the administration of an oath</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="rollCall" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>rollCall</name>
            <comment>
this element is a structural container for parts of a debates that contain a roll call of individuals</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="prayers" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>prayers</name>
            <comment>
this element is a structural container for parts of a debates that contain prayers</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="oralStatements" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>oralStatements</name>
            <comment>
this element is a structural container for parts of a debates that contain oral statements by participants</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="writtenStatements" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>writtenStatements</name>
            <comment>
this element is a structural container for parts of a debates that contain written statements by participants</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="personalStatements" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>personalStatements</name>
            <comment>
this element is a structural container for parts of a debates that contain written statements by participants</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="ministerialStatements" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>ministerialStatements</name>
            <comment>
this element is a structural container for parts of a debates that contain written statements by participants</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="resolutions" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>resolutions</name>
            <comment>
this element is a structural container for parts of a debates that contain resolutions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="nationalInterest" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>nationalInterest</name>
            <comment>
this element is a structural container for parts of a debates that contain resolutions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="declarationOfVote" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>declarationOfVote</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to the declaration of votes</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="communication" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>communication</name>
            <comment>
this element is a structural container for parts of a debates that contain communications from the house</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="petitions" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>petitions</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to petitions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="papers" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>papers</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to the display of papers</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="noticesOfMotion" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>noticesOfMotion</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to the notices of motions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="questions" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>questions</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to questions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="address" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>address</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to addresses</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="proceduralMotions" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>proceduralMotions</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to procedural motions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="pointOfOrder" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>pointOfOrder</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to points of order</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="adjournment" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>adjournment</name>
            <comment>
this element is a structural container for parts of a debates that contain adjournment notices</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="debateSection">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>debateSection</name>
            <comment>
this element is a generic structural container for all other parts of a debates that are not explicitly supported with a named element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="althierarchy">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="speechGroup">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>speechGroup</name>
            <comment>
the element speechGroup is a container of speech elements. This element is meant to pooint out, in a complex sequence of individual speech elements, the main contributor, i.e., the individual speech who was introducedand expected and that is causing the complex sequence that follows. Attributes by, as and to are those of the main speech.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="althierarchy">
					          <xsd:attributeGroup ref="speechAtts"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="speech" type="speechType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>speech</name>
            <comment>
the element speech is a container of a single speech utterance in a debate. Dialogs between speakers need a speech element each</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="question" type="speechType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>question</name>
            <comment>
the element question is a container of a single official question as proposed by an MP to a person holding an official position</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="answer" type="speechType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>answer</name>
            <comment>
the element answer is a container of a single official answer to a question</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="other" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>other</name>
            <comment>
the element other is a container of parts of a debate that are not speeches, nor scene comments (e.g., lists of papers, etc.)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="scene" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>scene</name>
            <comment>
the element scene is a container of descriptions of the scene elements happening in a given moment during a debate (e.g., applauses)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="narrative" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>narrative</name>
            <comment>
the element narrative is a block element in a debate to mark description in the third person of events taking place in the meeting, e.g. "Mr X. takes the Chair"</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="summary" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>summary</name>
            <comment>
the element summary is a block element in a debate to mark summaries of speeches that are individually not interesting (e.g.: "Question put and agreed to")</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="from" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>from</name>
            <comment>
the element from is a heading element in a debate that contains the name or role or a reference to the person doing the speech</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="vote">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>vote</name>
            <comment>
the element vote is an inline that wraps either the name of the voter (when organized by choice) or the vote (when organized by name) in a voting report.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="voteAtts"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="outcome" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>outcome</name>
            <comment>
the element outcome is an inline that wraps the outcome of a vote</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="introduction" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>introduction</name>
            <comment>
this element is a structural container for the section of a judgment containing introductory material</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="background" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>background</name>
            <comment>
this element is a structural container for the section of a judgment containing the background</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="arguments" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>arguments</name>
            <comment>
this element is a structural container for the section of a judgment containing the arguments</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="remedies" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>remedies</name>
            <comment>
this element is a structural container for the section of a judgment containing the remedies</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="motivation" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>motivation</name>
            <comment>
this element is a structural container for the section of a judgment containing the motivation</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="decision" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>decision</name>
            <comment>
this element is a structural container for the section of a judgment containing the decision</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="affectedDocument">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>affectedDocument</name>
            <comment>
the element affectedDocument is an inline element within preamble to identify the document that this amendment affects</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="link"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="relatedDocument">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>relatedDocument</name>
            <comment>
the element relatedDocument is an inline element to identify the document for which this document is a report of</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="link"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="change" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>change</name>
            <comment>
the element change is an inline element that identifies the changes expressed in the two columns of an amendment document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="amendmentHeading" type="blocksopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentHeading</name>
            <comment>
this element is a structural container for the section of an amendment containing the heading</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="amendmentContent" type="blocksopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentContent</name>
            <comment>
this element is a structural container for the section of an amendment containing the actual amendment text</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="amendmentReference" type="blocksopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentReference</name>
            <comment>
this element is a structural container for the section of an amendment containing the reference</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="amendmentJustification" type="blocksopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentJustification</name>
            <comment>
this element is a structural container for the section of an amendment containing the justification</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		AN block elements
		Comment: the only AN-specific block are the table of content (TOC),
		a structure containing toc items related to the various subparts of the
		main document, the list item and the titled block
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:complexType name="blockContainerType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>blockContainerType</name>
            <comment>
the element blockContainer is used as a container of many individual block elements in a block context</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:sequence>
					          <xsd:element ref="intro" minOccurs="0" maxOccurs="1"/>
					          <xsd:choice minOccurs="1" maxOccurs="unbounded">
						            <xsd:element ref="componentRef"/>
						            <xsd:group ref="blockElements"/>
					          </xsd:choice>
					          <xsd:element ref="wrapUp" minOccurs="0" maxOccurs="1"/>
				        </xsd:sequence>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="blockContainer" type="blockContainerType"/>
		


	  <xsd:element name="blockList">
      <xsd:annotation>
		       <xsd:documentation>
		 The element item is a container belonging to a list 
		</xsd:documentation>
	     </xsd:annotation>
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="listIntroduction" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="item" minOccurs="1" maxOccurs="unbounded"/>
				        <xsd:element ref="listWrapUp" minOccurs="0" maxOccurs="1"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="corereq"/>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="item" type="itemType"/>
		


	  <xsd:element name="listIntroduction" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>listIntroduction</name>
            <comment>
the element listIntroduction is an optional element of list before any item of the list itself.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="listWrapUp" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>listWrapUp</name>
            <comment>
the element listWrapUp is an optional element of list after all items of the list itself.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="tblock" type="itemType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>tblock</name>
            <comment>
The element tblock (titled block) is used to specify a container for blocks introduced by heading elements, similarly to a hierarchical structure</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="toc">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="tocItem"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="corereq"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="tocItem">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>tocItem</name>
            <comment>
the element tocItem is a component of the table of content and contains header information about sections or parts of the rest of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="link"/>
					          <xsd:attributeGroup ref="level"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="docType" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docType</name>
            <comment>
the element docType is an inline element within preface to identify the string used by the document for its own type</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docTitle" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docTitle</name>
            <comment>
the element docTitle is an inline element within preface to identify the string used by the document for its own title</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docNumber" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docNumber</name>
            <comment>
the element docNumber is an inline element within preface to identify the string used by the document for its own number</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docProponent">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docProponent</name>
            <comment>
the element docProponent is an inline element within preface to identify the string used by the document for its proponent</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="role"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="docDate">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docDate</name>
            <comment>
the element docDate is an inline element within preface to identify the string used by the document for its own date(s). Documents with multiple dates may use multiple docDate elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="date"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="legislature">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>legislature</name>
            <comment>
the element legislature is an inline element within preface to identify the string used by the document for the legislature relative to the document. Use #refersTo to a TLCEvent to refer to the event of the specific legislature.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="optvalue"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="session">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>session</name>
            <comment>
the element session is an inline element within preface to identify the string used by the document for the session of the legislature relative to the document. Use #refersTo to a TLCEvent to refer to the event of the specific session.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="optvalue"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="shortTitle" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>shortTitle</name>
            <comment>
the element shortTitle is an inline element within preface to identify the string used by the document for the short title of the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docAuthority" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docAuthority</name>
            <comment>
the element docAuthority is an inline element within preface to identify the string used by the document detailing the Auhtority to which the document was submitted</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docPurpose" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docPurpose</name>
            <comment>
the element docPurpose is an inline element within preface to identify the string used by the document detailing its own purpose</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docCommittee">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docCommittee</name>
            <comment>
the element docCommittee is an inline element within preface to identify the string used by the document detailing the committee within which the document originated</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="optvalue"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="docIntroducer" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docIntroducer</name>
            <comment>
the element docIntroducer is an inline element within preface to identify the string used by the document detailing the individual introducing of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docStage" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docStage</name>
            <comment>
the element docStage is an inline element within preface to identify the string used by the document detailing the stage in which the document sits</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docStatus" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docStatus</name>
            <comment>
the element docStatus is an inline element within preface to identify the string used by the document detailing the status of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docJurisdiction" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docJurisdiction</name>
            <comment>
the element docJurisdiction is an inline element within preface to identify the string used by the document detailing the jurisdiction of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docketNumber" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docketNumber</name>
            <comment>
the element docketNumber is an inline element within preface to identify the string used by the document for the number of the docket, case, file, etc which the document belongs to</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="courtType" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>courtType</name>
            <comment>
the element courtType is an inline element within judgments to identify the string used by the document for the type of the court doing the judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="neutralCitation" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>neutralCitation</name>
            <comment>
the element neutralCitation is an inline element within judgments to identify the string declared by the document as being the neutral citation for the judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="party">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>party</name>
            <comment>
the element party is an inline element within judgments to identify where the document defines one of the parties</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="role"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="judge">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>judge</name>
            <comment>
the element judge is an inline element within judgments to identify where the document defines one of the judges, and his/her role</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="role"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="lawyer">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>lawyer</name>
            <comment>
the element lawyer is an inline element within judgments to identify where the document defines one of the lawyers, his/her role, which party it represents, and the other lawyer, if any, this lawyer received the power delegation of power in some role</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="lawyerAtts"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="opinion">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>opinion</name>
            <comment>
the element opinion is an inline element within judgments to identify where the document defines the opinion of one of the judges</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="opinionType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="argument" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>argument</name>
            <comment>
the element argument is an inline element within judgments for classifying the arguments in the motivation part of the judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="signature" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>signature</name>
            <comment>
the element signature is an inline element within conclusions to identify where the document defines one of the signatures</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="date">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>date</name>
            <comment>
the element date is an inline element to identify a date expressed in the text and to propose a normalized representation in the date attribute.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="date"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="time">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>time</name>
            <comment>
the element time is an inline element to identify a time expressed in the text and to propose a normalized representation in the time attribute.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="time"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="entity">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>entity</name>
            <comment>
the element entity is a generic inline element to identify a text fragment introducing or referring to a concept in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="person">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>person</name>
            <comment>
the element person is an inline element to identify a text fragment introducing or referring to a person in the ontology. Attribute as allows to specify a TLCrole the person is holding in the context of the document's mention</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="role"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="organization" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>organization</name>
            <comment>
The element organization is an inline element to identify a text fragment introducing or referring to an organization in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="concept" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>concept</name>
            <comment>
The element concept is is an inline element to identify a text fragment introducing or referring to a concept in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="object" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>object</name>
            <comment>
The element object is is an inline element to identify a text fragment introducing or referring to an object in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="event" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>event</name>
            <comment>
The element event is an inline element to identify a text fragment introducing or referring to an event in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="location" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>location</name>
            <comment>
The element location is an inline element to identify a text fragment introducing or referring to a location in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="process" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>process</name>
            <comment>
The element process is an inline element to identify a text fragment introducing or referring to a process in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="role" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>role</name>
            <comment>
The element role is an inline element to identify a text fragment introducing or referring to a role in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="term" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>term</name>
            <comment>
The element term is an inline element to identify a text fragment introducing or referring to a term in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="quantity">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>quantity</name>
            <comment>
The element quantity is an inline element to identify a text fragment introducing or referring to a quantity. This could be a dimensionless number, or a number referring to a length, weight, duration, etc. or even a sum of money. The attribute normalized contains the value normalized in a number, if appropriate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="normalizedAtt"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="def" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>def</name>
            <comment>
the element def is an inline element used for the definition of a term used in the rest of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="ref">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>ref</name>
            <comment>
the element ref is an inline element containing a legal references (i.e. a reference to a document with legal status and for which an Akoma Ntoso IRI exists)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereq">
					          <xsd:attributeGroup ref="link"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="mref" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>mref</name>
            <comment>
the element mref is an inline element containing multiple references (each in turn represented by a ref element)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="rref">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>rref</name>
            <comment>
the element rref is an inline element containing a range of references between the IRI specified in the from attribute and the one specified in the upTo attribute.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereq">
					          <xsd:attributeGroup ref="range"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="modType" mixed="true">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>modType</name>
            <comment>
the complex type modType specifies the content that is allowed within mod, mmod and rmod elements, i.e. it adds quotedText and quotedStructure to the normal list of inline elements. Attribute for is used to point to the eId of the corresponding ref element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="inlineCM"/>
			      <xsd:element ref="quotedText"/>
			      <xsd:element ref="quotedStructure"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="corereq"/>
		    <xsd:attributeGroup ref="for"/>
	  </xsd:complexType>

	  <xsd:element name="mod" type="modType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>mod</name>
            <comment>
the element mod is an inline element containing the specification of a modification on another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="mmod" type="modType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>mmod</name>
            <comment>
the element mmod is an inline element containing multiple specifications of modifications on another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="rmod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>rmod</name>
            <comment>
the element rmod is an inline element containing the specification of a range of modifications on another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="modType">
					          <xsd:attributeGroup ref="range"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="quotedText">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>quotedText</name>
            <comment>
the element quotedText is an inline element containing a small string that is used either as the text being replaced, or the replacement, or the positioning at which some modification should take place. Attribute quote is used to specify the quote character used in the original; no quote attribute implies that the quote is left in the text; quote="" implies that there is no quote character. Attribute for is used to point to the eId of the corresponding ref element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="quote"/>
					          <xsd:attributeGroup ref="for"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="remark">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>remark</name>
            <comment>
the element remark is an inline element for the specification of editorial remarks (e.g., applauses, laughters, etc.) especially within debate records</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="remarkType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="recordedTime">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>recordedTime</name>
            <comment>
the element recordedTime is an inline element for the specification of an explicit mention of a time (e.g., in a debate)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="recordedTimeType"/>
					          <xsd:attributeGroup ref="time"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="ins" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>ins</name>
            <comment>
the element ins is an inline element for the specification of editorial insertions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="del" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>del</name>
            <comment>
the element del is an inline element for the specification of editorial deletions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="omissis" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>omissis</name>
            <comment>
the element omissis is an inline element for the specification of a text that substitutes a textual omission (e.g., dots, spaces, the word "omissis", etc.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="placeholder">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>placeholder</name>
            <comment>
the element placeholder is an inline element containing the text of a computable expression (e.g., '30 days after the publication of this act') that can be replaced editorially with an actual value</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="originalText"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="fillIn">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>fillIn</name>
            <comment>
the element fillIn is an inline element shown as a dotted line or any other typoaphical characteristics to represent a fill-in element in a printed form, that is as ane example of an actual form. It is NOT meant to be used for form elements as in HTML, i.e. as a way to collect input from the reader and deliver to some server-side process.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="fillInWidth"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="decoration" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>decoration</name>
            <comment>
the element decoration is an inline element to represent a decorative aspect that is present in the orignal text and that is meant as additional information to the text (e.g., the annotation 'new' on the side of a freshly inserted structure.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		AN marker elements
		Comment: AN-specific marker elements are connected to the
		identification of locations within the legal structure, anywhere where
		there is text.
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:element name="noteRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>noteRef</name>
            <comment>
the element noteRef is a reference to a editorial note placed in the notes metadata section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="markeropt">
					          <xsd:attributeGroup ref="notes"/>
					          <xsd:attributeGroup ref="link"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="eolType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>eolType</name>
            <comment>
the complex type eolType is shared by eol and eop elements as being able to specify a hyphen character and a position within the next word in which the break can happen, and the number if any, associated to the page or line at issue</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="markeropt">
				        <xsd:attributeGroup ref="number"/>
				        <xsd:attribute name="breakAt" type="xsd:integer"/>
				        <xsd:attribute name="breakWith" type="xsd:string"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="eol" type="eolType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>eol</name>
            <comment>
the element eol (end of line) is a marker for where in the original text the line breaks. If the line breaks within a word, place the element BEFORE the word and place the number of characters before the break in the attribute breakAt. One can also specify, if relevant, the hyphen or other character used to signal
the break of a word at the end of the line with the attribute breakWith.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="eop" type="eolType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>eop</name>
            <comment>
the element eop (end of page) is a marker for where in the original text the page breaks. Do NOT use a eol element, too. If the page breaks within a word, place the element BEFORE the word and place the number of characters before the break in the attribute breakAt. One can also specify, if relevant, the hyphen or other character used to signal the break of a word at the end of the page with the attribute breakWith.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		AN subFlow elements
		Comment: AN-specific subFlow elements are connected to the
		identification of structural elements that contain full structures
		within a text content, but are not of textual nature.
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:complexType name="subFlowStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>subFlowStructure</name>
            <comment>
the type subFlowStructure specifies the overall content model of the elements that are subFlows</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="documentType"/>
			      <xsd:choice minOccurs="1" maxOccurs="unbounded">
				        <xsd:group ref="blockElements"/>
				        <xsd:group ref="containerElements"/>
				        <xsd:element ref="tr"/>
				        <xsd:element ref="th"/>
				        <xsd:element ref="td"/>
				        <xsd:group ref="hierElements"/>
				        <xsd:element ref="content"/>
				        <xsd:group ref="ANcontainers"/>
				        <xsd:element ref="formula"/>
				        <xsd:element ref="recitals"/>
				        <xsd:element ref="citations"/>
				        <xsd:element ref="longTitle"/>
				        <xsd:element ref="recital"/>
				        <xsd:element ref="citation"/>
				        <xsd:element ref="componentRef"/>
				        <xsd:element ref="documentRef"/>
				        <xsd:element ref="intro"/>
				        <xsd:element ref="wrapUp"/>
				        <xsd:element ref="heading"/>
				        <xsd:element ref="subheading"/>
				        <xsd:element ref="num"/>
			      </xsd:choice>
		    </xsd:choice>
		    <xsd:attributeGroup ref="corereq"/>
	  </xsd:complexType>

	  <xsd:element name="quotedStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>quotedStructure</name>
            <comment>
the element quotedStructure is a subFlow element containing a full structure proposed as an insertion or a replacement. Attribute quote is used to specify the quote character used in the original; no quote attribute implies that the quote is left in the text; quote="" implies that there is no quote character. Attribute for is used in a mmod or rmod to point to the eId of the corresponding ref element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="subFlowStructure">
					          <xsd:attributeGroup ref="quote"/>
					          <xsd:attributeGroup ref="for"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="embeddedText">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>embeddedText</name>
            <comment>
the element embeddedText is an inline element containing a string used as an extract from another document. Attribute quote is used to specify the quote character used in the original; no quote attribute implies that the quote is left in the text; quote="" implies that there is no quote character.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="quote"/>
					          <xsd:attributeGroup ref="linkopt"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="embeddedStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>embeddedStructure</name>
            <comment>
the element embeddedStructure is a subFlow element containing a full structure used as an extract from another document or position. Attribute quote is used to specify the quote character used in the original; no quote attribute implies that the quote is left in the text; quote="" implies that there is no quote character. Attribute for is used in a mmod or rmod to point to the eId of the corresponding ref element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="subFlowStructure">
					          <xsd:attributeGroup ref="quote"/>
					          <xsd:attributeGroup ref="linkopt"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="authorialNote">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>authorialNote</name>
            <comment>
the element authorialNote is a subFlow element containing an authorial (non-editorial) note in the main flow of the text.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="subFlowStructure">
					          <xsd:attributeGroup ref="notes"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="subFlow">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subFlow</name>
            <comment>
the element subFlow is a generic element for a subFlow.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="subFlowStructure">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="foreign" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>foreign</name>
            <comment>
the element foreign is a generic container for elements not belonging to the Akoma Ntoso namespace (e.g., mathematical formulas). It is a block element and thus can be placed in a container.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Generic elements
		Comment: All elements in this schema fall under one of five
		content models: hierarchical container, container, block, inline
		and mark. Besides named elements, the schema also provides for a
		generic element for each of them, that can be used for markup that
		fits the content models but can be specified by a precise name that
		is not used in this schema. The 'name' attribute must be used for
		naming the element.
		It is also a characteristic of this schema that all named elements
		are equivalent, and can be substituted to, the corresponding generic
		element with their name as the value of the name attribute. Thus, by
		definition, &lt;p&gt;  = &lt;block name="p"&gt; , &lt;part&gt;  = &lt;hcontainer name="part"&gt; ,
		&lt;act&gt;  = &lt;container name="act"&gt; , etc.
		Pattern: Generic Document + Role Attribute
		Reference: http://www.xmlpatterns.com/GenericDocumentMain.shtml +
		http://www.xmlpatterns.com/RoleAttributeMain.shtml
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:element name="hcontainer">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>hcontainer</name>
            <comment>
the element hcontainer is a generic element for a hierarchical container. It can be placed in a hierarchy instead of any of the other hierarchical containers. The attribute name is required and gives a name to the element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="hierarchy">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="containerType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>containerType</name>
            <comment>
the complex type containerType is the content model for the generic element for a container. It can be placed in a container instead of any of the other containers. The attribute name is required and gives a name to the element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
			      <xsd:element ref="container"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="corereq"/>
		    <xsd:attributeGroup ref="name"/>
	  </xsd:complexType>

	  <xsd:element name="container" type="containerType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>container</name>
            <comment>
the element container is a generic element for a container.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="block">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>block</name>
            <comment>
the element block is a generic element for a container. It can be placed in a container instead of any of the other blocks. The attribute name is required and gives a name to the element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="inline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>inline</name>
            <comment>
the element inline is a generic element for an inline. It can be placed inside a block instead of any of the other inlines. The attribute name is required and gives a name to the element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="marker">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>marker</name>
            <comment>
the element marker is a generic element for a marker. It can be placed in a block instead of any of the other markers. The attribute name is required and gives a name to the element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="markerreq">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="div" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>div</name>
            <comment>
The element div is an HTML element, but is NOT used in Akoma Ntoso as in HTML. Instead of being used as a generic block, Akoma Ntoso uses div as a generic container (as in common practice)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="p" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>p</name>
            <comment>
The element p is an HTML element and is used in Akoma Ntoso as in HTML, for the generic paragraph of text (a block)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="span" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>span</name>
            <comment>
The element span is an HTML element and is used in Akoma Ntoso as in HTML, for the generic inline</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="br" type="markeropt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>br</name>
            <comment>
The element br is an HTML element and is used in Akoma Ntoso as in HTML, for the breaking of a line</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="b" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>b</name>
            <comment>
The element b is an HTML element and is used in Akoma Ntoso as in HTML, for the bold style (an inline)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="i" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>i</name>
            <comment>
The element i is an HTML element and is used in Akoma Ntoso as in HTML, for the italic style (an inline)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="u" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>u</name>
            <comment>
The element u is an HTML element and is used in Akoma Ntoso as in HTML, for the underline style (an inline)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="sup" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>sup</name>
            <comment>
The element sup is an HTML element and is used in Akoma Ntoso as in HTML, for the superscript style (an inline)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="sub" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>sub</name>
            <comment>
The element sub is an HTML element and is used in Akoma Ntoso as in HTML, for the subscript style (an inline)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="abbr" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>abbr</name>
            <comment>
The element abbr is an HTML element and is used in Akoma Ntoso as in HTML, for the specification of an abbreviation or an acronym (an inline). As in HTML, use attribute title to specify the full expansion of the abbreviation or acronym.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="a">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>a</name>
            <comment>
The element a is an HTML element and is used in Akoma Ntoso as in HTML, for the generic link to a web resource (NOT to an Akoma Ntoso document: use ref for that). It is an inline.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="link"/>
					          <xsd:attributeGroup ref="target"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="img">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>img</name>
            <comment>
The element img is an HTML element and is used in Akoma Ntoso as in HTML, for including an image. It is a marker.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="markeropt">
					          <xsd:attributeGroup ref="src"/>
					          <xsd:attributeGroup ref="imgAtts"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="listItems">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>listItems</name>
            <comment>
the complex type listItems specifies the content model of elements ul and ol, and specifies just a sequence of list items (elements li).</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="li" minOccurs="1" maxOccurs="unbounded"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="corereq"/>
	  </xsd:complexType>

	  <xsd:element name="ul" type="listItems">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>ul</name>
            <comment>
The element ul is an HTML element and is used in Akoma Ntoso as in HTML, for an unordered list of list item (elements li)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="ol" type="listItems">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>ol</name>
            <comment>
The element ol is an HTML element and is used in Akoma Ntoso as in HTML, for an ordered list of list item (elements li)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="li">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>li</name>
            <comment>
TYPE
Element
NAME
li
The element li is an HTML element and is used in Akoma Ntoso as in HTML, for the generic list item (not a pattern)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:choice minOccurs="0" maxOccurs="unbounded">
				        <xsd:group ref="inlineCM"/>
				        <xsd:element ref="ul"/>
				        <xsd:element ref="ol"/>
				        <xsd:element ref="p"/>
			      </xsd:choice>
				     <xsd:attributeGroup ref="optvalue"/>
				     <xsd:attributeGroup ref="coreopt"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="table">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="caption" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="tr" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="corereq"/>
				     <xsd:attributeGroup ref="tableAtts"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="caption" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>caption</name>
            <comment>
The element caption is an HTML element and is used in Akoma Ntoso as in HTML, for the caption of a table (a block)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="tr">
		    <xsd:complexType>
			      <xsd:choice minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="th"/>
				        <xsd:element ref="td"/>
			      </xsd:choice>
				     <xsd:attributeGroup ref="coreopt"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="th">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>th</name>
            <comment>
The element th is an HTML element and is used in Akoma Ntoso as in HTML, for a header cell of a table</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="blocksopt">
					          <xsd:attributeGroup ref="cellattrs"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="td">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>td</name>
            <comment>
The element td is an HTML element and is used in Akoma Ntoso as in HTML, for a data cell of a table</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="blocksopt">
					          <xsd:attributeGroup ref="cellattrs"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="meta">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="identification"/>
				        <xsd:element ref="publication" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="classification" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="lifecycle" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="workflow" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="analysis" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="temporalData" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="references" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="notes" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="proprietary" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="presentation" minOccurs="0" maxOccurs="unbounded"/>
			      </xsd:sequence>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="identification">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="FRBRWork"/>
				        <xsd:element ref="FRBRExpression"/>
				        <xsd:element ref="FRBRManifestation"/>
				        <xsd:element ref="FRBRItem" minOccurs="0" maxOccurs="1"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="coreProperties">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>coreProperties</name>
            <comment>
The complexType coreProperties lists the identifying properties available at any of the FRBR hierarchy levels.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="FRBRthis"/>
			      <xsd:element ref="FRBRuri" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRalias" minOccurs="0" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRdate" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRauthor" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="componentInfo" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preservation" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
	  </xsd:complexType>

	  <xsd:group name="workProperties">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>workProperties</name>
            <comment>
The group workProperties lists the properties that are characteristics of the FRBR Work level.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="FRBRcountry"/>
			      <xsd:element ref="FRBRsubtype" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRnumber" minOccurs="0" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRname" minOccurs="0" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRprescriptive" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRauthoritative" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
	  </xsd:group>

	  <xsd:group name="exprProperties">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>exprProperties</name>
            <comment>
The group exprProperties lists the properties that are characteristics of the FRBR Expression level.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="FRBRversionNumber" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRauthoritative" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRmasterExpression" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRlanguage" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRtranslation" minOccurs="0" maxOccurs="unbounded"/>
		    </xsd:sequence>
	  </xsd:group>

	  <xsd:group name="manifProperties">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>manifProperties</name>
            <comment>
The group manifProperties lists the properties that are characteristics of the FRBR Expression level.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="FRBRportion" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRformat" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
	  </xsd:group>

	  <xsd:element name="FRBRWork">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRWork</name>
            <comment>
The element FRBRWork is the metadata container of identifying properties related to the Work level according to the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="coreProperties">
					          <xsd:sequence>
						            <xsd:group ref="workProperties"/>
					          </xsd:sequence>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRExpression">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRExpression</name>
            <comment>
The element FRBRExpression is the metadata container of identifying properties related to the Expression level according to the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="coreProperties">
					          <xsd:sequence>
						            <xsd:group ref="exprProperties"/>
					          </xsd:sequence>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRManifestation">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRManifestation</name>
            <comment>
The element FRBRManifestation is the metadata container of identifying properties related to the Manifestation level according to the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="coreProperties">
					          <xsd:sequence>
						            <xsd:group ref="manifProperties"/>
					          </xsd:sequence>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRItem" type="coreProperties">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRItem</name>
            <comment>
The element FRBRItem is the metadata container of identifying properties related to the Item level according to the FRBR hierarchy.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="valueType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>valueType</name>
            <comment>
The type valueType specifies a value attribute to FRBR elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="metaopt">
				        <xsd:attributeGroup ref="value"/>
				        <xsd:attributeGroup ref="refers"/>
				        <xsd:attributeGroup ref="showopt"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="booleanValueType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>booleanValueType</name>
            <comment>
The type booleanValueType specifies a boolean value attribute to FRBR elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="metaopt">
				        <xsd:attributeGroup ref="booleanvalue"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="FRBRthis" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRthis</name>
            <comment>
The element FRBRthis is the metadata property containing the IRI of the specific component of the document in the respective level of the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRuri" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRuri</name>
            <comment>
The element FRBRuri is the metadata property containing the IRI of the whole document in the respective level of the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRalias">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRalias</name>
            <comment>
The element FRBRalias is the metadata property containing additional well-known names of the document in the respective level of the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="valueType">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRdate">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRdate</name>
            <comment>
The element FRBRdate is the metadata property containing a relevant date of the document in the respective level of the FRBR hierarchy. Attribute name specifies which actual date is contained here.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="date"/>
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRauthor">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRauthor</name>
            <comment>
The element FRBRauthor is the metadata property containing a relevant author of the document in the respective level of the FRBR hierarchy. Attribute as specifies the role of the author.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="link"/>
					          <xsd:attributeGroup ref="role"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRlanguage">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRlanguage</name>
            <comment>
The element FRBRlanguage is the metadata property containing a RFC4646 (three-letter code) of the main human language used in the content of this expression</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="language"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRtranslation">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRtranslation</name>
            <comment>
The element FRBRtranslation is the metadata property specifying the source of which this expression is a translation of.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="link"/>
					          <xsd:attributeGroup ref="fromLanguage"/>
					          <xsd:attributeGroup ref="authoritative"/>
					          <xsd:attributeGroup ref="pivot"/>
					          <xsd:attributeGroup ref="agent"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRsubtype" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRsubtype</name>
            <comment>
The element FRBRsubtype is the metadata property containing a string for the specific subtype of the document to be used in the work-level IRI of this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRcountry" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRcountry</name>
            <comment>
The element FRBRcountry is the metadata property containing a ISO 3166-1 Alpha-2 code for the country or jurisdiction to be used in the work-level IRI of this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRnumber" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRnumber</name>
            <comment>
The element FRBRnumber is the metadata property containing a string or number for the number to be used in the work-level IRI of this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRname" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRname</name>
            <comment>
The element FRBRname is the metadata property containing a string for the title to be used in the work-level IRI of this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRportion">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRportion</name>
            <comment>
The element FRBRportion is the metadata property containing the eId of the portion contained in a manifestation-level portion. If the portion contains an interval of elements, the range attributes specifies the first and last one.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="refers"/>
					          <xsd:attributeGroup ref="showopt"/>
					          <xsd:attributeGroup ref="rangeOpt"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRformat" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRformat</name>
            <comment>
The element FRBRformat is the metadata property containing the Internet Media Type specification for the data format to be used in the manifestation-level IRI of this document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRprescriptive" type="booleanValueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRprescriptive</name>
            <comment>
The element FRBRprescriptive is the metadata property containing a boolean value to determine whether the document contains prescriptive text (i.e., text that is or might become prescriptive, such as an act or a bill) or not (such as, for instance, a non-normative resolution from an assembly.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRversionNumber" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRversionNumber</name>
            <comment>
The element FRBRversionNumber is the metadata property containing the value of the version number if appropriate to identify the specific expression here contained. It allows an arbitrary string.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRauthoritative" type="booleanValueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRauthoritative</name>
            <comment>
The element FRBRauthoritative is the metadata property containing a boolean value to determine whether the document contains authoritative text (i.e., content that is the official, authoritative product of an official workflow from an entity that is entrusted with generating an official, authoriative version of the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRmasterExpression" type="linkType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRmasterExpression</name>
            <comment>
The element FRBRmasterExpression is the metadata property identifying the master expression, i.e., the expression whose ids are used as permanent ids in the wId attributes. An expression without the FRBRmasterExpression element is considered a master expression itself, i.e., the first version, or the most important version, of a document expressed in the only language, or in the most important language. Any other situation (subsequent versions, or language variants, or content variants) must have the FRBRmasterExpression element pointing to the URI of the master expression. If the FRBRmasterEpression is specified, but without a href pointing to the masterExpression, it is assumed that NO master expression exist in reality, but an UR-Expression exist, whose ids are used in this expression as wIds.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="componentInfo">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="0" maxOccurs="unbounded">
				        <xsd:element ref="componentData"/>
			      </xsd:sequence>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="componentData">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="0" maxOccurs="unbounded">
				        <xsd:element ref="componentData"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="core"/>
				     <xsd:attributeGroup ref="idreq"/>
				     <xsd:attributeGroup ref="name"/>
				     <xsd:attributeGroup ref="link"/>
				     <xsd:attributeGroup ref="show"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="preservation" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>preservation</name>
            <comment>
The element preservation is the metadata property containing an arbitrary list of elements detailing the preservation actions taken for the document is the respective level of the FRBR hierarchy..</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="publication">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>publication</name>
            <comment>
The element publication is the metadata container specifying an official publication event for the FRBR expression of the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="date"/>
					          <xsd:attributeGroup ref="show"/>
					          <xsd:attributeGroup ref="name"/>
					          <xsd:attributeGroup ref="number"/>
					          <xsd:attributeGroup ref="refers"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="classification">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="keyword" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="keyword">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>keyword</name>
            <comment>
The element keyword is a metadata element specifying a keyword  associated to the FRBR expression of the document. Attribute dictionary (required) specifies the thesaurus out of which the keyword has been taken. Attribute href points to the fragment of text this keyword is associated to. Keywords without href attribute refer to the content as a whole.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="linkopt"/>
					          <xsd:attributeGroup ref="value"/>
					          <xsd:attributeGroup ref="show"/>
					          <xsd:attributeGroup ref="dictionary"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="lifecycle">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="eventRef" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="eventRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>eventRef</name>
            <comment>
The element eventInfo is a metadata element specifying facts about an event that had an effect on the document. For each event, a date, a type and a document that generated the event must be referenced.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="date"/>
					          <xsd:attributeGroup ref="source"/>
					          <xsd:attributeGroup ref="refers"/>
					          <xsd:attributeGroup ref="eventType"/>
					          <xsd:attributeGroup ref="originating"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="workflow">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="step" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="step">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>step</name>
            <comment>
The element step is a metadata element specifying facts about a workflow step occurred to the document. For each event, a date, a type, an agent (and the corresponding role) that generated the action must be referenced. The outcome, too, can be specified.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="date"/>
					          <xsd:attributeGroup ref="agent"/>
					          <xsd:attributeGroup ref="role"/>
					          <xsd:attributeGroup ref="refers"/>
					          <xsd:attributeGroup ref="outcome"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="analysis">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="activeModifications" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="passiveModifications" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="restrictions" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="judicial" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="parliamentary" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="mappings" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="otherReferences" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="otherAnalysis" minOccurs="0" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="Amendments">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>Amendments</name>
            <comment>
The complex type Amendments is a list of all the amendment elements that can be used on a document analysis</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:element ref="textualMod"/>
			      <xsd:element ref="meaningMod"/>
			      <xsd:element ref="scopeMod"/>
			      <xsd:element ref="forceMod"/>
			      <xsd:element ref="efficacyMod"/>
			      <xsd:element ref="legalSystemMod"/>
		    </xsd:choice>
	  </xsd:complexType>

	  <xsd:complexType name="modificationType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>modificationType</name>
            <comment>
The complex type modificationType lists all the properties associated to modification elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="source" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="destination" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="force" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="efficacy" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="application" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="duration" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="condition" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="enactment"/>
		    <xsd:attributeGroup ref="modifiers"/>
		    <xsd:attributeGroup ref="refers"/>
	  </xsd:complexType>

	  <xsd:element name="activeModifications" type="Amendments">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>activeModifications</name>
            <comment>
The element activeModifications is a metadata container of the active modifications generated by the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="passiveModifications" type="Amendments">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>passiveModifications</name>
            <comment>
The element passiveModifications is a metadata container of the passive modifications affecting the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="textualMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>textualMod</name>
            <comment>
The element textualMod is a metadata element specifying an (active or passive) textual modification for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:sequence>
						            <xsd:element ref="previous" minOccurs="0" maxOccurs="1"/>
						            <xsd:element ref="old" minOccurs="0" maxOccurs="unbounded"/>
						            <xsd:element ref="new" minOccurs="0" maxOccurs="unbounded"/>
					          </xsd:sequence>
					          <xsd:attributeGroup ref="textualModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="meaningMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>meaningMod</name>
            <comment>
The element meaningMod is a metadata element specifying an (active or passive) modification in meaning for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:sequence>
						            <xsd:element ref="domain" minOccurs="0" maxOccurs="1"/>
					          </xsd:sequence>
					          <xsd:attributeGroup ref="meaningModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="scopeMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>scopeMod</name>
            <comment>
The element scopeMod is a metadata element specifying an (active or passive) modification in scope for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:sequence>
						            <xsd:element ref="domain" minOccurs="0" maxOccurs="1"/>
					          </xsd:sequence>
					          <xsd:attributeGroup ref="scopeModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="forceMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>forceMod</name>
            <comment>
The element forceMod is a metadata element specifying an (active or passive) modification in force for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:attributeGroup ref="forceModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="efficacyMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>efficacyMod</name>
            <comment>
The element efficacyMod is a metadata element specifying an (active or passive) modification in efficacy for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:attributeGroup ref="efficacyModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="legalSystemMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>legalSystemMod</name>
            <comment>
The element legalSystemMod is a metadata element specifying an (active or passive) modification in the legal system for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:attributeGroup ref="legalSystemModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="judicialArguments">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>judicialArguments</name>
            <comment>
The complex type judicialArguments is a list of all the judicial analysis elements that can be used on the analysis of a judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="result"/>
			      <xsd:choice minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="supports"/>
				        <xsd:element ref="isAnalogTo"/>
				        <xsd:element ref="applies"/>
				        <xsd:element ref="extends"/>
				        <xsd:element ref="restricts"/>
				        <xsd:element ref="derogates"/>
				        <xsd:element ref="contrasts"/>
				        <xsd:element ref="overrules"/>
				        <xsd:element ref="dissentsFrom"/>
				        <xsd:element ref="putsInQuestion"/>
				        <xsd:element ref="distinguishes"/>
			      </xsd:choice>
		    </xsd:sequence>
	  </xsd:complexType>

	  <xsd:complexType name="judicialArgumentType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>judicialArgumentType</name>
            <comment>
The complex type judicialArgumentType lists all the properties associated to judicial elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="source" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="destination" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="condition" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="enactment"/>
		    <xsd:attributeGroup ref="modifiers"/>
		    <xsd:attributeGroup ref="refers"/>
	  </xsd:complexType>

	  <xsd:element name="judicial" type="judicialArguments">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>judicial</name>
            <comment>
The element judicial is a metadata container of the analysis of the judicial arguments of a judgment.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="result">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>result</name>
            <comment>
The element result is a metadata element specifying the overall result of the judgment.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="resultType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="supports" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>supports</name>
            <comment>
The element supports is a metadata element specifying a reference to a source supported by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="isAnalogTo" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>isAnalogTo</name>
            <comment>
The element isAnalogTo is a metadata element specifying a reference to a source analog to the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="applies" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>applies</name>
            <comment>
The element applies is a metadata element specifying a reference to a source applyed by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="extends" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>extends</name>
            <comment>
The element extends is a metadata element specifying a reference to a source extended by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="restricts" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>restricts</name>
            <comment>
The element restricts is a metadata element specifying a reference to a source restricted by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="derogates" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>derogates</name>
            <comment>
The element derogates is a metadata element specifying a reference to a source derogated by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="contrasts" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>contrasts</name>
            <comment>
The element contrasts is a metadata element specifying a reference to a source contrasted by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="overrules" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>overrules</name>
            <comment>
The element overrules is a metadata element specifying a reference to a source overruled by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="dissentsFrom" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>dissentsFrom</name>
            <comment>
The element dissentsFrom is a metadata element specifying a reference to a source dissented from the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="putsInQuestion" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>putsInQuestion</name>
            <comment>
The element putsInQuestions is a metadata element specifying a reference to a source questioned by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="distinguishes" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>distinguishes</name>
            <comment>
The element distinguishes is a metadata element specifying a reference to a source being distinguished by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="restrictions">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="restriction"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="restriction">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>restriction</name>
            <comment>
The element restriction specifies information about a restriction (such as a jurisdiction specification) by pointing to a specific legislative, geographic or temporal events through the refersTo attribute</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="refers"/>
					          <xsd:attributeGroup ref="restrictionType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="parliamentaryAnalysis">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>parliamentaryAnalysis</name>
            <comment>
The complex type parliamentaryAnalysis is a list of all the parliamentary analysis elements that can be used on the analysis of a debate</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:element ref="quorumVerification"/>
			      <xsd:element ref="voting"/>
			      <xsd:element ref="recount"/>
		    </xsd:choice>
	  </xsd:complexType>

	  <xsd:element name="parliamentary" type="parliamentaryAnalysis">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>parliamentary</name>
            <comment>
The element parliamentary is a metadata container of the analysis of the events of a debate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="parliamentaryAnalysisType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>parliamentaryAnalysisType</name>
            <comment>
The complex type parliamentaryAnalysisType lists all the properties associated to elements in the parliamentary analysis.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:element ref="quorum"/>
			      <xsd:element ref="count"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="outcome"/>
		    <xsd:attributeGroup ref="refers"/>
		    <xsd:attributeGroup ref="linkopt"/>
	  </xsd:complexType>

	  <xsd:element name="quorumVerification" type="parliamentaryAnalysisType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>quorumVerification</name>
            <comment>
The element quorumVerification is a metadata container containing information about an event of quorum verification happened within a debate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="voting" type="parliamentaryAnalysisType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>voting</name>
            <comment>
The element voting is a metadata container containing information about an event of a vote happened within a debate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="recount" type="parliamentaryAnalysisType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>recount</name>
            <comment>
The element recount is a metadata container containing information about an event of a recount happened within a debate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="countType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>countType</name>
            <comment>
The complex type countType lists all the properties associated to elements of parliamentary count.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:any namespace="##other"
                  processContents="lax"
                  minOccurs="0"
                  maxOccurs="unbounded"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="value"/>
		    <xsd:attributeGroup ref="refers"/>
		    <xsd:attributeGroup ref="linkopt"/>
	  </xsd:complexType>

	  <xsd:element name="quorum" type="countType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>quorum</name>
            <comment>
The element quorum is a metadata container containing the value of a quorum in a vote or a quorum verification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="count" type="countType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>count</name>
            <comment>
The element count is a metadata container containing the value of a count in a vote or a quorum verification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="mappings">
      <xsd:annotation>
		       <xsd:documentation>
		 The element mapping contains a reference to the permanent wId (attribute original) of a structure, and to the eId (attribute current) such structure had during the time interval included between an initial temporalGroup and a final temporalGroup. This is useful for tracking the evolving ids of documents frequently renumbered (e,g., bills). Every single element whose wId does not match its eId needs to be represented here. 
		</xsd:documentation>
	     </xsd:annotation>
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="mapping"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="mapping">
		

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metareq">
					          <xsd:attributeGroup ref="mappingAtts"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="otherReferences">
      <xsd:annotation>
		       <xsd:documentation>
		 The element otherRef contains a legal reference to a document that is not explicitly mentioned in the content of the document. 
		</xsd:documentation>
	     </xsd:annotation>
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="otherRef"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="otherRef">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="source" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="core"/>
				     <xsd:attributeGroup ref="idreq"/>
				     <xsd:attributeGroup ref="enactment"/>
				     <xsd:attributeGroup ref="refers"/>
				     <xsd:attributeGroup ref="link"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="otherAnalysis">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>otherAnalysis</name>
            <comment>
The element otherAnalysis is a metadata container of any additional metadata analysis element that does not belong to the specific categories provided before. Anything can be placed in this element..</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="source"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="argumentType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>argumentType</name>
            <comment>
the complex type argumentType defines the empty content model and the list of attributes for metadata elements in the analysis section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="anyOtherType">
				        <xsd:attributeGroup ref="pos"/>
				        <xsd:attributeGroup ref="modifiers"/>
				        <xsd:attributeGroup ref="upToOpt"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="periodType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>periodType</name>
            <comment>
the complex type periodType defines the empty content model and the list of attributes for metadata elements in the analysis section using periods</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="anyOtherType">
				        <xsd:attributeGroup ref="period"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="source" type="argumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>source</name>
            <comment>
The element source is a metadata element specifying the IRI of the source of the modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="destination" type="argumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>destination</name>
            <comment>
The element destination is a metadata element specifying the IRI of the destination of the modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="force" type="periodType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>force</name>
            <comment>
The element force is a metadata element specifying the period of the force modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="efficacy" type="periodType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>efficacy</name>
            <comment>
The element efficacy is a metadata element specifying the period of the efficacy modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="application" type="periodType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>application</name>
            <comment>
The element application is a metadata element specifying the period of the application modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="duration" type="periodType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>duration</name>
            <comment>
The element duration is a metadata element specifying the period of the duration modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="condition">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>condition</name>
            <comment>
The element condition is a metadata element specifying an open set of conditions on the modification (non managed by Akoma Ntoso).</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="frozen"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="previous" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>previous</name>
            <comment>
The element previous is a metadata element referring to the element (rather than the text) of the modification in the previous version of the document. This is especially useful when renumbering occurs, so as to specify the eId of the instance that was modified in the previous version. Attribute href points to the eId of the element being modified in the old version, using a full expression-level URI.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="old" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>old</name>
            <comment>
The element old is a metadata element containing (in some non-managed form) the old text of the modification. Attribute href points to the eId of the element new it is being substituted by.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="new" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>new</name>
            <comment>
The element new is a metadata element containing (in some non-managed form) the new text of the modification. Attribute href points to the eId of the element old it is substituting.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="domain" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>domain</name>
            <comment>
The element domain is a metadata element containing (in some non-managed form) the domain to which the modification applies.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="temporalData">
      <xsd:annotation>
		       <xsd:documentation>
		 The element temporalGroup contains a list of time Intervals that need to be applied together to an element of the document that is associated to time information. Each temporal group can contain any number of time interval associated via the refersTo to a different time-specific concept. 
		</xsd:documentation>
	     </xsd:annotation>
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="temporalGroup" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="temporalGroup">
      <xsd:annotation>
		       <xsd:documentation>
		 The element timeInterval contains all the data needed to identify a specific time interval. It is built either with two dates or with a date and a duration (exactly two of the 'start', 'end' and 'duration' attributes can be specified). Values of the 'start' and 'end' attributes are NOT dates, but references to event elements in the corresponding metadata section. The refers attribute is a reference to a temporal concept belonging to the Akoma Ntoso ontology and specified in the references section 
		</xsd:documentation>
	     </xsd:annotation>
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="timeInterval" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="core"/>
				     <xsd:attributeGroup ref="idreq"/>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="timeInterval">
		

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="interval"/>
					          <xsd:attributeGroup ref="duration"/>
					          <xsd:attributeGroup ref="refersreq"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:group name="docRefs">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>docRefs</name>
            <comment>
The group docrefs is a list of types of legal references to documents.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="original"/>
			      <xsd:element ref="passiveRef"/>
			      <xsd:element ref="activeRef"/>
			      <xsd:element ref="jurisprudence"/>
			      <xsd:element ref="hasAttachment"/>
			      <xsd:element ref="attachmentOf"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="TLCs">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>TLCs</name>
            <comment>
The group TLCs is a list of types of Top Level classes of the Akoma Ntoso ontology.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="TLCPerson"/>
			      <xsd:element ref="TLCOrganization"/>
			      <xsd:element ref="TLCConcept"/>
			      <xsd:element ref="TLCObject"/>
			      <xsd:element ref="TLCEvent"/>
			      <xsd:element ref="TLCLocation"/>
			      <xsd:element ref="TLCProcess"/>
			      <xsd:element ref="TLCRole"/>
			      <xsd:element ref="TLCTerm"/>
			      <xsd:element ref="TLCReference"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:complexType name="refItems">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>refItems</name>
            <comment>
The complex type refItems is a list of types of references used in the references section.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="docRefs"/>
			      <xsd:group ref="TLCs"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="source"/>
	  </xsd:complexType>

	  <xsd:element name="references" type="refItems">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>references</name>
            <comment>
The element references is a metadata container of all the references to entities external to the document mentioned in the document. They include references to legal documents of any form,a s well as references to people, organizations, events, roles, concepts, and anything else is managed by the Akoma Ntoso ontology.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="original" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>original</name>
            <comment>
The element original is a metadata reference to the Akoma Ntoso IRI of the original version of this document (i.e., the first expression)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="passiveRef" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>passiveRef</name>
            <comment>
The element passiveRef is a metadata reference to the Akoma Ntoso IRI of a document providing modifications on this document (i.e., a passive references)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="activeRef" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>activeRef</name>
            <comment>
The element activeRef is a metadata reference to the Akoma Ntoso IRI of a document that is modified by this document (i.e., an active references)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="jurisprudence" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>jurisprudence</name>
            <comment>
The element jurisprudence is a metadata reference to the Akoma Ntoso IRI of a document providing jurisprudence on this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="hasAttachment">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>hasAttachment</name>
            <comment>
The element hasAttachment is a metadata reference to the Akoma Ntoso IRI of an attachment of this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="referenceType">
					          <xsd:attributeGroup ref="type"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="attachmentOf">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>attachmentOf</name>
            <comment>
The element attachmentOf is a metadata reference to the Akoma Ntoso IRI of a document of which this document is an attachment</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="referenceType">
					          <xsd:attributeGroup ref="type"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="TLCPerson" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCPerson</name>
            <comment>
The element TLCPerson is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Person</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCOrganization" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCOrganization</name>
            <comment>
The element TLCOrganization is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Organization</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCConcept" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCConcept</name>
            <comment>
The element TLCConcept is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Concept</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCObject" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCObject</name>
            <comment>
The element TLCObject is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Object</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCEvent" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCEvent</name>
            <comment>
The element TLCEvent is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Event</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCLocation" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCLocation</name>
            <comment>
The element TLCLocation is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Location</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCProcess" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCProcess</name>
            <comment>
The element TLCProcess is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Process</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCRole" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCRole</name>
            <comment>
The element TLCRole is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Role</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCTerm" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCTerm</name>
            <comment>
The element TLCTerm is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Term</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCReference">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCReference</name>
            <comment>
The element TLCreference is a generic metadata reference to the Akoma Ntoso IRI of an ontology instance of a class specified through the name attribute</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="referenceType">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="notes">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="note"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
				     <xsd:attributeGroup ref="xmllang"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="note" type="subFlowStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>note</name>
            <comment>
The element note is a metadata element containing the text of the footnote and endnote specified.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="proprietary">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>proprietary</name>
            <comment>
The element proprietary is a metadata container of any additional metadata property that does not belong to the Akoma Ntoso properties. Anything can be placed in this element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="source"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="presentation">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>presentation</name>
            <comment>
The element presentation is a metadata container of any presentation specification for the visual rendering of Akoam Ntoso elements. Anything can be placed in this element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="source"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	

</xsd:schema>
````

## File: tests/metadata/schemas/xml.xsd
````
<?xml version='1.0'?>
<!DOCTYPE xs:schema PUBLIC "-//W3C//DTD XMLSCHEMA 200102//EN" "XMLSchema.dtd" >
<xs:schema targetNamespace="http://www.w3.org/XML/1998/namespace" xmlns:xs="http://www.w3.org/2001/XMLSchema" xml:lang="en">

 <xs:annotation>
  <xs:documentation>
   See http://www.w3.org/XML/1998/namespace.html and
   http://www.w3.org/TR/REC-xml for information about this namespace.

    This schema document describes the XML namespace, in a form
    suitable for import by other schema documents.  

    Note that local names in this namespace are intended to be defined
    only by the World Wide Web Consortium or its subgroups.  The
    following names are currently defined in this namespace and should
    not be used with conflicting semantics by any Working Group,
    specification, or document instance:

    base (as an attribute name): denotes an attribute whose value
         provides a URI to be used as the base for interpreting any
         relative URIs in the scope of the element on which it
         appears; its value is inherited.  This name is reserved
         by virtue of its definition in the XML Base specification.

    lang (as an attribute name): denotes an attribute whose value
         is a language code for the natural language of the content of
         any element; its value is inherited.  This name is reserved
         by virtue of its definition in the XML specification.
  
    space (as an attribute name): denotes an attribute whose
         value is a keyword indicating what whitespace processing
         discipline is intended for the content of the element; its
         value is inherited.  This name is reserved by virtue of its
         definition in the XML specification.

    Father (in any context at all): denotes Jon Bosak, the chair of 
         the original XML Working Group.  This name is reserved by 
         the following decision of the W3C XML Plenary and 
         XML Coordination groups:

             In appreciation for his vision, leadership and dedication
             the W3C XML Plenary on this 10th day of February, 2000
             reserves for Jon Bosak in perpetuity the XML name
             xml:Father
  </xs:documentation>
 </xs:annotation>

 <xs:annotation>
  <xs:documentation>This schema defines attributes and an attribute group
        suitable for use by
        schemas wishing to allow xml:base, xml:lang or xml:space attributes
        on elements they define.

        To enable this, such a schema must import this schema
        for the XML namespace, e.g. as follows:
        &lt;schema . . .>
         . . .
         &lt;import namespace="http://www.w3.org/XML/1998/namespace"
                    schemaLocation="http://www.w3.org/2001/03/xml.xsd"/>

        Subsequently, qualified reference to any of the attributes
        or the group defined below will have the desired effect, e.g.

        &lt;type . . .>
         . . .
         &lt;attributeGroup ref="xml:specialAttrs"/>
 
         will define a type which will schema-validate an instance
         element with any of those attributes</xs:documentation>
 </xs:annotation>

 <xs:annotation>
  <xs:documentation>In keeping with the XML Schema WG's standard versioning
   policy, this schema document will persist at
   http://www.w3.org/2001/03/xml.xsd.
   At the date of issue it can also be found at
   http://www.w3.org/2001/xml.xsd.
   The schema document at that URI may however change in the future,
   in order to remain compatible with the latest version of XML Schema
   itself.  In other words, if the XML Schema namespace changes, the version
   of this document at
   http://www.w3.org/2001/xml.xsd will change
   accordingly; the version at
   http://www.w3.org/2001/03/xml.xsd will not change.
  </xs:documentation>
 </xs:annotation>

 <xs:attribute name="lang" type="xs:language">
  <xs:annotation>
   <xs:documentation>In due course, we should install the relevant ISO 2- and 3-letter
         codes as the enumerated possible values . . .</xs:documentation>
  </xs:annotation>
 </xs:attribute>

 <xs:attribute name="space" default="preserve">
  <xs:simpleType>
   <xs:restriction base="xs:NCName">
    <xs:enumeration value="default"/>
    <xs:enumeration value="preserve"/>
   </xs:restriction>
  </xs:simpleType>
 </xs:attribute>

 <xs:attribute name="base" type="xs:anyURI">
  <xs:annotation>
   <xs:documentation>See http://www.w3.org/TR/xmlbase/ for
                     information about this attribute.</xs:documentation>
  </xs:annotation>
 </xs:attribute>

 <xs:attributeGroup name="specialAttrs">
  <xs:attribute ref="xml:base"/>
  <xs:attribute ref="xml:lang"/>
  <xs:attribute ref="xml:space"/>
 </xs:attributeGroup>

</xs:schema>
````

## File: tests/parsers/html/test_cellar_html.py
````python
import unittest
import os
from tulit.parsers.html.cellar import CellarHTMLParser
import json

DATA_DIR = os.path.join(os.path.dirname(__file__), "..\\..\\data\\html")
file_path = os.path.join(DATA_DIR, "c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006.03\\DOC_1.html")


class TestCellarHTMLParser(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None  # Allow full diff if needed
        self.parser = CellarHTMLParser()
        
        # Ensure test file exists
        if not os.path.exists(file_path):
            raise FileNotFoundError(f"Test file not found at {file_path}")
        self.parser.get_root(file_path)
    
    def test_get_root(self):
        """Test parsing and root element retrieval from the Akoma Ntoso file."""
        self.assertTrue(os.path.exists(file_path), f"Test file not found at {file_path}")
        self.assertIsNotNone(self.parser.root, "Root element should not be None")      
    
    def test_get_body(self):
        self.parser.get_body()
        self.assertIsNotNone(self.parser.body, "Body element should not be None")      
    
    def test_get_preface(self):
        self.parser.get_preface()
        self.assertEqual(self.parser.preface, "REGULATION (EU) 2024/903 OF THE EUROPEAN PARLIAMENT AND OF THE COUNCIL of 13 March 2024 laying down measures for a high level of public sector interoperability across the Union (Interoperable Europe Act)")
    
    def test_get_preamble(self):
        self.parser.get_preamble()
        self.assertIsNotNone(self.parser.preamble, "Preamble element should not be None")      

    def test_get_formula(self):
        self.parser.get_preamble()
        self.parser.get_formula()
        formula = "THE EUROPEAN PARLIAMENT AND THE COUNCIL OF THE EUROPEAN UNION,"
        self.assertEqual(self.parser.formula, formula, "Formula should match expected value")

    def test_get_citations(self):
        self.parser.get_preamble()
        self.parser.get_citations()
        citations =  [
            {
                "eId": "cit_1",
                "text": "Having regard to the Treaty on the Functioning of the European Union, and in particular Article 172 thereof,"
            },
            {
                "eId": "cit_2",
                "text": "Having regard to the proposal from the European Commission,"
            },
            {
                "eId": "cit_3",
                "text": "After transmission of the draft legislative act to the national parliaments,"
            },
            {
                "eId": "cit_4",
                "text": "Having regard to the opinion of the European Economic and Social Committee,"
            },
            {
                "eId": "cit_5",
                "text": "Having regard to the opinion of the Committee of the Regions,"
            },
            {
                "eId": "cit_6",
                "text": "Acting in accordance with the ordinary legislative procedure,"
            }
        ]
        self.assertEqual(self.parser.citations, citations, "Citations should match expected values")
    
    def test_get_recitals(self):
        self.parser.get_preamble()
        self.parser.get_recitals()
        recital = {
            "eId": "rct_1",
            "text": "It is necessary to strengthen the development of the cross-border interoperability of network and information systems which are used to provide or manage public services in the Union, in order to allow public administrations in the Union to cooperate and make public services function across borders. The existing informal cooperation should be replaced by a clear legal framework to enable interoperability across different administrative levels and sectors and to facilitate seamless cross-border data flows for truly European digital services that strengthen the internal market while respecting the principle of subsidiarity. Public sector interoperability has an important impact on the right to free movement of goods, persons, services and capital laid down in the Treaties, as burdensome administrative procedures can create significant obstacles, especially for small and medium-sized enterprises (SMEs)."
        }
        self.assertEqual(self.parser.recitals[0], recital, "Recitals element should match expected value")
        

    def test_get_preamble_final(self):
        self.parser.get_preamble()
        self.parser.get_preamble_final()
        preamble_final = "HAVE ADOPTED THIS REGULATION:"
        self.assertEqual(self.parser.preamble_final, preamble_final, "Preamble final should match expected value")
    
    def test_get_chapters(self):
        self.parser.get_body()
        self.parser.get_chapters()
        chapters = [
            {
                "eId": "cpt_1",
                "num": "Chapter 1",
                "heading": "General provisions"
            },
            {
                "eId": "cpt_2",
                "num": "Chapter 2",
                "heading": "European Interoperability enablers"
            },
            {
                "eId": "cpt_3",
                "num": "Chapter 3",
                "heading": "Interoperable Europe support measures"
            },
            {
                "eId": "cpt_4",
                "num": "Chapter 4",
                "heading": "Governance of cross-border interoperability"
            },
            {
                "eId": "cpt_5",
                "num": "Chapter 5",
                "heading": "Interoperable Europe planning and monitoring"
            },
            {
                "eId": "cpt_6",
                "num": "Chapter 6",
                "heading": "Final provisions"
            }
        ]
        self.assertEqual(self.parser.chapters, chapters, "Chapters elements should match expected values")        
        

    def test_get_articles(self):
        """Test parsing articles from an HTML file."""
        # Parse the body and articles using the parser
        self.parser.get_body()
        self.parser.get_articles()
        article_1 = {
            "eId": "art_1",
            "num": "Article 1",
            "heading": "Subject matter and scope",
            "children": [
                {
                    "eId": "001.001",
                    "text": "1. This Regulation lays down measures that promote the cross-border interoperability of trans-European digital public services, thus contributing to the interoperability of the underlying network and information systems by establishing common rules and a governance framework."
                },
                {
                    "eId": "001.002",
                    "text": "2. This Regulation applies to Union entities and public sector bodies that regulate, provide, manage or implement trans-European digital public services."
                },
                {
                    "eId": "001.003",
                    "text": "3. This Regulation applies without prejudice to the competence of Member States to define what constitutes public services or to their ability to establish procedural rules for or to provide, manage or implement those services."
                },
                {
                    "eId": "001.004",
                    "text": "4. This Regulation is without prejudice to the competence of Member States with regard to their activities concerning public security, defence and national security."
                },
                {
                    "eId": "001.005",
                    "text": "5. This Regulation does not entail the supply of information the disclosure of which would be contrary to the essential interests of Member States’ public security, defence or national security."
                }
            ]
        }
        
        article_2 = {
            "eId": "art_2",
            "num": "Article 2",
            "heading": "Definitions",
            "children": [
                {
                    "eId": 0,
                    "text": "For the purposes of this Regulation, the following definitions apply:"
                },
                {
                    "eId": 1,
                    "text": "‘cross-border interoperability’ means the ability of Union entities and public sector bodies of Member States to interact with each other across borders by sharing data, information and knowledge through digital processes in line with the legal, organisational, semantic and technical requirements related to such cross-border interaction;"
                },
                {
                    "eId": 2,
                    "text": "‘trans-European digital public services’ means digital services provided by Union entities or public sector bodies to one another or to natural or legal persons in the Union, and requiring interaction across Member State borders, among Union entities or between Union entities and public sector bodies, by means of their network and information systems;"
                },
                {
                    "eId": 3,
                    "text": "‘network and information system’ means a network and information system as defined in Article 6, point (1), of Directive (EU) 2022/2555 of the European Parliament and of the Council;"
                },
                {
                    "eId": 4,
                    "text": "‘interoperability solution’ means a reusable asset concerning legal, organisational, semantic or technical requirements to enable cross-border interoperability, such as conceptual frameworks, guidelines, reference architectures, technical specifications, standards, services and applications, as well as documented technical components, such as source code;"
                },
                {
                    "eId": 5,
                    "text": "‘Union entities’ means the Union institutions, bodies, offices and agencies set up by, or on the basis of, the TEU, the Treaty on the functioning of European Union or the Treaty establishing the European Atomic Energy Community;"
                },
                {
                    "eId": 6,
                    "text": "‘public sector body’ means a public sector body as defined in Article 2, point (1), of Directive (EU) 2019/1024 of the European Parliament and of the Council;"
                },
                {
                    "eId": 7,
                    "text": "‘data’ means data as defined in Article 2, point (1), of Regulation (EU) 2022/868 of the European Parliament and of the Council;"
                },
                {
                    "eId": 8,
                    "text": "‘machine-readable format’ means a machine-readable format as defined in Article 2, point (13), of Directive (EU) 2019/1024;"
                },
                {
                    "eId": 9,
                    "text": "‘GovTech’ means technology-based cooperation between public and private sector actors supporting public sector digital transformation;"
                },
                {
                    "eId": 10,
                    "text": "‘standard’ means a standard as defined in Article 2, point (1), of Regulation (EU) No 1025/2012 of the European Parliament and of the Council;"
                },
                {
                    "eId": 11,
                    "text": "‘ICT technical specification’ means ICT technical specification as defined in Article 2, point (5), of Regulation (EU) No 1025/2012;"
                },
                {
                    "eId": 12,
                    "text": "‘open source licence’ means a licence whereby the reuse, redistribution and modification of software is permitted for all uses on the basis of a unilateral declaration by the right holder that may be subject to certain conditions, and where the source code of the software is made available to users indiscriminately;"
                },
                {
                    "eId": 13,
                    "text": "‘highest level of management’ means a manager, management or coordination and oversight body at the most senior administrative level, taking account of the high-level governance arrangements in each Union entity;"
                },
                {
                    "eId": 14,
                    "text": "‘interoperability regulatory sandbox’ means a controlled environment set up by a Union entity or a public sector body for the development, training, testing and validation of innovative interoperability solutions, where appropriate in real world conditions, supporting the cross-border interoperability of trans-European digital public services for a limited period of time under regulatory supervision;"
                },
                {
                    "eId": 15,
                    "text": "‘binding requirement’ means an obligation, prohibition, condition, criterion or limit of a legal, organisational, semantic or technical nature, which is set by a Union entity or a public sector body concerning one or more trans-European digital public services and which has an effect on cross-border interoperability."
                }
            ]
            }
        
        article_3 = {
            "eId": "art_3",
            "num": "Article 3",
            "heading": "Interoperability assessment",
            "children": [
                {
                    "eId": "003.001",
                    "text": "1. Before taking a decision on new or substantially modified binding requirements, a Union entity or a public sector body shall carry out an interoperability assessment. Where, in relation to binding requirements, an interoperability assessment has already been carried out or where binding requirements are implemented by solutions provided by Union entities, the public sector body concerned shall not be required to carry out a further interoperability assessment in relation to those requirements. A single interoperability assessment may be carried out to address a set of binding requirements. The Union entity or public sector body concerned may also carry out the interoperability assessment in other cases."
                },
                {
                    "eId": "003.002",
                    "text": "2. An interoperability assessment shall, in an appropriate manner, identify and assess the following: (a) the effects of the binding requirements on cross-border interoperability, using the European Interoperability Framework referred to in Article 6as a support tool; (b) the stakeholders to which the binding requirements are relevant; (c) the Interoperable Europe solutions referred to in Article 7 that support the implementation of the binding requirements. The Union entity or public sector body concerned shall publish, in a machine-readable format facilitating automated translation, a report presenting the outcome of the interoperability assessment, including the items listed in the Annex, on an official website. It shall share that report electronically with the Interoperable Europe Board established pursuant to Article 15 (the ‘Board’). The requirements laid down in this paragraph shall not restrict existing Member States’ rules on access to documents. The publication of that report shall not compromise intellectual property rights or trade secrets, public order or security."
                },
                {
                    "eId": "003.003",
                    "text": "3. Union entities and public sector bodies may decide which body is to provide the necessary support to carry out the interoperability assessment. The Commission shall provide technical tools to support the interoperability assessment, including an online tool to facilitate the completion of the report and its publication on the Interoperable Europe portal referred to in Article 8."
                },
                {
                    "eId": "003.004",
                    "text": "4. The Union entity or public sector body concerned shall consult recipients of the services directly affected, including citizens, or their representatives. That consultation shall be without prejudice to the protection of commercial or public interests or the security of such services."
                },
                {
                    "eId": "003.005",
                    "text": "5. By 12 January 2025, the Board shall adopt the guidelines referred to in Article 15(5), point (a)."
                }
            ]
        }
        self.assertEqual(self.parser.articles[0], article_1, "Article element should match expected value")
        self.assertEqual(self.parser.articles[1], article_2, "Article element should match expected value")
        self.assertEqual(self.parser.articles[2], article_3, "Article element should match expected value")
        
    def test_get_conclusions(self):
        self.parser.get_conclusions()
        self.assertIsNotNone(self.parser.conclusions, "Conclusions element should not be None")      
        
# Run the tests
if __name__ == "__main__":
    unittest.main()
````

## File: tests/parsers/html/test_html.py
````python
import unittest
import os
from tulit.parsers.html.xhtml import HTMLParser
import json

DATA_DIR = os.path.join(os.path.dirname(__file__), "..\\..\\data\\html")
file_path = os.path.join(DATA_DIR, "c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006.03\\DOC_1.html")


class TestHTMLParser(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None  # Allow full diff if needed
        self.parser = HTMLParser()        
        
        # Ensure test file exists
        if not os.path.exists(file_path):
            raise FileNotFoundError(f"Test file not found at {file_path}")
        self.parser.get_root(file_path)
    
    def test_get_root(self):
        """Test parsing and root element retrieval from the Akoma Ntoso file."""
        self.assertTrue(os.path.exists(file_path), f"Test file not found at {file_path}")
        self.assertIsNotNone(self.parser.root, "Root element should not be None")      
        
# Run the tests
if __name__ == "__main__":
    unittest.main()
````

## File: tests/parsers/xml/test_akomantoso.py
````python
import unittest
from tulit.parsers.xml.akomantoso import AkomaNtosoParser, AKN4EUParser
import os
import lxml.etree as etree

# Define constants for file paths and directories
file_path = os.path.join(os.path.dirname(__file__), '..\\..\\data\\akn\\eu', '32014L0092.akn')

class TestAkomaNtosoParser(unittest.TestCase):
    maxDiff = None

    def setUp(self):
        """Initialize the AkomaNtosoParser before each test."""
        self.parser = AkomaNtosoParser()
        self.parser.get_root(file_path)

    def tearDown(self):
        """Cleanup after each test."""
        self.parser = None

    def test_get_root(self):
        """Test parsing and root element retrieval from the Akoma Ntoso file."""
        self.assertTrue(os.path.exists(file_path), f"Test file not found at {file_path}")
        self.assertIsNotNone(self.parser.root, "Root element should not be None")
        
    def test_get_preface(self):
        """Test the content extracted from the preface section."""
        self.parser.get_preface()
        self.assertIsNotNone(self.parser.preface, "Preface element not found")
        
        expected_preface = "Directive 2014/92/EU of the European Parliament and of the Council of 23 July 2014 on the comparability of fees related to payment accounts, payment account switching and access to payment accounts with basic features (Text with EEA relevance)"

        self.assertEqual(self.parser.preface, expected_preface)

    def test_get_preamble(self):
        """Test retrieval of preamble data from the XML file."""
        self.parser.get_preamble()
        self.assertIsNotNone(self.parser.preamble, "Preamble element not found")
        

    def test_get_formula(self):
        """Test extraction of formula text within the preamble."""
        self.parser.get_preamble()

        self.parser.get_formula()
        self.assertEqual(self.parser.formula, "THE EUROPEAN PARLIAMENT AND THE COUNCIL OF THE EUROPEAN UNION,")

    def test_get_citations(self):
        """Test citation extraction in the preamble section."""
        self.parser.get_preamble()
        self.parser.get_citations()
        
        self.assertIsNotNone(self.parser.citations, "Citations data not found")

        first_citation = self.parser.citations[0]
        expected_text = "Having regard to the Treaty on the Functioning of the European Union, and in particular Article 114"
        self.assertIn(expected_text, first_citation['text'])

    def test_get_recitals(self):
        """Test retrieval and content verification of recitals in the preamble."""
        self.parser.get_preamble()
        self.parser.get_recitals()
        self.assertIsNotNone(self.parser.recitals, "Recitals section not found in <preamble>")
        self.assertEqual(len(self.parser.recitals), 58, "Incorrect number of recitals extracted")
        expected_recitals = {
            1: {'eId': "recs_1__rec_(2)", 'text': "In this respect, Directive 2007/64/EC of the European Parliament and of the Council established basic transparency requirements for fees charged by payment service providers in relation to services offered on payment accounts. This has substantially facilitated the activity of payment service providers, creating uniform rules with respect to the provision of payment services and the information to be provided, reduced the administrative burden and generated cost savings for payment service providers."},
            2: {'eId': "recs_1__rec_(3)", 'text': "The smooth functioning of the internal market and the development of a modern, socially inclusive economy increasingly depends on the universal provision of payment services. Any new legislation in this regard must be part of a smart economic strategy for the Union, which must effectively take into account the needs of more vulnerable consumers."},
            15: {'eId': "recs_1__rec_(16)", 'text': "Consumers would benefit most from information that is concise, standardised and easy to compare between different payment service providers. The tools made available to consumers to compare payment account offers would not have a positive impact if the time invested in going through lengthy lists of fees for different offers outweighed the benefit of choosing the offer that represents the best value. Those tools should be multifold and consumer testing should be conducted. At this stage, fee terminology should only be standardised for the most representative terms and definitions within Member States in order to avoid the risk of excessive information and to facilitate swift implementation."}
        }
        # Iterate over the selected recitals to verify content and ID
        for index, expected_values in expected_recitals.items():
            with self.subTest(recital=index):
                self.assertEqual(self.parser.recitals[index]['eId'], expected_values['eId'], 
                                 f"Recital {index} ID does not match expected value")
                self.assertIn(expected_values['text'], self.parser.recitals[index]['text'], 
                              f"Recital {index} text does not match expected content")

    def test_get_preamble_final(self):
        """Test extraction of the final preamble text."""
        self.parser.get_preamble()
        self.parser.get_preamble_final()
        preamble_final = "HAVE ADOPTED THIS DIRECTIVE:"
        self.assertEqual(self.parser.preamble_final, preamble_final, "Final preamble text does not match expected content")
    
    
    def test_get_body(self):
        """Test retrieval of the body element."""
        self.parser.get_body()
        self.assertIsInstance(self.parser.body, etree._Element, "Body element should be an etree._Element")

    def test_get_chapters(self):
        """Test retrieval and content of chapter headings."""
        self.parser.get_body()
        self.parser.get_chapters()

        expected_chapters = [
            {'eId': 'chp_I',   'num': 'CHAPTER I',   'heading': 'SUBJECT MATTER, SCOPE AND DEFINITIONS'},
            {'eId': 'chp_II',  'num': 'CHAPTER II',  'heading': 'COMPARABILITY OF FEES CONNECTED WITH PAYMENT ACCOUNTS'},
            {'eId': 'chp_III', 'num': 'CHAPTER III', 'heading': 'SWITCHING'},
            {'eId': 'chp_IV',  'num': 'CHAPTER IV',  'heading': 'ACCESS TO PAYMENT ACCOUNTS'},
            {'eId': 'chp_V',   'num': 'CHAPTER V',   'heading': 'COMPETENT AUTHORITIES AND ALTERNATIVE DISPUTE RESOLUTION'},
            {'eId': 'chp_VI',  'num': 'CHAPTER VI',  'heading': 'SANCTIONS'},
            {'eId': 'chp_VII', 'num': 'CHAPTER VII', 'heading': 'FINAL PROVISIONS'}
        ]
        self.assertEqual(self.parser.chapters, expected_chapters, "Chapters data does not match expected content")

    def test_get_articles(self):
        """Test retrieval of articles within the body."""
        self.parser.get_body()
        self.parser.get_articles()
        
        self.assertEqual(len(self.parser.articles), 31, "Incorrect number of articles extracted")
    
    def test_get_conclusions(self):
        # Expected output
        conclusions = {
            'date': '23 July 2014',
            'signatures': [
                ["Done at Brussels, 23 July 2014."],
                ['For the European Parliament', 'The President', 'M. Schulz'],
                ['For the Council', 'The President', 'S. Gozi']
            ]
        }
        self.parser.get_conclusions()
        self.assertEqual(self.parser.conclusions, conclusions, "Parsed conclusions do not match expected output")
        
class TestAKN4EUParser(unittest.TestCase):

    def setUp(self):
        """Initialize the AKN4EUParser before each test."""
        self.parser = AKN4EUParser()
        self.file_path = os.path.join(os.path.dirname(__file__), '..\\..\\data\\akn\\eu', 'akn4eu.xml')
        self.parser.get_root(self.file_path)
        
    def tearDown(self):
        """Cleanup after each test."""
        self.parser = None
        
    def test_parse(self):
        """Test parsing of an Akoma Ntoso 4.0 document."""
        self.parser.parse(self.file_path)
        self.assertIsNotNone(self.parser.root, "Root element should not be None after parsing")
        self.assertEqual(self.parser.root.tag, '{http://docs.oasis-open.org/legaldocml/ns/akn/3.0}akomaNtoso', 
                         "Root tag does not match expected Akoma Ntoso namespace")
    
    def test_get_body(self):
        """Test retrieval of the body element using AKN4EUParser."""
        self.parser.get_body()
        self.assertIsInstance(self.parser.body, etree._Element, "Body element should be an etree._Element")
    
    def test_get_chapters(self):
        """Test retrieval and content of chapter headings using AKN4EUParser."""
        self.parser.get_body()
        self.parser.get_chapters()
        expected_chapters = [
        ]
        self.assertEqual(self.parser.chapters, expected_chapters, "Chapters data does not match expected content")
        
    def test_get_articles(self):
        """Test retrieval of articles within the body using AKN4EUParser."""
        self.parser.get_body()
        self.parser.get_articles()
        
        self.assertEqual(len(self.parser.articles), 15, "Incorrect number of articles extracted")

if __name__ == '__main__':
    unittest.main()
````

## File: tests/parsers/xml/test_formex.py
````python
import unittest
from tulit.parsers.xml.formex import Formex4Parser
import xml.etree.ElementTree as ET

import os 

file_path = os.path.join(".\\tests\\data\\formex\\5cca93ee-e193-46bf-8416-c2f57cbc34c9.0004.05\\DOC_2.xml")

iopa = ".\\tests\\data\\formex\\c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006.02\\DOC_1\\L_202400903EN.000101.fmx.xml"

class TestFormex4Parser(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None  # Allow full diff if needed        
        self.parser = Formex4Parser()
        self.parser.get_root(file_path)

    def test_get_root(self):
        """Test parsing and root element retrieval from the Akoma Ntoso file."""
        self.assertTrue(os.path.exists(file_path), f"Test file not found at {file_path}")
        self.assertIsNotNone(self.parser.root, "Root element should not be None")
    
    def test_get_preface(self):
        
        self.parser.get_preface()
        preface = "Commission Implementing Regulation (EU) No 1319/2011 of 15 December 2011 fixing representative prices in the poultrymeat and egg sectors and for egg albumin, and amending Regulation (EC) No 1484/95"        
        self.assertEqual(self.parser.preface, preface)
    
    def test_get_preamble(self):        
        self.parser.get_preamble()
        self.assertIsNotNone(self.parser.preamble)
        
    def test_get_formula(self):
        
        self.parser.get_preamble()
        self.parser.get_formula()
                
        formula = "THE EUROPEAN COMMISSION,"
        
        self.assertEqual(self.parser.formula, formula)
        

    def test_get_citations(self):
    
        self.parser.get_preamble()    
        self.parser.get_citations()
        
        citations =  [
                {'eId': "cit_1", 'text': "Having regard to the Treaty on the Functioning of the European Union,"},
                {"eId": "cit_2", 'text':"Having regard to Council Regulation (EC) No 1234/2007 of 22 October 2007 establishing a common organisation of agricultural markets and on specific provisions for certain agricultural products (Single CMO Regulation), and in particular Article 143 thereof,"},
                {"eId": "cit_3", 'text':"Having regard to Council Regulation (EC) No 614/2009 of 7 July 2009 on the common system of trade for ovalbumin and lactalbumin, and in particular Article 3(4) thereof,"},
            ]
        
        self.assertEqual(self.parser.citations, citations)
   
    
    def test_get_recitals(self):
        
        self.parser.get_preamble()
        self.parser.get_recitals()
        
        recitals = [
                {"eId": "rct_1", "text": "Commission Regulation (EC) No 1484/95 lays down detailed rules for implementing the system of additional import duties and fixes representative prices for poultrymeat and egg products and for egg albumin."}, 
                {"eId": "rct_2", "text": "Regular monitoring of the data used to determine representative prices for poultrymeat and egg products and for egg albumin shows that the representative import prices for certain products should be amended to take account of variations in price according to origin. The representative prices should therefore be published."},
                {"eId": "rct_3", "text": "In view of the situation on the market, this amendment should be applied as soon as possible."},
                {"eId": "rct_4", "text": "The measures provided for in this Regulation are in accordance with the opinion of the Management Committee for the Common Organisation of Agricultural Markets,"},
        ]
        
        self.assertEqual(self.parser.recitals, recitals)      
    
    def test_get_preamble_final(self):
        self.parser.get_preamble()

        self.parser.get_preamble_final()
        preamble_final = "HAS ADOPTED THIS REGULATION:"
        
        self.assertEqual(self.parser.preamble_final, preamble_final)
    
    def test_get_body(self):
        self.parser.get_body()
        self.assertIsNotNone(self.parser.body, "Body element should not be None")    
    
    def test_get_chapters(self):
        self.parser = Formex4Parser()
        self.parser.get_root(iopa)
        self.parser.get_body()
        self.parser.get_chapters()

        expected_chapters = [
            {'eId': "cpt_1",  'num': 'Chapter 1', 'heading': 'General provisions'},
            {'eId': "cpt_2",  'num': 'Chapter 2', 'heading': 'European Interoperability enablers'}, 
            {'eId': "cpt_3",  'num': 'Chapter 3', 'heading': 'Interoperable Europe support measures'},
            {'eId': "cpt_4",  'num': 'Chapter 4', 'heading': 'Governance of cross-border interoperability'},
            {'eId': "cpt_5",  'num': 'Chapter 5', 'heading': 'Interoperable Europe planning and monitoring'},
            {'eId': "cpt_6",  'num': 'Chapter 6', 'heading': 'Final provisions'},
        ]
        
        self.assertEqual(self.parser.chapters[0], expected_chapters[0], "Chapters data does not match expected content")
        
    def test_get_articles(self):
        self.parser.get_body()
        self.parser.get_articles()
        
        # Expected articles based on sample data in XML file
        expected = [
            {
                "eId": "art_001",
                "num": "Article 1",
                "heading": None,
                "children": [
                    {"eId": '001', "text": "Annex I to Regulation (EC) No 1484/95 is replaced by the Annex to this Regulation.", "amendment": False}
                ]
            },
            {
                "eId": "art_002",
                "num": "Article 2",
                "heading": None,
                "children": [
                    {"eId": "001", "text": "This Regulation shall enter into force on the day of its publication in the Official Journal of the European Union.", "amendment": False}
                ]
            }
        ]
        
        self.assertEqual(self.parser.articles, expected)

    def test_get_conclusions(self):
        self.parser = Formex4Parser()
        self.parser.get_root(iopa)
        self.parser.get_body()
        self.parser.get_conclusions()
        conclusions = {
                "conclusion_text": "This Regulation shall be binding in its entirety and directly applicable in all Member States.",
                "signature": {
                    "place": "Done at Strasbourg,",
                    "date": "13 March 2024",
                    "signatory": "For the European Parliament",
                    "title": "The President"
                }
            }
        self.assertEqual(self.parser.conclusions, conclusions)

# Run the tests
if __name__ == "__main__":
    unittest.main()
````

## File: tests/parsers/test_parser.py
````python
import unittest

if __name__ == '__main__':
    unittest.main(verbosity=2)
````

## File: tulit/client/queries/formex_query.rq
````
PREFIX cdm: <http://publications.europa.eu/ontology/cdm#>
PREFIX purl: <http://purl.org/dc/elements/1.1/>

SELECT DISTINCT ?cellarURIs, ?manif, ?format, ?expr
WHERE {
    ?work owl:sameAs <http://publications.europa.eu/resource/celex/{CELEX}> .
    ?expr cdm:expression_belongs_to_work ?work ;
           cdm:expression_uses_language ?lang .
    ?lang purl:identifier ?langCode .
    ?manif cdm:manifestation_manifests_expression ?expr;
           cdm:manifestation_type ?format.
    ?cellarURIs cdm:item_belongs_to_manifestation ?manif.

    FILTER(str(?format)="fmx4" && str(?langCode)="ENG")
}
ORDER BY ?cellarURIs
LIMIT 10
````

## File: tulit/client/queries/html_query.rq
````
PREFIX cdm: <http://publications.europa.eu/ontology/cdm#>
PREFIX purl: <http://purl.org/dc/elements/1.1/>

SELECT DISTINCT ?cellarURIs, ?manif, ?format, ?expr
WHERE {
    ?work owl:sameAs <http://publications.europa.eu/resource/celex/{CELEX}> .
    ?expr cdm:expression_belongs_to_work ?work ;
           cdm:expression_uses_language ?lang .
    ?lang purl:identifier ?langCode .
    ?manif cdm:manifestation_manifests_expression ?expr;
           cdm:manifestation_type ?format.
    ?cellarURIs cdm:item_belongs_to_manifestation ?manif.

    FILTER(str(?format)="xhtml" && str(?langCode)="ENG")
}
ORDER BY ?cellarURIs
LIMIT 10
````

## File: tulit/client/__init__.py
````python
"""
This module provides functionality for [describe the main purpose of the module].

Classes:
    [ClassName]: [Brief description of the class]

Functions:
    [function_name_1]: [Brief description of the function]
    [function_name_2]: [Brief description of the function]
    ...

Usage:
    [Provide a brief example of how to use the module]

Notes:
    [Any additional notes or important information about the module]

Author:
    [Your Name]

Date:
    [Date of creation or last modification]
"""
````

## File: tulit/client/boe.py
````python
"""
Boletín Oficial del Estado (BOE) client.

This module contains the BOEClient class, which is used to download XML files from the BOE API endpoint.

The documentation for the BOE API can be found at https://www.boe.es/datosabiertos/documentos/APIsumarioBOE.pdf

"""

import requests
from tulit.client.client import Client
import argparse
import os

class BOEClient(Client):
    def __init__(self, download_dir, log_dir):
        super().__init__(download_dir=download_dir, log_dir=log_dir)

    def get_html(self, id):
        try:
            url = 'https://www.boe.es/diario_boe/xml.php?id='
            response = requests.get(url + id)
            response.raise_for_status()
            return response.text
        except requests.RequestException as e:
            print(f"An error occurred: {e}")
            return None


def main():
    parser = argparse.ArgumentParser(description='Downloads an XML file from the BVeneto website.')
    parser.add_argument('--id', type=str, default='BOE-A-2001-11814', help='BOE Id of the document to download.')
    parser.add_argument('--file', type=str, default='boe.xml', help='Path to the output HTML file.')
    args = parser.parse_args()
    
    client = BOEClient(download_dir=args.file, log_dir='../tests/metadata/logs')
    html_content = client.get_html(args.id)

    if html_content:
        # Ensure the directory exists
        output_dir = os.path.abspath('./tests/data/xml/spain/')
        os.makedirs(output_dir, exist_ok=True)

        # Write the HTML content to a file
        try:
            with open(os.path.join(output_dir, os.path.basename(args.file)), 'w', encoding='utf-8') as f:
                f.write(html_content)
            print(f"File saved successfully to {os.path.join(output_dir, os.path.basename(args.file))}")
        except PermissionError as e:
            print(f"Permission error: {e}")
        except Exception as e:
            print(f"An error occurred while writing the file: {e}")
    else:
        print("Failed to retrieve HTML content.")

if __name__ == "__main__":
    main()
````

## File: tulit/client/cellar.py
````python
import logging
import argparse
import requests
from tulit.client.client import Client
from SPARQLWrapper import SPARQLWrapper, JSON, POST

from importlib.resources import files



import logging

logger = logging.getLogger(__name__)

class CellarClient(Client):
    
    def __init__(self, download_dir, log_dir, proxies=None):
        super().__init__(download_dir, log_dir, proxies)
        self.endpoint = 'http://publications.europa.eu/resource/cellar/'
        self.sparql_endpoint = "http://publications.europa.eu/webapi/rdf/sparql"
    
    def send_sparql_query(self, sparql_query, celex=None):
        """
        Sends a SPARQL query to the EU SPARQL endpoint and stores the results in a JSON file.

        Parameters
        ----------
        sparql_query_filepath : str
            The path to the file containing the SPARQL query.
        response_file : str
            The path to the file where the results will be stored.

        Returns
        -------
        None

        Raises
        ------
        FileNotFoundError
            If the SPARQL query file is not found.
        Exception
            If there is an error sending the query or storing the results.

        Notes
        -----
        This function assumes that the SPARQL query file contains a valid SPARQL query.
        The results are stored in JSON format.

        """

        # Open SPARQL QUERY and print it to screen
        try:            
            
            if celex is not None:
                # Option 1: If you want to use format()
                sparql_query = sparql_query.replace("{CELEX}", celex) 

                # send query to cellar endpoint and retrieve results
                results = self.get_results_table(sparql_query)

            return results
    
        except FileNotFoundError as e:
            print(f"Error: The sparql query was not found.")
            raise e
        except Exception as e:
            print(f"An error occurred: {e}")
            raise e

    def get_results_table(self, sparql_query):
        """
        Sends a SPARQL query to the EU SPARQL endpoint and returns the results as a JSON object.

        Parameters
        ----------
        sparql_query : str
            The SPARQL query as a string.

        Returns
        -------
        dict
            The results of the SPARQL query in JSON format.

        Raises
        ------
        Exception
            If there is an error sending the query or retrieving the results.

        Notes
        -----
        This function uses the SPARQLWrapper library to send the query and retrieve the results.
        The results are returned in JSON format.

        """        

        try:
            # Create a SPARQLWrapper object with the endpoint URL
            sparql = SPARQLWrapper(self.sparql_endpoint)

            # Set the SPARQL query
            sparql.setQuery(sparql_query)

            # Set the query method to POST
            sparql.setMethod(POST)

            # Set the return format to JSON
            sparql.setReturnFormat(JSON)

            # Send the query and retrieve the results
            results = sparql.query().convert()

            return results
        except Exception as e:
            logger.error(f"An error occurred: {e}")
            raise e
    
    def fetch_content(self, url) -> requests.Response:
        """
        Send a GET request to download a file

        Parameters
        ----------
        url : str
            The URL to send the request to.

        Returns
        -------
        requests.Response
            The response from the server.

        Notes
        -----
        The request is sent with the following headers:
        - Accept: application/zip;mtype=fmx4, application/xml;mtype=fmx4, application/xhtml+xml, text/html, text/html;type=simplified, application/msword, text/plain, application/xml;notice=object
        - Accept-Language: eng
        - Content-Type: application/x-www-form-urlencoded
        - Host: publications.europa.eu

        Raises
        ------
        requests.RequestException
            If there is an error sending the request.

        See Also
        --------
        requests : The underlying library used for making HTTP requests.

        """
        try:
            headers = {
                'Accept': "*, application/zip, application/zip;mtype=fmx4, application/xml;mtype=fmx4, application/xhtml+xml, text/html, text/html;type=simplified, application/msword, text/plain, application/xml, application/xml;notice=object",
                'Accept-Language': "eng",
                'Content-Type': "application/x-www-form-urlencoded",
                'Host': "publications.europa.eu"
            }
            if self.proxies is not None:
                response = requests.request("GET", url, headers=headers, proxies=self.proxies)
            else:
                response = requests.request("GET", url, headers=headers)
            response.raise_for_status()
            return response
        except requests.RequestException as e:
            logger.error(f"Error sending GET request: {e}")
            return None
             
    def build_request_url(self, params):
        """
        Build the request URL based on the source and parameters.
        """
        url = f"{self.endpoint}{params['cellar']}"
        
        return url
    
    def get_cellar_ids_from_json_results(self, results, format):
        """
        Extract CELLAR ids from a JSON dictionary.

        Parameters
        ----------
        cellar_results : dict
            A dictionary containing the response of the CELLAR SPARQL query

        Returns
        -------
        list
            A list of CELLAR ids.

        Notes
        -----
        The function assumes that the JSON dictionary has the following structure:
        - The dictionary contains a key "results" that maps to another dictionary.
        - The inner dictionary contains a key "bindings" that maps to a list of dictionaries.
        - Each dictionary in the list contains a key "cellarURIs" that maps to a dictionary.
        - The innermost dictionary contains a key "value" that maps to a string representing the CELLAR URI.

        The function extracts the CELLAR id by splitting the CELLAR URI at "cellar/" and taking the second part.

        Examples
        --------
        >>> cellar_results = {
        ...     "results": {
        ...         "bindings": [
        ...             {"cellarURIs": {"value": "https://example.com/cellar/some_id"}},
        ...             {"cellarURIs": {"value": "https://example.com/cellar/another_id"}}
        ...         ]
        ...     }
        ... }
        >>> cellar_ids = get_cellar_ids_from_json_results(cellar_results)
        >>> print(cellar_ids)
        ['some_id', 'another_id']
        """
        cellar_ids = []
        results_list = results["results"]["bindings"]
        for i, file in enumerate(results_list):
            if file['format']['value'] == format:
                cellar_ids.append(file['cellarURIs']["value"].split("cellar/")[1])

        return cellar_ids

    def download(self, celex, format=None):
        """
        Sends a REST query to the specified source APIs and downloads the documents
        corresponding to the given results.

        Parameters
        ----------
        results : dict
            A dictionary containing the JSON results from the APIs.
        format : str, optional
            The format of the documents to download.        

        Returns
        -------
        list
            A list of paths to the downloaded documents.
        """
        if format == 'fmx4':            
            sparql_query = files("tulit.client.queries").joinpath("formex_query.rq").read_text()
        elif format == 'xhtml':
            sparql_query = files("tulit.client.queries").joinpath("html_query.rq").read_text()
        else:
            logger.error('No valid format provided. Please choose one between fmx4 or xhtml')
            return None
            
        results = self.send_sparql_query(sparql_query, celex)
        cellar_ids = self.get_cellar_ids_from_json_results(results, format=format)
        
        try:
            document_paths = []
            
            for id in cellar_ids:
                # Build the request URL
                url = self.build_request_url(params={'cellar': id})                
                # Send the GET request
                response = self.fetch_content(url)
                # Handle the response
                file_path = self.handle_response(response=response, filename=id)
                # Append the file path to the list
                document_paths.append(file_path)
                
            return document_paths

        except Exception as e:
            logging.error(f"Error processing range: {e}")
        
        return document_paths

def main():
    parser = argparse.ArgumentParser(description='Download a Cellar document to a folder')
    parser.add_argument('--celex', type=str, default='32024R0903', help='CELEX identifier of the document')
    parser.add_argument('--format', type=str, default='fmx4', help='Format of the document, either fmx4 or xhtml')
    parser.add_argument('--dir', type=str, default='tests/data/formex', help='Path to the directory')
    
    args = parser.parse_args()
    
    client = CellarClient(download_dir=args.dir, log_dir='./tests/logs')    
    
    documents = client.download(celex=args.celex, format=args.format)
    
    print(documents)

if __name__ == "__main__":
    main()
````

## File: tulit/client/client.py
````python
import os
import io
import logging
import zipfile
import requests

class Client:
    """	
    A generic document downloader class.
    """	
    def __init__(self, download_dir, log_dir, proxies=None):
        """
        Initializes the downloader with directories for downloads and logs.
        
        Parameters
        ----------
        download_dir : str
            Directory where downloaded files will be saved.
        log_dir : str
            Directory where log files will be saved.
        """
        self.download_dir = download_dir
        self.log_dir = log_dir
        self.proxies = proxies
        self._ensure_directories()

    def _ensure_directories(self):
        """
        Ensure that the download and log directories exist.
        """
        if not os.path.exists(self.download_dir):
            os.makedirs(self.download_dir)
        if not os.path.exists(self.log_dir):
            os.makedirs(self.log_dir)
    
    def handle_response(self, response, filename):
        """
        Handle a server response by saving or extracting its content.

        Parameters
        ----------
        response : requests.Response
            The HTTP response object.
        folder_path : str
            Directory where the file will be saved.
        cid : str
            CELLAR ID of the document.

        Returns
        -------
        str or None
            Path to the saved file or None if the response couldn't be processed.
        """
        content_type = response.headers.get('Content-Type', '')
        
        # The return file is usually either a zip file, or a file with the name DOC_* inside a folder named as the cellar_id
        target_path = os.path.join(self.download_dir, filename)
        os.makedirs(os.path.dirname(target_path), exist_ok=True)

        if 'zip' in content_type:
            self.extract_zip(response, target_path)
            return target_path
        else:
            extension = self.get_extension_from_content_type(content_type)
            if not extension:
                logging.warning(f"Unknown content type for ID {filename}: {content_type}")
                return None

            file_path = f"{target_path}.{extension}"
            file_path = os.path.normpath(file_path)
            
            with open(file_path, mode='wb+') as f:
                f.write(response.content)            
                
            return file_path
        
    def get_extension_from_content_type(self, content_type):
        """
        Map Content-Type to a file extension.
        
        Parameters
        ----------
        content_type : str
            The Content-Type header from the server response.
        
        Returns
        -------
        str or None
            File extension corresponding to the Content-Type
        """
        content_type_mapping = {
            'text/html': 'html',
            'application/json': 'json',
            'application/xml': 'xml',
            'text/plain': 'txt',
            'application/zip': 'zip',
            'text/xml': 'xml',
            'application/xhtml+xml': 'xhtml',
        }
        for ext, mapped_ext in content_type_mapping.items():
            if ext in content_type:
                return mapped_ext

    # Function to download a zip file and extract it
    def extract_zip(self, response: requests.Response, folder_path: str):
        """
        Extracts the content of a zip file.
        
        Parameters
        ----------
        response : requests.Response
            The HTTP response object.
        folder_path : str
            Directory where the zip file will be extracted.
        """
        try:
            z = zipfile.ZipFile(io.BytesIO(response.content))
            z.extractall(folder_path)
        except Exception as e:
            logging.error(f"Error downloading zip: {e}")
````

## File: tulit/client/legifrance.py
````python
import requests
from tulit.client.client import Client
import argparse

class LegifranceClient(Client):
    def __init__(self, client_id, client_secret, download_dir='./data/france/legifrance', log_dir='./data/logs'):
        super().__init__(download_dir, log_dir)
        self.client_id = client_id
        self.client_secret = client_secret
        self.base_url = "https://sandbox-api.piste.gouv.fr/dila/legifrance/lf-engine-app"

    def get_token(self):
        token_url = "https://sandbox-oauth.piste.gouv.fr/api/oauth/token"
        payload = {
            'grant_type': 'client_credentials',            
            "scope": "openid",
            "client_id": self.client_id,
            "client_secret": self.client_secret,        
            }
        response = requests.post(token_url, data=payload)
        response.raise_for_status()
        return response.json()['access_token']

    def get_dossier_legislatif(self, dossier_id):
        token = self.get_token()
        headers = {
            'Authorization': f'Bearer {token}',
            'Content-Type': 'application/json'
        }
        url = f"{self.base_url}/consult/legiPart"
        payload = {
            "searchedString": "constitution 1958",
            "date": "2021-04-15",
            "textId": "LEGITEXT000006075116"
            }
        response = requests.post(url, json=payload, headers=headers)
        response.raise_for_status()
        return response.json()
    
def main():
    parser = argparse.ArgumentParser(description='Legifrance Client')
    parser.add_argument('--client_id', type=str, help='Client ID for OAuth')
    parser.add_argument('--client_secret', type=str, help='Client Secret for OAuth')
    parser.add_argument('--dossier_id', type=str, required=True, help='Dossier ID to retrieve')
    args = parser.parse_args()
    
    client = LegifranceClient(args.client_id, args.client_secret)
    dossier = client.get_dossier_legislatif(args.dossier_id)
    print(dossier)

if __name__ == "__main__":
    main()
````

## File: tulit/client/legilux.py
````python
import requests
from tulit.client.client import Client

class LegiluxClient(Client):
    def __init__(self, download_dir, log_dir):
        super().__init__(download_dir, log_dir)
        #self.endpoint = "https://legilux.public.lu/eli/etat/leg/loi"

    def build_request_url(self, eli) -> str:
        """
        Build the request URL based on the source and parameters.
        """
        url = eli
        return url
    
    def fetch_content(self, url):
        """
        Fetch the content of the document.
        """
        headers = {"Accept": "application/xml"}
        response = requests.get(url, headers=headers)
        return response

    def download(self, eli):
        file_paths = []
        url = self.build_request_url(eli)
        response = self.fetch_content(url)        
        filename = eli.split('loi/')[1].replace('/', '_')
        if response.status_code == 200:
            file_paths.append(self.handle_response(response, filename=filename))
            print(f"Document downloaded successfully and saved to {file_paths}")
            return file_paths
        else:
            print(f"Failed to download document. Status code: {response.status_code}")
            return None

if __name__ == "__main__":
    downloader = LegiluxClient(download_dir='./tests/data/legilux', log_dir='./tests/metadata/logs')
    downloader.download(eli='http://data.legilux.public.lu/eli/etat/leg/loi/2006/07/31/n2/jo')
````

## File: tulit/client/normattiva.py
````python
from tulit.client.client import Client
import requests
import logging
from datetime import datetime


class NormattivaClient(Client):
    def __init__(self, download_dir, log_dir):
        super().__init__(download_dir, log_dir)
        self.endpoint = "https://www.normattiva.it/do/atto/caricaAKN"
    
    def build_request_url(self, params=None) -> str:
        """
        Build the request URL based on the source and parameters.
        """
        uri = f"https://www.normattiva.it/eli/id/{params['date']}//{params['codiceRedaz']}/CONSOLIDATED"
        # In case we want to use the NIR:URI instead of ELI
        #uri = f"https://www.normattiva.it/uri-res/N2Ls?urn:nir:stato:decreto.legislativo:{params['date']};{params['number']}"
        url = f"{self.endpoint}?dataGU={params['dataGU']}&codiceRedaz={params['codiceRedaz']}&dataVigenza={params['dataVigenza']}"
        
        return uri, url
                    
    def fetch_content(self, uri, url) -> requests.Response:
        """
        Send a GET request to download a file

        Parameters
        ----------
        url : str
            The URL to send the request to.

        Returns
        -------
        requests.Response
            The response from the server.

        Raises
        ------
        requests.RequestException
            If there is an error sending the request.
        """
        try:
            
            # Make a GET request to the URI to get the cookies        
            cookies_response = requests.get(uri)
            cookies_response.raise_for_status()
            cookies = cookies_response.cookies

            headers = {
                'Accept': "text/xml",
                'Accept-Encoding': "gzip, deflate, br, zstd",
                'Accept-Language': "en-US,en;q=0.9",
                
            }                     
            response = requests.get(url, headers=headers, cookies=cookies)
            response.raise_for_status()
            return response
        except requests.RequestException as e:
            logging.error(f"Error sending GET request: {e}")
            return None    
        
    def download(self, dataGU, codiceRedaz, dataVigenza = datetime.today().strftime('%Y%m%d')):     
        document_paths = []
        
        # Convert the dataGU to a datetime object
        dataGU = datetime.strptime(dataGU, '%Y%m%d')
            
        params = {
            # dataGU as a string in the format YYYYMMDD
            'dataGU': dataGU.strftime('%Y%m%d'),
            'codiceRedaz': codiceRedaz,
            'dataVigenza': dataVigenza,
            # dataGU as a string in the format YYYY/MM/DD
            'date': dataGU.strftime('%Y/%m/%d')
        }
        
        uri, url = self.build_request_url(params)
        response = self.fetch_content(uri, url)
        
        # If the response in HTML, raise an error saying that the date or codiceRedaz is wrong
        if 'text/html' in response.headers.get('Content-Type', ''):
            logging.error(f"Error downloading document: there is not an XML file with the following parameters: {params}")
            return None
        
        file_path = self.handle_response(response=response, filename=f"{params['dataGU']}_{params['codiceRedaz']}_VIGENZA_{params['dataVigenza']}")
        document_paths.append(file_path)
        return document_paths

# Example usage
if __name__ == "__main__":
    
    downloader = NormattivaClient(download_dir='./tests/data/akn/italy', log_dir='./tests/logs')
    #documents = downloader.download(dataGU='19410716', codiceRedaz='041U0633')
    documents = downloader.download(dataGU='19410716', codiceRedaz='041U0633', dataVigenza='20211231')

    
    print(documents)
````

## File: tulit/client/veneto.py
````python
import requests
from tulit.client.client import Client
import argparse
import os

class VenetoClient(Client):
    def __init__(self, download_dir, log_dir):
        super().__init__(download_dir=download_dir, log_dir=log_dir)

    def get_html(self, url):
        try:
            response = requests.get(url)
            response.raise_for_status()
            return response.text
        except requests.RequestException as e:
            print(f"An error occurred: {e}")
            return None
    
def main():
    parser = argparse.ArgumentParser(description='Downloads an HTML file from the Veneto website.')
    parser.add_argument('--url', type=str, default='https://www.consiglioveneto.it/web/crv/dettaglio-legge?numeroDocumento=10&id=69599315&backLink=https%3A%2F%2Fwww.consiglioveneto.it%2Fleggi-regionali%3Fp_p_id&p_p_lifecycle=0&p_p_state=normal&p_p_mode=view&pageTitle=&tab=vigente&annoSelezionato=2024', help='URL of the HTML file to download.')
    parser.add_argument('--file', type=str, default='esg.html', help='Path to the output HTML file.')
    args = parser.parse_args()
    
    client = VenetoClient(download_dir=args.file, log_dir='../tests/metadata/logs')
    html_content = client.get_html(args.url)

    if html_content:
        # Ensure the directory exists
        output_dir = os.path.abspath('./tests/data/html/veneto')
        os.makedirs(output_dir, exist_ok=True)

        # Write the HTML content to a file
        try:
            with open(os.path.join(output_dir, os.path.basename(args.file)), 'w', encoding='utf-8') as f:
                f.write(html_content)
            print(f"File saved successfully to {os.path.join(output_dir, os.path.basename(args.file))}")
        except PermissionError as e:
            print(f"Permission error: {e}")
        except Exception as e:
            print(f"An error occurred while writing the file: {e}")
    else:
        print("Failed to retrieve HTML content.")

if __name__ == "__main__":
    main()
````

## File: tulit/parsers/html/cellar.py
````python
from tulit.parsers.html.xhtml import HTMLParser
import json
import re
import argparse

class CellarHTMLParser(HTMLParser):
    def __init__(self):
        pass

    def get_preface(self):
        """
        Extracts the preface text from the HTML, if available.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted preface is stored in the 'preface' attribute.
        """
        try:
            preface_element = self.root.find('div', class_='eli-main-title')
            if preface_element:
                self.preface = preface_element.get_text(separator=' ', strip=True)
                print("Preface extracted successfully.")
            else:
                self.preface = None
                print("No preface found.")
        except Exception as e:
            print(f"Error extracting preface: {e}")
    
            
    def get_preamble(self):
        """
        Extracts the preamble text from the HTML, if available.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted preamble is stored in the 'preamble' attribute.
        """
        
        self.preamble = self.root.find('div', class_='eli-subdivision', id='pbl_1')
        # Remove all a tags from the preamble
        for a in self.preamble.find_all('a'):
            a.decompose()
            
    
    def get_formula(self):
        """
        Extracts the formula from the HTML, if present.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted formula is stored in the 'formula' attribute.
        """
        self.formula = self.preamble.find('p', class_='oj-normal').text


    
    def get_citations(self):
        """
        Extracts citations from the HTML.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted citations are stored in the 'citations' attribute
        """
        citations = self.preamble.find_all('div', class_='eli-subdivision', id=lambda x: x and x.startswith('cit_'))
        self.citations = []
        for citation in citations:
            eId = citation.get('id')
            text = citation.get_text(strip=True)
            self.citations.append({
                    'eId' : eId,
                    'text' : text
                }
            )

    def get_recitals(self):
        """
        Extracts recitals from the HTML.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted recitals are stored in the 'recitals' attribute.
        """
        recitals = self.preamble.find_all('div', class_='eli-subdivision', id=lambda x: x and x.startswith('rct_'))
        self.recitals = []
        for recital in recitals:
            eId = recital.get('id')
            
            text = recital.get_text()            
            text = re.sub(r'\s+', ' ', text).strip()
            text = re.sub(r'^\(\d+\)', '', text).strip()
            
            self.recitals.append({
                    'eId' : eId,
                    'text' : text
                }
            )
    def get_preamble_final(self):
        """
        Extracts the final preamble text from the HTML, if available.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted final preamble is stored in the 'preamble_final' attribute.
        """
        self.preamble_final = self.preamble.find_all('p', class_='oj-normal')[-1].get_text(strip=True)

    def get_body(self):
        """
        Extracts the body content from the HTML.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted body content is stored in the 'body' attribute
        """
        
        self.body = self.root.find('div', id=lambda x: x and x.startswith('enc_'))
        for a in self.body.find_all('a'):
            a.replace_with(' ')

    def get_chapters(self):
        """
        Extracts chapters from the HTML, grouping them by their IDs and headings.
        """
        
        chapters = self.body.find_all('div', id=lambda x: x and x.startswith('cpt_') and '.' not in x)
        self.chapters = []
        for chapter in chapters:
            eId = chapter.get('id')
            chapter_num = chapter.find('p', class_="oj-ti-section-1").get_text(strip=True)
            chapter_title = chapter.find('div', class_="eli-title").get_text(strip=True)
            self.chapters.append({
                'eId': eId,
                'num': chapter_num,
                'heading': chapter_title
            })

    def get_articles(self):
        """
        Extracts articles from the HTML. Each <div> with an id starting with "art" is treated as an article (eId).
        Subsequent subdivisions are processed based on the closest parent with an id.

        Returns:
            list[dict]: List of articles, each containing its eId and associated content.
        """
        
        articles = self.body.find_all('div', id=lambda x: x and x.startswith('art_') and '.' not in x)
        self.articles = []
        for article in articles:
            eId = article.get('id')  # Treat the id as the eId
            article_num = article.find('p', class_='oj-ti-art').get_text(strip=True)
            article_title_element = article.find('p', class_='oj-sti-art')
            if article_title_element is not None:
                article_title = article_title_element.get_text(strip=True)
            else:
                article_title = None
            
            # Extract paragraphs and lists within the article
            children = []
            
            # Handle articles with only paragraphs
            paragraphs = article.find_all('p', class_='oj-normal')            
            if paragraphs and len(article.find_all('table')) == 0:
                for paragraph in paragraphs:
                    text = ' '.join(paragraph.get_text(separator= ' ', strip=True).split())
                    text = re.sub(r'\s+([.,!?;:’])', r'\1', text)  # replace spaces before punctuation with nothing
                    children.append({
                        # Get parent of the paragraph: Use the id of the parent div as the eId
                        'eId': paragraph.find_parent('div').get('id'),
                        'text': text
                    })
            # Handle articles with only tables as first child:
            elif article.find_all('table') and article.find_all('table')[0].find_parent('div') == article:
                intro = article.find('p', class_='oj-normal')
                children.append({
                    'eId': 0,
                    'text': intro.get_text(strip=True)
                })
                tables = article.find_all('table')
                for table in tables:
                    rows = table.find_all('tr')
                    for row in rows:
                        cols = row.find_all('td')
                        if len(cols) == 2:
                            number = cols[0].get_text(strip=True)
                            number = number.strip('()')  # Remove parentheses
                            number = int(number)
                            text = ' '.join(cols[1].get_text(separator = ' ', strip=True).split())
                            text = re.sub(r'\s+([.,!?;:’])', r'\1', text)  # replace spaces before punctuation with nothing

                            children.append({
                                'eId': number,
                                'text': text
                            })
            # Handle articles with paragraphs and tables by treating tables as part of the same paragraph
            elif article.find_all('div', id=lambda x: x and '.' in x):
                paragraphs = article.find_all('div', id=lambda x: x and '.' in x)
                for paragraph in paragraphs:
                    if not paragraph.get('class'):
                        text = ' '.join(paragraph.get_text(separator = ' ', strip=True).split())
                        text = re.sub(r'\s+([.,!?;:’])', r'\1', text)  # replace spaces before punctuation with nothing
                        children.append({
                                'eId': paragraph.get('id'),
                                'text': text
                        })
            
            # Store the article with its eId and subdivisions
            self.articles.append({
                'eId': eId,
                'num': article_num,
                'heading': article_title,
                'children': children
            })


    def get_conclusions(self):
        """
        Extracts conclusions from the HTML, if present.
        """
        conclusions_element = self.root.find('div', class_='oj-final')
        self.conclusions = conclusions_element.get_text(separator=' ', strip=True)

    def parse(self, file):
        return super().parse(file)
        

def main():
    parser = argparse.ArgumentParser(description='Parse an Cellar XHTML document and output the results to a JSON file.')
    parser.add_argument('--input', type=str, default='tests/data/html/c008bcb6-e7ec-11ee-9ea8-01aa75ed71a1.0006.03/DOC_1.html', help='Path to the Cellar XHTML file to parse.')
    parser.add_argument('--output', type=str, default='tests/data/json/iopa_html.json', help='Path to the output JSON file.')
    args = parser.parse_args()
    
    html_parser = CellarHTMLParser()
    html_parser.parse(args.input)
    
    with open(args.output, 'w', encoding='utf-8') as f:
        # Get the parser's attributes as a dictionary
        parser_dict = html_parser.__dict__
    
        # Filter out non-serializable attributes
        serializable_dict = {k: v for k, v in parser_dict.items() if isinstance(v, (str, int, float, bool, list, dict, type(None)))}
    
        # Write to a JSON file
        json.dump(serializable_dict, f, ensure_ascii=False, indent=4)

if __name__ == "__main__":
    main()
````

## File: tulit/parsers/html/veneto.py
````python
from tulit.parsers.html.xhtml import HTMLParser
import json
import re
import argparse

class VenetoHTMLParser(HTMLParser):
    def __init__(self):
        pass
    
    def get_root(self, file):
        super().get_root(file)
        
        self.root = self.root.find_all('div', class_="row testo")[0]

    def get_preface(self):
        """
        Extracts the preface text from the HTML, if available.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted preface is stored in the 'preface' attribute.
        """
        try:
            preface_element = self.root.find('title')
            if preface_element:
                self.preface = preface_element.get_text(separator=' ', strip=True)
                print("Preface extracted successfully.")
            else:
                self.preface = None
                print("No preface found.")
        except Exception as e:
            print(f"Error extracting preface: {e}")
    
            
    def get_preamble(self):
        """
        Extracts the preamble text from the HTML, if available.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted preamble is stored in the 'preamble' attribute.
        """
        
        pass
        # self.preamble = self.root.find('div')        
        
            
    
    def get_formula(self):
        """
        Extracts the formula from the HTML, if present.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted formula is stored in the 'formula' attribute.
        """
        pass
        # self.formula = self.preamble.find('p', class_='oj-normal').text


    
    def get_citations(self):
        """
        Extracts citations from the HTML.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted citations are stored in the 'citations' attribute
        """
        self.citations = []
        pass

    def get_recitals(self):
        """
        Extracts recitals from the HTML.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted recitals are stored in the 'recitals' attribute.
        """
        self.recitals = []
        subtitle = self.root.find('b')        
        self.recitals.append({
                    'eId' : 0,
                    'text' : subtitle.text
                }
        )
    def get_preamble_final(self):
        """
        Extracts the final preamble text from the HTML, if available.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted final preamble is stored in the 'preamble_final' attribute.
        """
        pass

    def get_body(self):
        """
        Extracts the body content from the HTML.
        
        Parameters
        ----------
        None
        
        Returns
        -------
        None
            The extracted body content is stored in the 'body' attribute
        """
        pass
    
    def get_chapters(self):
        """
        Extracts chapters from the HTML, grouping them by their IDs and headings.
        """
        
        chapters = self.root.find_all('h3', class_='TITOLOCAPOTITOLO')
        chapters = self.root.find_all('h4', class_='TITOLOCAPOCAPO')

        self.chapters = []
        for index, chapter in enumerate(chapters):
            eId = index
            text = chapter.get_text(strip=True)
            num = text.split('-')[0].strip()
            heading = text.split('-')[1].strip()
            self.chapters.append({
                'eId': eId,
                'num': num,
                'heading': heading
            })



    def get_articles(self):
        """
        Extracts articles from the HTML. Each <div> with an id starting with "art" is treated as an article (eId).
        Subsequent subdivisions are processed based on the closest parent with an id.

        Returns:
            list[dict]: List of articles, each containing its eId and associated content.
        """
        
        articles = self.root.find_all('h6')
        self.articles = []
        
        for index, article in enumerate(articles):
            eId = index
            
            text = article.get_text(strip=True)
            text = text.replace('–', '-')
            
            num = text.split('-')[0].strip() 
            heading = text.split('-')[1].strip()
            
            children = []
            
            # Get the next sibling of the h6 tag, which should be the div containing the article content
            content_div = article.find_next_sibling('div')
            # Within the content div, separate all elements based on the presence
            # of a <br> tag, and store them as separate children
            if content_div:
                
                for element_index, element in enumerate(content_div.descendants):
                    separated_content = []
                    # Print the tag name for debugging
                    # print(element.name)
                    
                    
                    # element index needs to be lower than the index of the first <br> tag
                    if element.name == 'br' or element_index < list(content_div.descendants).index(content_div.find('br')):
                        previous_content = (element.previous_sibling)
                        next_content = (element.next_sibling)
                        # Paste the previous and next siblings together
                        if previous_content:
                            separated_content.append(next_content.get_text(strip=True) if next_content else '')
                        
                        children.append({
                            'eId': element_index,
                            'text': separated_content
                        })
                        
            # Store the article with its eId and subdivisions
            self.articles.append({
                'eId': eId,
                'num': num,
                'heading': heading,
                'children': children
            })


    def get_conclusions(self):
        """
        Extracts conclusions from the HTML, if present.
        """
        conclusions_element = self.root.find('div', class_='oj-final')
        self.conclusions = conclusions_element.get_text(separator=' ', strip=True)

    def parse(self, file):
        return super().parse(file)
        

def main():
    parser = argparse.ArgumentParser(description='Parse an Cellar XHTML document and output the results to a JSON file.')
    parser.add_argument('--input', type=str, default='tests/data/html/veneto/legge.html', help='Path to the Cellar XHTML file to parse.')
    parser.add_argument('--output', type=str, default='tests/data/json/veneto_html.json', help='Path to the output JSON file.')
    args = parser.parse_args()
    
    html_parser = VenetoHTMLParser()
    html_parser.parse(args.input)
    
    with open(args.output, 'w', encoding='utf-8') as f:
        # Get the parser's attributes as a dictionary
        parser_dict = html_parser.__dict__
    
        # Filter out non-serializable attributes
        serializable_dict = {k: v for k, v in parser_dict.items() if isinstance(v, (str, int, float, bool, list, dict, type(None)))}
    
        # Write to a JSON file
        json.dump(serializable_dict, f, ensure_ascii=False, indent=4)

if __name__ == "__main__":
    main()
````

## File: tulit/parsers/html/xhtml.py
````python
from bs4 import BeautifulSoup
from tulit.parsers.parser import Parser
import json

class HTMLParser(Parser):
    def __init__(self):
        """
        Initializes the HTML parser and sets up the BeautifulSoup instance.
        """
        super().__init__()
        
    def get_root(self, file):
        """
        Loads an HTML file and parses it with BeautifulSoup.

        Parameters
        ----------
        file : str
            The path to the HTML file.
        
        Returns
        -------
        None
            The root element is stored in the parser under the 'root' attribute.
        """
        try:
            with open(file, 'r', encoding='utf-8') as f:
                html = f.read()
            self.root = BeautifulSoup(html, 'html.parser')
            print("HTML loaded successfully.")
        except Exception as e:
            print(f"Error loading HTML: {e}")
            

    def parse(self, file: str) -> Parser:
        """
        Parses an HTML file and extracts the preface, preamble, formula, citations, recitals, preamble final, body, chapters, articles, and conclusions.
        
        Parameters
        ----------
        file : str
            Path to the XML file to parse.
        
        Returns
        -------
        Parser
            The parser object with the parsed elements stored in the attributes.
        """
            
        try:
            self.get_root(file)
            print("Root element loaded successfully.")
        except Exception as e:
            print(f"Error in get_root: {e}")
            
        try:
            self.get_preface()
            print(f"Preface parsed successfully. Preface: {self.preface}")
        except Exception as e:
            print(f"Error in get_preface: {e}")
        
        try:
            self.get_preamble()
            print(f"Preamble element found.")
        except Exception as e:
            print(f"Error in get_preamble: {e}")
        try:
            self.get_formula()
            print(f"Formula parsed successfully.")
        except Exception as e:
            print(f"Error in get_formula: {e}")
        try:
            self.get_citations()
            print(f"Citations parsed successfully. Number of citations: {len(self.citations)}")
        except Exception as e:
            print(f"Error in get_citations: {e}")
        try:
            self.get_recitals()
            print(f"Recitals parsed successfully. Number of recitals: {len(self.recitals)}")
        except Exception as e:
            print(f"Error in get_recitals: {e}")
        
        try:
            self.get_preamble_final()
            print(f"Preamble final parsed successfully.")
        except Exception as e:
            print(f"Error in get_preamble_final: {e}")
        
        try:
            self.get_body()
            print("Body element found.")
        except Exception as e:
            print(f"Error in get_body: {e}")
        try:
            self.get_chapters()
            print(f"Chapters parsed successfully. Number of chapters: {len(self.chapters)}")
        except Exception as e:
            print(f"Error in get_chapters: {e}")
        try:
            self.get_articles()
            print(f"Articles parsed successfully. Number of articles: {len(self.articles)}")
            print(f"Total number of children in articles: {sum([len(list(article)) for article in self.articles])}")                        
            
        except Exception as e:
            print(f"Error in get_articles: {e}")
        try:
            self.get_conclusions()                    
            print(f"Conclusions parsed successfully. ")
        except Exception as e:
            print(f"Error in get_conclusions: {e}")
                
        return self
````

## File: tulit/parsers/xml/assets/import/xml.xsd
````
<?xml version='1.0'?>
<!DOCTYPE xs:schema PUBLIC "-//W3C//DTD XMLSCHEMA 200102//EN" "XMLSchema.dtd" >
<xs:schema targetNamespace="http://www.w3.org/XML/1998/namespace" xmlns:xs="http://www.w3.org/2001/XMLSchema" xml:lang="en">

 <xs:annotation>
  <xs:documentation>
   See http://www.w3.org/XML/1998/namespace.html and
   http://www.w3.org/TR/REC-xml for information about this namespace.

    This schema document describes the XML namespace, in a form
    suitable for import by other schema documents.  

    Note that local names in this namespace are intended to be defined
    only by the World Wide Web Consortium or its subgroups.  The
    following names are currently defined in this namespace and should
    not be used with conflicting semantics by any Working Group,
    specification, or document instance:

    base (as an attribute name): denotes an attribute whose value
         provides a URI to be used as the base for interpreting any
         relative URIs in the scope of the element on which it
         appears; its value is inherited.  This name is reserved
         by virtue of its definition in the XML Base specification.

    lang (as an attribute name): denotes an attribute whose value
         is a language code for the natural language of the content of
         any element; its value is inherited.  This name is reserved
         by virtue of its definition in the XML specification.
  
    space (as an attribute name): denotes an attribute whose
         value is a keyword indicating what whitespace processing
         discipline is intended for the content of the element; its
         value is inherited.  This name is reserved by virtue of its
         definition in the XML specification.

    Father (in any context at all): denotes Jon Bosak, the chair of 
         the original XML Working Group.  This name is reserved by 
         the following decision of the W3C XML Plenary and 
         XML Coordination groups:

             In appreciation for his vision, leadership and dedication
             the W3C XML Plenary on this 10th day of February, 2000
             reserves for Jon Bosak in perpetuity the XML name
             xml:Father
  </xs:documentation>
 </xs:annotation>

 <xs:annotation>
  <xs:documentation>This schema defines attributes and an attribute group
        suitable for use by
        schemas wishing to allow xml:base, xml:lang or xml:space attributes
        on elements they define.

        To enable this, such a schema must import this schema
        for the XML namespace, e.g. as follows:
        &lt;schema . . .>
         . . .
         &lt;import namespace="http://www.w3.org/XML/1998/namespace"
                    schemaLocation="http://www.w3.org/2001/03/xml.xsd"/>

        Subsequently, qualified reference to any of the attributes
        or the group defined below will have the desired effect, e.g.

        &lt;type . . .>
         . . .
         &lt;attributeGroup ref="xml:specialAttrs"/>
 
         will define a type which will schema-validate an instance
         element with any of those attributes</xs:documentation>
 </xs:annotation>

 <xs:annotation>
  <xs:documentation>In keeping with the XML Schema WG's standard versioning
   policy, this schema document will persist at
   http://www.w3.org/2001/03/xml.xsd.
   At the date of issue it can also be found at
   http://www.w3.org/2001/xml.xsd.
   The schema document at that URI may however change in the future,
   in order to remain compatible with the latest version of XML Schema
   itself.  In other words, if the XML Schema namespace changes, the version
   of this document at
   http://www.w3.org/2001/xml.xsd will change
   accordingly; the version at
   http://www.w3.org/2001/03/xml.xsd will not change.
  </xs:documentation>
 </xs:annotation>

 <xs:attribute name="lang" type="xs:language">
  <xs:annotation>
   <xs:documentation>In due course, we should install the relevant ISO 2- and 3-letter
         codes as the enumerated possible values . . .</xs:documentation>
  </xs:annotation>
 </xs:attribute>

 <xs:attribute name="space" default="preserve">
  <xs:simpleType>
   <xs:restriction base="xs:NCName">
    <xs:enumeration value="default"/>
    <xs:enumeration value="preserve"/>
   </xs:restriction>
  </xs:simpleType>
 </xs:attribute>

 <xs:attribute name="base" type="xs:anyURI">
  <xs:annotation>
   <xs:documentation>See http://www.w3.org/TR/xmlbase/ for
                     information about this attribute.</xs:documentation>
  </xs:annotation>
 </xs:attribute>

 <xs:attributeGroup name="specialAttrs">
  <xs:attribute ref="xml:base"/>
  <xs:attribute ref="xml:lang"/>
  <xs:attribute ref="xml:space"/>
 </xs:attributeGroup>

</xs:schema>
````

## File: tulit/parsers/xml/assets/akomantoso30.xsd
````
<?xml version="1.0" encoding="UTF-8"?>
<!--
     Akoma Ntoso Version 1.0. Part 2: Specifications
     Committee Specification Draft 02 / Public Review Draft 02
     04 May 2016
     Copyright (c) OASIS Open 2016. All Rights Reserved.
     Source: http://docs.oasis-open.org/legaldocml/akn-core/v1.0/csprd02/part2-specs/schemas/
     Latest version of the specification: http://docs.oasis-open.org.legaldocml/akn-core/v1.0/akn-core-v1.0-part2-specs.html
     TC IPR Statement: https://www.oasis-open.org/committees/legaldocml/ipr.php
-->
<xsd:schema xmlns="http://docs.oasis-open.org/legaldocml/ns/akn/3.0"
            xmlns:xsd="http://www.w3.org/2001/XMLSchema"
            xmlns:xml="http://www.w3.org/XML/1998/namespace"
            targetNamespace="http://docs.oasis-open.org/legaldocml/ns/akn/3.0"
            elementFormDefault="qualified">

	<xsd:import namespace="http://www.w3.org/XML/1998/namespace"
				schemaLocation="./xml.xsd"/>


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		Akoma Ntoso main schema
		
		Release 01/04/2016 - Akoma Ntoso 3.0 WD17 with errata
		Complete version.
		
		
		technical supervision Fabio Vitali - University of Bologna
		legal domain supervision Monica Palmirani - University of Bologna
		
		The present schema uses systematically XML design patterns as
		described in http://www.xmlpatterns.com/
		A few of them are explicitly mentioned, when their use or approach
		was not felt clear or natural enough. Others are just used without
		mentions.
				
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Main element groups
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:group name="ANhier">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANhier</name>
            <comment>
The group ANhier lists the elements that belong to the Akoma Ntoso legislative hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="clause"/>
			      <xsd:element ref="section"/>
			      <xsd:element ref="part"/>
			      <xsd:element ref="paragraph"/>
			      <xsd:element ref="chapter"/>
			      <xsd:element ref="title"/>
			      <xsd:element ref="article"/>
			      <xsd:element ref="book"/>
			      <xsd:element ref="tome"/>
			      <xsd:element ref="division"/>
			      <xsd:element ref="list"/>
			      <xsd:element ref="point"/>
			      <xsd:element ref="indent"/>
			      <xsd:element ref="alinea"/>
			      <xsd:element ref="rule"/>
			      <xsd:element ref="subrule"/>
			      <xsd:element ref="proviso"/>
			      <xsd:element ref="subsection"/>
			      <xsd:element ref="subpart"/>
			      <xsd:element ref="subparagraph"/>
			      <xsd:element ref="subchapter"/>
			      <xsd:element ref="subtitle"/>
			      <xsd:element ref="subdivision"/>
			      <xsd:element ref="subclause"/>
			      <xsd:element ref="sublist"/>
			      <xsd:element ref="level"/>
			      <xsd:element ref="transitional"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANcontainers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANcontainers</name>
            <comment>
The group ANcontainers lists the elements that are containers and are specific to the Akoma Ntoso debate vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="speechGroup"/>
			      <xsd:element ref="speech"/>
			      <xsd:element ref="question"/>
			      <xsd:element ref="answer"/>
			      <xsd:element ref="other"/>
			      <xsd:element ref="scene"/>
			      <xsd:element ref="narrative"/>
			      <xsd:element ref="summary"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="basicContainers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>basicContainers</name>
            <comment>
The group basicContainers lists the elements that are containers and are specific to vocabulary of preambles, prefaces, conclusions and coverPages</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="longTitle"/>
			      <xsd:element ref="formula"/>
			      <xsd:element ref="container"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="preambleContainers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>preambleContainers</name>
            <comment>
The group preambleContainers lists the elements that are containers and are specific to the Akoma Ntoso preamble vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="recitals"/>
			      <xsd:element ref="citations"/>
			      <xsd:element ref="formula"/>
			      <xsd:element ref="container"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="prefaceContainers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>prefaceContainers</name>
            <comment>
The group prefaceContainers lists the elements that are containers and are specific to the Akoma Ntoso preface vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="longTitle"/>
			      <xsd:element ref="formula"/>
			      <xsd:element ref="container"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANblock">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANblock</name>
            <comment>
The group ANblock lists the elements that are blocks and are specific to the Akoma Ntoso vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="blockList"/>
			      <xsd:element ref="blockContainer"/>
			      <xsd:element ref="tblock"/>
			      <xsd:element ref="toc"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANinline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANinline</name>
            <comment>
The group ANinline lists the elements that are inline and are specific to the Akoma Ntoso vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="ref"/>
			      <xsd:element ref="mref"/>
			      <xsd:element ref="rref"/>
			      <xsd:element ref="mod"/>
			      <xsd:element ref="mmod"/>
			      <xsd:element ref="rmod"/>
			      <xsd:element ref="remark"/>
			      <xsd:element ref="recordedTime"/>
			      <xsd:element ref="vote"/>
			      <xsd:element ref="outcome"/>
			      <xsd:element ref="ins"/>
			      <xsd:element ref="del"/>
			      <xsd:element ref="omissis"/>
			      <xsd:element ref="embeddedText"/>
			      <xsd:element ref="embeddedStructure"/>
			      <xsd:element ref="placeholder"/>
			      <xsd:element ref="fillIn"/>
			      <xsd:element ref="decoration"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANtitleInline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANtitleInline</name>
            <comment>
The group ANtitleInline lists the elements that are inline, are specific to the Akoma Ntoso vocabulary, and should only be used within the title element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="docType"/>
			      <xsd:element ref="docTitle"/>
			      <xsd:element ref="docNumber"/>
			      <xsd:element ref="docProponent"/>
			      <xsd:element ref="docDate"/>
			      <xsd:element ref="legislature"/>
			      <xsd:element ref="session"/>
			      <xsd:element ref="shortTitle"/>
			      <xsd:element ref="docAuthority"/>
			      <xsd:element ref="docPurpose"/>
			      <xsd:element ref="docCommittee"/>
			      <xsd:element ref="docIntroducer"/>
			      <xsd:element ref="docStage"/>
			      <xsd:element ref="docStatus"/>
			      <xsd:element ref="docJurisdiction"/>
			      <xsd:element ref="docketNumber"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANheaderInline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANheaderInline</name>
            <comment>
The group ANheaderInline lists the elements that are inline, are specific to the Akoma Ntoso vocabulary, and should only be used within the header element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="courtType"/>
			      <xsd:element ref="neutralCitation"/>
			      <xsd:element ref="party"/>
			      <xsd:element ref="judge"/>
			      <xsd:element ref="lawyer"/>
			      <xsd:element ref="signature"/>
			      <xsd:element ref="opinion"/>
			      <xsd:element ref="argument"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANsemanticInline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANsemanticInline</name>
            <comment>
The group ANsemanticInline lists additional elements that are inline, and are specific to the Akoma Ntoso vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="date"/>
			      <xsd:element ref="time"/>
			      <xsd:element ref="person"/>
			      <xsd:element ref="organization"/>
			      <xsd:element ref="concept"/>
			      <xsd:element ref="object"/>
			      <xsd:element ref="event"/>
			      <xsd:element ref="location"/>
			      <xsd:element ref="process"/>
			      <xsd:element ref="role"/>
			      <xsd:element ref="term"/>
			      <xsd:element ref="quantity"/>
			      <xsd:element ref="def"/>
			      <xsd:element ref="entity"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANmarker">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANmarker</name>
            <comment>
The group ANmarker lists the elements that are markers and are specific to the Akoma Ntoso vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="noteRef"/>
			      <xsd:element ref="eol"/>
			      <xsd:element ref="eop"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="ANsubFlow">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>ANsubFlow</name>
            <comment>
The group ANsubFlow lists the elements that are subFlows and are specific to the Akoma Ntoso vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="authorialNote"/>
		    </xsd:sequence>
	  </xsd:group>

	  <xsd:group name="HTMLcontainers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>HTMLcontainers</name>
            <comment>
The group HTMLcontainers lists the elements that are containers and were inherited from the HTML vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="div"/>
		    </xsd:sequence>
	  </xsd:group>

	  <xsd:group name="HTMLblock">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>HTMLblock</name>
            <comment>
The group HTMLblock lists the elements that are blocks and were inherited from the HTML vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="ul"/>
			      <xsd:element ref="ol"/>
			      <xsd:element ref="table"/>
			      <xsd:element ref="p"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="HTMLinline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>HTMLinline</name>
            <comment>
The group HTMLinline lists the elements that are inline and were inherited from the HTML vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="b"/>
			      <xsd:element ref="i"/>
			      <xsd:element ref="a"/>
			      <xsd:element ref="u"/>
			      <xsd:element ref="sub"/>
			      <xsd:element ref="sup"/>
			      <xsd:element ref="abbr"/>
			      <xsd:element ref="span"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="HTMLmarker">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>HTMLmarker</name>
            <comment>
The group HTMLmarker lists the elements that are marker and were inherited from the HTML vocabulary</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="img"/>
			      <xsd:element ref="br"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="judgmentBlock">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>judgmentBlock</name>
            <comment>
The group judgmentBlock lists the structures that should be found in a judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="introduction"/>
			      <xsd:element ref="background"/>
			      <xsd:element ref="arguments"/>
			      <xsd:element ref="remedies"/>
			      <xsd:element ref="motivation"/>
			      <xsd:element ref="decision"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="amendmentBlock">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>amendmentBlock</name>
            <comment>
The group amendmentBlock lists the structures that should be found in an amendment</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="amendmentHeading"/>
			      <xsd:element ref="amendmentContent"/>
			      <xsd:element ref="amendmentReference"/>
			      <xsd:element ref="amendmentJustification"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="amendmentInline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>amendmentInline</name>
            <comment>
The group amendmentInline lists the inline elements that should be found in the preface of an amendment</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="affectedDocument"/>
			      <xsd:element ref="relatedDocument"/>
			      <xsd:element ref="change"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="speechSection">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>speechSection</name>
            <comment>
The group speechSection lists the structures that should contain speeches</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="administrationOfOath"/>
			      <xsd:element ref="rollCall"/>
			      <xsd:element ref="prayers"/>
			      <xsd:element ref="oralStatements"/>
			      <xsd:element ref="writtenStatements"/>
			      <xsd:element ref="personalStatements"/>
			      <xsd:element ref="ministerialStatements"/>
			      <xsd:element ref="resolutions"/>
			      <xsd:element ref="nationalInterest"/>
			      <xsd:element ref="declarationOfVote"/>
			      <xsd:element ref="communication"/>
			      <xsd:element ref="petitions"/>
			      <xsd:element ref="papers"/>
			      <xsd:element ref="noticesOfMotion"/>
			      <xsd:element ref="questions"/>
			      <xsd:element ref="address"/>
			      <xsd:element ref="proceduralMotions"/>
			      <xsd:element ref="pointOfOrder"/>
			      <xsd:element ref="adjournment"/>
			      <xsd:element ref="debateSection"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="hierElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>hierElements</name>
            <comment>
The group hierElements lists all the elements that are hierarchical</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="ANhier"/>
			      <xsd:element ref="hcontainer"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="containerElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>containerElements</name>
            <comment>
The group containerElements lists all the elements that are containers</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="speechSection"/>
			      <xsd:group ref="HTMLcontainers"/>
			      <xsd:element ref="container"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="blockElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>blockElements</name>
            <comment>
The group blockElements lists all the elements that are blocks</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="ANblock"/>
			      <xsd:group ref="HTMLblock"/>
			      <xsd:element ref="foreign"/>
			      <xsd:element ref="block"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="inlineElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>inlineElements</name>
            <comment>
The group inlineElements lists all the elements that are inline</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="ANinline"/>
			      <xsd:group ref="HTMLinline"/>
			      <xsd:group ref="ANtitleInline"/>
			      <xsd:group ref="ANsemanticInline"/>
			      <xsd:group ref="ANheaderInline"/>
			      <xsd:group ref="amendmentInline"/>
			      <xsd:element ref="inline"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="subFlowElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>subFlowElements</name>
            <comment>
The group subFlowElements lists all the elements that are subFlows</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="ANsubFlow"/>
			      <xsd:element ref="subFlow"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="markerElements">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>markerElements</name>
            <comment>
The group markerElements lists all the elements that are markers</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="ANmarker"/>
			      <xsd:group ref="HTMLmarker"/>
			      <xsd:element ref="marker"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="inlineCM">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>inlineCM</name>
            <comment>
The group inlineCM is the content model of blocks and inlines, and is composed of all the inlines and all the markers</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="inlineElements"/>
			      <xsd:group ref="markerElements"/>
			      <xsd:group ref="subFlowElements"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Attribute groups
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:attributeGroup name="contains">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>contains</name>
            <comment>
The attribute contains is used in the root of document types to indicate whether the document is original or modified wrt the orignal version</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="contains" type="versionType" default="originalVersion"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="alt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>alt</name>
            <comment>
The attribute alternativeTo is used to specify, when the element contains an alternative version of some content, the eId of the main element which this element is an alternative copy of</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="alternativeTo" type="eIdRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="name">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>name</name>
            <comment>
The attribute name is used to give a name to all generic elements</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="name" type="xsd:string" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="number">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>number</name>
            <comment>
The attribute number is used to specify a number in the publication of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="number" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="source">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>source</name>
            <comment>
The attribute source links to the agent (person, organization) providing the specific annotation or markup</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="source" type="agentRef" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="date">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>date</name>
            <comment>
The attribute date is used to give a normalized value for a date according to the XSD syntax YYYY-MM-DD or a normalized value for a dateTime according to the XSD syntax YYYY-MM-DDThh:mm:ss(zzzz)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="date" use="required">
			      <xsd:simpleType>
				        <xsd:union>
					          <xsd:simpleType>
						            <xsd:restriction base="xsd:date"/>
					          </xsd:simpleType>
					          <xsd:simpleType>
						            <xsd:restriction base="xsd:dateTime"/>
					          </xsd:simpleType>
				        </xsd:union>
			      </xsd:simpleType>
		    </xsd:attribute>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="time">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>time</name>
            <comment>
The attribute time is used to give a normalized value for a time according to the XSD syntax HH:MM:SS</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="time" use="required">
			      <xsd:simpleType>
				        <xsd:union>
					          <xsd:simpleType>
						            <xsd:restriction base="xsd:time"/>
					          </xsd:simpleType>
					          <xsd:simpleType>
						            <xsd:restriction base="xsd:dateTime"/>
					          </xsd:simpleType>
				        </xsd:union>
			      </xsd:simpleType>
		    </xsd:attribute>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="link">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>link</name>
            <comment>
The attribute href is used to specify an internal or external destination for a reference, a citation, an access to the ontology or a hypertext link. In elements using this attribute definition the href attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="href" type="xsd:anyURI" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="linkopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>linkopt</name>
            <comment>
The attribute href is used to specify an internal or external destination for a reference, a citation, an access to the ontology or a hypertext link. In elements using this attribute definition the href attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="href" type="xsd:anyURI"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="value">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>value</name>
            <comment>
The attribute value is used to specify a value for metadata elements. In elements using this attribute definition the value attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="value" type="xsd:string" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="optvalue">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>optvalue</name>
            <comment>
The attribute value is used to specify a value for metadata elements. In elements using this attribute definition the value attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="value" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="booleanvalue">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>booleanvalue</name>
            <comment>
The attribute value is used here to specify a boolean value for metadata elements. In elements using this attribute definition the value attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="value" type="xsd:boolean" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="interval">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>interval</name>
            <comment>
The start and end attributes are used to specify events that mark the beginning and the end of a time interval. Both are references to eventRef elements in the lifecycle element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="start" type="eventRefRef"/>
		    <xsd:attribute name="end" type="eventRefRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="duration">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>duration</name>
            <comment>
The duration attribute is used to specify a duration when either the start or the end event of a time interval is not known.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="duration" type="xsd:duration"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="speechAtts">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>speechAtts</name>
            <comment>
The attributes in speechAtts are used in speeches to identify agents and roles of speeches. In particular, attribute 'by' identifies the speaker, optional attribute 'as' identifies the role under which the speaker is speaking, optional attribute startTime specifies the absolute date and time where the individual speech item started, optional attribute endTime specifies the absolute date and time where the individual speech item ended, and  optional attribute to identifies the addressee of the speech. All of them are references to person or organization elements in the references section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="agent"/>
		    <xsd:attributeGroup ref="role"/>
		    <xsd:attribute name="startTime" type="xsd:dateTime"/>
		    <xsd:attribute name="endTime" type="xsd:dateTime"/>
		    <xsd:attribute name="to" type="agentRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="voteAtts">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>voteAtts</name>
            <comment>
The attributes in voteAtts are used in votes to identify agents and choices in votes. In particular, attribute 'by' identifies the voter, optional attribute 'as' identifies the role under which the voter is acting, optional attribute 'choice' specifies the vote that was casted.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="agent"/>
		    <xsd:attributeGroup ref="role"/>
		    <xsd:attribute name="choice" type="voteRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="frozen">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>frozen</name>
            <comment>
The attribute frozen for the condition element specifies whether the condition is frozen or not in the time period of this expression.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="frozen" type="xsd:boolean"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="lawyerAtts">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>lawyerAtts</name>
            <comment>
The attributes in lawyerAtts are used in the inline element lawyer to identify representation aspects of lawyers. In particular, attribute 'as' identifies his/her role, attribute 'for' identifies which party it represents, and attribute 'empoweredBy' identifies another lawyer, if any, this lawyer received the power delegation of power.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="role"/>
		    <xsd:attribute name="for" type="agentRef"/>
		    <xsd:attribute name="empoweredBy" type="agentRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="normalizedAtt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>normalizedAtt</name>
            <comment>
The attribute normalized is used in the inline element quantity to provide a normalized value of the number, if appropriate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="normalized" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="for">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>for</name>
            <comment>
The attribute for is used in modification elements to specify the element this is a modification of.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="for" type="eIdRef"/>
	  </xsd:attributeGroup>

	  <xsd:annotation>
		    <xsd:documentation>
		 The mappingAtts attributes represent the reference to the permanent wId (attribute original) of a structure, and to the eId (attribute current) such structure had during the time interval included between an initial temporalGroup (attribute start) and a final temporalGroup (attribute end). 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:attributeGroup name="mappingAtts">
		

		    <xsd:attribute name="original" type="wIdRef"/>
		    <xsd:attribute name="current" type="eIdRef"/>
		    <xsd:attribute name="start" type="eventRefRef"/>
		    <xsd:attribute name="end" type="eventRefRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="originating">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>originating</name>
            <comment>
The attribute originating is used to identify whether the event is the one originating this expression</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="originatingExpression" type="xsd:boolean"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="outcome">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>outcome</name>
            <comment>
The attribute outcome is used to identify the outcome of a step in a workflow. It is a reference to a TLCConcept element in the references section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="outcome" type="conceptRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="portionAtt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>portionAtt</name>
            <comment>
the attribute includedIn specifies the eId of the element of the references section identifying the document that this portion is a portion of.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="includedIn" type="referenceRef" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="level">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>level</name>
            <comment>
the attribute level specifies the level of the toc element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="level" type="xsd:integer" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="target">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>target</name>
            <comment>
the attribute target specifies the target of the a element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="target" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="imgAtts">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>imgAtts</name>
            <comment>
these attributes specify the dimension of the image.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="width" type="xsd:integer"/>
		    <xsd:attribute name="height" type="xsd:integer"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="tableAtts">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>tableAtts</name>
            <comment>
these attributes specify the characteristics of the table.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="width" type="xsd:integer"/>
		    <xsd:attribute name="border" type="xsd:integer"/>
		    <xsd:attribute name="cellspacing" type="xsd:integer"/>
		    <xsd:attribute name="cellpadding" type="xsd:integer"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="originalText">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>originalText</name>
            <comment>
the attribute originalText contains the original text of a placeholder</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="originalText" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="fillInWidth">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>fillInWidth</name>
            <comment>
the attribute width of the element fillIn specifies the width of the fillIn element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="width" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="upTo">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>upTo</name>
            <comment>
The attribute upTo is used in range-based elements to specify the eId of the ending element of the range.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="upTo" type="eIdRef" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="upToOpt">
		

		    <xsd:attribute name="upTo" type="eIdRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="range">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>range</name>
            <comment>
These attributes are used in range-based elements to specify the eId of the beginning and ending element of the range.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="from" type="eIdRef" use="required"/>
		    <xsd:attributeGroup ref="upTo"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="rangeOpt">
		

		    <xsd:attribute name="from" type="eIdRef" use="required"/>
		    <xsd:attributeGroup ref="upToOpt"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="pos">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>pos</name>
            <comment>
The attribute pos is used to identify the specific position of the reference (e.g., in source or destination)
with respect to the element being identified with the relative eId.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="pos" type="posType"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="language">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>language</name>
            <comment>
This attribute specifies the human language in which the document the element refers to is expressed. Values are taken from the RFC 4646.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="language" type="language" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="fromLanguage">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>fromLanguage</name>
            <comment>
This attribute specifies the human language of the expression this is a translation of. Values are taken from the RFC 4646.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="fromLanguage" type="language" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="pivot">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>pivot</name>
            <comment>
This attribute specifies the human language used as a pivot in the translation. Values are taken from the RFC 4646.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="pivot" type="language"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="authoritative">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>authoritative</name>
            <comment>
This attribute specifies whether the translation is authoritative or not</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="authoritative" type="xsd:boolean"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="dictionary">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>dictionary</name>
            <comment>
This attribute specifies the eId of the element in the references section detailing the dictionary used for the keyword.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="dictionary" type="referenceRef" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Attribute groups: attribute "type"
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:attributeGroup name="type">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>type</name>
            <comment>
The type attribute is used in various element to identify a type without restrictions as to its values.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="type" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="eventType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>eventType</name>
            <comment>
This is the list of allowed values for the type attribute of the event and action elements</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="generation"/>
			      <xsd:enumeration value="amendment"/>
			      <xsd:enumeration value="repeal"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="eventType">
		

		    <xsd:attribute name="type" type="eventType"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="opinionType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>opinionType</name>
            <comment>
This is the list of allowed values for the type attribute of the opinion element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="dissenting"/>
			      <xsd:enumeration value="agreeing"/>
			      <xsd:enumeration value="null"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="opinionType">
		

		    <xsd:attribute name="type" type="opinionType"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="resultType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>resultType</name>
            <comment>
This is the list of allowed values for the type attribute of the result element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="deny"/>
			      <xsd:enumeration value="dismiss"/>
			      <xsd:enumeration value="uphold"/>
			      <xsd:enumeration value="revert"/>
			      <xsd:enumeration value="replaceOrder"/>
			      <xsd:enumeration value="remit"/>
			      <xsd:enumeration value="decide"/>
			      <xsd:enumeration value="approve"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="resultType">
		

		    <xsd:attribute name="type" type="resultType" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="restrictionType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>restrictionType</name>
            <comment>
This is the list of allowed values for the restriction type attribute</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="jurisdiction"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="restrictionType">
		

		    <xsd:attribute name="type" type="restrictionType" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="remarkType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>remarkType</name>
            <comment>
This is the list of allowed values for the type attribute of the remark element for the specification of editorial remarks (e.g., applauses, laughters, etc.) especially within debate records</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="sceneDescription"/>
			      <xsd:enumeration value="phenomenon"/>
			      <xsd:enumeration value="caption"/>
			      <xsd:enumeration value="translation"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="remarkType">
		

		    <xsd:attribute name="type" type="remarkType"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="timeType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>timeType</name>
            <comment>
This is the list of allowed values for the type attribute of the recordedTime element for the specification of an explicit mention of a time (e.g., in a debate)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="startEvent"/>
			      <xsd:enumeration value="endEvent"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="recordedTimeType">
		

		    <xsd:attribute name="type" type="timeType"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="TextualMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>TextualMods</name>
            <comment>
The simple type TextualMods lists all the types of textual modifications as values for the type attribute of the textualMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="repeal"/>
			      <xsd:enumeration value="substitution"/>
			      <xsd:enumeration value="insertion"/>
			      <xsd:enumeration value="replacement"/>
			      <xsd:enumeration value="renumbering"/>
			      <xsd:enumeration value="split"/>
			      <xsd:enumeration value="join"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="textualModType">
		

		    <xsd:attribute name="type" type="TextualMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="MeaningMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>MeaningMods</name>
            <comment>
The simple type MeaningMods lists all the types of modifications in meaning as values for the type attribute of the meaningMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="variation"/>
			      <xsd:enumeration value="termModification"/>
			      <xsd:enumeration value="authenticInterpretation"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="meaningModType">
		

		    <xsd:attribute name="type" type="MeaningMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="ScopeMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>ScopeMods</name>
            <comment>
The simple type ScopeMods lists all the types of modifications in scope as values for the type attribute of the scopeMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="exceptionOfScope"/>
			      <xsd:enumeration value="extensionOfScope"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="scopeModType">
		

		    <xsd:attribute name="type" type="ScopeMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="ForceMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>ForceMods</name>
            <comment>
The simple type ForceMods lists all the types of modifications in force as values for the type attribute of the forceMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="entryIntoForce"/>
			      <xsd:enumeration value="endOfEnactment"/>
			      <xsd:enumeration value="postponementOfEntryIntoForce"/>
			      <xsd:enumeration value="prorogationOfForce"/>
			      <xsd:enumeration value="reEnactment"/>
			      <xsd:enumeration value="unconstitutionality"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="forceModType">
		

		    <xsd:attribute name="type" type="ForceMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="EfficacyMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>EfficacyMods</name>
            <comment>
The simple type EfficacyMods lists all the types of modifications in efficacy as values for the type attribute of the efficacyMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="entryIntoEfficacy"/>
			      <xsd:enumeration value="endOfEfficacy"/>
			      <xsd:enumeration value="inapplication"/>
			      <xsd:enumeration value="retroactivity"/>
			      <xsd:enumeration value="extraefficacy"/>
			      <xsd:enumeration value="postponementOfEfficacy"/>
			      <xsd:enumeration value="prorogationOfEfficacy"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="efficacyModType">
		

		    <xsd:attribute name="type" type="EfficacyMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:simpleType name="LegalSystemMods">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>LegalSystemMods</name>
            <comment>
The simple type LegalSystemMods lists all the types of modifications in the legal system as values for the type attribute of the LegalSystemMod element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="staticReference"/>
			      <xsd:enumeration value="implementation"/>
			      <xsd:enumeration value="ratification"/>
			      <xsd:enumeration value="application"/>
			      <xsd:enumeration value="legislativeDelegation"/>
			      <xsd:enumeration value="deregulation"/>
			      <xsd:enumeration value="conversion"/>
			      <xsd:enumeration value="expiration"/>
			      <xsd:enumeration value="reiteration"/>
			      <xsd:enumeration value="remaking"/>
			      <xsd:enumeration value="republication"/>
			      <xsd:enumeration value="coordination"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:attributeGroup name="legalSystemModType">
		

		    <xsd:attribute name="type" type="LegalSystemMods" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="show">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>show</name>
            <comment>
These attributes are used in metadata to propose visible representations of the metadata itself. Both a full visualization (attribute showAs) and an abbreviated one (attribute shortForm) can be specified</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="showAs" type="xsd:string" use="required"/>
		    <xsd:attribute name="shortForm" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="showopt">
		

		    <xsd:attribute name="showAs" type="xsd:string"/>
		    <xsd:attribute name="shortForm" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="src">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>src</name>
            <comment>
These attributes are used in manifestation-level references to specify external manifestation-level resources to be loaded in place. The src attribute holds the manifestation-level IRI of the resource, while the alt attribute holds the text to be displayed in case the loading of the external resource is not possible for any reason.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="src" type="manifestationURI" use="required"/>
		    <xsd:attribute name="alt" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="period">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>period</name>
            <comment>
The period attribute is used in versioned content and metadata elements to indicate a time interval in which they were in force, in efficacy, or in any other type of interval as specified in the corresponding temporalGroup.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="period" type="temporalGroupRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="enactment">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>enactment</name>
            <comment>
These attributes are those already defined in attribute list "period", plus the attribute status, that allows to specify the status of the piece of text it wraps.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="period"/>
		    <xsd:attribute name="status" type="statusType"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="notes">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>notes</name>
            <comment>
These attributes are used by notes, both authorial and editorial. The marker is the string to be used as a marker of the note, the placement is the position where the note should be placed, and placementBase is the eId of the element respect to which the note should be placed near.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="marker" type="xsd:string"/>
		    <xsd:attribute name="placement" type="placementType"/>
		    <xsd:attribute name="placementBase" type="eIdRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="modifiers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>modifiers</name>
            <comment>
These attributes are used in the analysis to allow manifestation editors to specify whether the analysis is complete and/or ignored in the text</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="exclusion" type="xsd:boolean"/>
		    <xsd:attribute name="incomplete" type="xsd:boolean"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="role">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>role</name>
            <comment>
The attribute role is used to identify the role of an individual mentioned in the text. It is a reference to a TLCRole element in the references section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="as" type="roleRef"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="agent">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>agent</name>
            <comment>
The attribute by is used to identify the agent of an action. It is a reference to a TLCPerson or TLCOrganization element in the references section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="by" type="agentRef" use="required"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="quote">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>quote</name>
            <comment>
The attributes startQuote and endQuote are used in quotedText, quotedStructure, embeddedText and embeddedStructure to specify the start and quote character delimiting the quoted or embedded part. If the characters are the same, one can use only startQuote.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="startQuote" type="xsd:string"/>
		    <xsd:attribute name="endQuote" type="xsd:string"/>
		    <xsd:attribute name="inlineQuote" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="cellattrs">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>cellattrs</name>
            <comment>
These attributes are used to specify that table cells span more than one row or one column, exactly as in HTML</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="rowspan" type="xsd:integer" default="1"/>
		    <xsd:attribute name="colspan" type="xsd:integer" default="1"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="HTMLattrs">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>HTMLattrs</name>
            <comment>
These attributes are used to specify class, style and title of the element, exactly as in HTML</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="class" type="xsd:string"/>
		    <xsd:attribute name="style" type="xsd:string"/>
		    <xsd:attribute name="title" type="xsd:string"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="core">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>core</name>
            <comment>
This attribute list are used to specify the fact that any attribute can be specified for this element if it belongs to a different namespace.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:anyAttribute namespace="##other" processContents="lax"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="idreq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>idreq</name>
            <comment>
These attributes identify the element in an absolute manner. In elements using this attribute definition the eId attribute is required whenever the Akoma Ntoso naming Convention is used. The wId is used to mark the identifier that the structure used to have in the original version, and is only needed when a renumbering occurred.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="eId" type="noWhiteSpace"/>
		    <xsd:attribute name="wId" type="noWhiteSpace"/>
		    <xsd:attribute name="GUID" type="noWhiteSpace"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="idopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>idopt</name>
            <comment>
These attributes identify the element in an absolute manner. In elements using this attribute definition the eId attribute is optional. The wId is used to mark the identifier that the structure used to have in the original version, and is only needed when a renumbering occurred.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="eId" type="noWhiteSpace"/>
		    <xsd:attribute name="wId" type="noWhiteSpace"/>
		    <xsd:attribute name="GUID" type="noWhiteSpace"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="refersreq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>refersreq</name>
            <comment>
This attribute creates a connection between the element and an element of the Akoma Ntoso ontology to which it refers. In elements using this attribute definition the refersTo attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="refersTo" use="required">
			      <xsd:simpleType>
				        <xsd:list>
					          <xsd:simpleType>
						            <xsd:restriction base="referenceRef"/>
					          </xsd:simpleType>
				        </xsd:list>
			      </xsd:simpleType>
		    </xsd:attribute>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="refers">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>refers</name>
            <comment>
This attribute creates a connection between the element and an element of the Akoma Ntoso ontology to which it refers. In elements using this attribute definition the refersTo attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute name="refersTo">
			      <xsd:simpleType>
				        <xsd:list>
					          <xsd:simpleType>
						            <xsd:restriction base="referenceRef"/>
					          </xsd:simpleType>
				        </xsd:list>
			      </xsd:simpleType>
		    </xsd:attribute>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="xmllang">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>xmllang</name>
            <comment>
These attribute specify the human language in which the content of the element is expressed as well as the rules for whitespace management in the element. Values for xml:lang are taken from the RFC 4646. Both xml:lang and xml:space are reserved attributes of the XML language, and cannot be used for any other purpose than these ones.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attribute ref="xml:lang"/>
		    <xsd:attribute ref="xml:space"/>
		    <!--<xsd:attribute ref="xml:id"/>-->
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="corereq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>corereq</name>
            <comment>
This is the list of the core attributes that all elements in the content part of the document must have. In elements using this attribute definition the refersTo attribute is optional but the eId attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="HTMLattrs"/>
		    <xsd:attributeGroup ref="enactment"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="refers"/>
		    <xsd:attributeGroup ref="xmllang"/>
		    <xsd:attributeGroup ref="alt"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="corereqreq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>corereqreq</name>
            <comment>
This is the list of the core attributes that all elements in the content part of the document must have. In elements using this attribute definition both the refersTo attribute and the eId attribute are required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="HTMLattrs"/>
		    <xsd:attributeGroup ref="enactment"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="refersreq"/>
		    <xsd:attributeGroup ref="xmllang"/>
		    <xsd:attributeGroup ref="alt"/>
	  </xsd:attributeGroup>

	  <xsd:attributeGroup name="coreopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Attlist</type>
            <name>coreopt</name>
            <comment>
This is the list of the core attributes that all elements in the content part of the document must have. In elements using this attribute definition both the refersTo attribute and the eId attribute are optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="HTMLattrs"/>
		    <xsd:attributeGroup ref="enactment"/>
		    <xsd:attributeGroup ref="idopt"/>
		    <xsd:attributeGroup ref="refers"/>
		    <xsd:attributeGroup ref="xmllang"/>
		    <xsd:attributeGroup ref="alt"/>
	  </xsd:attributeGroup>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Simple types (data values)
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:simpleType name="noWhiteSpace">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>noWhiteSpace</name>
            <comment>
These values are those of ids such as eId, wId and GUID as a collection of any printable character except whitespaces.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:pattern value="[^\s]+"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:simpleType name="eIdRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>eIdRef</name>
            <comment>
These values are references to existing eIds only, i.e., given an existing eId of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="wIdRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>wIdRef</name>
            <comment>
These values are references to existing wIds only, i.e., given an existing wId of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="roleRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>roleRef</name>
            <comment>
These values are references to existing roles (i.e., TLCRole) only, i.e., given an existing eId of a role of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="agentRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>agentRef</name>
            <comment>
These values are references to existing agents (i.e., TLCPerson or TLCOrganization) only, i.e., given an existing eId of an agent of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="conceptRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>conceptRef</name>
            <comment>
These values are references to existing concepts (i.e., TLCConcept) only, i.e., given an existing eId of a concept of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="referenceRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>referenceRef</name>
            <comment>
These values are references to existing References elements only, i.e., given an existing eId of an element of the References section of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="voteRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>voteRef</name>
            <comment>
These values are references to existing votes (i.e., quorum or count) only, i.e., given an existing eId of a vote element of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="eventRefRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>eventRefRef</name>
            <comment>
These values are references to existing event references (i.e., eventRef) only, i.e., given an existing eId of an eventRef of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="temporalGroupRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>temporalGroupRef</name>
            <comment>
These values are references to existing temporal groups only, i.e., given an existing eId of a temporal group of the form XYZ, the value is necessarily #XYZ</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="manifestationURI">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>manifestationURI</name>
            <comment>
These values are references of a manifestation-Level document only</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:anyURI"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="language">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>language</name>
            <comment>
This type specifies the human language in which the document the element refers to is expressed. Values are taken from the RFC 4646.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string"/>
	  </xsd:simpleType>

	  <xsd:simpleType name="versionType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>versionType</name>
            <comment>
This is the list of allowed values for the contains attribute</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="originalVersion"/>
			      <xsd:enumeration value="singleVersion"/>
			      <xsd:enumeration value="multipleVersions"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:simpleType name="placementType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>placementType</name>
            <comment>
This is the list of allowed values for the placement attribute of notes</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="side"/>
			      <xsd:enumeration value="left"/>
			      <xsd:enumeration value="right"/>
			      <xsd:enumeration value="bottom"/>
			      <xsd:enumeration value="inline"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:simpleType name="statusType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>statusType</name>
            <comment>
This is the list of allowed values for the status attribute. This is the list of possible reasons for a dscrepancy between the manifestation as it should be (e.g., a faithful representation of the content of an expression), and the manifestation as it actually is. Values should be interpreted as follows:
- removed: the content of the element is present in the markup (manifestation) but is not present in the real content of the document (expression level) because it has been definitely removed (either ex tunc, as in annullments, or ex nunc, as in abrogations).
- temporarily removed: the content of the element is present in the markup (manifestation) but is not present in the real content of the document (expression level) because it has been temporarily removed (e.g., for a temporary suspension or limitation of efficacy).
- translated: the content of the element is present in the markup (manifestation) in a different form than in the real content of the document (expression level) because it has been translated into a different language (e.g., to match the rest of the document or because of other editorial decisions).
- editorial: the content of the element is present in the markup (manifestation) but is not present in the real content of the document (expression level) because it has been inserted as an editorial process when creating the XML markup.
- edited: the content of the element is different in the markup (manifestation) than in the real content of the document (expression level) because it has been amended (e.g., to remove scurrilous or offensive remarks).
- verbatim: the content of the element is present in the markup (manifestation) is EXACTLY as it was in the real content of the document (expression level) because usual silent fixes and edits were NOT performed (e.g. to punctuation, grammatical errors or other usually non-debatable problems).
- incomplete: the content of the element or the value of a required attribute is NOT present in the markup (manifestation), although it should, because the missing data is not known at the moment, but in the future it might become known. This is especially appropriate for documents in drafting phase (e.g., the publication date of the act while drafting the bill)
- unknown: the content of the element or the value of a required attribute is NOT present in the markup (manifestation), although it should, because the author of the manifestation does not know it.
- undefined: the content of the element or the value of a required attribute is NOT present in the markup (manifestation), because the information is not defined in the original document, or it doesn't exist in some legal tradition (e.g. an anonymous speech cannot specify the attribute by, or some publications do not record the numbering of the items, etc.)
- ignored: the content of the element or the value of a required attribute is NOT present in the markup (manifestation) because the information exists but the author of the manifestation is not interested in reporting it (e.g., omitted parts of the document due to editorial reasons, etc.)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="removed"/>
			      <xsd:enumeration value="temporarilyRemoved"/>
			      <xsd:enumeration value="translated"/>
			      <xsd:enumeration value="editorial"/>
			      <xsd:enumeration value="edited"/>
			      <xsd:enumeration value="verbatim"/>
			      <xsd:enumeration value="incomplete"/>
			      <xsd:enumeration value="unknown"/>
			      <xsd:enumeration value="undefined"/>
			      <xsd:enumeration value="ignored"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:simpleType name="posType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Simple</type>
            <name>posType</name>
            <comment>
This is the list of possible positions of the text being analyzed by the element in the analysis section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:restriction base="xsd:string">
			      <xsd:enumeration value="start"/>
			      <xsd:enumeration value="before"/>
			      <xsd:enumeration value="inside"/>
			      <xsd:enumeration value="after"/>
			      <xsd:enumeration value="end"/>
			      <xsd:enumeration value="unspecified"/>
		    </xsd:restriction>
	  </xsd:simpleType>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Complex types (markup structures)
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:complexType name="basehierarchy">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>basehierarchy</name>
            <comment>
The complex type basehierarchy is not used by any element, but is derived by other types to contain the basic structure of hierarchical elements</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="0" maxOccurs="unbounded">
			      <xsd:element ref="num" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="heading" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="subheading" minOccurs="0" maxOccurs="1"/>
		    </xsd:choice>
	  </xsd:complexType>

	  <xsd:complexType name="hierarchy">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>hierarchy</name>
            <comment>
The complex type hierarchy is used by most or all the hierarchical elements of act-like documents.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:choice>
					          <xsd:sequence>
						            <xsd:element ref="intro" minOccurs="0" maxOccurs="1"/>
						            <xsd:choice minOccurs="0" maxOccurs="unbounded">
							              <xsd:element ref="componentRef"/>
							              <xsd:group ref="hierElements"/>
							              <xsd:element ref="crossHeading"/>
						            </xsd:choice>
						            <xsd:element ref="wrapUp" minOccurs="0" maxOccurs="1"/>
					          </xsd:sequence>
					          <xsd:element ref="content"/>
				        </xsd:choice>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="althierarchy">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>althierarchy</name>
            <comment>
The complex type althierarchy is used by most or all the hierarchical elements of documents that are not act-like.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:choice minOccurs="0" maxOccurs="unbounded">
					          <xsd:group ref="containerElements"/>
					          <xsd:element ref="componentRef"/>
					          <xsd:group ref="ANcontainers"/>
					          <xsd:group ref="blockElements"/>
				        </xsd:choice>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="blocksreq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>blocksreq</name>
            <comment>
the complex type blocksreq defines the content model and attributes shared by all containers. Here the eId attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="corereq"/>
	  </xsd:complexType>

	  <xsd:complexType name="blocksopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>blocksopt</name>
            <comment>
the complex type blocksopt defines the content model and attributes shared by all containers. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="inline" mixed="true">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>inline</name>
            <comment>
the complex type inline defines the content model and attributes shared by all blocks and inlines. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="inlineCM"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="inlinereq" mixed="true">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>inlinereq</name>
            <comment>
the complex type inlinereq defines the content model and attributes shared by all blocks and inlines. Here the eId attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="inlineCM"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="corereq"/>
	  </xsd:complexType>

	  <xsd:complexType name="inlinereqreq" mixed="true">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>inlinereqreq</name>
            <comment>
the complex type inlinereq defines the content model and attributes shared by all blocks and inlines. Here the eId attribute is required and also the refersTo is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="inlineCM"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="corereqreq"/>
	  </xsd:complexType>

	  <xsd:complexType name="markerreq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>markerreq</name>
            <comment>
the complex type markerreq defines the content model and attributes shared by all marker elements. Here the eId attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="corereq"/>
	  </xsd:complexType>

	  <xsd:complexType name="markeropt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>markeropt</name>
            <comment>
the complex type markeropt defines the content model and attributes shared by all marker elements. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="metareq">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>metareq</name>
            <comment>
the complex type metareq defines the content model and attributes shared by all metadata elements. Here the eId attribute is required</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
	  </xsd:complexType>

	  <xsd:complexType name="metaopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>metaopt</name>
            <comment>
the complex type metaopt defines the content model and attributes shared by all metadata elements. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="maincontent">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>maincontent</name>
            <comment>
the complex type maincontent is used by container elements that can contain basically any other Akoma Ntoso structure</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:element ref="componentRef"/>
			      <xsd:group ref="hierElements"/>
			      <xsd:group ref="blockElements"/>
			      <xsd:group ref="containerElements"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="speechType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>speechType</name>
            <comment>
the complex type speechType defines the content model and attributes shared by all speech elements. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:sequence>
					          <xsd:element ref="from" minOccurs="0" maxOccurs="1"/>
					          <xsd:sequence minOccurs="1" maxOccurs="unbounded">
						            <xsd:group ref="blockElements"/>
					          </xsd:sequence>
				        </xsd:sequence>
				        <xsd:attributeGroup ref="coreopt"/>
				        <xsd:attributeGroup ref="speechAtts"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="itemType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>itemType</name>
            <comment>
The complex type itemType is similar to a hierarchical element, but is isolated and does not belong to a full hierarchy.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:sequence minOccurs="1" maxOccurs="unbounded">
					          <xsd:group ref="blockElements"/>
				        </xsd:sequence>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="referenceType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>referenceType</name>
            <comment>
the complex type referenceType defines the empty content model and the list of attributes for metadata elements in the references section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="refers"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="link"/>
		    <xsd:attributeGroup ref="show"/>
	  </xsd:complexType>

	  <xsd:complexType name="srcType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>srcType</name>
            <comment>
the complex type srcType defines the empty content model and the list of attributes for manifestation-level references to external resources</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idopt"/>
		    <xsd:attributeGroup ref="src"/>
		    <xsd:attributeGroup ref="show"/>
	  </xsd:complexType>

	  <xsd:complexType name="linkType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>linkType</name>
            <comment>
the complex type linkType defines the empty content model and the list of attributes for Work- or Expression-level references to external resources</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="link"/>
		    <xsd:attributeGroup ref="show"/>
	  </xsd:complexType>

	  <xsd:complexType name="anyOtherType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>anyOtherType</name>
            <comment>
the complex type anyOtherType defines an open content model for elements that may use elements from other namespaces.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:any namespace="##other"
                  processContents="lax"
                  minOccurs="0"
                  maxOccurs="unbounded"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idopt"/>
		    <xsd:attributeGroup ref="linkopt"/>
	  </xsd:complexType>

	  <xsd:complexType name="docContainerType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>docContainerType</name>
            <comment>
the complex type docContainerType defines a shared content model for elements that contain whole documents, namely attachment, collectionItem, component.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:choice>
					          <xsd:group ref="documentType"/>
					          <xsd:element ref="interstitial"/>
					          <xsd:element ref="toc"/>
					          <xsd:element ref="documentRef"/>
				        </xsd:choice>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Root element
		Comment: One generic root element contains all document elements
		Pattern: Universal Root
		Reference: http://www.xmlpatterns.com/UniversalRootMain.shtml
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:group name="documentType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>documentType</name>
            <comment>
the type documentType lists all the document types that are managed by Akoma Ntoso</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="amendmentList"/>
			      <xsd:element ref="officialGazette"/>
			      <xsd:element ref="documentCollection"/>
			      <xsd:element ref="act"/>
			      <xsd:element ref="bill"/>
			      <xsd:element ref="debateReport"/>
			      <xsd:element ref="debate"/>
			      <xsd:element ref="statement"/>
			      <xsd:element ref="amendment"/>
			      <xsd:element ref="judgment"/>
			      <xsd:element ref="portion"/>
			      <xsd:element ref="doc"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Document elements
		Comment: All document elements share just two content models
		Pattern: Consistent Element Set
		Reference: http://www.xmlpatterns.com/ConsistentElementSetMain.shtml
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:complexType name="akomaNtosoType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>akomaNtosoType</name>
            <comment>
the complex type akomaNtosoType is the type for the root element in Akoma Ntoso.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:sequence>
				        <xsd:group ref="documentType"/>
			      </xsd:sequence>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
	  </xsd:complexType>

	  <xsd:element name="akomaNtoso" type="akomaNtosoType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element (root)</type>
            <name>akomaNtoso</name>
            <comment>
NAME
akomaNtoso
the element akomaNtoso is the root element of all document types in Akoma Ntoso. It follows the pattern Universal Root (http://www.xmlpatterns.com/UniversalRootMain.shtml)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="openStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>openStructure</name>
            <comment>
the type openStructure specifies the overall content model of all the document types that do not have a specific and peculiar structure</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preface" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preamble" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="mainBody"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="doc" type="openStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>doc</name>
            <comment>
Element doc is used for describing the structure and content of any other document that is not included in the list of document explicitly managed by Akoma Ntoso</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-doc">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-doc">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="mainBody" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>mainBody</name>
            <comment>
the element mainBody is the container of the main part of all other document types</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="statement" type="openStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>statement</name>
            <comment>
Element statement is used for describing the structure and content of a official documents of a legislative assembly that are not normative in structure (e.g., statements, non-normative resolutions, etc.).</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-statement">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-statement">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="components">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="component"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="idreq"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="component" type="docContainerType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>component</name>
            <comment>
The element component is a container of a subdocument specified in a composite document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="collectionStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>collectionStructure</name>
            <comment>
the type collectionStructure specifies the overall content model of the document types that are collections of other documents</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preface" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preamble" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="collectionBody"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="amendmentList" type="collectionStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentList</name>
            <comment>
Element amendmentList is used for describing the structure and content of a collection of amendments</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-amendmentList">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-amendmentList">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="officialGazette" type="collectionStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>officialGazette</name>
            <comment>
Element officialGazette is used for describing the structure and content of an issue of an official gazette</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-officialGazette">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-officialGazette">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="documentCollection" type="collectionStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>documentCollection</name>
            <comment>
Element documentCollection is used for describing the structure and content of a collection of other documents chosen and combined for any reason whatsoever</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-documentCollection">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-documentCollection">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="collectionBodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>collectionBodyType</name>
            <comment>
the type collectionBodyType specifies a content model of a container of a list of other documents (e.g, acts, bills, amendments, etc.) possibly  interspersed with interstitial elements with content that does not form an individual document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="component" minOccurs="1" maxOccurs="unbounded"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="collectionBody" type="collectionBodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>collectionBody</name>
            <comment>
the element collectionBody is the container of a list of other documents (e.g, acts, bills, amendments, etc.) possibly  interspersed with interstitial elements with content that does not form an individual document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="portionStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>portionStructure</name>
            <comment>
the type portionStructure specifies the overall content model of the document type that is a portion of another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="portionBody"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="portionAtt"/>
	  </xsd:complexType>

	  <xsd:element name="portion" type="portionStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>portion</name>
            <comment>
Element portion is used for describing the structure and content of an independent portion of a document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-portion">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-portion">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="portionBodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>portionBodyType</name>
            <comment>
the type portionBodyType specifies a content model of a container of a portion of another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="containerElements"/>
			      <xsd:group ref="hierElements"/>
			      <xsd:group ref="ANcontainers"/>
			      <xsd:element ref="recitals"/>
			      <xsd:element ref="recital"/>
			      <xsd:element ref="citations"/>
			      <xsd:element ref="citation"/>
			      <xsd:element ref="longTitle"/>
			      <xsd:element ref="formula"/>
			      <xsd:element ref="coverPage"/>
			      <xsd:element ref="preface"/>
			      <xsd:element ref="preamble"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="portionBody" type="portionBodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>portionBody</name>
            <comment>
the element portionBody is the container of a portion of another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="hierarchicalStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>hierarchicalStructure</name>
            <comment>
the type hierarchicalStructure specifies the overall content model of the document types that are hierarchical in nature, especially acts and bills</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preface" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preamble" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="body"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="act" type="hierarchicalStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>act</name>
            <comment>
Element act is used for describing the structure and content of an act</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-act">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-act">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="bill" type="hierarchicalStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>bill</name>
            <comment>
Element bill is used for describing the structure and content of a bill</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-bill">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-bill">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="bodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>bodyType</name>
            <comment>
the type bodyType specifies a content model of the main hierarchy of a hierarchical document (e.g, an act or a bill)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:element ref="componentRef"/>
			      <xsd:group ref="hierElements"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="body" type="bodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>body</name>
            <comment>
the element body is the container of the main hierarchy of a hierarchical document (e.g, an act or a bill)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="debateStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>debateStructure</name>
            <comment>
the type debateStructure specifies the overall content model of the document types that describe debates</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preface" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="debateBody"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="debateReport" type="openStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>debateReport</name>
            <comment>
Element debateReport is used for describing the structure and content of a report</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-debateReport">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-debateReport">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:element name="debate" type="debateStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>debate</name>
            <comment>
Element debate is used for describing the structure and content of a debate record</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-debate">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-debate">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="debateBodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>debateBodyType</name>
            <comment>
the type debateBodyType specifies a content model of the main hierarchy of a debate</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="speechSection"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="debateBody" type="debateBodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>debateBody</name>
            <comment>
the element debateBody is the container of the main hierarchy of a debate</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="judgmentStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>judgmentStructure</name>
            <comment>
the type judgmentStructure specifies the overall content model of the document types that describe judgments</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="header"/>
			      <xsd:element ref="judgmentBody"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="judgment" type="judgmentStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>judgment</name>
            <comment>
Element judgment is used for describing the structure and content of a judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-judgment">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-judgment">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="judgmentBodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>judgmentBodyType</name>
            <comment>
the type judgmentBodyType specifies a content model of the main hierarchy of a judgment document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="judgmentBlock"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="judgmentBody" type="judgmentBodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>judgmentBody</name>
            <comment>
the element judgmentBody is the container of the main hierarchy of a judgment document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="amendmentStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>amendmentStructure</name>
            <comment>
the type amendmentStructure specifies the overall content model of the document types that describe amendments</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="meta"/>
			      <xsd:element ref="coverPage" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preface" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="amendmentBody"/>
			      <xsd:element ref="conclusions" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="attachments" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="components" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="name"/>
		    <xsd:attributeGroup ref="contains"/>
	  </xsd:complexType>

	  <xsd:element name="amendment" type="amendmentStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendment</name>
            <comment>
Element amendment is used for describing the structure and content of an amendment</comment>
				     </xsd:documentation>
		    </xsd:annotation>
      <xsd:unique name="eId-amendment">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@eId"/>
      </xsd:unique>
      <xsd:unique name="GUID-amendment">
         <xsd:selector xpath=".//*"/>
         <xsd:field xpath="@GUID"/>
      </xsd:unique>
   </xsd:element>

	  <xsd:complexType name="amendmentBodyType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>amendmentBodyType</name>
            <comment>
the type amendmentBodyType specifies a content model of the main hierarchy of a amendment document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="amendmentBlock"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="amendmentBody" type="amendmentBodyType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentBody</name>
            <comment>
the element amendmentBody is the container of the main hierarchy of a amendment document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="recitalHierarchy">
		

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:sequence>
					          <xsd:element ref="intro" minOccurs="0" maxOccurs="1"/>
					          <xsd:choice minOccurs="1" maxOccurs="unbounded">
						            <xsd:element ref="componentRef"/>
						            <xsd:element ref="recital"/>
					          </xsd:choice>
					          <xsd:element ref="wrapUp" minOccurs="0" maxOccurs="1"/>
				        </xsd:sequence>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="recitals" type="recitalHierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>recitals</name>
            <comment>
the element recitals is the section of the preamble that contains recitals</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="recital" type="itemType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>recital</name>
            <comment>
the element recital is the individual element of the preface that is called recital</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="citationHierarchy">
		

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:sequence>
					          <xsd:element ref="intro" minOccurs="0" maxOccurs="1"/>
					          <xsd:choice minOccurs="1" maxOccurs="unbounded">
						            <xsd:element ref="componentRef"/>
						            <xsd:element ref="citation"/>
					          </xsd:choice>
					          <xsd:element ref="wrapUp" minOccurs="0" maxOccurs="1"/>
				        </xsd:sequence>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="citations" type="citationHierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>citations</name>
            <comment>
the element citations is the section of the preamble that contains citations</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="citation" type="itemType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>citation</name>
            <comment>
the element citation is the individual element of the preface that is called citation</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="longTitle" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>longTitle</name>
            <comment>
the element longTitle is the section of the preface or preamble that is called long title</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="formula">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>formula</name>
            <comment>
the element formula is a section of the preface or preamble that contains a formulaic expression that is systematically or frequently present in a preface or a preamble and has e precise legal meaning (e.g. an enacting formula). Use the refersTo attribute for the specification of the actual type of formula.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="blocksreq">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="basicopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>basicopt</name>
            <comment>
the complex type basicopt defines the content model and attributes used by basic containers such as coverPage and conclusions. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
			      <xsd:group ref="basicContainers"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="coverPage" type="basicopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>coverPage</name>
            <comment>
the element coverPage is used as a container of the text that acts as a cover page</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="preambleopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>preambleopt</name>
            <comment>
the complex type preambleopt defines the content model and attributes used by preambles. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
			      <xsd:group ref="preambleContainers"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="preamble" type="preambleopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>preamble</name>
            <comment>
the element preamble is used as a container of the text that opens the main body of the document as a preamble</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="prefaceopt">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>prefaceopt</name>
            <comment>
the complex type prefaceopt defines the content model and attributes used by preface. Here the eId attribute is optional</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
			      <xsd:group ref="prefaceContainers"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="coreopt"/>
	  </xsd:complexType>

	  <xsd:element name="preface" type="prefaceopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>preface</name>
            <comment>
the element preface is used as a container of all prefacing material (e.g. headers, formulas, etc.)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="conclusions" type="basicopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>conclusions</name>
            <comment>
the element conclusion is used as a container of all concluding material (e.g. dates, signatures, formulas, etc.)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="header" type="blocksopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>header</name>
            <comment>
the element header is used as a container of all prefacing material of judgments (e.g. headers, formulas, etc.)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="attachments">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="attachment"/>
			      </xsd:sequence>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="attachment" type="docContainerType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>attachment</name>
            <comment>
the element attachment is used as a container of individual attachment elements</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="interstitial" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>interstitial</name>
            <comment>
the element interstitial is used as a container of text elements and blocks that are placed for any reason between individual documents in a collection of documents</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="componentRef" type="srcType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>componentRef</name>
            <comment>
the element componentRef is a reference to a separate manifestation-level resource that holds the content of the component of the document not physically placed at the position specified. Actual resources can either be external (e.g. in the package or even in a different position) or internal (within the components element)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="documentRef" type="linkType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>documentRef</name>
            <comment>
the element documentRef is a reference to a separate work- or expression-level resource that should be placed in this position. Actual resources are external (e.g. in the package or even in a different position) and are (an expression or any expression of) a separate Work.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Elements for hierarchical documents
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:element name="clause" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>clause</name>
            <comment>
this element is a hierarchical container called "clause" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="section" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>section</name>
            <comment>
this element is a hierarchical container called "section" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="part" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>part</name>
            <comment>
this element is a hierarchical container called "part" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="paragraph" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>paragraph</name>
            <comment>
this element is a hierarchical container called "paragraph" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="chapter" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>chapter</name>
            <comment>
this element is a hierarchical container called "chapter" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="title" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>title</name>
            <comment>
this element is a hierarchical container called "title" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="book" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>book</name>
            <comment>
this element is a hierarchical container called "book" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="tome" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>tome</name>
            <comment>
this element is a hierarchical container called "tome" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="article" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>article</name>
            <comment>
this element is a hierarchical container called "article" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="division" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>division</name>
            <comment>
this element is a hierarchical container called "division" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="list" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>list</name>
            <comment>
this element is a hierarchical container called "list" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="point" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>point</name>
            <comment>
this element is a hierarchical container called "point" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="indent" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>indent</name>
            <comment>
this element is a hierarchical container called "indent" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="alinea" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>alinea</name>
            <comment>
this element is a hierarchical container called "alinea" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="rule" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>rule</name>
            <comment>
this element is a hierarchical container called "rule" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subrule" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subrule</name>
            <comment>
this element is a hierarchical container called "subrule" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="proviso" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>proviso</name>
            <comment>
this element is a hierarchical container called "proviso" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subsection" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subsection</name>
            <comment>
this element is a hierarchical container called "subsection" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subpart" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subpart</name>
            <comment>
this element is a hierarchical container called "subpart" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subparagraph" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subparagraph</name>
            <comment>
this element is a hierarchical container called "subparagraph" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subchapter" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subchapter</name>
            <comment>
this element is a hierarchical container called "subchapter" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subtitle" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subtitle</name>
            <comment>
this element is a hierarchical container called "subtitle" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subdivision" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subdivision</name>
            <comment>
this element is a hierarchical container called "subdivision" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subclause" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subclause</name>
            <comment>
this element is a hierarchical container called "subclause" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="sublist" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>sublist</name>
            <comment>
this element is a hierarchical container called "sublist" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="level" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>level</name>
            <comment>
this element is a hierarchical container called "level" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="transitional" type="hierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>transitional</name>
            <comment>
this element is a hierarchical container called "transitional" either explicitly or due to the local tradition</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="content" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>content</name>
            <comment>
the element content is the final container in a hierarchy, and is where the blocks of text of the content of the structure are finally specified</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="num" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>num</name>
            <comment>
the element num is a heading element in a hierarchy that contains a number or any other ordered mechanism to identify the structure.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="heading" type="inlinereq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>heading</name>
            <comment>
the element heading is a heading element in a hierarchy that contains a title or any other textual content to describe the structure.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="subheading" type="inlinereq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subheading</name>
            <comment>
the element subheading is a heading element in a hierarchy that contains a subtitle or any other textual content to further describe the structure.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="crossHeading" type="inlinereq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>crossHeading</name>
            <comment>
the element crossHeading is a heading element that is placed side by side with hierarchical containers .</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="intro" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>intro</name>
            <comment>
the element intro is a heading element in a hierarchy that contains paragraphs introducing one or more lower hierarchical elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="wrapUp" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>wrapUp</name>
            <comment>
the element wrapUp is a concluding element in a hierarchy that contains paragraphs wrapping up the preceding lower hierarchical elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Elements for debates documents
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:element name="administrationOfOath" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>administrationOfOath</name>
            <comment>
this element is a structural container for parts of a debates that contain the administration of an oath</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="rollCall" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>rollCall</name>
            <comment>
this element is a structural container for parts of a debates that contain a roll call of individuals</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="prayers" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>prayers</name>
            <comment>
this element is a structural container for parts of a debates that contain prayers</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="oralStatements" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>oralStatements</name>
            <comment>
this element is a structural container for parts of a debates that contain oral statements by participants</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="writtenStatements" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>writtenStatements</name>
            <comment>
this element is a structural container for parts of a debates that contain written statements by participants</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="personalStatements" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>personalStatements</name>
            <comment>
this element is a structural container for parts of a debates that contain written statements by participants</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="ministerialStatements" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>ministerialStatements</name>
            <comment>
this element is a structural container for parts of a debates that contain written statements by participants</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="resolutions" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>resolutions</name>
            <comment>
this element is a structural container for parts of a debates that contain resolutions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="nationalInterest" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>nationalInterest</name>
            <comment>
this element is a structural container for parts of a debates that contain resolutions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="declarationOfVote" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>declarationOfVote</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to the declaration of votes</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="communication" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>communication</name>
            <comment>
this element is a structural container for parts of a debates that contain communications from the house</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="petitions" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>petitions</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to petitions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="papers" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>papers</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to the display of papers</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="noticesOfMotion" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>noticesOfMotion</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to the notices of motions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="questions" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>questions</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to questions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="address" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>address</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to addresses</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="proceduralMotions" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>proceduralMotions</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to procedural motions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="pointOfOrder" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>pointOfOrder</name>
            <comment>
this element is a structural container for parts of a debates that are relevant to points of order</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="adjournment" type="althierarchy">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>adjournment</name>
            <comment>
this element is a structural container for parts of a debates that contain adjournment notices</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="debateSection">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>debateSection</name>
            <comment>
this element is a generic structural container for all other parts of a debates that are not explicitly supported with a named element</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="althierarchy">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="speechGroup">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>speechGroup</name>
            <comment>
the element speechGroup is a container of speech elements. This element is meant to pooint out, in a complex sequence of individual speech elements, the main contributor, i.e., the individual speech who was introducedand expected and that is causing the complex sequence that follows. Attributes by, as and to are those of the main speech.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="althierarchy">
					          <xsd:attributeGroup ref="speechAtts"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="speech" type="speechType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>speech</name>
            <comment>
the element speech is a container of a single speech utterance in a debate. Dialogs between speakers need a speech element each</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="question" type="speechType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>question</name>
            <comment>
the element question is a container of a single official question as proposed by an MP to a person holding an official position</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="answer" type="speechType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>answer</name>
            <comment>
the element answer is a container of a single official answer to a question</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="other" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>other</name>
            <comment>
the element other is a container of parts of a debate that are not speeches, nor scene comments (e.g., lists of papers, etc.)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="scene" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>scene</name>
            <comment>
the element scene is a container of descriptions of the scene elements happening in a given moment during a debate (e.g., applauses)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="narrative" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>narrative</name>
            <comment>
the element narrative is a block element in a debate to mark description in the third person of events taking place in the meeting, e.g. "Mr X. takes the Chair"</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="summary" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>summary</name>
            <comment>
the element summary is a block element in a debate to mark summaries of speeches that are individually not interesting (e.g.: "Question put and agreed to")</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="from" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>from</name>
            <comment>
the element from is a heading element in a debate that contains the name or role or a reference to the person doing the speech</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="vote">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>vote</name>
            <comment>
the element vote is an inline that wraps either the name of the voter (when organized by choice) or the vote (when organized by name) in a voting report.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="voteAtts"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="outcome" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>outcome</name>
            <comment>
the element outcome is an inline that wraps the outcome of a vote</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="introduction" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>introduction</name>
            <comment>
this element is a structural container for the section of a judgment containing introductory material</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="background" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>background</name>
            <comment>
this element is a structural container for the section of a judgment containing the background</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="arguments" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>arguments</name>
            <comment>
this element is a structural container for the section of a judgment containing the arguments</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="remedies" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>remedies</name>
            <comment>
this element is a structural container for the section of a judgment containing the remedies</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="motivation" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>motivation</name>
            <comment>
this element is a structural container for the section of a judgment containing the motivation</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="decision" type="maincontent">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>decision</name>
            <comment>
this element is a structural container for the section of a judgment containing the decision</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="affectedDocument">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>affectedDocument</name>
            <comment>
the element affectedDocument is an inline element within preamble to identify the document that this amendment affects</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="link"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="relatedDocument">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>relatedDocument</name>
            <comment>
the element relatedDocument is an inline element to identify the document for which this document is a report of</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="link"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="change" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>change</name>
            <comment>
the element change is an inline element that identifies the changes expressed in the two columns of an amendment document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="amendmentHeading" type="blocksopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentHeading</name>
            <comment>
this element is a structural container for the section of an amendment containing the heading</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="amendmentContent" type="blocksopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentContent</name>
            <comment>
this element is a structural container for the section of an amendment containing the actual amendment text</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="amendmentReference" type="blocksopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentReference</name>
            <comment>
this element is a structural container for the section of an amendment containing the reference</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="amendmentJustification" type="blocksopt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>amendmentJustification</name>
            <comment>
this element is a structural container for the section of an amendment containing the justification</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		AN block elements
		Comment: the only AN-specific block are the table of content (TOC),
		a structure containing toc items related to the various subparts of the
		main document, the list item and the titled block
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:complexType name="blockContainerType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>blockContainerType</name>
            <comment>
the element blockContainer is used as a container of many individual block elements in a block context</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="basehierarchy">
				        <xsd:sequence>
					          <xsd:element ref="intro" minOccurs="0" maxOccurs="1"/>
					          <xsd:choice minOccurs="1" maxOccurs="unbounded">
						            <xsd:element ref="componentRef"/>
						            <xsd:group ref="blockElements"/>
					          </xsd:choice>
					          <xsd:element ref="wrapUp" minOccurs="0" maxOccurs="1"/>
				        </xsd:sequence>
				        <xsd:attributeGroup ref="corereq"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="blockContainer" type="blockContainerType"/>
		


	  <xsd:element name="blockList">
      <xsd:annotation>
		       <xsd:documentation>
		 The element item is a container belonging to a list 
		</xsd:documentation>
	     </xsd:annotation>
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="listIntroduction" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="item" minOccurs="1" maxOccurs="unbounded"/>
				        <xsd:element ref="listWrapUp" minOccurs="0" maxOccurs="1"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="corereq"/>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="item" type="itemType"/>
		


	  <xsd:element name="listIntroduction" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>listIntroduction</name>
            <comment>
the element listIntroduction is an optional element of list before any item of the list itself.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="listWrapUp" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>listWrapUp</name>
            <comment>
the element listWrapUp is an optional element of list after all items of the list itself.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="tblock" type="itemType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>tblock</name>
            <comment>
The element tblock (titled block) is used to specify a container for blocks introduced by heading elements, similarly to a hierarchical structure</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="toc">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="tocItem"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="corereq"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="tocItem">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>tocItem</name>
            <comment>
the element tocItem is a component of the table of content and contains header information about sections or parts of the rest of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="link"/>
					          <xsd:attributeGroup ref="level"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="docType" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docType</name>
            <comment>
the element docType is an inline element within preface to identify the string used by the document for its own type</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docTitle" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docTitle</name>
            <comment>
the element docTitle is an inline element within preface to identify the string used by the document for its own title</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docNumber" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docNumber</name>
            <comment>
the element docNumber is an inline element within preface to identify the string used by the document for its own number</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docProponent">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docProponent</name>
            <comment>
the element docProponent is an inline element within preface to identify the string used by the document for its proponent</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="role"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="docDate">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docDate</name>
            <comment>
the element docDate is an inline element within preface to identify the string used by the document for its own date(s). Documents with multiple dates may use multiple docDate elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="date"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="legislature">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>legislature</name>
            <comment>
the element legislature is an inline element within preface to identify the string used by the document for the legislature relative to the document. Use #refersTo to a TLCEvent to refer to the event of the specific legislature.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="optvalue"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="session">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>session</name>
            <comment>
the element session is an inline element within preface to identify the string used by the document for the session of the legislature relative to the document. Use #refersTo to a TLCEvent to refer to the event of the specific session.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="optvalue"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="shortTitle" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>shortTitle</name>
            <comment>
the element shortTitle is an inline element within preface to identify the string used by the document for the short title of the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docAuthority" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docAuthority</name>
            <comment>
the element docAuthority is an inline element within preface to identify the string used by the document detailing the Auhtority to which the document was submitted</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docPurpose" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docPurpose</name>
            <comment>
the element docPurpose is an inline element within preface to identify the string used by the document detailing its own purpose</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docCommittee">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docCommittee</name>
            <comment>
the element docCommittee is an inline element within preface to identify the string used by the document detailing the committee within which the document originated</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="optvalue"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="docIntroducer" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docIntroducer</name>
            <comment>
the element docIntroducer is an inline element within preface to identify the string used by the document detailing the individual introducing of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docStage" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docStage</name>
            <comment>
the element docStage is an inline element within preface to identify the string used by the document detailing the stage in which the document sits</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docStatus" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docStatus</name>
            <comment>
the element docStatus is an inline element within preface to identify the string used by the document detailing the status of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docJurisdiction" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docJurisdiction</name>
            <comment>
the element docJurisdiction is an inline element within preface to identify the string used by the document detailing the jurisdiction of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="docketNumber" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>docketNumber</name>
            <comment>
the element docketNumber is an inline element within preface to identify the string used by the document for the number of the docket, case, file, etc which the document belongs to</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="courtType" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>courtType</name>
            <comment>
the element courtType is an inline element within judgments to identify the string used by the document for the type of the court doing the judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="neutralCitation" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>neutralCitation</name>
            <comment>
the element neutralCitation is an inline element within judgments to identify the string declared by the document as being the neutral citation for the judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="party">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>party</name>
            <comment>
the element party is an inline element within judgments to identify where the document defines one of the parties</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="role"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="judge">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>judge</name>
            <comment>
the element judge is an inline element within judgments to identify where the document defines one of the judges, and his/her role</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="role"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="lawyer">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>lawyer</name>
            <comment>
the element lawyer is an inline element within judgments to identify where the document defines one of the lawyers, his/her role, which party it represents, and the other lawyer, if any, this lawyer received the power delegation of power in some role</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="lawyerAtts"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="opinion">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>opinion</name>
            <comment>
the element opinion is an inline element within judgments to identify where the document defines the opinion of one of the judges</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="opinionType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="argument" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>argument</name>
            <comment>
the element argument is an inline element within judgments for classifying the arguments in the motivation part of the judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="signature" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>signature</name>
            <comment>
the element signature is an inline element within conclusions to identify where the document defines one of the signatures</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="date">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>date</name>
            <comment>
the element date is an inline element to identify a date expressed in the text and to propose a normalized representation in the date attribute.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="date"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="time">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>time</name>
            <comment>
the element time is an inline element to identify a time expressed in the text and to propose a normalized representation in the time attribute.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="time"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="entity">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>entity</name>
            <comment>
the element entity is a generic inline element to identify a text fragment introducing or referring to a concept in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="person">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>person</name>
            <comment>
the element person is an inline element to identify a text fragment introducing or referring to a person in the ontology. Attribute as allows to specify a TLCrole the person is holding in the context of the document's mention</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="role"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="organization" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>organization</name>
            <comment>
The element organization is an inline element to identify a text fragment introducing or referring to an organization in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="concept" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>concept</name>
            <comment>
The element concept is is an inline element to identify a text fragment introducing or referring to a concept in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="object" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>object</name>
            <comment>
The element object is is an inline element to identify a text fragment introducing or referring to an object in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="event" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>event</name>
            <comment>
The element event is an inline element to identify a text fragment introducing or referring to an event in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="location" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>location</name>
            <comment>
The element location is an inline element to identify a text fragment introducing or referring to a location in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="process" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>process</name>
            <comment>
The element process is an inline element to identify a text fragment introducing or referring to a process in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="role" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>role</name>
            <comment>
The element role is an inline element to identify a text fragment introducing or referring to a role in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="term" type="inlinereqreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>term</name>
            <comment>
The element term is an inline element to identify a text fragment introducing or referring to a term in the ontology</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="quantity">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>quantity</name>
            <comment>
The element quantity is an inline element to identify a text fragment introducing or referring to a quantity. This could be a dimensionless number, or a number referring to a length, weight, duration, etc. or even a sum of money. The attribute normalized contains the value normalized in a number, if appropriate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereqreq">
					          <xsd:attributeGroup ref="normalizedAtt"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="def" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>def</name>
            <comment>
the element def is an inline element used for the definition of a term used in the rest of the document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="ref">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>ref</name>
            <comment>
the element ref is an inline element containing a legal references (i.e. a reference to a document with legal status and for which an Akoma Ntoso IRI exists)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereq">
					          <xsd:attributeGroup ref="link"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="mref" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>mref</name>
            <comment>
the element mref is an inline element containing multiple references (each in turn represented by a ref element)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="rref">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>rref</name>
            <comment>
the element rref is an inline element containing a range of references between the IRI specified in the from attribute and the one specified in the upTo attribute.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inlinereq">
					          <xsd:attributeGroup ref="range"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="modType" mixed="true">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>modType</name>
            <comment>
the complex type modType specifies the content that is allowed within mod, mmod and rmod elements, i.e. it adds quotedText and quotedStructure to the normal list of inline elements. Attribute for is used to point to the eId of the corresponding ref element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="0" maxOccurs="unbounded">
			      <xsd:group ref="inlineCM"/>
			      <xsd:element ref="quotedText"/>
			      <xsd:element ref="quotedStructure"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="corereq"/>
		    <xsd:attributeGroup ref="for"/>
	  </xsd:complexType>

	  <xsd:element name="mod" type="modType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>mod</name>
            <comment>
the element mod is an inline element containing the specification of a modification on another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="mmod" type="modType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>mmod</name>
            <comment>
the element mmod is an inline element containing multiple specifications of modifications on another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="rmod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>rmod</name>
            <comment>
the element rmod is an inline element containing the specification of a range of modifications on another document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="modType">
					          <xsd:attributeGroup ref="range"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="quotedText">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>quotedText</name>
            <comment>
the element quotedText is an inline element containing a small string that is used either as the text being replaced, or the replacement, or the positioning at which some modification should take place. Attribute quote is used to specify the quote character used in the original; no quote attribute implies that the quote is left in the text; quote="" implies that there is no quote character. Attribute for is used to point to the eId of the corresponding ref element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="quote"/>
					          <xsd:attributeGroup ref="for"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="remark">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>remark</name>
            <comment>
the element remark is an inline element for the specification of editorial remarks (e.g., applauses, laughters, etc.) especially within debate records</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="remarkType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="recordedTime">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>recordedTime</name>
            <comment>
the element recordedTime is an inline element for the specification of an explicit mention of a time (e.g., in a debate)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="recordedTimeType"/>
					          <xsd:attributeGroup ref="time"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="ins" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>ins</name>
            <comment>
the element ins is an inline element for the specification of editorial insertions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="del" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>del</name>
            <comment>
the element del is an inline element for the specification of editorial deletions</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="omissis" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>omissis</name>
            <comment>
the element omissis is an inline element for the specification of a text that substitutes a textual omission (e.g., dots, spaces, the word "omissis", etc.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="placeholder">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>placeholder</name>
            <comment>
the element placeholder is an inline element containing the text of a computable expression (e.g., '30 days after the publication of this act') that can be replaced editorially with an actual value</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="originalText"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="fillIn">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>fillIn</name>
            <comment>
the element fillIn is an inline element shown as a dotted line or any other typoaphical characteristics to represent a fill-in element in a printed form, that is as ane example of an actual form. It is NOT meant to be used for form elements as in HTML, i.e. as a way to collect input from the reader and deliver to some server-side process.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="fillInWidth"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="decoration" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>decoration</name>
            <comment>
the element decoration is an inline element to represent a decorative aspect that is present in the orignal text and that is meant as additional information to the text (e.g., the annotation 'new' on the side of a freshly inserted structure.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		AN marker elements
		Comment: AN-specific marker elements are connected to the
		identification of locations within the legal structure, anywhere where
		there is text.
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:element name="noteRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>noteRef</name>
            <comment>
the element noteRef is a reference to a editorial note placed in the notes metadata section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="markeropt">
					          <xsd:attributeGroup ref="notes"/>
					          <xsd:attributeGroup ref="link"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="eolType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>eolType</name>
            <comment>
the complex type eolType is shared by eol and eop elements as being able to specify a hyphen character and a position within the next word in which the break can happen, and the number if any, associated to the page or line at issue</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="markeropt">
				        <xsd:attributeGroup ref="number"/>
				        <xsd:attribute name="breakAt" type="xsd:integer"/>
				        <xsd:attribute name="breakWith" type="xsd:string"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="eol" type="eolType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>eol</name>
            <comment>
the element eol (end of line) is a marker for where in the original text the line breaks. If the line breaks within a word, place the element BEFORE the word and place the number of characters before the break in the attribute breakAt. One can also specify, if relevant, the hyphen or other character used to signal
the break of a word at the end of the line with the attribute breakWith.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="eop" type="eolType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>eop</name>
            <comment>
the element eop (end of page) is a marker for where in the original text the page breaks. Do NOT use a eol element, too. If the page breaks within a word, place the element BEFORE the word and place the number of characters before the break in the attribute breakAt. One can also specify, if relevant, the hyphen or other character used to signal the break of a word at the end of the page with the attribute breakWith.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		AN subFlow elements
		Comment: AN-specific subFlow elements are connected to the
		identification of structural elements that contain full structures
		within a text content, but are not of textual nature.
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:complexType name="subFlowStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>subFlowStructure</name>
            <comment>
the type subFlowStructure specifies the overall content model of the elements that are subFlows</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:group ref="documentType"/>
			      <xsd:choice minOccurs="1" maxOccurs="unbounded">
				        <xsd:group ref="blockElements"/>
				        <xsd:group ref="containerElements"/>
				        <xsd:element ref="tr"/>
				        <xsd:element ref="th"/>
				        <xsd:element ref="td"/>
				        <xsd:group ref="hierElements"/>
				        <xsd:element ref="content"/>
				        <xsd:group ref="ANcontainers"/>
				        <xsd:element ref="formula"/>
				        <xsd:element ref="recitals"/>
				        <xsd:element ref="citations"/>
				        <xsd:element ref="longTitle"/>
				        <xsd:element ref="recital"/>
				        <xsd:element ref="citation"/>
				        <xsd:element ref="componentRef"/>
				        <xsd:element ref="documentRef"/>
				        <xsd:element ref="intro"/>
				        <xsd:element ref="wrapUp"/>
				        <xsd:element ref="heading"/>
				        <xsd:element ref="subheading"/>
				        <xsd:element ref="num"/>
			      </xsd:choice>
		    </xsd:choice>
		    <xsd:attributeGroup ref="corereq"/>
	  </xsd:complexType>

	  <xsd:element name="quotedStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>quotedStructure</name>
            <comment>
the element quotedStructure is a subFlow element containing a full structure proposed as an insertion or a replacement. Attribute quote is used to specify the quote character used in the original; no quote attribute implies that the quote is left in the text; quote="" implies that there is no quote character. Attribute for is used in a mmod or rmod to point to the eId of the corresponding ref element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="subFlowStructure">
					          <xsd:attributeGroup ref="quote"/>
					          <xsd:attributeGroup ref="for"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="embeddedText">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>embeddedText</name>
            <comment>
the element embeddedText is an inline element containing a string used as an extract from another document. Attribute quote is used to specify the quote character used in the original; no quote attribute implies that the quote is left in the text; quote="" implies that there is no quote character.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="quote"/>
					          <xsd:attributeGroup ref="linkopt"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="embeddedStructure">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>embeddedStructure</name>
            <comment>
the element embeddedStructure is a subFlow element containing a full structure used as an extract from another document or position. Attribute quote is used to specify the quote character used in the original; no quote attribute implies that the quote is left in the text; quote="" implies that there is no quote character. Attribute for is used in a mmod or rmod to point to the eId of the corresponding ref element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="subFlowStructure">
					          <xsd:attributeGroup ref="quote"/>
					          <xsd:attributeGroup ref="linkopt"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="authorialNote">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>authorialNote</name>
            <comment>
the element authorialNote is a subFlow element containing an authorial (non-editorial) note in the main flow of the text.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="subFlowStructure">
					          <xsd:attributeGroup ref="notes"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="subFlow">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>subFlow</name>
            <comment>
the element subFlow is a generic element for a subFlow.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="subFlowStructure">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="foreign" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>foreign</name>
            <comment>
the element foreign is a generic container for elements not belonging to the Akoma Ntoso namespace (e.g., mathematical formulas). It is a block element and thus can be placed in a container.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:annotation>
		    <xsd:documentation>
		 =====================================================================
		
		Generic elements
		Comment: All elements in this schema fall under one of five
		content models: hierarchical container, container, block, inline
		and mark. Besides named elements, the schema also provides for a
		generic element for each of them, that can be used for markup that
		fits the content models but can be specified by a precise name that
		is not used in this schema. The 'name' attribute must be used for
		naming the element.
		It is also a characteristic of this schema that all named elements
		are equivalent, and can be substituted to, the corresponding generic
		element with their name as the value of the name attribute. Thus, by
		definition, &lt;p&gt;  = &lt;block name="p"&gt; , &lt;part&gt;  = &lt;hcontainer name="part"&gt; ,
		&lt;act&gt;  = &lt;container name="act"&gt; , etc.
		Pattern: Generic Document + Role Attribute
		Reference: http://www.xmlpatterns.com/GenericDocumentMain.shtml +
		http://www.xmlpatterns.com/RoleAttributeMain.shtml
		
		===================================================================== 
		</xsd:documentation>
	  </xsd:annotation>

	  <xsd:element name="hcontainer">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>hcontainer</name>
            <comment>
the element hcontainer is a generic element for a hierarchical container. It can be placed in a hierarchy instead of any of the other hierarchical containers. The attribute name is required and gives a name to the element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="hierarchy">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="containerType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>containerType</name>
            <comment>
the complex type containerType is the content model for the generic element for a container. It can be placed in a container instead of any of the other containers. The attribute name is required and gives a name to the element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="blockElements"/>
			      <xsd:element ref="container"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="corereq"/>
		    <xsd:attributeGroup ref="name"/>
	  </xsd:complexType>

	  <xsd:element name="container" type="containerType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>container</name>
            <comment>
the element container is a generic element for a container.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="block">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>block</name>
            <comment>
the element block is a generic element for a container. It can be placed in a container instead of any of the other blocks. The attribute name is required and gives a name to the element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="inline">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>inline</name>
            <comment>
the element inline is a generic element for an inline. It can be placed inside a block instead of any of the other inlines. The attribute name is required and gives a name to the element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="marker">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>marker</name>
            <comment>
the element marker is a generic element for a marker. It can be placed in a block instead of any of the other markers. The attribute name is required and gives a name to the element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="markerreq">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="div" type="blocksreq">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>div</name>
            <comment>
The element div is an HTML element, but is NOT used in Akoma Ntoso as in HTML. Instead of being used as a generic block, Akoma Ntoso uses div as a generic container (as in common practice)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="p" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>p</name>
            <comment>
The element p is an HTML element and is used in Akoma Ntoso as in HTML, for the generic paragraph of text (a block)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="span" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>span</name>
            <comment>
The element span is an HTML element and is used in Akoma Ntoso as in HTML, for the generic inline</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="br" type="markeropt">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>br</name>
            <comment>
The element br is an HTML element and is used in Akoma Ntoso as in HTML, for the breaking of a line</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="b" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>b</name>
            <comment>
The element b is an HTML element and is used in Akoma Ntoso as in HTML, for the bold style (an inline)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="i" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>i</name>
            <comment>
The element i is an HTML element and is used in Akoma Ntoso as in HTML, for the italic style (an inline)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="u" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>u</name>
            <comment>
The element u is an HTML element and is used in Akoma Ntoso as in HTML, for the underline style (an inline)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="sup" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>sup</name>
            <comment>
The element sup is an HTML element and is used in Akoma Ntoso as in HTML, for the superscript style (an inline)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="sub" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>sub</name>
            <comment>
The element sub is an HTML element and is used in Akoma Ntoso as in HTML, for the subscript style (an inline)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="abbr" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>abbr</name>
            <comment>
The element abbr is an HTML element and is used in Akoma Ntoso as in HTML, for the specification of an abbreviation or an acronym (an inline). As in HTML, use attribute title to specify the full expansion of the abbreviation or acronym.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="a">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>a</name>
            <comment>
The element a is an HTML element and is used in Akoma Ntoso as in HTML, for the generic link to a web resource (NOT to an Akoma Ntoso document: use ref for that). It is an inline.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:complexContent>
				        <xsd:extension base="inline">
					          <xsd:attributeGroup ref="link"/>
					          <xsd:attributeGroup ref="target"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="img">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>img</name>
            <comment>
The element img is an HTML element and is used in Akoma Ntoso as in HTML, for including an image. It is a marker.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="markeropt">
					          <xsd:attributeGroup ref="src"/>
					          <xsd:attributeGroup ref="imgAtts"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="listItems">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>listItems</name>
            <comment>
the complex type listItems specifies the content model of elements ul and ol, and specifies just a sequence of list items (elements li).</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="li" minOccurs="1" maxOccurs="unbounded"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="corereq"/>
	  </xsd:complexType>

	  <xsd:element name="ul" type="listItems">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>ul</name>
            <comment>
The element ul is an HTML element and is used in Akoma Ntoso as in HTML, for an unordered list of list item (elements li)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="ol" type="listItems">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>ol</name>
            <comment>
The element ol is an HTML element and is used in Akoma Ntoso as in HTML, for an ordered list of list item (elements li)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="li">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>li</name>
            <comment>
TYPE
Element
NAME
li
The element li is an HTML element and is used in Akoma Ntoso as in HTML, for the generic list item (not a pattern)</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType mixed="true">
			      <xsd:choice minOccurs="0" maxOccurs="unbounded">
				        <xsd:group ref="inlineCM"/>
				        <xsd:element ref="ul"/>
				        <xsd:element ref="ol"/>
				        <xsd:element ref="p"/>
			      </xsd:choice>
				     <xsd:attributeGroup ref="optvalue"/>
				     <xsd:attributeGroup ref="coreopt"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="table">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="caption" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="tr" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="corereq"/>
				     <xsd:attributeGroup ref="tableAtts"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="caption" type="inline">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>caption</name>
            <comment>
The element caption is an HTML element and is used in Akoma Ntoso as in HTML, for the caption of a table (a block)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="tr">
		    <xsd:complexType>
			      <xsd:choice minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="th"/>
				        <xsd:element ref="td"/>
			      </xsd:choice>
				     <xsd:attributeGroup ref="coreopt"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="th">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>th</name>
            <comment>
The element th is an HTML element and is used in Akoma Ntoso as in HTML, for a header cell of a table</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="blocksopt">
					          <xsd:attributeGroup ref="cellattrs"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="td">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>td</name>
            <comment>
The element td is an HTML element and is used in Akoma Ntoso as in HTML, for a data cell of a table</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="blocksopt">
					          <xsd:attributeGroup ref="cellattrs"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="meta">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="identification"/>
				        <xsd:element ref="publication" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="classification" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="lifecycle" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="workflow" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="analysis" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="temporalData" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="references" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="notes" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="proprietary" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="presentation" minOccurs="0" maxOccurs="unbounded"/>
			      </xsd:sequence>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="identification">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="FRBRWork"/>
				        <xsd:element ref="FRBRExpression"/>
				        <xsd:element ref="FRBRManifestation"/>
				        <xsd:element ref="FRBRItem" minOccurs="0" maxOccurs="1"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="coreProperties">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>coreProperties</name>
            <comment>
The complexType coreProperties lists the identifying properties available at any of the FRBR hierarchy levels.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="FRBRthis"/>
			      <xsd:element ref="FRBRuri" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRalias" minOccurs="0" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRdate" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRauthor" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="componentInfo" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="preservation" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
	  </xsd:complexType>

	  <xsd:group name="workProperties">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>workProperties</name>
            <comment>
The group workProperties lists the properties that are characteristics of the FRBR Work level.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="FRBRcountry"/>
			      <xsd:element ref="FRBRsubtype" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRnumber" minOccurs="0" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRname" minOccurs="0" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRprescriptive" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRauthoritative" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
	  </xsd:group>

	  <xsd:group name="exprProperties">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>exprProperties</name>
            <comment>
The group exprProperties lists the properties that are characteristics of the FRBR Expression level.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="FRBRversionNumber" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRauthoritative" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRmasterExpression" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRlanguage" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="FRBRtranslation" minOccurs="0" maxOccurs="unbounded"/>
		    </xsd:sequence>
	  </xsd:group>

	  <xsd:group name="manifProperties">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>manifProperties</name>
            <comment>
The group manifProperties lists the properties that are characteristics of the FRBR Expression level.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="FRBRportion" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="FRBRformat" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
	  </xsd:group>

	  <xsd:element name="FRBRWork">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRWork</name>
            <comment>
The element FRBRWork is the metadata container of identifying properties related to the Work level according to the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="coreProperties">
					          <xsd:sequence>
						            <xsd:group ref="workProperties"/>
					          </xsd:sequence>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRExpression">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRExpression</name>
            <comment>
The element FRBRExpression is the metadata container of identifying properties related to the Expression level according to the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="coreProperties">
					          <xsd:sequence>
						            <xsd:group ref="exprProperties"/>
					          </xsd:sequence>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRManifestation">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRManifestation</name>
            <comment>
The element FRBRManifestation is the metadata container of identifying properties related to the Manifestation level according to the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="coreProperties">
					          <xsd:sequence>
						            <xsd:group ref="manifProperties"/>
					          </xsd:sequence>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRItem" type="coreProperties">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRItem</name>
            <comment>
The element FRBRItem is the metadata container of identifying properties related to the Item level according to the FRBR hierarchy.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="valueType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>valueType</name>
            <comment>
The type valueType specifies a value attribute to FRBR elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="metaopt">
				        <xsd:attributeGroup ref="value"/>
				        <xsd:attributeGroup ref="refers"/>
				        <xsd:attributeGroup ref="showopt"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="booleanValueType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>booleanValueType</name>
            <comment>
The type booleanValueType specifies a boolean value attribute to FRBR elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="metaopt">
				        <xsd:attributeGroup ref="booleanvalue"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="FRBRthis" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRthis</name>
            <comment>
The element FRBRthis is the metadata property containing the IRI of the specific component of the document in the respective level of the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRuri" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRuri</name>
            <comment>
The element FRBRuri is the metadata property containing the IRI of the whole document in the respective level of the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRalias">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRalias</name>
            <comment>
The element FRBRalias is the metadata property containing additional well-known names of the document in the respective level of the FRBR hierarchy</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="valueType">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRdate">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRdate</name>
            <comment>
The element FRBRdate is the metadata property containing a relevant date of the document in the respective level of the FRBR hierarchy. Attribute name specifies which actual date is contained here.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="date"/>
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRauthor">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRauthor</name>
            <comment>
The element FRBRauthor is the metadata property containing a relevant author of the document in the respective level of the FRBR hierarchy. Attribute as specifies the role of the author.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="link"/>
					          <xsd:attributeGroup ref="role"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRlanguage">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRlanguage</name>
            <comment>
The element FRBRlanguage is the metadata property containing a RFC4646 (three-letter code) of the main human language used in the content of this expression</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="language"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRtranslation">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRtranslation</name>
            <comment>
The element FRBRtranslation is the metadata property specifying the source of which this expression is a translation of.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="link"/>
					          <xsd:attributeGroup ref="fromLanguage"/>
					          <xsd:attributeGroup ref="authoritative"/>
					          <xsd:attributeGroup ref="pivot"/>
					          <xsd:attributeGroup ref="agent"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRsubtype" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRsubtype</name>
            <comment>
The element FRBRsubtype is the metadata property containing a string for the specific subtype of the document to be used in the work-level IRI of this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRcountry" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRcountry</name>
            <comment>
The element FRBRcountry is the metadata property containing a ISO 3166-1 Alpha-2 code for the country or jurisdiction to be used in the work-level IRI of this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRnumber" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRnumber</name>
            <comment>
The element FRBRnumber is the metadata property containing a string or number for the number to be used in the work-level IRI of this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRname" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRname</name>
            <comment>
The element FRBRname is the metadata property containing a string for the title to be used in the work-level IRI of this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRportion">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRportion</name>
            <comment>
The element FRBRportion is the metadata property containing the eId of the portion contained in a manifestation-level portion. If the portion contains an interval of elements, the range attributes specifies the first and last one.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="refers"/>
					          <xsd:attributeGroup ref="showopt"/>
					          <xsd:attributeGroup ref="rangeOpt"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="FRBRformat" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRformat</name>
            <comment>
The element FRBRformat is the metadata property containing the Internet Media Type specification for the data format to be used in the manifestation-level IRI of this document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRprescriptive" type="booleanValueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRprescriptive</name>
            <comment>
The element FRBRprescriptive is the metadata property containing a boolean value to determine whether the document contains prescriptive text (i.e., text that is or might become prescriptive, such as an act or a bill) or not (such as, for instance, a non-normative resolution from an assembly.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRversionNumber" type="valueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRversionNumber</name>
            <comment>
The element FRBRversionNumber is the metadata property containing the value of the version number if appropriate to identify the specific expression here contained. It allows an arbitrary string.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRauthoritative" type="booleanValueType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRauthoritative</name>
            <comment>
The element FRBRauthoritative is the metadata property containing a boolean value to determine whether the document contains authoritative text (i.e., content that is the official, authoritative product of an official workflow from an entity that is entrusted with generating an official, authoriative version of the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="FRBRmasterExpression" type="linkType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>FRBRmasterExpression</name>
            <comment>
The element FRBRmasterExpression is the metadata property identifying the master expression, i.e., the expression whose ids are used as permanent ids in the wId attributes. An expression without the FRBRmasterExpression element is considered a master expression itself, i.e., the first version, or the most important version, of a document expressed in the only language, or in the most important language. Any other situation (subsequent versions, or language variants, or content variants) must have the FRBRmasterExpression element pointing to the URI of the master expression. If the FRBRmasterEpression is specified, but without a href pointing to the masterExpression, it is assumed that NO master expression exist in reality, but an UR-Expression exist, whose ids are used in this expression as wIds.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="componentInfo">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="0" maxOccurs="unbounded">
				        <xsd:element ref="componentData"/>
			      </xsd:sequence>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="componentData">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="0" maxOccurs="unbounded">
				        <xsd:element ref="componentData"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="core"/>
				     <xsd:attributeGroup ref="idreq"/>
				     <xsd:attributeGroup ref="name"/>
				     <xsd:attributeGroup ref="link"/>
				     <xsd:attributeGroup ref="show"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="preservation" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>preservation</name>
            <comment>
The element preservation is the metadata property containing an arbitrary list of elements detailing the preservation actions taken for the document is the respective level of the FRBR hierarchy..</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="publication">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>publication</name>
            <comment>
The element publication is the metadata container specifying an official publication event for the FRBR expression of the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="date"/>
					          <xsd:attributeGroup ref="show"/>
					          <xsd:attributeGroup ref="name"/>
					          <xsd:attributeGroup ref="number"/>
					          <xsd:attributeGroup ref="refers"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="classification">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="keyword" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="keyword">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>keyword</name>
            <comment>
The element keyword is a metadata element specifying a keyword  associated to the FRBR expression of the document. Attribute dictionary (required) specifies the thesaurus out of which the keyword has been taken. Attribute href points to the fragment of text this keyword is associated to. Keywords without href attribute refer to the content as a whole.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="linkopt"/>
					          <xsd:attributeGroup ref="value"/>
					          <xsd:attributeGroup ref="show"/>
					          <xsd:attributeGroup ref="dictionary"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="lifecycle">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="eventRef" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="eventRef">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>eventRef</name>
            <comment>
The element eventInfo is a metadata element specifying facts about an event that had an effect on the document. For each event, a date, a type and a document that generated the event must be referenced.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="date"/>
					          <xsd:attributeGroup ref="source"/>
					          <xsd:attributeGroup ref="refers"/>
					          <xsd:attributeGroup ref="eventType"/>
					          <xsd:attributeGroup ref="originating"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="workflow">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="step" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="step">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>step</name>
            <comment>
The element step is a metadata element specifying facts about a workflow step occurred to the document. For each event, a date, a type, an agent (and the corresponding role) that generated the action must be referenced. The outcome, too, can be specified.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="date"/>
					          <xsd:attributeGroup ref="agent"/>
					          <xsd:attributeGroup ref="role"/>
					          <xsd:attributeGroup ref="refers"/>
					          <xsd:attributeGroup ref="outcome"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="analysis">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="activeModifications" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="passiveModifications" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="restrictions" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="judicial" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="parliamentary" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="mappings" minOccurs="0" maxOccurs="1"/>
				        <xsd:element ref="otherReferences" minOccurs="0" maxOccurs="unbounded"/>
				        <xsd:element ref="otherAnalysis" minOccurs="0" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="Amendments">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>Amendments</name>
            <comment>
The complex type Amendments is a list of all the amendment elements that can be used on a document analysis</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:element ref="textualMod"/>
			      <xsd:element ref="meaningMod"/>
			      <xsd:element ref="scopeMod"/>
			      <xsd:element ref="forceMod"/>
			      <xsd:element ref="efficacyMod"/>
			      <xsd:element ref="legalSystemMod"/>
		    </xsd:choice>
	  </xsd:complexType>

	  <xsd:complexType name="modificationType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>modificationType</name>
            <comment>
The complex type modificationType lists all the properties associated to modification elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="source" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="destination" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="force" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="efficacy" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="application" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="duration" minOccurs="0" maxOccurs="1"/>
			      <xsd:element ref="condition" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="enactment"/>
		    <xsd:attributeGroup ref="modifiers"/>
		    <xsd:attributeGroup ref="refers"/>
	  </xsd:complexType>

	  <xsd:element name="activeModifications" type="Amendments">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>activeModifications</name>
            <comment>
The element activeModifications is a metadata container of the active modifications generated by the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="passiveModifications" type="Amendments">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>passiveModifications</name>
            <comment>
The element passiveModifications is a metadata container of the passive modifications affecting the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="textualMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>textualMod</name>
            <comment>
The element textualMod is a metadata element specifying an (active or passive) textual modification for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:sequence>
						            <xsd:element ref="previous" minOccurs="0" maxOccurs="1"/>
						            <xsd:element ref="old" minOccurs="0" maxOccurs="unbounded"/>
						            <xsd:element ref="new" minOccurs="0" maxOccurs="unbounded"/>
					          </xsd:sequence>
					          <xsd:attributeGroup ref="textualModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="meaningMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>meaningMod</name>
            <comment>
The element meaningMod is a metadata element specifying an (active or passive) modification in meaning for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:sequence>
						            <xsd:element ref="domain" minOccurs="0" maxOccurs="1"/>
					          </xsd:sequence>
					          <xsd:attributeGroup ref="meaningModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="scopeMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>scopeMod</name>
            <comment>
The element scopeMod is a metadata element specifying an (active or passive) modification in scope for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:sequence>
						            <xsd:element ref="domain" minOccurs="0" maxOccurs="1"/>
					          </xsd:sequence>
					          <xsd:attributeGroup ref="scopeModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="forceMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>forceMod</name>
            <comment>
The element forceMod is a metadata element specifying an (active or passive) modification in force for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:attributeGroup ref="forceModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="efficacyMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>efficacyMod</name>
            <comment>
The element efficacyMod is a metadata element specifying an (active or passive) modification in efficacy for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:attributeGroup ref="efficacyModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="legalSystemMod">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>legalSystemMod</name>
            <comment>
The element legalSystemMod is a metadata element specifying an (active or passive) modification in the legal system for the document.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="modificationType">
					          <xsd:attributeGroup ref="legalSystemModType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="judicialArguments">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>judicialArguments</name>
            <comment>
The complex type judicialArguments is a list of all the judicial analysis elements that can be used on the analysis of a judgment</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="result"/>
			      <xsd:choice minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="supports"/>
				        <xsd:element ref="isAnalogTo"/>
				        <xsd:element ref="applies"/>
				        <xsd:element ref="extends"/>
				        <xsd:element ref="restricts"/>
				        <xsd:element ref="derogates"/>
				        <xsd:element ref="contrasts"/>
				        <xsd:element ref="overrules"/>
				        <xsd:element ref="dissentsFrom"/>
				        <xsd:element ref="putsInQuestion"/>
				        <xsd:element ref="distinguishes"/>
			      </xsd:choice>
		    </xsd:sequence>
	  </xsd:complexType>

	  <xsd:complexType name="judicialArgumentType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>judicialArgumentType</name>
            <comment>
The complex type judicialArgumentType lists all the properties associated to judicial elements.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:sequence>
			      <xsd:element ref="source" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="destination" minOccurs="1" maxOccurs="unbounded"/>
			      <xsd:element ref="condition" minOccurs="0" maxOccurs="1"/>
		    </xsd:sequence>
		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="enactment"/>
		    <xsd:attributeGroup ref="modifiers"/>
		    <xsd:attributeGroup ref="refers"/>
	  </xsd:complexType>

	  <xsd:element name="judicial" type="judicialArguments">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>judicial</name>
            <comment>
The element judicial is a metadata container of the analysis of the judicial arguments of a judgment.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="result">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>result</name>
            <comment>
The element result is a metadata element specifying the overall result of the judgment.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="resultType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="supports" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>supports</name>
            <comment>
The element supports is a metadata element specifying a reference to a source supported by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="isAnalogTo" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>isAnalogTo</name>
            <comment>
The element isAnalogTo is a metadata element specifying a reference to a source analog to the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="applies" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>applies</name>
            <comment>
The element applies is a metadata element specifying a reference to a source applyed by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="extends" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>extends</name>
            <comment>
The element extends is a metadata element specifying a reference to a source extended by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="restricts" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>restricts</name>
            <comment>
The element restricts is a metadata element specifying a reference to a source restricted by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="derogates" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>derogates</name>
            <comment>
The element derogates is a metadata element specifying a reference to a source derogated by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="contrasts" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>contrasts</name>
            <comment>
The element contrasts is a metadata element specifying a reference to a source contrasted by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="overrules" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>overrules</name>
            <comment>
The element overrules is a metadata element specifying a reference to a source overruled by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="dissentsFrom" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>dissentsFrom</name>
            <comment>
The element dissentsFrom is a metadata element specifying a reference to a source dissented from the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="putsInQuestion" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>putsInQuestion</name>
            <comment>
The element putsInQuestions is a metadata element specifying a reference to a source questioned by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="distinguishes" type="judicialArgumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>distinguishes</name>
            <comment>
The element distinguishes is a metadata element specifying a reference to a source being distinguished by the argument being described.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="restrictions">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="restriction"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="restriction">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>restriction</name>
            <comment>
The element restriction specifies information about a restriction (such as a jurisdiction specification) by pointing to a specific legislative, geographic or temporal events through the refersTo attribute</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="refers"/>
					          <xsd:attributeGroup ref="restrictionType"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="parliamentaryAnalysis">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>parliamentaryAnalysis</name>
            <comment>
The complex type parliamentaryAnalysis is a list of all the parliamentary analysis elements that can be used on the analysis of a debate</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:element ref="quorumVerification"/>
			      <xsd:element ref="voting"/>
			      <xsd:element ref="recount"/>
		    </xsd:choice>
	  </xsd:complexType>

	  <xsd:element name="parliamentary" type="parliamentaryAnalysis">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>parliamentary</name>
            <comment>
The element parliamentary is a metadata container of the analysis of the events of a debate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="parliamentaryAnalysisType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>parliamentaryAnalysisType</name>
            <comment>
The complex type parliamentaryAnalysisType lists all the properties associated to elements in the parliamentary analysis.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:element ref="quorum"/>
			      <xsd:element ref="count"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="outcome"/>
		    <xsd:attributeGroup ref="refers"/>
		    <xsd:attributeGroup ref="linkopt"/>
	  </xsd:complexType>

	  <xsd:element name="quorumVerification" type="parliamentaryAnalysisType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>quorumVerification</name>
            <comment>
The element quorumVerification is a metadata container containing information about an event of quorum verification happened within a debate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="voting" type="parliamentaryAnalysisType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>voting</name>
            <comment>
The element voting is a metadata container containing information about an event of a vote happened within a debate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="recount" type="parliamentaryAnalysisType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>recount</name>
            <comment>
The element recount is a metadata container containing information about an event of a recount happened within a debate.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:complexType name="countType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>countType</name>
            <comment>
The complex type countType lists all the properties associated to elements of parliamentary count.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:any namespace="##other"
                  processContents="lax"
                  minOccurs="0"
                  maxOccurs="unbounded"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="core"/>
		    <xsd:attributeGroup ref="idreq"/>
		    <xsd:attributeGroup ref="value"/>
		    <xsd:attributeGroup ref="refers"/>
		    <xsd:attributeGroup ref="linkopt"/>
	  </xsd:complexType>

	  <xsd:element name="quorum" type="countType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>quorum</name>
            <comment>
The element quorum is a metadata container containing the value of a quorum in a vote or a quorum verification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="count" type="countType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>count</name>
            <comment>
The element count is a metadata container containing the value of a count in a vote or a quorum verification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="mappings">
      <xsd:annotation>
		       <xsd:documentation>
		 The element mapping contains a reference to the permanent wId (attribute original) of a structure, and to the eId (attribute current) such structure had during the time interval included between an initial temporalGroup and a final temporalGroup. This is useful for tracking the evolving ids of documents frequently renumbered (e,g., bills). Every single element whose wId does not match its eId needs to be represented here. 
		</xsd:documentation>
	     </xsd:annotation>
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="mapping"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="mapping">
		

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metareq">
					          <xsd:attributeGroup ref="mappingAtts"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="otherReferences">
      <xsd:annotation>
		       <xsd:documentation>
		 The element otherRef contains a legal reference to a document that is not explicitly mentioned in the content of the document. 
		</xsd:documentation>
	     </xsd:annotation>
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="otherRef"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="otherRef">
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="source" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="core"/>
				     <xsd:attributeGroup ref="idreq"/>
				     <xsd:attributeGroup ref="enactment"/>
				     <xsd:attributeGroup ref="refers"/>
				     <xsd:attributeGroup ref="link"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="otherAnalysis">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>otherAnalysis</name>
            <comment>
The element otherAnalysis is a metadata container of any additional metadata analysis element that does not belong to the specific categories provided before. Anything can be placed in this element..</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="source"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:complexType name="argumentType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>argumentType</name>
            <comment>
the complex type argumentType defines the empty content model and the list of attributes for metadata elements in the analysis section</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="anyOtherType">
				        <xsd:attributeGroup ref="pos"/>
				        <xsd:attributeGroup ref="modifiers"/>
				        <xsd:attributeGroup ref="upToOpt"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:complexType name="periodType">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>periodType</name>
            <comment>
the complex type periodType defines the empty content model and the list of attributes for metadata elements in the analysis section using periods</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexContent>
			      <xsd:extension base="anyOtherType">
				        <xsd:attributeGroup ref="period"/>
			      </xsd:extension>
		    </xsd:complexContent>
	  </xsd:complexType>

	  <xsd:element name="source" type="argumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>source</name>
            <comment>
The element source is a metadata element specifying the IRI of the source of the modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="destination" type="argumentType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>destination</name>
            <comment>
The element destination is a metadata element specifying the IRI of the destination of the modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="force" type="periodType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>force</name>
            <comment>
The element force is a metadata element specifying the period of the force modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="efficacy" type="periodType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>efficacy</name>
            <comment>
The element efficacy is a metadata element specifying the period of the efficacy modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="application" type="periodType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>application</name>
            <comment>
The element application is a metadata element specifying the period of the application modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="duration" type="periodType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>duration</name>
            <comment>
The element duration is a metadata element specifying the period of the duration modification.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="condition">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>condition</name>
            <comment>
The element condition is a metadata element specifying an open set of conditions on the modification (non managed by Akoma Ntoso).</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="frozen"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="previous" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>previous</name>
            <comment>
The element previous is a metadata element referring to the element (rather than the text) of the modification in the previous version of the document. This is especially useful when renumbering occurs, so as to specify the eId of the instance that was modified in the previous version. Attribute href points to the eId of the element being modified in the old version, using a full expression-level URI.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="old" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>old</name>
            <comment>
The element old is a metadata element containing (in some non-managed form) the old text of the modification. Attribute href points to the eId of the element new it is being substituted by.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="new" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>new</name>
            <comment>
The element new is a metadata element containing (in some non-managed form) the new text of the modification. Attribute href points to the eId of the element old it is substituting.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="domain" type="anyOtherType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>domain</name>
            <comment>
The element domain is a metadata element containing (in some non-managed form) the domain to which the modification applies.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="temporalData">
      <xsd:annotation>
		       <xsd:documentation>
		 The element temporalGroup contains a list of time Intervals that need to be applied together to an element of the document that is associated to time information. Each temporal group can contain any number of time interval associated via the refersTo to a different time-specific concept. 
		</xsd:documentation>
	     </xsd:annotation>
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="temporalGroup" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="temporalGroup">
      <xsd:annotation>
		       <xsd:documentation>
		 The element timeInterval contains all the data needed to identify a specific time interval. It is built either with two dates or with a date and a duration (exactly two of the 'start', 'end' and 'duration' attributes can be specified). Values of the 'start' and 'end' attributes are NOT dates, but references to event elements in the corresponding metadata section. The refers attribute is a reference to a temporal concept belonging to the Akoma Ntoso ontology and specified in the references section 
		</xsd:documentation>
	     </xsd:annotation>
		    <xsd:complexType>
			      <xsd:sequence>
				        <xsd:element ref="timeInterval" minOccurs="1" maxOccurs="unbounded"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="core"/>
				     <xsd:attributeGroup ref="idreq"/>
		    </xsd:complexType>
	  </xsd:element>

	

	  <xsd:element name="timeInterval">
		

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="metaopt">
					          <xsd:attributeGroup ref="interval"/>
					          <xsd:attributeGroup ref="duration"/>
					          <xsd:attributeGroup ref="refersreq"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:group name="docRefs">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>docRefs</name>
            <comment>
The group docrefs is a list of types of legal references to documents.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="original"/>
			      <xsd:element ref="passiveRef"/>
			      <xsd:element ref="activeRef"/>
			      <xsd:element ref="jurisprudence"/>
			      <xsd:element ref="hasAttachment"/>
			      <xsd:element ref="attachmentOf"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:group name="TLCs">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Group</type>
            <name>TLCs</name>
            <comment>
The group TLCs is a list of types of Top Level classes of the Akoma Ntoso ontology.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice>
			      <xsd:element ref="TLCPerson"/>
			      <xsd:element ref="TLCOrganization"/>
			      <xsd:element ref="TLCConcept"/>
			      <xsd:element ref="TLCObject"/>
			      <xsd:element ref="TLCEvent"/>
			      <xsd:element ref="TLCLocation"/>
			      <xsd:element ref="TLCProcess"/>
			      <xsd:element ref="TLCRole"/>
			      <xsd:element ref="TLCTerm"/>
			      <xsd:element ref="TLCReference"/>
		    </xsd:choice>
	  </xsd:group>

	  <xsd:complexType name="refItems">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Complex</type>
            <name>refItems</name>
            <comment>
The complex type refItems is a list of types of references used in the references section.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:choice minOccurs="1" maxOccurs="unbounded">
			      <xsd:group ref="docRefs"/>
			      <xsd:group ref="TLCs"/>
		    </xsd:choice>
		    <xsd:attributeGroup ref="source"/>
	  </xsd:complexType>

	  <xsd:element name="references" type="refItems">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>references</name>
            <comment>
The element references is a metadata container of all the references to entities external to the document mentioned in the document. They include references to legal documents of any form,a s well as references to people, organizations, events, roles, concepts, and anything else is managed by the Akoma Ntoso ontology.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="original" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>original</name>
            <comment>
The element original is a metadata reference to the Akoma Ntoso IRI of the original version of this document (i.e., the first expression)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="passiveRef" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>passiveRef</name>
            <comment>
The element passiveRef is a metadata reference to the Akoma Ntoso IRI of a document providing modifications on this document (i.e., a passive references)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="activeRef" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>activeRef</name>
            <comment>
The element activeRef is a metadata reference to the Akoma Ntoso IRI of a document that is modified by this document (i.e., an active references)</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="jurisprudence" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>jurisprudence</name>
            <comment>
The element jurisprudence is a metadata reference to the Akoma Ntoso IRI of a document providing jurisprudence on this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="hasAttachment">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>hasAttachment</name>
            <comment>
The element hasAttachment is a metadata reference to the Akoma Ntoso IRI of an attachment of this document</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="referenceType">
					          <xsd:attributeGroup ref="type"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="attachmentOf">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>attachmentOf</name>
            <comment>
The element attachmentOf is a metadata reference to the Akoma Ntoso IRI of a document of which this document is an attachment</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="referenceType">
					          <xsd:attributeGroup ref="type"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="TLCPerson" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCPerson</name>
            <comment>
The element TLCPerson is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Person</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCOrganization" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCOrganization</name>
            <comment>
The element TLCOrganization is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Organization</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCConcept" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCConcept</name>
            <comment>
The element TLCConcept is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Concept</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCObject" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCObject</name>
            <comment>
The element TLCObject is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Object</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCEvent" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCEvent</name>
            <comment>
The element TLCEvent is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Event</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCLocation" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCLocation</name>
            <comment>
The element TLCLocation is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Location</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCProcess" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCProcess</name>
            <comment>
The element TLCProcess is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Process</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCRole" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCRole</name>
            <comment>
The element TLCRole is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Role</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCTerm" type="referenceType">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCTerm</name>
            <comment>
The element TLCTerm is a metadata reference to the Akoma Ntoso IRI of an ontology instance of the class Term</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="TLCReference">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>TLCReference</name>
            <comment>
The element TLCreference is a generic metadata reference to the Akoma Ntoso IRI of an ontology instance of a class specified through the name attribute</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="referenceType">
					          <xsd:attributeGroup ref="name"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="notes">
		    <xsd:complexType>
			      <xsd:sequence minOccurs="1" maxOccurs="unbounded">
				        <xsd:element ref="note"/>
			      </xsd:sequence>
				     <xsd:attributeGroup ref="source"/>
				     <xsd:attributeGroup ref="xmllang"/>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="note" type="subFlowStructure">
      <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>note</name>
            <comment>
The element note is a metadata element containing the text of the footnote and endnote specified.</comment>
				     </xsd:documentation>
		    </xsd:annotation>
   </xsd:element>
		


	  <xsd:element name="proprietary">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>proprietary</name>
            <comment>
The element proprietary is a metadata container of any additional metadata property that does not belong to the Akoma Ntoso properties. Anything can be placed in this element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="source"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	  <xsd:element name="presentation">
		    <xsd:annotation>
			      <xsd:documentation>
			         <type>Element</type>
            <name>presentation</name>
            <comment>
The element presentation is a metadata container of any presentation specification for the visual rendering of Akoam Ntoso elements. Anything can be placed in this element.</comment>
				     </xsd:documentation>
		    </xsd:annotation>

		    <xsd:complexType>
			      <xsd:complexContent>
				        <xsd:extension base="anyOtherType">
					          <xsd:attributeGroup ref="source"/>
				        </xsd:extension>
			      </xsd:complexContent>
		    </xsd:complexType>
	  </xsd:element>

	

</xsd:schema>
````

## File: tulit/parsers/xml/assets/eli.owl
````
<?xml version="1.0"?>
<rdf:RDF xmlns="http://data.europa.eu/eli/ontology#"
     xml:base="http://data.europa.eu/eli/ontology"
     xmlns:dc="http://purl.org/dc/elements/1.1/"
     xmlns:crm="http://www.cidoc-crm.org/cidoc-crm/"
     xmlns:eli="http://data.europa.eu/eli/ontology#"
     xmlns:owl="http://www.w3.org/2002/07/owl#"
     xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"
     xmlns:xml="http://www.w3.org/XML/1998/namespace"
     xmlns:xsd="http://www.w3.org/2001/XMLSchema#"
     xmlns:foaf="http://xmlns.com/foaf/0.1/"
     xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#"
     xmlns:skos="http://www.w3.org/2004/02/skos/core#"
     xmlns:vann="http://purl.org/vocab/vann/"
     xmlns:regap="http://metadataregistry.org/uri/profile/regap/"
     xmlns:terms="http://purl.org/dc/terms/"
     xmlns:frbroo="http://iflastandards.info/ns/fr/frbr/frbroo/"
     xmlns:rdakit="http://metadataregistry.org/uri/profile/rdakit/">
    <owl:Ontology rdf:about="http://data.europa.eu/eli/ontology#">
        <owl:imports rdf:resource="http://www.w3.org/2004/02/skos/core"/>
        <vann:preferredNamespacePrefix>eli</vann:preferredNamespacePrefix>
        <rdfs:comment xml:lang="en">An ontology to describe legislation, based on the FRBR model, that encodes the ELI metadata fields.
See the classes &quot;LegalResource&quot;, &quot;LegalExpression&quot; and &quot;Format&quot; as entry points.
Interested readers can find more information about ELI on the ELI register at http://eurlex.europa.eu/eli and on the Office of Publications vocabularies website at https://op.europa.eu/en/web/eu-vocabularies/eli.</rdfs:comment>
        <rdfs:label xml:lang="en">ELI Metadata Ontology</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:seeAlso rdf:resource="http://eurlex.europa.eu/eli"/>
        <rdfs:seeAlso rdf:resource="https://op.europa.eu/en/web/eu-vocabularies/eli"/>
        <owl:versionInfo rdf:datatype="http://www.w3.org/2001/XMLSchema#string">1.5</owl:versionInfo>
        <skos:editorialNote xml:lang="en">- The ELI ontology reuses the property names from the Metalex ontology (http://www.metalex.eu/) to express the FRBR skeleton hierarchy : is_realized_by/realizes, is_embodied_by/embodies, is_exemplified_by. This wording is also the one used in the original FRBR specifications. However, the decision has been made _not_ to align to the Metalex class names (BibliographicWork, BibliographicExpression, BibliographicManifestation), because ELI is specifically about publishing legal metadata on the web, and not structuring legal content in any document.

- first_date_entry_in_force and date_no_longer_in_force could be aligned with dcterms:date

- a class could be introduced for URI schemas to implement the URI schema registry

- ELI does not use rdfs:subPropertyOf to relate its properties with DublinCore Terms, but instead a custom annotation properties &quot;eli:isSpecificPropertyOf&quot;. ELI 1.1 chose to have 2 separate properties in certain situations to accomodate with cases where the value can be either a String or a reference to an identified object; this is the case for eli:publisher / eli:publisher_agent, or eli:rightsholder and eli:rightsholder_agent. This entails that this pair of properties cannot be both aligned with their respective DCTerms equivalent (dcterms:publisher and dcterms:rightsholder), since relating a property using rdfs:subPropertyOf necessary implies to declare the DCterms property as either a datatype property _or_ an objectProperty, but not both.</skos:editorialNote>
        <skos:historyNote xml:lang="en">## 1.5

### Semantic changes

- Introduced eli:refers_to / eli:is_referred_to_by on Work or Expression.
- Made eli:cites / eli:cited_by subproperties of eli:refers_to / eli:is_referred_to_by
- Made eli:cited_by_case_law a subproperty of eli:is_referred_to_by
- broaden eli:number on eli:Work or eli:Expression

### Non-semantic changes

_none_

## 1.4

### Semantic changes

- Introduced generic eli:ComplexWork class
- Introduced generic eli:WorkSubdivision class
- introducted generic eli:work_type property on eli:Work (and made eli:type_document a subproperty of this)
- introduced eli:is_derivative_of / eli:has_derivative on Work or Expression
- introduced eli:has_annex / eli:is_annex_of on Work or Expression
- broaden eli:date_document on eli:Work
- broaden eli:is_about on eli:Work
- broaden eli:changes / changed_by , eli:basis_for / based_on, eli:corrects / corrected_by, eli:amends / amended_by to eli:Work or eli:Expression

### Non-semantic changes

- adjusted the subclassOf relationship of eli:Organization, eli:Agent, eli:Person, eli:Manifestation to proper FRBRoo classes


## 1.3

### Semantic changes

  - Introduced High-Level FRBR classes Work, Expression and Manifestation, to be shared in ELI-DL;
  - Broaden the domain and range of structural FRBR properties to Work, Expression or Manifestation, so that they can be shared in ELI-DL :
    - eli:is_member_of/eli:has_member
    - eli:is_part_of/eli:has_part
    - eli:is_realized_in/eli:realizes
    - eli:is_embodied_by/eli:embodies
    - eli:exemplified_by
  - broaden the domain of eli:title, eli:title_short, eli:title_alternative, eli:language to:eli:Expression so that they can be shared in ELI-DL;
  - introduced eli:type_subdivision to express the type of a LegalResourceSubdivision
  - introduced eli:countersigned_by to capture countersignatures of legislation
  - introduced eli:ensures_implementation_of / eli:implementation_ensured_by to capture the fact that some legislation are modified to remain &quot;compatible&quot; with other EU-level or national-level regulations; this implies a legal meaning;

### Non-semantic changes

  - fixed the definition of the modified properties to match their new domain/range
  - updated the links to MDR to the EU vocabularies website
  - made eli:Manifestation a subclass of frbroo:F3_Manifestation to follow new LRM conceptual model;


## 1.2

### Semantic changes

  - Modification of eli:format and introduction of eli:media_type :
    - Modified the semantic of eli:format so that it now corresponds to a business / application format, and not a technical format (the technical format will be captured in eli:media_type). This is intended to distinguish e.g. authentic PDF from scanned PDF from generated-on-the-fly PDF; or e.g. XML in a standard (AkomaNtoso) format vs. XML in a proprietary format.
      - It is now possible to use the custom URIs &quot;application/pdf;type=signed&quot;, &quot;application/pdf;type=generated&quot;, &quot;application/pdf;type=archival&quot;, &quot;application/pdf;type=scanned&quot; as values of eli:format to indicate respectively electronically signed PDFs, generated-on-the-fly PDFs, PDF for archival, or scanned PDFs;
      - It is now possible to use the custom URI &quot;eli:print_format&quot; as a value of eli:format to indicate a printed/paper version of the legislation;
      - It is now possible to use the URI &quot;application/akn+xml&quot; as a value of eli:format to indicate an AkomaNtoso format;
    - Added a new property &quot;eli:media_type&quot; to capture the technical file format of the file; this is intended to serve as a basis for content negociation; IANA identifiers MUST be used in eli:media_type;

    - Existing ELI implementations using eli:format with IANA URIs remain valid; the eli:format property remains mandatory. ELI providers are however encouraged to add the &quot;eli:media_type&quot; property to their ELI metadata; in some situations the value of both properties might be the same (&quot;eli:format = application/xml&quot; and &quot;eli:media_type = application/xml&quot;). It is expected that in the future the &quot;eli:media_type&quot; property will become mandatory.

  - Added a couple of optional properties &quot;has_translation&quot; / &quot;is_translation_of&quot; to indicate that a LegalExpression is translated from another one;
  - Added the property &quot;cited_by_case_law&quot; and the corresponding literal property &quot;cited_by_case_law_reference&quot; :
    - Cited_by_case_law : Indicates that this LegalResource or LegalExpression is being cited in a case law, identified by a suitable URI. If the case law cannot be identified by a suitable URI, the property &quot;eli:cited_by_case_law_reference&quot; can be used with a textual reference to the case law. The actual citation link is expressed from the case law to the legislation, but legal portals may use the link from a legislation to a case law to e.g. refer to representative case laws about a legislation.
    - Cited_by_case_law_reference : Indicates that this LegalResource or LegalExpression is being cited in a case law that cannot be identified by a suitable URI and that is indicated by textual reference. An ECLI (European Case Law Identifier) can be used here. When the case law can be identified by a suitable URI, the property eli:cited_by_case_law should be used instead.

### Non-semantic changes

  - Referred to FRBRoo URIs instead of RDA URIs to extend the LegalResource, LegalExpression and Format classes, along with the FRBR backbone properties. This properly connects ELI with FRBRoo;
  - Added equivalence declarations between ELI and the schema.org legal extension; this is provided as a separate Excel and Turtle file;
  - Removed xsd:string range declaration on eli:description, title, title_short, title_alternative, rightsholder, rights, publisher, published_in so that can be declared with either a datatype or a language tag;
  - Fixed documentation of some properties;
  - Added link to Eur-Lex website in ontology documentation;

## 1.1

  - added the following properties :
    - is_member_of / has_member
    - number
    - jurisdiction
    - date_applicability
    - applies / applied_by
    - commences / commenced_by
    - repeals / repealed_by
    - corrects / corrected_by
    - amends / amended_by
    - is_another_publication_of / has_another_publication

  - additionnal properties were also added to accomodate with situations where a property value can be either a literal or a URI :
    - responsibility_of_agent vs. responsibility_of
    - published_in_format vs. published_in
    - publisher_agent vs. publisher
    - rightsholder_agent vs. rightsholder

  - the relations with DublinCore properties is now declared with a custom annotation property &quot;eli:isSpecificPropertyOf&quot; instead of rdfs:subPropertyOf to be able to relate both eli:publisher/eli:publisher_agent to dcterms:publisher, and eli:rightsholder/eli:rightsholder_agent to dcterms:rightsholder.

  - added the class LegalResourceSubdivision

  - the propery implements / implemented_by was deprecated in favor of applies / applied_by

  - numerous definitions were improved and clarified

  - all ontology items are documented with :
    - their version of creation;
    - their version of latest modification;
    - an history note documenting the changes on the item;

V1.1 is backward compatible with v1.0, with 1 following exception :
  - starting from v1.1, is_member_of / has_member should be used instead of is_part_of / has_part for conceptual inclusions;

Additionnally, ELI implementers are strongly encouraged to use applies / applied_by instead of implements / implemented_by; note however that implements/implemented_by currently remain in the ontology so publishers can take the time to transition smoothly.

## 1.0

  - added a set of values for the InForce table
  - added definitions to tables
  - minor fixes in definitions
  - added rdfs:seeAlso from properties to corresponding tables
  - cleaned the &quot;whyThisIsNeeded&quot; annotation

## 0.6.1

  - fixed typos in definitions

## 0.6

  - imported skos rather than redeclaring skos classes and properties
  - changed domain of in_force, first_date_entry_into_force, date_no_longer_in_force to [LegalResource or LegalExpression] instead of LegalResource
  - changed domain of language, title, title_short, title_alternative to LegalExpression instead of [LegalResource or LegalExpression]
  - reworked the tables : don&apos;t use country and resource type tables from MDR, but declare local tables.
  - integrated new definitions for all classes and properties

## 0.5

  - Renamed &quot;Interpretation&quot; into &quot;LegalExpression&quot; (and modified comments accordingly)
  - Renamed the structural FRBR relations in accordance with Metalex ontology :
    - is_legal_resource_of becomes &quot;is_realized_by&quot;
    - has_legalresource becomes &quot;realizes&quot;
    - is_interpretation_of becomes &quot;is_embodied_by&quot;
    - has_interpretation becomes &quot;embodies&quot;
    - file becomes &quot;is_exemplified_by&quot;
  - file has been made a subproperty of &quot;has exemplar of manifestation&quot; instead of &quot;dc:hasPart&quot;
  - relevant_for is no more considered mandatory. Deleted comment that stated &quot;we are seeking a way to express that relevant_for must have at least one value for LegalResource or one of its LegalInterpretation&quot;.

## 0.4.1

  - Added &quot;publishes&quot; property
  - Enriched the note on &quot;transposes&quot; and &quot;transposed by&quot; to indicate they must point to / be expressed on LegalResource</skos:historyNote>
    </owl:Ontology>
    


    <!-- 
    ///////////////////////////////////////////////////////////////////////////////////////
    //
    // Annotation properties
    //
    ///////////////////////////////////////////////////////////////////////////////////////
     -->

    


    <!-- http://data.europa.eu/eli/ontology#originalName -->

    <owl:AnnotationProperty rdf:about="http://data.europa.eu/eli/ontology#originalName"/>
    


    <!-- http://data.europa.eu/eli/ontology#specificPropertyOf -->

    <owl:AnnotationProperty rdf:about="http://data.europa.eu/eli/ontology#specificPropertyOf">
        <rdfs:comment xml:lang="en">Indicates that a property is, in a generic sense, a more specific property of another one. This does not entail the same logical reasoning as rdfs:subPropertyOf.
This is used to indicate the alignement to Dublin Core properties, without redeclaring the Dublin Core property in Protégé, and without commiting to choose that a DubinCore property is either a datatype or an object property.</rdfs:comment>
        <rdfs:label xml:lang="en">specific property of</rdfs:label>
    </owl:AnnotationProperty>
    


    <!-- http://purl.org/vocab/vann/preferredNamespacePrefix -->

    <owl:AnnotationProperty rdf:about="http://purl.org/vocab/vann/preferredNamespacePrefix"/>
    


    <!-- http://www.w3.org/2004/02/skos/core#broader -->

    <owl:AnnotationProperty rdf:about="http://www.w3.org/2004/02/skos/core#broader"/>
    


    <!-- http://www.w3.org/2004/02/skos/core#hasTopConcept -->

    <owl:AnnotationProperty rdf:about="http://www.w3.org/2004/02/skos/core#hasTopConcept"/>
    


    <!-- http://www.w3.org/2004/02/skos/core#narrower -->

    <owl:AnnotationProperty rdf:about="http://www.w3.org/2004/02/skos/core#narrower"/>
    


    <!-- http://www.w3.org/2004/02/skos/core#topConceptOf -->

    <owl:AnnotationProperty rdf:about="http://www.w3.org/2004/02/skos/core#topConceptOf"/>
    


    <!-- 
    ///////////////////////////////////////////////////////////////////////////////////////
    //
    // Datatypes
    //
    ///////////////////////////////////////////////////////////////////////////////////////
     -->

    


    <!-- http://www.w3.org/2001/XMLSchema#date -->

    <rdfs:Datatype rdf:about="http://www.w3.org/2001/XMLSchema#date"/>
    


    <!-- 
    ///////////////////////////////////////////////////////////////////////////////////////
    //
    // Object Properties
    //
    ///////////////////////////////////////////////////////////////////////////////////////
     -->

    


    <!-- http://data.europa.eu/eli/ontology#amended_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#amended_by">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#changed_by"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#amends"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <rdfs:comment xml:lang="en">Inverse of &quot;amends&quot;. Indicates a work that introduced legal changes in this resource. For modifications that don’t have a legal impact, use eli:corrected_by.</rdfs:comment>
        <rdfs:label xml:lang="en">amended by</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : broaden to Work or Expression</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#amends -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#amends">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#changes"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <rdfs:comment xml:lang="en">Indicates that this work introduces legal changes in another resource. For modifications that don’t have a legal impact, use eli:corrects.</rdfs:comment>
        <rdfs:label xml:lang="en">amends</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modfiied in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : broaden to Work or Expression</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#applied_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#applied_by">
        <owl:equivalentProperty rdf:resource="http://data.europa.eu/eli/ontology#implemented_by"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#applies"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Inverse of &quot;applies&quot;.

Note that this property is expressed on a legal resource, not on one of its language-specific legal expression.</rdfs:comment>
        <rdfs:label xml:lang="en">applied by</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#transposed_by"/>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#applies -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#applies">
        <owl:equivalentProperty rdf:resource="http://data.europa.eu/eli/ontology#implements"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates that this legislation (or part of a legislation) somehow conforms with another legislation. This is an informative link, and it has no legal value. For legally-binding links of transposition, use the property transposes. This can be used for example : 
  - when a pre-existing law already conforms to a recent european directive (in that case it does not &quot;transposes&quot; it strictly speaking);
  - when non-EU member states make sure their legislation is conformant with EU law without strictly speaking transposing it;
  - when a legislation from a local authority conforms with a national legislation;

Note that this should point to a LegalResource, not to a language-specific expression.</rdfs:comment>
        <rdfs:label xml:lang="en">applies</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#transposes"/>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#based_on -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#based_on">
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#basis_for"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Inverse of &quot;basis_for&quot;. Indicates that thiswork is empowered by another one, typically a constitution, a treaty or an enabling act.</rdfs:comment>
        <rdfs:label xml:lang="en">based on</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : broaden to Work or Expression
v1.1 : improved definition</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#basis_for -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#basis_for">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates that this work or expression empowers another . Typically primary legislation is the basis for secondary legislation.</rdfs:comment>
        <rdfs:label xml:lang="en">basis for</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : broaden to Work or Expression
v1.1 : improved definition</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#changed_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#changed_by">
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#changes"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Inverse of « changes ». Indicates that this work or expression is being legally changed by another. This encompasses the notions of amendment, replacement, repeal, or other types of change.</rdfs:comment>
        <rdfs:label xml:lang="en">changed by</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : broaden to Work or Expression
v1.1 : improved definition</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#changes -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#changes">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates that this work or expression legally changes another. This encompasses the notions of amendment, replacement, repeal, or other types of change. This may be a direct change (textual or non-textual amendment) or a consequential or indirect change. Note, the property is to be used to express the existence of a change relationship between two acts rather than the existence of a consolidated version of the text that shows the result of the change. For consolidation relationships, use the &quot;consolidates&quot; and &quot;consolidated_by&quot; properties.</rdfs:comment>
        <rdfs:label xml:lang="en">changes</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : broaden to Work or Expression
v1.1 : improved definition</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#cited_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#cited_by">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#is_referred_to_by"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#cites"/>
        <rdfs:domain rdf:resource="http://www.w3.org/2002/07/owl#Thing"/>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Inverse of &quot;cites&quot;.
Note that the intended meaning of this link is to indicate that &quot;something is cited by a legislation&quot; and not that &quot;this legislation is cited by something&quot;.</rdfs:comment>
        <rdfs:label xml:lang="en">cited by</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.5</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.5 : was made a subproperty of eli:is_referred_to_by
v1.2 : improved definition</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#cited_by_case_law -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#cited_by_case_law">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#is_referred_to_by"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2002/07/owl#Thing"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates that this LegalResource or LegalExpression is being cited in a case law, identified by a suitable URI. If the case law cannot be identified by a suitable URI, the property &quot;eli:cited_by_case_law_reference&quot; can be used with a textual reference to the case law.

The actual citation link is expressed from the case law to the legislation, but legal portals may use the link from a legislation to a case law to e.g. refer to representative case laws about a legislation.</rdfs:comment>
        <rdfs:label xml:lang="en">cited by case law</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#cited_by_case_law_reference"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.5</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.5 : was made a subproperty of eli:is_referred_to_by</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#cites -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#cites">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#refers_to"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2002/07/owl#Thing"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Citation in the text of the legislation. This may be at the legal resource or legal expression level, as required by the implementation context. This includes verbatim citation and citations in referrals.</rdfs:comment>
        <rdfs:label xml:lang="en">cites</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.5</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.5 : was made a subproperty of eli:refers_to</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#commenced_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#commenced_by">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#changed_by"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#commences"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <rdfs:comment xml:lang="en">Inverse of &quot;commences&quot;. Indicates that this legal resource was set in force by another legal resource.
Situations where a resource enters into force because of more than one resource are explicitely allowed.</rdfs:comment>
        <rdfs:label xml:lang="en">commenced by</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#commences -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#commences">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#changes"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <rdfs:comment xml:lang="en">Indicates that this legal resource sets another legal resource into force.
Note the the date of entry into force of the other resource should be modified accordingly.
Note also that it is not possible to indicate when the entry into force should happen.</rdfs:comment>
        <rdfs:label xml:lang="en">commences</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#consolidated_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#consolidated_by">
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#consolidates"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Inverse of &quot;consolidates&quot;. Indicates that this legal resource or expression is taken into account in a consolidated text (which is usually the product of an editorial process that revises the legislation).</rdfs:comment>
        <rdfs:label xml:lang="en">consolidated by</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : improved definition</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#consolidates -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#consolidates">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates that this consolidated legal resource or expression (which is usually the product of an editorial process that revises the legislation) takes into account another one. This property should be used multiple times to refer to both the original version or the previous consolidated version, and to the legislations making the change.</rdfs:comment>
        <rdfs:label xml:lang="en">consolidates</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : improved definition</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#corrected_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#corrected_by">
        <rdfs:subPropertyOf rdf:resource="http://www.w3.org/2002/07/owl#topObjectProperty"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#corrects"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Manifestation"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <rdfs:comment xml:lang="en">Inverse of &quot;corrects&quot;. Indicates a resource that introduces textual modifications (like correction of spelling mistakes) with no legal change in this work, expression or manifestation; typically corrigenda in EU legislation. For modifications that have a legal impact, use eli:amended_by.</rdfs:comment>
        <rdfs:label xml:lang="en">corrected by</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : broaden to Work or Expression</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#corrects -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#corrects">
        <rdfs:subPropertyOf rdf:resource="http://www.w3.org/2002/07/owl#topObjectProperty"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Manifestation"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <rdfs:comment xml:lang="en">Indicates that this work introduces textual modifications (like correction of spelling mistakes) with no legal change in another resource, expression or manifestation; typically corrigenda in EU legislation. For modifications that have a legal impact, use eli:amends.</rdfs:comment>
        <rdfs:label xml:lang="en">corrects</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : broaden to Work or Expression</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#countersigned_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#countersigned_by">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Agent"/>
        <rdfs:comment xml:lang="en">A person or organization that countersigned the legislation. Depending on the legal context, a countersignature can indicate that the signed authority undertakes to assume responsibility for texts emanating from a person who is inviolable and irresponsible, (for example a King, Grand Duc or President), or that the authority is in charge of the implementation of the text.</rdfs:comment>
        <rdfs:label xml:lang="en">countersigned by</rdfs:label>
        <owl:versionInfo>created in v1.3</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#embodies -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#embodies">
        <rdfs:subPropertyOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/R4i_comprises_carriers_of"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#is_embodied_by"/>
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Manifestation"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Expression"/>
        <rdfs:comment xml:lang="en">Relates a manifestation to the expression that it embodies. Inverse of &quot;is_embodied_by&quot;.</rdfs:comment>
        <rdfs:label xml:lang="en">embodies</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in 1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.2 : replaced reference of RDA with FRBRoo.</skos:historyNote>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Manifestation instead of Format and the range to Expression instead of LegalExpression.
In v1.1, this was declared functional (the constraint was missing in 1.0)</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#ensures_implementation_of -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#ensures_implementation_of">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#applies"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#implementation_ensured_by"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:comment xml:lang="en">Indicates that this LegalResource ensures the implementation of another LegalResource. This implies a legal meaning (contrary to eli:applies).

This can cover links from national legislation to EU legislation, or links from regional to national legislation. It can also cover links from EU implementing acts to EU legislative acts. In the case of implementation of EU legislation at national level, this covers links to EU regulations, decisions, etc. However the transpositions of EU Directives must be captured with eli:transposes.

Links between national primary and secondary legislation must be captured by eli:based_on / eli:basis_for.</rdfs:comment>
        <rdfs:label xml:lang="en">ensures implementation of</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#transposes"/>
        <owl:versionInfo>created in v1.3</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#format -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#format">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Manifestation"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/format"/>
        <rdfs:comment xml:lang="en">The format of the manifestation, expressed as a custom URI.
This field is intended to capture the format of the resource from an application or user perspective, as opposed to the &quot;media_type&quot; property that expresses its format from a technical point of view. This property allows to describe different XML schemas (Akoma N&apos;toso vs. proprietary), describe different types of PDFs (scanned PDF, generated-on-the-fly PDF, signed PDF, archival PDF) or describe the paper (printed) version of the legislation.
ELI includes a set of possible values for the most common use-cases and possible URIs values CAN also be taken from http://www.iana.org/assignments/media-types , or can de defined by the Member States.</rdfs:comment>
        <rdfs:label xml:lang="en">format</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#media_type"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Manifestation instead of Format.
v1.2 : this field should now not be used as basis of content negociation. Modified the description to introduce eli:media_type and declared a local ConceptScheme FormatType with values.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#has_annex -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#has_annex">
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#is_annex_of"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates an annex to this this work or expression .</rdfs:comment>
        <rdfs:label xml:lang="en">has annex</rdfs:label>
        <skos:historyNote xml:lang="en">created in v1.4</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#has_another_publication -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#has_another_publication">
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#is_another_publication_of"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Inverse of &quot;is_another_publication_of&quot;.</rdfs:comment>
        <rdfs:label xml:lang="en">has another publication</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#has_derivative -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#has_derivative">
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#is_derivative_of"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Inverse of is_derivative_of</rdfs:comment>
        <rdfs:label xml:lang="en">has derivative</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.4</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#has_member -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#has_member">
        <rdfs:subPropertyOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/R10_has_member"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#is_member_of"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/hasPart"/>
        <rdfs:comment xml:lang="en">Indicates that this work conceptually includes another one. For the notion of physical inclusion, use eli:has_part.</rdfs:comment>
        <rdfs:label xml:lang="en">has member</rdfs:label>
        <rdfs:seeAlso>modified in v1.2</rdfs:seeAlso>
        <owl:versionInfo>created in v1.1</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain and range to Work instead of LegalResource
v1.2 : replaced reference of RDA with FRBRoo.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#has_part -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#has_part">
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#is_part_of"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/hasPart"/>
        <rdfs:comment xml:lang="en">inverse of &quot;is_part_of&quot;</rdfs:comment>
        <rdfs:label xml:lang="en">has part</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain and range to Work instead of LegalResource</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#has_translation -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#has_translation">
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#is_translation_of"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalExpression"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#LegalExpression"/>
        <rdfs:comment xml:lang="en">Inverse of &quot;is_translation_of&quot;. Indicates that this expression has been translated into another derived expression. See the definition of &quot;is_translation_of&quot;.</rdfs:comment>
        <rdfs:label xml:lang="en">has translation</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#implementation_ensured_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#implementation_ensured_by">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#applied_by"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <rdfs:comment xml:lang="en">Indicates that the implementation of this LegalResource is ensured by another LegalResource. This implies a legal meaning (contrary to eli:applies). See the definition of eli:ensures_implementation_of.</rdfs:comment>
        <rdfs:label xml:lang="en">implementation ensured by</rdfs:label>
        <owl:versionInfo>created in v1.3</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#implemented_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#implemented_by">
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">This property is deprecated. Use &quot;applied_by&quot; instead.</rdfs:comment>
        <rdfs:label xml:lang="en">implemented by</rdfs:label>
        <owl:deprecated rdf:datatype="http://www.w3.org/2001/XMLSchema#boolean">true</owl:deprecated>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : this property was deprecated in favor of the more generic term &quot;applied_by&quot; to avoid confusion</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#implements -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#implements">
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">This property is deprecated. Use &quot;applies&quot; instead.</rdfs:comment>
        <rdfs:label xml:lang="en">implements</rdfs:label>
        <owl:deprecated rdf:datatype="http://www.w3.org/2001/XMLSchema#boolean">true</owl:deprecated>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : this property was deprecated in favor of the more generic term &quot;applies&quot; to avoid confusion.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#in_force -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#in_force">
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#InForce"/>
        <originalName xml:lang="en">status</originalName>
        <rdfs:comment xml:lang="en">A value indicating the legal force of a legal resource or a legal expression. A set of values is defined by ELI in the corresponding concept scheme. These values are :
  - in force
  - partially in force
  - not in force</rdfs:comment>
        <rdfs:label xml:lang="en">in force</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#InForce-inForce"/>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#InForce-notInForce"/>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#InForce-partiallyInForce"/>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#InForceTable"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_about -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_about">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:range rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/subject"/>
        <rdfs:comment xml:lang="en">A subject for this work. The use of Eurovoc (http://eurovoc.europa.eu) is encouraged to select values for this property. Member states are encouraged to align local values to Eurovoc.</rdfs:comment>
        <rdfs:label xml:lang="en">is about</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : broaden domain to Work</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_annex_of -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_annex_of">
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates this work or expression is an annex of another one.</rdfs:comment>
        <rdfs:label xml:lang="en">is annex of</rdfs:label>
        <skos:historyNote xml:lang="en">created in v1.4</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_another_publication_of -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_another_publication_of">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates that this resource is a new publication, in a different official journal, of another resource already published elsewhere, and cannot be considered to be the same resource (owl:sameAs cannot be used to avoid potential duplication of certain metadata, like the date of publication, or the publisher).

Note that this is different from the exceptionnal cases of &quot;republication&quot;, where the same resource is actually republished in the same official journal a few days after its initial publication, in case of errors.</rdfs:comment>
        <rdfs:label xml:lang="en">is another publication of</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_derivative_of -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_derivative_of">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">A Work or Expression from which this one derives</rdfs:comment>
        <rdfs:label xml:lang="en">is derivative of</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.4</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_embodied_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_embodied_by">
        <rdfs:subPropertyOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/R4_carriers_provided_by"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Expression"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Manifestation"/>
        <rdfs:comment xml:lang="en">Relates an expression to a manifestation of that expression. Inverse of &quot;embodies&quot;.</rdfs:comment>
        <rdfs:label xml:lang="en">is embodied by</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Expression instead of LegalExpression and the range to Manifestation instead of Format.
v1.2 : replaced reference of RDA with FRBRoo.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_exemplified_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_exemplified_by">
        <rdfs:subPropertyOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/R7i_has_example"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Manifestation"/>
        <rdfs:comment xml:lang="en">Link to a concrete file URL.
Relates a manifestation to a single exemplar or instance of that manifestation.</rdfs:comment>
        <rdfs:label xml:lang="en">is exemplified by</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Manifestation instead of Format.
v1.2 : replaced reference of RDA with FRBRoo.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_member_of -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_member_of">
        <rdfs:subPropertyOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/R10i_is_member_of"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/isPartOf"/>
        <rdfs:comment xml:lang="en">Indicates that this work is conceptually included in another one. In the case of a legislation, its successive temporal versions are conceptually members of a single « abstract » resource. For the notion of physical inclusion, use eli:is_part_of.</rdfs:comment>
        <rdfs:label xml:lang="en">is member of</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain and range to Work instead of LegalResource
v1.2 : replaced reference of RDA with FRBRoo.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_part_of -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_part_of">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <originalName xml:lang="en">composed</originalName>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/isPartOf"/>
        <rdfs:comment xml:lang="en">Indicates a work in which this one is physically included. Covers the case of text included in an Official Journal, or an article included in a text. For the notion of conceptual/temporal inclusion, use eli:is_member_of.</rdfs:comment>
        <rdfs:label xml:lang="en">is part of</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain and range to Work instead of LegalResource
Starting with v1.1, the notion of conceptual inclusion is expressed with eli:is_member_of instead of eli:is_part_of.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_realized_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_realized_by">
        <rdfs:subPropertyOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/R3_is_realised_in"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#realizes"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Expression"/>
        <rdfs:comment xml:lang="en">Relates a work to an expression of this work in the form of a &quot;sequence of signs&quot; (typically alpha-numeric characters in a legal context). Inverse of &quot;realizes&quot;.</rdfs:comment>
        <rdfs:label xml:lang="en">is realized by</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Work instead of LegalResource and range to Expression instead of LegalExpression.
v1.2 : replaced reference of RDA with FRBRoo.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_referred_to_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_referred_to_by">
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#refers_to"/>
        <rdfs:domain rdf:resource="http://www.w3.org/2002/07/owl#Thing"/>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates a work or expression that refers to this entity.</rdfs:comment>
        <rdfs:label xml:lang="en">is referred to by</rdfs:label>
        <owl:versionInfo>created in v1.5</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#is_translation_of -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#is_translation_of">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalExpression"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#LegalExpression"/>
        <rdfs:comment xml:lang="en">Indicates that this expression has been translated from another original expression; this can be used to distinguish original from derived expressions.

Note that asserting this link does not have any implication on the legal value of the original and translated expressions : depending on the context, a translation can be as legally binding as the original version, or can be published for informative purposes only (e.g. a country translating some legal acts in English).

The expressions linked with this property can be realisations of the same LegalResource, or different LegalResources. Multilingual legislations that do not need to distinguish between original and derived expressions of the same LegalResource (e.g. european legislation) will not use this property.</rdfs:comment>
        <rdfs:label xml:lang="en">is translation of</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#jurisdiction -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#jurisdiction">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#relevant_for"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#AdministrativeArea"/>
        <rdfs:comment xml:lang="en">The jurisdiction from which the legal resource originates. 

The place identifier can be taken from the Administrative Territorial Unit table published of the EU Publications Office at https://op.europa.eu/en/web/eu-vocabularies/at-dataset/-/resource/dataset/atu. Member States don&apos;t have to recreate their own list of values.</rdfs:comment>
        <rdfs:label xml:lang="en">jurisdiction</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : Updated link to EU vocabularies website in the definition.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#language -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#language">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Expression"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Language"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/language"/>
        <rdfs:comment xml:lang="en">The language of an expression.

EU Publications Office provides a list of languages at https://op.europa.eu/en/web/eu-vocabularies/at-dataset/-/resource/dataset/language. This list is large enough so that member states should not have to declare local values.

Note that, if needed, a language can also be stated on a legal resource using the DublinCore &quot;language&quot; property.</rdfs:comment>
        <rdfs:label xml:lang="en">language</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://publications.europa.eu/resource/authority/language"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Expression instead of LegalExpression and updated link to EU vocabularies website</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#legal_value -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#legal_value">
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Format"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#LegalValue"/>
        <rdfs:comment xml:lang="en">The legal value associated with a specific format of a resource. A set of values is defined by ELI in the corresponding concept scheme. These values are :
  - unofficial : no particular or special standing;
  - official : published by an organisation with the public task of making the information available (e.g. a consolidated version of a EU directive) ;
  - authoritative : the publisher gives some special status to the publication (e.g. &quot;the Queens Printer&quot; version of an Act of Parliament, or the OJ version of a EU Directive);
  - definitive : the text is conclusively what the law says, (e.g. the digitally signed version of an OJ).</rdfs:comment>
        <rdfs:label xml:lang="en">legal value</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#LegalValue-authoritative"/>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#LegalValue-definitive"/>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#LegalValue-official"/>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#LegalValue-unofficial"/>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#LegalValueTable"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#licence -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#licence">
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Format"/>
        <rdfs:range rdf:resource="http://www.w3.org/2002/07/owl#Thing"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/license"/>
        <rdfs:comment xml:lang="en">A legal document giving official permission to do something with the resource (Definition from Dublin Core)</rdfs:comment>
        <rdfs:label xml:lang="en">licence</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#media_type -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#media_type">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Manifestation"/>
        <rdfs:range rdf:resource="http://www.w3.org/2002/07/owl#Thing"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/format"/>
        <rdfs:comment xml:lang="en">The file format of the manifestation.
This field is intended to capture the technical file format and will serve as a basis for content negotiation for the server to return the appropriate file based on the client preference.
Although not mandatory, this property is highly encouraged.
Possible URIs values MUST be taken from http://www.iana.org/assignments/media-types (e.g. http://www.iana.org/assignments/media-types/application/xml).
See also the &quot;format&quot; property.</rdfs:comment>
        <rdfs:label xml:lang="en">media type</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#format"/>
        <owl:versionInfo>created in v1.2</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Manifestation instead of Format</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#passed_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#passed_by">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Agent"/>
        <originalName xml:lang="en">agent_document</originalName>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/creator"/>
        <rdfs:comment xml:lang="en">The person or organization that originally passed or made the law : typically parliament (for primary legislation) or government (for secondary legislation). This indicates the &quot;legal author&quot; of the law, as opposed to its physical author. 

This property can be used multiple times to indicate both the agent that authored the text, and/or the agent that signed the text, thus turning it into an actual legislation.

The relationship between current and any former law making body should be represented in the description of the agent itself.

Member states are encouraged to make their own list of Agents. EU Publications Office provides a list of corporate bodies at https://op.europa.eu/en/web/eu-vocabularies/at-dataset/-/resource/dataset/corporate-body.</rdfs:comment>
        <rdfs:label xml:lang="en">passed by</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : Updated link to EU vocabularies website in the definition.
In v1.1, the semantic of this property was broaden to encompass organisations/persons that « drafted the text » and « signed it ».</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#published_in_format -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#published_in_format">
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#publishes"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Format"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Format"/>
        <rdfs:comment xml:lang="en">Reference to the Official Journal or other publication manifestation in which this format is published.
This property should be used when the value can be identified by a suitable URI; in the absence of such a URI, the property &quot;published_in&quot; should be used with a string value.</rdfs:comment>
        <rdfs:label xml:lang="en">published in format</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#published_in"/>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#publisher_agent -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#publisher_agent">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Format"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Agent"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/publisher"/>
        <rdfs:comment xml:lang="en">An entity responsible for making the resource available (definition from Dublin Core).
This property should be used when the value can be identified by a suitable URI; in the absence of such a URI, the property &quot;publisher&quot; should be used with a string value.</rdfs:comment>
        <rdfs:label xml:lang="en">publisher agent</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#publisher"/>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#publishes -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#publishes">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Format"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Format"/>
        <rdfs:comment xml:lang="en">Inverse of &quot;published_in_format&quot;. Note this property does not link a publisher with a resource, but rather a specific Format of a resource with a specific Format of another resource, indicating that the subject Format publishes the object Format.</rdfs:comment>
        <rdfs:label xml:lang="en">publishes</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#realizes -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#realizes">
        <rdfs:subPropertyOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/R3i_realises"/>
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Expression"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:comment xml:lang="en">Relates an expression to a work realised through that expression. Inverse of &quot;is_realized_by&quot;.</rdfs:comment>
        <rdfs:label xml:lang="en">realizes</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.2</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Expression instead of LegalExpression and the range to Work instead of LegalResource
v1.2 : replaced reference of RDA with FRBRoo.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#refers_to -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#refers_to">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2002/07/owl#Thing"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Any entity that this work or expression refers to ; typically references are made to other Works, but it can be also to legislative processes, persons, etc.

Note that specific subproperties exist in ELI-DL to describe future legal impacts (\&quot;foresees_xxxx\&quot; links) and links between amendments and draft legislation.

Example : an opinion refers to the specific version of the draft legislation on which it is based.</rdfs:comment>
        <rdfs:label xml:lang="en">refers to</rdfs:label>
        <owl:versionInfo>created in v1.5</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#related_to -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#related_to">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2002/07/owl#Thing"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates a somehow related other document, not necessarily a legal resource. Note that citation links should use the cites property.</rdfs:comment>
        <rdfs:label xml:lang="en">related to</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#relevant_for -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#relevant_for">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#AdministrativeArea"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/coverage"/>
        <rdfs:comment xml:lang="en">Refers to a place or an area associated with the resource. This covers the notions of jurisdiction, sovereignty, applicability or administrative area. The place identifier can be taken from the Administrative Territorial Unit table published of the EU Publications Office at https://op.europa.eu/en/web/eu-vocabularies/at-dataset/-/resource/dataset/atu. Member States don&apos;t have to recreate their own list of values.

The group notes the limitations of what can be said with a single property; member states can refine this notion by declaring specific sub properties.</rdfs:comment>
        <rdfs:label xml:lang="en">relevant for</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : Updated link to EU vocabularies website in the definition.
Starting from v1.1, the values for this property can be taken from the named authority list published by the Office of Publications instead of defining new values.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#repealed_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#repealed_by">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#changed_by"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#repeals"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <rdfs:comment xml:lang="en">Inverse of &quot;repeals&quot;. Indicates that this legal resource or legal expression is being completely canceled, abrogated or replaced by another legal resource. If a resource is partially repealed by another, this link can be used at the corresponding subdivision level being completely repealed.</rdfs:comment>
        <rdfs:label xml:lang="en">repealed by</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#repeals -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#repeals">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#changes"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <rdfs:comment xml:lang="en">Indicates that this legal resource or legal expression completely cancels, abrogates or replaces another. If a resource partially repeals another, this link can be used at the corresponding subdivision level being completely repealed.</rdfs:comment>
        <rdfs:label xml:lang="en">repeals</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#responsibility_of_agent -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#responsibility_of_agent">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Agent"/>
        <rdfs:comment xml:lang="en">An individual, organisation or organisational unit that has some kind of responsibility for the legislation. Typically the ministry who is/was in charge of elaborating the legislation, or the adressee for potential questions about the legislation once it is published.
This property should be used when the value can be identified by a suitable URI; in the absence of such a URI, the property &quot;responsibility_of&quot; should be used with a string value.</rdfs:comment>
        <rdfs:label xml:lang="en">responsibility of agent</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#responsibility_of"/>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#rightsholder_agent -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#rightsholder_agent">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Format"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Agent"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/rightsholder"/>
        <rdfs:comment xml:lang="en">A person or organisation owning or managing rights over the resource (definition from Dublin Core).
This property should be used when the value can be identified by a suitable URI; in the absence of such a URI, the property &quot;rightsholder&quot; should be used with a string value.</rdfs:comment>
        <rdfs:label xml:lang="en">rightsholder agent</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#rightsholder"/>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#transposed_by -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#transposed_by">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#applied_by"/>
        <owl:inverseOf rdf:resource="http://data.europa.eu/eli/ontology#transposes"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:range>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:range>
        <rdfs:comment xml:lang="en">Inverse of &quot;transposes&quot;.

Note that this property is expressed on a legal resource, not on one of its language-specific legal expression.</rdfs:comment>
        <rdfs:label xml:lang="en">transposed by</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : this property was made a subproperty of &quot;applied_by&quot;</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#transposes -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#transposes">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#applies"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:comment xml:lang="en">Indicates that this legislation (or part of legislation) fulfills the objectives set by another legislation, by passing appropriate implementation measures. Typically, some legislations of European Union&apos;s member states or regions transpose European Directives. This indicates a legally binding link between the 2 legislations, at act or article level, from the original version of a national implementing measure to the legal resource Directive as published in the EU Official Journal. Can be used for transposition tables, once EU Publication Office has introduced ELI support down to the article level.

The implementation of EU legislation at national level, involving links to EU regulations, decisions, etc. must be captured with eli:ensures_implementation_of.

Note that this should point to the legal resource of the Directive itself, not to one of its language-specific legal expression.</rdfs:comment>
        <rdfs:label xml:lang="en">transposes</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#ensures_implementation_of"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : this property was made a subproperty of &quot;applies&quot;, and the definition was improved.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#type_document -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#type_document">
        <rdfs:subPropertyOf rdf:resource="http://data.europa.eu/eli/ontology#work_type"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#ResourceType"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/type"/>
        <rdfs:comment xml:lang="en">The type of a legal resource (e.g. &quot;Directive&quot;, &quot;Règlement grand ducal&quot;, &quot;law&quot;, &quot;règlement ministériel&quot;, &quot;draft proposition&quot;, &quot;Parliamentary act&quot;, etc.). 

Member states are encouraged to make their own list of values in the corresponding concept scheme. EU Publications Office provides a list of values for EU resource types at https://op.europa.eu/en/web/eu-vocabularies/at-dataset/-/resource/dataset/resource-type</rdfs:comment>
        <rdfs:label xml:lang="en">type of document</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#ResourceTypeTable"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : Moved under work_type
v1.3 : Updated link to EU vocabularies website in the definition.</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#type_subdivision -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#type_subdivision">
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#WorkSubdivision"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#SubdivisionType"/>
        <rdfs:comment xml:lang="en">The type of a document subdivision (e.g. &quot;Article&quot;, &quot;Paragraph&quot;, &quot;Section&quot;, etc.).
A subdivision can have only one type.

ELI does not specify a list of possible values. Member states are encouraged to make their own list of values in the corresponding concept scheme. EU Publication Office provies a list of values for EU resource types at https://op.europa.eu/en/web/eu-vocabularies/at-dataset/-/resource/dataset/subdivision</rdfs:comment>
        <rdfs:label xml:lang="en">type of subdivision</rdfs:label>
        <owl:versionInfo>created in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : Broaden range to WorkSubdivision</skos:historyNote>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#uri_schema -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#uri_schema">
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Format"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2002/07/owl#Thing"/>
        <rdfs:comment xml:lang="en">Schema describing the URI of an ELI instance. ELI uses URI template specifications (IETF RFC 6570). Schemes should be associated with member states and will be published in a registry.</rdfs:comment>
        <rdfs:label xml:lang="en">uri schema</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#version -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#version">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#Version"/>
        <rdfs:comment xml:lang="en">A skos concept scheme, could be locally defined? Group proposal is to start with an initial ELI scheme, that might include concepts of &quot;Official Journal&quot; &quot;made&quot; &quot;consolidated&quot; &quot;proposed&quot; &quot;prospective&quot;</rdfs:comment>
        <rdfs:label xml:lang="en">version</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#version_date"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:ObjectProperty>
    


    <!-- http://data.europa.eu/eli/ontology#work_type -->

    <owl:ObjectProperty rdf:about="http://data.europa.eu/eli/ontology#work_type">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:range rdf:resource="http://data.europa.eu/eli/ontology#WorkType"/>
        <rdfs:comment xml:lang="en">The type of a work, taken from a controlled vocabulary.

Member States need to define their own work type values.</rdfs:comment>
        <rdfs:label xml:lang="en">work type</rdfs:label>
    </owl:ObjectProperty>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/R10_has_member -->

    <owl:ObjectProperty rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/R10_has_member"/>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/R10i_is_member_of -->

    <owl:ObjectProperty rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/R10i_is_member_of"/>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/R3_is_realised_in -->

    <owl:ObjectProperty rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/R3_is_realised_in"/>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/R3i_realises -->

    <owl:ObjectProperty rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/R3i_realises"/>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/R4_carriers_provided_by -->

    <owl:ObjectProperty rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/R4_carriers_provided_by"/>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/R4i_comprises_carriers_of -->

    <owl:ObjectProperty rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/R4i_comprises_carriers_of"/>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/R7i_has_example -->

    <owl:ObjectProperty rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/R7i_has_example"/>
    


    <!-- 
    ///////////////////////////////////////////////////////////////////////////////////////
    //
    // Data properties
    //
    ///////////////////////////////////////////////////////////////////////////////////////
     -->

    


    <!-- http://data.europa.eu/eli/ontology#cited_by_case_law_reference -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#cited_by_case_law_reference">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/relation"/>
        <rdfs:comment xml:lang="en">Indicates that this LegalResource or LegalExpression is being cited in a case law that cannot be identified by a suitable URI and that is indicated by textual reference. An ECLI (European Case Law Identifier) can be used here.
When the case law can be identified by a suitable URI, the property eli:cited_by_case_law should be used instead.</rdfs:comment>
        <rdfs:label xml:lang="en">cited by case law reference</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#cited_by_case_law"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#date_applicability -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#date_applicability">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2001/XMLSchema#date"/>
        <rdfs:comment xml:lang="en">The date at which the legislation becomes applicable. This is distinct from the date of entry into force : a text may come in force today, and state it will become applicable in 3 months.

The group notes that applicability dates can lead to complex situations, e.g. a text with different applicability dates for different jurisdictions; specific extensions to the model should be used for such situations.</rdfs:comment>
        <rdfs:label xml:lang="en">date of applicability</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#date_document -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#date_document">
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:range rdf:resource="http://www.w3.org/2001/XMLSchema#date"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/date"/>
        <rdfs:comment xml:lang="en">Date of adoption or signature (of the form yyyy-mm-dd)</rdfs:comment>
        <rdfs:label xml:lang="en">date of the document</rdfs:label>
        <owl:versionInfo>created in v1.0
modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : broaden domain to Work</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#date_no_longer_in_force -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#date_no_longer_in_force">
        <rdfs:subPropertyOf rdf:resource="http://www.w3.org/2002/07/owl#topDataProperty"/>
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2001/XMLSchema#date"/>
        <rdfs:comment xml:lang="en">The last date any part of the legislation is in force, if the date is known (can be seen as the end date of a dc:valid range for this resource).</rdfs:comment>
        <rdfs:label xml:lang="en">date no longer in force</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#first_date_entry_in_force"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.2</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : improved definition so it symmetric to date_entry_into_force.</skos:historyNote>
        <skos:historyNote xml:lang="en">v1.2 : improved definition</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#date_publication -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#date_publication">
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2001/XMLSchema#date"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/issued"/>
        <rdfs:comment xml:lang="en">Date of publication of the official version of the legislation, in hard copy or online, depending on what the official publication is, and when it was published. Publication dates at the level of legal expressions can be separately asserted, using standard Dublin Core properties.</rdfs:comment>
        <rdfs:label xml:lang="en">date of publication</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:historyNote xml:lang="en">In v1.1, the domain was broaden to add LegalExpression.</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#description -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#description">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2000/01/rdf-schema#Literal"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/description"/>
        <rdfs:comment xml:lang="en">An account of the resource (definition from Dubin Core), e.g a summary.</rdfs:comment>
        <rdfs:label xml:lang="en">description</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#first_date_entry_in_force -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#first_date_entry_in_force">
        <rdfs:subPropertyOf rdf:resource="http://www.w3.org/2002/07/owl#topDataProperty"/>
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2001/XMLSchema#date"/>
        <rdfs:comment xml:lang="en">The first date any part of the legal resource or legal expression came into force (can be seen as the start date of a dc:valid range for this resource)</rdfs:comment>
        <rdfs:label xml:lang="en">first date of entry into force</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#date_no_longer_in_force"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#id_local -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#id_local">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Format"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range>
            <rdfs:Datatype>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://www.w3.org/2001/XMLSchema#anyURI"/>
                    <rdf:Description rdf:about="http://www.w3.org/2001/XMLSchema#string"/>
                </owl:unionOf>
            </rdfs:Datatype>
        </rdfs:range>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/identifier"/>
        <rdfs:comment xml:lang="en">The unique identifier used in a local reference system to maintain backwards compatibility. For examples the CELEX at EU level, or the NOR in France.</rdfs:comment>
        <rdfs:label xml:lang="en">local id</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#number -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#number">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Expression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Work"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2001/XMLSchema#string"/>
        <rdfs:comment xml:lang="en">An identifier or other disambiguating feature for a work or expression. This can be the number of a legislation, the number of an article, or the issue number of an official journal.</rdfs:comment>
        <rdfs:label xml:lang="en">number</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.5</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.5 : broaden the domain to Work or Expression instead of LegalResource or LegalExpression.</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#published_in -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#published_in">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Format"/>
        <rdfs:range rdf:resource="http://www.w3.org/2000/01/rdf-schema#Literal"/>
        <rdfs:comment xml:lang="en">Reference to the Official Journal or other publication manifestation in which this format is published.
This property should be used when the value cannot be identified by a suitable URI; if a URI is available, the property &quot;published_in_format&quot; should be used.</rdfs:comment>
        <rdfs:label xml:lang="en">published in</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#published_in_format"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.2</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : modified the range to xsd:string, and introduced the corresponding object property published_in_format.</skos:historyNote>
        <skos:historyNote xml:lang="en">v1.2 : changed range from xsd:string to rdfs:Literal</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#publisher -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#publisher">
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#Format"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2000/01/rdf-schema#Literal"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/publisher"/>
        <rdfs:comment xml:lang="en">An entity responsible for making the resource available (definition from Dublin Core).
This property should be used when the value cannot be identified by a suitable URI; if a URI is available, the property &quot;publisher_agent&quot; should be used.</rdfs:comment>
        <rdfs:label xml:lang="en">publisher</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#publisher_agent"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.2</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : modified the range to xsd:string and introduced the corresponding object property publisher_agent.</skos:historyNote>
        <skos:historyNote xml:lang="en">v1.2 : changed range from xsd:string to rdfs:Literal</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#responsibility_of -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#responsibility_of">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:range rdf:resource="http://www.w3.org/2001/XMLSchema#string"/>
        <originalName xml:lang="en">service</originalName>
        <rdfs:comment xml:lang="en">An individual, organisation or organisational unit that has some kind of responsibility for the legislation. Typically the ministry who is/was in charge of elaborating the legislation, or the adressee for potential questions about the legislation once it is published.
This property should be used when the value cannot be identified by a suitable URI; if a URI is available, the property &quot;responsibility_of_agent&quot; should be used.</rdfs:comment>
        <rdfs:label xml:lang="en">responsibility of</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#responsibility_of_agent"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : the definition was made more precise, with examples. Introduced the corresponding object property responsibility_of_agent.</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#rights -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#rights">
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Format"/>
        <rdfs:range rdf:resource="http://www.w3.org/2000/01/rdf-schema#Literal"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/rights"/>
        <rdfs:comment xml:lang="en">Information about rights held in and over the resource (definition from Dublin Core). For example, that property can be used to provide a link to a page that describes the licensing terms.</rdfs:comment>
        <rdfs:label xml:lang="en">rights</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.2</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : improved definition and changed the range to xsd:string</skos:historyNote>
        <skos:historyNote xml:lang="en">v1.2 : modified the range to rdfs:Literal</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#rightsholder -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#rightsholder">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Format"/>
        <rdfs:range rdf:resource="http://www.w3.org/2000/01/rdf-schema#Literal"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/rightsHolder"/>
        <rdfs:comment xml:lang="en">A person or organisation owning or managing rights over the resource (definition from Dublin Core).
This property should be used when the value cannot be identified by a suitable URI; if a URI is available, the property &quot;rightsholder_agent&quot; should be used.</rdfs:comment>
        <rdfs:label xml:lang="en">rights holder</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.2</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : modified the range to xsd:string, and introduced the corresponding object property rightsholder_agent.</skos:historyNote>
        <skos:historyNote xml:lang="en">v1.2 : modified the range to rdfs:Literal</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#title -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#title">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Expression"/>
        <rdfs:range rdf:resource="http://www.w3.org/2000/01/rdf-schema#Literal"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/title"/>
        <rdfs:comment xml:lang="en">The title, or name, of an expression. 

Note that, if needed, a title can also be stated on a legal resource using the Dublin Core &quot;title&quot; property.</rdfs:comment>
        <rdfs:label xml:lang="en">title</rdfs:label>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#title_alternative"/>
        <rdfs:seeAlso rdf:resource="http://data.europa.eu/eli/ontology#title_short"/>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Expression instead of LegalExpression
v1.2 : changed range from xsd:string to rdfs:Literal</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#title_alternative -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#title_alternative">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Expression"/>
        <rdfs:range rdf:resource="http://www.w3.org/2000/01/rdf-schema#Literal"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/alternative"/>
        <rdfs:comment xml:lang="en">An alternative title of the expression (if any).

Note that, if needed, an alternative title can also be stated on a work using the Dublin Core &quot;alternative&quot; property.</rdfs:comment>
        <rdfs:label xml:lang="en">alternative title</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Expression instead of LegalExpression
v1.2 : changed range from xsd:string to rdfs:Literal</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#title_short -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#title_short">
        <rdfs:domain rdf:resource="http://data.europa.eu/eli/ontology#Expression"/>
        <rdfs:range rdf:resource="http://www.w3.org/2000/01/rdf-schema#Literal"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/alternative"/>
        <rdfs:comment xml:lang="en">Established short title of the expression (if any)</rdfs:comment>
        <rdfs:label xml:lang="en">short title</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : broaden the domain to Expression instead of LegalExpression
v1.2 : changed range from xsd:string to rdfs:Literal</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- http://data.europa.eu/eli/ontology#version_date -->

    <owl:DatatypeProperty rdf:about="http://data.europa.eu/eli/ontology#version_date">
        <rdf:type rdf:resource="http://www.w3.org/2002/07/owl#FunctionalProperty"/>
        <rdfs:domain>
            <owl:Class>
                <owl:unionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalExpression"/>
                    <rdf:Description rdf:about="http://data.europa.eu/eli/ontology#LegalResource"/>
                </owl:unionOf>
            </owl:Class>
        </rdfs:domain>
        <rdfs:range rdf:resource="http://www.w3.org/2001/XMLSchema#date"/>
        <specificPropertyOf rdf:resource="http://purl.org/dc/terms/date"/>
        <rdfs:comment xml:lang="en">The point-in-time at which the provided description of the legislation is valid.</rdfs:comment>
        <rdfs:label xml:lang="en">version date</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.1 : improved definition</skos:historyNote>
    </owl:DatatypeProperty>
    


    <!-- 
    ///////////////////////////////////////////////////////////////////////////////////////
    //
    // Classes
    //
    ///////////////////////////////////////////////////////////////////////////////////////
     -->

    


    <!-- http://data.europa.eu/eli/ontology#AdministrativeArea -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#AdministrativeArea">
        <owl:equivalentClass>
            <owl:Class>
                <owl:intersectionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://www.w3.org/2004/02/skos/core#Concept"/>
                    <owl:Class>
                        <owl:unionOf rdf:parseType="Collection">
                            <owl:Restriction>
                                <owl:onProperty rdf:resource="http://www.w3.org/2004/02/skos/core#inScheme"/>
                                <owl:hasValue rdf:resource="http://data.europa.eu/eli/ontology#AdministrativeAreaTable"/>
                            </owl:Restriction>
                            <owl:Restriction>
                                <owl:onProperty rdf:resource="http://www.w3.org/2004/02/skos/core#inScheme"/>
                                <owl:hasValue rdf:resource="http://publications.europa.eu/resource/authority/atu"/>
                            </owl:Restriction>
                        </owl:unionOf>
                    </owl:Class>
                </owl:intersectionOf>
            </owl:Class>
        </owl:equivalentClass>
        <rdfs:subClassOf rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <rdfs:comment xml:lang="en">Formally defined as the set of skos:Concept either in concept scheme eli:AdministrativeAreaTable or in concept scheme https://op.europa.eu/en/web/eu-vocabularies/at-dataset/-/resource/dataset/atu (Administrative Territorial Unit table published of the EU Publications Office).</rdfs:comment>
        <rdfs:label xml:lang="en">Administrative Area</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : Updated the link to EU vocabularies website in the definition.
Starting from v1.1, the values for administrative areas can be taken from the Administrative Territorial Unit table published of the EU Publications Office.</skos:historyNote>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#Agent -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#Agent">
        <rdfs:subClassOf rdf:resource="http://www.cidoc-crm.org/cidoc-crm/E39_Actor"/>
        <rdfs:comment xml:lang="en">Individual people or groups of people, like an institution with a legal recognition, who have the potential to perform intentional actions for which they can be held responsible.</rdfs:comment>
        <rdfs:label xml:lang="en">Agent</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.4</owl:versionInfo>
        <skos:historyNote xml:lang="en">V1.4 : subClassOf E39_ACtor instead of equivalent.
V1.3 : added a definition.</skos:historyNote>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#ComplexWork -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#ComplexWork">
        <rdfs:subClassOf rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:comment xml:lang="en">Works that have other works as members. The members of a Complex Work may constitute alternatives to or derivatives of other members of the same Complex Work. Typically the members of the complex work represent the successive temporal versions of an act, while the complex work represents the act independantly of one of its version.</rdfs:comment>
        <rdfs:label>Complex Work</rdfs:label>
        <owl:versionInfo>created in v1.4</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#Expression -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#Expression">
        <rdfs:subClassOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/F22_Self-Contained_Expression"/>
        <rdfs:comment xml:lang="en">The intellectual realisation of a Work in the form of identifiable immaterial objects, such as, but not limited to, text.

The substance of Expression is signs.

An Expression cannot exist without a physical carrier but do not depend on a specific physical carrier and can exist on one or more carriers simultaneously.</rdfs:comment>
        <rdfs:label xml:lang="en">Expression</rdfs:label>
        <owl:versionInfo>created in v1.3</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#Format -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#Format">
        <rdfs:subClassOf rdf:resource="http://data.europa.eu/eli/ontology#Manifestation"/>
        <rdfs:subClassOf>
            <owl:Restriction>
                <owl:onProperty rdf:resource="http://data.europa.eu/eli/ontology#format"/>
                <owl:cardinality rdf:datatype="http://www.w3.org/2001/XMLSchema#nonNegativeInteger">1</owl:cardinality>
            </owl:Restriction>
        </rdfs:subClassOf>
        <rdfs:comment xml:lang="en">The physical embodiment of a legal expression, either on paper or in any electronic format.
For example, any electronic or physical format of the legal expression (XML, TIFF, PDF, etc.) ; e.g. PDF version of act 3 of 2005.  (adapted from Akoma Ntoso).
Note that the notion of Format in ELI is more precise than a FRBR Manifestation in the sense that a Format in ELI can carry a legal value (see the property eli:legal_value).</rdfs:comment>
        <rdfs:label xml:lang="en">Format</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : declared as a sublcass of eli:Manifestation
v1.2 : replaced reference of RDA with FRBRoo (for more information on FRBRoo, see https://www.ifla.org/node/10171) .</skos:historyNote>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#FormatType -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#FormatType">
        <owl:equivalentClass>
            <owl:Class>
                <owl:intersectionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://www.w3.org/2004/02/skos/core#Concept"/>
                    <owl:Restriction>
                        <owl:onProperty rdf:resource="http://www.w3.org/2004/02/skos/core#inScheme"/>
                        <owl:hasValue rdf:resource="http://data.europa.eu/eli/ontology#FormatTypeTable"/>
                    </owl:Restriction>
                </owl:intersectionOf>
            </owl:Class>
        </owl:equivalentClass>
        <rdfs:subClassOf rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <rdfs:comment xml:lang="en">Formally defined as the set of skos:Concept in concept scheme eli:MediaTypeTable.</rdfs:comment>
        <rdfs:label xml:lang="en">Format type</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#InForce -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#InForce">
        <owl:equivalentClass>
            <owl:Class>
                <owl:intersectionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://www.w3.org/2004/02/skos/core#Concept"/>
                    <owl:Restriction>
                        <owl:onProperty rdf:resource="http://www.w3.org/2004/02/skos/core#inScheme"/>
                        <owl:hasValue rdf:resource="http://data.europa.eu/eli/ontology#InForceTable"/>
                    </owl:Restriction>
                </owl:intersectionOf>
            </owl:Class>
        </owl:equivalentClass>
        <rdfs:subClassOf rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <rdfs:comment xml:lang="en">Formally defined as the set of skos:Concept in concept scheme eli:InForceTable</rdfs:comment>
        <rdfs:label xml:lang="en">In force</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#Language -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#Language">
        <owl:equivalentClass>
            <owl:Class>
                <owl:intersectionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://www.w3.org/2004/02/skos/core#Concept"/>
                    <owl:Restriction>
                        <owl:onProperty rdf:resource="http://www.w3.org/2004/02/skos/core#inScheme"/>
                        <owl:hasValue rdf:resource="http://publications.europa.eu/resource/authority/language"/>
                    </owl:Restriction>
                </owl:intersectionOf>
            </owl:Class>
        </owl:equivalentClass>
        <rdfs:subClassOf rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <rdfs:comment xml:lang="en">Formally defined as the set of skos:Concept in concept scheme http://publications.europa.eu/resource/authority/language</rdfs:comment>
        <rdfs:label xml:lang="en">Language</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.2</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.2 : fixed URI in definition</skos:historyNote>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#LegalExpression -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#LegalExpression">
        <rdfs:subClassOf rdf:resource="http://data.europa.eu/eli/ontology#Expression"/>
        <rdfs:comment xml:lang="en">The intellectual realisation of a legal resource in the form of a &quot;sequence of signs&quot; (typically alpha-numeric characters in a legal context).
For example, any version of the legal resource whose content is specified and different from others for any reason: language, versions, etc.;

Note that ELI ontology accommodates different point of view on what should be considered a new legal resource, or a new legal expression of the same resource. Typically, a consolidated version can be viewed, in the context of ELI, either as separate legal resource (linked to original version and previous consolidated version using corresponding ELI relations), or as a different legal expression of the same legal resource.</rdfs:comment>
        <rdfs:label xml:lang="en">Legal Expression</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : declared as a subclass of eli:Expression
v1.2 : replaced reference of RDA with FRBRoo (for more information on FRBRoo, see https://www.ifla.org/node/10171) .</skos:historyNote>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#LegalResource -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#LegalResource">
        <rdfs:subClassOf rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:comment xml:lang="en">A work in a legislative corpus. This applies to acts that have been legally enacted (whether or not they are still in force).
For example, the abstract concept of the legal resource; e.g. &quot;act 3 of 2005&quot; (adapted from Akoma Ntoso)

A legal resource can represent a legal act or any component of a legal act, like an article.  Legal resources can be linked together using properties defined in the model.

Note that ELI ontology accommodates different point of view on what should be considered a new legal resource, or a new legal expression of the same resource. Typically, a consolidated version can be viewed, in the context of ELI, either as separate legal resource (linked to original version and previous consolidated version using corresponding ELI relations), or as a different legal expression of the same legal resource.</rdfs:comment>
        <rdfs:label xml:lang="en">Legal Resource</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
        <owl:versionInfo>modified in v1.3</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.3 : declared as a subclass of eli:Work
v1.2 : replaced reference of RDA with FRBRoo (for more information on FRBRoo, see https://www.ifla.org/node/10171) .</skos:historyNote>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#LegalResourceSubdivision -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#LegalResourceSubdivision">
        <rdfs:subClassOf rdf:resource="http://data.europa.eu/eli/ontology#LegalResource"/>
        <rdfs:subClassOf rdf:resource="http://data.europa.eu/eli/ontology#WorkSubdivision"/>
        <rdfs:comment xml:lang="en">A component of a legal act, at an arbitrary level of precision, like a chapter, an article, an alinea, a paragraph or an list item. A subdivision can be linked with its parent and children subdivision using the properties eli:has_part / eli:is_part_of.</rdfs:comment>
        <rdfs:label xml:lang="en">Legal Resource Subdivision</rdfs:label>
        <owl:versionInfo xml:lang="en">created in v1.1</owl:versionInfo>
        <skos:historyNote xml:lang="en">v1.4 : added WorkSubdivision as a superclass</skos:historyNote>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#LegalValue -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#LegalValue">
        <owl:equivalentClass>
            <owl:Class>
                <owl:intersectionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://www.w3.org/2004/02/skos/core#Concept"/>
                    <owl:Restriction>
                        <owl:onProperty rdf:resource="http://www.w3.org/2004/02/skos/core#inScheme"/>
                        <owl:hasValue rdf:resource="http://data.europa.eu/eli/ontology#LegalValueTable"/>
                    </owl:Restriction>
                </owl:intersectionOf>
            </owl:Class>
        </owl:equivalentClass>
        <rdfs:subClassOf rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <rdfs:comment xml:lang="en">Formally defined as the set of skos:Concept in concept scheme eli:LegalValueTable</rdfs:comment>
        <rdfs:label xml:lang="en">Legal value</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#Manifestation -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#Manifestation">
        <rdfs:subClassOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/F3_Manifestation_Product_Type"/>
        <rdfs:comment xml:lang="en">The physical embodiment of an Expression, either on paper or in any electronic format.

The substance of Manifestation is not only signs, but also the manner in which they are presented to be consumed by users, including the kind of media adopted.</rdfs:comment>
        <rdfs:label xml:lang="en">Manifestation</rdfs:label>
        <owl:versionInfo>created v1.3</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#Organization -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#Organization">
        <rdfs:subClassOf rdf:resource="http://data.europa.eu/eli/ontology#Agent"/>
        <rdfs:subClassOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/F11_Corporate_Body"/>
        <rdfs:comment xml:lang="en">Institutions or groups of people that have obtained a legal recognition as a group and can act collectively as agents.</rdfs:comment>
        <rdfs:label xml:lang="en">Organization</rdfs:label>
        <owl:versionInfo>created in v1.3</owl:versionInfo>
        <owl:versionInfo>modified in v1.4</owl:versionInfo>
        <skos:historyNote>v1.4 : set as subclass of F11_Corporate_Body</skos:historyNote>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#Person -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#Person">
        <rdfs:subClassOf rdf:resource="http://data.europa.eu/eli/ontology#Agent"/>
        <rdfs:subClassOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/F10_Person"/>
        <rdfs:comment xml:lang="en">A real person who live or has lived.</rdfs:comment>
        <rdfs:label xml:lang="en">Person</rdfs:label>
        <owl:versionInfo>created in v1.3</owl:versionInfo>
        <owl:versionInfo>modified in v1.4</owl:versionInfo>
        <skos:historyNote>v1.4 : set as subclass of F10_Person</skos:historyNote>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#ResourceType -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#ResourceType">
        <owl:equivalentClass>
            <owl:Class>
                <owl:intersectionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://www.w3.org/2004/02/skos/core#Concept"/>
                    <owl:Restriction>
                        <owl:onProperty rdf:resource="http://www.w3.org/2004/02/skos/core#inScheme"/>
                        <owl:hasValue rdf:resource="http://data.europa.eu/eli/ontology#ResourceTypeTable"/>
                    </owl:Restriction>
                </owl:intersectionOf>
            </owl:Class>
        </owl:equivalentClass>
        <rdfs:subClassOf rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <rdfs:comment xml:lang="en">Formally defined as the set of skos:Concept in concept scheme eli:ResourceTypeTable</rdfs:comment>
        <rdfs:label xml:lang="en">Resource type</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#SubdivisionType -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#SubdivisionType">
        <owl:equivalentClass>
            <owl:Class>
                <owl:intersectionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://www.w3.org/2004/02/skos/core#Concept"/>
                    <owl:Restriction>
                        <owl:onProperty rdf:resource="http://www.w3.org/2004/02/skos/core#inScheme"/>
                        <owl:hasValue rdf:resource="http://data.europa.eu/eli/ontology#SubdivisionTypeTable"/>
                    </owl:Restriction>
                </owl:intersectionOf>
            </owl:Class>
        </owl:equivalentClass>
        <rdfs:subClassOf rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <rdfs:comment xml:lang="en">Formally defined as the set of skos:Concept in concept scheme eli:SubdivisionTypeTable</rdfs:comment>
        <rdfs:label xml:lang="en">Subdivision type</rdfs:label>
        <owl:versionInfo>created in v1.3</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#Version -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#Version">
        <owl:equivalentClass>
            <owl:Class>
                <owl:intersectionOf rdf:parseType="Collection">
                    <rdf:Description rdf:about="http://www.w3.org/2004/02/skos/core#Concept"/>
                    <owl:Restriction>
                        <owl:onProperty rdf:resource="http://www.w3.org/2004/02/skos/core#inScheme"/>
                        <owl:hasValue rdf:resource="http://data.europa.eu/eli/ontology#VersionTable"/>
                    </owl:Restriction>
                </owl:intersectionOf>
            </owl:Class>
        </owl:equivalentClass>
        <rdfs:subClassOf rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <rdfs:comment xml:lang="en">Formally defined as the set of skos:Concept in concept scheme eli:VersionTable</rdfs:comment>
        <rdfs:label xml:lang="en">Version</rdfs:label>
        <owl:versionInfo>created in v1.0</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#Work -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#Work">
        <rdfs:subClassOf rdf:resource="http://iflastandards.info/ns/fr/frbr/frbroo/F1_Work"/>
        <rdfs:comment xml:lang="en">Any distinct intellectual creation (i.e., the intellectual content), in the context of ELI.

The substance of Work is ideas.</rdfs:comment>
        <rdfs:label xml:lang="en">Work</rdfs:label>
        <owl:versionInfo>created in v1.3</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#WorkSubdivision -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#WorkSubdivision">
        <rdfs:subClassOf rdf:resource="http://data.europa.eu/eli/ontology#Work"/>
        <rdfs:comment xml:lang="en">A component of a work, at an arbitrary level of precision. A subdivision can be linked with its parent and children subdivision using eli:has_part / eli:is_part_of.</rdfs:comment>
        <rdfs:label xml:lang="en">Work Subdivision</rdfs:label>
        <owl:versionInfo>created in v1.4</owl:versionInfo>
    </owl:Class>
    


    <!-- http://data.europa.eu/eli/ontology#WorkType -->

    <owl:Class rdf:about="http://data.europa.eu/eli/ontology#WorkType">
        <owl:equivalentClass>
            <owl:Restriction>
                <owl:onProperty rdf:resource="http://www.w3.org/2004/02/skos/core#inScheme"/>
                <owl:hasValue rdf:resource="http://data.europa.eu/eli/ontology#WorkTypeTable"/>
            </owl:Restriction>
        </owl:equivalentClass>
        <rdfs:subClassOf rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <rdfs:label xml:lang="en">Formally defined as the set of skos:Concept in concept scheme eli:WorkTypeTable</rdfs:label>
        <rdfs:label xml:lang="en">Work Type</rdfs:label>
        <owl:versionInfo>created in v1.4</owl:versionInfo>
    </owl:Class>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/F10_Person -->

    <owl:Class rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/F10_Person"/>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/F11_Corporate_Body -->

    <owl:Class rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/F11_Corporate_Body"/>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/F1_Work -->

    <owl:Class rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/F1_Work"/>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/F22_Self-Contained_Expression -->

    <owl:Class rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/F22_Self-Contained_Expression"/>
    


    <!-- http://iflastandards.info/ns/fr/frbr/frbroo/F3_Manifestation_Product_Type -->

    <owl:Class rdf:about="http://iflastandards.info/ns/fr/frbr/frbroo/F3_Manifestation_Product_Type"/>
    


    <!-- http://www.cidoc-crm.org/cidoc-crm/E39_Actor -->

    <owl:Class rdf:about="http://www.cidoc-crm.org/cidoc-crm/E39_Actor"/>
    


    <!-- 
    ///////////////////////////////////////////////////////////////////////////////////////
    //
    // Individuals
    //
    ///////////////////////////////////////////////////////////////////////////////////////
     -->

    


    <!-- http://data.europa.eu/eli/ontology#AdministrativeAreaTable -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#AdministrativeAreaTable">
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#ConceptScheme"/>
        <rdfs:label xml:lang="en">Administrative area table</rdfs:label>
        <skos:definition xml:lang="en">A set of values for administrative areas.
ELI does not include a set of values for this table.</skos:definition>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#FormatTypeTable -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#FormatTypeTable">
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#ConceptScheme"/>
        <rdfs:comment xml:lang="en">A set of types for a Format.
ELI includes a list of values for this table.
This should contain user-oriented types to dinstinguish e.g. authentic PDF from scanned PDF from generated-on-the-fly PDF; or e.g. XML in a standard (AkomaNtoso) format vs. XML in a proprietary format.</rdfs:comment>
        <rdfs:label xml:lang="en">Format type table</rdfs:label>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#InForce-inForce -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#InForce-inForce">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#InForce"/>
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#InForceTable"/>
        <owl:versionInfo xml:lang="en">created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:definition xml:lang="en">Indicates that a legal resource is in force.</skos:definition>
        <skos:historyNote xml:lang="en">v1.1 : improved definition</skos:historyNote>
        <skos:prefLabel xml:lang="en">in force</skos:prefLabel>
        <skos:topConceptOf rdf:resource="http://data.europa.eu/eli/ontology#InForceTable"/>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#InForce-notInForce -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#InForce-notInForce">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#InForce"/>
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#InForceTable"/>
        <owl:versionInfo xml:lang="en">created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:definition xml:lang="en">Indicates that a legal resource is currently not in force.</skos:definition>
        <skos:historyNote xml:lang="en">v1.1 : improved definition</skos:historyNote>
        <skos:prefLabel xml:lang="en">not in force</skos:prefLabel>
        <skos:topConceptOf rdf:resource="http://data.europa.eu/eli/ontology#InForceTable"/>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#InForce-partiallyInForce -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#InForce-partiallyInForce">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#InForce"/>
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#InForceTable"/>
        <owl:versionInfo xml:lang="en">created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:definition xml:lang="en">Indicates that parts of the legal resource are in force, and parts are not.</skos:definition>
        <skos:historyNote xml:lang="en">v1.1 : improved definition</skos:historyNote>
        <skos:prefLabel xml:lang="en">partially in force</skos:prefLabel>
        <skos:topConceptOf rdf:resource="http://data.europa.eu/eli/ontology#InForceTable"/>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#InForceTable -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#InForceTable">
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#ConceptScheme"/>
        <rdfs:label xml:lang="en">In force table</rdfs:label>
        <skos:definition xml:lang="en">A set of values for the legal force of a resource.
ELI includes a set of values for this table.</skos:definition>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#LegalValue-authoritative -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#LegalValue-authoritative">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#LegalValue"/>
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#LegalValueTable"/>
        <owl:versionInfo xml:lang="en">created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:broader rdf:resource="http://data.europa.eu/eli/ontology#LegalValue-official"/>
        <skos:definition xml:lang="en">The publisher gives some special status to the publication of the document. (&quot;The Queens Printer&quot; version of a UK Act of Parliament). This status is specific to UK where no text can be considered « definitive ».</skos:definition>
        <skos:example xml:lang="en">&quot;The Queens Printer&quot; version of a UK Act of Parliament, or the Official Journal version of a EU Directive published by EU Publications Office.</skos:example>
        <skos:historyNote xml:lang="en">v1.1 : improved definition</skos:historyNote>
        <skos:narrower rdf:resource="http://data.europa.eu/eli/ontology#LegalValue-definitive"/>
        <skos:prefLabel xml:lang="en">authoritative</skos:prefLabel>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#LegalValue-definitive -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#LegalValue-definitive">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#LegalValue"/>
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#LegalValueTable"/>
        <owl:versionInfo xml:lang="en">created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:broader rdf:resource="http://data.europa.eu/eli/ontology#LegalValue-authoritative"/>
        <skos:definition xml:lang="en">Document for which the text is conclusively what the law says. (The digitally signed version of an Official Journal.)</skos:definition>
        <skos:example xml:lang="en">The digitally signed version of an Official Journal.</skos:example>
        <skos:historyNote xml:lang="en">v1.1 : improved definition</skos:historyNote>
        <skos:prefLabel xml:lang="en">definitive</skos:prefLabel>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#LegalValue-official -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#LegalValue-official">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#LegalValue"/>
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#LegalValueTable"/>
        <owl:versionInfo xml:lang="en">created in v1.0</owl:versionInfo>
        <owl:versionInfo xml:lang="en">modified in v1.1</owl:versionInfo>
        <skos:definition xml:lang="en">Document is published by an organisation with the public task of making the information available (A consolidated version of a EU directive published by EU Publications Office.)</skos:definition>
        <skos:example xml:lang="en">A consolidated version of a EU directive published by EU Publications Office.</skos:example>
        <skos:historyNote xml:lang="en">v1.1 : improved definition</skos:historyNote>
        <skos:narrower rdf:resource="http://data.europa.eu/eli/ontology#LegalValue-authoritative"/>
        <skos:prefLabel xml:lang="en">official</skos:prefLabel>
        <skos:topConceptOf rdf:resource="http://data.europa.eu/eli/ontology#LegalValueTable"/>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#LegalValue-unofficial -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#LegalValue-unofficial">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#LegalValue"/>
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#Concept"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#LegalValueTable"/>
        <owl:versionInfo xml:lang="en">created in v1.0</owl:versionInfo>
        <skos:definition xml:lang="en">Document has no particular or special standing.</skos:definition>
        <skos:prefLabel xml:lang="en">unofficial</skos:prefLabel>
        <skos:topConceptOf rdf:resource="http://data.europa.eu/eli/ontology#LegalValueTable"/>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#LegalValueTable -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#LegalValueTable">
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#ConceptScheme"/>
        <rdfs:label xml:lang="en">Legal value table</rdfs:label>
        <skos:definition xml:lang="en">A set of values for the legal value of a resource.
ELI ontology includes a set of values for this table.</skos:definition>
        <skos:hasTopConcept rdf:resource="http://data.europa.eu/eli/ontology#LegalValue-official"/>
        <skos:hasTopConcept rdf:resource="http://data.europa.eu/eli/ontology#LegalValue-unofficial"/>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#ResourceTypeTable -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#ResourceTypeTable">
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#ConceptScheme"/>
        <rdfs:label xml:lang="en">Resource type table</rdfs:label>
        <skos:definition xml:lang="en">A set of types for a legal resource.
ELI does not include a set of values for this table.</skos:definition>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#SubdivisionTypeTable -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#SubdivisionTypeTable">
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#ConceptScheme"/>
        <rdfs:comment xml:lang="en">A set of types for a legal resource subdivision.
ELI does not include a set of values for this table.</rdfs:comment>
        <rdfs:label xml:lang="en">Subdivision type table</rdfs:label>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#VersionTable -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#VersionTable">
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#ConceptScheme"/>
        <rdfs:label xml:lang="en">Version table</rdfs:label>
        <skos:definition xml:lang="en">A set of values for the version of a legal resource.
ELI does not include a set of values for this table.</skos:definition>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#WorkTypeTable -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#WorkTypeTable">
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#ConceptScheme"/>
        <rdfs:label xml:lang="en">Work Type Table</rdfs:label>
        <skos:definition xml:lang="en">A set of values for the type of a work.
ELI does not include a set of values for this table.</skos:definition>
    </owl:NamedIndividual>
    


    <!-- http://data.europa.eu/eli/ontology#print_format -->

    <owl:NamedIndividual rdf:about="http://data.europa.eu/eli/ontology#print_format">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#FormatTypeTable"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel xml:lang="en">print</skos:prefLabel>
        <skos:scopeNote xml:lang="en">ELI introduces this value to describe printed/paper versions of the legislation</skos:scopeNote>
    </owl:NamedIndividual>
    


    <!-- http://publications.europa.eu/resource/authority/atu -->

    <owl:NamedIndividual rdf:about="http://publications.europa.eu/resource/authority/atu">
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#ConceptScheme"/>
    </owl:NamedIndividual>
    


    <!-- http://publications.europa.eu/resource/authority/language -->

    <owl:NamedIndividual rdf:about="http://publications.europa.eu/resource/authority/language">
        <rdf:type rdf:resource="http://www.w3.org/2004/02/skos/core#ConceptScheme"/>
        <skos:definition xml:lang="en">A set of values for languages.
ELI does not include a set of values for this table.</skos:definition>
    </owl:NamedIndividual>
    


    <!-- https://www.iana.org/assignments/media-types/application/pdf -->

    <owl:NamedIndividual rdf:about="https://www.iana.org/assignments/media-types/application/pdf">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#FormatTypeTable"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel>application/pdf</skos:prefLabel>
    </owl:NamedIndividual>
    


    <!-- https://www.iana.org/assignments/media-types/application/xml -->

    <owl:NamedIndividual rdf:about="https://www.iana.org/assignments/media-types/application/xml">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#FormatTypeTable"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel>application/xml</skos:prefLabel>
    </owl:NamedIndividual>
    


    <!-- https://www.iana.org/assignments/media-types/application/akn+xml -->

    <owl:NamedIndividual rdf:about="https://www.iana.org/assignments/media-types/application/akn+xml">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#FormatTypeTable"/>
        <rdfs:seeAlso rdf:resource="https://docs.oasis-open.org/legaldocml/akn-media/v1.0/csprd01/akn-media-v1.0-csprd01.pdf"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel>application/akn+xml</skos:prefLabel>
    </owl:NamedIndividual>
    


    <!-- https://www.iana.org/assignments/media-types/application/pdf;type=archival -->

    <owl:NamedIndividual rdf:about="https://www.iana.org/assignments/media-types/application/pdf;type=archival">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel>application/pdf;type=archival</skos:prefLabel>
        <skos:scopeNote xml:lang="en">ELI introduces this value to describe PDF files that are intended for long-term archival.</skos:scopeNote>
    </owl:NamedIndividual>
    


    <!-- https://www.iana.org/assignments/media-types/application/pdf;type=generated -->

    <owl:NamedIndividual rdf:about="https://www.iana.org/assignments/media-types/application/pdf;type=generated">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel>application/pdf;type=generated</skos:prefLabel>
        <skos:scopeNote xml:lang="en">ELI introduces this value to describe PDF files that are generated on-the-fly from base XML formats.</skos:scopeNote>
    </owl:NamedIndividual>
    


    <!-- https://www.iana.org/assignments/media-types/application/pdf;type=scanned -->

    <owl:NamedIndividual rdf:about="https://www.iana.org/assignments/media-types/application/pdf;type=scanned">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel>application/pdf;type=scanned</skos:prefLabel>
        <skos:scopeNote xml:lang="en">ELI introduces this value to describe PDF files that are scanned copies of a paper version.</skos:scopeNote>
    </owl:NamedIndividual>
    


    <!-- https://www.iana.org/assignments/media-types/application/pdf;type=signed -->

    <owl:NamedIndividual rdf:about="https://www.iana.org/assignments/media-types/application/pdf;type=signed">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel>application/pdf;type=signed</skos:prefLabel>
        <skos:scopeNote xml:lang="en">ELI introduces this value to describe PDF files that are signed electronically.</skos:scopeNote>
    </owl:NamedIndividual>
    


    <!-- https://www.iana.org/assignments/media-types/application/xhtml+xml -->

    <owl:NamedIndividual rdf:about="https://www.iana.org/assignments/media-types/application/xhtml+xml">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#FormatTypeTable"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel>application/xhtml+xml</skos:prefLabel>
    </owl:NamedIndividual>
    


    <!-- https://www.iana.org/assignments/media-types/text/html -->

    <owl:NamedIndividual rdf:about="https://www.iana.org/assignments/media-types/text/html">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#FormatTypeTable"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel>text/html</skos:prefLabel>
    </owl:NamedIndividual>
    


    <!-- https://www.iana.org/assignments/media-types/text/plain -->

    <owl:NamedIndividual rdf:about="https://www.iana.org/assignments/media-types/text/plain">
        <rdf:type rdf:resource="http://data.europa.eu/eli/ontology#FormatType"/>
        <skos:inScheme rdf:resource="http://data.europa.eu/eli/ontology#FormatTypeTable"/>
        <owl:versionInfo xml:lang="en">created in v1.2</owl:versionInfo>
        <skos:prefLabel>text/plain</skos:prefLabel>
    </owl:NamedIndividual>
</rdf:RDF>



<!-- Generated by the OWL API (version 4.5.9.2019-02-01T07:24:44Z) https://github.com/owlcs/owlapi -->
````

## File: tulit/parsers/xml/assets/formex4.xsd
````
<?xml version="1.0" encoding="utf-8"?>
<xd:schema xmlns:xd="http://www.w3.org/2001/XMLSchema" xmlns:fmx="http://opoce" xml:space="preserve"><xd:annotation>
		<xd:documentation>
			<fmx:version>Current version 6.02.1</fmx:version>
			<fmx:history>
				<fmx:ti.history>History</fmx:ti.history>
				<fmx:line.history><fmx:line.date>20020117</fmx:line.date> 
					transform from DTD
				</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-ACT.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-ANNEX.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-CONSACT.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-AGR-act.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-AGR-letter.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-CONSOTH-1.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-CONSOTH-2.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-GENERAL.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-PROPOSAL.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-PROPOSAL-Amend.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-PROPOSAL-Amend-gen.xml</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020121</fmx:line.date> 
					test (XSV) with instance test-CJT.xml 
				</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-COMPETIT.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-COMPETIT-Gen.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-ECU-1.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-ECU-2.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-ECU-3.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-LIST-COM.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-OP-COM.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-OP-COM-Amend.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-STATEAID.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-STATEAID-Box.xml</fmx:line.history>
				<fmx:line.history>test (XSV) with instance test-SUMMARY-L.xml</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020122</fmx:line.date> 
					LG, LG.OJ -t_language 
				</fmx:line.history>
				<fmx:line.history>SADDR -ADDR.S, SFORMULA -FORMULA.S, SQUOT -QUOT.S</fmx:line.history>
				<fmx:line.history>extenstion of t_btx.seq (ADDR.S, FORMULAR.S, QUOT.S)</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020123</fmx:line.date> 
					extension of language codes according to ISO 639 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020124</fmx:line.date> 
					alternative table model 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020128</fmx:line.date> 
					NO.SEQ -IDENTIFFIER (ARTICLE, PARAG)
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020129</fmx:line.date> 
					alternative for publication structure 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020130</fmx:line.date> 
					EEA added to DOC.MAIN.PUB 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020204</fmx:line.date> 
					Modification of STI.DOC, NO.DOC in GENERAL 
				</fmx:line.history>
				<fmx:line.history>BIB.REF added</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020206</fmx:line.date> 
					integration of TI.CJT, modification of GENERAL 
				</fmx:line.history>
				<fmx:line.history>integration of NO.DOC.C</fmx:line.history>
				<fmx:line.history>redefinition of INDEX, integration of KEYWORD, LG.PROC, TRANS.REF</fmx:line.history>
				<fmx:line.history>integration of ADDR.S into GR.SEQ</fmx:line.history>
				<fmx:line.history>integration of optional attribute LEVEL for GR.SEQ</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020207</fmx:line.date> 
					integration of proposal for frequent text elements 
				</fmx:line.history>
				<fmx:line.history>(FTE, DICTIONARY, VOCABULARY, FTE.ENTRY, t_fte.id)</fmx:line.history>
				<fmx:line.history>OPINION added as value for DOC.TYPE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020211</fmx:line.date> 
					FT TYPE="NUMBER" added 
				</fmx:line.history>
				<fmx:line.history>attribute value TOTAL for ROW.ALT and CELL.ALT</fmx:line.history>
				<fmx:line.history>attribute value STATE.AID-AUTORISATION for DOC.TYPE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020212</fmx:line.date> 
					attribute value UNDERLINE for HT 
				</fmx:line.history>
				<fmx:line.history>value ANTI-DUMPING added to DOC.TYPE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020213</fmx:line.date> 
					value NOTICE added to DOC.TYPE 
				</fmx:line.history>
				<fmx:line.history>value RECRUITMENT added to DOC.TYPE</fmx:line.history>
				<fmx:line.history>frequence of PREAMBLE in AGR</fmx:line.history>
				<fmx:line.history>frequence of PL.DATE in SIGNATURE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020214</fmx:line.date> 
					value CONCENTRATION added to DOC.TYPE 
				</fmx:line.history>
				<fmx:line.history>element LIST added to CONTENTS</fmx:line.history>
				<fmx:line.history>element ADDR.S added to CONTENTS</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020218</fmx:line.date> 
					value LICENCE added to DOC.TYPE 
				</fmx:line.history>
				<fmx:line.history>value RECOMMENDATION added to DOC.TYPE</fmx:line.history>
				<fmx:line.history>value TENDER added to DOC.TYPE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020225</fmx:line.date> 
					element MARGIN integrated 
				</fmx:line.history>
				<fmx:line.history>element CORR integrated for handling of corrigenda</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020227</fmx:line.date> 
					value INFORMATION added to DOC.TYPE 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020304</fmx:line.date> 
					correction of model DIV.CONSID: SIT.DIV.CONSID optional 
				</fmx:line.history>
				<fmx:line.history>correction of model CONTENTS.CORR: minOccursw moved to element</fmx:line.history>
				<fmx:line.history>correction of model CORRECTION: sequence of OLD.CORR/NEW.CORR optional</fmx:line.history>
				<fmx:line.history>correction of model GR.TBL.ALT: TI.TBL.ALT optional, STI.TBL.ALT unbounded</fmx:line.history>
				<fmx:line.history>redefinition of model EXPL.REF: contents allowed</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020305</fmx:line.date> 
					attribute NUMBERING added to EXPL.REF 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020306</fmx:line.date> 
					type of STI.* changed from t_btx.seq to t_btx 
				</fmx:line.history>
				<fmx:line.history>attribute value DECIMAL added to element FT</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020311</fmx:line.date> 
					correction of attribute value for DATE: ENTER.INTO.FORCE to ENTRY.INTO.FORCE 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020312</fmx:line.date> 
					value APPOINTMENT added to DOC.TYPE 
				</fmx:line.history>
				<fmx:line.history>value CALL.PROPOSAL added to DOC.TYPE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020313</fmx:line.date> 
					value COMMON.POSITION added to DOC.TYPE 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020327</fmx:line.date> 
					optionally empty contents of GR.SEQ 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020403</fmx:line.date> 
					value TARIC for CELL.ALT 
				</fmx:line.history>
				<fmx:line.history>occurences for PL.DATE in LETTER</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020404</fmx:line.date> 
					frequence of FINAL in AGR 
				</fmx:line.history>
				<fmx:line.history>TXT added to t_btx.seq</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020408</fmx:line.date> 
					review of definitions/models, version 0.6 
					<fmx:ul type="dash">
						<fmx:li>t_btx.struct: GR.ANNOTATION, GR.SEQ, GR.TBL, GR.TBL.ALT, LIST.AMEND added</fmx:li>
						<fmx:li>t_no.oj: optional suffixes A or E</fmx:li>
						<fmx:li>t_page: prefixes c and e added to pattern</fmx:li>
						<fmx:li>ARTICLE/@IDENTIFIER: always three digits</fmx:li>
						<fmx:li>CELL.ALT/@CONT.TYPE: repl of NUMERIC by NUMBER</fmx:li>
						<fmx:li>CONTENTS: content type modified to t_btx.struct</fmx:li>
						<fmx:li>HT/@TYPE: STROKE added</fmx:li>
						<fmx:li>INCL.ELEMENT: direct child of GR.SEQ</fmx:li>
					</fmx:ul></fmx:line.history>
				<fmx:line.history><fmx:line.date>20020418</fmx:line.date> 
					value TYPE of element COLL completed by CA, CE and LA 
				</fmx:line.history>
				<fmx:line.history>distinction between SECTION and SUBSECTION, required attribute with predefined values for SECTION</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020419</fmx:line.date> 
					dictionaries organized in simpleType definitions 
				</fmx:line.history>
				<fmx:line.history>value CATALOG.PLANTS added to DOCTYPE</fmx:line.history>
				<fmx:line.history>TOC added to t_btx.struct</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020422</fmx:line.date> 
					element PRINTER optional in model BIB.SUM 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020423</fmx:line.date> 
					element VL added to t_btx, t_btx.seq, t_btx.struct 
				</fmx:line.history>
				<fmx:line.history>definition of element CAT.PLANT, integration into CONTENTS</fmx:line.history>
				<fmx:line.history>definition of element GR.PLANT</fmx:line.history>
				<fmx:line.history>definition of element TI.GR.PLANT</fmx:line.history>
				<fmx:line.history>definition of element SPECIE</fmx:line.history>
				<fmx:line.history>definition of element TI.SPECIE</fmx:line.history>
				<fmx:line.history>definition of element NO.SPECIE</fmx:line.history>
				<fmx:line.history>definition of element NAME.COMMON</fmx:line.history>
				<fmx:line.history>definition of element NAME.SCIENT</fmx:line.history>
				<fmx:line.history>definition of element VAR.INFO</fmx:line.history>
				<fmx:line.history>definition of element DEF.HEAD.VI</fmx:line.history>
				<fmx:line.history>definition of element DEF.VARIETY</fmx:line.history>
				<fmx:line.history>definition of element DEF.ADM.COUNTRY</fmx:line.history>
				<fmx:line.history>definition of element DEF.ADM.COUNTRY.HD</fmx:line.history>
				<fmx:line.history>definition of element DEF.EFTA.COUNTRY</fmx:line.history>
				<fmx:line.history>definition of element DEF.EFTA.COUNTRY.HD</fmx:line.history>
				<fmx:line.history>definition of element DEF.OBS.VAR</fmx:line.history>
				<fmx:line.history>definition of element DEF.OBS.VARX</fmx:line.history>
				<fmx:line.history>definition of element DEF.OBS.VARX.HD</fmx:line.history>
				<fmx:line.history>definition of element DEF.COUNTRY</fmx:line.history>
				<fmx:line.history>definition of element DEF.INDEX</fmx:line.history>
				<fmx:line.history>definition of element DEF.FORMA.H</fmx:line.history>
				<fmx:line.history>definition of element DEF.GEN.OBS</fmx:line.history>
				<fmx:line.history>definition of element GR.UNIT.VI</fmx:line.history>
				<fmx:line.history>definition of element TI.GR.UNIT.VI</fmx:line.history>
				<fmx:line.history>definition of element UNIT.VI</fmx:line.history>
				<fmx:line.history>definition of element NAME.VAR</fmx:line.history>
				<fmx:line.history>definition of element ADM.CRIT</fmx:line.history>
				<fmx:line.history>definition of element EFTA.CRIT</fmx:line.history>
				<fmx:line.history>definition of element QUALIF</fmx:line.history>
				<fmx:line.history>definition of element ID.RESP</fmx:line.history>
				<fmx:line.history>definition of element OBS.VAR</fmx:line.history>
				<fmx:line.history>definition of element OBS.VARX</fmx:line.history>
				<fmx:line.history>definition of element INDEX.MAT</fmx:line.history>
				<fmx:line.history>definition of element FORMA.H</fmx:line.history>
				<fmx:line.history>definition of element GEN.OBS</fmx:line.history>
				<fmx:line.history>all document titles of GENERAL made optional</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020429</fmx:line.date> 
					TXT added to t_btx
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020430</fmx:line.date> 
					CAT.VEGETABLES added to t_doctype 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020502</fmx:line.date> 
					revision of CAT.PLANTS
				</fmx:line.history>
				<fmx:line.history>DEF.ADM.COUNTRY and DEF.EFTA.COUNTRY deleted, DEF.AREA instead</fmx:line.history>
				<fmx:line.history>DEF.ADM.COUNTRY.HD and DEF.EFTA.COUNTRY.HD deleted, DEF.AREA.HD instead</fmx:line.history>
				<fmx:line.history>redefinition of UNIT.VI</fmx:line.history>
				<fmx:line.history>ADM.CRIT and EFTA.CRIT deleted, CRIT instead</fmx:line.history>
				<fmx:line.history>definition of type t_area</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020506</fmx:line.date> 
					definition of CAT.VEGET, added to t_btx.struct 
				</fmx:line.history>
				<fmx:line.history>NP as alternative element in TI.GR.SEQ</fmx:line.history>
				<fmx:line.history>TI.TBL defined as t_btx.struct</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020508</fmx:line.date> 
					correction of attribute value ATUOR to AUTO for EXPL.REF/NUMBERING
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020513</fmx:line.date> 
					DEFINITION redefined as optional
				</fmx:line.history>
				<fmx:line.history>redefinition of NOTE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020514</fmx:line.date> 
					redefinition of GR.NOTES 
				</fmx:line.history>
				<fmx:line.history>attribute NUMBER.ORG added to element NOTE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020515</fmx:line.date> 
					attribute value alpha added to numbering of NOTE 
				</fmx:line.history>
				<fmx:line.history>suppression of element NO.NOTE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020516</fmx:line.date> 
					element NOTE.REF added 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020611</fmx:line.date> 
					frequence of SIGNATURE modified to unbounded optional in CONS.FINAL, FINAL, and GENERAL 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020612</fmx:line.date> 
					integration of STYLE.xxx attributes 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020702</fmx:line.date> 
					optional attribute IDENTIFIER for element GR.SEQ
				</fmx:line.history>
				<fmx:line.history>TI.GR.SEQ/NP occurrence set to 1</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020703</fmx:line.date> 
					elements for publication of CN 
				</fmx:line.history>
				<fmx:line.history>redefinition of PARAG, suppression of ALINEA.NUM</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020704</fmx:line.date> 
					optional attribute TYPE for GR.ANNOTATION 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020708</fmx:line.date> 
					simplification of model ITEM
				</fmx:line.history>
				<fmx:line.history>suppression of NOTCOL, corresponding attribute value for ROW.ALT, CELL.ALT</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020709</fmx:line.date> 
					correction of CN.* models, GR.NOTES added to GR.SEQ and CONTENTS; element TOC added to ANNEX 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020729</fmx:line.date> 
					model TBL and specific children suppressed 
				</fmx:line.history>
				<fmx:line.history>model TBL.ALT and specific children renamed to TBL etc.</fmx:line.history>
				<fmx:line.history>element YEAR added to OJ, YEAR.OJ deleted from OJ</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020730</fmx:line.date> 
					redefinition of QUOT.S because XMLSpy does not treat anyType correctly
				</fmx:line.history>
				<fmx:line.history>element EXPL.TXT added to GR.NOTES</fmx:line.history>
				<fmx:line.history>pattern corrected for IDENTIFIER of elements ARTICLE and PARAG</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020731</fmx:line.date> 
					pattern of t_page corrected 
				</fmx:line.history>
				<fmx:line.history>GR.SEQ added to ACT for new kind of structure of proposal</fmx:line.history>
				<fmx:line.history>correction of REF.OJ: YEAR.OJ replaced by YEAR</fmx:line.history>
				<fmx:line.history>DOC.REF in LINK.OJ optional</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020801</fmx:line.date> 
					revision of t_btx.struct; creation of t_btx.struct.any 
				</fmx:line.history>
				<fmx:line.history>revision of COLL, DURAB, EEA, FTE, INCL.ELEMENT, INFO.CONSLEG, INFO.PROD, NEW, NEW.CORR, OLD, OLD.CORR, QUOT.S, TYPE.OJ</fmx:line.history>
				<fmx:line.history>VL deleted, substituted by LV</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020805</fmx:line.date> 
					integration of QUOT.START/QUOT.END with crossing references 
				</fmx:line.history>
				<fmx:line.history>NO.ITEM added</fmx:line.history>
				<fmx:line.history>CAPTION added to INCL.ELEMENT</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020806</fmx:line.date> 
					TOC added to ACT.GEN 
				</fmx:line.history>
				<fmx:line.history>attribute TYPE added to NOTE and EXPL.REF</fmx:line.history>
				<fmx:line.history>NOTE.REF deleted maintaining 2 techniques for footnote handling (NOTE and EXPL.TXT/EXPL.REF)</fmx:line.history>
				<fmx:line.history>TXT deleted from FINAL</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020807</fmx:line.date> 
					element APPLIC deleted, replaced by P
				</fmx:line.history>
				<fmx:line.history>model SUMMARY deleted, replaced by PUBLICATION (deleted: BIB.SUM, DOC.ASSOC, DOC.MAIN, DOC.SUMMARY, ITEM.SUMMARY, NO.INFO, SECTION.SUMMARY, STI.SECTION.SUMMARY, SUMMARY, TI.SECTION.SUMMARY)</fmx:line.history>
				<fmx:line.history>attribute CONT.TYPE deleted from CELL, replaced by use of FT (value CODE added)</fmx:line.history>
				<fmx:line.history>attribute CONTEXT deleted from DATE</fmx:line.history>
				<fmx:line.history>attribute value REFOJ deleted from FT/@TYPE, replaced by new element OJ.REF</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020808</fmx:line.date> 
					pattern for attribute DATE.PUB in element OJ.REF completed 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20020813</fmx:line.date> 
					QUOT.S integrated into ARTICLE, PARAG, LIST 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021017</fmx:line.date> 
					ADDR.S needs 2 P elements at least; LV suppressed
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021018</fmx:line.date> 
					element SUBDIV added to ARTICLE
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021023</fmx:line.date> 
					element IMG.CNT and attribute CONTENT added to INCL.ELEMENT
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021024</fmx:line.date> 
					element GR.SEQ added to t_btx 
				</fmx:line.history>
				<fmx:line.history>QUOT.S completed by QUOT.START/QUOT.END in t_btx.struct</fmx:line.history>
				<fmx:line.history>attribute CONTENT for element INCL.ELEMENT made optional</fmx:line.history>
				<fmx:line.history>attribute PAGE.FIRST for element OJ.REF made optional</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021028</fmx:line.date> 
					value STATEMENT added to t_doctype 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021030</fmx:line.date> 
					ACT.GEN supressed from the list of possible root elements
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021031</fmx:line.date> 
					GR.SEQ added to PREAMBLE 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021104</fmx:line.date> 
					suppression of QUOT (use of QUOT.START/QUOT.END instead)
				</fmx:line.history>
				<fmx:line.history>new value CHAPTER for attribute TYPE of element NOTE</fmx:line.history>
				<fmx:line.history>QUOT.S added to t_btx.struct</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021105</fmx:line.date> 
					revision of model PREAMBLE
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021107</fmx:line.date> 
					type of TI.DIV.CONSID and TI.DIVISION changed to t_btx.struct 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021111</fmx:line.date> 
					models CORPUS and BLK changed (seuqence choice)
				</fmx:line.history>
				<fmx:line.history>type t_fte.id and elements DICTIONARY, FTE, FTE.ENTRY, and VOCABULARY deleted</fmx:line.history>
				<fmx:line.history>value MANUAL added to attribute CLASS of element NOTE</fmx:line.history>
				<fmx:line.history>optional closing element LOC.NOTE (REF.NOTE) added to element GR.SEQ</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021118</fmx:line.date> 
					revision of models TI.GR.SEQ and STI.GR.SEQ
				</fmx:line.history>
				<fmx:line.history>revision of attributes ACCH and ACCV</fmx:line.history>
				<!-- version 0.91 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20021120</fmx:line.date> 
					attributes concerning style information grouped by means of xd:attributGroup 
				</fmx:line.history>
				<fmx:line.history>type of element NO.CURRENT changed to xd:string because of documents from CJ; elements YEAR and COM optional in NO.DOC</fmx:line.history>
				<fmx:line.history>revision of schema</fmx:line.history>
				<fmx:line.history>ACT: more flexibility for TOC</fmx:line.history>
				<fmx:line.history>CRIT: codes added for new member states</fmx:line.history>
				<fmx:line.history>HT: contents based on t_btx because of value BOX</fmx:line.history>
				<fmx:line.history>NO.DOC: no defaults for attributes</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021121</fmx:line.date> 
					correction of errors and incoherences: 
				</fmx:line.history>
				<fmx:line.history>attribute CLASS of element NO.OJ deleted, because information exists in COLL</fmx:line.history>
				<fmx:line.history>elements QUOT.END and QUOT.START deleted from t_btx.struct</fmx:line.history>
				<fmx:line.history>element TBL.INIT deleted</fmx:line.history>
				<fmx:line.history>revision of element ANNOTATION</fmx:line.history>
				<fmx:line.history>element LG in BIB.OJ replaced by LG.OJ</fmx:line.history>
				<fmx:line.history>no repetitions of PAGE.SUMMARY in BIB.OJ</fmx:line.history>
				<fmx:line.history>element EEA moved 1 level higher</fmx:line.history>
				<fmx:line.history>LINK.OJ suppressed from DOC.MAIN.PUB and DOC.SUB.PUB</fmx:line.history>
				<fmx:line.history>elements COLL, NO.OJ, YEAR, LG.OJ integrated into DOC.MAIN.PUB and DOC.SUB.PUB</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021127</fmx:line.date> 
					EEA raised to BIB.DOC, re-integration of LINK.OJ into DOC.MAIN.PUB and DOC.SUB.PUB
				</fmx:line.history>
				<fmx:line.history>revision of elements EEA and GR.SEQ</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021128</fmx:line.date> 
					revision of model of NO.ISSN
				</fmx:line.history>
				<fmx:line.history>suppression of NO.VOLUME from BIB.OJ, integration of VOLUME.REF into DOC.MAIN.PUB and DOC.SUB.PUB, suppression of NO.VOLUME</fmx:line.history>
				<fmx:line.history>extension of element CELL by elements TXT.COL and TXT.ROW</fmx:line.history>
				<!-- version 0.92 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20021204</fmx:line.date> 
					BIB.REF deleted from model GENERAL
				</fmx:line.history>
				<fmx:line.history>SIGNATURE deleted from model GENERAL (use of FINAL)</fmx:line.history>
				<fmx:line.history>revision of TI.*, maintained only for semanticall or structurall special types, otherwise replaced by generic element TITLE</fmx:line.history>
				<fmx:line.history>value TITLE added to attribute LOC of element NOTE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021209</fmx:line.date> 
					QUOT.S added to GR.SEQ model 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021212</fmx:line.date> 
					YEAR.OJ renamed to AGE.OJ
				</fmx:line.history>
				<fmx:line.history>attribute group gr.styles added to CELL</fmx:line.history>
				<fmx:line.history>element INT.INST replaced by COM</fmx:line.history>
				<fmx:line.history>elements BIB.REF, REF.DOC and TI.LETTER deleted</fmx:line.history>
				<fmx:line.history>elements EXPL.REF and EXPL.TXT replaced by NOTE, NOTE.ID no longer mandatory, t_btx.struct with minOccurs="0" declaration</fmx:line.history>
				<fmx:line.history><fmx:line.date>20021216</fmx:line.date> 
					element MARGIN deleted from t_btx.struct, t_btx.struct.any, t_btx.title, ADDR.S, AMEND, ANNOTATION, ARTICLE, BLK, GR.SEQ, LETTER, PL.DATE, SUBDIV
				</fmx:line.history>
				<!-- version 0.93 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20030128</fmx:line.date> 
					correction of non-deterministic models for ARTICLE and SUBDIV, correction of double declaration of attributes for TI and STI
				</fmx:line.history>
				<!-- version 0.94 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20030228</fmx:line.date> 
					revision of attribute group gr.attributes
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030304</fmx:line.date> 
					review of definitions/models, new types
					<fmx:ul type="dash">
						<fmx:li>TYPE.OJ renamed into LAST.OJ</fmx:li>
						<fmx:li>new type t_boolean with values NO/YES</fmx:li>
						<fmx:li>new type t_collection</fmx:li>
						<fmx:li>type t_no.oj reviewed</fmx:li>
						<fmx:li>element DATE.PUB modified</fmx:li>
						<fmx:li>new type t_date.publication</fmx:li>
						<fmx:li>type t_page extended (roman numbers)</fmx:li>
						<fmx:li>element OJ.REF renamed into REF.DOC.OJ, model extended</fmx:li>
						<fmx:li>new type t_page.sequence</fmx:li>
						<fmx:li>elements PAGE.FIRST, PAGE.LAST, PAGE.SEQ reviewed</fmx:li>
						<fmx:li>OJ renamed into BIB.INSTANCE, completed by LG.OJ</fmx:li>
						<fmx:li>element BIB.DATA reviewed</fmx:li>
						<fmx:li>type t_btx.title reviewed</fmx:li>
						<fmx:li>element NO.DOC reviewed</fmx:li>
						<fmx:li>new type t_format.no.doc</fmx:li>
						<fmx:li>element REF.DOC.CORR deleted</fmx:li>
					</fmx:ul></fmx:line.history>
				<fmx:line.history><fmx:line.date>20030310</fmx:line.date> 
					attribute group gr.attributes deleted from CELL
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030312</fmx:line.date> 
					element COMPETITION deleted from t_btx.struct; TOC added to ANNEX; document type PROPMODIF replaced by GENERAL; document type CJT added; revision of GENERAL, TOC added; ACT/GR.SEQ replaced by ACT/PROLOG
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030313</fmx:line.date> 
					review of element GENERAL
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030314</fmx:line.date> 
					type t_btx.struct.any deleted; element FRAGMENT created; type t_btx.struct completed; type t_btx revised; element BIB.INSTANCE mandatory everywhere; type changed to t_btx.seq for elements: CN.CODE, CN.DESC, CN.RATE, CN.SUP.UNIT, DEF.VARIETY, ITEM.REF, PERIOD; review of element GENERAL
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030317</fmx:line.date> 
					correction of element ITEM
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030318</fmx:line.date> 
					correction of element PREAMBLE 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030319</fmx:line.date> 
					correction of mixed content for btx, btx.form and btx.seq elements 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030527</fmx:line.date> 
					TOC added to ENACTING.TERMS 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030630</fmx:line.date> 
					reference to gr.styles added to element NO.DOC.C 
				</fmx:line.history>
				<!-- version 0.95 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20030709</fmx:line.date> 
					review of CONSLEG needs:
				</fmx:line.history>
				<fmx:line.history>CONS.FINAL replaced by FINAL</fmx:line.history>
				<fmx:line.history>suppression of CONS.FINAL</fmx:line.history>
				<fmx:line.history>LEG.VAL added to INFO.CONSLEG</fmx:line.history>
				<fmx:line.history>SPEC.ED: redefined, elements CHAP.SE and TOME.SE created</fmx:line.history>
				<fmx:line.history>attribute CELEX.BASIC deleted from INFO.CONSLEG</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030728</fmx:line.date> 
					review of bibliographical approach:
				</fmx:line.history>
				<fmx:line.history>publication level</fmx:line.history>
				<fmx:line.history>-new elements: OJ, BIB.VOLUME, GEN.PUB, BIB.GEN.PUB, PUBLISHER, DIMENSION, NO.ISBN, INFO.PUBLISHER, COPYRIGHT, HINT</fmx:line.history>
				<fmx:line.history>-modified elements: PUBLICATION, BIB.OJ, VOLUME, SECTION, SUBSECTION</fmx:line.history>
				<fmx:line.history>document level</fmx:line.history>
				<fmx:line.history>-new elements: PUBLICATION.REF</fmx:line.history>
				<fmx:line.history>-modified elements: DOC, DOC.MAIN.PUB, DOC.SUB.PUB, REF.PHYS</fmx:line.history>
				<fmx:line.history>instance level</fmx:line.history>
				<fmx:line.history>-new elements: DOCUMENT.REF</fmx:line.history>
				<fmx:line.history>-modified elements: BIB.INSTANCE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030729</fmx:line.date> 
					integration of TOC on bibliographical level: DOC, SECTION, SUBSECTION, VOLUME 
				</fmx:line.history>
				<fmx:line.history>file reference attributes added to elements ITEM.REF and TOC.HD.REF</fmx:line.history>
				<fmx:line.history>TARIC added to values for DOCTYPE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030806</fmx:line.date> 
					attribute value for TYPE in element INCL.ELEMENT renamed to FORMEX.DOC, value FORMEX.FRAGMENT added
				</fmx:line.history>
				<fmx:line.history>integration of TARIC publication</fmx:line.history>
				<fmx:line.history>-elements added: TARIC, DEF.HEAD.TA, UNIT.TA, DEF.CN.CODE, DEF.SH.TARIC, DEF.DESC, DEF.SUP.UNIT, DEF.REM, DEF.TA.CODE, DEF.THIRD.RATE, DEF.PART.RATE, DEF.IMPORT, DEF.EXPORT, DEF.SPG, DEF.OPREF, SH.TARIC, POS.TARIC, DESC, SUP.UIT, IMPORT, EXPORT, TA.CODE, THIRD.RATE, SPG, OPREF</fmx:line.history>
				<fmx:line.history>-modified elements: NOTE (OTHER added as attribute value of NUMBERING), TOC, TOC.BLK</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030807</fmx:line.date> 
					modification of element ITEM: contents only NP or P+
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20030813</fmx:line.date> 
					correction of some mixed content models based on t_btx, t_btx.seq and t_btx.form 
				</fmx:line.history>
				<!-- version 0.96 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20030930</fmx:line.date> 
					revision of TOC model 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20031001</fmx:line.date> 
					revision of GR.SEQ model (same as t_btx.struct, but with title) 
				</fmx:line.history>
				<fmx:line.history>optional identifier part added to identifier of articles and paragraphs (esp. for constitution)</fmx:line.history>
				<fmx:line.history><fmx:line.date>20031002</fmx:line.date> 
					improvement of pattern for ARTICLE and PARAG 
				</fmx:line.history>
				<fmx:line.history>new values for document type: CONSTITUTION, PROTOCOL</fmx:line.history>
				<fmx:line.history><fmx:line.date>20031124</fmx:line.date> 
					change of type for element TI.BLK into t_btx 
				</fmx:line.history>
				<fmx:line.history>FT and HT added to t_btx.form</fmx:line.history>
				<fmx:line.history><fmx:line.date>20031125</fmx:line.date> 
					TITLE added to TOC, TOC.BLK 
				</fmx:line.history>
				<fmx:line.history>attribute REF.XML added to ITEM.REF</fmx:line.history>
				<fmx:line.history>correction of non-deterministic model of element GR.SEQ</fmx:line.history>
				<fmx:line.history><fmx:line.date>20031201</fmx:line.date> 
					creation of type t_date 
				</fmx:line.history>
				<fmx:line.history>integration of CONS.LIST</fmx:line.history>
				<fmx:line.history><fmx:line.date>20031203</fmx:line.date> 
					reintegration of MARGIN to ARTICLE model; harmonisation of IDENTIFIER attributes (ARTICLE, PARAG)
				</fmx:line.history>
				<!-- version 0.97 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20031209</fmx:line.date> 
					TEMPLATE deleted from gr.styles 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20031211</fmx:line.date> 
					LEG.VAL attribute deleted from FAM.COMP.CL, as it is the same as on the higher level CONS.LIST 
				</fmx:line.history>
				<!-- version 0.98 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20031217</fmx:line.date> 
					attribute SYNOPTISM added to DOC.MAIN.PUB and DOC.SUB.PUB 
				</fmx:line.history>
				<fmx:line.history>implementation of element ENV.BIBLIO</fmx:line.history>
				<fmx:line.history>template integrated into root elements</fmx:line.history>
				<!-- version 0.99 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20040203</fmx:line.date> 
					attribute REF.DOC.SEQ added to ITEM.VOLUME 
				</fmx:line.history>
				<fmx:line.history>type of element DESC changed to t_btx</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040204</fmx:line.date> 
					attribute IDENTIFIER deleted from GR.SEQ 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040206</fmx:line.date> 
					correction of ambiguous model SUBDIV 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040219</fmx:line.date> 
					correction of patterns
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040223</fmx:line.date> 
					GR.ANNOTATION added to AGR, ANNEX and CJT
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040224</fmx:line.date> 
					correction of pattern definition for t_no.oj, t_page, t_page.sequence, AGE.OJ, CELL (@ACCH, @ACCV), ARTICLE (@IDENTIFIER), PARAG (@IDENTIFER)
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040225</fmx:line.date> 
					type of element PRINTER changed to xd:string
				</fmx:line.history>
				<!-- version 1.00 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20040302</fmx:line.date> 
					correction of model VOLUME
				</fmx:line.history>
				<fmx:line.history>ANNOTATION replaced by GR.ANNOTATION in model BIB.VOLUME</fmx:line.history>
				<fmx:line.history>ITEM.PUB/@DOC.INSTANCE, ITEM.REF/@REF.PDF, ITEM.REF/@REF.XML, INCL.ELEMENT/@FILEREF, DOCUMENT.REF/@FILE, PUBLICATION.REF/@FILE, REF.PHYS/@FILE: attribute type replaced byt xd:anyURI</fmx:line.history>
				<fmx:line.history><fmx:line.history>CONS.LIST/@LEG.VAL: alphabetical order of values</fmx:line.history><fmx:line.date>20040302</fmx:line.date> 
					correction of model VOLUME
				</fmx:line.history>
				<fmx:line.history>attributs NUMBERING, START, STEP deleted from element LIST</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040304</fmx:line.date> 
					type of attribute TEMPLATE modified to xd:anyURI
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040311</fmx:line.date> 
					type of element AUTHOR changed to xd:string
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040319</fmx:line.date> 
					t_doctype and DOCTYPE as well as all references deleted 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040322</fmx:line.date> 
					TBL added to model GR.NOTES 
				</fmx:line.history>
				<fmx:line.history>COLL added to DOC.CORR</fmx:line.history>
				<fmx:line.history>review of model FMX</fmx:line.history>
				<fmx:line.history>review of model GT.NOTES</fmx:line.history>
				<fmx:line.history>INC.ELEMENT added to model LETTER</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040325</fmx:line.date> 
					CONTENTS.CORR added to t_btx.struct, new element LV added to t_btx* 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040407</fmx:line.date> 
					element KEYWORD accepts HT 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040413</fmx:line.date> 
					correction of ambiguous model SUBDIV 
				</fmx:line.history>
				<fmx:line.history>GR.TBL added to t_btx.seq</fmx:line.history>
				<!-- version 1.01 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20040421</fmx:line.date> 
					TITLE deleted from VOLUME.PAPER 
				</fmx:line.history>
				<fmx:line.history>type of AGE.OJ.SUMMARY modified to t_btx</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040422</fmx:line.date> 
					gr.styles, attribute STYLE, attribute TEMPLATE deleted
				</fmx:line.history>
				<fmx:line.history>NO.DOC in BIB.DOC repeatable</fmx:line.history>
				<!-- version 1.02 = 2.00 .-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20040513</fmx:line.date> 
					revision of elements INCL.ELEMENT and IMG.CNT 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040526</fmx:line.date> 
					revision of element QUOT.S, QUOT.START and QUOT.END allowed
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040608</fmx:line.date> 
					attribute IDENTIFIER mandatory for ARTICLE and PARAG
				</fmx:line.history>
				<fmx:line.history>TBL allowed in IMG.CONT</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040720</fmx:line.date> 
					LV with mixed content model 
				</fmx:line.history>
				<fmx:line.history>attribute value EXPANDED for element HT (expanded character spacing)</fmx:line.history>
				<fmx:line.history>new element SPECIAL (in BIB.OJ and PUBLICATION.REF) for referring to special editions in MT or GA</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040823</fmx:line.date> 
					CONS.ANNEX added to t_btx.struct 
				</fmx:line.history>
				<fmx:line.history>type of element NO.OJ.SUMMARY changed to t_btx.seq</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040824</fmx:line.date> 
					value M added to t_no.oj 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040825</fmx:line.date> 
					new element BIB.INSTANCE.CONS in the context of codification 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040826</fmx:line.date> 
					attribute IN.SUMMARY added to DOC.MAIN.PUB 
				</fmx:line.history>
				<fmx:line.history>optional element NO.L2 added to VOLUME.ITEM</fmx:line.history>
				<fmx:line.history>DOCUMENT.REF.CONS added to BIB.INSTANCE.CONS</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040831</fmx:line.date> 
					generation of alternative structions within the element SPEC.ED 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040903</fmx:line.date> 
					elements DATE and YEAR alternatives in SPEC.ED 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040906</fmx:line.date> 
					type of element CONSID changed to t_btx.struct
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040914</fmx:line.date> 
					element BIB.INSTANCE added to CONS.DOC and CONS.OTH.DOC to mark up consolidated document 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20040929</fmx:line.date> 
					modification of rules of usage of attribute NO.SEQ for DOC.MAIN.PUB 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20041021</fmx:line.date> 
					values CH and IS added to element CRIT 
				</fmx:line.history>
				<!-- version 3.00 .-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20050103</fmx:line.date> 
					modification of IDENTIFIER attribute of ARTICLE and PARAG
				</fmx:line.history>
				<fmx:line.history>elements NOTE, QUOT.END, QUOT.START added to type t_btx.form</fmx:line.history>
				<fmx:line.history><fmx:line.date>20050321</fmx:line.date> 
					attribute values ROMAN/roman added to NOTE numbering scheme
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20050412</fmx:line.date> 
					suppression of INCLUSIONS from BIB.INSTANCE.CONS 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20050414</fmx:line.date> 
					TOC, ANNOTATION and GR.ANNOTATION added to CONS.DOC
				</fmx:line.history>
				<fmx:line.history>optional attribute ITEM.ID added to ITEM</fmx:line.history>
				<fmx:line.history>values BG/RO added to element CRIT</fmx:line.history>
				<fmx:line.history>specific elements for secondary legislation added (SE)</fmx:line.history>
				<fmx:line.history><fmx:line.date>20050427</fmx:line.date> 
					attribute IN.SUMMARY added to ITEM.REF
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20050525</fmx:line.date> 
					attribute NNC added to root elements and CONS.DOC, CONS.OTH.DOC, CONS.ANNEX
				</fmx:line.history>
				<fmx:line.history>attribute NCFOD added to element NOTE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20050627</fmx:line.date> 
					language attributes added for Basque (EU), Catalan (CA), and Galician (GL) 
				</fmx:line.history>
				<fmx:line.history>attribute NUMBERING.CONTINUED added to element NOTE</fmx:line.history>
				<fmx:line.history><fmx:line.date>20051010</fmx:line.date> 
					correction of element SE 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20051024</fmx:line.date> 
					correction of model SUBDIV 
				</fmx:line.history>
				<!-- Corrigendum .-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20060420</fmx:line.date> 
					attribute value for NO.SEQ and SYNOPTISM of DOC.MAIN.PUB and DOC.SUB.PUB modified to D\d{3,4}
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20060626</fmx:line.date> 
					empty element models revised (EEA, IE, OVER)
				</fmx:line.history>
				<!-- Version 4.00 .-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20060929</fmx:line.date> 
					addition of elements for CJ (Recueil): CASE, CONCLUSION, COSTS, COURT, DOC.CJT, DOC.VERSION, FINAL.CJT, FOOTING, HEADING, ID.DOC, INTRODUCTION, ITEM.LEFT, ITEM.RIGHT, METADATA, MOTIVES, PARTIES, PARTY PE, PROPERTIES, PROPERTY, RELATION, REPORT, ROLL, SUMMARY 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20061002</fmx:line.date> 
					revision of t_sectiontype 
				</fmx:line.history>
				<fmx:line.history>revision of SECTION.SUMMARY: TITLE optional</fmx:line.history>
				<fmx:line.history><fmx:line.date>20061005</fmx:line.date> 
					NO.L2 renamed to NO.DOC.TXT
				</fmx:line.history>
				<fmx:line.history>suppression of type t_format.no.doc</fmx:line.history>
				<fmx:line.history>SIGNATORY, SIGNATURE, SUBDIV added to QUOT.S</fmx:line.history>
				<fmx:line.history>attributes SYNOPTISM, NO.SEQ, REF.NO.SEQ on 4 positions if not defined else</fmx:line.history>
				<fmx:line.history>optional elements LIST and TBL added to DIV.CONSID and GR.CONSID</fmx:line.history>
				<fmx:line.history>NO.DOC in DOC.CORR multiple</fmx:line.history>
				<fmx:line.history>ITEM.PUB no longer allowed outside SECTION</fmx:line.history>
				<fmx:line.history>new optional element NOTICE added to VOLUME and SUMMARY.PDF</fmx:line.history>
				<fmx:line.history>optional GR.ANNOTATION added to SUMMARY.PDF</fmx:line.history>
				<fmx:line.history><fmx:line.date>20061010</fmx:line.date> 
					elements for CJ (Recueil) removed 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20070808</fmx:line.date> 
					elements for CJ (Recueil) 
				</fmx:line.history>
				<fmx:line.history><fmx:line.date>20071109</fmx:line.date> 
					revision of model for NO.ISBN 
				</fmx:line.history>
				<!-- Version 4.cj .-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:line.date>20080211</fmx:line.date> 
					elements for CJ (European Court Reports) added (only to be used in the context of instances related to European Court Reports): t_ecr.case, t_ecr.doc, t_ecr.pub, ABSTRACT, AGAINST, APPLICANT, ARGUMENTS, BIB.CASE, BIB.CONCLUSION, BIB.ECR, BIB.ECR.GENERAL, BIB.JDUGMENT, BIB.OPINION, BIB.ORDER, BIB.REPORT.HEARING, BIB.RULING, BIB.SUMMARY, CASE, CONCLUSION, CONTENTS.CONCLUSION, CONTENTS.JUDGMENT, CONTENTS.OPINION, CONTENTS.ORDER, CONTENTS.RULING, CONTENTS.SUMMARY.JUDGMENT, DEFENDANTS, ECR, ECR.GENERAL, INTERNAL.ANNEX, INTERVENERS, INTRO, INTRO.OPINION, ITEM.PART, JOINED.CASES, JUDGMENT, JUDGMENT.INIT, JURISDICTION, LG.CASE, NO.CASE, OPINION, OPINION.INIT, ORDER, ORDER.INIT, PART, PART.ECR, PARTIES, PLAINTIFS, PRELIM.RULING, PRESENCE, PRESENCE.INIT, REF.CASE, REF.CONCLUSION, REF.ECR, REF.ECR.DOC, REF.JUDGMENT, REF.NOTICE, REF.OPINION, REF.ORDER, REF.REPORT.HEARING, REF.REPORT.HEARING, REF.RULING, REF.SUMMARY.JUDGMENT, REPORT.HEARING, RULING, SIGNATURE.CASE, SUBJECT, SUBJECTS, SUMMARY, SUMMARY.JUDGMENT, TYPE.CASE, VOLUME.ECR 
				</fmx:line.history>
				<fmx:line.history><fmx:date>20080221</fmx:date> 
					ecr types modified 
				</fmx:line.history>
				<fmx:line.history>modification of elements ECR, CASE</fmx:line.history>
				<fmx:line.history>new elements: PDF.ECR, REF.PDF.ECR</fmx:line.history>
				<fmx:line.history><fmx:date>20081110</fmx:date> 
					new ECR root elements JUDGMENT.NP, ORDER.NP 
				</fmx:line.history>
				<fmx:line.history>modifications: t_ecr.case, t_ecr.doc, t_page, CASE</fmx:line.history>
				<fmx:line.history>new elements: REF.JUDGMENT.NP, REF.ORDER.NP, RELATED.MATERIAL, SUMMARY.NP</fmx:line.history>
				<!-- Version 5.00 .-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20080630</fmx:date> 
					integration of DOI (types, ITEM.REF, BIB.VOLUME)
				</fmx:line.history>
				<fmx:line.history>implementation of t_prod.id, t_fin.id, PROD.ID, FIN.ID</fmx:line.history>
				<fmx:line.history>IE added to t_btx.form</fmx:line.history>
				<fmx:line.history><fmx:date>20080701</fmx:date> 
					attribute YEAR added to REF.DOC.ECR
				</fmx:line.history>
				<fmx:line.history>optional element GR.ANNOTATION added root element (ACT, AGR, ANNEX, CJT, COMPETITION, CORR, GENERAL, LETTER)</fmx:line.history>
				<fmx:line.history>element NOTICE added to t_btx</fmx:line.history>
				<fmx:line.history>repeatable DATE in BIB.INSTANCE</fmx:line.history>
				<fmx:line.history>mandatory repeatable DATE ind DOC. MAIN.PUB</fmx:line.history>
				<fmx:line.history>attribute BOX added to GR.SEQ</fmx:line.history>
				<fmx:line.history><fmx:date>20080707</fmx:date> 
					revision of t_page.ecr, attributes EDITION and PAGE.SUM added to REF.DOC.ECR
				</fmx:line.history>
				<fmx:line.history><fmx:date>20081110</fmx:date> 
					new ECR root elements JUDGMENT.NP, ORDER.NP 
				</fmx:line.history>
				<fmx:line.history>modifications: t_ecr.case, t_ecr.doc, t_page, CASE</fmx:line.history>
				<fmx:line.history>new elements: REF.JUDGMENT.NP, REF.ORDER.NP, RELATED.MATERIAL, SUMMARY.NP</fmx:line.history>
				<fmx:line.history><fmx:date>20081204</fmx:date> 
					modificiation of ECR, CONTENTS.CORR, QUOT.S
				</fmx:line.history>
				<fmx:line.history>new element GR.CORRECTION</fmx:line.history>
				<fmx:line.history><fmx:date>20090108</fmx:date> 
					addition of elements to ECR context: APPEAL.TABLE, CHRON.TABLE, CHRON.TABLE.NP, REF.APPEAL.TABLE, REF.CHRON.TABLE, REF.CHRON.TABLE.NP, BIB.APPEAL.TABLE, BIB.CHRON.TABLE, BIB.CHRON.TABLE.NP, NO.CAT.GLOBAL
				</fmx:line.history>
				<fmx:line.history>review of t_page, t_ecr.doc</fmx:line.history>
				<fmx:line.history>redefinition of the element INTERVENERS</fmx:line.history>
				<fmx:line.history><fmx:date>20090130</fmx:date> 
					attribute value T1 suppressed from EDITION, GR.SEQ allowed after SIGNATURE.CASE in JUDGMENT
				</fmx:line.history>
				<!-- version 5.01 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20091103</fmx:date> 
					element DOI for general publication (to be used for General Report
				</fmx:line.history>
				<fmx:line.history><fmx:date>20091201</fmx:date> 
					revision of t_sectiontype
				</fmx:line.history>
				<!-- version 5.10 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20091215</fmx:date> 
					Revised pattern for t_date; simple type t_date.publication suppressed; revised pattern for t_page.ecr; new elements for t_sectiontype; new element NO.COM; revision of the elements CASE, ECR, GENERAL, KEYWORD, REF.DOC.ECR, REF.ECR.DOC, TBL, VOLUME, DECISION, OBJECT
				</fmx:line.history>
				<!-- version 5.11 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20100107</fmx:date> 
					ISBN and ISSN in BIB.GEN.PUB both possible 
				</fmx:line.history>
				<fmx:line.history><fmx:date>20100119</fmx:date> 
					redefinition of TYPE.CASE
				</fmx:line.history>
				<fmx:line.history><fmx:date>20100128</fmx:date> 
					redefinition of t_date; redefinition of HEADER.SUMARY.SE; definition of SERIES.SE and SUBSERIES.SE
				</fmx:line.history>
				<fmx:line.history><fmx:date>20100129</fmx:date> 
					redefinition of t_ecr...; new element REF.CASE.F1
				</fmx:line.history>
				<!-- version 5.20 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20100526</fmx:date> 
					integration of Croatian and Icelandic, ISSN and DOI in HEADER.SUMMARY.SE optional; DOI in SUMMARY.PDF; filenaming with U (urgent) OJ publications
				</fmx:line.history>
				<fmx:line.history><fmx:date>20100826</fmx:date> 
					integration of new elements for the handlng of Court decisions (DECISION.ECR) 
				</fmx:line.history>
				<fmx:line.history><fmx:date>20101027</fmx:date> 
					integration of NO.DOI in BIB.SE for secondary legislation 
				</fmx:line.history>
				<fmx:line.history><fmx:date>20110125</fmx:date> 
					suppression of U in filenames 
				</fmx:line.history>
				<!-- version 5.21 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20110407</fmx:date> 
					harmonisation of ISSN type (t_ISSN); integration of NO.ISSN.ELECTRONIC and NO.ISSN.PRINT; updated elements: BIB.SE, BIB.VOLUME, HEADER.SUMMARY, HEADER.SUMMARY.SE
				</fmx:line.history>
				<fmx:line.history><fmx:date>20110519</fmx:date> 
					element COMMENT added to ARTICLE, PARAG, SUBDIV, t_btx, t_btx.struct 
				</fmx:line.history>
				<!-- version 5.22 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20110810</fmx:date> 
					new attribute values for MOD.ACT/@TYPE, MOD.ACT.CL/@TYPE and INFO.MOD/@ACTION
				</fmx:line.history>
				<!-- version 5.23 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20111121</fmx:date> 
					extension of DOI pattern for documents in the context of secondary legislation 
				</fmx:line.history>
				<!-- version 5.24 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.- -->
				<fmx:line.history><fmx:date>20120810</fmx:date>new country list for CRIT/@COUNTRY</fmx:line.history>
				<fmx:line.history>ISSN.PRINT optional</fmx:line.history>
				<fmx:line.hisotry>New simple types t_charcode, t_quotation.end, t_quotation.start for encoding of quotation marks in QUOT.START and QUOT.END</fmx:line.hisotry>
				<!-- version 5.30 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.- -->
				<fmx:line.history><fmx:date>20120627</fmx:date>new document type LSEU (Legislative Summary EU), new elements BIB.LSEU, LAST.MODIFICATION, CONTENTS.LSEU, HEADER.LSEU, REFERENCES.LSEU, MENU.LSEU, SUMMARY.LSEU, REFERRED.DOCS.LSEU, ARCHIVED; REF.DOC added to t_btx and t_btx.seq, NO.DOC.LSEU, ARCHIVED, t_no.doc.lseu</fmx:line.history>
				<!-- version 5.50 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20121002</fmx:date> 
					new type definitions: t_ref.phys used by REF.PHYS, ASSOCIATED.TO, ASSOCIATES, t_volume used by attribute VOLUME of REF.DOC.OJ 
				</fmx:line.history>
				<fmx:line.history>new elements: ASSOCIATED.TO, ASSOCIATION, REF.BIB.RECORD, REF.CORE.METADATA</fmx:line.history>
				<fmx:line.history>modified elements: BIB.DOC, CRIT, DOC.MAIN.PUB, REF.DOC.OJ, REF.PHYS </fmx:line.history>
				<fmx:line.history><fmx:date>20121010</fmx:date> new model for formula: 
				</fmx:line.history>
				<fmx:line.history>suppression: t_btx.form, EXPR, IND, OVER, ROOT, FORMULA, FORMULA.S</fmx:line.history>
				<fmx:line.history>new: t_btx.formula, BAR, DEGREE, DIVIDEND, DIVISOR, EXPONENT, EXPR, FMT.VALUE, FORMULA, FRACTION, FUNCTION, IND, INTEGRAL, OP.CMP, OP.MATH, OVER, OVERLINE, PRODUCT, ROOT, SUM, VECTOR</fmx:line.history>
				<fmx:line.history><fmx:date>20121016</fmx:date>correction in CHRON.TABLE.NP:
				</fmx:line.history>
				<fmx:line.history><fmx:date>20121017</fmx:date>correction in DECISION:
				</fmx:line.history>
				<fmx:line.history><fmx:date>20121031</fmx:date>integration of LSEU elements: t_no.doc.lseu,  BIB.LSEU, CONTENTS.LSEU, HEADER.LSEU, LAST.MODIFICATION, LSEU, MENU.LSEU, OBSOLETE, REF.DOC, REFERENCES.LSEU, REFERRED.DOCS.LSEU, SUMMARY.LSEU 
				</fmx:line.history>
				<fmx:line.history><fmx:date>20121204</fmx:date>NO.ECLI introduced 
				</fmx:line.history>
				<!-- version 5.50 corrected-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20130218</fmx:date>correction: REF.BIB.RECORD and REF.CORE.METADATA added to BIB.DOC and BIB.GEN.PUB, removed from BIB.OPINION </fmx:line.history>
				<!-- version 5.51 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20130227</fmx:date>t_ref.ecr and t_ref.case: acceptance of value "n.a.", references from CASE to documents optional; integration of LSEU elements from 5.30 </fmx:line.history>
				<fmx:line.history><fmx:date>20130228</fmx:date>REF.BIB.RECORD and REF.CORE.METADATA added to BIB.APPEAL.TABLE, BIB.CASE, BIB.CONCLUSION, BIB.CORRIG.ECR, BIB.DECISION, BIB.JUDGMENT, BIB.OPINION, BIB.ORDER, BIB.REPORT.HEARING, BIB.RULING </fmx:line.history>
				<fmx:line.history><fmx:date>20130412</fmx:date>element DATE added to BIB.SE </fmx:line.history>
				<!-- version 5.52 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20130508</fmx:date>NO.DOC prepared for unique identifier (documentation), REF.CORE.METADATA and REF.BIB.RECORD added to BIB.OJ and HEADER.SUMMARY; TYPE.CASE prepared for accepting DECISION (documentation) </fmx:line.history>
				<fmx:line.history><fmx:date>20130516</fmx:date>FORMULA completed by generic element MATH.SYMBOL </fmx:line.history>
				<!-- version 5.53 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20130725</fmx:date>update of LSEU models due to legacy data </fmx:line.history>
				<fmx:line.history>LEFT/RIGHT in jurisdiction instances completed by a common page header (PAGE.HEADER)</fmx:line.history>
				<fmx:line.history>update of LSEU content model</fmx:line.history>
				<fmx:line.history><fmx:date>20130819</fmx:date>NO.ECLI added to all document types relevant for CJ </fmx:line.history>
				<!-- version 5.53L -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20140304</fmx:date>new pattern t_no.doc.lseu</fmx:line.history>
				<!-- version 5.54 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20130807</fmx:date>delete value F2 in t_ecr.case, t_ecr.doc and t_ecr.pub</fmx:line.history>
				<fmx:line.history><fmx:date>20131029</fmx:date>elements added to t_btx_seq for the markup of references to Court documents: NO.CASE, NO.ECLI, POINT</fmx:line.history>
				<fmx:line.history><fmx:date>20140313</fmx:date>adde attribute ECLI to NO.ECLI</fmx:line.history>
				<fmx:line.history><fmx:date>20140506</fmx:date>model of element REF.DOC.ECR modified; new elements NP.ECR, NAME.CASE, REF.NP.ECR</fmx:line.history>
				<!-- version 5.55 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20141027</fmx:date>changes to t_no.oj due to new OJ series</fmx:line.history>
				<fmx:line.history><fmx:date>20141030</fmx:date>changes to NO.DOC according to unique numbering</fmx:line.history>
				<fmx:line.history><fmx:date>20150513</fmx:date>Correction: NO.ECLI = extension of t_btx_seq; REF.DOC.ECR: HT possible in the context of old citation mode</fmx:line.history>
				<!-- version 5.56 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20160531</fmx:date>demand from B1: anonymisation of documents published in OJ (personal data) introduction of element ANONYMOUS</fmx:line.history>
				<fmx:line.history><fmx:date>20160602</fmx:date>added attribute REPUBLICATION (type t_boolean) to element BIB.OJ</fmx:line.history>
				<fmx:line.history><fmx:date>20160606</fmx:date>added attribute TYPE (GLOSSARY|SUMMARY) to BIB.LSEU</fmx:line.history>
				<!-- version 5.57 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20160914</fmx:date>following the entering into force of the new version of the vademecum of CJ: integration of new element INTERMEDIATE, modification, through integration of INTERMEDIATE, of elements JUDGMENT, OPINION, ORDER, CONCLUSION, added optional TOC to JUDGMENT.NP</fmx:line.history>
				<!-- version 5.58 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20161020</fmx:date>attributes SEPARATOR, IDX.OPEN, IDX.CLOSE added to the elements INDEX and SUBJECTS, value NDASH added to TYPE attribute of element LIST
				</fmx:line.history>
				<fmx:line.history><fmx:date>20161026</fmx:date>introduction of element BR (line break) due to special constraints in case law documents
				</fmx:line.history>
				<fmx:line.history><fmx:date>20170314</fmx:date>new value for TYPE attribute of HT: "SC" (small-caps) necessary in case law documents
				</fmx:line.history>
				<!-- version 5.59 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20170418</fmx:date>attribute value "INTRO" added to BIB.LSEU@TYPE, modification of pattern for element NO.DOC.LSEU
				</fmx:line.history>
				<fmx:line.history><fmx:date>20190401</fmx:date>new attribute value "0022" added to simpleType "t_quotation.end" and "t_quotation.start", for new character of quotation mark used in Dutch
				</fmx:line.history>
				<!-- version 6.00 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20210715</fmx:date>new schema for the production AbA
				</fmx:line.history>
				<!-- version 6.01 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20220520</fmx:date>new element NO.DOC added as optional in DOCUMENT.REF.CONS
				</fmx:line.history>
				<fmx:line.history>new element NO.ELI added as optional in DOCUMENT.REF.CONS</fmx:line.history>
				<fmx:line.history>added two new unicode characters "0406" and "0456" to the list allowed characters</fmx:line.history>
				<fmx:line.history>new element LINK created to be used for any uri link</fmx:line.history>
				<fmx:line.history>new element REF.DOC.SE created to be used for references to special editions</fmx:line.history>
				<fmx:line.history>creation of type t_chapter, needed for element REF.DOC.SE</fmx:line.history>
				<fmx:line.history>new pattern of type t_volume, needed for element REF.DOC.SE</fmx:line.history>
				<fmx:line.history>added new elements LINK and REF.DOC.SE inside type t_btx</fmx:line.history>
				<fmx:line.history>added new elements LINK and REF.DOC.SE inside type t_btx.seq</fmx:line.history>
				<!-- version 6.02 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20230206</fmx:date>new character added in allowed character ranges: unicode 25A1 for white square
				</fmx:line.history>
				<!-- version 6.02.1 -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. -->
				<fmx:line.history><fmx:date>20231031</fmx:date>Introduction of new type of document, SUMMARY and RESUME</fmx:line.history>
				<fmx:line.history>new element TITLE added as optional in INDEX</fmx:line.history>
				<fmx:line.history>element TITLE in SUMMARY made as optional</fmx:line.history>
				<fmx:line.history>RESUME added as value REF.PDF.ECR element</fmx:line.history>
				<fmx:line.history>DATE.PUB no longer mandatory for REF.DOC.SE element</fmx:line.history>
			</fmx:history>
		</xd:documentation>
	</xd:annotation><!-- -.-.-.-. Type definitions -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. --><xd:simpleType name="t_area">
		<xd:restriction base="xd:string">
			<xd:enumeration value="EFTA"/>
			<xd:enumeration value="EU"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_boolean">
		<xd:restriction base="xd:string">
			<xd:enumeration value="NO"/>
			<xd:enumeration value="YES"/>
		</xd:restriction>
	</xd:simpleType><xd:complexType name="t_btx" mixed="true">
		<xd:sequence>
			<xd:element ref="MARGIN" minOccurs="0"/>
			<xd:choice minOccurs="0" maxOccurs="unbounded">
				<xd:element ref="ACT.GEN"/>
				<xd:element ref="ADDR"/>
				<xd:element ref="ADDR.S"/>
				<xd:element ref="ALINEA"/>
				<xd:element ref="ANNOTATION"/>
				<xd:element ref="ANONYMOUS"/>
				<xd:element ref="ARTICLE"/>
				<xd:element ref="BR"/>
				<xd:element ref="CAT.PLANT"/>
				<xd:element ref="CAT.VEGET"/>
				<xd:element ref="CN"/>
				<xd:element ref="COLL.LETTERS"/>
				<xd:element ref="COMMENT"/>
				<xd:element ref="COMPETITION.INIT"/>
				<xd:element ref="CONSID"/>
				<xd:element ref="CONTENTS"/>
				<xd:element ref="DATE"/>
				<xd:element ref="DIV.CONSID"/>
				<xd:element ref="DIVISION"/>
				<xd:element ref="DLIST"/>
				<xd:element ref="DLIST.ITEM"/>
				<xd:element ref="ENACTING.TERMS"/>
				<xd:element ref="ENACTING.TERMS.CJT"/>
				<xd:element ref="FINAL"/>
				<xd:element ref="FORMULA"/>
				<xd:element ref="FT"/>
				<xd:element ref="FULL.POWER"/>
				<xd:element ref="FULL.POWER.FINAL"/>
				<xd:element ref="FULL.POWER.INIT"/>
				<xd:element ref="GR.AMEND"/>
				<xd:element ref="GR.ANNOTATION"/>
				<xd:element ref="GR.CONSID"/>
				<xd:element ref="GR.CONSID.INIT"/>
				<xd:element ref="GR.NOTES"/>
				<xd:element ref="GR.SEQ"/>
				<xd:element ref="GR.TBL"/>
				<xd:element ref="GR.VISA"/>
				<xd:element ref="GR.VISA.INIT"/>
				<xd:element ref="HT"/>
				<xd:element ref="IE"/>
				<xd:element ref="INCL.ELEMENT"/>
				<xd:element ref="INDEX"/>
				<xd:element ref="ITEM"/>
				<xd:element ref="KEYWORD"/>
				<xd:element ref="LETTER"/>
				<xd:element ref="LINK"/>
				<xd:element ref="LIST"/>
				<xd:element ref="LIST.AMEND"/>
				<xd:element ref="LV"/>
				<xd:element ref="NOTE"/>
				<xd:element ref="NP"/>
				<xd:element ref="P"/>
				<xd:element ref="PARAG"/>
				<xd:element ref="PL.DATE"/>
				<xd:element ref="PLENIPOTENTIARY"/>
				<xd:element ref="PREAMBLE"/>
				<xd:element ref="PREAMBLE.FINAL"/>
				<xd:element ref="PREAMBLE.GEN"/>
				<xd:element ref="PREAMBLE.INIT"/>
				<xd:element ref="QUOT.END"/>
				<xd:element ref="QUOT.S"/>
				<xd:element ref="QUOT.START"/>
				<xd:element ref="REF.DOC"/>
				<xd:element ref="REF.DOC.ECR"/>
				<xd:element ref="REF.DOC.OJ"/>
				<xd:element ref="REF.DOC.SE"/>
				<xd:element ref="STI.ART"/>
				<xd:element ref="TBL"/>
				<xd:element ref="TI.ART"/>
				<xd:element ref="TI.CJT"/>
				<xd:element ref="TITLE"/>
				<xd:element ref="TOC"/>
				<xd:element ref="TXT"/>
				<xd:element ref="VISA"/>
			</xd:choice>
		</xd:sequence>
	</xd:complexType><xd:complexType name="t_btx.ecr" mixed="true">
		<xd:sequence>
			<xd:element ref="MARGIN" minOccurs="0"/>
			<xd:choice minOccurs="0" maxOccurs="unbounded">
				<xd:element ref="ADDR"/>
				<xd:element ref="ADDR.S"/>
				<xd:element ref="ALINEA"/>
				<xd:element ref="ANNOTATION"/>
				<xd:element ref="ANONYMOUS"/>
				<xd:element ref="ARTICLE"/>
				<xd:element ref="BR"/>
				<xd:element ref="COLL.LETTERS"/>
				<xd:element ref="COMMENT"/>
				<xd:element ref="CONSID"/>
				<xd:element ref="CONTENTS"/>
				<xd:element ref="DATE"/>
				<xd:element ref="DIV.CONSID"/>
				<xd:element ref="DIVISION"/>
				<xd:element ref="DLIST"/>
				<xd:element ref="DLIST.ITEM"/>
				<xd:element ref="ENACTING.TERMS"/>
				<xd:element ref="ENACTING.TERMS.CJT"/>
				<xd:element ref="FINAL"/>
				<xd:element ref="FORMULA"/>
				<xd:element ref="FT"/>
				<xd:element ref="GR.ANNOTATION"/>
				<xd:element ref="GR.CONSID"/>
				<xd:element ref="GR.CONSID.INIT"/>
				<xd:element ref="GR.NOTES"/>
				<xd:element ref="GR.SEQ"/>
				<xd:element ref="GR.TBL"/>
				<xd:element ref="GR.VISA"/>
				<xd:element ref="GR.VISA.INIT"/>
				<xd:element ref="HT"/>
				<xd:element ref="IE"/>
				<xd:element ref="INCL.ELEMENT"/>
				<xd:element ref="INDEX"/>
				<xd:element ref="ITEM"/>
				<xd:element ref="KEYWORD"/>
				<xd:element ref="LETTER"/>
				<xd:element ref="LIST"/>
				<xd:element ref="LIST.AMEND"/>
				<xd:element ref="LV"/>
				<xd:element ref="NOTE"/>
				<xd:element ref="NP"/>
				<xd:element ref="NP.ECR"/>
				<xd:element ref="P"/>
				<xd:element ref="PARAG"/>
				<xd:element ref="PL.DATE"/>
				<xd:element ref="PREAMBLE"/>
				<xd:element ref="PREAMBLE.FINAL"/>
				<xd:element ref="PREAMBLE.GEN"/>
				<xd:element ref="PREAMBLE.INIT"/>
				<xd:element ref="QUOT.END"/>
				<xd:element ref="QUOT.S"/>
				<xd:element ref="QUOT.START"/>
				<xd:element ref="REF.DOC"/>
				<xd:element ref="REF.DOC.ECR"/>
				<xd:element ref="REF.DOC.OJ"/>
				<xd:element ref="STI.ART"/>
				<xd:element ref="TBL"/>
				<xd:element ref="TI.ART"/>
				<xd:element ref="TI.CJT"/>
				<xd:element ref="TITLE"/>
				<xd:element ref="TOC"/>
				<xd:element ref="TXT"/>
				<xd:element ref="VISA"/>
			</xd:choice>
		</xd:sequence>
	</xd:complexType><xd:complexType name="t_btx.formula" mixed="true">
		<xd:sequence>
			<xd:element ref="MARGIN" minOccurs="0"/>
			<xd:choice minOccurs="0" maxOccurs="unbounded">
				<xd:element ref="BAR"/>
				<xd:element ref="EXPONENT"/>
				<xd:element ref="EXPR"/>
				<xd:element ref="FMT.VALUE"/>
				<xd:element ref="FRACTION"/>
				<xd:element ref="FUNCTION"/>
				<xd:element ref="HT"/>
				<xd:element ref="IE"/>
				<xd:element ref="IND"/>
				<xd:element ref="INTEGRAL"/>
				<xd:element ref="NOTE"/>
				<xd:element ref="OP.CMP"/>
				<xd:element ref="OP.MATH"/>
				<xd:element ref="OVERLINE"/>
				<xd:element ref="PRODUCT"/>
				<xd:element ref="QUOT.END"/>
				<xd:element ref="QUOT.START"/>
				<xd:element ref="ROOT"/>
				<xd:element ref="SUM"/>
				<xd:element ref="VECTOR"/>
			</xd:choice>
		</xd:sequence>
	</xd:complexType><xd:complexType name="t_btx.seq" mixed="true">
		<xd:sequence>
			<xd:element ref="MARGIN" minOccurs="0"/>
			<xd:choice minOccurs="0" maxOccurs="unbounded">
				<xd:element ref="ADDR"/>
				<xd:element ref="ADDR.S"/>
				<xd:element ref="ANONYMOUS"/>
				<xd:element ref="BR"/>
				<xd:element ref="DATE"/>
				<xd:element ref="DLIST"/>
				<xd:element ref="FORMULA"/>
				<xd:element ref="FT"/>
				<xd:element ref="GR.TBL"/>
				<xd:element ref="HT"/>
				<xd:element ref="IE"/>
				<xd:element ref="INCL.ELEMENT"/>
				<xd:element ref="INDEX"/>
				<xd:element ref="LINK"/>
				<xd:element ref="LIST"/>
				<xd:element ref="LV"/>
				<xd:element ref="NO.CASE"/>
				<xd:element ref="NO.ECLI"/>
				<xd:element ref="NOTE"/>
				<xd:element ref="REF.DOC"/>
				<xd:element ref="REF.DOC.ECR"/>
				<xd:element ref="REF.DOC.OJ"/>
				<xd:element ref="REF.DOC.SE"/>
				<xd:element ref="REF.NP.ECR"/>
				<xd:element ref="QUOT.END"/>
				<xd:element ref="QUOT.S"/>
				<xd:element ref="QUOT.START"/>
				<xd:element ref="TBL"/>
				<xd:element ref="TXT"/>
			</xd:choice>
		</xd:sequence>
	</xd:complexType><xd:complexType name="t_btx.struct">
		<xd:choice minOccurs="0" maxOccurs="unbounded">
			<xd:element ref="ACT.GEN"/>
			<xd:element ref="ADDR.S"/>
			<xd:element ref="ALINEA"/>
			<xd:element ref="ANNOTATION"/>
			<xd:element ref="ARTICLE"/>
			<xd:element ref="CAT.PLANT"/>
			<xd:element ref="CAT.VEGET"/>
			<xd:element ref="CN"/>
			<xd:element ref="COLL.LETTERS"/>
			<xd:element ref="COMMENT"/>
			<xd:element ref="COMPETITION.INIT"/>
			<xd:element ref="CONS.ANNEX"/>
			<xd:element ref="CONSID"/>
			<xd:element ref="CONTENTS"/>
			<xd:element ref="CONTENTS.CORR"/>
			<xd:element ref="DIV.CONSID"/>
			<xd:element ref="DIVISION"/>
			<xd:element ref="DLIST"/>
			<xd:element ref="DLIST.ITEM"/>
			<xd:element ref="ENACTING.TERMS"/>
			<xd:element ref="ENACTING.TERMS.CJT"/>
			<xd:element ref="FINAL"/>
			<xd:element ref="FORMULA"/>
			<xd:element ref="FULL.POWER"/>
			<xd:element ref="FULL.POWER.FINAL"/>
			<xd:element ref="FULL.POWER.INIT"/>
			<xd:element ref="GR.AMEND"/>
			<xd:element ref="GR.ANNOTATION"/>
			<xd:element ref="GR.CONSID"/>
			<xd:element ref="GR.CONSID.INIT"/>
			<xd:element ref="GR.NOTES"/>
			<xd:element ref="GR.SEQ"/>
			<xd:element ref="GR.TBL"/>
			<xd:element ref="GR.VISA"/>
			<xd:element ref="GR.VISA.INIT"/>
			<xd:element ref="INCL.ELEMENT"/>
			<xd:element ref="INDEX"/>
			<xd:element ref="ITEM"/>
			<xd:element ref="LETTER"/>
			<xd:element ref="LIST"/>
			<xd:element ref="LIST.AMEND"/>
			<xd:element ref="LV"/>
			<xd:element ref="NOTE"/>
			<xd:element ref="NP"/>
			<xd:element ref="NP.ECR"/>
			<xd:element ref="P"/>
			<xd:element ref="PARAG"/>
			<xd:element ref="PL.DATE"/>
			<xd:element ref="PLENIPOTENTIARY"/>
			<xd:element ref="PREAMBLE"/>
			<xd:element ref="PREAMBLE.FINAL"/>
			<xd:element ref="PREAMBLE.GEN"/>
			<xd:element ref="PREAMBLE.INIT"/>
			<xd:element ref="QUOT.S"/>
			<xd:element ref="REFERRED.DOCS.LSEU"/>
			<xd:element ref="STI.ART"/>
			<xd:element ref="TARIC"/>
			<xd:element ref="TBL"/>
			<xd:element ref="TI.ART"/>
			<xd:element ref="TI.CJT"/>
			<xd:element ref="TITLE"/>
			<xd:element ref="TOC"/>
			<xd:element ref="VISA"/>
		</xd:choice>
	</xd:complexType><xd:complexType name="t_btx.title">
		<xd:choice minOccurs="0" maxOccurs="unbounded">
			<xd:element ref="LIST"/>
			<xd:element ref="NO.DOC.C"/>
			<xd:element ref="NP"/>
			<xd:element ref="P"/>
		</xd:choice>
	</xd:complexType><xd:simpleType name="t_chapter">
		<xd:restriction base="xd:string">
			<xd:pattern value="[1-9][0-9]?|([ilvxILVX]+)"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_charcode">
		<xd:restriction base="xd:hexBinary">
			<xd:pattern value="[0-9A-F]{4}"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_collection">
		<xd:restriction base="xd:string">
			<xd:enumeration value="A"/>
			<xd:enumeration value="C"/>
			<xd:enumeration value="L"/>
			<xd:enumeration value="P"/>
			<xd:enumeration value="S"/>
			<xd:enumeration value="X"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_date">
		<xd:restriction base="xd:string">
			<xd:pattern value="\d{4}(\d{2}(\d{2})?)?"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_doi.doc">
		<xd:restriction base="xd:string">
			<xd:pattern value="doi:10\.\d{4}/[ACLP][_AEGM]\d{4}\.\d{3}\.\d{2}(\d|c|e|s)\d{3}\d{2}\.[a-z]{3}"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_doi.gen">
		<xd:restriction base="xd:string">
			<xd:pattern value="doi:10\.\d{4}/.+"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_doi.oj">
		<xd:restriction base="xd:string">
			<xd:pattern value="doi:10\.\d{4}/\d{7}(\d|X)\.[CL][_AEGM]\d{4}\.\d{3}\.[a-z]{3}"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_ecr.case">
		<xd:restriction base="xd:string">
			<xd:pattern value="(ECR(CJ|T1|F1|F2)\d{4}[A-Z]{2}[ABC][12]?\.\d{2}[ars\d]\d{5}\d{2}\.case\.xml)|(n\.a\.)|(ECR_\d{5}[A-Z]{2}\d{4}_SUM_[A-Z]{2}_\d{2}\.xml)|(n\.a\.)"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_ecr.doc">
		<xd:restriction base="xd:string">
			<xd:pattern value="(ECR(CJ|T1|F1)\d{4}[A-Z]{2}[ABC][12]?\.\d{2}[aprs\d]\d{5}\d{2}\.xml)|(ECR_\d{5}[A-Z]{2}\d{4}(_INF)?_[A-Z]{2}_\.xml)"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_ecr.pub">
		<xd:restriction base="xd:string">
			<xd:pattern value="(ECR(CJ|T1|F1)\d{4}[A-Z]{2}[ABC][12]?\.\d{2}\.xml)|(n\.a\.)"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_fin.id">
		<xd:restriction base="xd:string">
			<xd:pattern value="\d{1,38}"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_ISSN">
		<xd:restriction base="xd:string">
			<xd:pattern value="\d{7}(\d|X)"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_language">
		<xd:annotation>
			<xd:documentation>based on ISO 639-1 (ISO 639-3 as comments)</xd:documentation>
		</xd:annotation>
		<xd:restriction base="xd:string">
			<xd:enumeration value="BG"/>
			<xd:enumeration value="CA"/>
			<xd:enumeration value="CS"/>
			<xd:enumeration value="DA"/>
			<xd:enumeration value="DE"/>
			<xd:enumeration value="EL"/>
			<xd:enumeration value="EN"/>
			<xd:enumeration value="ES"/>
			<xd:enumeration value="ET"/>
			<xd:enumeration value="EU"/>
			<xd:enumeration value="FI"/>
			<xd:enumeration value="FR"/>
			<xd:enumeration value="GA"/>
			<xd:enumeration value="GL"/>
			<xd:enumeration value="HR"/>
			<xd:enumeration value="HU"/>
			<xd:enumeration value="IS"/>
			<xd:enumeration value="IT"/>
			<xd:enumeration value="LT"/>
			<xd:enumeration value="LV"/>
			<xd:enumeration value="MT"/>
			<xd:enumeration value="NL"/>
			<xd:enumeration value="NO"/>
			<xd:enumeration value="PL"/>
			<xd:enumeration value="PT"/>
			<xd:enumeration value="RO"/>
			<xd:enumeration value="SK"/>
			<xd:enumeration value="SL"/>
			<xd:enumeration value="SV"/>
			<xd:enumeration value="TR"/>
			<xd:enumeration value="XM"/>
			<xd:enumeration value="XX"/>
			<!-- bul -->
			<!-- cat -->
			<!-- ces -->
			<!-- dan -->
			<!-- deu -->
			<!-- ell -->
			<!-- eng -->
			<!-- spa -->
			<!-- est -->
			<!-- eus -->
			<!-- fin -->
			<!-- fra -->
			<!-- gle -->
			<!-- gle -->
			<!-- hrv -->
			<!-- hun -->
			<!-- isl -->
			<!-- ita -->
			<!-- lit -->
			<!-- lav -->
			<!-- mlt -->
			<!-- nld -->
			<!-- nor -->
			<!-- pol -->
			<!-- por -->
			<!-- ron -->
			<!-- slk -->
			<!-- slv -->
			<!-- swe -->
			<!-- tur -->
			<!-- mul -->
			<!-- multilingual: not ISO -->
			<!-- und -->
			<!-- other language: not ISO -->
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_no.doc.lseu">
		<xd:restriction base="xd:string">
			<xd:pattern value="LSEU\-((GLOSSARY\-[A-Z0-9_]+)|(INTRO\-(\d{2})+)|([A-Z]([A-Z]|\d)\d{4}[A-Z]?)|(\d{2}(\d{2}){0,5}_[1-9]\d{0,3})|([4-9](\d{4}|\d{6})))\-\d{4}\-\d{2}\-\d{2}\-\d{4}"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_no.oj">
		<xd:restriction base="xd:string">
			<xd:pattern value="\d{3}[AEIM]?"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_page">
		<xd:restriction base="xd:string">
			<xd:pattern value="([ceps]{0,1}\-{0,1}\d{1,5}[a-z*]?(/\d{2})?)|([cdilvxCDILVX]+)"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_page.ecr">
		<xd:restriction base="xd:string">
			<xd:pattern value="[ceps]{0,1}(([I]{1,3}\-)?([ABP]\-(\d{1,2}\-)?)?)?[\dcdilvxCDILVX]+\*?(;(([I]{1,3}\-)?([ABP]\-(\d{1,2}\-)?)?)?[\dcdilvxCDILVX]+\*?)*"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_page.sequence">
		<xd:restriction base="xd:string">
			<xd:pattern value="[1-9][0-9]?"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_prod.id">
		<xd:restriction base="xd:string">
			<xd:pattern value="\d{11}"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_quotation.end">
		<xd:restriction base="t_charcode">
			<xd:enumeration value="0022"/>
			<xd:enumeration value="00AB"/>
			<xd:enumeration value="00BB"/>
			<xd:enumeration value="2018"/>
			<xd:enumeration value="2019"/>
			<xd:enumeration value="201C"/>
			<xd:enumeration value="201D"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_quotation.start">
		<xd:restriction base="t_charcode">
			<xd:enumeration value="0022"/>
			<xd:enumeration value="00AB"/>
			<xd:enumeration value="00BB"/>
			<xd:enumeration value="2018"/>
			<xd:enumeration value="2019"/>
			<xd:enumeration value="201A"/>
			<xd:enumeration value="201C"/>
			<xd:enumeration value="201D"/>
			<xd:enumeration value="201E"/>
		</xd:restriction>
	</xd:simpleType><xd:complexType name="t_ref.phys">
		<xd:attribute name="FILE" type="xd:anyURI" use="required"/>
		<xd:attribute name="TYPE" use="required">
			<xd:simpleType>
				<xd:restriction base="xd:string">
					<xd:enumeration value="DOC.PDF"/>
					<xd:enumeration value="DOC.XML"/>
					<xd:enumeration value="FRAGMENT.XML"/>
					<xd:enumeration value="IMAGE"/>
				</xd:restriction>
			</xd:simpleType>
		</xd:attribute>
		<xd:attribute name="IMAGE.TYPE">
			<xd:simpleType>
				<xd:restriction base="xd:string">
					<xd:enumeration value="TIFF"/>
				</xd:restriction>
			</xd:simpleType>
		</xd:attribute>
	</xd:complexType><xd:simpleType name="t_sectiontype">
		<xd:restriction base="xd:string">
			<xd:enumeration value="C0"/>
			<xd:enumeration value="C1"/>
			<xd:enumeration value="C2"/>
			<xd:enumeration value="C3"/>
			<xd:enumeration value="C4"/>
			<xd:enumeration value="C5"/>
			<xd:enumeration value="CC"/>
			<xd:enumeration value="CI"/>
			<xd:enumeration value="CX"/>
			<xd:enumeration value="L0"/>
			<xd:enumeration value="L1"/>
			<xd:enumeration value="L2"/>
			<xd:enumeration value="L3"/>
			<xd:enumeration value="L4"/>
			<xd:enumeration value="L5"/>
			<xd:enumeration value="LC"/>
			<xd:enumeration value="LI"/>
			<xd:enumeration value="LX"/>
		</xd:restriction>
	</xd:simpleType><xd:simpleType name="t_volume">
		<xd:restriction base="xd:string">
			<xd:pattern value="[1-9][0-9]?|([ilvxILVX]+)"/>
		</xd:restriction>
	</xd:simpleType><!-- -.-.-.-. Element definitions .-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-. --><xd:element name="ABSTRACT" type="t_btx.struct"/><xd:element name="ACT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.INSTANCE"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="TOC"/>
				</xd:choice>
				<xd:element ref="TITLE"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="PROLOG"/>
					<xd:element ref="TOC"/>
				</xd:choice>
				<xd:element ref="PREAMBLE"/>
				<xd:element ref="ENACTING.TERMS"/>
				<xd:element ref="FINAL" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="ACT.GEN">
		<xd:complexType>
			<xd:sequence>
				<xd:choice minOccurs="0">
					<xd:element ref="PREAMBLE.GEN"/>
					<xd:element ref="P" maxOccurs="unbounded"/>
				</xd:choice>
				<xd:element ref="TOC" minOccurs="0"/>
				<xd:element ref="ENACTING.TERMS"/>
				<xd:element ref="FINAL" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="ADDITIONAL.DOCS.LSEU" type="t_btx.struct"/><xd:element name="ADDR" type="t_btx.seq"/><xd:element name="ADDR.S">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="P" minOccurs="2" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="AGAINST" type="t_btx.seq"/><xd:element name="AGE.OJ">
		<xd:simpleType>
			<xd:restriction base="xd:string">
				<xd:pattern value="[1-9][0-9]?"/>
			</xd:restriction>
		</xd:simpleType>
	</xd:element><xd:element name="AGE.OJ.SUMMARY" type="t_btx"/><xd:element name="AGR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.INSTANCE"/>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="TITLE"/>
					<xd:element ref="TOC"/>
				</xd:choice>
				<xd:choice>
					<xd:sequence>
						<xd:element ref="PREAMBLE" minOccurs="0"/>
						<xd:element ref="ENACTING.TERMS"/>
						<xd:element ref="FINAL" minOccurs="0"/>
					</xd:sequence>
					<xd:element ref="COLL.LETTERS"/>
				</xd:choice>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="ALINEA" type="t_btx"/><xd:element name="AMEND">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:element ref="OLD" minOccurs="0"/>
				<xd:element ref="NEW" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="ANNEX">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.INSTANCE"/>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="TITLE"/>
					<xd:element ref="TOC"/>
				</xd:choice>
				<xd:element ref="CURR.TITLE" minOccurs="0"/>
				<xd:element ref="CONTENTS"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="ANNOTATION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:choice>
					<xd:element ref="NP"/>
					<xd:element ref="P" maxOccurs="unbounded"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="ANONYMOUS">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="NOTE"/>
			</xd:sequence>
			<xd:attribute name="PLACEHOLDER" type="xd:string" use="optional"/>
		</xd:complexType>
	</xd:element><xd:element name="APPEAL.TABLE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.APPEAL.TABLE"/>
				<xd:element ref="TITLE"/>
				<xd:element ref="CONTENTS"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="APPELANT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="P" maxOccurs="unbounded"/>
				<xd:element ref="PARTY.STATUS" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="APPLICANT" type="t_btx"/><xd:element name="ARCHIVED">
		<xd:complexType>
			<xd:attribute name="VALUE" default="NO">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="NO"/>
						<xd:enumeration value="YES"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="ARGUMENTS" type="t_btx.struct"/><xd:element name="ARTICLE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TI.ART"/>
				<xd:element ref="STI.ART" minOccurs="0"/>
				<xd:choice minOccurs="0">
					<xd:sequence>
						<xd:element ref="ALINEA"/>
						<xd:choice minOccurs="0" maxOccurs="unbounded">
							<xd:element ref="ALINEA"/>
							<xd:element ref="COMMENT"/>
							<xd:element ref="QUOT.S"/>
						</xd:choice>
					</xd:sequence>
					<xd:sequence>
						<xd:element ref="PARAG"/>
						<xd:choice minOccurs="0" maxOccurs="unbounded">
							<xd:element ref="COMMENT"/>
							<xd:element ref="PARAG"/>
						</xd:choice>
					</xd:sequence>
				</xd:choice>
				<xd:sequence minOccurs="0">
					<xd:element ref="SUBDIV"/>
					<xd:choice minOccurs="0" maxOccurs="unbounded">
						<xd:element ref="COMMENT"/>
						<xd:element ref="SUBDIV"/>
					</xd:choice>
				</xd:sequence>
			</xd:sequence>
			<xd:attribute name="IDENTIFIER" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:pattern value="\d{3}(\-\d{3})?[A-Z]*"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="ASSOCIATED.TO" type="t_ref.phys"/><xd:element name="ASSOCIATES" type="t_ref.phys"/><xd:element name="AUTHOR" type="xd:string"/><xd:element name="BAR">
		<xd:complexType/>
		<!-- single | -->
	</xd:element><xd:element name="BIB.APPEAL.TABLE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.ECR"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.CASE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.ECR"/>
				<xd:element ref="LG.DOC"/>
				<xd:element ref="LG.CASE"/>
				<xd:element ref="DATE" minOccurs="0"/>
				<xd:element ref="YEAR" minOccurs="0"/>
				<xd:element ref="NO.CASE" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="TYPE.CASE"/>
				<xd:element ref="NO.CELEX" minOccurs="0"/>
				<xd:element ref="NO.ECLI" minOccurs="0"/>
				<xd:element ref="NO.SEQ"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="AUTHOR" minOccurs="0"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.CHRON.TABLE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.ECR"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.CHRON.TABLE.NP">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.ECR"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.CONCLUSION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.CASE.F1" minOccurs="0"/>
				<xd:element ref="REF.CASE"/>
				<xd:element ref="NO.CELEX" minOccurs="0"/>
				<xd:element ref="NO.ECLI" minOccurs="0"/>
				<xd:element ref="NO.SEQ"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.CORRIG.ECR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.CASE.F1" minOccurs="0"/>
				<xd:element ref="REF.CASE"/>
				<xd:element ref="NO.SEQ"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.DATA">
		<xd:complexType>
			<xd:sequence>
				<xd:choice>
					<xd:element ref="BIB.INSTANCE.CONS"/>
					<xd:element ref="OTH.PUB"/>
				</xd:choice>
				<xd:element ref="NO.CELEX" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="DATE" minOccurs="0"/>
				<xd:element ref="TITLE"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.DATA.CL">
		<xd:complexType>
			<xd:sequence>
				<xd:choice>
					<xd:sequence>
						<xd:element ref="OJ.CL"/>
						<xd:element ref="SPEC.ED" minOccurs="0" maxOccurs="unbounded"/>
					</xd:sequence>
					<xd:element ref="OTH.PUB.CL"/>
				</xd:choice>
				<xd:element ref="NO.CELEX"/>
				<xd:element ref="DATE.ADOPT" minOccurs="0"/>
				<xd:element ref="TITLE" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="FILENAME" type="xd:string"/>
		</xd:complexType>
	</xd:element><xd:element name="BIB.DECISION.ECR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.CASE.F1" minOccurs="0"/>
				<xd:element ref="REF.CASE"/>
				<xd:element ref="NO.CELEX" minOccurs="0"/>
				<xd:element ref="NO.ECLI" minOccurs="0"/>
				<xd:element ref="NO.SEQ"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="AUTHOR"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.DOC">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="PROD.ID" minOccurs="0"/>
				<xd:element ref="FIN.ID" minOccurs="0"/>
				<xd:element ref="NO.DOC" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="DURAB"/>
				<xd:element ref="AUTHOR" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="COM" minOccurs="0"/>
				<xd:element ref="DOC.CORR" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="DOC.CORR.SE" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="EEA" minOccurs="0"/>
				<xd:element ref="NO.ELI" minOccurs="0"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.ECR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="YEAR"/>
				<xd:element ref="NO.FASCICLE"/>
				<xd:element ref="PART.ECR"/>
				<xd:element ref="LG.PUB"/>
				<xd:choice>
					<xd:element ref="NO.ISSN"/>
					<xd:sequence>
						<xd:element ref="NO.ISSN.ELECTRONIC"/>
						<xd:element ref="NO.ISSN.PRINT" minOccurs="0"/>
					</xd:sequence>
				</xd:choice>
				<xd:element ref="NO.CAT" minOccurs="0"/>
				<xd:element ref="NO.CAT.GLOBAL" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.ECR.GENERAL">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.ECR"/>
				<xd:element ref="LG.DOC"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.GEN.PUB">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="AUTHOR" minOccurs="0"/>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:element ref="PUBLISHER" minOccurs="0"/>
				<xd:element ref="SIZE" minOccurs="0"/>
				<xd:element ref="NO.CAT" minOccurs="0"/>
				<xd:element ref="NO.DOI" minOccurs="0"/>
				<xd:element ref="NO.ISBN" minOccurs="0"/>
				<xd:choice minOccurs="0">
					<xd:element ref="NO.ISSN"/>
					<xd:sequence>
						<xd:element ref="NO.ISSN.ELECTRONIC"/>
						<xd:element ref="NO.ISSN.PRINT" minOccurs="0"/>
					</xd:sequence>
				</xd:choice>
				<xd:element ref="INFO.PUBLISHER" minOccurs="0"/>
				<xd:element ref="P" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="FMX.GEN"/>
				<xd:element ref="PAPER.GEN" minOccurs="0"/>
				<xd:element ref="PDF.GEN" minOccurs="0"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.INSTANCE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DOCUMENT.REF"/>
				<xd:element ref="SPEC.ED" minOccurs="0"/>
				<xd:element ref="EEA" minOccurs="0"/>
				<xd:element ref="DATE" maxOccurs="unbounded"/>
				<xd:element ref="LG.DOC"/>
				<xd:element ref="NO.SEQ" minOccurs="0"/>
				<xd:element ref="PAGE.FIRST"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="INCLUSIONS" minOccurs="0"/>
				<xd:element ref="NO.DOC" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.INSTANCE.CONS">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DOCUMENT.REF.CONS"/>
				<xd:element ref="SPEC.ED" minOccurs="0"/>
				<xd:element ref="EEA" minOccurs="0"/>
				<xd:element ref="DATE" maxOccurs="unbounded"/>
				<xd:element ref="LG.DOC"/>
				<xd:element ref="NO.SEQ" minOccurs="0"/>
				<xd:element ref="PAGE.FIRST"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="NO.DOC" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.JUDGMENT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.CASE.F1" minOccurs="0"/>
				<xd:element ref="REF.CASE"/>
				<xd:element ref="NO.CELEX" minOccurs="0"/>
				<xd:element ref="NO.ECLI" minOccurs="0"/>
				<xd:element ref="NO.SEQ"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="AUTHOR"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.LSEU">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="NO.DOC.LSEU"/>
				<xd:element ref="LG.DOC"/>
				<xd:element ref="LAST.MODIFICATION" minOccurs="0"/>
				<xd:element ref="ARCHIVED"/>
			</xd:sequence>
			<xd:attribute name="TYPE" default="SUMMARY">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="GLOSSARY"/>
						<xd:enumeration value="INTRO"/>
						<xd:enumeration value="SUMMARY"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="BIB.OJ">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="COLL" minOccurs="0"/>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:choice minOccurs="0">
					<xd:element ref="NO.DOC"/>
					<xd:element ref="NO.OJ"/>
					<xd:element ref="SPECIAL"/>
				</xd:choice>
				<xd:element ref="AGE.OJ" minOccurs="0"/>
				<xd:element ref="DATE" minOccurs="0"/>
				<xd:element ref="LAST.OJ" minOccurs="0"/>
				<xd:element ref="LG.OJ"/>
				<xd:element ref="PRINTER" minOccurs="0"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="REPUBLICATION" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="BIB.OPINION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.CASE.F1" minOccurs="0"/>
				<xd:element ref="REF.CASE"/>
				<xd:element ref="NO.CELEX" minOccurs="0"/>
				<xd:element ref="NO.ECLI" minOccurs="0"/>
				<xd:element ref="NO.SEQ"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="AUTHOR"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.ORDER">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.CASE.F1" minOccurs="0"/>
				<xd:element ref="REF.CASE"/>
				<xd:element ref="NO.CELEX" minOccurs="0"/>
				<xd:element ref="NO.ECLI" minOccurs="0"/>
				<xd:element ref="NO.SEQ"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="AUTHOR"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.REPORT.HEARING">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.CASE.F1" minOccurs="0"/>
				<xd:element ref="REF.CASE"/>
				<xd:element ref="NO.CELEX" minOccurs="0"/>
				<xd:element ref="NO.ECLI" minOccurs="0"/>
				<xd:element ref="NO.SEQ"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.RULING">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.CASE.F1" minOccurs="0"/>
				<xd:element ref="REF.CASE"/>
				<xd:element ref="NO.CELEX" minOccurs="0"/>
				<xd:element ref="NO.ECLI" minOccurs="0"/>
				<xd:element ref="NO.SEQ"/>
				<xd:element ref="PAGE.FIRST.ECR"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST.ECR"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="AUTHOR"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				<xd:element ref="REF.BIB.RECORD" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.SE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="LG.PUB"/>
				<xd:element ref="YEAR"/>
				<xd:element ref="CHAP.SE"/>
				<xd:element ref="TOME.SE"/>
				<xd:element ref="PRINTER"/>
				<xd:element ref="NO.CAT"/>
				<xd:choice>
					<xd:element ref="NO.ISSN"/>
					<xd:sequence>
						<xd:element ref="NO.ISSN.PRINT"/>
						<xd:element ref="NO.ISSN.ELECTRONIC"/>
					</xd:sequence>
				</xd:choice>
				<xd:element ref="NO.DOI" minOccurs="0"/>
				<xd:element ref="DATE"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.SUMMARY">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.CASE.F1" minOccurs="0"/>
				<xd:element ref="REF.CASE"/>
				<xd:element ref="NO.CELEX" minOccurs="0"/>
				<xd:element ref="NO.ECLI" minOccurs="0"/>
				<xd:element ref="NO.SEQ"/>
				<xd:element ref="PAGE.FIRST"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="PAGE.LAST"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="AUTHOR"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BIB.VOLUME">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="VOLUME.ID" minOccurs="0"/>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:element ref="NO.CAT" minOccurs="0"/>
				<xd:choice minOccurs="0">
					<xd:element ref="NO.ISSN"/>
					<xd:sequence>
						<xd:element ref="NO.ISSN.PRINT" minOccurs="0"/>
						<xd:element ref="NO.ISSN.ELECTRONIC"/>
					</xd:sequence>
				</xd:choice>
				<xd:element ref="DOI.OJ" minOccurs="0"/>
				<xd:element ref="GR.ANNOTATION" minOccurs="0"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="PAGE.SUMMARY" minOccurs="0"/>
				<xd:element ref="PAGE.CONTENTS"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BLK">
		<xd:complexType>
			<xd:sequence>
				<xd:sequence minOccurs="0">
					<xd:element ref="TI.BLK"/>
					<xd:element ref="STI.BLK" minOccurs="0"/>
				</xd:sequence>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="ROW"/>
					<xd:element ref="BLK"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="BR">
		<xd:complexType/>
	</xd:element><xd:element name="CAPTION" type="t_btx.struct"/><xd:element name="CASE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.CASE"/>
				<xd:element ref="CURR.TITLE" minOccurs="0"/>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:element ref="PARTIES" minOccurs="0"/>
				<xd:element ref="SUBJECT" minOccurs="0"/>
				<xd:element ref="PRELIM.RULING" minOccurs="0"/>
				<xd:element ref="INDEX" minOccurs="0"/>
				<xd:element ref="JOINED.CASES" minOccurs="0"/>
				<xd:element ref="P" minOccurs="0" maxOccurs="unbounded"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="DECISION"/>
					<xd:element ref="OBJECT"/>
					<xd:element ref="REF.CONCLUSION"/>
					<xd:element ref="REF.CORRIG.ECR"/>
					<xd:element ref="REF.DECISION.ECR"/>
					<xd:element ref="REF.JUDGMENT"/>
					<xd:element ref="REF.JUDGMENT.NP"/>
					<xd:element ref="REF.OPINION"/>
					<xd:element ref="REF.ORDER"/>
					<xd:element ref="REF.ORDER.NP"/>
					<xd:element ref="REF.REPORT.HEARING"/>
					<xd:element ref="REF.RULING"/>
					<xd:element ref="REF.SUMMARY.JUDGMENT"/>
					<xd:element ref="SUMMARY"/>
				</xd:choice>
				<xd:element ref="PDF.ECR" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CAT.PLANT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="GR.NOTES" minOccurs="0"/>
				<xd:element ref="GR.PLANT" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CAT.VEGET">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="GR.NOTES" minOccurs="0"/>
				<xd:element ref="SPECIE" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CELL">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:sequence>
						<xd:element ref="TXT.COL" minOccurs="0"/>
						<xd:element ref="TXT.ROW" minOccurs="0"/>
					</xd:sequence>
					<xd:attribute name="COL" type="xd:positiveInteger" use="required"/>
					<xd:attribute name="COLSPAN" type="xd:positiveInteger"/>
					<xd:attribute name="ROWSPAN" type="xd:positiveInteger"/>
					<xd:attribute name="ACCH">
						<xd:simpleType>
							<xd:restriction base="xd:string">
								<xd:pattern value="\d{1,3}\.[BT]\.[BT]"/>
							</xd:restriction>
						</xd:simpleType>
					</xd:attribute>
					<xd:attribute name="ACCV">
						<xd:simpleType>
							<xd:restriction base="xd:string">
								<xd:pattern value="\d{1,3}\.[LR]\.[LR]"/>
							</xd:restriction>
						</xd:simpleType>
					</xd:attribute>
					<xd:attribute name="TYPE" default="NORMAL">
						<xd:simpleType>
							<xd:restriction base="xd:string">
								<xd:enumeration value="ALIAS"/>
								<xd:enumeration value="HEADER"/>
								<xd:enumeration value="NORMAL"/>
								<xd:enumeration value="NOTCOL"/>
								<xd:enumeration value="TOTAL"/>
							</xd:restriction>
						</xd:simpleType>
					</xd:attribute>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="CHAP.SE" type="xd:string"/><xd:element name="CHRON.TABLE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.CHRON.TABLE"/>
				<xd:element ref="TITLE"/>
				<xd:element ref="CONTENTS"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CHRON.TABLE.NP">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.CHRON.TABLE.NP"/>
				<xd:element ref="TITLE"/>
				<xd:element ref="CONTENTS"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CJT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.INSTANCE"/>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="TI.CJT"/>
				</xd:choice>
				<xd:element ref="CONTENTS"/>
				<xd:element ref="ENACTING.TERMS.CJT" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="CN">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="GR.NOTES" minOccurs="0"/>
				<xd:element ref="CN.HEADER"/>
				<xd:choice>
					<xd:element ref="GR.CN.UNIT" maxOccurs="unbounded"/>
					<xd:element ref="CN.UNIT" maxOccurs="unbounded"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CN.AUT.RATE">
		<xd:complexType>
			<xd:choice>
				<xd:element ref="TXT"/>
				<xd:element ref="PERIOD" maxOccurs="unbounded"/>
			</xd:choice>
		</xd:complexType>
	</xd:element><xd:element name="CN.CODE">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx.seq">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="CN.CONV.RATE">
		<xd:complexType>
			<xd:choice>
				<xd:element ref="TXT"/>
				<xd:element ref="PERIOD" maxOccurs="unbounded"/>
			</xd:choice>
			<xd:attribute name="ALIAS" type="xd:string"/>
		</xd:complexType>
	</xd:element><xd:element name="CN.DESC">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx.seq">
					<xd:attribute name="ALIAS" type="xd:string"/>
					<xd:attribute name="ASSV" type="xd:positiveInteger"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="CN.HEADER">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="CN.CODE"/>
				<xd:element ref="CN.DESC"/>
				<xd:sequence minOccurs="0">
					<xd:element ref="CN.RATE"/>
					<xd:element ref="CN.SUP.UNIT" minOccurs="0"/>
				</xd:sequence>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CN.RATE">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx.seq">
					<xd:sequence>
						<xd:element ref="CN.AUT.RATE" minOccurs="0"/>
						<xd:element ref="CN.CONV.RATE" minOccurs="0"/>
					</xd:sequence>
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="CN.SUP.UNIT">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx.seq">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="CN.UNIT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="CN.CODE" minOccurs="0"/>
				<xd:element ref="CN.DESC" minOccurs="0"/>
				<xd:element ref="CN.AUT.RATE" minOccurs="0"/>
				<xd:element ref="CN.CONV.RATE" minOccurs="0"/>
				<xd:element ref="CN.SUP.UNIT" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="TYPE" default="OLD">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="OLD"/>
						<xd:enumeration value="NEW"/>
						<xd:enumeration value="MODIF"/>
						<xd:enumeration value="DEL"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="HS" default="YES">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="YES"/>
						<xd:enumeration value="NO"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="LEVEL" type="xd:nonNegativeInteger" use="required"/>
		</xd:complexType>
	</xd:element><xd:element name="COLL" type="t_collection"/><xd:element name="COLL.LETTERS">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="LETTER" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="COM" type="xd:string"/><xd:element name="COMMENT" type="t_btx.struct"/><xd:element name="COMPETITION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.INSTANCE"/>
				<xd:element ref="GR.ANNOTATION" minOccurs="0"/>
				<xd:element ref="TITLE"/>
				<xd:element ref="GR.ANNOTATION" minOccurs="0"/>
				<xd:element ref="COMPETITION.INIT"/>
				<xd:element ref="TOC" minOccurs="0"/>
				<xd:element ref="CONTENTS"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="COMPETITION.INIT" type="t_btx.struct"/><xd:element name="CONCLUSION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.CONCLUSION"/>
				<xd:element ref="CURR.TITLE"/>
				<xd:element ref="TITLE"/>
				<xd:element ref="INTERMEDIATE" minOccurs="0"/>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:element ref="CONTENTS.CONCLUSION"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="CONS.ACT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="INFO.CONSLEG"/>
				<xd:element ref="INFO.PROD"/>
				<xd:element ref="CONS.DOC"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONS.ANNEX">
		<xd:complexType>
			<xd:sequence>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="TOC"/>
					<xd:element ref="TITLE"/>
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="ANNOTATION"/>
				</xd:choice>
				<xd:element ref="CONTENTS" minOccurs="0"/>
				<xd:element ref="CONS.ANNEX" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="CONS.DOC">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.INSTANCE"/>
				<xd:element ref="FAM.COMP"/>
				<xd:element ref="TOC" minOccurs="0"/>
				<xd:element ref="TITLE"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="TOC"/>
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="ANNOTATION"/>
				</xd:choice>
				<xd:element ref="PREAMBLE"/>
				<xd:element ref="ENACTING.TERMS"/>
				<xd:element ref="FINAL" minOccurs="0"/>
				<xd:element ref="CONS.ANNEX" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="CONS.DOC.GEN">
		<xd:complexType>
			<xd:sequence>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="TOC"/>
					<xd:element ref="TITLE"/>
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="ANNOTATION"/>
				</xd:choice>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="CONTENTS"/>
					<xd:element ref="FINAL"/>
					<xd:element ref="PREAMBLE"/>
					<xd:element ref="ENACTING.TERMS"/>
					<xd:element ref="ACT.GEN"/>
				</xd:choice>
				<xd:element ref="CONS.DOC.GEN" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
			<xd:attribute name="TYPE" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="ACT"/>
						<xd:enumeration value="ANNEX"/>
						<xd:enumeration value="GENERAL"/>
						<xd:enumeration value="OTH.ACT"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="CONS.LIST">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="FAM.COMP.CL"/>
				<xd:sequence maxOccurs="unbounded">
					<xd:element ref="JOB.CONSLEG"/>
					<xd:element ref="EVENT" minOccurs="0" maxOccurs="unbounded"/>
				</xd:sequence>
			</xd:sequence>
			<xd:attribute name="CONSLEG.REF" type="xd:string" use="required"/>
			<xd:attribute name="CELEX.BASIC" type="xd:string" use="required"/>
			<xd:attribute name="LEG.VAL" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="DEC"/>
						<xd:enumeration value="DECDEL"/>
						<xd:enumeration value="DEC.EEA"/>
						<xd:enumeration value="DEC.ECSC"/>
						<xd:enumeration value="DECIMP"/>
						<xd:enumeration value="DIR"/>
						<xd:enumeration value="DIRDEL"/>
						<xd:enumeration value="DIRIMP"/>
						<xd:enumeration value="GENGUID"/>
						<xd:enumeration value="JOINT.ACT"/>
						<xd:enumeration value="OTHER"/>
						<xd:enumeration value="PROC"/>
						<xd:enumeration value="REC.ECSC"/>
						<xd:enumeration value="REG"/>
						<xd:enumeration value="REGDEL"/>
						<xd:enumeration value="REGIMP"/>
						<xd:enumeration value="TREATY"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="CELEX.PREP" type="xd:string" use="required"/>
			<xd:attribute name="DATE.LAST" type="t_date" use="required"/>
			<xd:attribute name="LG.EXISTS" type="xd:string" use="required"/>
			<xd:attribute name="PRODUCER" type="xd:string" use="required"/>
			<xd:attribute name="CREATED.BY" type="xd:string" use="required"/>
		</xd:complexType>
	</xd:element><xd:element name="CONS.OTH">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="INFO.CONSLEG"/>
				<xd:element ref="INFO.PROD"/>
				<xd:element ref="CONS.OTH.DOC"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONS.OTH.DOC">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.INSTANCE"/>
				<xd:element ref="FAM.COMP"/>
				<xd:element ref="CONS.DOC.GEN"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="CONSID" type="t_btx.struct"/><xd:element name="CONTENTS">
		<xd:complexType>
			<xd:complexContent>
				<xd:extension base="t_btx.struct">
					<xd:attribute name="TYPE" default="NORMAL">
						<xd:simpleType>
							<xd:restriction base="xd:string">
								<xd:enumeration value="BOX"/>
								<xd:enumeration value="NORMAL"/>
							</xd:restriction>
						</xd:simpleType>
					</xd:attribute>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.CONCLUSION" type="t_btx.ecr"/><xd:element name="CONTENTS.CORR">
		<xd:complexType>
			<xd:choice>
				<xd:element ref="CORRECTION" maxOccurs="unbounded"/>
				<xd:element ref="GR.CORRECTION" maxOccurs="unbounded"/>
			</xd:choice>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.DECISION.ECR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="GR.SEQ" maxOccurs="unbounded"/>
				<xd:element ref="JURISDICTION"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.JUDGMENT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="GR.SEQ" maxOccurs="unbounded"/>
				<xd:element ref="JURISDICTION"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.LSEU">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="HEADER.LSEU" minOccurs="0" maxOccurs="unbounded"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="ADDITIONAL.DOCS.LSEU"/>
					<xd:element ref="REFERENCE.TABLE"/>
					<xd:element ref="REFERENCES.LSEU"/>
					<xd:element ref="REFERRED.DOCS.LSEU"/>
					<xd:element ref="SUMMARY.LSEU"/>
					<xd:element ref="MENU.LSEU" minOccurs="0"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.OPINION">
		<xd:complexType>
			<xd:sequence>
				<xd:choice>
					<xd:sequence>
						<xd:element ref="TITLE"/>
						<xd:element ref="SUMMARY"/>
						<xd:element ref="ARGUMENTS" minOccurs="0"/>
						<xd:element ref="PREAMBLE.GEN"/>
						<xd:element ref="ENACTING.TERMS.CJT"/>
					</xd:sequence>
					<xd:element ref="GR.SEQ"/>
				</xd:choice>
				<xd:element ref="FINAL"/>
				<xd:element ref="INTERNAL.ANNEX" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.ORDER">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="GR.SEQ" maxOccurs="unbounded"/>
				<xd:element ref="JURISDICTION"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.RULING">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE"/>
				<xd:element ref="SUMMARY"/>
				<xd:element ref="ARGUMENTS" minOccurs="0"/>
				<xd:element ref="PREAMBLE.GEN"/>
				<xd:element ref="ENACTING.TERMS.CJT"/>
				<xd:element ref="FINAL"/>
				<xd:element ref="INTERNAL.ANNEX" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.SE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE"/>
				<xd:element ref="GR.ANNOTATION" minOccurs="0"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="PAGE.SUMMARY"/>
				<xd:element ref="PAGE.CONTENTS"/>
				<xd:element ref="ITEM.PUB" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.SUMMARY">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="EDITION"/>
				<xd:element ref="TI.COLL"/>
				<xd:element ref="TI.INFO.NO" minOccurs="0"/>
				<xd:element ref="TI.CONTENTS"/>
				<xd:element ref="TI.PAGE"/>
				<xd:element ref="GR.ANNOTATION" minOccurs="0"/>
				<xd:element ref="SECTION.SUMMARY" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.SUMMARY.JUDGMENT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE"/>
				<xd:element ref="P" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="PARTIES"/>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="GR.SEQ"/>
					<xd:element ref="P"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CONTENTS.SUMMARY.SE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="EDITION"/>
				<xd:element ref="TI.COLL"/>
				<xd:element ref="ITEM.SUMMARY" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="COPYRIGHT" type="t_btx.seq"/><xd:element name="CORPUS">
		<xd:complexType>
			<xd:choice maxOccurs="unbounded">
				<xd:element ref="ROW"/>
				<xd:element ref="BLK"/>
			</xd:choice>
		</xd:complexType>
	</xd:element><xd:element name="CORR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.INSTANCE"/>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="TITLE"/>
				</xd:choice>
				<xd:element ref="CONTENTS.CORR"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="CORR.ACTION" type="xd:string"/><xd:element name="CORRECTION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DESCRIPTION"/>
				<xd:sequence minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="OLD.CORR" minOccurs="0"/>
					<xd:element ref="NEW.CORR" minOccurs="0"/>
				</xd:sequence>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CORRIG">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.DATA"/>
				<xd:element ref="GR.CORRIG" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="RELEVANT" default="YES">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="NO"/>
						<xd:enumeration value="YES"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="CORRIG.CL">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.DATA.CL"/>
				<xd:element ref="GR.CORRIG.CL" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="RELEVANT" default="YES">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="NO"/>
						<xd:enumeration value="YES"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="CORRIG.ECR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.ORIGINAL" minOccurs="0"/>
				<xd:element ref="CORR.ACTION" minOccurs="0"/>
				<xd:choice minOccurs="1" maxOccurs="1">
					<xd:element ref="GR.SEQ"/>
					<xd:element ref="NP"/>
					<xd:element ref="P"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="CORRIGENDUM.ECR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.CORRIG.ECR"/>
				<xd:element ref="CURR.TITLE" minOccurs="0"/>
				<xd:element ref="TITLE"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="TOC"/>
				</xd:choice>
				<xd:element ref="GR.CORRIG.ECR" maxOccurs="unbounded"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="CRIT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="QUALIF" minOccurs="0"/>
				<xd:element ref="ID.RESP" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="AREA" type="t_area" use="required"/>
			<xd:attribute name="COUNTRY" use="required">
				<!-- based on ISO 3166-1-2006, taking into account the definitions of the
				Interinstitutional Style Guide -->
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="AT"/>
						<xd:enumeration value="BE"/>
						<xd:enumeration value="BG"/>
						<xd:enumeration value="CH"/>
						<xd:enumeration value="CY"/>
						<xd:enumeration value="CZ"/>
						<xd:enumeration value="DE"/>
						<xd:enumeration value="DK"/>
						<xd:enumeration value="EE"/>
						<xd:enumeration value="EL"/>
						<xd:enumeration value="ES"/>
						<xd:enumeration value="FI"/>
						<xd:enumeration value="FR"/>
						<xd:enumeration value="HR"/>
						<xd:enumeration value="HU"/>
						<xd:enumeration value="IE"/>
						<xd:enumeration value="IS"/>
						<xd:enumeration value="IT"/>
						<xd:enumeration value="LI"/>
						<xd:enumeration value="LT"/>
						<xd:enumeration value="LU"/>
						<xd:enumeration value="LV"/>
						<xd:enumeration value="MT"/>
						<xd:enumeration value="NL"/>
						<xd:enumeration value="NO"/>
						<xd:enumeration value="PL"/>
						<xd:enumeration value="PT"/>
						<xd:enumeration value="RO"/>
						<xd:enumeration value="SE"/>
						<xd:enumeration value="SI"/>
						<xd:enumeration value="SK"/>
						<xd:enumeration value="UK"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="CURR.TITLE">
		<xd:complexType>
			<xd:choice>
				<xd:sequence>
					<xd:element ref="LEFT"/>
					<xd:element ref="RIGHT"/>
				</xd:sequence>
				<xd:element ref="PAGE.HEADER"/>
			</xd:choice>
		</xd:complexType>
	</xd:element><xd:element name="DATE">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx.seq">
					<xd:attribute name="ISO" type="t_date" use="required"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DATE.ADOPT" type="t_date"/><xd:element name="DECISION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE"/>
				<xd:element ref="P" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DECISION.ECR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.DECISION.ECR"/>
				<xd:element ref="CURR.TITLE"/>
				<xd:element ref="TITLE"/>
				<xd:element ref="INDEX" minOccurs="0"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="TOC"/>
				</xd:choice>
				<xd:element ref="DECISION.ECR.INIT"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="LIST"/>
					<xd:element ref="NP"/>
					<xd:element ref="P"/>
				</xd:choice>
				<xd:element ref="PREAMBLE"/>
				<xd:element ref="CONTENTS.DECISION.ECR"/>
				<xd:element ref="SIGNATURE.CASE"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="GR.SEQ"/>
					<xd:element ref="P"/>
					<xd:element ref="TOC"/>
				</xd:choice>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="DECISION.ECR.INIT" type="t_btx"/><xd:element name="DEF.AREA">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DEF.AREA.HD"/>
				<xd:element ref="DEF.COUNTRY" maxOccurs="unbounded"/>
			</xd:sequence>
			<xd:attribute name="ALIAS" type="xd:positiveInteger" use="optional"/>
			<xd:attribute name="AREA" type="t_area" use="required"/>
		</xd:complexType>
	</xd:element><xd:element name="DEF.AREA.HD" type="t_btx.seq"/><xd:element name="DEF.CN.CODE">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.COUNTRY" type="t_btx.seq"/><xd:element name="DEF.DESC">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.EXPORT">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.FORMA.H" type="t_btx.seq"/><xd:element name="DEF.GEN.OBS" type="t_btx.seq"/><xd:element name="DEF.HEAD.TA">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DEF.CN.CODE"/>
				<xd:element ref="DEF.SH.TARIC"/>
				<xd:element ref="DEF.DESC"/>
				<xd:element ref="DEF.SUP.UNIT" minOccurs="0"/>
				<xd:element ref="DEF.REM"/>
				<xd:element ref="DEF.TA.CODE"/>
				<xd:element ref="DEF.THIRD.RATE"/>
				<xd:element ref="DEF.PART.RATE"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DEF.HEAD.VI">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DEF.VARIETY"/>
				<xd:element ref="DEF.AREA" minOccurs="2" maxOccurs="2"/>
				<xd:choice>
					<xd:element ref="DEF.OBS.VAR"/>
					<xd:element ref="DEF.OBS.VARX"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DEF.IMPORT">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.INDEX" type="t_btx.seq"/><xd:element name="DEF.OBS.VAR">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx.seq">
					<xd:attribute name="ALIAS" type="xd:positiveInteger" use="optional"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.OBS.VARX">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DEF.OBS.VARX.HD"/>
				<xd:element ref="DEF.INDEX"/>
				<xd:element ref="DEF.FORMA.H"/>
				<xd:element ref="DEF.GEN.OBS"/>
			</xd:sequence>
			<xd:attribute name="ALIAS" type="xd:positiveInteger" use="optional"/>
		</xd:complexType>
	</xd:element><xd:element name="DEF.OBS.VARX.HD" type="t_btx.seq"/><xd:element name="DEF.OPREF">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.PART.RATE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="P"/>
				<xd:element ref="DEF.SPG"/>
				<xd:element ref="DEF.OPREF" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DEF.REM">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="P"/>
				<xd:element ref="DEF.IMPORT"/>
				<xd:element ref="DEF.EXPORT"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DEF.SH.TARIC">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.SPG">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.SUP.UNIT">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.TA.CODE">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.THIRD.RATE">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ALIAS" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEF.VARIETY">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx.seq">
					<xd:attribute name="ALIAS" type="xd:positiveInteger" use="optional"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DEFENDANTS">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="P" maxOccurs="unbounded"/>
				<xd:element ref="PARTY.STATUS" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DEFINITION" type="t_btx"/><xd:element name="DEGREE" type="t_btx.formula"/><xd:element name="DESC">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx">
					<xd:attribute name="ASSV" type="xd:string"/>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="DESCRIPTION" type="t_btx"/><xd:element name="DIV.CONSID">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="LIST"/>
					<xd:element ref="TBL"/>
				</xd:choice>
				<xd:element ref="CONSID" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="DIV.CONSID" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DIVIDEND" type="t_btx.formula"/><xd:element name="DIVISION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE"/>
				<xd:element ref="ARTICLE" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="DIVISION" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DIVISOR" type="t_btx.formula"/><xd:element name="DLIST">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DLIST.ITEM" maxOccurs="unbounded"/>
			</xd:sequence>
			<xd:attribute name="TYPE">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="FORM"/>
						<xd:enumeration value="TBL"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="SEPARATOR" type="xd:string"/>
		</xd:complexType>
	</xd:element><xd:element name="DLIST.ITEM">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="MARGIN" minOccurs="0"/>
				<xd:element ref="PREFIX" minOccurs="0"/>
				<xd:element ref="TERM"/>
				<xd:element ref="DEFINITION" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DOC">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.DOC" minOccurs="0"/>
				<xd:choice>
					<xd:element ref="PUBLICATION.REF"/>
					<xd:element ref="PUBLICATION.REF.SE"/>
				</xd:choice>
				<xd:element ref="FMX"/>
				<xd:element ref="PAPER" minOccurs="0"/>
				<xd:element ref="PDF" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DOC.CORR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="COLL"/>
				<xd:element ref="NO.OJ" minOccurs="0"/>
				<xd:element ref="DATE"/>
				<xd:element ref="NO.DOC" maxOccurs="unbounded"/>
				<xd:element ref="PAGE.FIRST"/>
				<xd:element ref="PAGE.SEQ"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DOC.CORR.SE">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="CHAP.SE"/>
				<xd:element ref="TOME.SE"/>
				<xd:element ref="YEAR"/>
				<xd:element ref="NO.DOC" minOccurs="0"/>
				<xd:element ref="PAGE.FIRST"/>
				<xd:element ref="PAGE.SEQ"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DOC.MAIN.PUB">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="LG.DOC"/>
				<xd:element ref="LEGAL.VALUE" minOccurs="0"/>
				<xd:element ref="NO.CELEX" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="DATE" maxOccurs="unbounded"/>
				<xd:element ref="PAGE.FIRST"/>
				<xd:element ref="PAGE.LAST"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="REF.PHYS" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="LINK.OJ" minOccurs="0"/>
				<xd:element ref="ASSOCIATES" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="ASSOCIATED.TO" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="NO.SEQ" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:pattern value="\d{4}"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="SYNOPTISM" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:pattern value="D\d{4}"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="IN.SUMMARY" default="YES">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="NO"/>
						<xd:enumeration value="YES"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="RELATION" default="AUTONOMOUS">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="ASSOCIATION"/>
						<xd:enumeration value="AUTONOMOUS"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="DOC.NOR">
		<xd:complexType>
			<xd:attribute name="NO.CELEX" type="xd:string" use="required"/>
			<xd:attribute name="COMMENT" type="xd:string"/>
		</xd:complexType>
	</xd:element><xd:element name="DOC.REF" type="xd:string"/><xd:element name="DOC.SUB.PUB">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="LG.DOC" minOccurs="0"/>
				<xd:element ref="LEGAL.VALUE" minOccurs="0"/>
				<xd:element ref="NO.CELEX" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="DATE" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="PAGE.FIRST"/>
				<xd:element ref="PAGE.LAST"/>
				<xd:element ref="PAGE.TOTAL"/>
				<xd:element ref="PAGE.SEQ"/>
				<xd:element ref="REF.PHYS" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="DOC.SUB.PUB" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="LINK.OJ" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="NO.SEQ" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:pattern value="\d{4}(\.\d{4})+"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="SYNOPTISM" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:pattern value="D\d{4}(\.\d{4})+"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="TYPE" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="ANNEX"/>
						<xd:enumeration value="ASSOCIATION"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="IN.SUMMARY" default="NO">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="NO"/>
						<xd:enumeration value="YES"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="DOCUMENT.REF">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="COLL" minOccurs="0"/>
				<xd:choice minOccurs="0">
					<xd:element ref="NO.DOC" minOccurs="0"/>
					<xd:sequence>
						<xd:element ref="NO.OJ" minOccurs="0"/>
						<xd:element ref="YEAR" minOccurs="0"/>
					</xd:sequence>
				</xd:choice>
				<xd:element ref="LG.OJ" minOccurs="0"/>
				<xd:element ref="PAGE.FIRST" minOccurs="0"/>
				<xd:element ref="PAGE.SEQ" minOccurs="0"/>
				<xd:element ref="VOLUME.REF" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="FILE" type="xd:anyURI" use="required"/>
		</xd:complexType>
	</xd:element><xd:element name="DOCUMENT.REF.CONS">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="COLL" minOccurs="0"/>
				<xd:choice minOccurs="0">
					<xd:element ref="NO.DOC" minOccurs="0"/>
					<xd:sequence>
						<xd:element ref="NO.OJ" minOccurs="0"/>
						<xd:element ref="YEAR" minOccurs="0"/>
					</xd:sequence>
				</xd:choice>
				<xd:element ref="LG.OJ" minOccurs="0"/>
				<xd:element ref="PAGE.FIRST" minOccurs="0"/>
				<xd:element ref="PAGE.SEQ" minOccurs="0"/>
				<xd:element ref="VOLUME.REF" minOccurs="0"/>
				<xd:element ref="NO.ELI" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="DOI.OJ" type="t_doi.oj"/><xd:element name="DURAB">
		<xd:complexType>
			<xd:attribute name="TYPE" default="NA">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="DUR"/>
						<xd:enumeration value="EPH"/>
						<xd:enumeration value="NA"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="ECR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.ECR"/>
				<xd:element ref="TITLE"/>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="GR.SEQ"/>
					<xd:element ref="REF.NOTICE"/>
					<xd:element ref="TOC"/>
					<xd:element ref="REF.APPEAL.TABLE"/>
					<xd:element ref="REF.CHRON.TABLE"/>
					<xd:element ref="REF.CHRON.TABLE.NP"/>
					<xd:element ref="REF.PDF.ECR"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="ECR.GENERAL">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.ECR.GENERAL"/>
				<xd:element ref="TITLE" minOccurs="0" maxOccurs="1"/>
				<xd:element ref="CONTENTS"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="EDITION" type="t_btx.seq"/><xd:element name="EEA">
		<xd:complexType/>
	</xd:element><xd:element name="ENACTING.TERMS">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TOC" minOccurs="0"/>
				<xd:choice>
					<xd:sequence>
						<xd:element ref="ARTICLE" minOccurs="0" maxOccurs="unbounded"/>
						<xd:element ref="DIVISION" minOccurs="0" maxOccurs="unbounded"/>
					</xd:sequence>
					<xd:element ref="NP" maxOccurs="unbounded"/>
					<xd:element ref="P" maxOccurs="unbounded"/>
					<xd:element ref="GR.SEQ" maxOccurs="unbounded"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="ENACTING.TERMS.CJT" type="t_btx.struct"/><xd:element name="ENV.BIBLIO">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="PUBLICATION"/>
				<xd:element ref="DOC" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="EVENT">
		<xd:complexType>
			<xd:attribute name="LG.EXISTS" type="xd:string" use="required"/>
			<xd:attribute name="PROD.SEQ" type="xd:string" use="required"/>
			<xd:attribute name="END.DATE" type="t_date" default="99999999"/>
			<xd:attribute name="END" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="GIVEN"/>
						<xd:enumeration value="NEXT.VERS"/>
						<xd:enumeration value="REPEALED"/>
						<xd:enumeration value="NEW.MOD"/>
						<xd:enumeration value="NEW.COR"/>
						<xd:enumeration value="NONE"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
			<xd:attribute name="SOURCE.END" type="xd:string"/>
			<xd:attribute name="EVENT.DATE" type="t_date"/>
		</xd:complexType>
	</xd:element><xd:element name="EXPONENT" type="t_btx.formula"/><xd:element name="EXPORT" type="t_btx.seq"/><xd:element name="EXPR">
		<xd:complexType>
			<xd:complexContent>
				<xd:extension base="t_btx.formula">
					<xd:attribute name="TYPE" default="NONE">
						<xd:simpleType>
							<xd:restriction base="xd:string">
								<xd:enumeration value="BAR">
									<!-- |...| -->
								</xd:enumeration>
								<xd:enumeration value="BRACE">
									<!-- {..} -->
								</xd:enumeration>
								<xd:enumeration value="BRACKET">
									<!-- (...) -->
								</xd:enumeration>
								<xd:enumeration value="NONE"/>
								<xd:enumeration value="SQBRACKET">
									<!-- [...] -->
								</xd:enumeration>
							</xd:restriction>
						</xd:simpleType>
					</xd:attribute>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="FAM.COMP">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.DATA"/>
				<xd:element ref="GR.CORRIG" minOccurs="0"/>
				<xd:element ref="GR.MOD.ACT" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="LEG.VAL" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="DEC"/>
						<xd:enumeration value="DECDEL"/>
						<xd:enumeration value="DEC.EEA"/>
						<xd:enumeration value="DEC.ECSC"/>
						<xd:enumeration value="DECIMP"/>
						<xd:enumeration value="DIR"/>
						<xd:enumeration value="DIRDEL"/>
						<xd:enumeration value="DIRIMP"/>
						<xd:enumeration value="GENGUID"/>
						<xd:enumeration value="JOINT.ACT"/>
						<xd:enumeration value="OTHER"/>
						<xd:enumeration value="PROC"/>
						<xd:enumeration value="REC.ECSC"/>
						<xd:enumeration value="REG"/>
						<xd:enumeration value="REGDEL"/>
						<xd:enumeration value="REGIMP"/>
						<xd:enumeration value="TREATY"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="FAM.COMP.CL">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.DATA.CL"/>
				<xd:element ref="GR.CORRIG.CL" minOccurs="0"/>
				<xd:element ref="GR.MOD.ACT.CL" minOccurs="0"/>
				<xd:element ref="GR.DOC.NOR" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="FIN.ID" type="t_fin.id"/><xd:element name="FINAL">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="P" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="SIGNATURE" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="FINAL.SUMMARY">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="P" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="FMT.VALUE">
		<xd:simpleType>
			<xd:restriction base="xd:string">
				<xd:pattern value="[+-±]?[0-9]+(,[0-9]+)?"/>
			</xd:restriction>
		</xd:simpleType>
	</xd:element><xd:element name="FMX">
		<xd:complexType>
			<xd:choice>
				<xd:element ref="DOC.SUB.PUB" maxOccurs="unbounded"/>
				<xd:sequence>
					<xd:element ref="DOC.MAIN.PUB"/>
					<xd:element ref="DOC.SUB.PUB" minOccurs="0" maxOccurs="unbounded"/>
				</xd:sequence>
			</xd:choice>
		</xd:complexType>
	</xd:element><xd:element name="FMX.GEN">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="REF.PHYS" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="FORMA.H" type="t_btx.seq"/><xd:element name="FORMULA">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="EXPR"/>
				<xd:sequence minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="OP.CMP"/>
					<xd:element ref="EXPR"/>
				</xd:sequence>
			</xd:sequence>
			<xd:attribute name="TYPE" use="required">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="INLINE"/>
						<xd:enumeration value="OUTLINE"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="FRACTION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DIVIDEND"/>
				<xd:element ref="DIVISOR"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="FRAGMENT" type="t_btx.struct"/><xd:element name="FT">
		<xd:complexType mixed="true">
			<xd:complexContent>
				<xd:extension base="t_btx.seq">
					<xd:attribute name="TYPE" use="required">
						<xd:simpleType>
							<xd:restriction base="xd:string">
								<xd:enumeration value="CN"/>
								<xd:enumeration value="CODE"/>
								<xd:enumeration value="DECIMAL"/>
								<xd:enumeration value="NUMBER"/>
								<xd:enumeration value="TARIC"/>
							</xd:restriction>
						</xd:simpleType>
					</xd:attribute>
				</xd:extension>
			</xd:complexContent>
		</xd:complexType>
	</xd:element><xd:element name="FULL.POWER">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="FULL.POWER.INIT"/>
				<xd:element ref="PLENIPOTENTIARY" maxOccurs="unbounded"/>
				<xd:element ref="FULL.POWER.FINAL"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="FULL.POWER.FINAL" type="t_btx.seq"/><xd:element name="FULL.POWER.INIT" type="t_btx.seq"/><xd:element name="FUNCTION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="OVER" minOccurs="0"/>
				<xd:element ref="UNDER" minOccurs="0"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GEN.OBS" type="t_btx.seq"/><xd:element name="GENERAL">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="BIB.INSTANCE"/>
				<xd:element ref="PL.DATE" minOccurs="0"/>
				<xd:element ref="NO.COM" minOccurs="0"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="TOC"/>
					<xd:element ref="TITLE"/>
				</xd:choice>
				<xd:element ref="CURR.TITLE" minOccurs="0"/>
				<xd:element ref="PROLOG" minOccurs="0"/>
				<xd:element ref="CONTENTS"/>
				<xd:element ref="FINAL" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="NNC" type="t_boolean" default="NO"/>
		</xd:complexType>
	</xd:element><xd:element name="GR.AMEND">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:element ref="AMEND" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.ANNOTATION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:element ref="ANNOTATION" maxOccurs="unbounded"/>
			</xd:sequence>
			<xd:attribute name="TYPE">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="ADD"/>
						<xd:enumeration value="GEN"/>
						<xd:enumeration value="SUBH"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="GR.CN.UNIT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TI.GR.CN.UNIT"/>
				<xd:element ref="CN.UNIT" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.CONSID">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="GR.CONSID.INIT" minOccurs="0"/>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="LIST"/>
					<xd:element ref="TBL"/>
				</xd:choice>
				<xd:element ref="CONSID" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="DIV.CONSID" minOccurs="0" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.CONSID.INIT" type="t_btx.seq"/><xd:element name="GR.CORRECTION">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DESCRIPTION"/>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="CORRECTION"/>
					<xd:element ref="GR.CORRECTION"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.CORRIG">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="CORRIG" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.CORRIG.CL">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="CORRIG.CL" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.CORRIG.ECR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:element ref="CORRIG.ECR" minOccurs="1" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.DOC.NOR">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="DOC.NOR" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.MOD.ACT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="MOD.ACT" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.MOD.ACT.CL">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="MOD.ACT.CL" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.NOTES">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:choice maxOccurs="unbounded">
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="LIST"/>
					<xd:element ref="NOTE"/>
					<xd:element ref="NP"/>
					<xd:element ref="P"/>
					<xd:element ref="TBL"/>
				</xd:choice>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.PLANT">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="GR.NOTES" minOccurs="0"/>
				<xd:element ref="TI.GR.PLANT"/>
				<xd:element ref="SPECIE" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.SEQ">
		<xd:complexType>
			<xd:sequence>
				<xd:choice minOccurs="0">
					<xd:element ref="TITLE"/>
					<xd:element ref="NO.GR.SEQ"/>
				</xd:choice>
				<xd:choice minOccurs="0" maxOccurs="unbounded">
					<xd:element ref="ACT.GEN"/>
					<xd:element ref="ADDR.S"/>
					<xd:element ref="ALINEA"/>
					<xd:element ref="ANNOTATION"/>
					<xd:element ref="ARTICLE"/>
					<xd:element ref="CAT.PLANT"/>
					<xd:element ref="CAT.VEGET"/>
					<xd:element ref="CN"/>
					<xd:element ref="COLL.LETTERS"/>
					<xd:element ref="COMPETITION.INIT"/>
					<xd:element ref="CONSID"/>
					<xd:element ref="CONTENTS"/>
					<xd:element ref="DIV.CONSID"/>
					<xd:element ref="DIVISION"/>
					<xd:element ref="DLIST"/>
					<xd:element ref="DLIST.ITEM"/>
					<xd:element ref="ENACTING.TERMS"/>
					<xd:element ref="ENACTING.TERMS.CJT"/>
					<xd:element ref="FINAL"/>
					<xd:element ref="FORMULA"/>
					<xd:element ref="FULL.POWER"/>
					<xd:element ref="FULL.POWER.FINAL"/>
					<xd:element ref="FULL.POWER.INIT"/>
					<xd:element ref="GR.AMEND"/>
					<xd:element ref="GR.ANNOTATION"/>
					<xd:element ref="GR.CONSID"/>
					<xd:element ref="GR.CONSID.INIT"/>
					<xd:element ref="GR.NOTES"/>
					<xd:element ref="GR.SEQ"/>
					<xd:element ref="GR.TBL"/>
					<xd:element ref="GR.VISA"/>
					<xd:element ref="GR.VISA.INIT"/>
					<xd:element ref="INCL.ELEMENT"/>
					<xd:element ref="ITEM"/>
					<xd:element ref="LETTER"/>
					<xd:element ref="LIST"/>
					<xd:element ref="LIST.AMEND"/>
					<xd:element ref="NOTE"/>
					<xd:element ref="NP"/>
					<xd:element ref="NP.ECR"/>
					<xd:element ref="P"/>
					<xd:element ref="PARAG"/>
					<xd:element ref="PL.DATE"/>
					<xd:element ref="PLENIPOTENTIARY"/>
					<xd:element ref="PREAMBLE"/>
					<xd:element ref="PREAMBLE.FINAL"/>
					<xd:element ref="PREAMBLE.GEN"/>
					<xd:element ref="PREAMBLE.INIT"/>
					<xd:element ref="QUOT.S"/>
					<xd:element ref="STI.ART"/>
					<xd:element ref="TARIC"/>
					<xd:element ref="TBL"/>
					<xd:element ref="TI.ART"/>
					<xd:element ref="TI.CJT"/>
					<xd:element ref="TOC"/>
					<xd:element ref="VISA"/>
				</xd:choice>
				<xd:element ref="LOC.NOTES" minOccurs="0"/>
			</xd:sequence>
			<xd:attribute name="LEVEL" type="xd:positiveInteger" default="1"/>
			<xd:attribute name="BOX" default="NO">
				<xd:simpleType>
					<xd:restriction base="xd:string">
						<xd:enumeration value="NO"/>
						<xd:enumeration value="YES"/>
					</xd:restriction>
				</xd:simpleType>
			</xd:attribute>
		</xd:complexType>
	</xd:element><xd:element name="GR.TBL">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE" minOccurs="0"/>
				<xd:element ref="GR.SEQ" minOccurs="0" maxOccurs="unbounded"/>
				<xd:element ref="GR.NOTES" minOccurs="0"/>
				<xd:element ref="TBL" minOccurs="2" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.UNIT.VI">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TI.GR.UNIT.VI"/>
				<xd:element ref="UNIT.VI" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.VISA">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="GR.VISA.INIT" minOccurs="0"/>
				<xd:element ref="VISA" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="GR.VISA.INIT" type="t_btx.seq"/><xd:element name="HEADER.LIST.AMEND">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="HEADER.OLD"/>
				<xd:element ref="HEADER.NEW" maxOccurs="unbounded"/>
			</xd:sequence>
		</xd:complexType>
	</xd:element><xd:element name="HEADER.LSEU" type="t_btx.struct"/><xd:element name="HEADER.NEW" type="t_btx.seq"/><xd:element name="HEADER.OLD" type="t_btx.seq"/><xd:element name="HEADER.SUMMARY">
		<xd:complexType>
			<xd:sequence>
				<xd:element ref="TITLE"/>
				<xd:element ref="LG.OJ"/>
				<xd:choice minOccurs="0">
					<xd:element ref="NO.ISSN"/>
					<xd:sequence>
						<xd:element ref="NO.ISSN.PRINT" minOccurs="0"/>
						<xd:element ref="NO.ISSN.ELECTRONIC"/>
					</xd:sequence>
				</xd:choice>
				<xd:element ref="NO.DOI.OJ" minOccurs="0"/>
				<xd:element ref="COLL"/>
				<xd:element ref="NO.OJ.SUMMARY"/>
				<xd:element ref="AGE.OJ.SUMMARY"/>
				<xd:element ref="DATE"/>
				<xd:element ref="REF.CORE.METADATA" minOccurs="0"/>
				