"""
Try me out in your browser:

[![Binder](https://img.shields.io/badge/try%20me%20out-launch%20notebook-579ACA.svg?logo=data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAFkAAABZCAMAAABi1XidAAAB8lBMVEX///9XmsrmZYH1olJXmsr1olJXmsrmZYH1olJXmsr1olJXmsrmZYH1olL1olJXmsr1olJXmsrmZYH1olL1olJXmsrmZYH1olJXmsr1olL1olJXmsrmZYH1olL1olJXmsrmZYH1olL1olL0nFf1olJXmsrmZYH1olJXmsq8dZb1olJXmsrmZYH1olJXmspXmspXmsr1olL1olJXmsrmZYH1olJXmsr1olL1olJXmsrmZYH1olL1olLeaIVXmsrmZYH1olL1olL1olJXmsrmZYH1olLna31Xmsr1olJXmsr1olJXmsrmZYH1olLqoVr1olJXmsr1olJXmsrmZYH1olL1olKkfaPobXvviGabgadXmsqThKuofKHmZ4Dobnr1olJXmsr1olJXmspXmsr1olJXmsrfZ4TuhWn1olL1olJXmsqBi7X1olJXmspZmslbmMhbmsdemsVfl8ZgmsNim8Jpk8F0m7R4m7F5nLB6jbh7jbiDirOEibOGnKaMhq+PnaCVg6qWg6qegKaff6WhnpKofKGtnomxeZy3noG6dZi+n3vCcpPDcpPGn3bLb4/Mb47UbIrVa4rYoGjdaIbeaIXhoWHmZYHobXvpcHjqdHXreHLroVrsfG/uhGnuh2bwj2Hxk17yl1vzmljzm1j0nlX1olL3AJXWAAAAbXRSTlMAEBAQHx8gICAuLjAwMDw9PUBAQEpQUFBXV1hgYGBkcHBwcXl8gICAgoiIkJCQlJicnJ2goKCmqK+wsLC4usDAwMjP0NDQ1NbW3Nzg4ODi5+3v8PDw8/T09PX29vb39/f5+fr7+/z8/Pz9/v7+zczCxgAABC5JREFUeAHN1ul3k0UUBvCb1CTVpmpaitAGSLSpSuKCLWpbTKNJFGlcSMAFF63iUmRccNG6gLbuxkXU66JAUef/9LSpmXnyLr3T5AO/rzl5zj137p136BISy44fKJXuGN/d19PUfYeO67Znqtf2KH33Id1psXoFdW30sPZ1sMvs2D060AHqws4FHeJojLZqnw53cmfvg+XR8mC0OEjuxrXEkX5ydeVJLVIlV0e10PXk5k7dYeHu7Cj1j+49uKg7uLU61tGLw1lq27ugQYlclHC4bgv7VQ+TAyj5Zc/UjsPvs1sd5cWryWObtvWT2EPa4rtnWW3JkpjggEpbOsPr7F7EyNewtpBIslA7p43HCsnwooXTEc3UmPmCNn5lrqTJxy6nRmcavGZVt/3Da2pD5NHvsOHJCrdc1G2r3DITpU7yic7w/7Rxnjc0kt5GC4djiv2Sz3Fb2iEZg41/ddsFDoyuYrIkmFehz0HR2thPgQqMyQYb2OtB0WxsZ3BeG3+wpRb1vzl2UYBog8FfGhttFKjtAclnZYrRo9ryG9uG/FZQU4AEg8ZE9LjGMzTmqKXPLnlWVnIlQQTvxJf8ip7VgjZjyVPrjw1te5otM7RmP7xm+sK2Gv9I8Gi++BRbEkR9EBw8zRUcKxwp73xkaLiqQb+kGduJTNHG72zcW9LoJgqQxpP3/Tj//c3yB0tqzaml05/+orHLksVO+95kX7/7qgJvnjlrfr2Ggsyx0eoy9uPzN5SPd86aXggOsEKW2Prz7du3VID3/tzs/sSRs2w7ovVHKtjrX2pd7ZMlTxAYfBAL9jiDwfLkq55Tm7ifhMlTGPyCAs7RFRhn47JnlcB9RM5T97ASuZXIcVNuUDIndpDbdsfrqsOppeXl5Y+XVKdjFCTh+zGaVuj0d9zy05PPK3QzBamxdwtTCrzyg/2Rvf2EstUjordGwa/kx9mSJLr8mLLtCW8HHGJc2R5hS219IiF6PnTusOqcMl57gm0Z8kanKMAQg0qSyuZfn7zItsbGyO9QlnxY0eCuD1XL2ys/MsrQhltE7Ug0uFOzufJFE2PxBo/YAx8XPPdDwWN0MrDRYIZF0mSMKCNHgaIVFoBbNoLJ7tEQDKxGF0kcLQimojCZopv0OkNOyWCCg9XMVAi7ARJzQdM2QUh0gmBozjc3Skg6dSBRqDGYSUOu66Zg+I2fNZs/M3/f/Grl/XnyF1Gw3VKCez0PN5IUfFLqvgUN4C0qNqYs5YhPL+aVZYDE4IpUk57oSFnJm4FyCqqOE0jhY2SMyLFoo56zyo6becOS5UVDdj7Vih0zp+tcMhwRpBeLyqtIjlJKAIZSbI8SGSF3k0pA3mR5tHuwPFoa7N7reoq2bqCsAk1HqCu5uvI1n6JuRXI+S1Mco54YmYTwcn6Aeic+kssXi8XpXC4V3t7/ADuTNKaQJdScAAAAAElFTkSuQmCC)](https://mybinder.org/v2/gh/VorTECHsa/python-sdk/master?filepath=docs%2Fexamples%2Ftry_me_out%2Fattributes.ipynb)
"""

from typing import Any, Dict, List, Optional, Union
from vortexasdk.endpoints.endpoints import ATTRIBUTES_REFERENCE
from vortexasdk.endpoints.attributes_result import AttributeResult
from vortexasdk.operations import Reference, Search
from vortexasdk.utils import convert_to_list


class Attributes(Reference, Search):
    """
    Attributes endpoint.

    An Attribute is a reference value that corresponds to an ID associated with other entities.

    For example, a vessel object from the Vessel reference endpoint may have the following keys:

    ```json
    {
        "ice_class": "b09ed4e2bd6904dd",
        "propulsion": "3ace0e050724707b"
    }
    ```

    These IDs represent attributes which can be found via the Attributes reference endpoint.

    When the attributes endpoint is searched with those ids as parameters:

    ```python
        >>> from vortexasdk import Attributes
        >>> df = Attributes().search(ids=["b09ed4e2bd6904dd", "3ace0e050724707b"]).to_df()

    ```

    Returns

    |    | id               | type       | label    |
    |---:|:-----------------|:-----------|:---------|
    |  0 | b09ed4e2bd6904dd | ice_class  | UNKNOWN  |
    |  1 | 3ace0e050724707b | propulsion | DFDE     |


    """

    def __init__(self) -> None:
        Reference.__init__(self, ATTRIBUTES_REFERENCE)
        Search.__init__(self, ATTRIBUTES_REFERENCE)

    def load_all(self) -> AttributeResult:
        """
        Load all attributes.
        """
        return self.search()

    # noinspection PyShadowingBuiltins
    def search(
        self,
        type: Optional[str] = None,
        term: Optional[Union[str, List[str]]] = None,
        ids: Optional[Union[str, List[str]]] = None,
    ) -> AttributeResult:
        """
        Find all attributes matching given type.

        # Arguments
            type: The type of attribute we're filtering on. Type can be: `ice_class`, `propulsion`, `scrubber`

        # Returns
        List of attributes matching `type`


        # Examples

        Find all attributes with a type of `ice_class`.
        ```python
        >>> from vortexasdk import Attributes
        >>> df = Attributes().search(type="scrubber").to_df()

        ```
        returns

        |    | id               | name       | type        |
        |---:|:-----------------|:-----------|:------------|
        |  0 | 14c7b073809eb565 | Open Loop  | scrubber    |
        |  1 | 478fca39000c49d6 | Unknown     | scrubber   |

        """

        search_params: Dict[str, Any] = {
            "term": [str(e) for e in convert_to_list(term)],
            "ids": convert_to_list(ids),
            "type": type,
        }

        response = super().search_with_client(
            exact_term_match=False,
            response_type=None,
            headers=None,
            **search_params
        )

        return AttributeResult(
            records=response["data"], reference=response["reference"]
        )
