from typing import List, Optional, Union
from typing_extensions import Literal
import pandas as pd

from vortexasdk.api import Product
from vortexasdk.api.entity_flattening import flatten_dictionary
from vortexasdk.api.search_result import Result
from vortexasdk.logger import get_logger
from vortexasdk.result_conversions import create_dataframe, create_list

logger = get_logger(__name__)


DEFAULT_COLUMNS = ["id", "name", "layer.0", "parent.0.name"]


class ProductResult(Result):
    """Container class that holds the result obtained from calling the `Product` endpoint."""

    def to_list(self) -> List[Product]:
        """Represent products as a list."""
        # noinspection PyTypeChecker
        return create_list(super().to_list(), Product)

    def to_df(
        self,
        columns: Optional[Union[List[str], Literal["all"]]] = DEFAULT_COLUMNS,
    ) -> pd.DataFrame:
        """
        Represent products as a `pd.DataFrame`.

        # Arguments
            columns: The product features we want in the dataframe. Enter `columns='all'` to include all features.
            Defaults to `columns = ['id', 'name', 'layer.0', 'parent.0.name']`.


        # Returns
        `pd.DataFrame` of products.

        """
        flattened_dicts = [flatten_dictionary(p) for p in super().to_list()]

        return create_dataframe(
            data=flattened_dicts,
            logger_description="Products",
            columns=columns,
        )
