# SPDX-FileCopyrightText: 2023 Espressif Systems (Shanghai) CO LTD
# SPDX-License-Identifier: Apache-2.0

import datetime
import json
import sys
from argparse import Namespace
from typing import Any, Dict, List

from rich.align import Align
from rich.console import Console
from rich.table import Table

from esp_idf_sbom import __version__
from esp_idf_sbom.libsbom import utils

empty_record = {
    'vulnerable': '',
    'pkg_name': '',
    'pkg_version': '',
    'cve_id': '',
    'cvss_base_score': '',
    'cvss_base_severity': '',
    'cvss_version': '',
    'cvss_vector_string': '',
    'cpe': '',
    'cve_link': '',
    'cve_desc': '',
    'exclude_reason': '',
}


def show(records: List[Dict[str,str]],
         args: Namespace,
         proj_name: str='',
         proj_ver: str='') -> None:
    # Sort records based on CVSS base score
    records_sorted = sorted(records, key=lambda r: float(r['cvss_base_score'] or 0), reverse=True)
    record_list = [r for r in records_sorted if r['vulnerable'] == 'YES']
    record_list += [r for r in records_sorted if r['vulnerable'] == 'EXCLUDED']
    record_list += [r for r in records_sorted if r['vulnerable'] == 'NO']
    record_list += [r for r in records_sorted if r['vulnerable'] == 'SKIPPED']

    pkgs_cnt = len({record['pkg_name'] for record in record_list})

    # Get summary
    summary: Dict[str, Any] = {
        'date': datetime.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ'),
        'database': 'NATIONAL VULNERABILITY DATABASE (https://nvd.nist.gov)',
        'tool': {
            'name': 'esp-idf-sbom',
            'version': __version__,
            'cmdl': ' '.join('\"' + arg + '\"' if ' ' in arg else arg for arg in sys.argv),
        },
        'project': {
            'name': proj_name,
            'version': proj_ver,
        },
        'cves_summary': {
            'critical': {
                'count': 0,
                'cves': [],
                'packages': [],
            },
            'high': {
                'count': 0,
                'cves': [],
                'packages': [],
            },
            'medium': {
                'count': 0,
                'cves': [],
                'packages': [],
            },
            'low': {
                'count': 0,
                'cves': [],
                'packages': [],
            },
            'unknown': {
                'count': 0,
                'cves': [],
                'packages': [],
            },
            'total_cves_count': 0,
            'packages_count': pkgs_cnt,
            'all_cves': [],
            'all_packages': [],
        }
    }

    # Add information about CVE into summary
    for r in record_list:
        if r['vulnerable'] != 'YES':
            continue
        summary['cves_summary']['total_cves_count'] += 1
        summary['cves_summary']['all_cves'].append(r['cve_id'])
        if r['pkg_name'] not in summary['cves_summary']['all_packages']:
            summary['cves_summary']['all_packages'].append(r['pkg_name'])
        severity = r['cvss_base_severity'] or 'unknown'
        severity_dict = summary['cves_summary'][severity.lower()]
        severity_dict['count'] += 1
        severity_dict['cves'].append(r['cve_id'])
        if r['pkg_name'] not in severity_dict['packages']:
            severity_dict['packages'].append(r['pkg_name'])

    console = Console(no_color=args.no_colors, emoji=False)

    if args.format == 'json':
        summary['records'] = record_list
        console.print_json(json.dumps(summary))
        return
    elif args.format == 'csv':
        console.print(','.join(utils.csv_escape(empty_record.keys())))
        for r in record_list:
            console.print(','.join(utils.csv_escape(r.values())))
        return

    cvss_severity_color_map = {
        'CRITICAL': '[red]',
        'HIGH': '[dark_orange]',
        'MEDIUM': '[yellow]',
        'LOW': '[green]',
        '': ''
    }

    # Table with report summary
    table = Table(title='Report summary', show_header=False)
    table.add_column('key', overflow='fold')
    table.add_column('value', overflow='fold')
    table.add_row('Date:', summary['date']),
    table.add_row('Project name:', summary['project']['name']),
    table.add_row('Project version:', summary['project']['version']),
    table.add_row('Vulnerability database:', summary['database']),
    table.add_row('Generated by tool:', f'{summary["tool"]["name"]} ({summary["tool"]["version"]})'),
    table.add_row('Generated with command:', f'{summary["tool"]["cmdl"]}'),
    table.add_row('Number of scanned packages:', f'{summary["cves_summary"]["packages_count"]}', end_section=True),

    severity_dict = summary['cves_summary']['critical']
    table.add_row('[red]CRITICAL CVEs found:', ', '.join(severity_dict['cves']))
    table.add_row('[red]Packages affect by CRITICAL CVEs:', ', '.join(severity_dict['packages']))
    table.add_row('[red]Number of CRITICAL CVEs:', str(severity_dict['count']), end_section=True)

    severity_dict = summary['cves_summary']['high']
    table.add_row('[dark_orange]HIGH CVEs found:', ', '.join(severity_dict['cves']))
    table.add_row('[dark_orange]Packages affect by HIGH CVEs:', ', '.join(severity_dict['packages']))
    table.add_row('[dark_orange]Number of HIGH CVEs:', str(severity_dict['count']), end_section=True)

    severity_dict = summary['cves_summary']['medium']
    table.add_row('[yellow]MEDIUM CVEs found:', ', '.join(severity_dict['cves']))
    table.add_row('[yellow]Packages affect by MEDIUM CVEs:', ', '.join(severity_dict['packages']))
    table.add_row('[yellow]Number of MEDIUM CVEs:', str(severity_dict['count']), end_section=True)

    severity_dict = summary['cves_summary']['low']
    table.add_row('[green]LOW CVEs found:', ', '.join(severity_dict['cves']))
    table.add_row('[green]Packages affect by LOW CVEs:', ', '.join(severity_dict['packages']))
    table.add_row('[green]Number of LOW CVEs:', str(severity_dict['count']), end_section=True)

    severity_dict = summary['cves_summary']['unknown']
    table.add_row('UNKNOWN CVEs found:', ', '.join(severity_dict['cves']))
    table.add_row('Packages affect by UNKNOWN CVEs:', ', '.join(severity_dict['packages']))
    table.add_row('Number of UNKNOWN CVEs:', str(severity_dict['count']), end_section=True)

    table.add_row('[bright_blue]All CVEs found:', ', '.join(summary['cves_summary']['all_cves']))
    table.add_row('[bright_blue]All packages affect by CVEs:', ', '.join(summary['cves_summary']['all_packages']))
    table.add_row('[bright_blue]Total number of CVEs:', str(summary['cves_summary']['total_cves_count']))

    console.print(Align(table, align='center'), '\n')

    # Table with newly identified vulnerabilities
    table = Table(title='[red]Packages with Identified Vulnerabilities',
                  caption='Newly identified vulnerabilities. Further analysis may be required for confirmation.')
    table.add_column('Package', vertical='middle', justify='center', overflow='fold')
    table.add_column('Version', vertical='middle', justify='center', overflow='fold')
    table.add_column('CVE ID', vertical='middle', justify='center', overflow='fold')
    table.add_column('Base Score', vertical='middle', justify='center', overflow='fold')
    table.add_column('Base Severity', vertical='middle', justify='center', overflow='fold')
    table.add_column('Information', vertical='middle', justify='center', overflow='fold')

    for r in record_list:
        if r['vulnerable'] != 'YES':
            continue
        info_table = Table(show_edge=False, show_header=False, box=None)
        if any([r['cvss_vector_string'],
               r['cvss_version'],
               r['cpe'],
               r['cve_link'],
               r['cve_desc']]):
            info_table.add_column('key', overflow='fold')
            info_table.add_column('value', overflow='fold')
            info_table.add_row('[yellow]CVSS', r['cvss_version'])
            info_table.add_row(f'[yellow]Vec.', r['cvss_vector_string'])
            info_table.add_row('[yellow]CPE', r['cpe'])
            info_table.add_row('[yellow]Link', r['cve_link'])
            info_table.add_row('[yellow]Desc.', r['cve_desc'])

        table.add_row('[bright_blue]' + r['pkg_name'],
                      r['pkg_version'],
                      cvss_severity_color_map[r['cvss_base_severity']] + r['cve_id'],
                      cvss_severity_color_map[r['cvss_base_severity']] + r['cvss_base_score'],
                      cvss_severity_color_map[r['cvss_base_severity']] + r['cvss_base_severity'],
                      info_table,
                      end_section=True)

    if table.row_count:
        console.print(Align(table, align='center'), '\n')

    # Table with vulnerabilities in cve-exclude-list
    table = Table(title='[green]Packages with Excluded Vulnerabilities',
                  caption='Already assessed vulnerabilities that do not apply to packages.')
    table.add_column('Package', vertical='middle', justify='center', overflow='fold')
    table.add_column('Version', vertical='middle', justify='center', overflow='fold')
    table.add_column('CVE ID', vertical='middle', justify='center', overflow='fold')
    table.add_column('Base Score', vertical='middle', justify='center', overflow='fold')
    table.add_column('Base Severity', vertical='middle', justify='center', overflow='fold')
    table.add_column('Information', vertical='middle', justify='center', overflow='fold')

    for r in record_list:
        if r['vulnerable'] != 'EXCLUDED':
            continue
        info_table = Table(show_edge=False, show_header=False, box=None)
        if any([r['cvss_vector_string'],
               r['cvss_version'],
               r['cpe'],
               r['cve_link'],
               r['cve_desc'],
               r['exclude_reason']]):
            info_table.add_column('key', overflow='fold')
            info_table.add_column('value', overflow='fold')
            info_table.add_row('[yellow]CVSS', r['cvss_version'])
            info_table.add_row(f'[yellow]Vec.', r['cvss_vector_string'])
            info_table.add_row('[yellow]CPE', r['cpe'])
            info_table.add_row('[yellow]Link', r['cve_link'])
            info_table.add_row('[yellow]Desc.', r['cve_desc'])
            info_table.add_row('[yellow]Reason', r['exclude_reason'])

        table.add_row('[bright_blue]' + r['pkg_name'],
                      r['pkg_version'],
                      cvss_severity_color_map[r['cvss_base_severity']] + r['cve_id'],
                      cvss_severity_color_map[r['cvss_base_severity']] + r['cvss_base_score'],
                      cvss_severity_color_map[r['cvss_base_severity']] + r['cvss_base_severity'],
                      info_table,
                      end_section=True)

    if table.row_count:
        console.print(Align(table, align='center'), '\n')

    # Table with packages which were scanned and no vulnerability was found
    table = Table(title='[green]Packages with No Identified Vulnerabilities',
                  caption='Packages checked against NVD with no vulnerabilities found.')
    table.add_column('Package', vertical='middle', justify='center', overflow='fold')
    table.add_column('Version', vertical='middle', justify='center', overflow='fold')
    table.add_column('CPE', vertical='middle', justify='center', overflow='fold')

    for r in record_list:
        if r['vulnerable'] != 'NO':
            continue
        table.add_row('[bright_blue]' + r['pkg_name'],
                      r['pkg_version'],
                      '[yellow]' + r['cpe'],
                      end_section=True)

    if table.row_count:
        console.print(Align(table, align='center'), '\n')

    # Table with packages which were not scanned because of missing CPE
    table = Table(title='[green]Packages without CPE Information',
                  caption='Packages not checked against NVD.')
    table.add_column('Package', vertical='middle', justify='center', overflow='fold')
    table.add_column('Version', vertical='middle', justify='center', overflow='fold')

    for r in record_list:
        if r['vulnerable'] != 'SKIPPED':
            continue
        table.add_row('[bright_blue]' + r['pkg_name'],
                      r['pkg_version'],
                      end_section=True)

    if table.row_count:
        console.print(Align(table, align='center'))
