"""
Configuration schema validation module for IC.

This module provides data models and validation for IC configuration files.
"""

import re
from dataclasses import dataclass, field
from typing import Dict, List, Any, Optional, Union
from pathlib import Path
import logging

logger = logging.getLogger(__name__)


@dataclass
class LoggingConfig:
    """Logging configuration data model."""
    console_level: str = "ERROR"
    file_level: str = "INFO"
    file_path: str = "logs/ic_{date}.log"
    max_files: int = 30
    format: str = "%(asctime)s [%(levelname)s] - %(message)s"
    mask_sensitive: bool = True
    
    def validate(self) -> List[str]:
        """Validate logging configuration."""
        errors = []
        
        valid_levels = ["DEBUG", "INFO", "WARNING", "ERROR", "CRITICAL"]
        if self.console_level not in valid_levels:
            errors.append(f"Invalid console_level: {self.console_level}. Must be one of {valid_levels}")
        
        if self.file_level not in valid_levels:
            errors.append(f"Invalid file_level: {self.file_level}. Must be one of {valid_levels}")
        
        if self.max_files < 1:
            errors.append("max_files must be at least 1")
        
        if not self.file_path:
            errors.append("file_path cannot be empty")
        
        return errors


@dataclass
class TagConfig:
    """Tag validation configuration."""
    required: List[str] = field(default_factory=lambda: ["User", "Team", "Environment"])
    optional: List[str] = field(default_factory=lambda: ["Service", "Application"])
    rules: Dict[str, str] = field(default_factory=lambda: {
        "User": "^.+$",
        "Team": "^\\d+$",
        "Environment": "^(PROD|STG|DEV|TEST|QA)$",
    })
    
    def validate(self) -> List[str]:
        """Validate tag configuration."""
        errors = []
        
        # Validate regex patterns
        for tag_name, pattern in self.rules.items():
            try:
                re.compile(pattern)
            except re.error as e:
                errors.append(f"Invalid regex pattern for tag '{tag_name}': {e}")
        
        return errors


@dataclass
class AWSConfig:
    """AWS configuration data model."""
    accounts: List[str] = field(default_factory=list)
    regions: List[str] = field(default_factory=lambda: ["ap-northeast-2"])
    cross_account_role: str = "OrganizationAccountAccessRole"
    session_duration: int = 3600
    max_workers: int = 10
    tags: TagConfig = field(default_factory=TagConfig)
    default_profile: Optional[str] = None
    default_region: Optional[str] = None
    
    def validate(self) -> List[str]:
        """Validate AWS configuration."""
        errors = []
        
        # Validate account IDs
        for account_id in self.accounts:
            if not isinstance(account_id, str) or not re.match(r'^\d{12}$', account_id):
                errors.append(f"Invalid AWS account ID: {account_id}. Must be 12 digits")
        
        # Validate regions
        aws_regions = [
            "us-east-1", "us-east-2", "us-west-1", "us-west-2",
            "ap-northeast-1", "ap-northeast-2", "ap-northeast-3",
            "ap-southeast-1", "ap-southeast-2", "ap-south-1",
            "eu-west-1", "eu-west-2", "eu-west-3", "eu-central-1",
            "ca-central-1", "sa-east-1"
        ]
        for region in self.regions:
            if region not in aws_regions:
                errors.append(f"Unknown AWS region: {region}")
        
        # Validate session duration
        if not (900 <= self.session_duration <= 43200):  # 15 minutes to 12 hours
            errors.append("session_duration must be between 900 and 43200 seconds")
        
        # Validate max_workers
        if self.max_workers < 1:
            errors.append("max_workers must be at least 1")
        
        # Validate tags
        errors.extend(self.tags.validate())
        
        return errors


@dataclass
class AzureConfig:
    """Azure configuration data model."""
    subscriptions: List[str] = field(default_factory=list)
    locations: List[str] = field(default_factory=lambda: ["Korea Central"])
    max_workers: int = 10
    tenant_id: Optional[str] = None
    client_id: Optional[str] = None
    client_secret: Optional[str] = None
    subscription_id: Optional[str] = None
    
    def validate(self) -> List[str]:
        """Validate Azure configuration."""
        errors = []
        
        # Validate subscription IDs (UUIDs)
        uuid_pattern = r'^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$'
        for sub_id in self.subscriptions:
            if not re.match(uuid_pattern, sub_id, re.IGNORECASE):
                errors.append(f"Invalid Azure subscription ID format: {sub_id}")
        
        # Validate max_workers
        if self.max_workers < 1:
            errors.append("max_workers must be at least 1")
        
        # Validate tenant_id if provided
        if self.tenant_id and not re.match(uuid_pattern, self.tenant_id, re.IGNORECASE):
            errors.append(f"Invalid Azure tenant ID format: {self.tenant_id}")
        
        # Validate client_id if provided
        if self.client_id and not re.match(uuid_pattern, self.client_id, re.IGNORECASE):
            errors.append(f"Invalid Azure client ID format: {self.client_id}")
        
        return errors


@dataclass
class GCPMCPConfig:
    """GCP MCP configuration."""
    enabled: bool = True
    endpoint: str = "http://localhost:8080/gcp"
    auth_method: str = "service_account"
    prefer_mcp: bool = True
    
    def validate(self) -> List[str]:
        """Validate GCP MCP configuration."""
        errors = []
        
        valid_auth_methods = ["service_account", "oauth", "default"]
        if self.auth_method not in valid_auth_methods:
            errors.append(f"Invalid auth_method: {self.auth_method}. Must be one of {valid_auth_methods}")
        
        return errors


@dataclass
class GCPConfig:
    """GCP configuration data model."""
    mcp: GCPMCPConfig = field(default_factory=GCPMCPConfig)
    projects: List[str] = field(default_factory=list)
    regions: List[str] = field(default_factory=lambda: ["asia-northeast3"])
    zones: List[str] = field(default_factory=lambda: ["asia-northeast3-a"])
    max_workers: int = 10
    service_account_key_path: Optional[str] = None
    project_id: Optional[str] = None
    
    def validate(self) -> List[str]:
        """Validate GCP configuration."""
        errors = []
        
        # Validate project IDs
        project_pattern = r'^[a-z][a-z0-9-]{4,28}[a-z0-9]$'
        for project_id in self.projects:
            if not re.match(project_pattern, project_id):
                errors.append(f"Invalid GCP project ID: {project_id}")
        
        # Validate max_workers
        if self.max_workers < 1:
            errors.append("max_workers must be at least 1")
        
        # Validate service account key path if provided
        if self.service_account_key_path:
            key_path = Path(self.service_account_key_path).expanduser()
            if not key_path.exists():
                errors.append(f"Service account key file not found: {self.service_account_key_path}")
        
        # Validate MCP configuration
        errors.extend(self.mcp.validate())
        
        return errors


@dataclass
class OCIConfig:
    """OCI configuration data model."""
    config_path: str = "~/.oci/config"
    max_workers: int = 10
    
    def validate(self) -> List[str]:
        """Validate OCI configuration."""
        errors = []
        
        if self.max_workers < 1:
            errors.append("max_workers must be at least 1")
        
        # Check if config file exists
        config_path = Path(self.config_path).expanduser()
        if not config_path.exists():
            errors.append(f"OCI config file not found: {self.config_path}")
        
        return errors


@dataclass
class CloudFlareConfig:
    """CloudFlare configuration data model."""
    accounts: List[str] = field(default_factory=list)
    zones: List[str] = field(default_factory=list)
    email: Optional[str] = None
    api_token: Optional[str] = None
    
    def validate(self) -> List[str]:
        """Validate CloudFlare configuration."""
        errors = []
        
        # Validate email format if provided
        if self.email:
            email_pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
            if not re.match(email_pattern, self.email):
                errors.append(f"Invalid email format: {self.email}")
        
        return errors


@dataclass
class SSHTimeoutConfig:
    """SSH timeout configuration."""
    port_scan: float = 0.5
    ssh_connect: int = 5
    
    def validate(self) -> List[str]:
        """Validate SSH timeout configuration."""
        errors = []
        
        if self.port_scan <= 0:
            errors.append("port_scan timeout must be positive")
        
        if self.ssh_connect <= 0:
            errors.append("ssh_connect timeout must be positive")
        
        return errors


@dataclass
class SSHConfig:
    """SSH configuration data model."""
    config_file: str = "~/.ssh/config"
    key_dir: str = "~/aws-key"
    max_workers: int = 70
    timeouts: SSHTimeoutConfig = field(default_factory=SSHTimeoutConfig)
    
    def validate(self) -> List[str]:
        """Validate SSH configuration."""
        errors = []
        
        if self.max_workers < 1:
            errors.append("max_workers must be at least 1")
        
        # Validate timeouts
        errors.extend(self.timeouts.validate())
        
        return errors


@dataclass
class MCPServerConfig:
    """MCP server configuration."""
    enabled: bool = True
    auto_approve: List[str] = field(default_factory=list)
    personal_access_token: Optional[str] = None
    
    def validate(self) -> List[str]:
        """Validate MCP server configuration."""
        return []  # Basic validation, can be extended


@dataclass
class MCPConfig:
    """MCP configuration data model."""
    servers: Dict[str, MCPServerConfig] = field(default_factory=lambda: {
        "github": MCPServerConfig(),
        "terraform": MCPServerConfig(),
        "aws_docs": MCPServerConfig(auto_approve=["read_documentation", "search_documentation"]),
        "azure": MCPServerConfig(auto_approve=["documentation"]),
    })
    
    def validate(self) -> List[str]:
        """Validate MCP configuration."""
        errors = []
        
        for server_name, server_config in self.servers.items():
            server_errors = server_config.validate()
            errors.extend([f"MCP server '{server_name}': {error}" for error in server_errors])
        
        return errors


@dataclass
class SlackConfig:
    """Slack configuration data model."""
    enabled: bool = False
    webhook_url: Optional[str] = None
    
    def validate(self) -> List[str]:
        """Validate Slack configuration."""
        errors = []
        
        if self.enabled and not self.webhook_url:
            errors.append("webhook_url is required when Slack is enabled")
        
        if self.webhook_url and not self.webhook_url.startswith('https://hooks.slack.com/'):
            errors.append("Invalid Slack webhook URL format")
        
        return errors


@dataclass
class SecurityConfig:
    """Security configuration data model."""
    sensitive_keys: List[str] = field(default_factory=lambda: [
        "password", "passwd", "pwd",
        "token", "access_token", "refresh_token", "auth_token",
        "key", "api_key", "access_key", "secret_key", "private_key",
        "secret", "client_secret", "webhook_secret",
        "webhook_url", "webhook",
        "credential", "credentials",
        "cert", "certificate",
        "session", "session_token",
    ])
    mask_pattern: str = "***MASKED***"
    warn_on_sensitive_in_config: bool = True
    git_hooks_enabled: bool = True
    
    def validate(self) -> List[str]:
        """Validate security configuration."""
        errors = []
        
        if not self.sensitive_keys:
            errors.append("sensitive_keys cannot be empty")
        
        if not self.mask_pattern:
            errors.append("mask_pattern cannot be empty")
        
        return errors


@dataclass
class ICConfig:
    """Main IC configuration data model."""
    version: str = "1.0"
    logging: LoggingConfig = field(default_factory=LoggingConfig)
    aws: AWSConfig = field(default_factory=AWSConfig)
    azure: AzureConfig = field(default_factory=AzureConfig)
    gcp: GCPConfig = field(default_factory=GCPConfig)
    oci: OCIConfig = field(default_factory=OCIConfig)
    cloudflare: CloudFlareConfig = field(default_factory=CloudFlareConfig)
    ssh: SSHConfig = field(default_factory=SSHConfig)
    mcp: MCPConfig = field(default_factory=MCPConfig)
    slack: SlackConfig = field(default_factory=SlackConfig)
    security: SecurityConfig = field(default_factory=SecurityConfig)
    
    def validate(self) -> List[str]:
        """Validate entire configuration."""
        errors = []
        
        # Validate version
        if not self.version:
            errors.append("version cannot be empty")
        
        # Validate each section
        errors.extend([f"logging: {error}" for error in self.logging.validate()])
        errors.extend([f"aws: {error}" for error in self.aws.validate()])
        errors.extend([f"azure: {error}" for error in self.azure.validate()])
        errors.extend([f"gcp: {error}" for error in self.gcp.validate()])
        errors.extend([f"oci: {error}" for error in self.oci.validate()])
        errors.extend([f"cloudflare: {error}" for error in self.cloudflare.validate()])
        errors.extend([f"ssh: {error}" for error in self.ssh.validate()])
        errors.extend([f"mcp: {error}" for error in self.mcp.validate()])
        errors.extend([f"slack: {error}" for error in self.slack.validate()])
        errors.extend([f"security: {error}" for error in self.security.validate()])
        
        return errors


class ConfigValidator:
    """Configuration validator with comprehensive error reporting."""
    
    def __init__(self):
        """Initialize ConfigValidator."""
        self.errors: List[str] = []
        self.warnings: List[str] = []
    
    def validate_config_dict(self, config_data: Dict[str, Any]) -> Dict[str, List[str]]:
        """
        Validate configuration dictionary and return detailed results.
        
        Args:
            config_data: Configuration dictionary to validate
            
        Returns:
            Dictionary with 'errors' and 'warnings' lists
        """
        self.errors = []
        self.warnings = []
        
        try:
            # Convert dict to dataclass for validation
            ic_config = self._dict_to_dataclass(config_data)
            
            # Validate the configuration
            validation_errors = ic_config.validate()
            self.errors.extend(validation_errors)
            
        except Exception as e:
            self.errors.append(f"Configuration structure error: {e}")
        
        return {
            'errors': self.errors,
            'warnings': self.warnings
        }
    
    def _dict_to_dataclass(self, config_data: Dict[str, Any]) -> ICConfig:
        """
        Convert configuration dictionary to ICConfig dataclass.
        
        Args:
            config_data: Configuration dictionary
            
        Returns:
            ICConfig instance
        """
        # This is a simplified conversion - in a real implementation,
        # you might want to use a library like dacite or cattrs
        
        # Extract and convert each section
        logging_data = config_data.get('logging', {})
        logging_config = LoggingConfig(
            console_level=logging_data.get('console_level', 'ERROR'),
            file_level=logging_data.get('file_level', 'INFO'),
            file_path=logging_data.get('file_path', 'logs/ic_{date}.log'),
            max_files=logging_data.get('max_files', 30),
            format=logging_data.get('format', '%(asctime)s [%(levelname)s] - %(message)s'),
            mask_sensitive=logging_data.get('mask_sensitive', True),
        )
        
        # AWS configuration
        aws_data = config_data.get('aws', {})
        tags_data = aws_data.get('tags', {})
        tag_config = TagConfig(
            required=tags_data.get('required', ["User", "Team", "Environment"]),
            optional=tags_data.get('optional', ["Service", "Application"]),
            rules=tags_data.get('rules', {
                "User": "^.+$",
                "Team": "^\\d+$",
                "Environment": "^(PROD|STG|DEV|TEST|QA)$",
            }),
        )
        
        aws_config = AWSConfig(
            accounts=aws_data.get('accounts', []),
            regions=aws_data.get('regions', ["ap-northeast-2"]),
            cross_account_role=aws_data.get('cross_account_role', 'OrganizationAccountAccessRole'),
            session_duration=aws_data.get('session_duration', 3600),
            max_workers=aws_data.get('max_workers', 10),
            tags=tag_config,
            default_profile=aws_data.get('default_profile'),
            default_region=aws_data.get('default_region'),
        )
        
        # Continue with other sections...
        # For brevity, I'll create a basic version
        
        return ICConfig(
            version=config_data.get('version', '1.0'),
            logging=logging_config,
            aws=aws_config,
            # Add other sections as needed
        )
    
    def validate_json_schema(self, config_data: Dict[str, Any]) -> Dict[str, List[str]]:
        """
        Validate configuration against JSON schema.
        
        Args:
            config_data: Configuration data to validate
            
        Returns:
            Dictionary with validation results
        """
        # This would use jsonschema library for validation
        # For now, return basic validation
        return self.validate_config_dict(config_data)


def get_json_schema() -> Dict[str, Any]:
    """
    Get JSON schema for IC configuration.
    
    Returns:
        JSON schema dictionary
    """
    return {
        "$schema": "http://json-schema.org/draft-07/schema#",
        "title": "IC Configuration Schema",
        "type": "object",
        "required": ["version"],
        "properties": {
            "version": {
                "type": "string",
                "pattern": "^\\d+\\.\\d+$"
            },
            "logging": {
                "type": "object",
                "properties": {
                    "console_level": {
                        "type": "string",
                        "enum": ["DEBUG", "INFO", "WARNING", "ERROR", "CRITICAL"]
                    },
                    "file_level": {
                        "type": "string",
                        "enum": ["DEBUG", "INFO", "WARNING", "ERROR", "CRITICAL"]
                    },
                    "file_path": {"type": "string"},
                    "max_files": {"type": "integer", "minimum": 1},
                    "format": {"type": "string"},
                    "mask_sensitive": {"type": "boolean"}
                }
            },
            "aws": {
                "type": "object",
                "properties": {
                    "accounts": {
                        "type": "array",
                        "items": {
                            "type": "string",
                            "pattern": "^\\d{12}$"
                        }
                    },
                    "regions": {
                        "type": "array",
                        "items": {"type": "string"}
                    },
                    "cross_account_role": {"type": "string"},
                    "session_duration": {
                        "type": "integer",
                        "minimum": 900,
                        "maximum": 43200
                    },
                    "max_workers": {
                        "type": "integer",
                        "minimum": 1
                    }
                }
            },
            "security": {
                "type": "object",
                "properties": {
                    "sensitive_keys": {
                        "type": "array",
                        "items": {"type": "string"}
                    },
                    "mask_pattern": {"type": "string"},
                    "warn_on_sensitive_in_config": {"type": "boolean"},
                    "git_hooks_enabled": {"type": "boolean"}
                }
            }
        }
    }