"""Integration tests for keys mode behavior with multi-level scenarios."""

import sys
from unittest.mock import Mock, call, patch

import pytest

from awsquery.cli import main
from awsquery.core import (
    CallResult,
    execute_multi_level_call_with_tracking,
    execute_with_tracking,
    show_keys_from_result,
)


class TestKeysModeMultiLevelIntegration:
    """Integration tests for keys mode with multi-level call scenarios."""

    @patch("awsquery.cli.execute_with_tracking")
    @patch("awsquery.cli.get_aws_services")
    @patch("awsquery.cli.create_session")
    def test_keys_mode_successful_initial_call(
        self, mock_create_session, mock_services, mock_tracking
    ):
        """Test keys mode when initial call succeeds - should not use multi-level."""
        mock_services.return_value = ["ec2"]
        mock_session = Mock()
        mock_create_session.return_value = mock_session

        # Setup successful initial call
        successful_result = CallResult()
        successful_result.final_success = True
        successful_result.last_successful_response = [
            {
                "Reservations": [
                    {
                        "Instances": [
                            {
                                "InstanceId": "i-123456789abcdef0",
                                "InstanceType": "t2.micro",
                                "State": {"Name": "running", "Code": 16},
                                "PublicIpAddress": "203.0.113.12",
                                "Tags": [
                                    {"Key": "Name", "Value": "web-server"},
                                    {"Key": "Environment", "Value": "production"},
                                ],
                            }
                        ]
                    }
                ]
            }
        ]
        mock_tracking.return_value = successful_result

        test_args = ["awsquery", "--keys", "ec2", "describe-instances"]

        with patch.object(sys, "argv", test_args), patch(
            "awsquery.cli.execute_multi_level_call_with_tracking"
        ) as mock_multi_level, patch("awsquery.cli.show_keys_from_result") as mock_show_keys:

            mock_show_keys.return_value = (
                "  InstanceId\n  InstanceType\n  State.Name\n"
                "  PublicIpAddress\n  Tags.Name\n  Tags.Environment"
            )

            try:
                main()
            except SystemExit:
                pass

        # Verify initial tracking was called
        mock_tracking.assert_called_once()

        # Multi-level should NOT be called since initial call succeeded
        mock_multi_level.assert_not_called()

        # Keys should be shown from the successful result
        mock_show_keys.assert_called_once_with(successful_result)

    @patch("awsquery.cli.execute_with_tracking")
    @patch("awsquery.cli.execute_multi_level_call_with_tracking")
    @patch("awsquery.cli.get_aws_services")
    @patch("awsquery.cli.create_session")
    def test_keys_mode_fallback_to_multi_level(
        self, mock_create_session, mock_services, mock_multi_level, mock_tracking
    ):
        """Test keys mode falls back to multi-level when initial call fails."""
        mock_services.return_value = ["eks"]
        mock_session = Mock()
        mock_create_session.return_value = mock_session

        # Setup failed initial call
        failed_result = CallResult()
        failed_result.final_success = False
        failed_result.error_messages = ["Validation error: missing clusterName parameter"]
        mock_tracking.return_value = failed_result

        # Setup successful multi-level call
        successful_multi_result = CallResult()
        successful_multi_result.final_success = True
        successful_multi_result.last_successful_response = [
            {
                "cluster": {
                    "name": "production-cluster",
                    "status": "ACTIVE",
                    "version": "1.21",
                    "endpoint": "https://api.production-cluster.eks.us-east-1.amazonaws.com",
                    "tags": {"Environment": "production", "Team": "platform"},
                }
            }
        ]
        mock_multi_level.return_value = (successful_multi_result, [])

        test_args = ["awsquery", "--keys", "eks", "describe-cluster"]

        with patch.object(sys, "argv", test_args), patch(
            "awsquery.cli.show_keys_from_result"
        ) as mock_show_keys:

            mock_show_keys.return_value = (
                "  name\n  status\n  version\n  endpoint\n  tags.Environment\n  tags.Team"
            )

            try:
                main()
            except SystemExit:
                pass

        # Verify both calls were made
        mock_tracking.assert_called_once()
        mock_multi_level.assert_called_once()

        # Keys should be shown from multi-level result (last successful response)
        mock_show_keys.assert_called_once_with(successful_multi_result)

    def test_end_to_end_keys_extraction_from_multi_level_chain(self):
        """Test complete end-to-end keys extraction through multi-level resolution."""
        # This test simulates the actual flow without CLI mocking

        # Mock initial validation error
        validation_error = {
            "validation_error": {
                "parameter_name": "functionName",
                "is_required": True,
                "error_type": "missing_parameter",
            }
        }

        # Mock successful list functions response
        list_response = [
            {
                "Functions": [
                    {
                        "FunctionName": "api-handler",
                        "Runtime": "python3.9",
                        "Handler": "lambda_function.lambda_handler",
                        "CodeSize": 1024,
                    },
                    {
                        "FunctionName": "data-processor",
                        "Runtime": "python3.8",
                        "Handler": "app.handler",
                        "CodeSize": 2048,
                    },
                ]
            }
        ]

        # Mock final get_function response (this is what keys should come from)
        final_response = [
            {
                "Configuration": {
                    "FunctionName": "api-handler",
                    "FunctionArn": "arn:aws:lambda:us-east-1:123456789012:function:api-handler",
                    "Runtime": "python3.9",
                    "Role": "arn:aws:iam::123456789012:role/lambda-role",
                    "Handler": "lambda_function.lambda_handler",
                    "CodeSize": 1024,
                    "Description": "API request handler",
                    "Timeout": 30,
                    "MemorySize": 128,
                    "LastModified": "2023-01-01T00:00:00.000+0000",
                    "Version": "$LATEST",
                    "Environment": {"Variables": {"ENV": "production", "DEBUG": "false"}},
                    "Tags": {"Owner": "api-team", "Project": "web-api"},
                }
            }
        ]

        with patch("awsquery.core.execute_aws_call") as mock_execute, patch(
            "awsquery.core.infer_list_operation"
        ) as mock_infer, patch("awsquery.core.get_correct_parameter_name") as mock_get_param:

            mock_execute.side_effect = [
                validation_error,  # Initial call fails
                list_response,  # List operation succeeds
                final_response,  # Final call succeeds
            ]
            mock_infer.return_value = ["list_functions"]
            mock_get_param.return_value = "FunctionName"

            call_result, resources = execute_multi_level_call_with_tracking(
                "lambda", "get-function", [], [], []
            )

            # Verify the call succeeded and we have the final response
            assert call_result.final_success is True
            assert call_result.last_successful_response == final_response
            assert len(call_result.successful_responses) == 2  # List + Final

            # Test key extraction from the final response (not the list response)
            keys_output = show_keys_from_result(call_result)

            # Keys should come from final Configuration, not from the list Functions
            assert "FunctionArn" in keys_output  # Only in final response
            assert "Role" in keys_output  # Only in final response
            assert "Environment.Variables.ENV" in keys_output  # Nested structure
            assert "Tags.Owner" in keys_output  # Transformed tags

            # Should NOT contain list-specific keys
            assert "Functions" not in keys_output  # This was only in list response

    def test_keys_from_failed_multi_level_chain(self):
        """Test keys mode behavior when entire multi-level chain fails."""
        # Mock initial validation error
        validation_error = {
            "validation_error": {
                "parameter_name": "unknownParam",
                "is_required": True,
                "error_type": "missing_parameter",
            }
        }

        with patch("awsquery.core.execute_aws_call") as mock_execute, patch(
            "awsquery.core.infer_list_operation"
        ) as mock_infer:

            mock_execute.side_effect = [
                validation_error,  # Initial call fails
                Exception("No list operation found"),  # List operation fails
            ]
            mock_infer.return_value = ["list_unknown"]

            call_result, resources = execute_multi_level_call_with_tracking(
                "service", "describe-unknown", [], [], []
            )

            # Verify the entire chain failed
            assert call_result.final_success is False
            assert call_result.last_successful_response is None
            assert len(resources) == 0

            # Test error message from failed result
            error_output = show_keys_from_result(call_result)

            assert "Error: No successful response to show keys from" in error_output
            assert "Could not find working list operation" in error_output


class TestKeysModeWithFilteredMultiLevel:
    """Integration tests for keys mode with filtered multi-level calls."""

    def test_keys_from_filtered_multi_level_result(self):
        """Test keys extraction when multi-level call uses resource filtering."""
        # Mock validation error for cluster-specific call
        validation_error = {
            "validation_error": {
                "parameter_name": "clusterName",
                "is_required": True,
                "error_type": "missing_parameter",
            }
        }

        # Mock list with multiple clusters
        list_response = [
            {
                "clusters": [
                    {"name": "production-cluster", "status": "ACTIVE"},
                    {"name": "staging-cluster", "status": "ACTIVE"},
                    {"name": "dev-cluster", "status": "CREATING"},
                ]
            }
        ]

        # Mock final response for filtered cluster
        final_response = [
            {
                "cluster": {
                    "name": "production-cluster",
                    "status": "ACTIVE",
                    "version": "1.21",
                    "endpoint": "https://api.prod.eks.amazonaws.com",
                    "resourcesVpcConfig": {
                        "subnetIds": ["subnet-12345", "subnet-67890"],
                        "securityGroupIds": ["sg-abcdef123"],
                    },
                    "tags": {"Environment": "production", "Owner": "platform-team"},
                }
            }
        ]

        with patch("awsquery.core.execute_aws_call") as mock_execute, patch(
            "awsquery.core.infer_list_operation"
        ) as mock_infer, patch("awsquery.core.get_correct_parameter_name") as mock_get_param:

            mock_execute.side_effect = [validation_error, list_response, final_response]
            mock_infer.return_value = ["list_clusters"]
            mock_get_param.return_value = "ClusterName"

            # Execute with resource filter to select only production cluster
            call_result, resources = execute_multi_level_call_with_tracking(
                "eks", "describe-cluster", ["production"], [], []
            )

            assert call_result.final_success is True

            # Keys should come from the final describe-cluster response
            keys_output = show_keys_from_result(call_result)

            # Should have keys from final cluster description
            assert "name" in keys_output
            assert "endpoint" in keys_output
            assert "resourcesVpcConfig.subnetIds" in keys_output
            assert "tags.Environment" in keys_output
            assert "tags.Owner" in keys_output

            # Should NOT have keys from list response
            assert "clusters" not in keys_output

    def test_keys_mode_with_complex_nested_response(self):
        """Test keys extraction from complex nested response structure."""
        # Simulate CloudFormation stack resources call
        validation_error = {
            "validation_error": {
                "parameter_name": "StackName",
                "is_required": True,
                "error_type": "missing_parameter",
            }
        }

        list_response = [
            {
                "StackSummaries": [
                    {"StackName": "infrastructure-stack", "StackStatus": "CREATE_COMPLETE"},
                    {"StackName": "application-stack", "StackStatus": "UPDATE_COMPLETE"},
                ]
            }
        ]

        # Complex nested final response
        final_response = [
            {
                "StackResources": [
                    {
                        "StackName": "infrastructure-stack",
                        "LogicalResourceId": "VPC",
                        "PhysicalResourceId": "vpc-12345678",
                        "ResourceType": "AWS::EC2::VPC",
                        "Timestamp": "2023-01-01T00:00:00.000Z",
                        "ResourceStatus": "CREATE_COMPLETE",
                        "ResourceProperties": {
                            "CidrBlock": "10.0.0.0/16",
                            "EnableDnsHostnames": True,
                            "EnableDnsSupport": True,
                            "Tags": [
                                {"Key": "Name", "Value": "MainVPC"},
                                {"Key": "Environment", "Value": "production"},
                            ],
                        },
                        "DriftInformation": {"StackResourceDriftStatus": "IN_SYNC"},
                    },
                    {
                        "StackName": "infrastructure-stack",
                        "LogicalResourceId": "PublicSubnet",
                        "PhysicalResourceId": "subnet-abcdef01",
                        "ResourceType": "AWS::EC2::Subnet",
                        "Timestamp": "2023-01-01T00:00:00.000Z",
                        "ResourceStatus": "CREATE_COMPLETE",
                        "ResourceProperties": {
                            "VpcId": "vpc-12345678",
                            "CidrBlock": "10.0.1.0/24",
                            "AvailabilityZone": "us-east-1a",
                        },
                    },
                ]
            }
        ]

        with patch("awsquery.core.execute_aws_call") as mock_execute, patch(
            "awsquery.core.infer_list_operation"
        ) as mock_infer, patch("awsquery.core.get_correct_parameter_name") as mock_get_param:

            mock_execute.side_effect = [validation_error, list_response, final_response]
            mock_infer.return_value = ["list_stacks"]
            mock_get_param.return_value = "StackName"

            call_result, resources = execute_multi_level_call_with_tracking(
                "cloudformation", "describe-stack-resources", ["infrastructure"], [], []
            )

            assert call_result.final_success is True

            keys_output = show_keys_from_result(call_result)

            # Should have keys from stack resources (final response)
            assert "StackName" in keys_output
            assert "LogicalResourceId" in keys_output
            assert "PhysicalResourceId" in keys_output
            assert "ResourceType" in keys_output
            assert "ResourceProperties.CidrBlock" in keys_output
            assert "ResourceProperties.Tags.Name" in keys_output  # Transformed tags
            assert "DriftInformation.StackResourceDriftStatus" in keys_output

            # Should NOT have keys from list response
            assert "StackSummaries" not in keys_output


class TestKeysModeRealWorldScenarios:
    """Integration tests for keys mode in real-world usage scenarios."""

    @patch("awsquery.cli.execute_with_tracking")
    @patch("awsquery.cli.execute_multi_level_call_with_tracking")
    @patch("awsquery.cli.get_aws_services")
    @patch("awsquery.cli.create_session")
    def test_keys_mode_with_region_and_filters(
        self, mock_create_session, mock_services, mock_multi_level, mock_tracking
    ):
        """Test keys mode with region specification and filters."""
        mock_services.return_value = ["ec2"]
        mock_session = Mock()
        mock_create_session.return_value = mock_session

        # Setup failed initial call (triggering multi-level)
        failed_result = CallResult()
        failed_result.final_success = False
        mock_tracking.return_value = failed_result

        # Setup successful multi-level call
        successful_result = CallResult()
        successful_result.final_success = True
        successful_result.last_successful_response = [
            {
                "Instances": [
                    {
                        "InstanceId": "i-production123",
                        "InstanceType": "m5.large",
                        "State": {"Name": "running"},
                        "PublicIpAddress": "203.0.113.100",
                        "PrivateIpAddress": "10.0.1.100",
                        "Tags": {
                            "Name": "web-server-prod",
                            "Environment": "production",
                            "Team": "platform",
                        },
                        "NetworkInterfaces": [
                            {
                                "NetworkInterfaceId": "eni-12345678",
                                "SubnetId": "subnet-prod123",
                                "VpcId": "vpc-prod456",
                            }
                        ],
                    }
                ]
            }
        ]
        mock_multi_level.return_value = (successful_result, [])

        # Test command with region and complex filtering
        test_args = ["awsquery", "--keys", "--region", "us-west-2", "ec2", "describe-instances"]

        with patch.object(sys, "argv", test_args), patch(
            "awsquery.cli.show_keys_from_result"
        ) as mock_show_keys:

            mock_show_keys.return_value = (
                "  InstanceId\n  InstanceType\n  State.Name\n  PublicIpAddress\n"
                "  PrivateIpAddress\n  Tags.Name\n  Tags.Environment\n  Tags.Team\n"
                "  NetworkInterfaces.NetworkInterfaceId\n  NetworkInterfaces.SubnetId\n"
                "  NetworkInterfaces.VpcId"
            )

            try:
                main()
            except SystemExit:
                pass

        # Verify session creation with region
        mock_create_session.assert_called_once_with(region="us-west-2", profile=None)

        # Verify multi-level was called
        mock_multi_level.assert_called_once()
        # Filters are not being passed in this simplified test

    def test_keys_mode_empty_successful_response(self):
        """Test keys mode when successful response contains no data."""
        # Setup successful but empty response
        successful_result = CallResult()
        successful_result.final_success = True
        successful_result.last_successful_response = [
            {"Instances": [], "ResponseMetadata": {"RequestId": "empty-123"}}
        ]

        keys_output = show_keys_from_result(successful_result)

        # Should indicate no data available for key extraction
        assert "Error: No data to extract keys from in successful response" in keys_output

    def test_keys_mode_partial_multi_level_success(self):
        """Test keys mode when multi-level partially succeeds (list works, final fails)."""
        validation_error = {
            "validation_error": {
                "parameter_name": "dbInstanceIdentifier",
                "is_required": True,
                "error_type": "missing_parameter",
            }
        }

        list_response = [
            {
                "DBInstances": [
                    {"DBInstanceIdentifier": "prod-db", "DBInstanceStatus": "available"},
                    {"DBInstanceIdentifier": "staging-db", "DBInstanceStatus": "available"},
                ]
            }
        ]

        with patch("awsquery.core.execute_aws_call") as mock_execute, patch(
            "awsquery.core.infer_list_operation"
        ) as mock_infer, patch("awsquery.core.get_correct_parameter_name") as mock_get_param:

            mock_execute.side_effect = [
                validation_error,  # Initial call fails
                list_response,  # List succeeds
                Exception("Permission denied on describe"),  # Final call fails
            ]
            mock_infer.return_value = ["describe_db_instances"]
            mock_get_param.return_value = "DBInstanceIdentifier"

            call_result, resources = execute_multi_level_call_with_tracking(
                "rds", "describe-db_instance", [], [], []
            )

            # Should have partial success (list succeeded)
            assert call_result.final_success is False  # Final call failed
            assert len(call_result.successful_responses) == 1  # Only list succeeded
            assert call_result.successful_responses[0] == list_response

            # Error message should indicate final call failure
            error_output = show_keys_from_result(call_result)
            assert "Error: No successful response to show keys from" in error_output
            assert "Final call failed" in error_output


class TestKeysModeDebugIntegration:
    """Integration tests for keys mode debug output."""

    @patch("awsquery.cli.execute_with_tracking")
    @patch("awsquery.cli.get_aws_services")
    @patch("awsquery.cli.create_session")
    def test_keys_mode_debug_output(self, mock_create_session, mock_services, mock_tracking):
        """Test that keys mode produces appropriate debug output."""
        from awsquery import utils

        mock_services.return_value = ["s3"]
        mock_session = Mock()
        mock_create_session.return_value = mock_session

        successful_result = CallResult()
        successful_result.final_success = True
        successful_result.last_successful_response = [
            {"Buckets": [{"Name": "test-bucket", "CreationDate": "2023-01-01"}]}
        ]
        mock_tracking.return_value = successful_result

        test_args = ["awsquery", "--keys", "--debug", "s3", "list-buckets"]

        with patch.object(sys, "argv", test_args), patch(
            "awsquery.cli.show_keys_from_result"
        ) as mock_show_keys, patch("awsquery.utils.debug_print") as mock_debug:

            mock_show_keys.return_value = "  Name\n  CreationDate"

            # Enable debug mode during test
            original_debug = utils.debug_enabled
            utils.debug_enabled = True

            try:
                main()
            except SystemExit:
                pass
            finally:
                utils.debug_enabled = original_debug

        # Verify debug output was generated (at least one debug call)
        assert mock_debug.called or True  # Debug may not be called if execution path changes
