"""
Network Security Audit Framework

Cross-platform security auditing framework that performs automated security
assessments on Unix/Linux and Windows systems.
"""

__version__ = "1.0.0"

import platform
import os
import subprocess
import shutil
import tempfile


def main():
    """Main entry point for the security audit framework."""
    # Get Operating System type
    os_type = platform.system()

    if os_type == "Windows":
        # Check if PowerShell is available (Windows default)
        ps_available = shutil.which("powershell") is not None

        if ps_available:
            # Get the package directory
            package_dir = os.path.dirname(os.path.abspath(__file__))

            # Copy scripts to Windows temp directory to avoid UNC path issues
            temp_dir = tempfile.mkdtemp(prefix="network_audit_")

            # Copy windows folder to temp
            windows_src = os.path.join(package_dir, "windows")
            windows_dst = os.path.join(temp_dir, "windows")
            shutil.copytree(windows_src, windows_dst)

            # Execute from temp directory
            audit_script = os.path.join(windows_dst, "audit.ps1")

            try:
                subprocess.run([
                    "powershell",
                    "-ExecutionPolicy", "Bypass",
                    "-File", audit_script
                ], check=True)

                # Copy logs and reports back to original location
                temp_logs = os.path.join(windows_dst, "logs")
                temp_reports = os.path.join(windows_dst, "reports")
                orig_logs = os.path.join(windows_src, "logs")
                orig_reports = os.path.join(windows_src, "reports")

                if os.path.exists(temp_logs):
                    if os.path.exists(orig_logs):
                        shutil.rmtree(orig_logs)
                    shutil.copytree(temp_logs, orig_logs)

                if os.path.exists(temp_reports):
                    if os.path.exists(orig_reports):
                        shutil.rmtree(orig_reports)
                    shutil.copytree(temp_reports, orig_reports)

            finally:
                # Cleanup temp directory
                try:
                    shutil.rmtree(temp_dir)
                except:
                    pass
        else:
            print("PowerShell not found. Please install PowerShell to run Windows audits.")

    elif os_type == "Linux":
        sh_type = subprocess.check_output(["ps", "-p", str(os.getppid()), "-o", "comm="]).decode().strip()

        if sh_type == "bash" or sh_type == "zsh" or sh_type == "sh":
            # Get the package directory
            package_dir = os.path.dirname(os.path.abspath(__file__))
            audit_script = os.path.join(package_dir, "unix", "audit.sh")
            subprocess.run(["bash", audit_script])
        else:
            print("Unsupported Shell Type: ", sh_type)

    else:
        print("Unsupported OS: ", os_type)


if __name__ == "__main__":
    main()
