### Functions ###

## ANSI 코드
RED=$'\e[1;31m'         # BOLD + RED
GREEN=$'\e[1;32m'       # BOLD + GREEN
BOLD=$'\e[1m'           # BOLD
RESET=$'\e[0m'          # 원래 글씨로 전환

## ASCII 배너 출력
print_banner() {
    if [ -z "$BANNER_PRINTED" ]; then
        local DARK_RED=$'\e[2;31m'

        printf "${DARK_RED}"
        cat << 'EOF'


        ███████╗███████╗██████╗ ██╗   ██╗               █████╗ ██╗   ██╗██████╗ ██╗████████╗
        ██╔════╝██╔════╝██╔══██╗██║   ██║              ██╔══██╗██║   ██║██╔══██╗██║╚══██╔══╝
        ███████╗█████╗  ██████╔╝██║   ██║    █████╗    ███████║██║   ██║██║  ██║██║   ██║
        ╚════██║██╔══╝  ██╔══██╗╚██╗ ██╔╝    ╚════╝    ██╔══██║██║   ██║██║  ██║██║   ██║
        ███████║███████╗██║  ██║ ╚████╔╝               ██║  ██║╚██████╔╝██████╔╝██║   ██║
        ╚══════╝╚══════╝╚═╝  ╚═╝  ╚═══╝                ╚═╝  ╚═╝ ╚═════╝ ╚═════╝ ╚═╝   ╚═╝


EOF
        printf "$RESET\n"
        export BANNER_PRINTED=1
    fi
}

## 배너 출력
print_banner

## 화면에 출력한 내용을 모두 log 파일에 저장
log() {
    printf '%s\n' "$*" | tee -a "$LOG"
}

## 표준 에러 출력 후 log 파일에 저장
err() {
    printf "$RED[ERROR] %s$RESET\n" "$*" | tee -a "$LOG" >&2
}

## 백업 파일 생성 EX: /etc/securetty -> /etc/securetty.bak.20251031-000000
backup() {
    local src="$1"
    local ts
    ts=$(date '+%Y%m%d-%H%M%S')
    local name="$src.bak.$ts"

    if [ -f "$src" ]; then
        cp -p "$src" "$name"
        log "$BOLD[BACKUP]$RESET $src -> $name"
    fi
}

## U-37: Apache AllowOverride AuthConfig 보안 점검 및 조치
U-37() {
    local TS="$(date '+%Y-%m-%dT%H:%M:%S%z')"            # 점검 실행 날짜(KST) EX: 2025-10-31T00:00:00+0900
    local DIR="$(cd "$(dirname "$0")" && pwd)"           # 점검 스크립트 경로
    LOG="$DIR/logs/U-37.log"                             # 로그 파일 저장 경로 및 파일명
    local REPORT="$DIR/reports/U-37.json"                # 보고서 저장 경로 및 파일명

    mkdir -p "$(dirname "$LOG")" "$(dirname "$REPORT")"  # 로그, 보고서 파일의 상위 디렉터리가 없으면 생성

    ## 점검 할 파일 (우선순위: httpd.conf > apache2.conf)
    local AUDIT_FILE=""
    if [ -f "/etc/httpd/conf/httpd.conf" ]; then
        AUDIT_FILE="/etc/httpd/conf/httpd.conf"          # Apache 2.4 이하 버전
    elif [ -f "/etc/apache2/apache2.conf" ]; then
        AUDIT_FILE="/etc/apache2/apache2.conf"           # Apache 2.4 이상 버전
    fi

    ## Apache 활성화 여부 및 AuthConfig 설정 여부
    local APACHE_ACTIVE="no"
    local ALLOWOVERRIDE_AUTHCONFIG="no"

    ## 점검 시 탐색 및 수정 결과 기록 (PASS/FAIL)
    local AUDIT_STATUS="FAIL"
    local REMEDIATE_STATUS="FAIL"

    log ""
    log "$BOLD============= [U-37] Apache AllowOverride AuthConfig Assessment =============$RESET"
    log ""

    if [ -z "$AUDIT_FILE" ]; then
        log "$RED[ERROR] Apache configuration file not found$RESET"
        AUDIT_STATUS="FAIL"
        REMEDIATE_STATUS="FAIL"
    else
        log "$GREEN[INFO] Apache assessment file: $AUDIT_FILE$RESET"
    fi

    ## TCP/80번 또는 443번 포트 LISTENING이면 Apache 서비스 활성화로 판단
    if ss -ltn '( sport = :80 or sport = :443 )' 2>/dev/null | grep -q .; then
        APACHE_ACTIVE="yes"
    fi

    if [ "$APACHE_ACTIVE" = "yes" ] && [ -n "$AUDIT_FILE" ]; then
        ## 주석이 아닌 AllowOverride 설정 중 None이 있는지 확인
        ## AllowOverride None이 하나라도 있으면 FAIL
        if [ -r "$AUDIT_FILE" ]; then
            if grep -Ev '^[[:space:]]*#' "$AUDIT_FILE" | \
               grep -Eq '^[[:space:]]*AllowOverride[[:space:]]+None[[:space:]]*$'; then
                ALLOWOVERRIDE_AUTHCONFIG="no"
            else
                ## None이 없으면 AuthConfig 또는 All이 설정되어 있는지 확인
                if grep -Ev '^[[:space:]]*#' "$AUDIT_FILE" | \
                   grep -Eq '^[[:space:]]*AllowOverride[[:space:]]+.*(AuthConfig|All)'; then
                    ALLOWOVERRIDE_AUTHCONFIG="yes"
                fi
            fi
        fi
    fi

    ## Apache가 비활성화 상태이거나 None이 없고 AuthConfig/All이 있으면 PASS
    if [ "$APACHE_ACTIVE" = "no" ] || [ "$ALLOWOVERRIDE_AUTHCONFIG" = "yes" ]; then
        AUDIT_STATUS="PASS"
    fi

    log ""
    log "$BOLD[DETECTION RESULT]$RESET"
    log "  Apache Service: $APACHE_ACTIVE"
    log "  AllowOverride AuthConfig: $ALLOWOVERRIDE_AUTHCONFIG"
    log ""

    ## 점검 결과 임시 저장
    local BF_ALLOWOVERRIDE_AUTHCONFIG=$ALLOWOVERRIDE_AUTHCONFIG

    if [ "$AUDIT_STATUS" = "FAIL" ]; then
        if [ "$ALLOWOVERRIDE_AUTHCONFIG" = "no" ]; then
            backup "$AUDIT_FILE"

            ## Step 1&2: Apache 설정 파일에서 설정된 모든 디렉터리의 AllowOverride 지시자에서 None을 AuthConfig로 변경
            sed -i 's/^\([[:space:]]*AllowOverride[[:space:]]\+\)None/\1AuthConfig/' "$AUDIT_FILE"
            log "$GREEN[Step 1&2]$RESET Changed AllowOverride None to AuthConfig"
            ALLOWOVERRIDE_AUTHCONFIG="yes"

            ## Step 3: 사용자 인증을 설정할 디렉터리에 .htaccess 파일 생성
            local DOCROOT=$(grep -E '^[[:space:]]*DocumentRoot' "$AUDIT_FILE" 2>/dev/null | head -1 | awk '{print $2}' | tr -d '"')

            ## 못 찾으면 기본값 사용
            if [ -z "$DOCROOT" ]; then
                if [ -d "/var/www/html" ]; then
                    DOCROOT="/var/www/html"
                    log "$YELLOW[WARNING] DocumentRoot not found in config, using default: $DOCROOT$RESET"
                fi
            fi

            if [ -n "$DOCROOT" ] && [ -d "$DOCROOT" ]; then
                local HTACCESS="$DOCROOT/.htaccess"
                local HTPASSWD_FILE="$DOCROOT/.auth"

                if [ ! -f "$HTACCESS" ]; then
                    cat > "$HTACCESS" <<HTACCESS_EOF
AuthName "Directory User Authentication"
AuthType Basic
AuthUserFile $HTPASSWD_FILE
Require valid-user
HTACCESS_EOF
                    log "$GREEN[Step 3]$RESET Created .htaccess at $HTACCESS"
                else
                    log "$GREEN[Step 3]$RESET .htaccess already exists (skipped)"
                fi

                ## Step 4: 사용자 인증에 사용할 아이디 및 패스워드 생성
                if [ ! -f "$HTPASSWD_FILE" ]; then
                    if command -v htpasswd >/dev/null 2>&1; then
                        log "$GREEN[Step 4]$RESET htpasswd -c $HTPASSWD_FILE webadmin"
                        htpasswd -c "$HTPASSWD_FILE" webadmin
                    else
                        log "$YELLOW[WARNING] htpasswd command not found. Install apache2-utils$RESET"
                        log "  Manual: htpasswd -c $HTPASSWD_FILE webadmin"
                    fi
                else
                    log "$GREEN[Step 4]$RESET htpasswd file already exists (skipped)"
                fi
            else
                log "$YELLOW[WARNING] DocumentRoot not found. Skipping Steps 3-4$RESET"
            fi

            ## Step 5: Apache 데몬 재시작
            if command -v systemctl >/dev/null 2>&1; then
                if systemctl is-active --quiet apache2 2>/dev/null; then
                    systemctl restart apache2 2>/dev/null && \
                        log "$GREEN[Step 5]$RESET Apache restarted successfully" || \
                        log "$RED[Step 5]$RESET Failed to restart Apache"
                elif systemctl is-active --quiet httpd 2>/dev/null; then
                    systemctl restart httpd 2>/dev/null && \
                        log "$GREEN[Step 5]$RESET Apache (httpd) restarted successfully" || \
                        log "$RED[Step 5]$RESET Failed to restart Apache (httpd)"
                else
                    log "$BOLD[Step 5]$RESET Apache not running (skipped)"
                fi
            else
                log "$BOLD[Step 5]$RESET systemctl not available (manual restart required)"
            fi

            ## 수정 후 재확인
            if grep -Ev '^[[:space:]]*#' "$AUDIT_FILE" | \
               grep -Eq '^[[:space:]]*AllowOverride[[:space:]]+.*AuthConfig'; then
                REMEDIATE_STATUS="PASS"
            fi
        fi
    else
        log "$GREEN[RESULT] AllowOverride AuthConfig already configured$RESET"
        REMEDIATE_STATUS="PASS"
    fi

    cat > "$REPORT" <<EOF
{
  "date": "$TS",
  "control_family": "3. Service Management > 3.19 Web Service",
  "check_target": "Prevent upper directory access through authentication",
  "discussion": "Good: When upper directory access is restricted by requiring user authentication, preventing unauthorized access to parent directories.\\nVulnerable: When AllowOverride is set to None, .htaccess authentication cannot be used, allowing potential unauthorized access to sensitive directories without authentication.",
  "check_content": "Confirm that the AllowOverride directive's AuthConfig option is set in httpd.conf file to enable directory-level authentication\\n#vi /[Apache_home]/conf/httpd.conf\\nCheck if 'AllowOverride None' needs to be changed to 'AllowOverride AuthConfig'",
  "fix_text": "[SOLARIS, LINUX, AIX, HP-UX]\\nStep 1) Open /[Apache_home]/conf/httpd.conf with vi editor\\nStep 2) Change AllowOverride from None to AuthConfig in Directory directives to enable .htaccess authentication\\n(Before) <Directory \\\"/usr/local/apache2/htdocs\\\">\\n    AllowOverride None\\n    Allow from all\\n</Directory>\\n(After) <Directory \\\"/usr/local/apache2/htdocs\\\">\\n    AllowOverride AuthConfig\\n    Allow from all\\n</Directory>\\nStep 3) Create .htaccess file in directories requiring authentication\\nStep 4) Create user accounts with htpasswd command",
  "payload": {
    "severity": "high",
    "port": [80, 443],
    "service": ["apache", "httpd"],
    "protocol": "TCP",
    "threat": ["Unauthorized Directory Access", "Insufficient Access Control", "Missing Authentication"],
    "TTP": ["T1190", "T1078"],
    "files_checked": ["$AUDIT_FILE"]
  },
  "results": [
    {
      "phase": "detect",
      "status": "$AUDIT_STATUS",
      "apache_active": "$APACHE_ACTIVE",
      "allowoverride_authconfig": "$BF_ALLOWOVERRIDE_AUTHCONFIG"
    },
    {
      "phase": "remediate",
      "status": "$REMEDIATE_STATUS",
      "apache_active": "$APACHE_ACTIVE",
      "allowoverride_authconfig": "$ALLOWOVERRIDE_AUTHCONFIG"
    }
  ]
}
EOF

    log ""
    log "$BOLD[REPORT]$RESET"
    cat "$REPORT" | tee -a "$LOG"
    log ""
    log "$BOLD============= [U-37] Assessment Complete =============$RESET"
    log ""
}

## U-39: Apache FollowSymLinks 보안 점검 및 조치
U-39() {
    local TS="$(date '+%Y-%m-%dT%H:%M:%S%z')"              # 점검 실행 날짜(KST) EX: 2025-10-31T00:00:00+0900
    local DIR="$(cd "$(dirname "$0")" && pwd)"             # 점검 스크립트 경로
    LOG="$DIR/logs/U-39.log"                               # 로그 파일 저장 경로 및 파일명
    local REPORT="$DIR/reports/U-39.json"                  # 보고서 저장 경로 및 파일명

    mkdir -p "$(dirname "$LOG")" "$(dirname "$REPORT")"    # 로그, 보고서 파일의 상위 디렉터리가 없으면 생성

    ## 점검 할 파일 (우선순위: httpd.conf > apache2.conf)
    local AUDIT_FILE=""
    if [ -f "/etc/httpd/conf/httpd.conf" ]; then
        AUDIT_FILE="/etc/httpd/conf/httpd.conf"          # Apache 2.4 이하 버전
    elif [ -f "/etc/apache2/apache2.conf" ]; then
        AUDIT_FILE="/etc/apache2/apache2.conf"           # Apache 2.4 이상 버전
    fi

    ## Apache 활성화 여부 및 FollowSymLinks 설정 여부
    local APACHE_ACTIVE="no"
    local FOLLOWSYMLINKS_ENABLED="no"
    local ALIAS_ENABLED="no"

    ## 점검 시 탐색 및 수정 결과 기록 (PASS/FAIL)
    local AUDIT_STATUS="FAIL"
    local REMEDIATE_STATUS="FAIL"

    log ""
    log "$BOLD============= [U-39] Apache FollowSymLinks Assessment =============$RESET"
    log ""

    if [ -z "$AUDIT_FILE" ]; then
        log "$RED[ERROR] Apache configuration file not found$RESET"
        AUDIT_STATUS="FAIL"
        REMEDIATE_STATUS="FAIL"
    else
        log "$GREEN[INFO] Apache assessment file: $AUDIT_FILE$RESET"
    fi

    ## TCP/80번 또는 443번 포트 LISTENING이면 Apache 서비스 활성화로 판단
    if ss -ltn '( sport = :80 or sport = :443 )' 2>/dev/null | grep -q .; then
        APACHE_ACTIVE="yes"
    fi

    if [ "$APACHE_ACTIVE" = "yes" ] && [ -n "$AUDIT_FILE" ]; then
        ## Options 지시자에서 FollowSymLinks 옵션이 설정되어 있는지 확인
        ## 주석이 아닌 Options 설정 중 FollowSymLinks를 포함하는 경우 취약으로 판단
        ## -FollowSymLinks는 명시적 비활성화이므로 양호로 판단
        if [ -r "$AUDIT_FILE" ]; then
            if grep -Ev '^[[:space:]]*#' "$AUDIT_FILE" | \
               grep -Eq '^[[:space:]]*Options[[:space:]]+.*[^-]FollowSymLinks'; then
                FOLLOWSYMLINKS_ENABLED="yes"
            fi

            ## Alias 및 ScriptAlias 지시자 확인
            if grep -Ev '^[[:space:]]*#' "$AUDIT_FILE" | \
               grep -Eq '^[[:space:]]*(Alias|ScriptAlias)[[:space:]]+'; then
                ALIAS_ENABLED="yes"
            fi
        fi
    fi

    ## Apache가 비활성화 상태이거나 FollowSymLinks와 Alias 모두 비활성화 상태면 PASS
    if [ "$APACHE_ACTIVE" = "no" ] || ([ "$FOLLOWSYMLINKS_ENABLED" = "no" ] && [ "$ALIAS_ENABLED" = "no" ]); then
        AUDIT_STATUS="PASS"
    fi

    log ""
    log "$BOLD[DETECTION RESULT]$RESET"
    log "  Apache Service: $APACHE_ACTIVE"
    log "  FollowSymLinks: $FOLLOWSYMLINKS_ENABLED"
    log "  Alias/ScriptAlias: $ALIAS_ENABLED"
    log ""

    ## 점검 결과 임시 저장
    local BF_FOLLOWSYMLINKS_ENABLED=$FOLLOWSYMLINKS_ENABLED
    local BF_ALIAS_ENABLED=$ALIAS_ENABLED

    if [ "$AUDIT_STATUS" = "FAIL" ]; then
        if [ "$FOLLOWSYMLINKS_ENABLED" = "yes" ]; then
            backup "$AUDIT_FILE"

            ## Step 1&2: 설정된 모든 디렉터리의 Options 지시자에서 FollowSymLinks 옵션 제거
            ## 가이드: "FollowSymLinks 삭제 또는 -FollowSymLinks"
            sed -i 's/^\([[:space:]]*Options[[:space:]]\+\)\(.*\)FollowSymLinks\(.*\)/\1\2\3/' "$AUDIT_FILE"
            log "$GREEN[Step 1&2]$RESET Removed FollowSymLinks from Options"
            FOLLOWSYMLINKS_ENABLED="no"

            ## 수정 후 재확인
            if ! grep -Ev '^[[:space:]]*#' "$AUDIT_FILE" | \
                 grep -Eq '^[[:space:]]*Options[[:space:]]+.*[^-]FollowSymLinks'; then
                FOLLOWSYMLINKS_ENABLED="no"
            fi
        fi

        if [ "$ALIAS_ENABLED" = "yes" ]; then
            if [ ! -f "$AUDIT_FILE.bak" ]; then
                backup "$AUDIT_FILE"
            fi

            ## Alias 및 ScriptAlias 지시자를 주석 처리
            ## 가이드: "심볼릭 링크, aliases 사용 제한"
            sed -i 's/^\([[:space:]]*\)\(Alias\|ScriptAlias\)\([[:space:]]\+\)/\1#\2\3/' "$AUDIT_FILE"
            log "$GREEN[Step 3]$RESET Commented out Alias and ScriptAlias directives"
            ALIAS_ENABLED="no"

            ## 수정 후 재확인
            if ! grep -Ev '^[[:space:]]*#' "$AUDIT_FILE" | \
                 grep -Eq '^[[:space:]]*(Alias|ScriptAlias)[[:space:]]+'; then
                ALIAS_ENABLED="no"
            fi
        fi

        ## 모든 취약점이 해결되면 PASS
        if [ "$FOLLOWSYMLINKS_ENABLED" = "no" ] && [ "$ALIAS_ENABLED" = "no" ]; then
            REMEDIATE_STATUS="PASS"
        fi
    else
        log "$GREEN[RESULT] FollowSymLinks and Alias restrictions already configured$RESET"
        REMEDIATE_STATUS="PASS"
    fi

    cat > "$REPORT" <<EOF
{
  "date": "$TS",
  "control_family": "3. Service Management > 3.21 Web Service Symbolic Link Restriction",
  "check_target": "Prohibit symbolic link and aliases usage in web service",
  "discussion": "Good: When symbolic link and aliases usage is restricted to prevent system privilege escalation.\\nVulnerable: When FollowSymLinks option is enabled or Alias/ScriptAlias directives are configured, attackers can access sensitive system directories outside DocumentRoot, potentially exposing critical files and escalating privileges.",
  "check_content": "Confirm that FollowSymLinks option is NOT set in Options directive and Alias/ScriptAlias directives are restricted in httpd.conf file\\n#vi /[Apache_home]/conf/httpd.conf\\nCheck: Options Indexes FollowSymLinks (should be removed or -FollowSymLinks)\\nCheck: Alias and ScriptAlias directives (should be reviewed and restricted)",
  "fix_text": "[SOLARIS, LINUX, AIX, HP-UX]\\nStep 1) Open /[Apache_home]/conf/httpd.conf with vi editor\\n#vi /[Apache_home]/conf/httpd.conf\\nStep 2) Remove FollowSymLinks option from Options directive in all configured directories\\n(Before) <Directory />\\n    Options Indexes FollowSymLinks\\n    AllowOverride None\\n</Directory>\\n(After) <Directory />\\n    Options Indexes\\n    AllowOverride None\\n</Directory>\\nStep 3) Comment out or review Alias and ScriptAlias directives that may expose sensitive directories",
  "payload": {
    "severity": "high",
    "port": [80, 443],
    "service": ["apache", "httpd"],
    "protocol": "TCP",
    "threat": ["Symbolic Link Exploitation", "Privilege Escalation", "Unauthorized File Access", "Information Disclosure"],
    "TTP": ["T1190", "T1068", "T1083"],
    "files_checked": ["$AUDIT_FILE"]
  },
  "results": [
    {
      "phase": "detect",
      "status": "$AUDIT_STATUS",
      "apache_active": "$APACHE_ACTIVE",
      "followsymlinks_enabled": "$BF_FOLLOWSYMLINKS_ENABLED",
      "alias_enabled": "$BF_ALIAS_ENABLED"
    },
    {
      "phase": "remediate",
      "status": "$REMEDIATE_STATUS",
      "apache_active": "$APACHE_ACTIVE",
      "followsymlinks_enabled": "$FOLLOWSYMLINKS_ENABLED",
      "alias_enabled": "$ALIAS_ENABLED"
    }
  ]
}
EOF

    log ""
    log "$BOLD[REPORT]$RESET"
    cat "$REPORT" | tee -a "$LOG"
    log ""
    log "$BOLD============= [U-39] Assessment Complete =============$RESET"
    log ""
}