﻿# Windows Security Functions
# Encoding: UTF-8

# ASCII 배너 출력
function Print-Banner {
    if (-not $Global:BANNER_PRINTED) {
        Write-Host @"
███████╗███████╗██████╗ ██╗   ██╗               █████╗ ██╗   ██╗██████╗ ██╗████████╗
██╔════╝██╔════╝██╔══██╗██║   ██║              ██╔══██╗██║   ██║██╔══██╗██║╚══██╔══╝
███████╗█████╗  ██████╔╝██║   ██║    █████╗    ███████║██║   ██║██║  ██║██║   ██║
╚════██║██╔══╝  ██╔══██╗╚██╗ ██╔╝    ╚════╝    ██╔══██║██║   ██║██║  ██║██║   ██║
███████║███████╗██║  ██║ ╚████╔╝               ██║  ██║╚██████╔╝██████╔╝██║   ██║
╚══════╝╚══════╝╚═╝  ╚═╝  ╚═══╝                ╚═╝  ╚═╝ ╚═════╝ ╚═════╝ ╚═╝   ╚═╝
"@ -ForegroundColor DarkRed
        Write-Host ""
        $Global:BANNER_PRINTED = $true
    }
}

# 배너 출력
Print-Banner

# 화면에 출력한 내용 그대로 log 파일로 저장 (자동 색상 적용)
function Write-Log {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $output = "[$timestamp] $Message"

    # 메시지 타입에 따라 색상 자동 지정
    $color = "White"
    if ($Message -match '^\[INFO\]') {
        $color = "Cyan"
    } elseif ($Message -match '^\[SUCCESS\]') {
        $color = "Green"
    } elseif ($Message -match '^\[WARNING\]') {
        $color = "Yellow"
    } elseif ($Message -match '^\[RESULT\]') {
        $color = "Magenta"
    } elseif ($Message -match '^\[BACKUP\]') {
        $color = "Gray"
    } elseif ($Message -match '^={3,}') {
        # 타이틀 라인 (===)
        $color = "Cyan"
        Write-Host $output -ForegroundColor $color
        Add-Content -Path $script:LOGFILE -Value $output
        return
    } elseif ($Message -match '^-{3,}') {
        # 구분선 (---)
        $color = "DarkGray"
    }

    Write-Host $output -ForegroundColor $color
    Add-Content -Path $script:LOGFILE -Value $output
}

# 표준 에러로 출력 후 log 파일에 저장
function Write-ErrorLog {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $output = "[$timestamp] [ERROR] $Message"
    Write-Host $output -ForegroundColor Red
    Add-Content -Path $script:LOGFILE -Value $output
}

# 백업 파일 생성 EX: C:\file.inf -> C:\file.inf.bak.20250829-000000
function Backup-File {
    param([string]$SourcePath)

    if (Test-Path $SourcePath) {
        $timestamp = Get-Date -Format "yyyyMMdd-HHmmss"
        $backupPath = "$SourcePath.bak.$timestamp"
        Copy-Item -Path $SourcePath -Destination $backupPath -Force
        Write-Log "[BACKUP] $SourcePath -> $backupPath"
        return $backupPath
    } else {
        Write-Log "[BACKUP] $SourcePath no backup (file not found)"
        return $null
    }
}

# W-40: 원격 시스템에서 강제로 시스템 종료 권한 제한 (SeRemoteShutdownPrivilege)
function Invoke-W40 {
    $TS = Get-Date -Format "yyyy-MM-ddTHH:mm:sszzz"
    $DIR = Split-Path -Parent $PSScriptRoot
    $script:LOGFILE = Join-Path $DIR "logs\W-40.log"
    $JSON = Join-Path $DIR "reports\W-40.json"

    # 로그 및 리포트 디렉터리 생성
    $logsDir = Join-Path $DIR "logs"
    $reportsDir = Join-Path $DIR "reports"
    if (-not (Test-Path $logsDir)) { New-Item -ItemType Directory -Path $logsDir -Force | Out-Null }
    if (-not (Test-Path $reportsDir)) { New-Item -ItemType Directory -Path $reportsDir -Force | Out-Null }

    # 임시 파일 경로
    $tempSecPol = Join-Path $env:TEMP "secpol_export.inf"
    $tempSecPolClean = Join-Path $env:TEMP "secpol_clean.inf"
    $tempSecDb = Join-Path $env:TEMP "secedit.sdb"

    # 상태 플래그
    $DETECT_STATUS = "FAIL"
    $REMEDIATE_STATUS = "FAIL"
    $REMOTE_SHUTDOWN_USERS = @()
    $BF_REMOTE_SHUTDOWN_USERS = @()
    $AUTHORIZED_ONLY = "no"
    $BF_AUTHORIZED_ONLY = "no"

    Write-Log "============= [W-40] Remote Shutdown Privilege Assessment ============="
    Write-Log "[INFO] Checking SeRemoteShutdownPrivilege policy"

    try {
        # Step 1: 현재 보안 정책 내보내기
        Write-Log "[INFO] Exporting current security policy..."
        $exportResult = secedit /export /cfg $tempSecPol /quiet

        if ($LASTEXITCODE -ne 0) {
            Write-ErrorLog "Failed to export security policy"
            throw "secedit export failed"
        }

        # Step 2: SeRemoteShutdownPrivilege 설정 확인
        Write-Log "[INFO] Parsing security policy for SeRemoteShutdownPrivilege..."

        $policyContent = Get-Content $tempSecPol -Encoding Unicode
        $remoteShutdownLine = $policyContent | Where-Object { $_ -match '^\s*SeRemoteShutdownPrivilege\s*=\s*(.*)$' }

        if ($remoteShutdownLine) {
            $privilegeValue = $Matches[1].Trim()
            Write-Log "[INFO] Current SeRemoteShutdownPrivilege = $privilegeValue"

            # SID 또는 계정명을 배열로 분리
            if ($privilegeValue -ne "" -and $privilegeValue -ne "*S-1-5-32-544") {
                $REMOTE_SHUTDOWN_USERS = $privilegeValue -split ',' | ForEach-Object { $_.Trim() }
            } elseif ($privilegeValue -eq "*S-1-5-32-544") {
                $REMOTE_SHUTDOWN_USERS = @("*S-1-5-32-544")
            } else {
                $REMOTE_SHUTDOWN_USERS = @()
            }

            $BF_REMOTE_SHUTDOWN_USERS = $REMOTE_SHUTDOWN_USERS

            # Administrator 외 다른 계정이 있는지 확인
            $nonAdminUsers = $REMOTE_SHUTDOWN_USERS | Where-Object {
                $_ -ne "*S-1-5-32-544" -and $_ -ne ""
            }

            if ($nonAdminUsers.Count -eq 0 -and $REMOTE_SHUTDOWN_USERS -contains "*S-1-5-32-544") {
                $AUTHORIZED_ONLY = "yes"
                $DETECT_STATUS = "PASS"
            } elseif ($REMOTE_SHUTDOWN_USERS.Count -eq 0) {
                $AUTHORIZED_ONLY = "no"
                $DETECT_STATUS = "FAIL"
            } else {
                $AUTHORIZED_ONLY = "no"
                $DETECT_STATUS = "FAIL"
            }

            $BF_AUTHORIZED_ONLY = $AUTHORIZED_ONLY
        } else {
            Write-Log "[INFO] SeRemoteShutdownPrivilege not found in policy (using default)"
            $DETECT_STATUS = "PASS"
            $AUTHORIZED_ONLY = "yes"
            $BF_AUTHORIZED_ONLY = "yes"
        }

        Write-Log "------------- Detect Result -------------"
        Write-Log "[RESULT] SeRemoteShutdownPrivilege users: $($REMOTE_SHUTDOWN_USERS -join ', ')"
        Write-Log "[RESULT] Authorized only (Administrators): $AUTHORIZED_ONLY"
        Write-Log "[RESULT] Detection status: $DETECT_STATUS"
        Write-Log "------------------------------------"

        # Step 3: 조치 (Administrators 그룹만 남기고 제거)
        if ($DETECT_STATUS -eq "FAIL") {
            Write-Log "[INFO] Remediating: Setting SeRemoteShutdownPrivilege to Administrators only..."

            Backup-File -SourcePath $tempSecPol

            # 새로운 보안 정책 파일 생성
            $newPolicyContent = $policyContent | ForEach-Object {
                if ($_ -match '^\s*SeRemoteShutdownPrivilege\s*=') {
                    "SeRemoteShutdownPrivilege = *S-1-5-32-544"
                } else {
                    $_
                }
            }

            # 정책 파일에 SeRemoteShutdownPrivilege가 없으면 추가
            if (-not ($policyContent | Where-Object { $_ -match '^\s*SeRemoteShutdownPrivilege\s*=' })) {
                Write-Log "[INFO] SeRemoteShutdownPrivilege not found, adding to policy..."
                $privilegeRightsSection = $false
                $newPolicyContent = $policyContent | ForEach-Object {
                    if ($_ -match '^\s*\[Privilege Rights\]') {
                        $privilegeRightsSection = $true
                        $_
                        "SeRemoteShutdownPrivilege = *S-1-5-32-544"
                    } else {
                        $_
                    }
                }

                # [Privilege Rights] 섹션이 없으면 생성
                if (-not $privilegeRightsSection) {
                    $newPolicyContent += "`r`n[Privilege Rights]`r`nSeRemoteShutdownPrivilege = *S-1-5-32-544"
                }
            }

            # UTF-16 LE 인코딩으로 저장
            $newPolicyContent | Out-File -FilePath $tempSecPolClean -Encoding Unicode -Force

            # 정책 적용
            Write-Log "[INFO] Applying new security policy..."
            $importResult = secedit /configure /db $tempSecDb /cfg $tempSecPolClean /quiet

            if ($LASTEXITCODE -ne 0) {
                Write-ErrorLog "Failed to apply security policy"
                $REMEDIATE_STATUS = "FAIL"
            } else {
                Write-Log "[SUCCESS] Security policy applied successfully"

                # gpupdate로 정책 갱신
                Write-Log "[INFO] Refreshing group policy..."
                gpupdate /force /target:computer | Out-Null

                # Step 4: 재확인
                Start-Sleep -Seconds 2
                $verifyResult = secedit /export /cfg $tempSecPol /quiet
                $verifyContent = Get-Content $tempSecPol -Encoding Unicode
                $verifyLine = $verifyContent | Where-Object { $_ -match '^\s*SeRemoteShutdownPrivilege\s*=\s*(.*)$' }

                if ($verifyLine) {
                    $verifyValue = $Matches[1].Trim()
                    $REMOTE_SHUTDOWN_USERS = @($verifyValue)

                    if ($verifyValue -eq "*S-1-5-32-544") {
                        $AUTHORIZED_ONLY = "yes"
                        $REMEDIATE_STATUS = "PASS"
                        Write-Log "[SUCCESS] Verification passed: Only Administrators have remote shutdown privilege"
                    } else {
                        $AUTHORIZED_ONLY = "no"
                        $REMEDIATE_STATUS = "FAIL"
                        Write-ErrorLog "Verification failed: Policy not applied correctly"
                    }
                } else {
                    Write-ErrorLog "Verification failed: Cannot read policy after remediation"
                    $REMEDIATE_STATUS = "FAIL"
                }
            }
        } else {
            Write-Log "[RESULT] Already compliant: Only Administrators have remote shutdown privilege"
            $REMEDIATE_STATUS = "PASS"
        }

    } catch {
        Write-ErrorLog "Exception occurred: $_"
        $DETECT_STATUS = "FAIL"
        $REMEDIATE_STATUS = "FAIL"
    } finally {
        # 임시 파일 정리
        if (Test-Path $tempSecPol) { Remove-Item $tempSecPol -Force -ErrorAction SilentlyContinue }
        if (Test-Path $tempSecPolClean) { Remove-Item $tempSecPolClean -Force -ErrorAction SilentlyContinue }
        if (Test-Path $tempSecDb) { Remove-Item $tempSecDb -Force -ErrorAction SilentlyContinue }
    }

    # Step 5: JSON 리포트 생성
    $jsonContent = @{
        date = $TS
        control_family = "1. Account Management > 1.15 User Rights Assignment"
        check_target = "Restrict remote shutdown privilege to Administrators only"
        discussion = "Good: When only Administrators group has the SeRemoteShutdownPrivilege, preventing unauthorized remote system shutdown.`nVulnerable: When unauthorized users or groups have remote shutdown privilege, they may disrupt system availability or cause denial of service."
        check_content = "Confirm that only Administrators group has 'Force shutdown from a remote system' privilege in Local Security Policy`n1. Run > secpol.msc`n2. Security Settings > Local Policies > User Rights Assignment`n3. Check 'Force shutdown from a remote system' policy`n4. Verify only Administrators group is listed"
        fix_text = "[Windows Server 2012/2016/2019]`nStep 1) Run > secpol.msc`nStep 2) Navigate to Security Settings > Local Policies > User Rights Assignment`nStep 3) Open 'Force shutdown from a remote system' policy`nStep 4) Remove all accounts and groups except Administrators`nStep 5) Apply and run 'gpupdate /force'"
        payload = @{
            severity = "medium"
            port = @(135, 445, 3389)
            service = @("RPC", "SMB", "RDP")
            protocol = "TCP"
            threat = @("Denial of Service", "Unauthorized System Shutdown", "Service Disruption")
            TTP = @("T1489", "T1529")
            policy_checked = @("SeRemoteShutdownPrivilege")
        }
        results = @(
            @{
                phase = "detect"
                status = $DETECT_STATUS
                authorized_only = $BF_AUTHORIZED_ONLY
                users_with_privilege = $BF_REMOTE_SHUTDOWN_USERS
            },
            @{
                phase = "remediate"
                status = $REMEDIATE_STATUS
                authorized_only = $AUTHORIZED_ONLY
                users_with_privilege = $REMOTE_SHUTDOWN_USERS
            }
        )
    }

    $jsonContent | ConvertTo-Json -Depth 10 | Out-File -FilePath $JSON -Encoding UTF8 -Force
    Write-Log "[INFO] Report saved to $JSON"
    Write-Log "============= [W-40] Assessment Complete ============="
}

# W-30: IIS MSADC 가상 디렉토리 존재 여부 및 RDS(Remote Data Service) 취약점 점검
function Invoke-W30 {
    $TS = Get-Date -Format "yyyy-MM-ddTHH:mm:sszzz"
    $DIR = Split-Path -Parent $PSScriptRoot
    $script:LOGFILE = Join-Path $DIR "logs\W-30.log"
    $JSON = Join-Path $DIR "reports\W-30.json"

    # 로그 및 리포트 디렉터리 생성
    $logsDir = Join-Path $DIR "logs"
    $reportsDir = Join-Path $DIR "reports"
    if (-not (Test-Path $logsDir)) { New-Item -ItemType Directory -Path $logsDir -Force | Out-Null }
    if (-not (Test-Path $reportsDir)) { New-Item -ItemType Directory -Path $reportsDir -Force | Out-Null }

    # 상태 플래그
    $DETECT_STATUS = "FAIL"
    $REMEDIATE_STATUS = "FAIL"
    $IIS_INSTALLED = "no"
    $OS_VERSION_SAFE = "no"
    $MSADC_EXISTS = "yes"
    $REGISTRY_EXISTS = "yes"
    $BF_IIS_INSTALLED = "no"
    $BF_OS_VERSION_SAFE = "no"
    $BF_MSADC_EXISTS = "yes"
    $BF_REGISTRY_EXISTS = "yes"

    Write-Log "============= [W-30] IIS MSADC Virtual Directory Assessment ============="
    Write-Log "[INFO] Checking for MSADC virtual directory and RDS vulnerability"

    try {
        # Step 1: Windows 버전 확인
        Write-Log "[INFO] Checking Windows version..."
        $osInfo = Get-WmiObject -Class Win32_OperatingSystem
        $osVersion = $osInfo.Version
        $osCaption = $osInfo.Caption
        $osServicePack = $osInfo.ServicePackMajorVersion

        Write-Log "[INFO] OS: $osCaption (Version: $osVersion, SP: $osServicePack)"

        # Windows 버전 파싱 (major.minor.build)
        $versionParts = $osVersion -split '\.'
        $majorVersion = [int]$versionParts[0]
        $minorVersion = [int]$versionParts[1]

        # Windows 2008 이상 판단 (Vista/2008 = 6.0, 2008 R2 = 6.1, 이상은 자동 양호)
        if ($majorVersion -gt 6 -or ($majorVersion -eq 6 -and $minorVersion -ge 0)) {
            Write-Log "[INFO] Windows 2008 or later detected - automatically considered safe"
            $OS_VERSION_SAFE = "yes"
            $DETECT_STATUS = "PASS"
            $BF_OS_VERSION_SAFE = "yes"

            # Windows 2008 이상이면 추가 점검 불필요
            Write-Log "------------- Detect Result -------------"
            Write-Log "[RESULT] OS Version Safe: $OS_VERSION_SAFE"
            Write-Log "[RESULT] Detection status: $DETECT_STATUS"
            Write-Log "------------------------------------"
            $REMEDIATE_STATUS = "PASS"
        } else {
            # Windows 2008 미만: 추가 점검 필요
            Write-Log "[INFO] Windows version older than 2008 - performing additional checks"

            # Step 2: IIS 설치 여부 확인
            Write-Log "[INFO] Checking if IIS is installed..."
            $iisService = Get-Service -Name W3SVC -ErrorAction SilentlyContinue

            if ($null -eq $iisService) {
                Write-Log "[INFO] IIS (W3SVC) service not found - IIS not installed"
                $IIS_INSTALLED = "no"
                $DETECT_STATUS = "PASS"
                $BF_IIS_INSTALLED = "no"
            } else {
                Write-Log "[INFO] IIS (W3SVC) service found - IIS is installed"
                $IIS_INSTALLED = "yes"
                $BF_IIS_INSTALLED = "yes"

                # Step 3: Windows 2000 SP4 또는 2003 SP2 이상 확인
                Write-Log "[INFO] Checking service pack version..."
                $spSafe = $false

                # Windows 2000 (5.0) SP4 이상
                if ($majorVersion -eq 5 -and $minorVersion -eq 0 -and $osServicePack -ge 4) {
                    Write-Log "[INFO] Windows 2000 SP4 or later detected"
                    $spSafe = $true
                }
                # Windows 2003 (5.2) SP2 이상
                elseif ($majorVersion -eq 5 -and $minorVersion -eq 2 -and $osServicePack -ge 2) {
                    Write-Log "[INFO] Windows 2003 SP2 or later detected"
                    $spSafe = $true
                }

                if ($spSafe) {
                    Write-Log "[INFO] Service pack version is safe"
                    $DETECT_STATUS = "PASS"
                } else {
                    # Step 4: MSADC 가상 디렉토리 존재 여부 확인
                    Write-Log "[INFO] Checking for MSADC virtual directory in IIS..."

                    # IIS 메타베이스 또는 파일 시스템에서 MSADC 확인
                    $msadcPath = "$env:SystemDrive\inetpub\wwwroot\msadc"
                    $msadcExists = $false

                    # 파일 시스템 확인
                    if (Test-Path $msadcPath) {
                        Write-Log "[WARNING] MSADC directory found at $msadcPath"
                        $msadcExists = $true
                    }

                    # IIS 메타베이스 확인 (레거시 IIS 6.0 이하)
                    try {
                        $iisVdir = Get-WmiObject -Namespace "root\MicrosoftIISv2" -Class IIsWebVirtualDirSetting -ErrorAction SilentlyContinue | Where-Object { $_.Name -like "*msadc*" }
                        if ($iisVdir) {
                            Write-Log "[WARNING] MSADC virtual directory found in IIS metabase"
                            $msadcExists = $true
                        }
                    } catch {
                        Write-Log "[INFO] Could not query IIS metabase (may not be applicable for this IIS version)"
                    }

                    if (-not $msadcExists) {
                        Write-Log "[INFO] MSADC virtual directory not found"
                        $MSADC_EXISTS = "no"
                        $BF_MSADC_EXISTS = "no"
                        $DETECT_STATUS = "PASS"
                    } else {
                        $MSADC_EXISTS = "yes"
                        $BF_MSADC_EXISTS = "yes"

                        # Step 5: 레지스트리 키 확인
                        Write-Log "[INFO] Checking RDS registry keys..."
                        $registryExists = $false
                        $registryPaths = @(
                            "HKLM:\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\RDSServer.DataFactory",
                            "HKLM:\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\AdvancedDataFactory",
                            "HKLM:\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\VbBusObj.VbBusObjCls"
                        )

                        foreach ($regPath in $registryPaths) {
                            if (Test-Path $regPath) {
                                Write-Log "[WARNING] Registry key exists: $regPath"
                                $registryExists = $true
                            }
                        }

                        if (-not $registryExists) {
                            Write-Log "[INFO] No RDS registry keys found"
                            $REGISTRY_EXISTS = "no"
                            $BF_REGISTRY_EXISTS = "no"
                            $DETECT_STATUS = "PASS"
                        } else {
                            $REGISTRY_EXISTS = "yes"
                            $BF_REGISTRY_EXISTS = "yes"
                            $DETECT_STATUS = "FAIL"
                        }
                    }
                }
            }

            Write-Log "------------- Detect Result -------------"
            Write-Log "[RESULT] IIS Installed: $IIS_INSTALLED"
            Write-Log "[RESULT] OS Version Safe: $OS_VERSION_SAFE"
            Write-Log "[RESULT] MSADC Exists: $MSADC_EXISTS"
            Write-Log "[RESULT] Registry Exists: $REGISTRY_EXISTS"
            Write-Log "[RESULT] Detection status: $DETECT_STATUS"
            Write-Log "------------------------------------"

            # Step 6: 조치 (MSADC 디렉토리 및 레지스트리 제거)
            if ($DETECT_STATUS -eq "FAIL") {
                Write-Log "[INFO] Remediating: Removing MSADC virtual directory and RDS registry keys..."

                $remediationSuccess = $true

                # MSADC 디렉토리 제거
                if (Test-Path $msadcPath) {
                    try {
                        Write-Log "[INFO] Removing MSADC directory: $msadcPath"
                        Backup-File -SourcePath $msadcPath
                        Remove-Item -Path $msadcPath -Recurse -Force -ErrorAction Stop
                        Write-Log "[SUCCESS] MSADC directory removed"
                        $MSADC_EXISTS = "no"
                    } catch {
                        Write-ErrorLog "Failed to remove MSADC directory: $_"
                        $remediationSuccess = $false
                    }
                }

                # IIS 메타베이스에서 가상 디렉토리 제거 (IIS 6.0 이하)
                try {
                    $iisVdir = Get-WmiObject -Namespace "root\MicrosoftIISv2" -Class IIsWebVirtualDirSetting -ErrorAction SilentlyContinue | Where-Object { $_.Name -like "*msadc*" }
                    if ($iisVdir) {
                        foreach ($vdir in $iisVdir) {
                            Write-Log "[INFO] Removing IIS virtual directory: $($vdir.Name)"
                            $vdir.Delete()
                        }
                        Write-Log "[SUCCESS] MSADC virtual directory removed from IIS"
                    }
                } catch {
                    Write-Log "[INFO] Could not remove from IIS metabase (may not be applicable): $_"
                }

                # 레지스트리 키 제거
                foreach ($regPath in $registryPaths) {
                    if (Test-Path $regPath) {
                        try {
                            Write-Log "[INFO] Removing registry key: $regPath"
                            Remove-Item -Path $regPath -Recurse -Force -ErrorAction Stop
                            Write-Log "[SUCCESS] Registry key removed: $regPath"
                        } catch {
                            Write-ErrorLog "Failed to remove registry key $regPath : $_"
                            $remediationSuccess = $false
                        }
                    }
                }

                # Step 7: 재확인
                Write-Log "[INFO] Verifying remediation..."
                Start-Sleep -Seconds 1

                $msadcStillExists = Test-Path $msadcPath
                $registryStillExists = $false
                foreach ($regPath in $registryPaths) {
                    if (Test-Path $regPath) {
                        $registryStillExists = $true
                        break
                    }
                }

                if (-not $msadcStillExists) {
                    $MSADC_EXISTS = "no"
                }

                if (-not $registryStillExists) {
                    $REGISTRY_EXISTS = "no"
                }

                if (-not $msadcStillExists -and -not $registryStillExists -and $remediationSuccess) {
                    $REMEDIATE_STATUS = "PASS"
                    Write-Log "[SUCCESS] Remediation completed successfully"
                } else {
                    $REMEDIATE_STATUS = "FAIL"
                    Write-ErrorLog "Remediation verification failed"
                }
            } else {
                Write-Log "[RESULT] Already compliant: No MSADC vulnerability detected"
                $REMEDIATE_STATUS = "PASS"
            }
        }

    } catch {
        Write-ErrorLog "Exception occurred: $_"
        $DETECT_STATUS = "FAIL"
        $REMEDIATE_STATUS = "FAIL"
    }

    # Step 8: JSON 리포트 생성
    $jsonContent = @{
        date = $TS
        control_family = "3. Service Management > 3.3 IIS Service"
        check_target = "Remove IIS MSADC virtual directory and RDS registry keys"
        discussion = "Good: When Windows 2008 or later is used, or MSADC virtual directory does not exist, or RDS registry keys are not present, the system is protected from RDS Data Factory vulnerability.`nVulnerable: When MSADC virtual directory exists in IIS default website or RDS registry keys are present on older Windows versions, attackers may exploit Remote Data Service to execute arbitrary code or access database."
        check_content = "[Windows 2000/2003]`n1. Check MSADC virtual directory: Start > Run > INETMGR > Select website > Check if 'msadc' directory exists`n2. Check registry keys:`n   - HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\RDSServer.DataFactory`n   - HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\AdvancedDataFactory`n   - HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\VbBusObj.VbBusObjCls`n3. Verify that Windows 2000 SP4 or Windows 2003 SP2 or later is installed"
        fix_text = "[Windows 2000/2003]`nStep 1) Remove '/msadc' virtual directory from website`n   Start > Run > INETMGR > Select website > Right-click on 'msadc' directory > Delete`nStep 2) Remove the following registry keys/directories:`n   - HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\RDSServer.DataFactory`n   - HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\AdvancedDataFactory`n   - HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\VbBusObj.VbBusObjCls`nStep 3) Restart IIS service: iisreset"
        payload = @{
            severity = "high"
            port = @(80, 443)
            service = @("IIS", "World Wide Web Publishing Service")
            protocol = "TCP"
            threat = @("Remote Code Execution", "Information Disclosure", "Unauthorized Database Access", "RDS Data Factory Exploitation")
            TTP = @("T1190", "T1505.003", "T1046")
            directories_checked = @("C:\inetpub\wwwroot\msadc")
            registry_checked = @(
                "HKLM\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\RDSServer.DataFactory",
                "HKLM\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\AdvancedDataFactory",
                "HKLM\SYSTEM\CurrentControlSet\Services\W3SVC\Parameters\ADCLaunch\VbBusObj.VbBusObjCls"
            )
        }
        results = @(
            @{
                phase = "detect"
                status = $DETECT_STATUS
                iis_installed = $BF_IIS_INSTALLED
                os_version_safe = $BF_OS_VERSION_SAFE
                msadc_exists = $BF_MSADC_EXISTS
                registry_exists = $BF_REGISTRY_EXISTS
            },
            @{
                phase = "remediate"
                status = $REMEDIATE_STATUS
                iis_installed = $IIS_INSTALLED
                os_version_safe = $OS_VERSION_SAFE
                msadc_exists = $MSADC_EXISTS
                registry_exists = $REGISTRY_EXISTS
            }
        )
    }

    $jsonContent | ConvertTo-Json -Depth 10 | Out-File -FilePath $JSON -Encoding UTF8 -Force
    Write-Log "[INFO] Report saved to $JSON"
    Write-Log "============= [W-30] Assessment Complete ============="
}

# W-41: 보안 감사를 로그할 수 없는 경우 즉시 시스템 종료 정책 설정 여부 점검
function Invoke-W41 {
    $TS = Get-Date -Format "yyyy-MM-ddTHH:mm:sszzz"
    $DIR = Split-Path -Parent $PSScriptRoot
    $script:LOGFILE = Join-Path $DIR "logs\W-41.log"
    $JSON = Join-Path $DIR "reports\W-41.json"

    # 로그 및 리포트 디렉터리 생성
    $logsDir = Join-Path $DIR "logs"
    $reportsDir = Join-Path $DIR "reports"
    if (-not (Test-Path $logsDir)) { New-Item -ItemType Directory -Path $logsDir -Force | Out-Null }
    if (-not (Test-Path $reportsDir)) { New-Item -ItemType Directory -Path $reportsDir -Force | Out-Null }

    # 레지스트리 경로
    $regPath = "HKLM:\SYSTEM\CurrentControlSet\Control\Lsa"
    $regName = "CrashOnAuditFail"

    # 상태 플래그
    $DETECT_STATUS = "FAIL"
    $REMEDIATE_STATUS = "FAIL"
    $POLICY_ENABLED = "yes"
    $BF_POLICY_ENABLED = "yes"
    $CURRENT_VALUE = -1
    $BF_CURRENT_VALUE = -1

    Write-Log "============= [W-41] CrashOnAuditFail Policy Assessment ============="
    Write-Log "[INFO] Checking 'Audit: Shut down system immediately if unable to log security audits' policy"

    try {
        # Step 1: 레지스트리에서 현재 설정 확인
        Write-Log "[INFO] Reading registry: $regPath\$regName"

        if (Test-Path $regPath) {
            $regValue = Get-ItemProperty -Path $regPath -Name $regName -ErrorAction SilentlyContinue

            if ($null -ne $regValue) {
                $CURRENT_VALUE = $regValue.$regName
                $BF_CURRENT_VALUE = $CURRENT_VALUE
                Write-Log "[INFO] Current CrashOnAuditFail value: $CURRENT_VALUE"

                # 값 해석
                # 0 = 사용 안 함 (양호)
                # 1 = 사용 (취약)
                # 2 = 감사 로그가 가득 찬 상태 (취약)
                if ($CURRENT_VALUE -eq 0) {
                    $POLICY_ENABLED = "no"
                    $DETECT_STATUS = "PASS"
                    Write-Log "[INFO] Policy is disabled (Good)"
                } else {
                    $POLICY_ENABLED = "yes"
                    $DETECT_STATUS = "FAIL"
                    Write-Log "[WARNING] Policy is enabled (Vulnerable): value=$CURRENT_VALUE"
                }

                $BF_POLICY_ENABLED = $POLICY_ENABLED
            } else {
                # 레지스트리 값이 없으면 기본값 0 (사용 안 함) 간주
                Write-Log "[INFO] Registry value not found, assuming default (disabled)"
                $CURRENT_VALUE = 0
                $BF_CURRENT_VALUE = 0
                $POLICY_ENABLED = "no"
                $BF_POLICY_ENABLED = "no"
                $DETECT_STATUS = "PASS"
            }
        } else {
            Write-ErrorLog "Registry path not found: $regPath"
            throw "Registry path not found"
        }

        Write-Log "------------- Detect Result -------------"
        Write-Log "[RESULT] CrashOnAuditFail value: $CURRENT_VALUE"
        Write-Log "[RESULT] Policy enabled: $POLICY_ENABLED"
        Write-Log "[RESULT] Detection status: $DETECT_STATUS"
        Write-Log "------------------------------------"

        # Step 2: 조치 (정책 사용 안 함으로 설정)
        if ($DETECT_STATUS -eq "FAIL") {
            Write-Log "[INFO] Remediating: Setting CrashOnAuditFail to 0 (disabled)..."

            try {
                # 레지스트리 값을 0으로 설정
                Set-ItemProperty -Path $regPath -Name $regName -Value 0 -Type DWord -ErrorAction Stop
                Write-Log "[SUCCESS] Registry value set to 0"

                # Step 3: 재확인
                Start-Sleep -Seconds 1
                $verifyValue = Get-ItemProperty -Path $regPath -Name $regName -ErrorAction SilentlyContinue

                if ($null -ne $verifyValue) {
                    $CURRENT_VALUE = $verifyValue.$regName

                    if ($CURRENT_VALUE -eq 0) {
                        $POLICY_ENABLED = "no"
                        $REMEDIATE_STATUS = "PASS"
                        Write-Log "[SUCCESS] Verification passed: CrashOnAuditFail is now disabled"
                    } else {
                        $POLICY_ENABLED = "yes"
                        $REMEDIATE_STATUS = "FAIL"
                        Write-ErrorLog "Verification failed: Value is still $CURRENT_VALUE"
                    }
                } else {
                    Write-ErrorLog "Verification failed: Cannot read registry after remediation"
                    $REMEDIATE_STATUS = "FAIL"
                }
            } catch {
                Write-ErrorLog "Failed to set registry value: $_"
                $REMEDIATE_STATUS = "FAIL"
            }
        } else {
            Write-Log "[RESULT] Already compliant: Policy is disabled"
            $REMEDIATE_STATUS = "PASS"
        }

    } catch {
        Write-ErrorLog "Exception occurred: $_"
        $DETECT_STATUS = "FAIL"
        $REMEDIATE_STATUS = "FAIL"
    }

    # Step 4: JSON 리포트 생성
    $jsonContent = @{
        date = $TS
        control_family = "2. Security Audit > 2.2 Audit Policy"
        check_target = "Disable 'Audit: Shut down system immediately if unable to log security audits' policy"
        discussion = "Good: When the 'CrashOnAuditFail' policy is disabled (0), the system continues to operate even if security audits cannot be logged, preventing unnecessary system shutdowns.`nVulnerable: When the policy is enabled (1 or 2), the system may shut down unexpectedly if the security log is full or cannot be written, causing denial of service."
        check_content = "Verify that 'Audit: Shut down system immediately if unable to log security audits' policy is disabled`n1. Run > secpol.msc`n2. Navigate to Security Settings > Local Policies > Security Options`n3. Check 'Audit: Shut down system immediately if unable to log security audits' policy`n4. Verify it is set to 'Disabled'`n`nAlternatively, check registry:`n1. Run > regedit`n2. Navigate to HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\Lsa`n3. Check 'CrashOnAuditFail' value`n4. Verify it is set to 0 (disabled)"
        fix_text = "[All Windows Versions]`nStep 1) Run > secpol.msc`nStep 2) Navigate to Security Settings > Local Policies > Security Options`nStep 3) Open 'Audit: Shut down system immediately if unable to log security audits' policy`nStep 4) Set to 'Disabled'`nStep 5) Apply settings`n`nAlternative (Registry):`nStep 1) Run > regedit`nStep 2) Navigate to HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\Lsa`nStep 3) Set 'CrashOnAuditFail' value to 0`nStep 4) Restart the system or run 'gpupdate /force'"
        payload = @{
            severity = "medium"
            port = @()
            service = @("Security Audit", "Event Log")
            protocol = "N/A"
            threat = @("Denial of Service", "System Crash", "Service Disruption", "Audit Log Overflow")
            TTP = @("T1489", "T1562.002")
            registry_checked = @("HKLM\SYSTEM\CurrentControlSet\Control\Lsa\CrashOnAuditFail")
        }
        results = @(
            @{
                phase = "detect"
                status = $DETECT_STATUS
                policy_enabled = $BF_POLICY_ENABLED
                registry_value = $BF_CURRENT_VALUE
            },
            @{
                phase = "remediate"
                status = $REMEDIATE_STATUS
                policy_enabled = $POLICY_ENABLED
                registry_value = $CURRENT_VALUE
            }
        )
    }

    $jsonContent | ConvertTo-Json -Depth 10 | Out-File -FilePath $JSON -Encoding UTF8 -Force
    Write-Log "[INFO] Report saved to $JSON"
    Write-Log "============= [W-41] Assessment Complete ============="
}
