"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueueExtension = exports.TopicSubscription = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const events_targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const subscription = require("aws-cdk-lib/aws-sns-subscriptions");
const sqs = require("aws-cdk-lib/aws-sqs");
const extension_interfaces_1 = require("../extension-interfaces");
/**
 * The `TopicSubscription` class represents an SNS Topic resource that can be subscribed to by the service queues.
 */
class TopicSubscription {
    constructor(props) {
        this.topic = props.topic;
        if (props.topicSubscriptionQueue && props.queue) {
            throw Error('Either provide the `subscriptionQueue` or the `queue` (deprecated) for the topic subscription, but not both.');
        }
        this.subscriptionQueue = props.topicSubscriptionQueue;
        this.queue = props.queue ?? props.topicSubscriptionQueue?.queue;
    }
    /**
     * This method sets up SNS Topic subscriptions for the SQS queue provided by the user. If a `queue` is not provided,
     * the default `eventsQueue` subscribes to the given topic.
     *
     * @param extension `QueueExtension` added to the service
     * @returns the queue subscribed to the given topic
     */
    subscribe(extension) {
        const queue = this.subscriptionQueue?.queue ?? this.queue ?? extension.eventsQueue;
        this.topic.addSubscription(new subscription.SqsSubscription(queue));
        return queue;
    }
}
exports.TopicSubscription = TopicSubscription;
_a = JSII_RTTI_SYMBOL_1;
TopicSubscription[_a] = { fqn: "@aws-cdk-containers/ecs-service-extensions.TopicSubscription", version: "2.0.1-alpha.218" };
/**
 * This hook modifies the application container's environment to
 * add the queue URL for the events queue of the service.
 */
class QueueExtensionMutatingHook extends extension_interfaces_1.ContainerMutatingHook {
    constructor(props) {
        super();
        this.environment = props.environment;
    }
    mutateContainerDefinition(props) {
        return {
            ...props,
            environment: { ...(props.environment || {}), ...this.environment },
        };
    }
}
/**
 * This extension creates a default `eventsQueue` for the service (if not provided) and accepts a list of objects of
 * type `ISubscribable` that the `eventsQueue` subscribes to. It creates the subscriptions and sets up permissions
 * for the service to consume messages from the SQS Queues.
 *
 * It also configures a target tracking scaling policy for the service to maintain an acceptable queue latency by tracking
 * the backlog per task. For more information, please refer: https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-using-sqs-queue.html .
 *
 * The default queue for this service can be accessed using the getter `<extension>.eventsQueue`.
 */
class QueueExtension extends extension_interfaces_1.ServiceExtension {
    constructor(props) {
        super('queue');
        this.subscriptionQueues = new Set();
        this.environment = {};
        this.props = props;
    }
    /**
     * This hook creates (if required) and sets the default queue `eventsQueue`. It also sets up the subscriptions for
     * the provided `ISubscribable` objects.
     *
     * @param service The parent service which this extension has been added to
     * @param scope The scope that this extension should create resources in
     */
    prehook(service, scope) {
        this.parentService = service;
        this.scope = scope;
        let eventsQueue = this.props?.eventsQueue;
        if (!eventsQueue) {
            const deadLetterQueue = new sqs.Queue(this.scope, 'EventsDeadLetterQueue', {
                retentionPeriod: aws_cdk_lib_1.Duration.days(14),
            });
            eventsQueue = new sqs.Queue(this.scope, 'EventsQueue', {
                deadLetterQueue: {
                    queue: deadLetterQueue,
                    maxReceiveCount: 3,
                },
            });
        }
        this._eventsQueue = eventsQueue;
        this._autoscalingOptions = this.props?.scaleOnLatency;
        this.environment[`${this.parentService.id.toUpperCase()}_QUEUE_URI`] = this._eventsQueue.queueUrl;
        if (this.props?.subscriptions) {
            for (const subs of this.props.subscriptions) {
                const subsQueue = subs.subscribe(this);
                if (subsQueue !== this._eventsQueue) {
                    if (subs.subscriptionQueue?.scaleOnLatency && !this._autoscalingOptions) {
                        throw Error(`Autoscaling for a topic-specific queue cannot be configured as autoscaling based on SQS Queues hasn’t been set up for the service '${this.parentService.id}'. If you want to enable autoscaling for this service, please also specify 'scaleOnLatency' in the 'QueueExtension'.`);
                    }
                    const subscriptionQueue = subs.subscriptionQueue ?? {
                        queue: subsQueue,
                    };
                    this.subscriptionQueues.add(subscriptionQueue);
                }
            }
        }
    }
    /**
     * Add hooks to the main application extension so that it is modified to
     * add the events queue URL to the container environment.
     */
    addHooks() {
        const container = this.parentService.serviceDescription.get('service-container');
        if (!container) {
            throw new Error('Queue Extension requires an application extension');
        }
        container.addContainerMutatingHook(new QueueExtensionMutatingHook({
            environment: this.environment,
        }));
    }
    /**
     * After the task definition has been created, this hook grants SQS permissions to the task role.
     *
     * @param taskDefinition The created task definition
     */
    useTaskDefinition(taskDefinition) {
        this._eventsQueue.grantConsumeMessages(taskDefinition.taskRole);
        for (const subsQueue of this.subscriptionQueues) {
            subsQueue.queue.grantConsumeMessages(taskDefinition.taskRole);
        }
    }
    /**
     * When this hook is implemented by extension, it allows the extension
     * to use the service which has been created. It is used to add target tracking
     * scaling policies for the SQS Queues of the service. It also creates an AWS Lambda
     * Function for calculating the backlog per task metric.
     *
     * @param service - The generated service.
     */
    useService(service) {
        if (!this._autoscalingOptions) {
            return;
        }
        if (!this.parentService.scalableTaskCount) {
            throw Error(`Auto scaling target for the service '${this.parentService.id}' hasn't been configured. Please use Service construct to configure 'minTaskCount' and 'maxTaskCount'.`);
        }
        this.addQueueScalingPolicy(this._eventsQueue, this._autoscalingOptions);
        for (const subsQueue of this.subscriptionQueues) {
            const autoscalingOpts = subsQueue.scaleOnLatency ?? this._autoscalingOptions;
            this.addQueueScalingPolicy(subsQueue.queue, autoscalingOpts);
        }
        this.parentService.enableAutoScalingPolicy();
        this.createLambdaFunction(service);
    }
    /**
     * This method adds a target tracking policy based on the backlog per task custom metric
     * to the auto scaling target configured for this service.
     *
     * @param queue The queue for which backlog per task metric is being configured
     * @param queueDelay The auto scaling options for the queue
     */
    addQueueScalingPolicy(queue, queueDelay) {
        const messageProcessingTime = queueDelay.messageProcessingTime.toSeconds();
        const acceptableLatency = queueDelay.acceptableLatency.toSeconds();
        if (messageProcessingTime > acceptableLatency) {
            throw Error(`Message processing time (${messageProcessingTime}s) for the queue cannot be greater acceptable queue latency (${acceptableLatency}s).`);
        }
        const acceptableBacklog = acceptableLatency / messageProcessingTime;
        this.parentService.scalableTaskCount?.scaleToTrackCustomMetric(`${queue.node.id}-autoscaling-policy`, {
            metric: new cloudwatch.Metric({
                namespace: `${this.parentService.environment.id}-${this.parentService.id}`,
                metricName: 'BacklogPerTask',
                dimensionsMap: { QueueName: queue.queueName },
                unit: cloudwatch.Unit.COUNT,
            }),
            targetValue: acceptableBacklog,
        });
    }
    /**
     * This method is used to create the AWS Lambda Function for calculating backlog
     * per task metric and a Cloudwatch event trigger for this function.
     *
     * @param service - The generated service.
     */
    createLambdaFunction(service) {
        const queueNames = [this._eventsQueue.queueName];
        this.subscriptionQueues.forEach(subs => queueNames.push(subs.queue.queueName));
        const backLogPerTaskCalculator = new lambda.Function(this.scope, 'BackLogPerTaskCalculatorFunction', {
            runtime: lambda.Runtime.PYTHON_3_9,
            code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', '..', 'lambda', 'queue')),
            handler: 'index.queue_handler',
            environment: {
                CLUSTER_NAME: this.parentService.cluster.clusterName,
                SERVICE_NAME: service.serviceName,
                NAMESPACE: `${this.parentService.environment.id}-${this.parentService.id}`,
                QUEUE_NAMES: queueNames.join(','),
            },
            initialPolicy: [new iam.PolicyStatement({
                    actions: ['ecs:DescribeServices'],
                    resources: [`${service.serviceArn}`],
                    conditions: {
                        ArnEquals: {
                            'ecs:cluster': this.parentService.cluster.clusterArn,
                        },
                    },
                })],
        });
        const queueArns = [this._eventsQueue.queueArn];
        this.subscriptionQueues.forEach(subs => queueArns.push(subs.queue.queueArn));
        backLogPerTaskCalculator.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: [
                'sqs:GetQueueAttributes',
                'sqs:GetQueueUrl',
            ],
            resources: queueArns,
        }));
        new events.Rule(this.scope, 'BacklogPerTaskScheduledRule', {
            schedule: events.Schedule.rate(aws_cdk_lib_1.Duration.seconds(60)),
            targets: [new events_targets.LambdaFunction(backLogPerTaskCalculator)],
        });
        this.logGroup = new logs.LogGroup(this.scope, `${this.parentService.id}-BackLogPerTaskCalculatorLogs`, {
            logGroupName: `/aws/lambda/${backLogPerTaskCalculator.functionName}`,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            retention: logs.RetentionDays.THREE_DAYS,
        });
    }
    get eventsQueue() {
        return this._eventsQueue;
    }
    get autoscalingOptions() {
        return this._autoscalingOptions;
    }
}
exports.QueueExtension = QueueExtension;
_b = JSII_RTTI_SYMBOL_1;
QueueExtension[_b] = { fqn: "@aws-cdk-containers/ecs-service-extensions.QueueExtension", version: "2.0.1-alpha.218" };
//# sourceMappingURL=data:application/json;base64,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