import uuid
from dataclasses import asdict
from datetime import datetime
from typing import TYPE_CHECKING, Dict, List, Optional, Union

import aiofiles
import aiohttp

from mtmai.chainlit.context import context
from mtmai.chainlit.data import BaseDataLayer, queue_until_user_message
from mtmai.chainlit.element import ElementDict
from mtmai.chainlit.logger import logger
from mtmai.chainlit.step import StepDict
from mtmai.chainlit.types import (
    Feedback,
    FeedbackDict,
    PageInfo,
    PaginatedResponse,
    Pagination,
    ThreadDict,
    ThreadFilter,
)
from mtmai.chainlit.user import PersistedUser
from mtmai.chainlit.user import User as ClUser
from mtmai.db.db import get_async_session
from mtmai.crud import curd_chat
from mtmai.crud.curd import get_user_by_username
from mtmai.crud.curd_chat import get_user_threads
from mtmai.models.chat import ChatStep
from mtmai.models.models import User

if TYPE_CHECKING:
    from mtmai.chainlit.element import Element, ElementDict
    from mtmai.chainlit.step import StepDict

user_thread_limit = 50


class SQLAlchemyDataLayer(BaseDataLayer):
    def __init__(
        self,
    ):
        pass

    async def build_debug_url(self) -> str:
        return ""

    async def get_current_timestamp(self) -> str:
        return datetime.now().isoformat() + "Z"

    def clean_result(self, obj):
        """Recursively change UUID -> str and serialize dictionaries"""
        if isinstance(obj, dict):
            return {k: self.clean_result(v) for k, v in obj.items()}
        elif isinstance(obj, list):
            return [self.clean_result(item) for item in obj]
        elif isinstance(obj, uuid.UUID):
            return str(obj)
        return obj

    def _convert_dbuser_to_PersistedUser(self, db_user: User):
        return PersistedUser(
            id=db_user.id,
            display_name=db_user.username,
            createdAt=str(db_user.created_at),
            metadata=db_user.meta,
            identifier=db_user.username,
        )

    ###### Elements ######
    async def get_element(
        self, thread_id: str, element_id: str
    ) -> Optional["ElementDict"]:
        item = await curd_chat.get_chat_element(
            thread_id=thread_id, element_id=element_id
        )
        if item:
            return item.model_dump()
        return None

    ###### User ######
    async def get_user(self, identifier: str) -> Optional[PersistedUser]:
        async with get_async_session() as session:
            db_user = await get_user_by_username(session=session, username=identifier)
        if db_user:
            return self._convert_dbuser_to_PersistedUser(db_user)
        return None

    async def create_user(self, user: ClUser) -> Optional[PersistedUser]:
        """注意： 这里似乎 会在聊天用户会话开始时就会被调用，功能更像是 获取用户信息，而不是创建用户"""
        return await self.get_user(user.identifier)

    ###### Threads ######
    async def get_thread_author(self, thread_id: str) -> str:
        return await curd_chat.get_thread_author(thread_id)

    async def get_thread(self, thread_id: str) -> Optional[ThreadDict]:
        result = await curd_chat.get_thread(thread_id)
        if result:
            return result.model_dump()
        return None

    async def update_thread(
        self,
        thread_id: str,
        name: Optional[str] = None,
        user_id: Optional[str] = None,
        metadata: Optional[Dict] = None,
        tags: Optional[List[str]] = None,
    ):
        if context.session.user is None:
            raise ValueError("User not found in session context")
        if context.session.user.id is None:
            raise ValueError("User identifier not found in session context")

        result = await curd_chat.update_thread(
            thread_id=thread_id,
            name=name,
            metadata=metadata,
            tags=tags,
            userId=user_id or context.session.user.id,
        )
        if result:
            return result.model_dump()
        return

    async def delete_thread(self, thread_id: str):
        return curd_chat.delete_thread(thread_id)

    async def list_threads(
        self, pagination: Pagination, filters: ThreadFilter
    ) -> PaginatedResponse:
        if not filters.userId:
            raise ValueError("userId is required")
        all_user_threads: List[ThreadDict] = (
            await self.get_all_user_threads(user_id=filters.userId) or []
        )

        search_keyword = filters.search.lower() if filters.search else None
        feedback_value = int(filters.feedback) if filters.feedback else None

        filtered_threads = []
        for thread in all_user_threads:
            keyword_match = True
            feedback_match = True
            if search_keyword or feedback_value is not None:
                if search_keyword:
                    keyword_match = any(
                        search_keyword in step["output"].lower()
                        for step in thread["steps"]
                        if "output" in step
                    )
                if feedback_value is not None:
                    feedback_match = False  # Assume no match until found
                    for step in thread["steps"]:
                        feedback = step.get("feedback")
                        if feedback and feedback.get("value") == feedback_value:
                            feedback_match = True
                            break
            if keyword_match and feedback_match:
                filtered_threads.append(thread)

        start = 0
        if pagination.cursor:
            for i, thread in enumerate(filtered_threads):
                if (
                    thread["id"] == pagination.cursor
                ):  # Find the start index using pagination.cursor
                    start = i + 1
                    break
        end = start + pagination.first
        paginated_threads = filtered_threads[start:end] or []

        has_next_page = len(filtered_threads) > end
        start_cursor = paginated_threads[0]["id"] if paginated_threads else None
        end_cursor = paginated_threads[-1]["id"] if paginated_threads else None

        return PaginatedResponse(
            pageInfo=PageInfo(
                hasNextPage=has_next_page,
                startCursor=str(start_cursor),
                endCursor=str(end_cursor),
            ),
            data=paginated_threads,
        )

    ###### Steps ######
    @queue_until_user_message()
    async def create_step(self, step_dict: "StepDict"):
        """upsert chat step"""
        step_dict["show_input"] = (
            str(step_dict.get("showInput", "")).lower()
            if "showInput" in step_dict
            else None
        )

        chat_step = ChatStep(
            id=step_dict.get("id"),
            name=step_dict.get("name"),
            type=step_dict.get("type"),
            thread_id=step_dict.get("threadId"),
            parent_id=step_dict.get("parentId"),
            disable_feedback=step_dict.get("disableFeedback", False),
            streaming=step_dict.get("streaming", False),
            wait_for_answer=step_dict.get("waitForAnswer"),
            is_error=step_dict.get("isError"),
            meta=step_dict.get("metadata", {}),
            tags=step_dict.get("tags", []),
            input=step_dict.get("input"),
            output=step_dict.get("output"),
            created_at=step_dict.get("createdAt"),
            start=step_dict.get("start"),
            end=step_dict.get("end"),
            generation=step_dict.get("generation", {}),
            show_input=step_dict.get("show_input"),
            language=step_dict.get("language"),
            indent=step_dict.get("indent"),
        )

        # async with AsyncSession(get_async_engine()) as session:
        async with get_async_session() as session:
            await ChatStep.upsert(chat_step, session)
            await session.commit()

        return chat_step

    @queue_until_user_message()
    async def update_step(self, step_dict: "StepDict"):
        await self.create_step(step_dict)

    @queue_until_user_message()
    async def delete_step(self, step_id: str):
        feedbacks_query = """DELETE FROM feedbacks WHERE "forId" = :id"""
        elements_query = """DELETE FROM elements WHERE "forId" = :id"""
        steps_query = """DELETE FROM steps WHERE "id" = :id"""
        parameters = {"id": step_id}
        await self.execute_sql(query=feedbacks_query, parameters=parameters)
        await self.execute_sql(query=elements_query, parameters=parameters)
        await self.execute_sql(query=steps_query, parameters=parameters)

    ###### Feedback ######
    async def upsert_feedback(self, feedback: Feedback) -> str:
        feedback.id = feedback.id or str(uuid.uuid4())
        feedback_dict = asdict(feedback)
        parameters = {
            key: value for key, value in feedback_dict.items() if value is not None
        }

        columns = ", ".join(f'"{key}"' for key in parameters.keys())
        values = ", ".join(f":{key}" for key in parameters.keys())
        updates = ", ".join(
            f'"{key}" = :{key}' for key in parameters.keys() if key != "id"
        )
        query = f"""
            INSERT INTO feedbacks ({columns})
            VALUES ({values})
            ON CONFLICT (id) DO UPDATE
            SET {updates};
        """
        await self.execute_sql(query=query, parameters=parameters)
        return feedback.id

    async def delete_feedback(self, feedback_id: str) -> bool:
        if self.show_logger:
            logger.info(f"SQLAlchemy: delete_feedback, feedback_id={feedback_id}")
        query = """DELETE FROM feedbacks WHERE "id" = :feedback_id"""
        parameters = {"feedback_id": feedback_id}
        await self.execute_sql(query=query, parameters=parameters)
        return True

    ###### Elements ######
    @queue_until_user_message()
    async def create_element(self, element: "Element"):
        # if not getattr(context.session.user, "id", None):
        #     raise ValueError("No authenticated user in context")
        # if not self.storage_provider:
        #     logger.warn(
        #         "SQLAlchemy: create_element error. No blob_storage_client is configured!"
        #     )
        #     return
        if not element.for_id:
            return

        content: Optional[Union[bytes, str]] = None

        if element.path:
            async with aiofiles.open(element.path, "rb") as f:
                content = await f.read()
        elif element.url:
            async with aiohttp.ClientSession() as session:
                async with session.get(element.url) as response:
                    if response.status == 200:
                        content = await response.read()
                    else:
                        content = None
        elif element.content:
            content = element.content
        else:
            raise ValueError("Element url, path or content must be provided")
        if content is None:
            raise ValueError("Content is None, cannot upload file")

        context_user = context.session.user
        await curd_chat.create_chat_element(
            # session=session,
            user_id=context_user.id,
            chat_element_dict=element.to_dict(),
        )

    @queue_until_user_message()
    async def delete_element(self, element_id: str, thread_id: Optional[str] = None):
        if self.show_logger:
            logger.info(f"SQLAlchemy: delete_element, element_id={element_id}")
        query = """DELETE FROM elements WHERE "id" = :id"""
        parameters = {"id": element_id}
        await self.execute_sql(query=query, parameters=parameters)

    async def delete_user_session(self, id: str) -> bool:
        return False  # Not sure why documentation wants this

    async def get_all_user_threads(
        self, user_id: Optional[str] = None, thread_id: Optional[str] = None
    ) -> Optional[List[ThreadDict]]:
        """Fetch all user threads up to self.user_thread_limit, or one thread by id if thread_id is provided."""

        async with get_async_session() as session:
            return await get_user_threads(
                session=session,
                user_id=user_id,
                thread_id=thread_id,
                limit=user_thread_limit,
            )

    async def get_all_user_feedbacks(
        self, thread_ids: list[str] = None
    ) -> Optional[List[FeedbackDict]]:
        pass
