# This file is part of Karabo.
#
# http://www.karabo.eu
#
# Copyright (C) European XFEL GmbH Schenefeld. All rights reserved.
#
# Karabo is free software: you can redistribute it and/or modify it under
# the terms of the MPL-2 Mozilla Public License.
#
# You should have received a copy of the MPL-2 Public License along with
# Karabo. If not, see <https://www.mozilla.org/en-US/MPL/2.0/>.
#
# Karabo is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.
""" This module contains the OLD project code.
"""

import hashlib
import os.path
import urllib.error
import urllib.parse
import urllib.request
from io import BytesIO, StringIO
from uuid import uuid4
from zipfile import ZIP_DEFLATED, ZipFile

from karabo.common.project.api import read_macro
from karabo.common.scenemodel.api import read_scene
from karabo.native.data import decodeXML, encodeXML


class Project:
    def __init__(self, filename):
        super().__init__()

        self.version = 1
        self.filename = filename
        self.uuid = str(uuid4())

        # List of devices
        self.devices = []
        # Map for {deviceId, [ProjectConfiguration]}
        self.configurations = {}
        # List of macro models
        self.macros = []
        self.monitors = []
        self.resources = {}
        # List of scene models
        self.scenes = []

        self.monitorFilename = ""
        self.monitorInterval = 0
        self.isMonitoring = False

    @property
    def name(self):
        """
        This function returns the name of the project excluding the suffix.
        """
        r = os.path.basename(self.filename)
        if r.endswith(".krb"):
            return r[:-4]
        else:
            return r

    def addConfiguration(self, deviceId, configuration, index=-1):
        if deviceId in self.configurations:
            if index < 1:
                self.configurations[deviceId].append(configuration)
            else:
                self.configurations[deviceId].insert(index, configuration)
        else:
            self.configurations[deviceId] = [configuration]

    def addDevice(self, device):
        self.devices.append(device)

    def addDeviceGroup(self, deviceGroup):
        self.devices.append(deviceGroup)

    def addMacro(self, macro):
        self.macros.append(macro)

    def addMonitor(self, monitor):
        self.monitors.append(monitor)

    def addResource(self, category, data):
        """Add the data into the resources of given category

        This returns a URL under which the resource can be opened again
        """
        def _hasPath(zf, path):
            try:
                zf.getinfo(path)
                return True
            except KeyError:
                return False

        with ZipFile(self.filename, mode="a", compression=ZIP_DEFLATED) as zf:
            digest = hashlib.sha1(data).hexdigest()
            respath = f"resources/{category}/{digest}"
            # Duplicate items compute the same hash. Don't write them twice.
            if not _hasPath(zf, respath):
                zf.writestr(respath, data)
        self.resources.setdefault(category, set()).add(digest)
        return f"project:{respath}"

    def addScene(self, scene):
        self.scenes.append(scene)

    def getURL(self, url):
        """Retrieve the URL and return its content

        This method retrieves the content for the URL, where the URL might
        have been generated by addResource, or point to a local file or some
        HTTP site."""
        u = urllib.parse.urlparse(url)
        if u.scheme == "project":
            with ZipFile(self.filename, mode="r") as zf:
                return zf.read(u.path)
        elif u.scheme == "":  # for old projects, delete later
            with open(url) as fin:
                return fin.read()
        else:
            return urllib.request.urlopen(url).read()

    def unzip(self):
        """ Initialize from the zip file `self.filename`.
        """
        read_project(self.filename, instance=self)


class ProjectConfiguration:
    def __init__(self, project, name, hash=None):
        super().__init__()

        if name.endswith(".xml"):
            self.filename = name
        else:
            self.filename = f"{name}.xml"

        self.hash = hash

    def fromXml(self, xmlString):
        self.hash = decodeXML(xmlString)

    def toXml(self):
        return encodeXML(self.hash)


class BaseDevice:
    def __init__(self, serverId, classId, deviceId, ifexists):
        assert ifexists in ("ignore", "restart")

        self.serverId = serverId
        self.classId = classId

        self.filename = f"{deviceId}.xml"
        self.ifexists = ifexists


class BaseDeviceGroup(BaseDevice):
    def __init__(self, serverId, classId, id, ifexists):
        BaseDevice.__init__(self, serverId, classId, id, ifexists)

        self.devices = []

    def addDevice(self, device):
        self.devices.append(device)


class Monitor:
    def __init__(self, name, config=None):
        super().__init__()

        self.name = name
        self.config = config

    @property
    def name(self):
        r = os.path.basename(self.filename)
        if r.endswith(".xml"):
            return r[:-4]
        else:
            return r

    @name.setter
    def name(self, name):
        self.filename = f"{name}.xml"

    def fromXml(self, xmlString):
        self.config = decodeXML(xmlString)

    def toXml(self):
        return encodeXML(self.config)


# -----------------------------------------------------------------------------

PROJECT_KEY = "project"
DEVICES_KEY = "devices"
SCENES_KEY = "scenes"
CONFIGURATIONS_KEY = "configurations"
MACROS_KEY = "macros"
MONITORS_KEY = "monitors"
RESOURCES_KEY = "resources"


def read_project(path, instance=None):
    """ Create a new project instance from a config Hash and open zipfile.
    """
    if instance is not None:
        proj = instance
    else:
        proj = Project(path)

    with ZipFile(path, "r") as zf:
        projectHash = _read_xml_hash(zf, f"{PROJECT_KEY}.xml")

        proj.version = projectHash[PROJECT_KEY, "version"]
        proj.uuid = projectHash[PROJECT_KEY, ...].get("uuid", proj.uuid)

        projectConfig = projectHash[PROJECT_KEY]
        dev_groups, devices = _read_devices(zf, projectConfig)
        for grp in dev_groups:
            proj.addDeviceGroup(grp)
        for dev in devices:
            proj.addDevice(dev)

        _read_configurations(zf, projectConfig, proj)

        macros = _read_macros(zf, projectConfig)
        for m in macros:
            proj.addMacro(m)

        monitors = _read_monitors(zf, projectConfig)
        for mon in monitors:
            proj.addMonitor(mon)

        scenes = _read_scenes(zf, projectConfig)
        for s in scenes:
            proj.addScene(s)

        proj.resources = {k: set(v) for k, v in
                          projectConfig[RESOURCES_KEY].items()}


def _read_configurations(zf, projectConfig, projInstance):
    """ Read all the project configurations from a project zipfile.
    """
    for devId, configList in projectConfig[CONFIGURATIONS_KEY].items():
        # Vector of hashes
        for c in configList:
            filename = c.get("filename")
            configuration = ProjectConfiguration(projInstance, filename)
            data = zf.read(f"{CONFIGURATIONS_KEY}/{filename}")
            configuration.fromXml(data)
            projInstance.addConfiguration(devId, configuration)


def _read_devices(zf, projectConfig):
    """ Read all the devices from a project zipfile.
    """
    device_groups, devices = [], []

    for dev in projectConfig[DEVICES_KEY]:
        group = dev.get("group")
        if group is not None:
            filename = dev.getAttribute("group", "filename")
            assert filename.endswith(".xml")
            filename = filename[:-4]

            configPath = f"{DEVICES_KEY}/{filename}.xml"
            grpConfig = _read_xml_hash(zf, configPath)
            for _, config in grpConfig.items():
                serverId = dev.getAttribute("group", "serverId")
                classId = dev.getAttribute("group", "classId")
                # This is currently for backporting
                if dev.hasAttribute("group", "ifexists"):
                    ifexists = dev.getAttribute("group", "ifexists")
                else:
                    ifexists = "ignore"  # Use default

                devGroup = BaseDeviceGroup(filename, serverId, classId,
                                           ifexists)
                devGroup.initConfig = config
                break  # there better be only one!

            for item in group:
                serverId = item.get("serverId")
                filename = item.get("filename")
                assert filename.endswith(".xml")
                filename = filename[:-4]

                configPath = f"{DEVICES_KEY}/{filename}.xml"
                devConfig = _read_xml_hash(zf, configPath)
                for classId, config in devConfig.items():
                    device = BaseDevice(serverId, classId, filename,
                                        item.get("ifexists"))

                    device.initConfig = config
                    devGroup.addDevice(device)
                    break  # there better be only one!
            device_groups.append(devGroup)
        else:
            serverId = dev.get("serverId")
            filename = dev.get("filename")
            assert filename.endswith(".xml")
            filename = filename[:-4]

            configPath = f"{DEVICES_KEY}/{filename}.xml"
            devConfig = _read_xml_hash(zf, configPath)
            for classId, config in devConfig.items():
                device = BaseDevice(serverId, classId, filename,
                                    dev.get("ifexists"))
                device.initConfig = config
                devices.append(device)
                break  # there better be only one!

    return device_groups, devices


def _read_macros(zf, projectConfig):
    """ Read all the macros from a project zipfile.
    """
    macros = []

    for name in projectConfig[MACROS_KEY]:
        data = zf.read("{}/{}".format(MACROS_KEY, f"{name}.py"))
        macro_model = read_macro(StringIO(data.decode()))
        macro_model.simple_name = name
        macros.append(macro_model)

    return macros


def _read_monitors(zf, projectConfig):
    """ Read all the monitors from a project zipfile.
    """
    monitors = []

    monitorHash = projectConfig.get(MONITORS_KEY)
    if monitorHash is not None:
        for m in monitorHash:
            filename = m.get("filename")
            assert filename.endswith(".xml")
            data = zf.read(f"{MONITORS_KEY}/{filename}")
            filename = filename[:-4]
            monitor = Monitor(filename)
            monitor.fromXml(data)
            monitors.append(monitor)

    return monitors


def _read_scenes(zf, projectConfig):
    """ Read all the scenes from a project zipfile.
    """
    scenes = []

    for s in projectConfig[SCENES_KEY]:
        title = s["filename"]
        data = zf.read(f"{SCENES_KEY}/{title}")
        scene_model = read_scene(BytesIO(data))
        scene_model.simple_name = title
        scenes.append(scene_model)

    return scenes


def _read_xml_hash(zf, path):
    """ Read a Hash object (serialized as XML) from a zipfile.
    """
    return decodeXML(zf.read(path))
