# AUTOGENERATED! DO NOT EDIT! File to edit: ../source_nbs/lib_nbs/analysis.ipynb.

# %% auto 0
__all__ = ['get_angle', 'dataset_angles', 'msd_analysis', 'vacf', 'CH_changepoints', 'CRLB_D', 'generate_covariance_matrix',
           'derivative_covariance_matrix_alpha', 'derivative_covariance_matrix_K_alpha', 'fisher_information_alpha',
           'fisher_information_K_alpha', 'fisher_information_cross', 'fisher_information_matrix', 'p_variation',
           'p_variation_FBM', 'psd']

# %% ../source_nbs/lib_nbs/analysis.ipynb 2
import numpy as np
import math

# %% ../source_nbs/lib_nbs/analysis.ipynb 5
def get_angle(a:tuple, # 2d position point A
              b:tuple, # 2d position point B
              c:tuple # 2d position point C       
             ) -> tuple:  # angle between segments AB and BC points
                
    ''' Calculates the angle between the segments generate by three points '''
    ang = math.degrees(math.atan2(c[1]-b[1], c[0]-b[0]) - math.atan2(a[1]-b[1], a[0]-b[0]))
    return ang + 360 if ang < 0 else ang

def dataset_angles(trajs:list, # set of trajectories from which to calculate angles
                  ) -> list: # list of angles between displacements
    '''Given a set of trajectories, calculate all angles between displacements'''
    angles = []
    for traj in trajs:
        for a, b, c in zip(traj[:, :-2].transpose(), traj[:, 1:-1].transpose(), traj[:, 2:].transpose()):
            angles.append(get_angle(a, b, c))
    return angles

# %% ../source_nbs/lib_nbs/analysis.ipynb 7
class msd_analysis():
    def __init__(self):
        ''' Contains mean squared displacement (MSD) based methods to analyze trajectories.  '''
        

    def tamsd(self, 
              trajs:np.ndarray, 
              t_lags:np.ndarray,
              dim = 1
             ):
        '''
        Calculates the time average mean squared displacement (TA-MSD) of a trajectory at various time lags,
        
        Parameters
        ----------
        trajs : np.array
            Set of trajectories of dimenions NxTxD (N: number of trajectories, T: lenght, D: dimension)
        
        t_lags : list | np.array
            Time lags used for the TA-MSD
        
        dim : int
            Dimension of the trajectories (currently only 1 and 2 supported)
        
        Returns       
        -------
        np.array
            TA-MSD of each trayectory / t_lag
            
        '''
        tamsd = np.zeros((len(t_lags), trajs.shape[0]), dtype= float)
        
        for idx, tlag in enumerate(t_lags):                  
            tamsd[idx, :] = ((trajs[:, tlag:, :]-trajs[:, :-tlag, :])**2).sum(-1).mean(1)
                                   
            
        return tamsd    

    def get_diff_coeff(self, 
                       trajs:np.ndarray, 
                       t_lags:list = None):
        '''
        Calculates the diffusion coefficient of a trajectory by means of the linear
        fitting of the TA-MSD.
        
        Parameters
        ----------
        traj : np.array
            Set of trajectories of dimenions NxTxD (N: number of trajectories, T: lenght, D: dimension)
        
        t_lags : bool | list
            Time lags used for the TA-MSD.
        
        Returns       
        -------
        np.array
            Diffusion coefficient of the given trajectory.          
        
        '''
        
        # To account for previous versions of this function, we correct if given a single 1D trajectory
        if len(trajs.shape) == 1:
            trajs = trajs[np.newaxis, :, np.newaxis]
        
        if not t_lags:
            N_t_lags = max(4, int(trajs.shape[1]*0.1))
            t_lags = np.arange(1, N_t_lags)

        tasmd = self.tamsd(trajs, t_lags)
        
        return np.polyfit(t_lags, tasmd, deg = 1)[0, :]/2/trajs.shape[-1]

    def get_exponent(self, 
                     trajs:np.ndarray, 
                     t_lags:list = None):
        '''
        Calculates the diffusion coefficient of a trajectory by means of the linear
        fitting of the TA-MSD.
        
        Parameters
        ----------
        traj : np.array
            Set of trajectories of dimenions NxTxD (N: number of trajectories, T: lenght, D: dimension)
        
        t_lags : bool | list
            Time lags used for the TA-MSD.
        
        Returns       
        -------
        np.array
            Diffusion coefficient of the given trajectory.          
        
        '''
        
        # To account for previous versions of this function, we correct if given a single 1D trajectory
        if len(trajs.shape) == 1:
            trajs = trajs[np.newaxis, :, np.newaxis]
        
        if not t_lags:
            N_t_lags = max(4, int(trajs.shape[1]*0.1))
            t_lags = np.arange(1, N_t_lags)

        tasmd = self.tamsd(trajs, t_lags)
        
        return np.polyfit(np.log(t_lags), np.log(tasmd), deg = 1)[0]


    def get_optimal_tlags(self, 
                          x : float, # Reduced square localization error                          
                          T : int, # Lenght of the trajectory
                         ) -> tuple: # Optimal number of t_lags for fitting the intercept a and slope b of the tMSD
        '''
        Given the length of a trajectory and the reduced square localization error (see Eq. (4) in PRE E 85, 061916 (2012),
        returns a tuple with the optimal number of time lags for the fitting the intercept a and slope b of the tMSD.
        '''
        return self._p_min_a(x, T), self._p_min_b(x, T)

    @staticmethod
    def _fa(x): return 2 + 1.6 * x**0.51
    @staticmethod
    def _La(T): return 3 + (4.5 * T**0.4 - 8.5)**1.2
    @staticmethod
    def _fb(x): return 2 + 1.35 * x**0.6
    @staticmethod
    def _Lb(T): return 0.8 + 0.564 * T

    @staticmethod
    def _get_reduced_loc_error(sigma, D, deltaT, R):
        return sigma**2/(D*deltaT)-2*R
    
    def _p_min_a(self,
                 x : float, # Reduced square localization error
                 T : int # Lenght of the trajectory
                ) -> int: # Result of Eq. B3 from PRE E 85, 061916 (2012)
        fa_x = self._fa(x)
        La_N = self._La(T)
        E_fa_La = fa_x * La_N / (fa_x**3 + La_N**3)**(1/3)
        return int(np.round(E_fa_La))
    
    
    
    def _p_min_b(self,
                 x : float, # Reduced square localization error
                 T : int # Lenght of the trajectory
                ) -> int: # Result of Eq. B4 from PRE E 85, 061916 (2012)
        fb_x = self._fb(x)
        Lb_N = self._Lb(T)
        E_fb_Lb = fb_x * Lb_N / (fb_x**3 + Lb_N**3)**(1/3)
        return int(np.round(min(Lb_N, E_fb_Lb)))
    
    

# %% ../source_nbs/lib_nbs/analysis.ipynb 16
def vacf(trajs, 
         delta_t:int | list | np.ndarray  = 1, 
         taus:bool | list | np.ndarray = None):
    '''
    Calculates the velocity autocorrelation function for 
    the given set of trajectories.
    
    Parameters
    ----------
    trajs : np.array
     NxT matrix containing N trajectories of length T.
        
    delta_t : int | list | array
     If not None, the vacf is calculated in the demanded time lags. 
        
    taus : bool |  list | array
     Time windows at wich the vacf is calculated. 
    
    Returns
    -------
    np.array
        VACF of the given trajectories and the given time windows.        
    '''
    if isinstance(delta_t, int): delta_t = [delta_t]
    
    if taus is None: taus = np.arange(1, trajs.shape[1]).astype(int)
        
    V = np.zeros((len(delta_t), len(taus)))
    for idx_d, delta in enumerate(delta_t):
        # Calculate the velocity
        velocity = trajs[: ,delta:] - trajs[:,:-delta]
        velocity /= delta_t   
    
    
        for idx_t, tau in enumerate(taus):
            if tau == 0:
                V[idx_d, idx_t] = (velocity**2).mean()
            else:
                V[idx_d, idx_t] = (velocity[:, :-tau]*velocity[:, tau:]).mean()
        V[idx_d, :] /= V[idx_d, 0]
        
    return V

# %% ../source_nbs/lib_nbs/analysis.ipynb 20
from scipy.spatial import ConvexHull

def CH_changepoints(trajs, 
                    tau:int = 10, 
                    metric:{'volume', 'area'} = 'volume'):
    ''' 
    Computes the changes points a multistate trajectory based on the Convex Hull approach proposed in PRE 96 (022144), 2017.
    
    Parameters
    ----------
    trajs : np.array
     NxT matrix containing N trajectories of length T.
        
    tau : int
     Time window over which the CH is calculated.
        
    metric : {'volume', 'area'} 
     Calculate change points w.r.t. area or volume of CH.
    
    Returns
    -------
    list
        Change points of the given trajectory.
    '''
    
    CPs = []
    for traj in trajs:
        traj = np.array(traj)

        Sd = np.zeros(traj.shape[0]-2*tau)
        for k in range(traj.shape[0]-2*tau):
            if metric == 'volume':
                Sd[k] = ConvexHull(traj[k:(k+2*tau)]).volume
            elif metric == 'area':
                Sd[k] = ConvexHull(traj[k:(k+2*tau)]).area

        below_mean = Sd < Sd.mean()
        cp_traj = np.argwhere(below_mean[1:] != below_mean[:-1])+1
        CPs.append(cp_traj+tau)

    return CPs

# %% ../source_nbs/lib_nbs/analysis.ipynb 24
def CRLB_D(T:int, # Length of the trajectory
           dim:int = 1 # Dimension of the trajectoy
          ) ->float: # Cramér-Rao bound 
    '''
    Returns the Cramer-Rao lower bound S(D)/D for Brownian motion without noise, namely

    S(D) / D = sqrt(2 / (d·T) )
    '''
    return np.sqrt(2 / (dim*T))

# %% ../source_nbs/lib_nbs/analysis.ipynb 29
def generate_covariance_matrix(alpha, K_alpha, N, delta_t=1.0):
    """Generate the covariance matrix for FBM using vectorized operations."""
    indices = np.arange(N)
    diff = np.abs(indices[:, None] - indices)  # Create matrix of |i-j|
    
    term1 = np.abs(diff + 1)**alpha
    term2 = np.abs(diff - 1)**alpha
    term3 = np.abs(diff)**alpha

    covariance_matrix = K_alpha * delta_t**alpha * (term1 + term2 - 2 * term3)
    
    # Correct diagonal elements manually (as term1 and term2 would add 0 at diagonal)
    np.fill_diagonal(covariance_matrix, K_alpha * delta_t**alpha * 2)
    
    return covariance_matrix

def derivative_covariance_matrix_alpha(alpha, K_alpha, N, delta_t=1.0):
    """Calculate the derivative of the covariance matrix with respect to alpha using vectorized operations."""
    indices = np.arange(N)
    diff = np.abs(indices[:, None] - indices).astype(float)  # Create matrix of |i-j|  

    # Calculate the log terms, avoiding log(0) by setting these terms to zero
    log_term1 = np.log(np.abs(diff + 1), where=(diff + 1) > 0, out=np.zeros_like(diff))
    log_term2 = np.log(np.abs(diff - 1), where=(diff - 1) > 0, out=np.zeros_like(diff))
    log_term3 = np.log(np.abs(diff), where=diff > 0, out=np.zeros_like(diff))

    term1 = log_term1 * np.abs(diff + 1)**alpha
    term2 = log_term2 * np.abs(diff - 1)**alpha
    term3 = log_term3 * np.abs(diff)**alpha

    derivative_matrix = K_alpha * delta_t**alpha * (term1 + term2 - 2 * term3)
    
    return derivative_matrix

def derivative_covariance_matrix_K_alpha(alpha, K_alpha, N, delta_t=1.0):
    """Calculate the derivative of the covariance matrix with respect to K_alpha."""
    # The derivative of the covariance matrix Sigma with respect to K_alpha is Sigma / K_alpha,
    # we just compute Sigma for K_alpha = 1
    derivative_matrix = generate_covariance_matrix(alpha, 1.0, N, delta_t)
    return derivative_matrix
    

# %% ../source_nbs/lib_nbs/analysis.ipynb 30
def fisher_information_alpha(alpha, N, delta_t=1.0):
    """Calculate the Fisher Information component I_alpha,alpha."""

    # Note that the K_alpha terms will cancel when performing the dot product, hence we just 
    # set it to a placeholder value:
    K_alpha = 1
    
    Sigma = generate_covariance_matrix(alpha, K_alpha, N, delta_t)
    Sigma_inv = np.linalg.inv(Sigma)
    derivative_Sigma_alpha = derivative_covariance_matrix_alpha(alpha, K_alpha, N, delta_t)
    
    term = np.dot(Sigma_inv, derivative_Sigma_alpha)
    I_alpha_alpha = 0.5 * np.trace(np.dot(term, term))
    
    return I_alpha_alpha


def fisher_information_K_alpha(K_alpha, N, delta_t=1.0):
    """Calculate the Fisher Information component I_K_alpha,K_alpha.
    
    In this case, it can be analytically calculate as N/(2*K_alpha**2)    
    """    
    return N/(2*K_alpha**2) 

def fisher_information_cross(alpha, K_alpha, N, delta_t=1.0):
    """Calculate the Fisher Information cross term I_alpha,K_alpha."""
    Sigma = generate_covariance_matrix(alpha, K_alpha, N, delta_t)
    Sigma_inv = np.linalg.inv(Sigma)
    derivative_Sigma_alpha = derivative_covariance_matrix_alpha(alpha, K_alpha, N, delta_t)
    derivative_Sigma_K_alpha = derivative_covariance_matrix_K_alpha(alpha, K_alpha, N, delta_t)
    
    term_alpha = np.dot(Sigma_inv, derivative_Sigma_alpha)
    term_K_alpha = np.dot(Sigma_inv, derivative_Sigma_K_alpha)
    I_cross = 0.5 * np.trace(np.dot(term_alpha, term_K_alpha))
    
    return I_cross

def fisher_information_matrix(alpha, K_alpha, N, delta_t=1.0):
    """Calculate the full Fisher Information Matrix and its inverse."""
    I_alpha_alpha = fisher_information_alpha(alpha, N, delta_t)
    I_K_alpha_K_alpha = fisher_information_K_alpha(K_alpha, N, delta_t)
    I_cross = fisher_information_cross(alpha, K_alpha, N, delta_t)

    I_matrix = np.array([
        [I_alpha_alpha, I_cross],
        [I_cross, I_K_alpha_K_alpha]
    ])
    
    I_inverse = np.linalg.inv(I_matrix)
    
    return I_matrix, I_inverse

# %% ../source_nbs/lib_nbs/analysis.ipynb 34
def p_variation(traj : np.array, # Trajectory for which to compute the p-variation
                m : int, # Time lag between positions considered in the p-variation (see Eq. (8) of https://doi.org/10.1103/PhysRevE.82.021130)
                p : int # Integer defining the p-variation
               )-> np.array : # p-variation of the current trajectory 
    ''' 
    Given a trajectory, computes the p-variation for the given lag m and variation p.
    '''
    
    N = len(traj)
    sample_indexes = np.arange(0, N-m, m)
    pvar = np.zeros_like(traj)
    
    for t in range(N):
        sample_indexes_t = sample_indexes[sample_indexes < t]
        
        x_diff = np.abs(np.take(traj, sample_indexes_t+m) - np.take(traj, sample_indexes_t))
    
        pvar[t] = sum(x_diff ** p)
        
    return pvar

# %% ../source_nbs/lib_nbs/analysis.ipynb 35
from scipy.special import gamma

def p_variation_FBM(H : float, # Hurst exponent
                    sigma : float, # Standard deviation of the FBM process
                    t : np.array # times for which to compute p^(1/H) (t)
                   )-> np.array: # Theoretical prediction for the p^(1/H) (t)
    ''' 
    Given a Hurst exponent and the std of a FBM process with MSD = sigma^2 t^(2H), computes the predicted p-variation:
    
    p^(1/H) (t) = E(|B(1)|^(1/H))

    See more at Eq. (8) in https://doi.org/10.1103/PhysRevE.82.011129
    '''
    expected_value = sigma**(1/H) * 2**(1/(2*H)) / np.sqrt(np.pi) * gamma(1/(2*H) + 0.5) 
    pvar = t * expected_value
    return pvar

# %% ../source_nbs/lib_nbs/analysis.ipynb 42
from scipy.signal import periodogram

def psd(traj : np.ndarray, # Trajectory of size (T,) (i.e. 1D) or (T, dim)
       )-> tuple: # (f, psd): frequencies f and power spectral density. Sizes f: (floor(T/2)), psd: (floor(T/2))
    
    ''' Calculates the approximate power spectral density based on scipy.periodogram'''

    # Checking dimensions of input trajectory
    if len(traj.shape) == 1:
        dim = 1
    else:
        dim = traj.shape[1]

    if dim == 1:
        return periodogram(traj)

    else:
        psd = []
        for t_dim in traj.transpose(1,0):
            f, psd_dim = periodogram(t_dim)
            psd.append(psd_dim)
        
        return f, np.array(psd).sum(0)
    
