#!/usr/bin/env python3
"""Create a latex document with signs (For printing and glueing to boxes).

Todo:
    rename font size to text scale. Because it is used to scale instead of setting
    a font size. So the 'normal' is not 12 (pt) but 1.
"""
import importlib.resources
import os.path
import logging
import subprocess
from typing import Union, Iterable

from .box import BoxedThing
from .sign import Sign

from . import thingtemplate_latex

TEMPLATE_PACKAGE = thingtemplate_latex
TEMPLATE_PRE = "signlist-header.tex"
    #importlib.resources.read_text("flinventory", "signlist-header.tex"))
TEMPLATE_POST = "signlist-footer.tex"
TEMPLATE_SIGN = "sign.tex"
DUMMY_IMAGE = "dummyImage.jpg"
AUX_DIR = "latexAux"

UNIT = "cm"  # todo: replace by length_unit in options, includes scaling the constants
# estimated by one example word
# at standard fontsize german
STANDARD_LETTER_LENGTH = 0.2  # in cm at font size 12pt
STANDARD_LETTER_HEIGHT = (12 + 1) * 0.03515  # in cm at font size 12 + linesep
# weird number is size of pt in LaTeX in cm
STANDARD_FONT_SIZE_GERMAN = 1
STANDARD_FONTSIZE_ENGLISH = 0.8
# amount of space the location string is shifted down
# (usually negative amount to shift up)
STANDARD_LOCATION_SHIFT_DOWN = r"-.8\baselineskip"
# in symbols:
STANDARD_LINE_LENGTH = 15
# how much bigger german should roughly be than english
GERMAN_TO_ENGLISH_SHARE = 1.25
# how big the image should be in parts of entire sign (in portrait mode)
IMAGE_SHARE = 0.5
# how much of the sign height can actually be used for text (due to margins)
# (in landscape mode)
TEXT_SHARE = 0.7
# minimum ration width/height for using landscape mode
LANDSCAPE_MIN_RATIO = 2
# paper text width (a4 = 21 cm wide) in cm
PAPER_TEXT_WIDTH = 20
# minimal height of sign for including an image (in cm)
MINIMAL_HEIGHT_FOR_IMAGE = 3
STANDARD_WIDTH = 8
# width of signs without given width [cm]
STANDARD_HEIGHT = 8
# height of signs without given height [cm]
MAX_WIDTH = 18  # otherwise too wide for A4 page
# maximum width of a sign [cm]
MAX_HEIGHT = 28  # otherwise too long for A4 page
# maximum height for a sign [cm]

def sanitize_latex_code(latex: Union[str, int, float]):
    """Make a string insertable into LaTeX code.

    Replace &, \\, ....
    """
    try:
        for orig, new in [
            ("\\", "\\textbackslash{}"),
            ("{", "\\{"),
            ("}", "\\}"),
            ("$", "\\$"),
            ("&", "\\&"),
            ("#", "\\#"),
            ("^", ""),  # better: → \textasciicircum{} (requires the textcomp package)
            ("_", "\\_"),
            ("~", "\\textasciitilde{}"),
            ("%", "\\%"),
            ("<", "\\textless{}"),
            (">", "\\textgreater{}"),
            ("|", "\\textbar{}"),
        ]:
            latex = latex.replace(orig, new)
    except AttributeError:
        pass
    return latex


class SignPrinterLaTeX:
    # pylint: disable=too-many-instance-attributes
    """Class encapsulating algorithm for creating printable signs from thing list.

    Bad design choice: you have to create an object but actually this object is never used,
    could all just be module or class functions.
    """

    def __init__(self, paths):
        """Create SignPrinter by reading in template files."""
        # pylint: disable-next=unbalanced-tuple-unpacking
        self.pre, self.sign, self.post = self._read_templates()
        self.output_dir = paths.output_dir
        self.signs_file = paths.output_signs_latex
        self.logger = logging.getLogger(__name__)

    @staticmethod
    def _read_templates() -> list[str]:
        """Read the templates from the files.

        Returns:
            (iterable) main template, css header,
            single size sign html file, double size sign html file
        """
        # todo remove function by using importlib, which gives directly the text
        file_contents = []
        for filename in [TEMPLATE_PRE, TEMPLATE_SIGN, TEMPLATE_POST]:
            file_contents.append(importlib.resources.read_text(TEMPLATE_PACKAGE, filename))
        return file_contents

    def controlled_length(self, sign:Sign, key: str, backup: Union[int, float],
                          max: Union[int, float]):
        """Get the value for the key if it exists and is a number and < max, otherwise backup."""
        value = sign.get(key, backup)
        try:
            value = float(value)
        except ValueError:
            self.logger(f"{key} {value} is not a number. Use {backup} {UNIT} instead.")
            return backup
        if value <= 0:
            self.logger(f"{key} {value} is non-positive and therefore not a valid sign {key}. "
                        f"Use {backup} {UNIT} instead.")
            return backup
        if value == int(value):
            return min(int(value), max)
        return min(value, max)

    def width(self, thing: BoxedThing) -> Union[int, float]:
        """Get the width of a sign, using backup and max size.

        In UNIT.
        """
        return self.controlled_length(thing.sign, key='width', backup=STANDARD_WIDTH, max=MAX_WIDTH)

    def height(self, thing: BoxedThing) -> Union[int, float]:
        """Get the height of the sign of a boxed thing, using backup and max size."""
        return self.controlled_length(thing.sign, key='height', backup=STANDARD_HEIGHT, max=MAX_HEIGHT)

    def if_use_landscape_template(self, thing: BoxedThing):
        """Return if we want to use the landscape template for this thing.

        The landscape template has the image to the right of the text
        instead of below it. So a portrait sign is usually still wider
        than high.
        """
        try:
            return thing.sign.get('landscape')
        except KeyError:
            try:
                return self.width(thing) >= self.width(thing) * LANDSCAPE_MIN_RATIO
            except KeyError:
                return True

    def guess_font_size(self, thing):
        """Guess good font size.

        Based on the length of the name and the size of the sign.

        Returns:
            guessed font size primary language, guessed font size secondary language in UNIT
        """
        # at first we do not support landscape signs
        # if self.if_use_landscape_template(thing):
        #    return self.guess_font_size_landscape(thing)
        return self.guess_font_size_portrait(thing)

    @staticmethod
    def guess_font_size_landscape(thing):
        """Guess good font sizes for the landscape template."""
        assert False, "landscape latex signs are not supperted"
        # for simplicity assume only one line
        width = self.width(thing)
        height = self.height(thing)
        used_width = width - height  # that is approximately the part the image uses
        german = thing.sign.get(('name', 0), '')
        english = thing.sign.get(('name', 1), '')
        german_expected_width_at_standard = len(german) * STANDARD_LETTER_LENGTH
        german_max_by_width = (
                STANDARD_FONT_SIZE_GERMAN * used_width / german_expected_width_at_standard
        )
        german_max_by_height = (
            height
            * TEXT_SHARE
            / (GERMAN_TO_ENGLISH_SHARE + 1)
            * GERMAN_TO_ENGLISH_SHARE
        )
        english_expected_width_at_standard = (
                len(english)
                * STANDARD_LETTER_LENGTH
                * STANDARD_FONTSIZE_ENGLISH
                / STANDARD_FONT_SIZE_GERMAN
        )
        english_max_by_width = (
            STANDARD_FONTSIZE_ENGLISH * used_width / english_expected_width_at_standard
        )
        english_max_by_height = height * TEXT_SHARE / (GERMAN_TO_ENGLISH_SHARE + 1)
        return (
            min(german_max_by_width, german_max_by_height),
            min(english_max_by_width, english_max_by_height),
        )

    def guess_font_size_portrait(self, thing: BoxedThing):
        # pylint: disable=too-many-locals
        """Guess what a good font size is for this sign.

        Based on the length of the name and the size of the sign.

        Returns:
            guessed font size primary, guessed font size secondary in UNIT
        """
        # use german and english as aliases for primary and secondary language
        # because it's easier to read
        german = thing.get(('name', 0), '')
        english = thing.get(('name', 1), '')
        german_words = german.replace("-", "- ").split() or [" "]
        english_words = english.replace("-", " ").split() or [" "]
        # ignore cases with more than 2 lines, should be considered by hand
        max_font_sizes = [[0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0]]
        # there are 4 line number cases, see below in for loop
        GERMAN_INDEX = 0  # pylint: disable=invalid-name
        ENGLISH_INDEX = 1  # pylint: disable=invalid-name
        SUM_INDEX = 2  # pylint: disable=invalid-name
        # self.logger.debug("g: {}; e: {}, width: {}, heigth: {}".format(
        #     german, english, *thing.sign.size  # unpacking pair
        # ))
        width = self.width(thing)
        height = self.width(thing)
        for german_number_lines, english_number_lines, case in [
            (1, 1, 0),
            (1, 2, 1),
            (2, 1, 2),
            (2, 2, 3),
        ]:
            german_length = max(
                (
                    max(len(word) for word in german_words)
                    if german_number_lines == 2
                    else len(german)
                ),
                1,
            )
            english_length = max(
                (
                    max(len(word) for word in english_words)
                    if english_number_lines == 2
                    else len(english)
                ),
                1,
            )
            german_expected_width_at_standard = german_length * STANDARD_LETTER_LENGTH
            german_max_by_width = (
                    STANDARD_FONT_SIZE_GERMAN * width / german_expected_width_at_standard
            )
            text_height = height * (
                1 if height < MINIMAL_HEIGHT_FOR_IMAGE else IMAGE_SHARE
            )
            german_max_by_height = (
                    STANDARD_FONT_SIZE_GERMAN
                    * text_height
                    / STANDARD_LETTER_HEIGHT
                    / (german_number_lines * GERMAN_TO_ENGLISH_SHARE + english_number_lines)
                    * GERMAN_TO_ENGLISH_SHARE
            )
            english_expected_width_at_standard = (
                    english_length
                    * STANDARD_LETTER_LENGTH
                    * STANDARD_FONTSIZE_ENGLISH
                    / STANDARD_FONT_SIZE_GERMAN
            )
            # in factor size compared to normal size
            english_max_by_width = (
                STANDARD_FONTSIZE_ENGLISH * width / english_expected_width_at_standard
            )
            # in cm:
            english_max_by_height = (
                STANDARD_FONTSIZE_ENGLISH
                * text_height
                / STANDARD_LETTER_HEIGHT
                / (german_number_lines * GERMAN_TO_ENGLISH_SHARE + english_number_lines)
            )
            logging.info(
                f"german: {german}, english: {english}, case: {case} "
                f"lines: ({german_number_lines}, {english_number_lines}, {case},"
                f"german_max_by_height: {german_max_by_height}, "
                f"german_max_by_width: {german_max_by_width}, "
                f"english_max_by_height: {english_max_by_height}, "
                f"english_max_by_width: {english_max_by_width}"
            )
            max_font_sizes[case][GERMAN_INDEX] = min(
                [german_max_by_height, german_max_by_width]
            )
            max_font_sizes[case][ENGLISH_INDEX] = min(
                [english_max_by_height, english_max_by_width]
            )
            max_font_sizes[case][SUM_INDEX] = (
                max_font_sizes[case][GERMAN_INDEX] + max_font_sizes[case][ENGLISH_INDEX]
            )
            # self.logger.debug(
            #     "case: {}; gmaxH: {:.3f}; gmaxW: {:.3f}; emaxH: {:.3f}; emaxW: {:.3f}".format(
            #         case, german_max_by_height, german_max_by_width,
            #         english_max_by_height, english_max_by_width
            #     )
            # )
        german_max, english_max, _ = max(
            max_font_sizes, key=lambda case: case[SUM_INDEX]
        )
        # self.logger.debug("used fs: g: {:.3f}, e: {:.3f}".format(german_max, english_max))
        return german_max, english_max

    def get_font_size(self, thing: BoxedThing):
        """Determine font size of sign for thing.

        Take font size in the thing.
        Guess font size if not specified.

        Returns:
            (german font size, english font size)
        """
        default_german_font_size, default_english_font_size = self.guess_font_size(thing)
        german_font_size = thing.sign.get(('fontsize', 0), default_german_font_size)
        english_font_size = thing.sign.get(('fontsize', 1), default_english_font_size)
        logging.info(
            f"{thing.best('name')} font factor: (de) {german_font_size} (en) {english_font_size}"
        )
        return german_font_size, english_font_size

    def create_latex(self, things: Iterable[BoxedThing]):
        """Create latex code (as str) that shows all signs.

        Arguments:
            things: list of things to be described
        """
        content_latex = [self.pre]
        current_line_filled_width = 0
        for thing in [
            tmp_thing for tmp_thing in things if tmp_thing.sign.should_be_printed()
        ]:
            if current_line_filled_width + self.width(thing) > PAPER_TEXT_WIDTH:
                content_latex.append(
                    ""
                )  # empty line in code makes new line (= row) in pdf
                current_line_filled_width = 0
            content_latex.append(self.create_sign(thing))
            current_line_filled_width += self.width(thing)
        content_latex.append(self.post)
        return "\n".join(content_latex)

    def get_values_for_template(self, thing: BoxedThing):
        """Get values for the insertion into the templates.

        Only the values that are common for portrait
        and landscape template:
            widthAdjusted (width - 0.14cm)
            textscaleGerman
            GermanName
            textscaleEnglish
            EnglishName
            imagepath
            imageheight
            location
        """
        german = thing.sign.get(('name', 0), '')
        english = thing.sign.get(('name', 1), '')

        width = self.width(thing)
        height = self.height(thing)
        width = width - 0.14  # todo: fix latex template or create constant
        insertions = {
            "PrimaryName": sanitize_latex_code(german),
            "SecondaryName": sanitize_latex_code(english),
            "location": sanitize_latex_code(str(thing.where)),
            "height": height,
            "widthAdjusted": width,
        }

        if not insertions["location"]:  # empty string
            logging.warning(
                f"Print sign for {german} ({english}) without location."
            )

        # todo: make image height configurable
        if height < MINIMAL_HEIGHT_FOR_IMAGE:
            insertions["imageheight"] = 0
            insertions["vspace"] = r"-1.5\baselineskip"
        else:
            insertions["imageheight"] = height * IMAGE_SHARE
            insertions["vspace"] = "1pt"

        # at first only portrait sign, landscape sign can be implemented later
        # if self.if_use_landscape_template(thing):  # would need different condition
        if image_path := thing.thing.image_path():  # not None or ""
            rel_path_to_image = os.path.relpath(
                image_path,
                self.output_dir
            )
            insertions["imagepath"] = rel_path_to_image
        else:
            if insertions["imageheight"] > 0:
                # otherwise no image is printed and we do not need an unneccessary warning
                logging.getLogger(__name__).warning(
                    f"Missing image for {thing.best('name')}."
                )
            insertions["imagepath"] = DUMMY_IMAGE
            try:
                image_file = open(os.path.join(self.output_dir, DUMMY_IMAGE), mode="bw")
            except FileExistsError:
                pass
            else:
                dummy_image = importlib.resources.read_binary(TEMPLATE_PACKAGE, DUMMY_IMAGE)
                with image_file:
                    image_file.write(dummy_image)
        font_sizes = self.get_font_size(thing)
        insertions["textscalePrimary"] = font_sizes[0]  # /12
        insertions["textscaleSecondary"] = font_sizes[1]  # /12

        insertions["locationShiftDown"] = thing.sign.get('location_shift_down',
                                                         STANDARD_LOCATION_SHIFT_DOWN)
        return insertions

    def create_sign(self, thing):
        """Create a sign based on the template sign.tex."""
        # text that is to be inserted into the template
        insertions = self.get_values_for_template(thing)
        # no landscape yet
        # if self.if_use_landscape_template(thing):
        #    return self.signhtml_landscape.format(
        #        **insertions # unpacking dictionary
        #        )
        return self.sign.format(**insertions)

    def save_signs_latex(self, things: list[BoxedThing]) -> str:
        """Save signs as tex-file to file path.

        Ignore things that should not be printed as saved in things.sign['printed'].

        Arguments:
            things: list of things to visualize
        Returns:
            path to created file
        """
        things.sort(key=lambda t: self.height(t))
        file_signs = os.path.join(self.output_dir, self.signs_file)
        print(f"Print LaTeX file to {file_signs}.")
        with open(file_signs, mode="w", encoding="UTF-8") as latex_file:
            latex_file.write(self.create_latex(things))
        return file_signs

    def create_signs_pdf(self, things: list[BoxedThing]):
        """Create a pdf and latex files with signs."""
        self.save_signs_latex(things)
        latex = subprocess.run(['latexmk', self.signs_file,
                                f'-aux-directory={AUX_DIR}',
                                "-pdf"],
                               cwd=self.output_dir,
                               capture_output=True,
                               text=True)  # text=True makes output a str instead of bytes
        with open(os.path.join(self.output_dir, AUX_DIR, 'latex_output.txt'), mode="w") as stdout_file:
            stdout_file.write(latex.stdout)
        with open(os.path.join(self.output_dir, AUX_DIR, 'latex_error.txt'), mode="w") as stderr_file:
            stderr_file.write(latex.stderr)
        if latex.returncode != 0:
            print(f"Latex finished with returncode {latex.returncode}."
                  f"Check in {os.path.join(self.output_dir, AUX_DIR)} for details.")
