#pragma once

#include <algorithm>
#include <exception>  // std::terminate
#include <format>
#include <functional>
#include <iostream>
#include <string>

#include "scaler/ymq/timestamp.h"
#include "scaler/ymq/utils.h"

namespace scaler {
namespace ymq {

struct Error: public std::exception {
    enum struct ErrorCode {
        Uninit,
        InvalidPortFormat,
        InvalidAddressFormat,
        ConfigurationError,
        SignalNotSupported,
        CoreBug,
        RepetetiveIOSocketIdentity,
        RedundantIOSocketRefCount,
        MultipleConnectToNotSupported,
        MultipleBindToNotSupported,
        InitialConnectFailedWithInProgress,
        SendMessageRequestCouldNotComplete,
        SetSockOptNonFatalFailure,
        IPv6NotSupported,
        RemoteEndDisconnectedOnSocketWithoutGuaranteedDelivery,
        ConnectorSocketClosedByRemoteEnd,
        IOSocketStopRequested,
        BinderSendMessageWithNoAddress,
    };

    // NOTE:
    // Format:
    //    [Timestamp, ": ", Error Explanation, ": ", Other]
    // For user calls errors:
    //     Other := ["Originated from", Function Name, items...]
    // For system calls errors:
    //     Other := ["Originated from", Function Name, "Errno is", strerror(errno), items...]
    template <typename... Args>
    constexpr Error(ErrorCode e, Args&&... args) noexcept
        : _errorCode(e), _logMsg(argsToString(Timestamp {}, convertErrorToExplanation(e), std::forward<Args>(args)...))
    {
    }

    Error() noexcept: _errorCode(ErrorCode::Uninit) {}

    static inline constexpr std::string_view convertErrorToExplanation(ErrorCode e) noexcept
    {
        switch (e) {
            case ErrorCode::Uninit: return "";
            case ErrorCode::InvalidPortFormat: return "Invalid port format, example input \"tcp://127.0.0.1:2345\"";
            case ErrorCode::InvalidAddressFormat:
                return "Invalid address format, example input \"tcp://127.0.0.1:2345\"";
            case ErrorCode::ConfigurationError:
                return "An error generated by system call that's likely due to mis-configuration";
            case ErrorCode::SignalNotSupported:
                return "A function call was interrupted by signal, but signal handling is not supported";
            case ErrorCode::CoreBug: return "Likely a bug within the library";
            case ErrorCode::RepetetiveIOSocketIdentity:
                return "It is NOT allowed to create two IOSocket with the same identity";
            case ErrorCode::RedundantIOSocketRefCount:
                return "It is NOT allowed to hold IOSocket shared_ptr after you try to remove it";
            case ErrorCode::MultipleConnectToNotSupported:
                return "Connect to remote end without the previous such request successfully completed or exceeds "
                       "retry limit is currently NOT supported";
            case ErrorCode::MultipleBindToNotSupported:
                return "Bind to multiple IP addresses is currently not supported";
            case ErrorCode::InitialConnectFailedWithInProgress:
                return "The first connect(2) call to an socket failed with EINPROGRESS, expected";
            case ErrorCode::SendMessageRequestCouldNotComplete:
                return "sendMessage request could not complete. Possibly because the underlying connection has been "
                       "destructed because it exceeds maximum retry limit";
            case ErrorCode::SetSockOptNonFatalFailure: return "sendsockopt(3) gets unfatal failure";
            case ErrorCode::IPv6NotSupported: return "IPv6 is currently not supported";
            case ErrorCode::RemoteEndDisconnectedOnSocketWithoutGuaranteedDelivery:
                return "You are using IOSocket::Unicast or IOSocket::Multicast, which do not support guaranteed "
                       "message delivery, and the connection(s) disconnects";
            case ErrorCode::ConnectorSocketClosedByRemoteEnd:
                return "You have an IOSocket with Connector type but the only connection is closed by remote end";
            case ErrorCode::IOSocketStopRequested: return "Current IOSocket is requested to stop by another thread";
            case ErrorCode::BinderSendMessageWithNoAddress:
                return "You call sendMessage with a Binder IOSocket but failed to provide an address";
        }
        std::cerr << "Unrecognized ErrorCode value, program exits\n";
        std::exit(1);
    }

    constexpr const char* what() const noexcept override { return _logMsg.c_str(); }

    ErrorCode _errorCode;
    std::string _logMsg;
};

}  // namespace ymq
}  // namespace scaler

template <>
struct std::formatter<scaler::ymq::Error, char> {
    template <class ParseContext>
    constexpr ParseContext::iterator parse(ParseContext& ctx) noexcept
    {
        return ctx.begin();
    }

    template <class FmtContext>
    constexpr FmtContext::iterator format(scaler::ymq::Error e, FmtContext& ctx) const noexcept
    {
        return std::ranges::copy(e._logMsg, ctx.out()).out;
    }
};

using UnrecoverableErrorFunctionHookPtr = std::function<void(scaler::ymq::Error)>;

[[noreturn]] inline void defaultUnrecoverableError(scaler::ymq::Error e) noexcept
{
    std::cerr << e.what() << '\n';
    std::exit(1);
}

inline UnrecoverableErrorFunctionHookPtr unrecoverableErrorFunctionHookPtr = defaultUnrecoverableError;

[[noreturn]] inline void unrecoverableError(scaler::ymq::Error e) noexcept
{
    unrecoverableErrorFunctionHookPtr(std::move(e));
    std::exit(1);
}
