import array
import unittest
from enum import IntEnum

from scaler.io.ymq import ymq


class TestTypes(unittest.TestCase):
    def test_exception(self):
        # type checkers misidentify this as "unnecessary" due to the type hints file
        self.assertTrue(issubclass(ymq.YMQException, Exception))  # type: ignore

        exc = ymq.YMQException(ymq.ErrorCode.CoreBug, "oh no")
        self.assertEqual(exc.args, (ymq.ErrorCode.CoreBug, "oh no"))
        self.assertEqual(exc.code, ymq.ErrorCode.CoreBug)
        self.assertEqual(exc.message, "oh no")

    def test_error_code(self):
        self.assertTrue(issubclass(ymq.ErrorCode, IntEnum))  # type: ignore
        self.assertEqual(
            ymq.ErrorCode.ConfigurationError.explanation(),
            "An error generated by system call that's likely due to mis-configuration",
        )

    def test_bytes(self):
        b = ymq.Bytes(b"data")
        self.assertEqual(b.len, len(b))
        self.assertEqual(b.len, 4)
        self.assertEqual(b.data, b"data")

        # would raise an exception if ymq.Bytes didn't support the buffer interface
        m = memoryview(b)
        self.assertTrue(m.obj is b)
        self.assertEqual(m.tobytes(), b"data")

        b = ymq.Bytes()
        self.assertEqual(b.len, 0)
        self.assertTrue(b.data is None)

        b = ymq.Bytes(b"")
        self.assertEqual(b.len, 0)
        self.assertEqual(b.data, b"")

        b = ymq.Bytes(array.array("B", [115, 99, 97, 108, 101, 114]))
        assert b.len == 6
        assert b.data == b"scaler"

    def test_message(self):
        m = ymq.Message(b"address", b"payload")
        assert m.address is not None
        self.assertEqual(m.address.data, b"address")
        self.assertEqual(m.payload.data, b"payload")

        m = ymq.Message(address=None, payload=ymq.Bytes(b"scaler"))
        self.assertTrue(m.address is None)
        self.assertEqual(m.payload.data, b"scaler")

        m = ymq.Message(b"address", payload=b"payload")
        assert m.address is not None
        self.assertEqual(m.address.data, b"address")
        self.assertEqual(m.payload.data, b"payload")

    def test_io_context(self):
        ctx = ymq.IOContext()
        self.assertEqual(ctx.num_threads, 1)

        ctx = ymq.IOContext(2)
        self.assertEqual(ctx.num_threads, 2)

        ctx = ymq.IOContext(num_threads=3)
        self.assertEqual(ctx.num_threads, 3)

    # TODO: backporting to 3.8 broke this somehow
    # it causes a segmentation fault
    # re-enable once fixed
    @unittest.skip("causes segmentation fault")
    def test_io_socket(self):
        # check that we can't create io socket instances directly
        self.assertRaises(TypeError, lambda: ymq.IOSocket())  # type: ignore

    def test_io_socket_type(self):
        self.assertTrue(issubclass(ymq.IOSocketType, IntEnum))  # type: ignore

    def test_bad_socket_type(self):
        ctx = ymq.IOContext()

        # TODO: should the core reject this?
        socket = ctx.createIOSocket_sync("identity", ymq.IOSocketType.Uninit)
        self.assertEqual(socket.socket_type, ymq.IOSocketType.Uninit)
