import * as cloudwatch from "aws-cdk-lib/aws-cloudwatch";
import { Construct } from "constructs";
import { HealthCheckBase, HealthCheckOptions } from "./health-check";
export interface EndpointHealthCheckProps extends HealthCheckOptions {
    /**
     * The domain name that Route53 performs health checks on. Route53 resolves the IP address and performs the lookup.
     *
     * If IP address is given, it's used as the host name.
     *
     * <b>Either DomainName or IpAddress must be specified</b>
     */
    readonly domainName?: string;
    /**
     * The ip address that Route53 performs health checks on. Optionally a domain name may be given.
     *
     * <b>An IP address must be specified if protocol TCP</b>
     */
    readonly ipAddress?: string;
    /**
     * The protocol that Route53 uses to communicate with the endpoint.
     *
     * <b>An IP address must be specified if protocol TCP</b>
     *
     * @default HTTPS
     */
    readonly protocol?: Protocol;
    /**
     * The port that Route53 performs health checks. The port must be between 1 and 65535.
     *
     * @default 80 for HTTP; 443 for HTTPS
     */
    readonly port?: number;
    /**
     * Specify that Route53 sends the host name for TLS negotiation.
     *
     * @default true for HTTPS
     */
    readonly enableSni?: boolean;
    /**
     * The path for HTTP or HTTPS health checks. Provide a string between 1 and 255 length.
     */
    readonly resourcePath?: string;
    /**
     * The search string for HTTP or HTTPS health checks. Route53 will search in the response body. Provide a string between 1 and 255 length.
     */
    readonly searchString?: string;
    /**
     * The number of seconds between the time that Route53 gets a response from your endpoint and the time that it sends the next health check request. Each Route53 health checker makes requests at this interval. Provide a number between 10 and 30.
     *
     * <i>If you choose an interval of 10 and there are 15 health checkers, the endpoint will receive approximately 1 request per second.</i>
     *
     * <b>Can't be changed after HealthCheck is deployed</b>
     */
    readonly requestInterval?: number;
    /**
     * The number of consecutive health checks that an endpoint must pass or fail for Route53 to change the current status of the endpoint between healthy and unhealthy. Provide a number between 1 and 10.
     */
    readonly failureThreshold?: number;
    /**
     * Whether Route53 measures the latency between health checkers in multiple AWS regions and your endpoint, and displays a CloudWatch latency graphs in the Route53 console.
     *
     * <b>Can't be changed after HealthCheck is deployed</b>
     */
    readonly latencyGraphs?: boolean;
    /**
     * The list of regions from which Route53 health checkers check the endpoint.
     *
     * <b>If omitted Route53 performs checks from all health checker regions.</b>
     */
    readonly regions?: HealthCheckerRegions[];
}
/**
 * Create a Route53 HealthCheck that monitors an endpoint either by domain name or by IP address.
 *
 * <b>Example</b>
 * ```typescript
 * new EndpointHealthCheck(stack, "HealthCheck", {
 *   domainName: "pepperize.com",
 * });
 * ```
 * Generates
 * ```yaml
 * Resources:
 *   Type: AWS::Route53::HealthCheck
 *   Properties:
 *     HealthCheckConfig:
 *       FullyQualifiedDomainName: "pepperize.com"
 *       Port: 443
 *       Type: "HTTPS"
 *       EnableSNI: true
 * ```
 * @link https://docs.aws.amazon.com/de_de/AWSCloudFormation/latest/UserGuide/aws-resource-route53-healthcheck.html#aws-resource-route53-healthcheck-properties
 *
 * @resource AWS::Route53::HealthCheck
 */
export declare class EndpointHealthCheck extends HealthCheckBase {
    readonly healthCheckId: string;
    constructor(scope: Construct, id: string, props: EndpointHealthCheckProps);
    /**
     * Returns the CFN HealthCheckType for the given protocol. If undefined returns default HTTPS.
     */
    private healthCheckType;
    /**
     * Sets the default if undefined for HTTP and HTTPS
     */
    private defaultPort;
    /**
     * Enables SNI by default for HTTPS if omitted, otherwise undefined
     */
    private enableSniForHttps;
    /**
     * The percentage of Route53 health checkers that report that the status of the health check is healthy
     *
     * <b>LatencyGraphs has to be enabled</b>
     *
     * Valid statistics: Average (recommended), Minimum, Maximum
     */
    metricHealthCheckPercentageHealthy(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The time in milliseconds that it took Route53 health checkers to establish a TCP connection with the endpoint
     *
     * Valid statistics: Average (recommended), Minimum, Maximum
     */
    metricConnectionTime(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The time in milliseconds that it took Route53 health checkers to complete the SSL/TLS handshake
     *
     * Valid statistics: Average, Minimum, Maximum
     */
    metricSSLHandshakeTime(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The time in milliseconds that it took Route53 health checkers to receive the first byte of the response to an HTTP or HTTPS request
     *
     * Valid statistics: Average (recommended), Minimum, Maximum
     */
    metricTimeToFirstByte(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * The protocol that Route53 uses to communicate with the endpoint.
 */
export declare enum Protocol {
    HTTP = "HTTP",
    HTTPS = "HTTPS",
    TCP = "TCP"
}
/**
 * The regions of health checker from which Route53 performs checks on the endpoint.
 */
export declare enum HealthCheckerRegions {
    US_EAST_1 = "us-east-1",
    US_WEST_1 = "us-west-1",
    US_WEST_2 = "us-west-2",
    EU_WEST_1 = "eu-west-1",
    AP_SOUTHEAST_1 = "ap-southeast-1",
    AP_SOUTHEAST_2 = "ap-southeast-2",
    AP_NORTHEAST_1 = "ap-northeast-1",
    SA_EAST_1 = "sa-east-1"
}
