"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Failover = exports.HealthCheckBase = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const route53 = require("aws-cdk-lib/aws-route53");
/**
 * @internal
 */
class HealthCheckBase extends aws_cdk_lib_1.Resource {
    constructor() {
        super(...arguments);
        this.tags = new aws_cdk_lib_1.TagManager(aws_cdk_lib_1.TagType.STANDARD, "AWS::Route53::HealthCheck");
    }
    /**
     * Import an existing Route53 HealthCheck.
     */
    static fromHealthCheckId(scope, id, healthCheckId) {
        return new (class Imported extends HealthCheckBase {
            constructor() {
                super(scope, id);
                this.healthCheckId = healthCheckId;
            }
        })();
    }
    failover(recordSet, evaluateTargetHealth, failover) {
        const resource = recordSet.node.defaultChild;
        if (!(resource instanceof route53.CfnRecordSet)) {
            throw new Error(`RecordSet ${recordSet.node.path} doesn't have a default child of type CfnRecordSet`);
        }
        resource.healthCheckId = this.healthCheckId;
        if (failover) {
            resource.failover = failover;
        }
        if (evaluateTargetHealth) {
            resource.addPropertyOverride("AliasTarget.EvaluateTargetHealth", evaluateTargetHealth);
        }
    }
    failoverPrimary(recordSet, evaluateTargetHealth) {
        this.failover(recordSet, evaluateTargetHealth, Failover.PRIMARY);
    }
    failoverSecondary(recordSet, evaluateTargetHealth) {
        this.failover(recordSet, evaluateTargetHealth, Failover.SECONDARY);
    }
    /**
     * Return the given named metric for this HealthCheck.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: "AWS/Route53",
            metricName,
            dimensionsMap: { HealthCheckId: this.healthCheckId },
            ...props,
        }).attachTo(this);
    }
    /**
     * Route53 health checkers report that the HealthCheck is healthy or unhealthy.
     */
    metricHealthCheckStatus(props) {
        return this.metric("HealthCheckStatus", { statistic: cloudwatch.Statistic.MINIMUM, ...props });
    }
}
exports.HealthCheckBase = HealthCheckBase;
var Failover;
(function (Failover) {
    /**
     * The primary record set
     */
    Failover["PRIMARY"] = "PRIMARY";
    /**
     * The secondary record set
     */
    Failover["SECONDARY"] = "SECONDARY";
})(Failover = exports.Failover || (exports.Failover = {}));
//# sourceMappingURL=data:application/json;base64,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