"""
Datasheet

Largely adapted from the Atomic Simulation Environment (ASE) database:
    /path_to_ase/data/__init__.py

Data herein:
    chemical_symbols
    atomic_numbers
    atomic_names
    covalent_radii
    reference_states
    molar_volumes
    inteplanar_dist
"""

import numpy as np

# from scipy.constants import N_A

N_A = 6.02214076e23

chemical_symbols = [
    # 0
    "X",
    # 1
    "H",
    "He",
    # 2
    "Li",
    "Be",
    "B",
    "C",
    "N",
    "O",
    "F",
    "Ne",
    # 3
    "Na",
    "Mg",
    "Al",
    "Si",
    "P",
    "S",
    "Cl",
    "Ar",
    # 4
    "K",
    "Ca",
    "Sc",
    "Ti",
    "V",
    "Cr",
    "Mn",
    "Fe",
    "Co",
    "Ni",
    "Cu",
    "Zn",
    "Ga",
    "Ge",
    "As",
    "Se",
    "Br",
    "Kr",
    # 5
    "Rb",
    "Sr",
    "Y",
    "Zr",
    "Nb",
    "Mo",
    "Tc",
    "Ru",
    "Rh",
    "Pd",
    "Ag",
    "Cd",
    "In",
    "Sn",
    "Sb",
    "Te",
    "I",
    "Xe",
    # 6
    "Cs",
    "Ba",
    "La",
    "Ce",
    "Pr",
    "Nd",
    "Pm",
    "Sm",
    "Eu",
    "Gd",
    "Tb",
    "Dy",
    "Ho",
    "Er",
    "Tm",
    "Yb",
    "Lu",
    "Hf",
    "Ta",
    "W",
    "Re",
    "Os",
    "Ir",
    "Pt",
    "Au",
    "Hg",
    "Tl",
    "Pb",
    "Bi",
    "Po",
    "At",
    "Rn",
    # 7
    "Fr",
    "Ra",
    "Ac",
    "Th",
    "Pa",
    "U",
    "Np",
    "Pu",
    "Am",
    "Cm",
    "Bk",
    "Cf",
    "Es",
    "Fm",
    "Md",
    "No",
    "Lr",
    "Rf",
    "Db",
    "Sg",
    "Bh",
    "Hs",
    "Mt",
    "Ds",
    "Rg",
    "Cn",
    "Nh",
    "Fl",
    "Mc",
    "Lv",
    "Ts",
    "Og",
]

atomic_numbers = {symbol: Z for Z, symbol in enumerate(chemical_symbols)}
# IUPAC version dated 28 November 2016
atomic_names = [
    "",
    "Hydrogen",
    "Helium",
    "Lithium",
    "Beryllium",
    "Boron",
    "Carbon",
    "Nitrogen",
    "Oxygen",
    "Fluorine",
    "Neon",
    "Sodium",
    "Magnesium",
    "Aluminium",
    "Silicon",
    "Phosphorus",
    "Sulfur",
    "Chlorine",
    "Argon",
    "Potassium",
    "Calcium",
    "Scandium",
    "Titanium",
    "Vanadium",
    "Chromium",
    "Manganese",
    "Iron",
    "Cobalt",
    "Nickel",
    "Copper",
    "Zinc",
    "Gallium",
    "Germanium",
    "Arsenic",
    "Selenium",
    "Bromine",
    "Krypton",
    "Rubidium",
    "Strontium",
    "Yttrium",
    "Zirconium",
    "Niobium",
    "Molybdenum",
    "Technetium",
    "Ruthenium",
    "Rhodium",
    "Palladium",
    "Silver",
    "Cadmium",
    "Indium",
    "Tin",
    "Antimony",
    "Tellurium",
    "Iodine",
    "Xenon",
    "Caesium",
    "Barium",
    "Lanthanum",
    "Cerium",
    "Praseodymium",
    "Neodymium",
    "Promethium",
    "Samarium",
    "Europium",
    "Gadolinium",
    "Terbium",
    "Dysprosium",
    "Holmium",
    "Erbium",
    "Thulium",
    "Ytterbium",
    "Lutetium",
    "Hafnium",
    "Tantalum",
    "Tungsten",
    "Rhenium",
    "Osmium",
    "Iridium",
    "Platinum",
    "Gold",
    "Mercury",
    "Thallium",
    "Lead",
    "Bismuth",
    "Polonium",
    "Astatine",
    "Radon",
    "Francium",
    "Radium",
    "Actinium",
    "Thorium",
    "Protactinium",
    "Uranium",
    "Neptunium",
    "Plutonium",
    "Americium",
    "Curium",
    "Berkelium",
    "Californium",
    "Einsteinium",
    "Fermium",
    "Mendelevium",
    "Nobelium",
    "Lawrencium",
    "Rutherfordium",
    "Dubnium",
    "Seaborgium",
    "Bohrium",
    "Hassium",
    "Meitnerium",
    "Darmastadtium",
    "Roentgenium",
    "Copernicium",
    "Nihonium",
    "Flerovium",
    "Moscovium",
    "Livermorium",
    "Tennessine",
    "Oganesson",
]


# Covalent radii from:
#
#  Covalent radii revisited,
#  Beatriz Cordero, Verónica Gómez, Ana E. Platero-Prats, Marc Revés,
#  Jorge Echeverría, Eduard Cremades, Flavia Barragán and Santiago Alvarez,
#  Dalton Trans., 2008, 2832-2838 DOI:10.1039/B801115J
missing = 0.2
covalent_radii = np.array(
    [
        missing,  # X
        0.31,  # H
        0.28,  # He
        1.28,  # Li
        0.96,  # Be
        0.84,  # B
        0.76,  # C
        0.71,  # N
        0.66,  # O
        0.57,  # F
        0.58,  # Ne
        1.66,  # Na
        1.41,  # Mg
        1.21,  # Al
        1.11,  # Si
        1.07,  # P
        1.05,  # S
        1.02,  # Cl
        1.06,  # Ar
        2.03,  # K
        1.76,  # Ca
        1.70,  # Sc
        1.60,  # Ti
        1.53,  # V
        1.39,  # Cr
        1.39,  # Mn
        1.32,  # Fe
        1.26,  # Co
        1.24,  # Ni
        1.32,  # Cu
        1.22,  # Zn
        1.22,  # Ga
        1.20,  # Ge
        1.19,  # As
        1.20,  # Se
        1.20,  # Br
        1.16,  # Kr
        2.20,  # Rb
        1.95,  # Sr
        1.90,  # Y
        1.75,  # Zr
        1.64,  # Nb
        1.54,  # Mo
        1.47,  # Tc
        1.46,  # Ru
        1.42,  # Rh
        1.39,  # Pd
        1.45,  # Ag
        1.44,  # Cd
        1.42,  # In
        1.39,  # Sn
        1.39,  # Sb
        1.38,  # Te
        1.39,  # I
        1.40,  # Xe
        2.44,  # Cs
        2.15,  # Ba
        2.07,  # La
        2.04,  # Ce
        2.03,  # Pr
        2.01,  # Nd
        1.99,  # Pm
        1.98,  # Sm
        1.98,  # Eu
        1.96,  # Gd
        1.94,  # Tb
        1.92,  # Dy
        1.92,  # Ho
        1.89,  # Er
        1.90,  # Tm
        1.87,  # Yb
        1.87,  # Lu
        1.75,  # Hf
        1.70,  # Ta
        1.62,  # W
        1.51,  # Re
        1.44,  # Os
        1.41,  # Ir
        1.36,  # Pt
        1.36,  # Au
        1.32,  # Hg
        1.45,  # Tl
        1.46,  # Pb
        1.48,  # Bi
        1.40,  # Po
        1.50,  # At
        1.50,  # Rn
        2.60,  # Fr
        2.21,  # Ra
        2.15,  # Ac
        2.06,  # Th
        2.00,  # Pa
        1.96,  # U
        1.90,  # Np
        1.87,  # Pu
        1.80,  # Am
        1.69,  # Cm
        missing,  # Bk
        missing,  # Cf
        missing,  # Es
        missing,  # Fm
        missing,  # Md
        missing,  # No
        missing,  # Lr
        missing,  # Rf
        missing,  # Db
        missing,  # Sg
        missing,  # Bh
        missing,  # Hs
        missing,  # Mt
        missing,  # Ds
        missing,  # Rg
        missing,  # Cn
        missing,  # Nh
        missing,  # Fl
        missing,  # Mc
        missing,  # Lv
        missing,  # Ts
        missing,  # Og
    ]
)


# This data is from Ashcroft and Mermin.
# Most constants are listed in periodic table, inside front cover.
# Reference states that have a non-trivial basis have a 'basis' key.
# If the basis is None, it means it has a basis but we have not tabulated it.
# For basis of RHL systems (represented here as basis_x) see page 127.
# For TET systems see page 127, too.
#
# (Values for magnetic moments are not from A&M and indeed are not taken
# from anywhere in particular.)
reference_states = [
    None,  # X
    {"symmetry": "diatom", "d": 0.74},  # H
    {"symmetry": "atom"},  # He
    {"symmetry": "bcc", "a": 3.49},  # Li
    {"symmetry": "hcp", "c/a": 1.567, "a": 2.29},  # Be
    {"symmetry": "tetragonal", "c/a": 0.576, "a": 8.73, "basis": None},  # B
    {"symmetry": "diamond", "a": 3.57},  # C
    {"symmetry": "diatom", "d": 1.10},  # N
    {"symmetry": "diatom", "d": 1.21},  # O
    {"symmetry": "diatom", "d": 1.42},  # F
    {"symmetry": "fcc", "a": 4.43},  # Ne
    {"symmetry": "bcc", "a": 4.23},  # Na
    {"symmetry": "hcp", "c/a": 1.624, "a": 3.21},  # Mg
    {"symmetry": "fcc", "a": 4.05},  # Al
    {"symmetry": "diamond", "a": 5.43},  # Si
    {"symmetry": "cubic", "a": 7.17, "basis": None},  # P
    {
        "symmetry": "orthorhombic",
        "c/a": 2.339,
        "a": 10.47,
        "b/a": 1.229,  # S
        "basis": None,
    },
    {
        "symmetry": "orthorhombic",
        "c/a": 1.324,
        "a": 6.24,
        "b/a": 0.718,  # Cl
        "basis": None,
    },
    {"symmetry": "fcc", "a": 5.26},  # Ar
    {"symmetry": "bcc", "a": 5.23},  # K
    {"symmetry": "fcc", "a": 5.58},  # Ca
    {"symmetry": "hcp", "c/a": 1.594, "a": 3.31},  # Sc
    {"symmetry": "hcp", "c/a": 1.588, "a": 2.95},  # Ti
    {"symmetry": "bcc", "a": 3.02},  # V
    {"symmetry": "bcc", "a": 2.88},  # Cr
    {"symmetry": "cubic", "a": 8.89, "basis": None},  # Mn
    {"symmetry": "bcc", "a": 2.87, "magmom_per_atom": 2.3},  # Fe
    {"symmetry": "hcp", "c/a": 1.622, "a": 2.51, "magmom_per_atom": 1.2},  # Co
    {"symmetry": "fcc", "a": 3.52, "magmom_per_atom": 0.6},  # Ni
    {"symmetry": "fcc", "a": 3.61},  # Cu
    {"symmetry": "hcp", "c/a": 1.856, "a": 2.66},  # Zn
    {
        "symmetry": "orthorhombic",
        "c/a": 1.695,
        "a": 4.51,
        "b/a": 1.001,  # Ga
        "basis": None,
    },
    {"symmetry": "diamond", "a": 5.66},  # Ge
    {
        "symmetry": "rhombohedral",
        "a": 4.13,
        "alpha": 54.10,  # As
        "basis_x": np.array(0.226) * (-1, 1),
    },
    {
        "symmetry": "hcp",
        "c/a": 1.136,
        "a": 4.36,  # Se
        "basis": None,
    },  # Needs 3-atom basis
    {
        "symmetry": "orthorhombic",
        "c/a": 1.307,
        "a": 6.67,
        "b/a": 0.672,  # Br
        "basis": None,
    },
    {"symmetry": "fcc", "a": 5.72},  # Kr
    {"symmetry": "bcc", "a": 5.59},  # Rb
    {"symmetry": "fcc", "a": 6.08},  # Sr
    {"symmetry": "hcp", "c/a": 1.571, "a": 3.65},  # Y
    {"symmetry": "hcp", "c/a": 1.593, "a": 3.23},  # Zr
    {"symmetry": "bcc", "a": 3.30},  # Nb
    {"symmetry": "bcc", "a": 3.15},  # Mo
    {"symmetry": "hcp", "c/a": 1.604, "a": 2.74},  # Tc
    {"symmetry": "hcp", "c/a": 1.584, "a": 2.70},  # Ru
    {"symmetry": "fcc", "a": 3.80},  # Rh
    {"symmetry": "fcc", "a": 3.89},  # Pd
    {"symmetry": "fcc", "a": 4.09},  # Ag
    {"symmetry": "hcp", "c/a": 1.886, "a": 2.98},  # Cd
    # For In, A&M give a face-centered cell; we need some sqrt2 conversions.
    {"symmetry": "bct", "c/a": 1.076 * 2**0.5, "a": 4.59 / 2**0.5},  # In
    {
        "symmetry": "bct",
        "c/a": 0.546,
        "a": 5.82,  # Sn
        "basis": [[0.0, 0.0, 0.0], [0.25, 0.75, 0.5]],
    },
    {
        "symmetry": "rhombohedral",
        "a": 4.51,
        "alpha": 57.60,  # Sb
        "basis_x": np.array(0.233) * (-1, 1),
    },
    {
        "symmetry": "hcp",
        "c/a": 1.330,
        "a": 4.45,  # Te
        "basis": None,
    },  # Te needs a 3-atom basis.
    {
        "symmetry": "orthorhombic",
        "c/a": 1.347,
        "a": 7.27,
        "b/a": 0.659,  # I
        "basis": None,
    },
    {"symmetry": "fcc", "a": 6.20},  # Xe
    {"symmetry": "bcc", "a": 6.05},  # Cs
    {"symmetry": "bcc", "a": 5.02},  # Ba
    {"symmetry": "hcp", "c/a": 1.619, "a": 3.75},  # La
    {"symmetry": "fcc", "a": 5.16},  # Ce
    {"symmetry": "hcp", "c/a": 1.614, "a": 3.67},  # Pr
    {"symmetry": "hcp", "c/a": 1.614, "a": 3.66},  # Nd
    None,  # Pm
    {
        "symmetry": "rhombohedral",
        "a": 9.00,
        "alpha": 23.13,
        "basis_x": np.array(0.222) * (0, -1, 1),
    },  # Sm
    {"symmetry": "bcc", "a": 4.61},  # Eu
    {"symmetry": "hcp", "c/a": 1.588, "a": 3.64},  # Gd
    {"symmetry": "hcp", "c/a": 1.581, "a": 3.60},  # Th
    {"symmetry": "hcp", "c/a": 1.573, "a": 3.59},  # Dy
    {"symmetry": "hcp", "c/a": 1.570, "a": 3.58},  # Ho
    {"symmetry": "hcp", "c/a": 1.570, "a": 3.56},  # Er
    {"symmetry": "hcp", "c/a": 1.570, "a": 3.54},  # Tm
    {"symmetry": "fcc", "a": 5.49},  # Yb
    {"symmetry": "hcp", "c/a": 1.585, "a": 3.51},  # Lu
    {"symmetry": "hcp", "c/a": 1.582, "a": 3.20},  # Hf
    {"symmetry": "bcc", "a": 3.31},  # Ta
    {"symmetry": "bcc", "a": 3.16},  # W
    {"symmetry": "hcp", "c/a": 1.615, "a": 2.76},  # Re
    {"symmetry": "hcp", "c/a": 1.579, "a": 2.74},  # Os
    {"symmetry": "fcc", "a": 3.84},  # Ir
    {"symmetry": "fcc", "a": 3.92},  # Pt
    {"symmetry": "fcc", "a": 4.08},  # Au
    {
        "symmetry": "rhombohedral",
        "a": 2.99,
        "alpha": 70.45,  # Hg
        "basis_x": np.zeros(1),
    },
    {"symmetry": "hcp", "c/a": 1.599, "a": 3.46},  # Tl
    {"symmetry": "fcc", "a": 4.95},  # Pb
    {
        "symmetry": "rhombohedral",
        "a": 4.75,
        "alpha": 57.14,
        "basis_x": np.array(0.237) * (-1, 1),
    },  # Bi
    {"symmetry": "sc", "a": 3.35},  # Po
    None,  # At
    None,  # Rn
    None,  # Fr
    None,  # Ra
    {"symmetry": "fcc", "a": 5.31},  # Ac
    {"symmetry": "fcc", "a": 5.08},  # Th
    {"symmetry": "tetragonal", "c/a": 0.825, "a": 3.92},  # Pa
    {"symmetry": "orthorhombic", "c/a": 2.056, "a": 2.85, "b/a": 1.736},  # U
    {"symmetry": "orthorhombic", "c/a": 1.411, "a": 4.72, "b/a": 1.035},  # Np
    {"symmetry": "monoclinic"},  # Pu
    None,  # Am
    None,  # Cm
    None,  # Bk
    None,  # Cf
    None,  # Es
    None,  # Fm
    None,  # Md
    None,  # No
    None,  # Lr
    None,  # Rf
    None,  # Db
    None,  # Sg
    None,  # Bh
    None,  # Hs
    None,  # Mt
    None,  # Ds
    None,  # Rg
    None,  # Cn
    None,  # Nh
    None,  # Fl
    None,  # Mc
    None,  # Lv
    None,  # Ts
    None,  # Og
]


"""
the below is gotten by:

from ase.build import bulk
from ase.data import chemical_symbols
from scipy.constants import N_A
import json

molar_vol = {}

for Z, sym in enumerate(chemical_symbols):
    if sym == "X":         # Z = 0 placeholder
        continue
    try:
        atoms = bulk(sym)                           
        V_atom = atoms.get_volume() / len(atoms)    # A3/atom
        molar_vol[sym] = N_A * V_atom #A3/mol
    except Exception:
        # No crystal data; skip element
        continue

print(json.dumps(molar_vol, indent=2, sort_keys=True))
"""
# all in A3/mol
molar_volumes = {
    "Ac": 2.2541067229273025e25,
    "Ag": 1.030055997364215e25,
    "Al": 1.0001289086359875e25,
    "Ar": 2.191029089241595e25,
    "As": 1.2932573882130303e25,
    "Au": 1.022519032262093e25,
    "Ba": 3.809184935808428e25,
    "Be": 4.90712342862515e24,
    "Bi": 2.1314764341933746e25,
    "C": 3.4250393365810323e24,
    "Ca": 2.6157335806573126e25,
    "Cd": 1.301493546681067e25,
    "Ce": 2.0684261321509823e25,
    "Co": 6.688420244604578e24,
    "Cr": 7.192806382043136e24,
    "Cs": 6.667868566828973e25,
    "Cu": 7.082922939005239e24,
    "Dy": 1.8978575276039413e25,
    "Er": 1.8471458338718941e25,
    "Eu": 2.950011322730988e25,
    "Fe": 7.118141170937316e24,
    "Gd": 1.9971293192108506e25,
    "Ge": 1.3649294646572212e25,
    "Hf": 1.3517860256843859e25,
    "Hg": 1.3838529205475248e25,
    "Ho": 1.8784527297819117e25,
    "In": 1.5665390995752235e25,
    "Ir": 8.524807563902976e24,
    "K": 4.3075068159484365e25,
    "Kr": 2.8175977864603706e25,
    "La": 2.2263435942462644e25,
    "Li": 1.2799623279067861e25,
    "Lu": 1.787319338431579e25,
    "Mg": 1.4007242922833397e25,
    "Mo": 9.411363941348245e24,
    "Na": 2.2789878448573746e25,
    "Nb": 1.0820883624605997e25,
    "Nd": 2.0634707876165025e25,
    "Ne": 1.3088868054752332e25,
    "Ni": 6.566272492797953e24,
    "Np": 9.247924208723399e25,
    "Os": 8.470037963558634e24,
    "Pb": 1.826024111652263e25,
    "Pd": 8.86216261990501e24,
    "Po": 2.2640464017498497e25,
    "Pr": 2.08043073696602e25,
    "Pt": 9.068785129897469e24,
    "Rb": 5.259643764277437e25,
    "Re": 8.854240441183479e24,
    "Rh": 8.261172694567998e24,
    "Ru": 8.130139700324425e24,
    "Sb": 1.8453942686400805e25,
    "Sc": 1.50738377656577e25,
    "Si": 1.2052035553165811e25,
    "Sm": 1.982108956096168e25,
    "Sn": 1.6205129565423706e25,
    "Sr": 3.383776335695053e25,
    "Ta": 1.0919553690995255e25,
    "Tb": 1.9234943308960588e25,
    "Tc": 8.60414242783284e24,
    "Th": 1.9737041210225723e25,
    "Ti": 1.0630852303207755e25,
    "Tl": 1.727143007051994e25,
    "Tm": 1.8161887320220726e25,
    "U": 4.975747272036323e25,
    "V": 8.293574220713104e24,
    "W": 9.501280826142848e24,
    "Xe": 3.5881119076232015e25,
    "Y": 1.9920778970913784e25,
    "Yb": 2.4911962667885323e25,
    "Zn": 9.109077985059764e24,
    "Zr": 1.3998286607294745e25,
}

"""
The following was gotten thus:

from itertools import product
from ase.build import surface
from ase.data import chemical_symbols
import numpy as np
from ase.build import bulk

miller_planes = [(h, k, l)
                 for h, k, l in product(range(4), repeat=3)
                 if (h, k, l) != (0, 0, 0)]

tol = 0.05  # A; just in case some atom.z != 0 exactly
interplanar_dist = {}  # {element: {(h,k,l): d}}

for sym in chemical_symbols[1:]:                
    facet_d = {}
    for plane in miller_planes:
        try:
            atoms = surface(lattice=bulk(sym,cubic=False), indices=plane, layers=2)
            z = atoms.positions[:, 2]
            base = z.min()
            above = z[z > base + tol]           
            if above.size:
                facet_d[plane] = above.min() - base
        except Exception:                        # no bulk data / bad Miller index
            continue
    if facet_d:                                 
        interplanar_dist[sym] = facet_d

for el in sorted(interplanar_dist):
    items = [f'"{k}": {round(v, 3)}' for k, v in interplanar_dist[el].items()]
    print(f'  "{el}": {{', end='')

    for i, text in enumerate(items):
        end_char = ', ' if i < len(items) - 1 else ''
        newline  = '\n' + ' ' * (len(el) + 6) if (i + 1) % 6 == 0 and i < len(items) - 1 else ''
        print(text + end_char + newline, end='')

    print('},\n')

print("}")
"""

interplanar_dist = {
    "Ac": {
        "(0, 0, 1)": 2.655,
        "(0, 0, 2)": 2.655,
        "(0, 0, 3)": 2.655,
        "(0, 1, 0)": 2.655,
        "(0, 1, 1)": 1.877,
        "(0, 1, 2)": 1.187,
        "(0, 1, 3)": 0.84,
        "(0, 2, 0)": 2.655,
        "(0, 2, 1)": 1.187,
        "(0, 2, 2)": 1.877,
        "(0, 2, 3)": 0.736,
        "(0, 3, 0)": 2.655,
        "(0, 3, 1)": 0.84,
        "(0, 3, 2)": 0.736,
        "(0, 3, 3)": 1.877,
        "(1, 0, 0)": 2.655,
        "(1, 0, 1)": 1.877,
        "(1, 0, 2)": 1.187,
        "(1, 0, 3)": 0.84,
        "(1, 1, 0)": 1.877,
        "(1, 1, 1)": 3.066,
        "(1, 1, 2)": 1.084,
        "(1, 1, 3)": 1.601,
        "(1, 2, 0)": 1.187,
        "(1, 2, 1)": 1.084,
        "(1, 2, 2)": 0.885,
        "(1, 2, 3)": 0.71,
        "(1, 3, 0)": 0.84,
        "(1, 3, 1)": 1.601,
        "(1, 3, 2)": 0.71,
        "(1, 3, 3)": 1.218,
        "(2, 0, 0)": 2.655,
        "(2, 0, 1)": 1.187,
        "(2, 0, 2)": 1.877,
        "(2, 0, 3)": 0.736,
        "(2, 1, 0)": 1.187,
        "(2, 1, 1)": 1.084,
        "(2, 1, 2)": 0.885,
        "(2, 1, 3)": 0.71,
        "(2, 2, 0)": 1.877,
        "(2, 2, 1)": 0.885,
        "(2, 2, 2)": 3.066,
        "(2, 2, 3)": 0.644,
        "(2, 3, 0)": 0.736,
        "(2, 3, 1)": 0.71,
        "(2, 3, 2)": 0.644,
        "(2, 3, 3)": 0.566,
        "(3, 0, 0)": 2.655,
        "(3, 0, 1)": 0.84,
        "(3, 0, 2)": 0.736,
        "(3, 0, 3)": 1.877,
        "(3, 1, 0)": 0.84,
        "(3, 1, 1)": 1.601,
        "(3, 1, 2)": 0.71,
        "(3, 1, 3)": 1.218,
        "(3, 2, 0)": 0.736,
        "(3, 2, 1)": 0.71,
        "(3, 2, 2)": 0.644,
        "(3, 2, 3)": 0.566,
        "(3, 3, 0)": 1.877,
        "(3, 3, 1)": 1.218,
        "(3, 3, 2)": 0.566,
        "(3, 3, 3)": 3.066,
    },
    "Ag": {
        "(0, 0, 1)": 2.045,
        "(0, 0, 2)": 2.045,
        "(0, 0, 3)": 2.045,
        "(0, 1, 0)": 2.045,
        "(0, 1, 1)": 1.446,
        "(0, 1, 2)": 0.915,
        "(0, 1, 3)": 0.647,
        "(0, 2, 0)": 2.045,
        "(0, 2, 1)": 0.915,
        "(0, 2, 2)": 1.446,
        "(0, 2, 3)": 0.567,
        "(0, 3, 0)": 2.045,
        "(0, 3, 1)": 0.647,
        "(0, 3, 2)": 0.567,
        "(0, 3, 3)": 1.446,
        "(1, 0, 0)": 2.045,
        "(1, 0, 1)": 1.446,
        "(1, 0, 2)": 0.915,
        "(1, 0, 3)": 0.647,
        "(1, 1, 0)": 1.446,
        "(1, 1, 1)": 2.361,
        "(1, 1, 2)": 0.835,
        "(1, 1, 3)": 1.233,
        "(1, 2, 0)": 0.915,
        "(1, 2, 1)": 0.835,
        "(1, 2, 2)": 0.682,
        "(1, 2, 3)": 0.547,
        "(1, 3, 0)": 0.647,
        "(1, 3, 1)": 1.233,
        "(1, 3, 2)": 0.547,
        "(1, 3, 3)": 0.938,
        "(2, 0, 0)": 2.045,
        "(2, 0, 1)": 0.915,
        "(2, 0, 2)": 1.446,
        "(2, 0, 3)": 0.567,
        "(2, 1, 0)": 0.915,
        "(2, 1, 1)": 0.835,
        "(2, 1, 2)": 0.682,
        "(2, 1, 3)": 0.547,
        "(2, 2, 0)": 1.446,
        "(2, 2, 1)": 0.682,
        "(2, 2, 2)": 2.361,
        "(2, 2, 3)": 0.496,
        "(2, 3, 0)": 0.567,
        "(2, 3, 1)": 0.547,
        "(2, 3, 2)": 0.496,
        "(2, 3, 3)": 0.436,
        "(3, 0, 0)": 2.045,
        "(3, 0, 1)": 0.647,
        "(3, 0, 2)": 0.567,
        "(3, 0, 3)": 1.446,
        "(3, 1, 0)": 0.647,
        "(3, 1, 1)": 1.233,
        "(3, 1, 2)": 0.547,
        "(3, 1, 3)": 0.938,
        "(3, 2, 0)": 0.567,
        "(3, 2, 1)": 0.547,
        "(3, 2, 2)": 0.496,
        "(3, 2, 3)": 0.436,
        "(3, 3, 0)": 1.446,
        "(3, 3, 1)": 0.938,
        "(3, 3, 2)": 0.436,
        "(3, 3, 3)": 2.361,
    },
    "Al": {
        "(0, 0, 1)": 2.025,
        "(0, 0, 2)": 2.025,
        "(0, 0, 3)": 2.025,
        "(0, 1, 0)": 2.025,
        "(0, 1, 1)": 1.432,
        "(0, 1, 2)": 0.906,
        "(0, 1, 3)": 0.64,
        "(0, 2, 0)": 2.025,
        "(0, 2, 1)": 0.906,
        "(0, 2, 2)": 1.432,
        "(0, 2, 3)": 0.562,
        "(0, 3, 0)": 2.025,
        "(0, 3, 1)": 0.64,
        "(0, 3, 2)": 0.562,
        "(0, 3, 3)": 1.432,
        "(1, 0, 0)": 2.025,
        "(1, 0, 1)": 1.432,
        "(1, 0, 2)": 0.906,
        "(1, 0, 3)": 0.64,
        "(1, 1, 0)": 1.432,
        "(1, 1, 1)": 2.338,
        "(1, 1, 2)": 0.827,
        "(1, 1, 3)": 1.221,
        "(1, 2, 0)": 0.906,
        "(1, 2, 1)": 0.827,
        "(1, 2, 2)": 0.675,
        "(1, 2, 3)": 0.541,
        "(1, 3, 0)": 0.64,
        "(1, 3, 1)": 1.221,
        "(1, 3, 2)": 0.541,
        "(1, 3, 3)": 0.929,
        "(2, 0, 0)": 2.025,
        "(2, 0, 1)": 0.906,
        "(2, 0, 2)": 1.432,
        "(2, 0, 3)": 0.562,
        "(2, 1, 0)": 0.906,
        "(2, 1, 1)": 0.827,
        "(2, 1, 2)": 0.675,
        "(2, 1, 3)": 0.541,
        "(2, 2, 0)": 1.432,
        "(2, 2, 1)": 0.675,
        "(2, 2, 2)": 2.338,
        "(2, 2, 3)": 0.491,
        "(2, 3, 0)": 0.562,
        "(2, 3, 1)": 0.541,
        "(2, 3, 2)": 0.491,
        "(2, 3, 3)": 0.432,
        "(3, 0, 0)": 2.025,
        "(3, 0, 1)": 0.64,
        "(3, 0, 2)": 0.562,
        "(3, 0, 3)": 1.432,
        "(3, 1, 0)": 0.64,
        "(3, 1, 1)": 1.221,
        "(3, 1, 2)": 0.541,
        "(3, 1, 3)": 0.929,
        "(3, 2, 0)": 0.562,
        "(3, 2, 1)": 0.541,
        "(3, 2, 2)": 0.491,
        "(3, 2, 3)": 0.432,
        "(3, 3, 0)": 1.432,
        "(3, 3, 1)": 0.929,
        "(3, 3, 2)": 0.432,
        "(3, 3, 3)": 2.338,
    },
    "Ar": {
        "(0, 0, 1)": 2.63,
        "(0, 0, 2)": 2.63,
        "(0, 0, 3)": 2.63,
        "(0, 1, 0)": 2.63,
        "(0, 1, 1)": 1.86,
        "(0, 1, 2)": 1.176,
        "(0, 1, 3)": 0.832,
        "(0, 2, 0)": 2.63,
        "(0, 2, 1)": 1.176,
        "(0, 2, 2)": 1.86,
        "(0, 2, 3)": 0.729,
        "(0, 3, 0)": 2.63,
        "(0, 3, 1)": 0.832,
        "(0, 3, 2)": 0.729,
        "(0, 3, 3)": 1.86,
        "(1, 0, 0)": 2.63,
        "(1, 0, 1)": 1.86,
        "(1, 0, 2)": 1.176,
        "(1, 0, 3)": 0.832,
        "(1, 1, 0)": 1.86,
        "(1, 1, 1)": 3.037,
        "(1, 1, 2)": 1.074,
        "(1, 1, 3)": 1.586,
        "(1, 2, 0)": 1.176,
        "(1, 2, 1)": 1.074,
        "(1, 2, 2)": 0.877,
        "(1, 2, 3)": 0.703,
        "(1, 3, 0)": 0.832,
        "(1, 3, 1)": 1.586,
        "(1, 3, 2)": 0.703,
        "(1, 3, 3)": 1.207,
        "(2, 0, 0)": 2.63,
        "(2, 0, 1)": 1.176,
        "(2, 0, 2)": 1.86,
        "(2, 0, 3)": 0.729,
        "(2, 1, 0)": 1.176,
        "(2, 1, 1)": 1.074,
        "(2, 1, 2)": 0.877,
        "(2, 1, 3)": 0.703,
        "(2, 2, 0)": 1.86,
        "(2, 2, 1)": 0.877,
        "(2, 2, 2)": 3.037,
        "(2, 2, 3)": 0.638,
        "(2, 3, 0)": 0.729,
        "(2, 3, 1)": 0.703,
        "(2, 3, 2)": 0.638,
        "(2, 3, 3)": 0.561,
        "(3, 0, 0)": 2.63,
        "(3, 0, 1)": 0.832,
        "(3, 0, 2)": 0.729,
        "(3, 0, 3)": 1.86,
        "(3, 1, 0)": 0.832,
        "(3, 1, 1)": 1.586,
        "(3, 1, 2)": 0.703,
        "(3, 1, 3)": 1.207,
        "(3, 2, 0)": 0.729,
        "(3, 2, 1)": 0.703,
        "(3, 2, 2)": 0.638,
        "(3, 2, 3)": 0.561,
        "(3, 3, 0)": 1.86,
        "(3, 3, 1)": 1.207,
        "(3, 3, 2)": 0.561,
        "(3, 3, 3)": 3.037,
    },
    "As": {
        "(0, 0, 1)": 1.703,
        "(0, 0, 2)": 1.703,
        "(0, 0, 3)": 1.703,
        "(0, 1, 0)": 1.703,
        "(0, 1, 1)": 0.266,
        "(0, 1, 2)": 0.59,
        "(0, 1, 3)": 0.9,
        "(0, 2, 0)": 1.703,
        "(0, 2, 1)": 0.59,
        "(0, 2, 2)": 0.266,
        "(0, 2, 3)": 0.786,
        "(0, 3, 0)": 1.703,
        "(0, 3, 1)": 0.9,
        "(0, 3, 2)": 0.786,
        "(0, 3, 3)": 0.266,
        "(1, 0, 0)": 1.703,
        "(1, 0, 1)": 0.266,
        "(1, 0, 2)": 0.59,
        "(1, 0, 3)": 0.9,
        "(1, 1, 0)": 0.266,
        "(1, 1, 1)": 1.251,
        "(1, 1, 2)": 1.655,
        "(1, 1, 3)": 0.953,
        "(1, 2, 0)": 0.59,
        "(1, 2, 1)": 1.655,
        "(1, 2, 2)": 1.309,
        "(1, 2, 3)": 0.37,
        "(1, 3, 0)": 0.9,
        "(1, 3, 1)": 0.953,
        "(1, 3, 2)": 0.37,
        "(1, 3, 3)": 0.181,
        "(2, 0, 0)": 1.703,
        "(2, 0, 1)": 0.59,
        "(2, 0, 2)": 0.266,
        "(2, 0, 3)": 0.786,
        "(2, 1, 0)": 0.59,
        "(2, 1, 1)": 1.655,
        "(2, 1, 2)": 1.309,
        "(2, 1, 3)": 0.37,
        "(2, 2, 0)": 0.266,
        "(2, 2, 1)": 1.309,
        "(2, 2, 2)": 1.251,
        "(2, 2, 3)": 0.224,
        "(2, 3, 0)": 0.786,
        "(2, 3, 1)": 0.37,
        "(2, 3, 2)": 0.224,
        "(2, 3, 3)": 0.752,
        "(3, 0, 0)": 1.703,
        "(3, 0, 1)": 0.9,
        "(3, 0, 2)": 0.786,
        "(3, 0, 3)": 0.266,
        "(3, 1, 0)": 0.9,
        "(3, 1, 1)": 0.953,
        "(3, 1, 2)": 0.37,
        "(3, 1, 3)": 0.181,
        "(3, 2, 0)": 0.786,
        "(3, 2, 1)": 0.37,
        "(3, 2, 2)": 0.224,
        "(3, 2, 3)": 0.752,
        "(3, 3, 0)": 0.266,
        "(3, 3, 1)": 0.181,
        "(3, 3, 2)": 0.752,
        "(3, 3, 3)": 1.251,
    },
    "Au": {
        "(0, 0, 1)": 2.04,
        "(0, 0, 2)": 2.04,
        "(0, 0, 3)": 2.04,
        "(0, 1, 0)": 2.04,
        "(0, 1, 1)": 1.442,
        "(0, 1, 2)": 0.912,
        "(0, 1, 3)": 0.645,
        "(0, 2, 0)": 2.04,
        "(0, 2, 1)": 0.912,
        "(0, 2, 2)": 1.442,
        "(0, 2, 3)": 0.566,
        "(0, 3, 0)": 2.04,
        "(0, 3, 1)": 0.645,
        "(0, 3, 2)": 0.566,
        "(0, 3, 3)": 1.442,
        "(1, 0, 0)": 2.04,
        "(1, 0, 1)": 1.442,
        "(1, 0, 2)": 0.912,
        "(1, 0, 3)": 0.645,
        "(1, 1, 0)": 1.442,
        "(1, 1, 1)": 2.356,
        "(1, 1, 2)": 0.833,
        "(1, 1, 3)": 1.23,
        "(1, 2, 0)": 0.912,
        "(1, 2, 1)": 0.833,
        "(1, 2, 2)": 0.68,
        "(1, 2, 3)": 0.545,
        "(1, 3, 0)": 0.645,
        "(1, 3, 1)": 1.23,
        "(1, 3, 2)": 0.545,
        "(1, 3, 3)": 0.936,
        "(2, 0, 0)": 2.04,
        "(2, 0, 1)": 0.912,
        "(2, 0, 2)": 1.442,
        "(2, 0, 3)": 0.566,
        "(2, 1, 0)": 0.912,
        "(2, 1, 1)": 0.833,
        "(2, 1, 2)": 0.68,
        "(2, 1, 3)": 0.545,
        "(2, 2, 0)": 1.442,
        "(2, 2, 1)": 0.68,
        "(2, 2, 2)": 2.356,
        "(2, 2, 3)": 0.495,
        "(2, 3, 0)": 0.566,
        "(2, 3, 1)": 0.545,
        "(2, 3, 2)": 0.495,
        "(2, 3, 3)": 0.435,
        "(3, 0, 0)": 2.04,
        "(3, 0, 1)": 0.645,
        "(3, 0, 2)": 0.566,
        "(3, 0, 3)": 1.442,
        "(3, 1, 0)": 0.645,
        "(3, 1, 1)": 1.23,
        "(3, 1, 2)": 0.545,
        "(3, 1, 3)": 0.936,
        "(3, 2, 0)": 0.566,
        "(3, 2, 1)": 0.545,
        "(3, 2, 2)": 0.495,
        "(3, 2, 3)": 0.435,
        "(3, 3, 0)": 1.442,
        "(3, 3, 1)": 0.936,
        "(3, 3, 2)": 0.435,
        "(3, 3, 3)": 2.356,
    },
    "Ba": {
        "(0, 0, 1)": 2.51,
        "(0, 0, 2)": 2.51,
        "(0, 0, 3)": 2.51,
        "(0, 1, 0)": 2.51,
        "(0, 1, 1)": 3.55,
        "(0, 1, 2)": 1.123,
        "(0, 1, 3)": 1.587,
        "(0, 2, 0)": 2.51,
        "(0, 2, 1)": 1.123,
        "(0, 2, 2)": 3.55,
        "(0, 2, 3)": 0.696,
        "(0, 3, 0)": 2.51,
        "(0, 3, 1)": 1.587,
        "(0, 3, 2)": 0.696,
        "(0, 3, 3)": 3.55,
        "(1, 0, 0)": 2.51,
        "(1, 0, 1)": 3.55,
        "(1, 0, 2)": 1.123,
        "(1, 0, 3)": 1.587,
        "(1, 1, 0)": 3.55,
        "(1, 1, 1)": 1.449,
        "(1, 1, 2)": 2.049,
        "(1, 1, 3)": 0.757,
        "(1, 2, 0)": 1.123,
        "(1, 2, 1)": 2.049,
        "(1, 2, 2)": 0.837,
        "(1, 2, 3)": 1.342,
        "(1, 3, 0)": 1.587,
        "(1, 3, 1)": 0.757,
        "(1, 3, 2)": 1.342,
        "(1, 3, 3)": 0.576,
        "(2, 0, 0)": 2.51,
        "(2, 0, 1)": 1.123,
        "(2, 0, 2)": 3.55,
        "(2, 0, 3)": 0.696,
        "(2, 1, 0)": 1.123,
        "(2, 1, 1)": 2.049,
        "(2, 1, 2)": 0.837,
        "(2, 1, 3)": 1.342,
        "(2, 2, 0)": 3.55,
        "(2, 2, 1)": 0.837,
        "(2, 2, 2)": 1.449,
        "(2, 2, 3)": 0.609,
        "(2, 3, 0)": 0.696,
        "(2, 3, 1)": 1.342,
        "(2, 3, 2)": 0.609,
        "(2, 3, 3)": 1.07,
        "(3, 0, 0)": 2.51,
        "(3, 0, 1)": 1.587,
        "(3, 0, 2)": 0.696,
        "(3, 0, 3)": 3.55,
        "(3, 1, 0)": 1.587,
        "(3, 1, 1)": 0.757,
        "(3, 1, 2)": 1.342,
        "(3, 1, 3)": 0.576,
        "(3, 2, 0)": 0.696,
        "(3, 2, 1)": 1.342,
        "(3, 2, 2)": 0.609,
        "(3, 2, 3)": 1.07,
        "(3, 3, 0)": 3.55,
        "(3, 3, 1)": 0.576,
        "(3, 3, 2)": 1.07,
        "(3, 3, 3)": 1.449,
    },
    "Be": {
        "(0, 0, 1)": 1.794,
        "(0, 0, 2)": 1.794,
        "(0, 0, 3)": 1.794,
        "(0, 1, 0)": 1.322,
        "(0, 1, 1)": 0.289,
        "(0, 1, 2)": 0.887,
        "(0, 1, 3)": 0.171,
        "(0, 2, 0)": 1.322,
        "(0, 2, 1)": 0.796,
        "(0, 2, 2)": 0.289,
        "(0, 2, 3)": 0.636,
        "(0, 3, 0)": 1.322,
        "(0, 3, 1)": 0.325,
        "(0, 3, 2)": 0.62,
        "(0, 3, 3)": 0.289,
        "(1, 0, 0)": 0.661,
        "(1, 0, 1)": 1.446,
        "(1, 0, 2)": 0.444,
        "(1, 0, 3)": 0.854,
        "(1, 1, 0)": 1.145,
        "(1, 1, 1)": 0.545,
        "(1, 1, 2)": 0.965,
        "(1, 1, 3)": 0.414,
        "(1, 2, 0)": 0.5,
        "(1, 2, 1)": 0.122,
        "(1, 2, 2)": 0.461,
        "(1, 2, 3)": 0.106,
        "(1, 3, 0)": 0.183,
        "(1, 3, 1)": 0.453,
        "(1, 3, 2)": 0.175,
        "(1, 3, 3)": 0.416,
        "(2, 0, 0)": 0.661,
        "(2, 0, 1)": 0.159,
        "(2, 0, 2)": 1.446,
        "(2, 0, 3)": 0.127,
        "(2, 1, 0)": 0.25,
        "(2, 1, 1)": 0.611,
        "(2, 1, 2)": 0.231,
        "(2, 1, 3)": 0.529,
        "(2, 2, 0)": 1.145,
        "(2, 2, 1)": 0.283,
        "(2, 2, 2)": 0.545,
        "(2, 2, 3)": 0.258,
        "(2, 3, 0)": 0.303,
        "(2, 3, 1)": 0.451,
        "(2, 3, 2)": 0.294,
        "(2, 3, 3)": 0.425,
        "(3, 0, 0)": 0.661,
        "(3, 0, 1)": 0.325,
        "(3, 0, 2)": 0.62,
        "(3, 0, 3)": 1.446,
        "(3, 1, 0)": 0.367,
        "(3, 1, 1)": 0.544,
        "(3, 1, 2)": 0.351,
        "(3, 1, 3)": 0.5,
        "(3, 2, 0)": 0.152,
        "(3, 2, 1)": 0.376,
        "(3, 2, 2)": 0.147,
        "(3, 2, 3)": 0.354,
        "(3, 3, 0)": 1.145,
        "(3, 3, 1)": 0.19,
        "(3, 3, 2)": 0.373,
        "(3, 3, 3)": 0.545,
    },
    "Bi": {
        "(0, 0, 1)": 1.965,
        "(0, 0, 2)": 1.965,
        "(0, 0, 3)": 1.965,
        "(0, 1, 0)": 1.965,
        "(0, 1, 1)": 0.171,
        "(0, 1, 2)": 0.832,
        "(0, 1, 3)": 1.191,
        "(0, 2, 0)": 1.965,
        "(0, 2, 1)": 0.832,
        "(0, 2, 2)": 0.171,
        "(0, 2, 3)": 0.794,
        "(0, 3, 0)": 1.965,
        "(0, 3, 1)": 1.191,
        "(0, 3, 2)": 0.794,
        "(0, 3, 3)": 0.171,
        "(1, 0, 0)": 1.965,
        "(1, 0, 1)": 0.171,
        "(1, 0, 2)": 0.832,
        "(1, 0, 3)": 1.191,
        "(1, 1, 0)": 0.171,
        "(1, 1, 1)": 1.671,
        "(1, 1, 2)": 2.124,
        "(1, 1, 3)": 0.955,
        "(1, 2, 0)": 0.832,
        "(1, 2, 1)": 2.124,
        "(1, 2, 2)": 1.281,
        "(1, 2, 3)": 0.233,
        "(1, 3, 0)": 1.191,
        "(1, 3, 1)": 0.955,
        "(1, 3, 2)": 0.233,
        "(1, 3, 3)": 0.409,
        "(2, 0, 0)": 1.965,
        "(2, 0, 1)": 0.832,
        "(2, 0, 2)": 0.171,
        "(2, 0, 3)": 0.794,
        "(2, 1, 0)": 0.832,
        "(2, 1, 1)": 2.124,
        "(2, 1, 2)": 1.281,
        "(2, 1, 3)": 0.233,
        "(2, 2, 0)": 0.171,
        "(2, 2, 1)": 1.281,
        "(2, 2, 2)": 1.671,
        "(2, 2, 3)": 0.496,
        "(2, 3, 0)": 0.794,
        "(2, 3, 1)": 0.233,
        "(2, 3, 2)": 0.496,
        "(2, 3, 3)": 1.1,
        "(3, 0, 0)": 1.965,
        "(3, 0, 1)": 1.191,
        "(3, 0, 2)": 0.794,
        "(3, 0, 3)": 0.171,
        "(3, 1, 0)": 1.191,
        "(3, 1, 1)": 0.955,
        "(3, 1, 2)": 0.233,
        "(3, 1, 3)": 0.409,
        "(3, 2, 0)": 0.794,
        "(3, 2, 1)": 0.233,
        "(3, 2, 2)": 0.496,
        "(3, 2, 3)": 1.1,
        "(3, 3, 0)": 0.171,
        "(3, 3, 1)": 0.409,
        "(3, 3, 2)": 1.1,
        "(3, 3, 3)": 1.671,
    },
    "C": {
        "(0, 0, 1)": 0.892,
        "(0, 0, 2)": 0.892,
        "(0, 0, 3)": 0.892,
        "(0, 1, 0)": 0.892,
        "(0, 1, 1)": 1.262,
        "(0, 1, 2)": 0.399,
        "(0, 1, 3)": 0.564,
        "(0, 2, 0)": 0.892,
        "(0, 2, 1)": 0.399,
        "(0, 2, 2)": 1.262,
        "(0, 2, 3)": 0.248,
        "(0, 3, 0)": 0.892,
        "(0, 3, 1)": 0.564,
        "(0, 3, 2)": 0.248,
        "(0, 3, 3)": 1.262,
        "(1, 0, 0)": 0.892,
        "(1, 0, 1)": 1.262,
        "(1, 0, 2)": 0.399,
        "(1, 0, 3)": 0.564,
        "(1, 1, 0)": 1.262,
        "(1, 1, 1)": 1.546,
        "(1, 1, 2)": 0.729,
        "(1, 1, 3)": 0.269,
        "(1, 2, 0)": 0.399,
        "(1, 2, 1)": 0.729,
        "(1, 2, 2)": 0.297,
        "(1, 2, 3)": 0.477,
        "(1, 3, 0)": 0.564,
        "(1, 3, 1)": 0.269,
        "(1, 3, 2)": 0.477,
        "(1, 3, 3)": 0.614,
        "(2, 0, 0)": 0.892,
        "(2, 0, 1)": 0.399,
        "(2, 0, 2)": 1.262,
        "(2, 0, 3)": 0.248,
        "(2, 1, 0)": 0.399,
        "(2, 1, 1)": 0.729,
        "(2, 1, 2)": 0.297,
        "(2, 1, 3)": 0.477,
        "(2, 2, 0)": 1.262,
        "(2, 2, 1)": 0.297,
        "(2, 2, 2)": 1.546,
        "(2, 2, 3)": 0.216,
        "(2, 3, 0)": 0.248,
        "(2, 3, 1)": 0.477,
        "(2, 3, 2)": 0.216,
        "(2, 3, 3)": 0.381,
        "(3, 0, 0)": 0.892,
        "(3, 0, 1)": 0.564,
        "(3, 0, 2)": 0.248,
        "(3, 0, 3)": 1.262,
        "(3, 1, 0)": 0.564,
        "(3, 1, 1)": 0.269,
        "(3, 1, 2)": 0.477,
        "(3, 1, 3)": 0.614,
        "(3, 2, 0)": 0.248,
        "(3, 2, 1)": 0.477,
        "(3, 2, 2)": 0.216,
        "(3, 2, 3)": 0.381,
        "(3, 3, 0)": 1.262,
        "(3, 3, 1)": 0.614,
        "(3, 3, 2)": 0.381,
        "(3, 3, 3)": 1.546,
    },
    "Ca": {
        "(0, 0, 1)": 2.79,
        "(0, 0, 2)": 2.79,
        "(0, 0, 3)": 2.79,
        "(0, 1, 0)": 2.79,
        "(0, 1, 1)": 1.973,
        "(0, 1, 2)": 1.248,
        "(0, 1, 3)": 0.882,
        "(0, 2, 0)": 2.79,
        "(0, 2, 1)": 1.248,
        "(0, 2, 2)": 1.973,
        "(0, 2, 3)": 0.774,
        "(0, 3, 0)": 2.79,
        "(0, 3, 1)": 0.882,
        "(0, 3, 2)": 0.774,
        "(0, 3, 3)": 1.973,
        "(1, 0, 0)": 2.79,
        "(1, 0, 1)": 1.973,
        "(1, 0, 2)": 1.248,
        "(1, 0, 3)": 0.882,
        "(1, 1, 0)": 1.973,
        "(1, 1, 1)": 3.222,
        "(1, 1, 2)": 1.139,
        "(1, 1, 3)": 1.682,
        "(1, 2, 0)": 1.248,
        "(1, 2, 1)": 1.139,
        "(1, 2, 2)": 0.93,
        "(1, 2, 3)": 0.746,
        "(1, 3, 0)": 0.882,
        "(1, 3, 1)": 1.682,
        "(1, 3, 2)": 0.746,
        "(1, 3, 3)": 1.28,
        "(2, 0, 0)": 2.79,
        "(2, 0, 1)": 1.248,
        "(2, 0, 2)": 1.973,
        "(2, 0, 3)": 0.774,
        "(2, 1, 0)": 1.248,
        "(2, 1, 1)": 1.139,
        "(2, 1, 2)": 0.93,
        "(2, 1, 3)": 0.746,
        "(2, 2, 0)": 1.973,
        "(2, 2, 1)": 0.93,
        "(2, 2, 2)": 3.222,
        "(2, 2, 3)": 0.677,
        "(2, 3, 0)": 0.774,
        "(2, 3, 1)": 0.746,
        "(2, 3, 2)": 0.677,
        "(2, 3, 3)": 0.595,
        "(3, 0, 0)": 2.79,
        "(3, 0, 1)": 0.882,
        "(3, 0, 2)": 0.774,
        "(3, 0, 3)": 1.973,
        "(3, 1, 0)": 0.882,
        "(3, 1, 1)": 1.682,
        "(3, 1, 2)": 0.746,
        "(3, 1, 3)": 1.28,
        "(3, 2, 0)": 0.774,
        "(3, 2, 1)": 0.746,
        "(3, 2, 2)": 0.677,
        "(3, 2, 3)": 0.595,
        "(3, 3, 0)": 1.973,
        "(3, 3, 1)": 1.28,
        "(3, 3, 2)": 0.595,
        "(3, 3, 3)": 3.222,
    },
    "Cd": {
        "(0, 0, 1)": 2.81,
        "(0, 0, 2)": 2.81,
        "(0, 0, 3)": 2.81,
        "(0, 1, 0)": 1.721,
        "(0, 1, 1)": 0.391,
        "(0, 1, 2)": 1.267,
        "(0, 1, 3)": 0.253,
        "(0, 2, 0)": 1.721,
        "(0, 2, 1)": 1.048,
        "(0, 2, 2)": 0.391,
        "(0, 2, 3)": 0.886,
        "(0, 3, 0)": 1.721,
        "(0, 3, 1)": 0.425,
        "(0, 3, 2)": 0.823,
        "(0, 3, 3)": 0.391,
        "(1, 0, 0)": 0.86,
        "(1, 0, 1)": 1.954,
        "(1, 0, 2)": 0.634,
        "(1, 0, 3)": 1.263,
        "(1, 1, 0)": 1.49,
        "(1, 1, 1)": 0.72,
        "(1, 1, 2)": 1.316,
        "(1, 1, 3)": 0.583,
        "(1, 2, 0)": 0.65,
        "(1, 2, 1)": 0.16,
        "(1, 2, 2)": 0.614,
        "(1, 2, 3)": 0.144,
        "(1, 3, 0)": 0.239,
        "(1, 3, 1)": 0.592,
        "(1, 3, 2)": 0.231,
        "(1, 3, 3)": 0.557,
        "(2, 0, 0)": 0.86,
        "(2, 0, 1)": 0.21,
        "(2, 0, 2)": 1.954,
        "(2, 0, 3)": 0.177,
        "(2, 1, 0)": 0.325,
        "(2, 1, 1)": 0.801,
        "(2, 1, 2)": 0.307,
        "(2, 1, 3)": 0.721,
        "(2, 2, 0)": 1.49,
        "(2, 2, 1)": 0.369,
        "(2, 2, 2)": 0.72,
        "(2, 2, 3)": 0.346,
        "(2, 3, 0)": 0.395,
        "(2, 3, 1)": 0.589,
        "(2, 3, 2)": 0.386,
        "(2, 3, 3)": 0.565,
        "(3, 0, 0)": 0.86,
        "(3, 0, 1)": 0.425,
        "(3, 0, 2)": 0.823,
        "(3, 0, 3)": 1.954,
        "(3, 1, 0)": 0.477,
        "(3, 1, 1)": 0.118,
        "(3, 1, 2)": 0.462,
        "(3, 1, 3)": 0.111,
        "(3, 2, 0)": 0.197,
        "(3, 2, 1)": 0.491,
        "(3, 2, 2)": 0.193,
        "(3, 2, 3)": 0.47,
        "(3, 3, 0)": 1.49,
        "(3, 3, 1)": 0.247,
        "(3, 3, 2)": 0.489,
        "(3, 3, 3)": 0.72,
    },
    "Ce": {
        "(0, 0, 1)": 2.58,
        "(0, 0, 2)": 2.58,
        "(0, 0, 3)": 2.58,
        "(0, 1, 0)": 2.58,
        "(0, 1, 1)": 1.824,
        "(0, 1, 2)": 1.154,
        "(0, 1, 3)": 0.816,
        "(0, 2, 0)": 2.58,
        "(0, 2, 1)": 1.154,
        "(0, 2, 2)": 1.824,
        "(0, 2, 3)": 0.716,
        "(0, 3, 0)": 2.58,
        "(0, 3, 1)": 0.816,
        "(0, 3, 2)": 0.716,
        "(0, 3, 3)": 1.824,
        "(1, 0, 0)": 2.58,
        "(1, 0, 1)": 1.824,
        "(1, 0, 2)": 1.154,
        "(1, 0, 3)": 0.816,
        "(1, 1, 0)": 1.824,
        "(1, 1, 1)": 2.979,
        "(1, 1, 2)": 1.053,
        "(1, 1, 3)": 1.556,
        "(1, 2, 0)": 1.154,
        "(1, 2, 1)": 1.053,
        "(1, 2, 2)": 0.86,
        "(1, 2, 3)": 0.69,
        "(1, 3, 0)": 0.816,
        "(1, 3, 1)": 1.556,
        "(1, 3, 2)": 0.69,
        "(1, 3, 3)": 1.184,
        "(2, 0, 0)": 2.58,
        "(2, 0, 1)": 1.154,
        "(2, 0, 2)": 1.824,
        "(2, 0, 3)": 0.716,
        "(2, 1, 0)": 1.154,
        "(2, 1, 1)": 1.053,
        "(2, 1, 2)": 0.86,
        "(2, 1, 3)": 0.69,
        "(2, 2, 0)": 1.824,
        "(2, 2, 1)": 0.86,
        "(2, 2, 2)": 2.979,
        "(2, 2, 3)": 0.626,
        "(2, 3, 0)": 0.716,
        "(2, 3, 1)": 0.69,
        "(2, 3, 2)": 0.626,
        "(2, 3, 3)": 0.55,
        "(3, 0, 0)": 2.58,
        "(3, 0, 1)": 0.816,
        "(3, 0, 2)": 0.716,
        "(3, 0, 3)": 1.824,
        "(3, 1, 0)": 0.816,
        "(3, 1, 1)": 1.556,
        "(3, 1, 2)": 0.69,
        "(3, 1, 3)": 1.184,
        "(3, 2, 0)": 0.716,
        "(3, 2, 1)": 0.69,
        "(3, 2, 2)": 0.626,
        "(3, 2, 3)": 0.55,
        "(3, 3, 0)": 1.824,
        "(3, 3, 1)": 1.184,
        "(3, 3, 2)": 0.55,
        "(3, 3, 3)": 2.979,
    },
    "Co": {
        "(0, 0, 1)": 2.036,
        "(0, 0, 2)": 2.036,
        "(0, 0, 3)": 2.036,
        "(0, 1, 0)": 1.449,
        "(0, 1, 1)": 0.32,
        "(0, 1, 2)": 0.991,
        "(0, 1, 3)": 0.192,
        "(0, 2, 0)": 1.449,
        "(0, 2, 1)": 0.875,
        "(0, 2, 2)": 0.32,
        "(0, 2, 3)": 0.707,
        "(0, 3, 0)": 1.449,
        "(0, 3, 1)": 0.357,
        "(0, 3, 2)": 0.683,
        "(0, 3, 3)": 0.32,
        "(1, 0, 0)": 0.725,
        "(1, 0, 1)": 1.598,
        "(1, 0, 2)": 0.495,
        "(1, 0, 3)": 0.959,
        "(1, 1, 0)": 1.255,
        "(1, 1, 1)": 0.6,
        "(1, 1, 2)": 1.068,
        "(1, 1, 3)": 0.461,
        "(1, 2, 0)": 0.548,
        "(1, 2, 1)": 0.134,
        "(1, 2, 2)": 0.508,
        "(1, 2, 3)": 0.117,
        "(1, 3, 0)": 0.201,
        "(1, 3, 1)": 0.497,
        "(1, 3, 2)": 0.193,
        "(1, 3, 3)": 0.459,
        "(2, 0, 0)": 0.725,
        "(2, 0, 1)": 0.175,
        "(2, 0, 2)": 1.598,
        "(2, 0, 3)": 0.141,
        "(2, 1, 0)": 0.274,
        "(2, 1, 1)": 0.671,
        "(2, 1, 2)": 0.254,
        "(2, 1, 3)": 0.586,
        "(2, 2, 0)": 1.255,
        "(2, 2, 1)": 0.31,
        "(2, 2, 2)": 0.6,
        "(2, 2, 3)": 0.285,
        "(2, 3, 0)": 0.332,
        "(2, 3, 1)": 0.495,
        "(2, 3, 2)": 0.323,
        "(2, 3, 3)": 0.468,
        "(3, 0, 0)": 0.725,
        "(3, 0, 1)": 0.357,
        "(3, 0, 2)": 0.683,
        "(3, 0, 3)": 1.598,
        "(3, 1, 0)": 0.402,
        "(3, 1, 1)": 0.596,
        "(3, 1, 2)": 0.385,
        "(3, 1, 3)": 0.551,
        "(3, 2, 0)": 0.166,
        "(3, 2, 1)": 0.412,
        "(3, 2, 2)": 0.161,
        "(3, 2, 3)": 0.39,
        "(3, 3, 0)": 1.255,
        "(3, 3, 1)": 0.208,
        "(3, 3, 2)": 0.41,
        "(3, 3, 3)": 0.6,
    },
    "Cr": {
        "(0, 0, 1)": 1.44,
        "(0, 0, 2)": 1.44,
        "(0, 0, 3)": 1.44,
        "(0, 1, 0)": 1.44,
        "(0, 1, 1)": 2.036,
        "(0, 1, 2)": 0.644,
        "(0, 1, 3)": 0.911,
        "(0, 2, 0)": 1.44,
        "(0, 2, 1)": 0.644,
        "(0, 2, 2)": 2.036,
        "(0, 2, 3)": 0.399,
        "(0, 3, 0)": 1.44,
        "(0, 3, 1)": 0.911,
        "(0, 3, 2)": 0.399,
        "(0, 3, 3)": 2.036,
        "(1, 0, 0)": 1.44,
        "(1, 0, 1)": 2.036,
        "(1, 0, 2)": 0.644,
        "(1, 0, 3)": 0.911,
        "(1, 1, 0)": 2.036,
        "(1, 1, 1)": 0.831,
        "(1, 1, 2)": 1.176,
        "(1, 1, 3)": 0.434,
        "(1, 2, 0)": 0.644,
        "(1, 2, 1)": 1.176,
        "(1, 2, 2)": 0.48,
        "(1, 2, 3)": 0.77,
        "(1, 3, 0)": 0.911,
        "(1, 3, 1)": 0.434,
        "(1, 3, 2)": 0.77,
        "(1, 3, 3)": 0.33,
        "(2, 0, 0)": 1.44,
        "(2, 0, 1)": 0.644,
        "(2, 0, 2)": 2.036,
        "(2, 0, 3)": 0.399,
        "(2, 1, 0)": 0.644,
        "(2, 1, 1)": 1.176,
        "(2, 1, 2)": 0.48,
        "(2, 1, 3)": 0.77,
        "(2, 2, 0)": 2.036,
        "(2, 2, 1)": 0.48,
        "(2, 2, 2)": 0.831,
        "(2, 2, 3)": 0.349,
        "(2, 3, 0)": 0.399,
        "(2, 3, 1)": 0.77,
        "(2, 3, 2)": 0.349,
        "(2, 3, 3)": 0.614,
        "(3, 0, 0)": 1.44,
        "(3, 0, 1)": 0.911,
        "(3, 0, 2)": 0.399,
        "(3, 0, 3)": 2.036,
        "(3, 1, 0)": 0.911,
        "(3, 1, 1)": 0.434,
        "(3, 1, 2)": 0.77,
        "(3, 1, 3)": 0.33,
        "(3, 2, 0)": 0.399,
        "(3, 2, 1)": 0.77,
        "(3, 2, 2)": 0.349,
        "(3, 2, 3)": 0.614,
        "(3, 3, 0)": 2.036,
        "(3, 3, 1)": 0.33,
        "(3, 3, 2)": 0.614,
        "(3, 3, 3)": 0.831,
    },
    "Cs": {
        "(0, 0, 1)": 3.025,
        "(0, 0, 2)": 3.025,
        "(0, 0, 3)": 3.025,
        "(0, 1, 0)": 3.025,
        "(0, 1, 1)": 4.278,
        "(0, 1, 2)": 1.353,
        "(0, 1, 3)": 1.913,
        "(0, 2, 0)": 3.025,
        "(0, 2, 1)": 1.353,
        "(0, 2, 2)": 4.278,
        "(0, 2, 3)": 0.839,
        "(0, 3, 0)": 3.025,
        "(0, 3, 1)": 1.913,
        "(0, 3, 2)": 0.839,
        "(0, 3, 3)": 4.278,
        "(1, 0, 0)": 3.025,
        "(1, 0, 1)": 4.278,
        "(1, 0, 2)": 1.353,
        "(1, 0, 3)": 1.913,
        "(1, 1, 0)": 4.278,
        "(1, 1, 1)": 1.746,
        "(1, 1, 2)": 2.47,
        "(1, 1, 3)": 0.912,
        "(1, 2, 0)": 1.353,
        "(1, 2, 1)": 2.47,
        "(1, 2, 2)": 1.008,
        "(1, 2, 3)": 1.617,
        "(1, 3, 0)": 1.913,
        "(1, 3, 1)": 0.912,
        "(1, 3, 2)": 1.617,
        "(1, 3, 3)": 0.694,
        "(2, 0, 0)": 3.025,
        "(2, 0, 1)": 1.353,
        "(2, 0, 2)": 4.278,
        "(2, 0, 3)": 0.839,
        "(2, 1, 0)": 1.353,
        "(2, 1, 1)": 2.47,
        "(2, 1, 2)": 1.008,
        "(2, 1, 3)": 1.617,
        "(2, 2, 0)": 4.278,
        "(2, 2, 1)": 1.008,
        "(2, 2, 2)": 1.746,
        "(2, 2, 3)": 0.734,
        "(2, 3, 0)": 0.839,
        "(2, 3, 1)": 1.617,
        "(2, 3, 2)": 0.734,
        "(2, 3, 3)": 1.29,
        "(3, 0, 0)": 3.025,
        "(3, 0, 1)": 1.913,
        "(3, 0, 2)": 0.839,
        "(3, 0, 3)": 4.278,
        "(3, 1, 0)": 1.913,
        "(3, 1, 1)": 0.912,
        "(3, 1, 2)": 1.617,
        "(3, 1, 3)": 0.694,
        "(3, 2, 0)": 0.839,
        "(3, 2, 1)": 1.617,
        "(3, 2, 2)": 0.734,
        "(3, 2, 3)": 1.29,
        "(3, 3, 0)": 4.278,
        "(3, 3, 1)": 0.694,
        "(3, 3, 2)": 1.29,
        "(3, 3, 3)": 1.746,
    },
    "Cu": {
        "(0, 0, 1)": 1.805,
        "(0, 0, 2)": 1.805,
        "(0, 0, 3)": 1.805,
        "(0, 1, 0)": 1.805,
        "(0, 1, 1)": 1.276,
        "(0, 1, 2)": 0.807,
        "(0, 1, 3)": 0.571,
        "(0, 2, 0)": 1.805,
        "(0, 2, 1)": 0.807,
        "(0, 2, 2)": 1.276,
        "(0, 2, 3)": 0.501,
        "(0, 3, 0)": 1.805,
        "(0, 3, 1)": 0.571,
        "(0, 3, 2)": 0.501,
        "(0, 3, 3)": 1.276,
        "(1, 0, 0)": 1.805,
        "(1, 0, 1)": 1.276,
        "(1, 0, 2)": 0.807,
        "(1, 0, 3)": 0.571,
        "(1, 1, 0)": 1.276,
        "(1, 1, 1)": 2.084,
        "(1, 1, 2)": 0.737,
        "(1, 1, 3)": 1.088,
        "(1, 2, 0)": 0.807,
        "(1, 2, 1)": 0.737,
        "(1, 2, 2)": 0.602,
        "(1, 2, 3)": 0.482,
        "(1, 3, 0)": 0.571,
        "(1, 3, 1)": 1.088,
        "(1, 3, 2)": 0.482,
        "(1, 3, 3)": 0.828,
        "(2, 0, 0)": 1.805,
        "(2, 0, 1)": 0.807,
        "(2, 0, 2)": 1.276,
        "(2, 0, 3)": 0.501,
        "(2, 1, 0)": 0.807,
        "(2, 1, 1)": 0.737,
        "(2, 1, 2)": 0.602,
        "(2, 1, 3)": 0.482,
        "(2, 2, 0)": 1.276,
        "(2, 2, 1)": 0.602,
        "(2, 2, 2)": 2.084,
        "(2, 2, 3)": 0.438,
        "(2, 3, 0)": 0.501,
        "(2, 3, 1)": 0.482,
        "(2, 3, 2)": 0.438,
        "(2, 3, 3)": 0.385,
        "(3, 0, 0)": 1.805,
        "(3, 0, 1)": 0.571,
        "(3, 0, 2)": 0.501,
        "(3, 0, 3)": 1.276,
        "(3, 1, 0)": 0.571,
        "(3, 1, 1)": 1.088,
        "(3, 1, 2)": 0.482,
        "(3, 1, 3)": 0.828,
        "(3, 2, 0)": 0.501,
        "(3, 2, 1)": 0.482,
        "(3, 2, 2)": 0.438,
        "(3, 2, 3)": 0.385,
        "(3, 3, 0)": 1.276,
        "(3, 3, 1)": 0.828,
        "(3, 3, 2)": 0.385,
        "(3, 3, 3)": 2.084,
    },
    "Dy": {
        "(0, 0, 1)": 2.824,
        "(0, 0, 2)": 2.824,
        "(0, 0, 3)": 2.824,
        "(0, 1, 0)": 2.073,
        "(0, 1, 1)": 0.454,
        "(0, 1, 2)": 1.393,
        "(0, 1, 3)": 0.268,
        "(0, 2, 0)": 2.073,
        "(0, 2, 1)": 1.249,
        "(0, 2, 2)": 0.454,
        "(0, 2, 3)": 0.999,
        "(0, 3, 0)": 2.073,
        "(0, 3, 1)": 0.51,
        "(0, 3, 2)": 0.973,
        "(0, 3, 3)": 0.454,
        "(1, 0, 0)": 1.036,
        "(1, 0, 1)": 2.27,
        "(1, 0, 2)": 0.697,
        "(1, 0, 3)": 1.342,
        "(1, 1, 0)": 1.795,
        "(1, 1, 1)": 0.855,
        "(1, 1, 2)": 1.515,
        "(1, 1, 3)": 0.65,
        "(1, 2, 0)": 0.783,
        "(1, 2, 1)": 0.192,
        "(1, 2, 2)": 0.723,
        "(1, 2, 3)": 0.166,
        "(1, 3, 0)": 0.287,
        "(1, 3, 1)": 0.71,
        "(1, 3, 2)": 0.275,
        "(1, 3, 3)": 0.653,
        "(2, 0, 0)": 1.036,
        "(2, 0, 1)": 0.25,
        "(2, 0, 2)": 2.27,
        "(2, 0, 3)": 0.2,
        "(2, 1, 0)": 0.392,
        "(2, 1, 1)": 0.959,
        "(2, 1, 2)": 0.362,
        "(2, 1, 3)": 0.831,
        "(2, 2, 0)": 1.795,
        "(2, 2, 1)": 0.443,
        "(2, 2, 2)": 0.855,
        "(2, 2, 3)": 0.405,
        "(2, 3, 0)": 0.476,
        "(2, 3, 1)": 0.118,
        "(2, 3, 2)": 0.461,
        "(2, 3, 3)": 0.111,
        "(3, 0, 0)": 1.036,
        "(3, 0, 1)": 0.51,
        "(3, 0, 2)": 0.973,
        "(3, 0, 3)": 2.27,
        "(3, 1, 0)": 0.575,
        "(3, 1, 1)": 0.142,
        "(3, 1, 2)": 0.55,
        "(3, 1, 3)": 0.131,
        "(3, 2, 0)": 0.238,
        "(3, 2, 1)": 0.59,
        "(3, 2, 2)": 0.231,
        "(3, 2, 3)": 0.556,
        "(3, 3, 0)": 1.795,
        "(3, 3, 1)": 0.298,
        "(3, 3, 2)": 0.585,
        "(3, 3, 3)": 0.855,
    },
    "Er": {
        "(0, 0, 1)": 2.795,
        "(0, 0, 2)": 2.795,
        "(0, 0, 3)": 2.795,
        "(0, 1, 0)": 2.055,
        "(0, 1, 1)": 0.45,
        "(0, 1, 2)": 1.38,
        "(0, 1, 3)": 0.266,
        "(0, 2, 0)": 2.055,
        "(0, 2, 1)": 1.238,
        "(0, 2, 2)": 0.45,
        "(0, 2, 3)": 0.99,
        "(0, 3, 0)": 2.055,
        "(0, 3, 1)": 0.505,
        "(0, 3, 2)": 0.965,
        "(0, 3, 3)": 0.45,
        "(1, 0, 0)": 1.028,
        "(1, 0, 1)": 2.25,
        "(1, 0, 2)": 0.69,
        "(1, 0, 3)": 1.329,
        "(1, 1, 0)": 1.78,
        "(1, 1, 1)": 0.848,
        "(1, 1, 2)": 1.501,
        "(1, 1, 3)": 0.644,
        "(1, 2, 0)": 0.777,
        "(1, 2, 1)": 0.19,
        "(1, 2, 2)": 0.717,
        "(1, 2, 3)": 0.165,
        "(1, 3, 0)": 0.285,
        "(1, 3, 1)": 0.704,
        "(1, 3, 2)": 0.273,
        "(1, 3, 3)": 0.648,
        "(2, 0, 0)": 1.028,
        "(2, 0, 1)": 0.248,
        "(2, 0, 2)": 2.25,
        "(2, 0, 3)": 0.198,
        "(2, 1, 0)": 0.388,
        "(2, 1, 1)": 0.951,
        "(2, 1, 2)": 0.359,
        "(2, 1, 3)": 0.823,
        "(2, 2, 0)": 1.78,
        "(2, 2, 1)": 0.439,
        "(2, 2, 2)": 0.848,
        "(2, 2, 3)": 0.402,
        "(2, 3, 0)": 0.472,
        "(2, 3, 1)": 0.117,
        "(2, 3, 2)": 0.457,
        "(2, 3, 3)": 0.11,
        "(3, 0, 0)": 1.028,
        "(3, 0, 1)": 0.505,
        "(3, 0, 2)": 0.965,
        "(3, 0, 3)": 2.25,
        "(3, 1, 0)": 0.57,
        "(3, 1, 1)": 0.141,
        "(3, 1, 2)": 0.545,
        "(3, 1, 3)": 0.13,
        "(3, 2, 0)": 0.236,
        "(3, 2, 1)": 0.585,
        "(3, 2, 2)": 0.229,
        "(3, 2, 3)": 0.551,
        "(3, 3, 0)": 1.78,
        "(3, 3, 1)": 0.295,
        "(3, 3, 2)": 0.58,
        "(3, 3, 3)": 0.848,
    },
    "Eu": {
        "(0, 0, 1)": 2.305,
        "(0, 0, 2)": 2.305,
        "(0, 0, 3)": 2.305,
        "(0, 1, 0)": 2.305,
        "(0, 1, 1)": 3.26,
        "(0, 1, 2)": 1.031,
        "(0, 1, 3)": 1.458,
        "(0, 2, 0)": 2.305,
        "(0, 2, 1)": 1.031,
        "(0, 2, 2)": 3.26,
        "(0, 2, 3)": 0.639,
        "(0, 3, 0)": 2.305,
        "(0, 3, 1)": 1.458,
        "(0, 3, 2)": 0.639,
        "(0, 3, 3)": 3.26,
        "(1, 0, 0)": 2.305,
        "(1, 0, 1)": 3.26,
        "(1, 0, 2)": 1.031,
        "(1, 0, 3)": 1.458,
        "(1, 1, 0)": 3.26,
        "(1, 1, 1)": 1.331,
        "(1, 1, 2)": 1.882,
        "(1, 1, 3)": 0.695,
        "(1, 2, 0)": 1.031,
        "(1, 2, 1)": 1.882,
        "(1, 2, 2)": 0.768,
        "(1, 2, 3)": 1.232,
        "(1, 3, 0)": 1.458,
        "(1, 3, 1)": 0.695,
        "(1, 3, 2)": 1.232,
        "(1, 3, 3)": 0.529,
        "(2, 0, 0)": 2.305,
        "(2, 0, 1)": 1.031,
        "(2, 0, 2)": 3.26,
        "(2, 0, 3)": 0.639,
        "(2, 1, 0)": 1.031,
        "(2, 1, 1)": 1.882,
        "(2, 1, 2)": 0.768,
        "(2, 1, 3)": 1.232,
        "(2, 2, 0)": 3.26,
        "(2, 2, 1)": 0.768,
        "(2, 2, 2)": 1.331,
        "(2, 2, 3)": 0.559,
        "(2, 3, 0)": 0.639,
        "(2, 3, 1)": 1.232,
        "(2, 3, 2)": 0.559,
        "(2, 3, 3)": 0.983,
        "(3, 0, 0)": 2.305,
        "(3, 0, 1)": 1.458,
        "(3, 0, 2)": 0.639,
        "(3, 0, 3)": 3.26,
        "(3, 1, 0)": 1.458,
        "(3, 1, 1)": 0.695,
        "(3, 1, 2)": 1.232,
        "(3, 1, 3)": 0.529,
        "(3, 2, 0)": 0.639,
        "(3, 2, 1)": 1.232,
        "(3, 2, 2)": 0.559,
        "(3, 2, 3)": 0.983,
        "(3, 3, 0)": 3.26,
        "(3, 3, 1)": 0.529,
        "(3, 3, 2)": 0.983,
        "(3, 3, 3)": 1.331,
    },
    "Fe": {
        "(0, 0, 1)": 1.435,
        "(0, 0, 2)": 1.435,
        "(0, 0, 3)": 1.435,
        "(0, 1, 0)": 1.435,
        "(0, 1, 1)": 2.029,
        "(0, 1, 2)": 0.642,
        "(0, 1, 3)": 0.908,
        "(0, 2, 0)": 1.435,
        "(0, 2, 1)": 0.642,
        "(0, 2, 2)": 2.029,
        "(0, 2, 3)": 0.398,
        "(0, 3, 0)": 1.435,
        "(0, 3, 1)": 0.908,
        "(0, 3, 2)": 0.398,
        "(0, 3, 3)": 2.029,
        "(1, 0, 0)": 1.435,
        "(1, 0, 1)": 2.029,
        "(1, 0, 2)": 0.642,
        "(1, 0, 3)": 0.908,
        "(1, 1, 0)": 2.029,
        "(1, 1, 1)": 0.828,
        "(1, 1, 2)": 1.172,
        "(1, 1, 3)": 0.433,
        "(1, 2, 0)": 0.642,
        "(1, 2, 1)": 1.172,
        "(1, 2, 2)": 0.478,
        "(1, 2, 3)": 0.767,
        "(1, 3, 0)": 0.908,
        "(1, 3, 1)": 0.433,
        "(1, 3, 2)": 0.767,
        "(1, 3, 3)": 0.329,
        "(2, 0, 0)": 1.435,
        "(2, 0, 1)": 0.642,
        "(2, 0, 2)": 2.029,
        "(2, 0, 3)": 0.398,
        "(2, 1, 0)": 0.642,
        "(2, 1, 1)": 1.172,
        "(2, 1, 2)": 0.478,
        "(2, 1, 3)": 0.767,
        "(2, 2, 0)": 2.029,
        "(2, 2, 1)": 0.478,
        "(2, 2, 2)": 0.828,
        "(2, 2, 3)": 0.348,
        "(2, 3, 0)": 0.398,
        "(2, 3, 1)": 0.767,
        "(2, 3, 2)": 0.348,
        "(2, 3, 3)": 0.612,
        "(3, 0, 0)": 1.435,
        "(3, 0, 1)": 0.908,
        "(3, 0, 2)": 0.398,
        "(3, 0, 3)": 2.029,
        "(3, 1, 0)": 0.908,
        "(3, 1, 1)": 0.433,
        "(3, 1, 2)": 0.767,
        "(3, 1, 3)": 0.329,
        "(3, 2, 0)": 0.398,
        "(3, 2, 1)": 0.767,
        "(3, 2, 2)": 0.348,
        "(3, 2, 3)": 0.612,
        "(3, 3, 0)": 2.029,
        "(3, 3, 1)": 0.329,
        "(3, 3, 2)": 0.612,
        "(3, 3, 3)": 0.828,
    },
    "Gd": {
        "(0, 0, 1)": 2.89,
        "(0, 0, 2)": 2.89,
        "(0, 0, 3)": 2.89,
        "(0, 1, 0)": 2.102,
        "(0, 1, 1)": 0.461,
        "(0, 1, 2)": 1.42,
        "(0, 1, 3)": 0.274,
        "(0, 2, 0)": 2.102,
        "(0, 2, 1)": 1.267,
        "(0, 2, 2)": 0.461,
        "(0, 2, 3)": 1.017,
        "(0, 3, 0)": 2.102,
        "(0, 3, 1)": 0.517,
        "(0, 3, 2)": 0.988,
        "(0, 3, 3)": 0.461,
        "(1, 0, 0)": 1.051,
        "(1, 0, 1)": 2.306,
        "(1, 0, 2)": 0.71,
        "(1, 0, 3)": 1.37,
        "(1, 1, 0)": 1.82,
        "(1, 1, 1)": 0.868,
        "(1, 1, 2)": 1.54,
        "(1, 1, 3)": 0.662,
        "(1, 2, 0)": 0.794,
        "(1, 2, 1)": 0.194,
        "(1, 2, 2)": 0.734,
        "(1, 2, 3)": 0.169,
        "(1, 3, 0)": 0.291,
        "(1, 3, 1)": 0.72,
        "(1, 3, 2)": 0.279,
        "(1, 3, 3)": 0.663,
        "(2, 0, 0)": 1.051,
        "(2, 0, 1)": 0.253,
        "(2, 0, 2)": 2.306,
        "(2, 0, 3)": 0.203,
        "(2, 1, 0)": 0.397,
        "(2, 1, 1)": 0.972,
        "(2, 1, 2)": 0.367,
        "(2, 1, 3)": 0.844,
        "(2, 2, 0)": 1.82,
        "(2, 2, 1)": 0.449,
        "(2, 2, 2)": 0.868,
        "(2, 2, 3)": 0.411,
        "(2, 3, 0)": 0.482,
        "(2, 3, 1)": 0.12,
        "(2, 3, 2)": 0.468,
        "(2, 3, 3)": 0.113,
        "(3, 0, 0)": 1.051,
        "(3, 0, 1)": 0.517,
        "(3, 0, 2)": 0.988,
        "(3, 0, 3)": 2.306,
        "(3, 1, 0)": 0.583,
        "(3, 1, 1)": 0.144,
        "(3, 1, 2)": 0.558,
        "(3, 1, 3)": 0.133,
        "(3, 2, 0)": 0.241,
        "(3, 2, 1)": 0.598,
        "(3, 2, 2)": 0.234,
        "(3, 2, 3)": 0.564,
        "(3, 3, 0)": 1.82,
        "(3, 3, 1)": 0.302,
        "(3, 3, 2)": 0.594,
        "(3, 3, 3)": 0.868,
    },
    "Ge": {
        "(0, 0, 1)": 1.415,
        "(0, 0, 2)": 1.415,
        "(0, 0, 3)": 1.415,
        "(0, 1, 0)": 1.415,
        "(0, 1, 1)": 2.001,
        "(0, 1, 2)": 0.633,
        "(0, 1, 3)": 0.895,
        "(0, 2, 0)": 1.415,
        "(0, 2, 1)": 0.633,
        "(0, 2, 2)": 2.001,
        "(0, 2, 3)": 0.392,
        "(0, 3, 0)": 1.415,
        "(0, 3, 1)": 0.895,
        "(0, 3, 2)": 0.392,
        "(0, 3, 3)": 2.001,
        "(1, 0, 0)": 1.415,
        "(1, 0, 1)": 2.001,
        "(1, 0, 2)": 0.633,
        "(1, 0, 3)": 0.895,
        "(1, 1, 0)": 2.001,
        "(1, 1, 1)": 2.451,
        "(1, 1, 2)": 1.155,
        "(1, 1, 3)": 0.427,
        "(1, 2, 0)": 0.633,
        "(1, 2, 1)": 1.155,
        "(1, 2, 2)": 0.472,
        "(1, 2, 3)": 0.756,
        "(1, 3, 0)": 0.895,
        "(1, 3, 1)": 0.427,
        "(1, 3, 2)": 0.756,
        "(1, 3, 3)": 0.974,
        "(2, 0, 0)": 1.415,
        "(2, 0, 1)": 0.633,
        "(2, 0, 2)": 2.001,
        "(2, 0, 3)": 0.392,
        "(2, 1, 0)": 0.633,
        "(2, 1, 1)": 1.155,
        "(2, 1, 2)": 0.472,
        "(2, 1, 3)": 0.756,
        "(2, 2, 0)": 2.001,
        "(2, 2, 1)": 0.472,
        "(2, 2, 2)": 2.451,
        "(2, 2, 3)": 0.343,
        "(2, 3, 0)": 0.392,
        "(2, 3, 1)": 0.756,
        "(2, 3, 2)": 0.343,
        "(2, 3, 3)": 0.603,
        "(3, 0, 0)": 1.415,
        "(3, 0, 1)": 0.895,
        "(3, 0, 2)": 0.392,
        "(3, 0, 3)": 2.001,
        "(3, 1, 0)": 0.895,
        "(3, 1, 1)": 0.427,
        "(3, 1, 2)": 0.756,
        "(3, 1, 3)": 0.974,
        "(3, 2, 0)": 0.392,
        "(3, 2, 1)": 0.756,
        "(3, 2, 2)": 0.343,
        "(3, 2, 3)": 0.603,
        "(3, 3, 0)": 2.001,
        "(3, 3, 1)": 0.974,
        "(3, 3, 2)": 0.603,
        "(3, 3, 3)": 2.451,
    },
    "Hf": {
        "(0, 0, 1)": 2.531,
        "(0, 0, 2)": 2.531,
        "(0, 0, 3)": 2.531,
        "(0, 1, 0)": 1.848,
        "(0, 1, 1)": 0.405,
        "(0, 1, 2)": 1.246,
        "(0, 1, 3)": 0.24,
        "(0, 2, 0)": 1.848,
        "(0, 2, 1)": 1.114,
        "(0, 2, 2)": 0.405,
        "(0, 2, 3)": 0.892,
        "(0, 3, 0)": 1.848,
        "(0, 3, 1)": 0.454,
        "(0, 3, 2)": 0.868,
        "(0, 3, 3)": 0.405,
        "(1, 0, 0)": 0.924,
        "(1, 0, 1)": 2.026,
        "(1, 0, 2)": 0.623,
        "(1, 0, 3)": 1.201,
        "(1, 1, 0)": 1.6,
        "(1, 1, 1)": 0.763,
        "(1, 1, 2)": 1.352,
        "(1, 1, 3)": 0.581,
        "(1, 2, 0)": 0.698,
        "(1, 2, 1)": 0.171,
        "(1, 2, 2)": 0.645,
        "(1, 2, 3)": 0.148,
        "(1, 3, 0)": 0.256,
        "(1, 3, 1)": 0.633,
        "(1, 3, 2)": 0.245,
        "(1, 3, 3)": 0.583,
        "(2, 0, 0)": 0.924,
        "(2, 0, 1)": 0.223,
        "(2, 0, 2)": 2.026,
        "(2, 0, 3)": 0.178,
        "(2, 1, 0)": 0.349,
        "(2, 1, 1)": 0.855,
        "(2, 1, 2)": 0.323,
        "(2, 1, 3)": 0.742,
        "(2, 2, 0)": 1.6,
        "(2, 2, 1)": 0.395,
        "(2, 2, 2)": 0.763,
        "(2, 2, 3)": 0.361,
        "(2, 3, 0)": 0.424,
        "(2, 3, 1)": 0.105,
        "(2, 3, 2)": 0.411,
        "(2, 3, 3)": 0.595,
        "(3, 0, 0)": 0.924,
        "(3, 0, 1)": 0.454,
        "(3, 0, 2)": 0.868,
        "(3, 0, 3)": 2.026,
        "(3, 1, 0)": 0.512,
        "(3, 1, 1)": 0.127,
        "(3, 1, 2)": 0.49,
        "(3, 1, 3)": 0.117,
        "(3, 2, 0)": 0.212,
        "(3, 2, 1)": 0.526,
        "(3, 2, 2)": 0.206,
        "(3, 2, 3)": 0.496,
        "(3, 3, 0)": 1.6,
        "(3, 3, 1)": 0.265,
        "(3, 3, 2)": 0.522,
        "(3, 3, 3)": 0.763,
    },
    "Hg": {
        "(0, 0, 1)": 2.728,
        "(0, 0, 2)": 2.728,
        "(0, 0, 3)": 2.728,
        "(0, 1, 0)": 2.728,
        "(0, 1, 1)": 2.228,
        "(0, 1, 2)": 1.364,
        "(0, 1, 3)": 0.936,
        "(0, 2, 0)": 2.728,
        "(0, 2, 1)": 1.364,
        "(0, 2, 2)": 2.228,
        "(0, 2, 3)": 0.863,
        "(0, 3, 0)": 2.728,
        "(0, 3, 1)": 0.936,
        "(0, 3, 2)": 0.863,
        "(0, 3, 3)": 2.228,
        "(1, 0, 0)": 2.728,
        "(1, 0, 1)": 2.228,
        "(1, 0, 2)": 1.364,
        "(1, 0, 3)": 0.936,
        "(1, 1, 0)": 2.228,
        "(1, 1, 1)": 2.23,
        "(1, 1, 2)": 1.459,
        "(1, 1, 3)": 0.997,
        "(1, 2, 0)": 1.364,
        "(1, 2, 1)": 1.459,
        "(1, 2, 2)": 1.221,
        "(1, 2, 3)": 0.936,
        "(1, 3, 0)": 0.936,
        "(1, 3, 1)": 0.997,
        "(1, 3, 2)": 0.936,
        "(1, 3, 3)": 0.805,
        "(2, 0, 0)": 2.728,
        "(2, 0, 1)": 1.364,
        "(2, 0, 2)": 2.228,
        "(2, 0, 3)": 0.863,
        "(2, 1, 0)": 1.364,
        "(2, 1, 1)": 1.459,
        "(2, 1, 2)": 1.221,
        "(2, 1, 3)": 0.936,
        "(2, 2, 0)": 2.228,
        "(2, 2, 1)": 1.221,
        "(2, 2, 2)": 2.23,
        "(2, 2, 3)": 0.91,
        "(2, 3, 0)": 0.863,
        "(2, 3, 1)": 0.936,
        "(2, 3, 2)": 0.91,
        "(2, 3, 3)": 0.805,
        "(3, 0, 0)": 2.728,
        "(3, 0, 1)": 0.936,
        "(3, 0, 2)": 0.863,
        "(3, 0, 3)": 2.228,
        "(3, 1, 0)": 0.936,
        "(3, 1, 1)": 0.997,
        "(3, 1, 2)": 0.936,
        "(3, 1, 3)": 0.805,
        "(3, 2, 0)": 0.863,
        "(3, 2, 1)": 0.936,
        "(3, 2, 2)": 0.91,
        "(3, 2, 3)": 0.805,
        "(3, 3, 0)": 2.228,
        "(3, 3, 1)": 0.805,
        "(3, 3, 2)": 0.805,
        "(3, 3, 3)": 2.23,
    },
    "Ho": {
        "(0, 0, 1)": 2.81,
        "(0, 0, 2)": 2.81,
        "(0, 0, 3)": 2.81,
        "(0, 1, 0)": 2.067,
        "(0, 1, 1)": 0.452,
        "(0, 1, 2)": 1.388,
        "(0, 1, 3)": 0.267,
        "(0, 2, 0)": 2.067,
        "(0, 2, 1)": 1.245,
        "(0, 2, 2)": 0.452,
        "(0, 2, 3)": 0.995,
        "(0, 3, 0)": 2.067,
        "(0, 3, 1)": 0.508,
        "(0, 3, 2)": 0.97,
        "(0, 3, 3)": 0.452,
        "(1, 0, 0)": 1.033,
        "(1, 0, 1)": 2.262,
        "(1, 0, 2)": 0.694,
        "(1, 0, 3)": 1.336,
        "(1, 1, 0)": 1.79,
        "(1, 1, 1)": 0.853,
        "(1, 1, 2)": 1.51,
        "(1, 1, 3)": 0.647,
        "(1, 2, 0)": 0.781,
        "(1, 2, 1)": 0.191,
        "(1, 2, 2)": 0.721,
        "(1, 2, 3)": 0.166,
        "(1, 3, 0)": 0.287,
        "(1, 3, 1)": 0.708,
        "(1, 3, 2)": 0.274,
        "(1, 3, 3)": 0.651,
        "(2, 0, 0)": 1.033,
        "(2, 0, 1)": 0.249,
        "(2, 0, 2)": 2.262,
        "(2, 0, 3)": 0.199,
        "(2, 1, 0)": 0.391,
        "(2, 1, 1)": 0.956,
        "(2, 1, 2)": 0.361,
        "(2, 1, 3)": 0.828,
        "(2, 2, 0)": 1.79,
        "(2, 2, 1)": 0.442,
        "(2, 2, 2)": 0.853,
        "(2, 2, 3)": 0.404,
        "(2, 3, 0)": 0.474,
        "(2, 3, 1)": 0.118,
        "(2, 3, 2)": 0.46,
        "(2, 3, 3)": 0.111,
        "(3, 0, 0)": 1.033,
        "(3, 0, 1)": 0.508,
        "(3, 0, 2)": 0.97,
        "(3, 0, 3)": 2.262,
        "(3, 1, 0)": 0.573,
        "(3, 1, 1)": 0.142,
        "(3, 1, 2)": 0.548,
        "(3, 1, 3)": 0.13,
        "(3, 2, 0)": 0.237,
        "(3, 2, 1)": 0.588,
        "(3, 2, 2)": 0.23,
        "(3, 2, 3)": 0.554,
        "(3, 3, 0)": 1.79,
        "(3, 3, 1)": 0.297,
        "(3, 3, 2)": 0.584,
        "(3, 3, 3)": 0.853,
    },
    "In": {
        "(0, 0, 1)": 2.295,
        "(0, 0, 2)": 2.295,
        "(0, 0, 3)": 2.295,
        "(0, 1, 0)": 2.712,
        "(0, 1, 1)": 1.393,
        "(0, 1, 2)": 0.886,
        "(0, 1, 3)": 0.644,
        "(0, 2, 0)": 2.712,
        "(0, 2, 1)": 0.948,
        "(0, 2, 2)": 1.393,
        "(0, 2, 3)": 0.543,
        "(0, 3, 0)": 2.712,
        "(0, 3, 1)": 0.71,
        "(0, 3, 2)": 0.566,
        "(0, 3, 3)": 1.393,
        "(1, 0, 0)": 2.712,
        "(1, 0, 1)": 1.393,
        "(1, 0, 2)": 0.886,
        "(1, 0, 3)": 0.644,
        "(1, 1, 0)": 1.681,
        "(1, 1, 1)": 1.041,
        "(1, 1, 2)": 0.731,
        "(1, 1, 3)": 0.559,
        "(1, 2, 0)": 1.089,
        "(1, 2, 1)": 0.79,
        "(1, 2, 2)": 0.604,
        "(1, 2, 3)": 0.484,
        "(1, 3, 0)": 0.789,
        "(1, 3, 1)": 0.626,
        "(1, 3, 2)": 0.507,
        "(1, 3, 3)": 0.423,
        "(2, 0, 0)": 2.712,
        "(2, 0, 1)": 0.948,
        "(2, 0, 2)": 1.393,
        "(2, 0, 3)": 0.543,
        "(2, 1, 0)": 1.089,
        "(2, 1, 1)": 0.79,
        "(2, 1, 2)": 0.604,
        "(2, 1, 3)": 0.484,
        "(2, 2, 0)": 1.681,
        "(2, 2, 1)": 0.65,
        "(2, 2, 2)": 1.041,
        "(2, 2, 3)": 0.43,
        "(2, 3, 0)": 0.665,
        "(2, 3, 1)": 0.542,
        "(2, 3, 2)": 0.451,
        "(2, 3, 3)": 0.383,
        "(3, 0, 0)": 2.712,
        "(3, 0, 1)": 0.71,
        "(3, 0, 2)": 0.566,
        "(3, 0, 3)": 1.393,
        "(3, 1, 0)": 0.789,
        "(3, 1, 1)": 0.626,
        "(3, 1, 2)": 0.507,
        "(3, 1, 3)": 0.423,
        "(3, 2, 0)": 0.665,
        "(3, 2, 1)": 0.542,
        "(3, 2, 2)": 0.451,
        "(3, 2, 3)": 0.383,
        "(3, 3, 0)": 1.681,
        "(3, 3, 1)": 0.471,
        "(3, 3, 2)": 0.401,
        "(3, 3, 3)": 1.041,
    },
    "Ir": {
        "(0, 0, 1)": 1.92,
        "(0, 0, 2)": 1.92,
        "(0, 0, 3)": 1.92,
        "(0, 1, 0)": 1.92,
        "(0, 1, 1)": 1.358,
        "(0, 1, 2)": 0.859,
        "(0, 1, 3)": 0.607,
        "(0, 2, 0)": 1.92,
        "(0, 2, 1)": 0.859,
        "(0, 2, 2)": 1.358,
        "(0, 2, 3)": 0.533,
        "(0, 3, 0)": 1.92,
        "(0, 3, 1)": 0.607,
        "(0, 3, 2)": 0.533,
        "(0, 3, 3)": 1.358,
        "(1, 0, 0)": 1.92,
        "(1, 0, 1)": 1.358,
        "(1, 0, 2)": 0.859,
        "(1, 0, 3)": 0.607,
        "(1, 1, 0)": 1.358,
        "(1, 1, 1)": 2.217,
        "(1, 1, 2)": 0.784,
        "(1, 1, 3)": 1.158,
        "(1, 2, 0)": 0.859,
        "(1, 2, 1)": 0.784,
        "(1, 2, 2)": 0.64,
        "(1, 2, 3)": 0.513,
        "(1, 3, 0)": 0.607,
        "(1, 3, 1)": 1.158,
        "(1, 3, 2)": 0.513,
        "(1, 3, 3)": 0.881,
        "(2, 0, 0)": 1.92,
        "(2, 0, 1)": 0.859,
        "(2, 0, 2)": 1.358,
        "(2, 0, 3)": 0.533,
        "(2, 1, 0)": 0.859,
        "(2, 1, 1)": 0.784,
        "(2, 1, 2)": 0.64,
        "(2, 1, 3)": 0.513,
        "(2, 2, 0)": 1.358,
        "(2, 2, 1)": 0.64,
        "(2, 2, 2)": 2.217,
        "(2, 2, 3)": 0.466,
        "(2, 3, 0)": 0.533,
        "(2, 3, 1)": 0.513,
        "(2, 3, 2)": 0.466,
        "(2, 3, 3)": 0.409,
        "(3, 0, 0)": 1.92,
        "(3, 0, 1)": 0.607,
        "(3, 0, 2)": 0.533,
        "(3, 0, 3)": 1.358,
        "(3, 1, 0)": 0.607,
        "(3, 1, 1)": 1.158,
        "(3, 1, 2)": 0.513,
        "(3, 1, 3)": 0.881,
        "(3, 2, 0)": 0.533,
        "(3, 2, 1)": 0.513,
        "(3, 2, 2)": 0.466,
        "(3, 2, 3)": 0.409,
        "(3, 3, 0)": 1.358,
        "(3, 3, 1)": 0.881,
        "(3, 3, 2)": 0.409,
        "(3, 3, 3)": 2.217,
    },
    "K": {
        "(0, 0, 1)": 2.615,
        "(0, 0, 2)": 2.615,
        "(0, 0, 3)": 2.615,
        "(0, 1, 0)": 2.615,
        "(0, 1, 1)": 3.698,
        "(0, 1, 2)": 1.169,
        "(0, 1, 3)": 1.654,
        "(0, 2, 0)": 2.615,
        "(0, 2, 1)": 1.169,
        "(0, 2, 2)": 3.698,
        "(0, 2, 3)": 0.725,
        "(0, 3, 0)": 2.615,
        "(0, 3, 1)": 1.654,
        "(0, 3, 2)": 0.725,
        "(0, 3, 3)": 3.698,
        "(1, 0, 0)": 2.615,
        "(1, 0, 1)": 3.698,
        "(1, 0, 2)": 1.169,
        "(1, 0, 3)": 1.654,
        "(1, 1, 0)": 3.698,
        "(1, 1, 1)": 1.51,
        "(1, 1, 2)": 2.135,
        "(1, 1, 3)": 0.788,
        "(1, 2, 0)": 1.169,
        "(1, 2, 1)": 2.135,
        "(1, 2, 2)": 0.872,
        "(1, 2, 3)": 1.398,
        "(1, 3, 0)": 1.654,
        "(1, 3, 1)": 0.788,
        "(1, 3, 2)": 1.398,
        "(1, 3, 3)": 0.6,
        "(2, 0, 0)": 2.615,
        "(2, 0, 1)": 1.169,
        "(2, 0, 2)": 3.698,
        "(2, 0, 3)": 0.725,
        "(2, 1, 0)": 1.169,
        "(2, 1, 1)": 2.135,
        "(2, 1, 2)": 0.872,
        "(2, 1, 3)": 1.398,
        "(2, 2, 0)": 3.698,
        "(2, 2, 1)": 0.872,
        "(2, 2, 2)": 1.51,
        "(2, 2, 3)": 0.634,
        "(2, 3, 0)": 0.725,
        "(2, 3, 1)": 1.398,
        "(2, 3, 2)": 0.634,
        "(2, 3, 3)": 1.115,
        "(3, 0, 0)": 2.615,
        "(3, 0, 1)": 1.654,
        "(3, 0, 2)": 0.725,
        "(3, 0, 3)": 3.698,
        "(3, 1, 0)": 1.654,
        "(3, 1, 1)": 0.788,
        "(3, 1, 2)": 1.398,
        "(3, 1, 3)": 0.6,
        "(3, 2, 0)": 0.725,
        "(3, 2, 1)": 1.398,
        "(3, 2, 2)": 0.634,
        "(3, 2, 3)": 1.115,
        "(3, 3, 0)": 3.698,
        "(3, 3, 1)": 0.6,
        "(3, 3, 2)": 1.115,
        "(3, 3, 3)": 1.51,
    },
    "Kr": {
        "(0, 0, 1)": 2.86,
        "(0, 0, 2)": 2.86,
        "(0, 0, 3)": 2.86,
        "(0, 1, 0)": 2.86,
        "(0, 1, 1)": 2.022,
        "(0, 1, 2)": 1.279,
        "(0, 1, 3)": 0.904,
        "(0, 2, 0)": 2.86,
        "(0, 2, 1)": 1.279,
        "(0, 2, 2)": 2.022,
        "(0, 2, 3)": 0.793,
        "(0, 3, 0)": 2.86,
        "(0, 3, 1)": 0.904,
        "(0, 3, 2)": 0.793,
        "(0, 3, 3)": 2.022,
        "(1, 0, 0)": 2.86,
        "(1, 0, 1)": 2.022,
        "(1, 0, 2)": 1.279,
        "(1, 0, 3)": 0.904,
        "(1, 1, 0)": 2.022,
        "(1, 1, 1)": 3.302,
        "(1, 1, 2)": 1.168,
        "(1, 1, 3)": 1.725,
        "(1, 2, 0)": 1.279,
        "(1, 2, 1)": 1.168,
        "(1, 2, 2)": 0.953,
        "(1, 2, 3)": 0.764,
        "(1, 3, 0)": 0.904,
        "(1, 3, 1)": 1.725,
        "(1, 3, 2)": 0.764,
        "(1, 3, 3)": 1.312,
        "(2, 0, 0)": 2.86,
        "(2, 0, 1)": 1.279,
        "(2, 0, 2)": 2.022,
        "(2, 0, 3)": 0.793,
        "(2, 1, 0)": 1.279,
        "(2, 1, 1)": 1.168,
        "(2, 1, 2)": 0.953,
        "(2, 1, 3)": 0.764,
        "(2, 2, 0)": 2.022,
        "(2, 2, 1)": 0.953,
        "(2, 2, 2)": 3.302,
        "(2, 2, 3)": 0.694,
        "(2, 3, 0)": 0.793,
        "(2, 3, 1)": 0.764,
        "(2, 3, 2)": 0.694,
        "(2, 3, 3)": 0.61,
        "(3, 0, 0)": 2.86,
        "(3, 0, 1)": 0.904,
        "(3, 0, 2)": 0.793,
        "(3, 0, 3)": 2.022,
        "(3, 1, 0)": 0.904,
        "(3, 1, 1)": 1.725,
        "(3, 1, 2)": 0.764,
        "(3, 1, 3)": 1.312,
        "(3, 2, 0)": 0.793,
        "(3, 2, 1)": 0.764,
        "(3, 2, 2)": 0.694,
        "(3, 2, 3)": 0.61,
        "(3, 3, 0)": 2.022,
        "(3, 3, 1)": 1.312,
        "(3, 3, 2)": 0.61,
        "(3, 3, 3)": 3.302,
    },
    "La": {
        "(0, 0, 1)": 3.036,
        "(0, 0, 2)": 3.036,
        "(0, 0, 3)": 3.036,
        "(0, 1, 0)": 2.165,
        "(0, 1, 1)": 0.477,
        "(0, 1, 2)": 1.478,
        "(0, 1, 3)": 0.286,
        "(0, 2, 0)": 2.165,
        "(0, 2, 1)": 1.307,
        "(0, 2, 2)": 0.477,
        "(0, 2, 3)": 1.055,
        "(0, 3, 0)": 2.165,
        "(0, 3, 1)": 0.533,
        "(0, 3, 2)": 1.02,
        "(0, 3, 3)": 0.477,
        "(1, 0, 0)": 1.083,
        "(1, 0, 1)": 2.386,
        "(1, 0, 2)": 0.739,
        "(1, 0, 3)": 1.431,
        "(1, 1, 0)": 1.875,
        "(1, 1, 1)": 0.896,
        "(1, 1, 2)": 1.595,
        "(1, 1, 3)": 0.688,
        "(1, 2, 0)": 0.818,
        "(1, 2, 1)": 0.201,
        "(1, 2, 2)": 0.759,
        "(1, 2, 3)": 0.175,
        "(1, 3, 0)": 0.3,
        "(1, 3, 1)": 0.742,
        "(1, 3, 2)": 0.288,
        "(1, 3, 3)": 0.686,
        "(2, 0, 0)": 1.083,
        "(2, 0, 1)": 0.261,
        "(2, 0, 2)": 2.386,
        "(2, 0, 3)": 0.211,
        "(2, 1, 0)": 0.409,
        "(2, 1, 1)": 1.003,
        "(2, 1, 2)": 0.379,
        "(2, 1, 3)": 0.875,
        "(2, 2, 0)": 1.875,
        "(2, 2, 1)": 0.463,
        "(2, 2, 2)": 0.896,
        "(2, 2, 3)": 0.425,
        "(2, 3, 0)": 0.497,
        "(2, 3, 1)": 0.123,
        "(2, 3, 2)": 0.482,
        "(2, 3, 3)": 0.117,
        "(3, 0, 0)": 1.083,
        "(3, 0, 1)": 0.533,
        "(3, 0, 2)": 1.02,
        "(3, 0, 3)": 2.386,
        "(3, 1, 0)": 0.6,
        "(3, 1, 1)": 0.148,
        "(3, 1, 2)": 0.576,
        "(3, 1, 3)": 0.137,
        "(3, 2, 0)": 0.248,
        "(3, 2, 1)": 0.616,
        "(3, 2, 2)": 0.241,
        "(3, 2, 3)": 0.583,
        "(3, 3, 0)": 1.875,
        "(3, 3, 1)": 0.311,
        "(3, 3, 2)": 0.612,
        "(3, 3, 3)": 0.896,
    },
    "Li": {
        "(0, 0, 1)": 1.745,
        "(0, 0, 2)": 1.745,
        "(0, 0, 3)": 1.745,
        "(0, 1, 0)": 1.745,
        "(0, 1, 1)": 2.468,
        "(0, 1, 2)": 0.78,
        "(0, 1, 3)": 1.104,
        "(0, 2, 0)": 1.745,
        "(0, 2, 1)": 0.78,
        "(0, 2, 2)": 2.468,
        "(0, 2, 3)": 0.484,
        "(0, 3, 0)": 1.745,
        "(0, 3, 1)": 1.104,
        "(0, 3, 2)": 0.484,
        "(0, 3, 3)": 2.468,
        "(1, 0, 0)": 1.745,
        "(1, 0, 1)": 2.468,
        "(1, 0, 2)": 0.78,
        "(1, 0, 3)": 1.104,
        "(1, 1, 0)": 2.468,
        "(1, 1, 1)": 1.007,
        "(1, 1, 2)": 1.425,
        "(1, 1, 3)": 0.526,
        "(1, 2, 0)": 0.78,
        "(1, 2, 1)": 1.425,
        "(1, 2, 2)": 0.582,
        "(1, 2, 3)": 0.933,
        "(1, 3, 0)": 1.104,
        "(1, 3, 1)": 0.526,
        "(1, 3, 2)": 0.933,
        "(1, 3, 3)": 0.4,
        "(2, 0, 0)": 1.745,
        "(2, 0, 1)": 0.78,
        "(2, 0, 2)": 2.468,
        "(2, 0, 3)": 0.484,
        "(2, 1, 0)": 0.78,
        "(2, 1, 1)": 1.425,
        "(2, 1, 2)": 0.582,
        "(2, 1, 3)": 0.933,
        "(2, 2, 0)": 2.468,
        "(2, 2, 1)": 0.582,
        "(2, 2, 2)": 1.007,
        "(2, 2, 3)": 0.423,
        "(2, 3, 0)": 0.484,
        "(2, 3, 1)": 0.933,
        "(2, 3, 2)": 0.423,
        "(2, 3, 3)": 0.744,
        "(3, 0, 0)": 1.745,
        "(3, 0, 1)": 1.104,
        "(3, 0, 2)": 0.484,
        "(3, 0, 3)": 2.468,
        "(3, 1, 0)": 1.104,
        "(3, 1, 1)": 0.526,
        "(3, 1, 2)": 0.933,
        "(3, 1, 3)": 0.4,
        "(3, 2, 0)": 0.484,
        "(3, 2, 1)": 0.933,
        "(3, 2, 2)": 0.423,
        "(3, 2, 3)": 0.744,
        "(3, 3, 0)": 2.468,
        "(3, 3, 1)": 0.4,
        "(3, 3, 2)": 0.744,
        "(3, 3, 3)": 1.007,
    },
    "Lu": {
        "(0, 0, 1)": 2.782,
        "(0, 0, 2)": 2.782,
        "(0, 0, 3)": 2.782,
        "(0, 1, 0)": 2.026,
        "(0, 1, 1)": 0.445,
        "(0, 1, 2)": 1.368,
        "(0, 1, 3)": 0.264,
        "(0, 2, 0)": 2.026,
        "(0, 2, 1)": 1.222,
        "(0, 2, 2)": 0.445,
        "(0, 2, 3)": 0.98,
        "(0, 3, 0)": 2.026,
        "(0, 3, 1)": 0.498,
        "(0, 3, 2)": 0.952,
        "(0, 3, 3)": 0.445,
        "(1, 0, 0)": 1.013,
        "(1, 0, 1)": 2.223,
        "(1, 0, 2)": 0.684,
        "(1, 0, 3)": 1.319,
        "(1, 1, 0)": 1.755,
        "(1, 1, 1)": 0.837,
        "(1, 1, 2)": 1.484,
        "(1, 1, 3)": 0.637,
        "(1, 2, 0)": 0.766,
        "(1, 2, 1)": 0.188,
        "(1, 2, 2)": 0.708,
        "(1, 2, 3)": 0.163,
        "(1, 3, 0)": 0.281,
        "(1, 3, 1)": 0.695,
        "(1, 3, 2)": 0.269,
        "(1, 3, 3)": 0.64,
        "(2, 0, 0)": 1.013,
        "(2, 0, 1)": 0.244,
        "(2, 0, 2)": 2.223,
        "(2, 0, 3)": 0.196,
        "(2, 1, 0)": 0.383,
        "(2, 1, 1)": 0.938,
        "(2, 1, 2)": 0.354,
        "(2, 1, 3)": 0.814,
        "(2, 2, 0)": 1.755,
        "(2, 2, 1)": 0.433,
        "(2, 2, 2)": 0.837,
        "(2, 2, 3)": 0.397,
        "(2, 3, 0)": 0.465,
        "(2, 3, 1)": 0.115,
        "(2, 3, 2)": 0.451,
        "(2, 3, 3)": 0.109,
        "(3, 0, 0)": 1.013,
        "(3, 0, 1)": 0.498,
        "(3, 0, 2)": 0.952,
        "(3, 0, 3)": 2.223,
        "(3, 1, 0)": 0.562,
        "(3, 1, 1)": 0.139,
        "(3, 1, 2)": 0.538,
        "(3, 1, 3)": 0.128,
        "(3, 2, 0)": 0.232,
        "(3, 2, 1)": 0.577,
        "(3, 2, 2)": 0.225,
        "(3, 2, 3)": 0.544,
        "(3, 3, 0)": 1.755,
        "(3, 3, 1)": 0.291,
        "(3, 3, 2)": 0.572,
        "(3, 3, 3)": 0.837,
    },
    "Mg": {
        "(0, 0, 1)": 2.607,
        "(0, 0, 2)": 2.607,
        "(0, 0, 3)": 2.607,
        "(0, 1, 0)": 1.853,
        "(0, 1, 1)": 0.409,
        "(0, 1, 2)": 1.268,
        "(0, 1, 3)": 0.246,
        "(0, 2, 0)": 1.853,
        "(0, 2, 1)": 1.119,
        "(0, 2, 2)": 0.409,
        "(0, 2, 3)": 0.905,
        "(0, 3, 0)": 1.853,
        "(0, 3, 1)": 0.456,
        "(0, 3, 2)": 0.873,
        "(0, 3, 3)": 0.409,
        "(1, 0, 0)": 0.927,
        "(1, 0, 1)": 2.044,
        "(1, 0, 2)": 0.634,
        "(1, 0, 3)": 1.228,
        "(1, 1, 0)": 1.605,
        "(1, 1, 1)": 0.767,
        "(1, 1, 2)": 1.367,
        "(1, 1, 3)": 0.59,
        "(1, 2, 0)": 0.7,
        "(1, 2, 1)": 0.172,
        "(1, 2, 2)": 0.65,
        "(1, 2, 3)": 0.15,
        "(1, 3, 0)": 0.257,
        "(1, 3, 1)": 0.636,
        "(1, 3, 2)": 0.246,
        "(1, 3, 3)": 0.587,
        "(2, 0, 0)": 0.927,
        "(2, 0, 1)": 0.224,
        "(2, 0, 2)": 2.044,
        "(2, 0, 3)": 0.181,
        "(2, 1, 0)": 0.35,
        "(2, 1, 1)": 0.858,
        "(2, 1, 2)": 0.325,
        "(2, 1, 3)": 0.749,
        "(2, 2, 0)": 1.605,
        "(2, 2, 1)": 0.397,
        "(2, 2, 2)": 0.767,
        "(2, 2, 3)": 0.364,
        "(2, 3, 0)": 0.425,
        "(2, 3, 1)": 0.106,
        "(2, 3, 2)": 0.413,
        "(2, 3, 3)": 0.599,
        "(3, 0, 0)": 0.927,
        "(3, 0, 1)": 0.456,
        "(3, 0, 2)": 0.873,
        "(3, 0, 3)": 2.044,
        "(3, 1, 0)": 0.514,
        "(3, 1, 1)": 0.127,
        "(3, 1, 2)": 0.493,
        "(3, 1, 3)": 0.117,
        "(3, 2, 0)": 0.213,
        "(3, 2, 1)": 0.528,
        "(3, 2, 2)": 0.206,
        "(3, 2, 3)": 0.499,
        "(3, 3, 0)": 1.605,
        "(3, 3, 1)": 0.266,
        "(3, 3, 2)": 0.524,
        "(3, 3, 3)": 0.767,
    },
    "Mo": {
        "(0, 0, 1)": 1.575,
        "(0, 0, 2)": 1.575,
        "(0, 0, 3)": 1.575,
        "(0, 1, 0)": 1.575,
        "(0, 1, 1)": 2.227,
        "(0, 1, 2)": 0.704,
        "(0, 1, 3)": 0.996,
        "(0, 2, 0)": 1.575,
        "(0, 2, 1)": 0.704,
        "(0, 2, 2)": 2.227,
        "(0, 2, 3)": 0.437,
        "(0, 3, 0)": 1.575,
        "(0, 3, 1)": 0.996,
        "(0, 3, 2)": 0.437,
        "(0, 3, 3)": 2.227,
        "(1, 0, 0)": 1.575,
        "(1, 0, 1)": 2.227,
        "(1, 0, 2)": 0.704,
        "(1, 0, 3)": 0.996,
        "(1, 1, 0)": 2.227,
        "(1, 1, 1)": 0.909,
        "(1, 1, 2)": 1.286,
        "(1, 1, 3)": 0.475,
        "(1, 2, 0)": 0.704,
        "(1, 2, 1)": 1.286,
        "(1, 2, 2)": 0.525,
        "(1, 2, 3)": 0.842,
        "(1, 3, 0)": 0.996,
        "(1, 3, 1)": 0.475,
        "(1, 3, 2)": 0.842,
        "(1, 3, 3)": 0.361,
        "(2, 0, 0)": 1.575,
        "(2, 0, 1)": 0.704,
        "(2, 0, 2)": 2.227,
        "(2, 0, 3)": 0.437,
        "(2, 1, 0)": 0.704,
        "(2, 1, 1)": 1.286,
        "(2, 1, 2)": 0.525,
        "(2, 1, 3)": 0.842,
        "(2, 2, 0)": 2.227,
        "(2, 2, 1)": 0.525,
        "(2, 2, 2)": 0.909,
        "(2, 2, 3)": 0.382,
        "(2, 3, 0)": 0.437,
        "(2, 3, 1)": 0.842,
        "(2, 3, 2)": 0.382,
        "(2, 3, 3)": 0.672,
        "(3, 0, 0)": 1.575,
        "(3, 0, 1)": 0.996,
        "(3, 0, 2)": 0.437,
        "(3, 0, 3)": 2.227,
        "(3, 1, 0)": 0.996,
        "(3, 1, 1)": 0.475,
        "(3, 1, 2)": 0.842,
        "(3, 1, 3)": 0.361,
        "(3, 2, 0)": 0.437,
        "(3, 2, 1)": 0.842,
        "(3, 2, 2)": 0.382,
        "(3, 2, 3)": 0.672,
        "(3, 3, 0)": 2.227,
        "(3, 3, 1)": 0.361,
        "(3, 3, 2)": 0.672,
        "(3, 3, 3)": 0.909,
    },
    "Na": {
        "(0, 0, 1)": 2.115,
        "(0, 0, 2)": 2.115,
        "(0, 0, 3)": 2.115,
        "(0, 1, 0)": 2.115,
        "(0, 1, 1)": 2.991,
        "(0, 1, 2)": 0.946,
        "(0, 1, 3)": 1.338,
        "(0, 2, 0)": 2.115,
        "(0, 2, 1)": 0.946,
        "(0, 2, 2)": 2.991,
        "(0, 2, 3)": 0.587,
        "(0, 3, 0)": 2.115,
        "(0, 3, 1)": 1.338,
        "(0, 3, 2)": 0.587,
        "(0, 3, 3)": 2.991,
        "(1, 0, 0)": 2.115,
        "(1, 0, 1)": 2.991,
        "(1, 0, 2)": 0.946,
        "(1, 0, 3)": 1.338,
        "(1, 1, 0)": 2.991,
        "(1, 1, 1)": 1.221,
        "(1, 1, 2)": 1.727,
        "(1, 1, 3)": 0.638,
        "(1, 2, 0)": 0.946,
        "(1, 2, 1)": 1.727,
        "(1, 2, 2)": 0.705,
        "(1, 2, 3)": 1.131,
        "(1, 3, 0)": 1.338,
        "(1, 3, 1)": 0.638,
        "(1, 3, 2)": 1.131,
        "(1, 3, 3)": 0.485,
        "(2, 0, 0)": 2.115,
        "(2, 0, 1)": 0.946,
        "(2, 0, 2)": 2.991,
        "(2, 0, 3)": 0.587,
        "(2, 1, 0)": 0.946,
        "(2, 1, 1)": 1.727,
        "(2, 1, 2)": 0.705,
        "(2, 1, 3)": 1.131,
        "(2, 2, 0)": 2.991,
        "(2, 2, 1)": 0.705,
        "(2, 2, 2)": 1.221,
        "(2, 2, 3)": 0.513,
        "(2, 3, 0)": 0.587,
        "(2, 3, 1)": 1.131,
        "(2, 3, 2)": 0.513,
        "(2, 3, 3)": 0.902,
        "(3, 0, 0)": 2.115,
        "(3, 0, 1)": 1.338,
        "(3, 0, 2)": 0.587,
        "(3, 0, 3)": 2.991,
        "(3, 1, 0)": 1.338,
        "(3, 1, 1)": 0.638,
        "(3, 1, 2)": 1.131,
        "(3, 1, 3)": 0.485,
        "(3, 2, 0)": 0.587,
        "(3, 2, 1)": 1.131,
        "(3, 2, 2)": 0.513,
        "(3, 2, 3)": 0.902,
        "(3, 3, 0)": 2.991,
        "(3, 3, 1)": 0.485,
        "(3, 3, 2)": 0.902,
        "(3, 3, 3)": 1.221,
    },
    "Nb": {
        "(0, 0, 1)": 1.65,
        "(0, 0, 2)": 1.65,
        "(0, 0, 3)": 1.65,
        "(0, 1, 0)": 1.65,
        "(0, 1, 1)": 2.333,
        "(0, 1, 2)": 0.738,
        "(0, 1, 3)": 1.044,
        "(0, 2, 0)": 1.65,
        "(0, 2, 1)": 0.738,
        "(0, 2, 2)": 2.333,
        "(0, 2, 3)": 0.458,
        "(0, 3, 0)": 1.65,
        "(0, 3, 1)": 1.044,
        "(0, 3, 2)": 0.458,
        "(0, 3, 3)": 2.333,
        "(1, 0, 0)": 1.65,
        "(1, 0, 1)": 2.333,
        "(1, 0, 2)": 0.738,
        "(1, 0, 3)": 1.044,
        "(1, 1, 0)": 2.333,
        "(1, 1, 1)": 0.953,
        "(1, 1, 2)": 1.347,
        "(1, 1, 3)": 0.497,
        "(1, 2, 0)": 0.738,
        "(1, 2, 1)": 1.347,
        "(1, 2, 2)": 0.55,
        "(1, 2, 3)": 0.882,
        "(1, 3, 0)": 1.044,
        "(1, 3, 1)": 0.497,
        "(1, 3, 2)": 0.882,
        "(1, 3, 3)": 0.379,
        "(2, 0, 0)": 1.65,
        "(2, 0, 1)": 0.738,
        "(2, 0, 2)": 2.333,
        "(2, 0, 3)": 0.458,
        "(2, 1, 0)": 0.738,
        "(2, 1, 1)": 1.347,
        "(2, 1, 2)": 0.55,
        "(2, 1, 3)": 0.882,
        "(2, 2, 0)": 2.333,
        "(2, 2, 1)": 0.55,
        "(2, 2, 2)": 0.953,
        "(2, 2, 3)": 0.4,
        "(2, 3, 0)": 0.458,
        "(2, 3, 1)": 0.882,
        "(2, 3, 2)": 0.4,
        "(2, 3, 3)": 0.704,
        "(3, 0, 0)": 1.65,
        "(3, 0, 1)": 1.044,
        "(3, 0, 2)": 0.458,
        "(3, 0, 3)": 2.333,
        "(3, 1, 0)": 1.044,
        "(3, 1, 1)": 0.497,
        "(3, 1, 2)": 0.882,
        "(3, 1, 3)": 0.379,
        "(3, 2, 0)": 0.458,
        "(3, 2, 1)": 0.882,
        "(3, 2, 2)": 0.4,
        "(3, 2, 3)": 0.704,
        "(3, 3, 0)": 2.333,
        "(3, 3, 1)": 0.379,
        "(3, 3, 2)": 0.704,
        "(3, 3, 3)": 0.953,
    },
    "Nd": {
        "(0, 0, 1)": 2.954,
        "(0, 0, 2)": 2.954,
        "(0, 0, 3)": 2.954,
        "(0, 1, 0)": 2.113,
        "(0, 1, 1)": 0.465,
        "(0, 1, 2)": 1.441,
        "(0, 1, 3)": 0.279,
        "(0, 2, 0)": 2.113,
        "(0, 2, 1)": 1.276,
        "(0, 2, 2)": 0.465,
        "(0, 2, 3)": 1.029,
        "(0, 3, 0)": 2.113,
        "(0, 3, 1)": 0.52,
        "(0, 3, 2)": 0.995,
        "(0, 3, 3)": 0.465,
        "(1, 0, 0)": 1.057,
        "(1, 0, 1)": 2.327,
        "(1, 0, 2)": 0.72,
        "(1, 0, 3)": 1.394,
        "(1, 1, 0)": 1.83,
        "(1, 1, 1)": 0.874,
        "(1, 1, 2)": 1.556,
        "(1, 1, 3)": 0.67,
        "(1, 2, 0)": 0.799,
        "(1, 2, 1)": 0.196,
        "(1, 2, 2)": 0.74,
        "(1, 2, 3)": 0.171,
        "(1, 3, 0)": 0.293,
        "(1, 3, 1)": 0.725,
        "(1, 3, 2)": 0.281,
        "(1, 3, 3)": 0.669,
        "(2, 0, 0)": 1.057,
        "(2, 0, 1)": 0.255,
        "(2, 0, 2)": 2.327,
        "(2, 0, 3)": 0.206,
        "(2, 1, 0)": 0.399,
        "(2, 1, 1)": 0.978,
        "(2, 1, 2)": 0.37,
        "(2, 1, 3)": 0.853,
        "(2, 2, 0)": 1.83,
        "(2, 2, 1)": 0.452,
        "(2, 2, 2)": 0.874,
        "(2, 2, 3)": 0.415,
        "(2, 3, 0)": 0.485,
        "(2, 3, 1)": 0.12,
        "(2, 3, 2)": 0.471,
        "(2, 3, 3)": 0.114,
        "(3, 0, 0)": 1.057,
        "(3, 0, 1)": 0.52,
        "(3, 0, 2)": 0.995,
        "(3, 0, 3)": 2.327,
        "(3, 1, 0)": 0.586,
        "(3, 1, 1)": 0.145,
        "(3, 1, 2)": 0.562,
        "(3, 1, 3)": 0.134,
        "(3, 2, 0)": 0.242,
        "(3, 2, 1)": 0.601,
        "(3, 2, 2)": 0.235,
        "(3, 2, 3)": 0.568,
        "(3, 3, 0)": 1.83,
        "(3, 3, 1)": 0.303,
        "(3, 3, 2)": 0.597,
        "(3, 3, 3)": 0.874,
    },
    "Ne": {
        "(0, 0, 1)": 2.215,
        "(0, 0, 2)": 2.215,
        "(0, 0, 3)": 2.215,
        "(0, 1, 0)": 2.215,
        "(0, 1, 1)": 1.566,
        "(0, 1, 2)": 0.991,
        "(0, 1, 3)": 0.7,
        "(0, 2, 0)": 2.215,
        "(0, 2, 1)": 0.991,
        "(0, 2, 2)": 1.566,
        "(0, 2, 3)": 0.614,
        "(0, 3, 0)": 2.215,
        "(0, 3, 1)": 0.7,
        "(0, 3, 2)": 0.614,
        "(0, 3, 3)": 1.566,
        "(1, 0, 0)": 2.215,
        "(1, 0, 1)": 1.566,
        "(1, 0, 2)": 0.991,
        "(1, 0, 3)": 0.7,
        "(1, 1, 0)": 1.566,
        "(1, 1, 1)": 2.558,
        "(1, 1, 2)": 0.904,
        "(1, 1, 3)": 1.336,
        "(1, 2, 0)": 0.991,
        "(1, 2, 1)": 0.904,
        "(1, 2, 2)": 0.738,
        "(1, 2, 3)": 0.592,
        "(1, 3, 0)": 0.7,
        "(1, 3, 1)": 1.336,
        "(1, 3, 2)": 0.592,
        "(1, 3, 3)": 1.016,
        "(2, 0, 0)": 2.215,
        "(2, 0, 1)": 0.991,
        "(2, 0, 2)": 1.566,
        "(2, 0, 3)": 0.614,
        "(2, 1, 0)": 0.991,
        "(2, 1, 1)": 0.904,
        "(2, 1, 2)": 0.738,
        "(2, 1, 3)": 0.592,
        "(2, 2, 0)": 1.566,
        "(2, 2, 1)": 0.738,
        "(2, 2, 2)": 2.558,
        "(2, 2, 3)": 0.537,
        "(2, 3, 0)": 0.614,
        "(2, 3, 1)": 0.592,
        "(2, 3, 2)": 0.537,
        "(2, 3, 3)": 0.472,
        "(3, 0, 0)": 2.215,
        "(3, 0, 1)": 0.7,
        "(3, 0, 2)": 0.614,
        "(3, 0, 3)": 1.566,
        "(3, 1, 0)": 0.7,
        "(3, 1, 1)": 1.336,
        "(3, 1, 2)": 0.592,
        "(3, 1, 3)": 1.016,
        "(3, 2, 0)": 0.614,
        "(3, 2, 1)": 0.592,
        "(3, 2, 2)": 0.537,
        "(3, 2, 3)": 0.472,
        "(3, 3, 0)": 1.566,
        "(3, 3, 1)": 1.016,
        "(3, 3, 2)": 0.472,
        "(3, 3, 3)": 2.558,
    },
    "Ni": {
        "(0, 0, 1)": 1.76,
        "(0, 0, 2)": 1.76,
        "(0, 0, 3)": 1.76,
        "(0, 1, 0)": 1.76,
        "(0, 1, 1)": 1.245,
        "(0, 1, 2)": 0.787,
        "(0, 1, 3)": 0.557,
        "(0, 2, 0)": 1.76,
        "(0, 2, 1)": 0.787,
        "(0, 2, 2)": 1.245,
        "(0, 2, 3)": 0.488,
        "(0, 3, 0)": 1.76,
        "(0, 3, 1)": 0.557,
        "(0, 3, 2)": 0.488,
        "(0, 3, 3)": 1.245,
        "(1, 0, 0)": 1.76,
        "(1, 0, 1)": 1.245,
        "(1, 0, 2)": 0.787,
        "(1, 0, 3)": 0.557,
        "(1, 1, 0)": 1.245,
        "(1, 1, 1)": 2.032,
        "(1, 1, 2)": 0.719,
        "(1, 1, 3)": 1.061,
        "(1, 2, 0)": 0.787,
        "(1, 2, 1)": 0.719,
        "(1, 2, 2)": 0.587,
        "(1, 2, 3)": 0.47,
        "(1, 3, 0)": 0.557,
        "(1, 3, 1)": 1.061,
        "(1, 3, 2)": 0.47,
        "(1, 3, 3)": 0.808,
        "(2, 0, 0)": 1.76,
        "(2, 0, 1)": 0.787,
        "(2, 0, 2)": 1.245,
        "(2, 0, 3)": 0.488,
        "(2, 1, 0)": 0.787,
        "(2, 1, 1)": 0.719,
        "(2, 1, 2)": 0.587,
        "(2, 1, 3)": 0.47,
        "(2, 2, 0)": 1.245,
        "(2, 2, 1)": 0.587,
        "(2, 2, 2)": 2.032,
        "(2, 2, 3)": 0.427,
        "(2, 3, 0)": 0.488,
        "(2, 3, 1)": 0.47,
        "(2, 3, 2)": 0.427,
        "(2, 3, 3)": 0.375,
        "(3, 0, 0)": 1.76,
        "(3, 0, 1)": 0.557,
        "(3, 0, 2)": 0.488,
        "(3, 0, 3)": 1.245,
        "(3, 1, 0)": 0.557,
        "(3, 1, 1)": 1.061,
        "(3, 1, 2)": 0.47,
        "(3, 1, 3)": 0.808,
        "(3, 2, 0)": 0.488,
        "(3, 2, 1)": 0.47,
        "(3, 2, 2)": 0.427,
        "(3, 2, 3)": 0.375,
        "(3, 3, 0)": 1.245,
        "(3, 3, 1)": 0.808,
        "(3, 3, 2)": 0.375,
        "(3, 3, 3)": 2.032,
    },
    "Np": {
        "(0, 0, 1)": 6.66,
        "(0, 0, 2)": 6.66,
        "(0, 0, 3)": 6.66,
        "(0, 1, 0)": 4.885,
        "(0, 1, 1)": 3.939,
        "(0, 1, 2)": 2.752,
        "(0, 1, 3)": 2.021,
        "(0, 2, 0)": 4.885,
        "(0, 2, 1)": 2.293,
        "(0, 2, 2)": 3.939,
        "(0, 2, 3)": 1.643,
        "(0, 3, 0)": 4.885,
        "(0, 3, 1)": 1.582,
        "(0, 3, 2)": 1.463,
        "(0, 3, 3)": 3.939,
        "(1, 0, 0)": 4.72,
        "(1, 0, 1)": 3.851,
        "(1, 0, 2)": 2.721,
        "(1, 0, 3)": 2.009,
        "(1, 1, 0)": 3.394,
        "(1, 1, 1)": 3.024,
        "(1, 1, 2)": 2.377,
        "(1, 1, 3)": 1.858,
        "(1, 2, 0)": 2.169,
        "(1, 2, 1)": 2.063,
        "(1, 2, 2)": 1.818,
        "(1, 2, 3)": 1.552,
        "(1, 3, 0)": 1.539,
        "(1, 3, 1)": 1.5,
        "(1, 3, 2)": 1.397,
        "(1, 3, 3)": 1.265,
        "(2, 0, 0)": 4.72,
        "(2, 0, 1)": 2.224,
        "(2, 0, 2)": 3.851,
        "(2, 0, 3)": 1.617,
        "(2, 1, 0)": 2.125,
        "(2, 1, 1)": 2.024,
        "(2, 1, 2)": 1.791,
        "(2, 1, 3)": 1.535,
        "(2, 2, 0)": 3.394,
        "(2, 2, 1)": 1.645,
        "(2, 2, 2)": 3.024,
        "(2, 2, 3)": 1.348,
        "(2, 3, 0)": 1.34,
        "(2, 3, 1)": 1.314,
        "(2, 3, 2)": 1.243,
        "(2, 3, 3)": 1.147,
        "(3, 0, 0)": 4.72,
        "(3, 0, 1)": 1.531,
        "(3, 0, 2)": 1.423,
        "(3, 0, 3)": 3.851,
        "(3, 1, 0)": 1.498,
        "(3, 1, 1)": 1.461,
        "(3, 1, 2)": 1.366,
        "(3, 1, 3)": 1.242,
        "(3, 2, 0)": 1.323,
        "(3, 2, 1)": 1.297,
        "(3, 2, 2)": 1.229,
        "(3, 2, 3)": 1.136,
        "(3, 3, 0)": 3.394,
        "(3, 3, 1)": 1.115,
        "(3, 3, 2)": 1.071,
        "(3, 3, 3)": 3.024,
    },
    "Os": {
        "(0, 0, 1)": 2.163,
        "(0, 0, 2)": 2.163,
        "(0, 0, 3)": 2.163,
        "(0, 1, 0)": 1.582,
        "(0, 1, 1)": 0.347,
        "(0, 1, 2)": 1.066,
        "(0, 1, 3)": 0.205,
        "(0, 2, 0)": 1.582,
        "(0, 2, 1)": 0.954,
        "(0, 2, 2)": 0.347,
        "(0, 2, 3)": 0.764,
        "(0, 3, 0)": 1.582,
        "(0, 3, 1)": 0.389,
        "(0, 3, 2)": 0.743,
        "(0, 3, 3)": 0.347,
        "(1, 0, 0)": 0.791,
        "(1, 0, 1)": 1.734,
        "(1, 0, 2)": 0.533,
        "(1, 0, 3)": 1.027,
        "(1, 1, 0)": 1.37,
        "(1, 1, 1)": 0.653,
        "(1, 1, 2)": 1.157,
        "(1, 1, 3)": 0.497,
        "(1, 2, 0)": 0.598,
        "(1, 2, 1)": 0.146,
        "(1, 2, 2)": 0.552,
        "(1, 2, 3)": 0.127,
        "(1, 3, 0)": 0.219,
        "(1, 3, 1)": 0.542,
        "(1, 3, 2)": 0.21,
        "(1, 3, 3)": 0.499,
        "(2, 0, 0)": 0.791,
        "(2, 0, 1)": 0.191,
        "(2, 0, 2)": 1.734,
        "(2, 0, 3)": 0.153,
        "(2, 1, 0)": 0.299,
        "(2, 1, 1)": 0.732,
        "(2, 1, 2)": 0.276,
        "(2, 1, 3)": 0.635,
        "(2, 2, 0)": 1.37,
        "(2, 2, 1)": 0.338,
        "(2, 2, 2)": 0.653,
        "(2, 2, 3)": 0.309,
        "(2, 3, 0)": 0.363,
        "(2, 3, 1)": 0.54,
        "(2, 3, 2)": 0.352,
        "(2, 3, 3)": 0.509,
        "(3, 0, 0)": 0.791,
        "(3, 0, 1)": 0.389,
        "(3, 0, 2)": 0.743,
        "(3, 0, 3)": 1.734,
        "(3, 1, 0)": 0.439,
        "(3, 1, 1)": 0.108,
        "(3, 1, 2)": 0.42,
        "(3, 1, 3)": 0.599,
        "(3, 2, 0)": 0.181,
        "(3, 2, 1)": 0.45,
        "(3, 2, 2)": 0.176,
        "(3, 2, 3)": 0.424,
        "(3, 3, 0)": 1.37,
        "(3, 3, 1)": 0.227,
        "(3, 3, 2)": 0.447,
        "(3, 3, 3)": 0.653,
    },
    "Pb": {
        "(0, 0, 1)": 2.475,
        "(0, 0, 2)": 2.475,
        "(0, 0, 3)": 2.475,
        "(0, 1, 0)": 2.475,
        "(0, 1, 1)": 1.75,
        "(0, 1, 2)": 1.107,
        "(0, 1, 3)": 0.783,
        "(0, 2, 0)": 2.475,
        "(0, 2, 1)": 1.107,
        "(0, 2, 2)": 1.75,
        "(0, 2, 3)": 0.686,
        "(0, 3, 0)": 2.475,
        "(0, 3, 1)": 0.783,
        "(0, 3, 2)": 0.686,
        "(0, 3, 3)": 1.75,
        "(1, 0, 0)": 2.475,
        "(1, 0, 1)": 1.75,
        "(1, 0, 2)": 1.107,
        "(1, 0, 3)": 0.783,
        "(1, 1, 0)": 1.75,
        "(1, 1, 1)": 2.858,
        "(1, 1, 2)": 1.01,
        "(1, 1, 3)": 1.492,
        "(1, 2, 0)": 1.107,
        "(1, 2, 1)": 1.01,
        "(1, 2, 2)": 0.825,
        "(1, 2, 3)": 0.661,
        "(1, 3, 0)": 0.783,
        "(1, 3, 1)": 1.492,
        "(1, 3, 2)": 0.661,
        "(1, 3, 3)": 1.136,
        "(2, 0, 0)": 2.475,
        "(2, 0, 1)": 1.107,
        "(2, 0, 2)": 1.75,
        "(2, 0, 3)": 0.686,
        "(2, 1, 0)": 1.107,
        "(2, 1, 1)": 1.01,
        "(2, 1, 2)": 0.825,
        "(2, 1, 3)": 0.661,
        "(2, 2, 0)": 1.75,
        "(2, 2, 1)": 0.825,
        "(2, 2, 2)": 2.858,
        "(2, 2, 3)": 0.6,
        "(2, 3, 0)": 0.686,
        "(2, 3, 1)": 0.661,
        "(2, 3, 2)": 0.6,
        "(2, 3, 3)": 0.528,
        "(3, 0, 0)": 2.475,
        "(3, 0, 1)": 0.783,
        "(3, 0, 2)": 0.686,
        "(3, 0, 3)": 1.75,
        "(3, 1, 0)": 0.783,
        "(3, 1, 1)": 1.492,
        "(3, 1, 2)": 0.661,
        "(3, 1, 3)": 1.136,
        "(3, 2, 0)": 0.686,
        "(3, 2, 1)": 0.661,
        "(3, 2, 2)": 0.6,
        "(3, 2, 3)": 0.528,
        "(3, 3, 0)": 1.75,
        "(3, 3, 1)": 1.136,
        "(3, 3, 2)": 0.528,
        "(3, 3, 3)": 2.858,
    },
    "Pd": {
        "(0, 0, 1)": 1.945,
        "(0, 0, 2)": 1.945,
        "(0, 0, 3)": 1.945,
        "(0, 1, 0)": 1.945,
        "(0, 1, 1)": 1.375,
        "(0, 1, 2)": 0.87,
        "(0, 1, 3)": 0.615,
        "(0, 2, 0)": 1.945,
        "(0, 2, 1)": 0.87,
        "(0, 2, 2)": 1.375,
        "(0, 2, 3)": 0.539,
        "(0, 3, 0)": 1.945,
        "(0, 3, 1)": 0.615,
        "(0, 3, 2)": 0.539,
        "(0, 3, 3)": 1.375,
        "(1, 0, 0)": 1.945,
        "(1, 0, 1)": 1.375,
        "(1, 0, 2)": 0.87,
        "(1, 0, 3)": 0.615,
        "(1, 1, 0)": 1.375,
        "(1, 1, 1)": 2.246,
        "(1, 1, 2)": 0.794,
        "(1, 1, 3)": 1.173,
        "(1, 2, 0)": 0.87,
        "(1, 2, 1)": 0.794,
        "(1, 2, 2)": 0.648,
        "(1, 2, 3)": 0.52,
        "(1, 3, 0)": 0.615,
        "(1, 3, 1)": 1.173,
        "(1, 3, 2)": 0.52,
        "(1, 3, 3)": 0.892,
        "(2, 0, 0)": 1.945,
        "(2, 0, 1)": 0.87,
        "(2, 0, 2)": 1.375,
        "(2, 0, 3)": 0.539,
        "(2, 1, 0)": 0.87,
        "(2, 1, 1)": 0.794,
        "(2, 1, 2)": 0.648,
        "(2, 1, 3)": 0.52,
        "(2, 2, 0)": 1.375,
        "(2, 2, 1)": 0.648,
        "(2, 2, 2)": 2.246,
        "(2, 2, 3)": 0.472,
        "(2, 3, 0)": 0.539,
        "(2, 3, 1)": 0.52,
        "(2, 3, 2)": 0.472,
        "(2, 3, 3)": 0.415,
        "(3, 0, 0)": 1.945,
        "(3, 0, 1)": 0.615,
        "(3, 0, 2)": 0.539,
        "(3, 0, 3)": 1.375,
        "(3, 1, 0)": 0.615,
        "(3, 1, 1)": 1.173,
        "(3, 1, 2)": 0.52,
        "(3, 1, 3)": 0.892,
        "(3, 2, 0)": 0.539,
        "(3, 2, 1)": 0.52,
        "(3, 2, 2)": 0.472,
        "(3, 2, 3)": 0.415,
        "(3, 3, 0)": 1.375,
        "(3, 3, 1)": 0.892,
        "(3, 3, 2)": 0.415,
        "(3, 3, 3)": 2.246,
    },
    "Po": {
        "(0, 0, 1)": 3.35,
        "(0, 0, 2)": 3.35,
        "(0, 0, 3)": 3.35,
        "(0, 1, 0)": 3.35,
        "(0, 1, 1)": 2.369,
        "(0, 1, 2)": 1.498,
        "(0, 1, 3)": 1.059,
        "(0, 2, 0)": 3.35,
        "(0, 2, 1)": 1.498,
        "(0, 2, 2)": 2.369,
        "(0, 2, 3)": 0.929,
        "(0, 3, 0)": 3.35,
        "(0, 3, 1)": 1.059,
        "(0, 3, 2)": 0.929,
        "(0, 3, 3)": 2.369,
        "(1, 0, 0)": 3.35,
        "(1, 0, 1)": 2.369,
        "(1, 0, 2)": 1.498,
        "(1, 0, 3)": 1.059,
        "(1, 1, 0)": 2.369,
        "(1, 1, 1)": 1.934,
        "(1, 1, 2)": 1.368,
        "(1, 1, 3)": 1.01,
        "(1, 2, 0)": 1.498,
        "(1, 2, 1)": 1.368,
        "(1, 2, 2)": 1.117,
        "(1, 2, 3)": 0.895,
        "(1, 3, 0)": 1.059,
        "(1, 3, 1)": 1.01,
        "(1, 3, 2)": 0.895,
        "(1, 3, 3)": 0.769,
        "(2, 0, 0)": 3.35,
        "(2, 0, 1)": 1.498,
        "(2, 0, 2)": 2.369,
        "(2, 0, 3)": 0.929,
        "(2, 1, 0)": 1.498,
        "(2, 1, 1)": 1.368,
        "(2, 1, 2)": 1.117,
        "(2, 1, 3)": 0.895,
        "(2, 2, 0)": 2.369,
        "(2, 2, 1)": 1.117,
        "(2, 2, 2)": 1.934,
        "(2, 2, 3)": 0.812,
        "(2, 3, 0)": 0.929,
        "(2, 3, 1)": 0.895,
        "(2, 3, 2)": 0.812,
        "(2, 3, 3)": 0.714,
        "(3, 0, 0)": 3.35,
        "(3, 0, 1)": 1.059,
        "(3, 0, 2)": 0.929,
        "(3, 0, 3)": 2.369,
        "(3, 1, 0)": 1.059,
        "(3, 1, 1)": 1.01,
        "(3, 1, 2)": 0.895,
        "(3, 1, 3)": 0.769,
        "(3, 2, 0)": 0.929,
        "(3, 2, 1)": 0.895,
        "(3, 2, 2)": 0.812,
        "(3, 2, 3)": 0.714,
        "(3, 3, 0)": 2.369,
        "(3, 3, 1)": 0.769,
        "(3, 3, 2)": 0.714,
        "(3, 3, 3)": 1.934,
    },
    "Pr": {
        "(0, 0, 1)": 2.962,
        "(0, 0, 2)": 2.962,
        "(0, 0, 3)": 2.962,
        "(0, 1, 0)": 2.119,
        "(0, 1, 1)": 0.467,
        "(0, 1, 2)": 1.445,
        "(0, 1, 3)": 0.28,
        "(0, 2, 0)": 2.119,
        "(0, 2, 1)": 1.279,
        "(0, 2, 2)": 0.467,
        "(0, 2, 3)": 1.032,
        "(0, 3, 0)": 2.119,
        "(0, 3, 1)": 0.521,
        "(0, 3, 2)": 0.998,
        "(0, 3, 3)": 0.467,
        "(1, 0, 0)": 1.059,
        "(1, 0, 1)": 2.334,
        "(1, 0, 2)": 0.722,
        "(1, 0, 3)": 1.398,
        "(1, 1, 0)": 1.835,
        "(1, 1, 1)": 0.876,
        "(1, 1, 2)": 1.56,
        "(1, 1, 3)": 0.672,
        "(1, 2, 0)": 0.801,
        "(1, 2, 1)": 0.196,
        "(1, 2, 2)": 0.742,
        "(1, 2, 3)": 0.171,
        "(1, 3, 0)": 0.294,
        "(1, 3, 1)": 0.727,
        "(1, 3, 2)": 0.282,
        "(1, 3, 3)": 0.671,
        "(2, 0, 0)": 1.059,
        "(2, 0, 1)": 0.256,
        "(2, 0, 2)": 2.334,
        "(2, 0, 3)": 0.206,
        "(2, 1, 0)": 0.4,
        "(2, 1, 1)": 0.981,
        "(2, 1, 2)": 0.371,
        "(2, 1, 3)": 0.855,
        "(2, 2, 0)": 1.835,
        "(2, 2, 1)": 0.453,
        "(2, 2, 2)": 0.876,
        "(2, 2, 3)": 0.416,
        "(2, 3, 0)": 0.486,
        "(2, 3, 1)": 0.121,
        "(2, 3, 2)": 0.472,
        "(2, 3, 3)": 0.114,
        "(3, 0, 0)": 1.059,
        "(3, 0, 1)": 0.521,
        "(3, 0, 2)": 0.998,
        "(3, 0, 3)": 2.334,
        "(3, 1, 0)": 0.588,
        "(3, 1, 1)": 0.145,
        "(3, 1, 2)": 0.563,
        "(3, 1, 3)": 0.134,
        "(3, 2, 0)": 0.243,
        "(3, 2, 1)": 0.603,
        "(3, 2, 2)": 0.236,
        "(3, 2, 3)": 0.57,
        "(3, 3, 0)": 1.835,
        "(3, 3, 1)": 0.304,
        "(3, 3, 2)": 0.599,
        "(3, 3, 3)": 0.876,
    },
    "Pt": {
        "(0, 0, 1)": 1.96,
        "(0, 0, 2)": 1.96,
        "(0, 0, 3)": 1.96,
        "(0, 1, 0)": 1.96,
        "(0, 1, 1)": 1.386,
        "(0, 1, 2)": 0.877,
        "(0, 1, 3)": 0.62,
        "(0, 2, 0)": 1.96,
        "(0, 2, 1)": 0.877,
        "(0, 2, 2)": 1.386,
        "(0, 2, 3)": 0.544,
        "(0, 3, 0)": 1.96,
        "(0, 3, 1)": 0.62,
        "(0, 3, 2)": 0.544,
        "(0, 3, 3)": 1.386,
        "(1, 0, 0)": 1.96,
        "(1, 0, 1)": 1.386,
        "(1, 0, 2)": 0.877,
        "(1, 0, 3)": 0.62,
        "(1, 1, 0)": 1.386,
        "(1, 1, 1)": 2.263,
        "(1, 1, 2)": 0.8,
        "(1, 1, 3)": 1.182,
        "(1, 2, 0)": 0.877,
        "(1, 2, 1)": 0.8,
        "(1, 2, 2)": 0.653,
        "(1, 2, 3)": 0.524,
        "(1, 3, 0)": 0.62,
        "(1, 3, 1)": 1.182,
        "(1, 3, 2)": 0.524,
        "(1, 3, 3)": 0.899,
        "(2, 0, 0)": 1.96,
        "(2, 0, 1)": 0.877,
        "(2, 0, 2)": 1.386,
        "(2, 0, 3)": 0.544,
        "(2, 1, 0)": 0.877,
        "(2, 1, 1)": 0.8,
        "(2, 1, 2)": 0.653,
        "(2, 1, 3)": 0.524,
        "(2, 2, 0)": 1.386,
        "(2, 2, 1)": 0.653,
        "(2, 2, 2)": 2.263,
        "(2, 2, 3)": 0.475,
        "(2, 3, 0)": 0.544,
        "(2, 3, 1)": 0.524,
        "(2, 3, 2)": 0.475,
        "(2, 3, 3)": 0.418,
        "(3, 0, 0)": 1.96,
        "(3, 0, 1)": 0.62,
        "(3, 0, 2)": 0.544,
        "(3, 0, 3)": 1.386,
        "(3, 1, 0)": 0.62,
        "(3, 1, 1)": 1.182,
        "(3, 1, 2)": 0.524,
        "(3, 1, 3)": 0.899,
        "(3, 2, 0)": 0.544,
        "(3, 2, 1)": 0.524,
        "(3, 2, 2)": 0.475,
        "(3, 2, 3)": 0.418,
        "(3, 3, 0)": 1.386,
        "(3, 3, 1)": 0.899,
        "(3, 3, 2)": 0.418,
        "(3, 3, 3)": 2.263,
    },
    "Rb": {
        "(0, 0, 1)": 2.795,
        "(0, 0, 2)": 2.795,
        "(0, 0, 3)": 2.795,
        "(0, 1, 0)": 2.795,
        "(0, 1, 1)": 3.953,
        "(0, 1, 2)": 1.25,
        "(0, 1, 3)": 1.768,
        "(0, 2, 0)": 2.795,
        "(0, 2, 1)": 1.25,
        "(0, 2, 2)": 3.953,
        "(0, 2, 3)": 0.775,
        "(0, 3, 0)": 2.795,
        "(0, 3, 1)": 1.768,
        "(0, 3, 2)": 0.775,
        "(0, 3, 3)": 3.953,
        "(1, 0, 0)": 2.795,
        "(1, 0, 1)": 3.953,
        "(1, 0, 2)": 1.25,
        "(1, 0, 3)": 1.768,
        "(1, 1, 0)": 3.953,
        "(1, 1, 1)": 1.614,
        "(1, 1, 2)": 2.282,
        "(1, 1, 3)": 0.843,
        "(1, 2, 0)": 1.25,
        "(1, 2, 1)": 2.282,
        "(1, 2, 2)": 0.932,
        "(1, 2, 3)": 1.494,
        "(1, 3, 0)": 1.768,
        "(1, 3, 1)": 0.843,
        "(1, 3, 2)": 1.494,
        "(1, 3, 3)": 0.641,
        "(2, 0, 0)": 2.795,
        "(2, 0, 1)": 1.25,
        "(2, 0, 2)": 3.953,
        "(2, 0, 3)": 0.775,
        "(2, 1, 0)": 1.25,
        "(2, 1, 1)": 2.282,
        "(2, 1, 2)": 0.932,
        "(2, 1, 3)": 1.494,
        "(2, 2, 0)": 3.953,
        "(2, 2, 1)": 0.932,
        "(2, 2, 2)": 1.614,
        "(2, 2, 3)": 0.678,
        "(2, 3, 0)": 0.775,
        "(2, 3, 1)": 1.494,
        "(2, 3, 2)": 0.678,
        "(2, 3, 3)": 1.192,
        "(3, 0, 0)": 2.795,
        "(3, 0, 1)": 1.768,
        "(3, 0, 2)": 0.775,
        "(3, 0, 3)": 3.953,
        "(3, 1, 0)": 1.768,
        "(3, 1, 1)": 0.843,
        "(3, 1, 2)": 1.494,
        "(3, 1, 3)": 0.641,
        "(3, 2, 0)": 0.775,
        "(3, 2, 1)": 1.494,
        "(3, 2, 2)": 0.678,
        "(3, 2, 3)": 1.192,
        "(3, 3, 0)": 3.953,
        "(3, 3, 1)": 0.641,
        "(3, 3, 2)": 1.192,
        "(3, 3, 3)": 1.614,
    },
    "Re": {
        "(0, 0, 1)": 2.229,
        "(0, 0, 2)": 2.229,
        "(0, 0, 3)": 2.229,
        "(0, 1, 0)": 1.593,
        "(0, 1, 1)": 0.351,
        "(0, 1, 2)": 1.087,
        "(0, 1, 3)": 0.21,
        "(0, 2, 0)": 1.593,
        "(0, 2, 1)": 0.962,
        "(0, 2, 2)": 0.351,
        "(0, 2, 3)": 0.776,
        "(0, 3, 0)": 1.593,
        "(0, 3, 1)": 0.392,
        "(0, 3, 2)": 0.75,
        "(0, 3, 3)": 0.351,
        "(1, 0, 0)": 0.797,
        "(1, 0, 1)": 1.755,
        "(1, 0, 2)": 0.543,
        "(1, 0, 3)": 1.052,
        "(1, 1, 0)": 1.38,
        "(1, 1, 1)": 0.659,
        "(1, 1, 2)": 1.173,
        "(1, 1, 3)": 0.506,
        "(1, 2, 0)": 0.602,
        "(1, 2, 1)": 0.148,
        "(1, 2, 2)": 0.558,
        "(1, 2, 3)": 0.129,
        "(1, 3, 0)": 0.221,
        "(1, 3, 1)": 0.546,
        "(1, 3, 2)": 0.212,
        "(1, 3, 3)": 0.505,
        "(2, 0, 0)": 0.797,
        "(2, 0, 1)": 0.192,
        "(2, 0, 2)": 1.755,
        "(2, 0, 3)": 0.155,
        "(2, 1, 0)": 0.301,
        "(2, 1, 1)": 0.738,
        "(2, 1, 2)": 0.279,
        "(2, 1, 3)": 0.643,
        "(2, 2, 0)": 1.38,
        "(2, 2, 1)": 0.341,
        "(2, 2, 2)": 0.659,
        "(2, 2, 3)": 0.313,
        "(2, 3, 0)": 0.366,
        "(2, 3, 1)": 0.544,
        "(2, 3, 2)": 0.355,
        "(2, 3, 3)": 0.514,
        "(3, 0, 0)": 0.797,
        "(3, 0, 1)": 0.392,
        "(3, 0, 2)": 0.75,
        "(3, 0, 3)": 1.755,
        "(3, 1, 0)": 0.442,
        "(3, 1, 1)": 0.109,
        "(3, 1, 2)": 0.424,
        "(3, 1, 3)": 0.101,
        "(3, 2, 0)": 0.183,
        "(3, 2, 1)": 0.454,
        "(3, 2, 2)": 0.177,
        "(3, 2, 3)": 0.429,
        "(3, 3, 0)": 1.38,
        "(3, 3, 1)": 0.229,
        "(3, 3, 2)": 0.451,
        "(3, 3, 3)": 0.659,
    },
    "Rh": {
        "(0, 0, 1)": 1.9,
        "(0, 0, 2)": 1.9,
        "(0, 0, 3)": 1.9,
        "(0, 1, 0)": 1.9,
        "(0, 1, 1)": 1.344,
        "(0, 1, 2)": 0.85,
        "(0, 1, 3)": 0.601,
        "(0, 2, 0)": 1.9,
        "(0, 2, 1)": 0.85,
        "(0, 2, 2)": 1.344,
        "(0, 2, 3)": 0.527,
        "(0, 3, 0)": 1.9,
        "(0, 3, 1)": 0.601,
        "(0, 3, 2)": 0.527,
        "(0, 3, 3)": 1.344,
        "(1, 0, 0)": 1.9,
        "(1, 0, 1)": 1.344,
        "(1, 0, 2)": 0.85,
        "(1, 0, 3)": 0.601,
        "(1, 1, 0)": 1.344,
        "(1, 1, 1)": 2.194,
        "(1, 1, 2)": 0.776,
        "(1, 1, 3)": 1.146,
        "(1, 2, 0)": 0.85,
        "(1, 2, 1)": 0.776,
        "(1, 2, 2)": 0.633,
        "(1, 2, 3)": 0.508,
        "(1, 3, 0)": 0.601,
        "(1, 3, 1)": 1.146,
        "(1, 3, 2)": 0.508,
        "(1, 3, 3)": 0.872,
        "(2, 0, 0)": 1.9,
        "(2, 0, 1)": 0.85,
        "(2, 0, 2)": 1.344,
        "(2, 0, 3)": 0.527,
        "(2, 1, 0)": 0.85,
        "(2, 1, 1)": 0.776,
        "(2, 1, 2)": 0.633,
        "(2, 1, 3)": 0.508,
        "(2, 2, 0)": 1.344,
        "(2, 2, 1)": 0.633,
        "(2, 2, 2)": 2.194,
        "(2, 2, 3)": 0.461,
        "(2, 3, 0)": 0.527,
        "(2, 3, 1)": 0.508,
        "(2, 3, 2)": 0.461,
        "(2, 3, 3)": 0.405,
        "(3, 0, 0)": 1.9,
        "(3, 0, 1)": 0.601,
        "(3, 0, 2)": 0.527,
        "(3, 0, 3)": 1.344,
        "(3, 1, 0)": 0.601,
        "(3, 1, 1)": 1.146,
        "(3, 1, 2)": 0.508,
        "(3, 1, 3)": 0.872,
        "(3, 2, 0)": 0.527,
        "(3, 2, 1)": 0.508,
        "(3, 2, 2)": 0.461,
        "(3, 2, 3)": 0.405,
        "(3, 3, 0)": 1.344,
        "(3, 3, 1)": 0.872,
        "(3, 3, 2)": 0.405,
        "(3, 3, 3)": 2.194,
    },
    "Ru": {
        "(0, 0, 1)": 2.138,
        "(0, 0, 2)": 2.138,
        "(0, 0, 3)": 2.138,
        "(0, 1, 0)": 1.559,
        "(0, 1, 1)": 0.342,
        "(0, 1, 2)": 1.052,
        "(0, 1, 3)": 0.203,
        "(0, 2, 0)": 1.559,
        "(0, 2, 1)": 0.94,
        "(0, 2, 2)": 0.342,
        "(0, 2, 3)": 0.753,
        "(0, 3, 0)": 1.559,
        "(0, 3, 1)": 0.383,
        "(0, 3, 2)": 0.732,
        "(0, 3, 3)": 0.342,
        "(1, 0, 0)": 0.779,
        "(1, 0, 1)": 1.71,
        "(1, 0, 2)": 0.526,
        "(1, 0, 3)": 1.014,
        "(1, 1, 0)": 1.35,
        "(1, 1, 1)": 0.644,
        "(1, 1, 2)": 1.142,
        "(1, 1, 3)": 0.49,
        "(1, 2, 0)": 0.589,
        "(1, 2, 1)": 0.144,
        "(1, 2, 2)": 0.545,
        "(1, 2, 3)": 0.125,
        "(1, 3, 0)": 0.216,
        "(1, 3, 1)": 0.534,
        "(1, 3, 2)": 0.207,
        "(1, 3, 3)": 0.492,
        "(2, 0, 0)": 0.779,
        "(2, 0, 1)": 0.188,
        "(2, 0, 2)": 1.71,
        "(2, 0, 3)": 0.151,
        "(2, 1, 0)": 0.295,
        "(2, 1, 1)": 0.721,
        "(2, 1, 2)": 0.272,
        "(2, 1, 3)": 0.626,
        "(2, 2, 0)": 1.35,
        "(2, 2, 1)": 0.333,
        "(2, 2, 2)": 0.644,
        "(2, 2, 3)": 0.305,
        "(2, 3, 0)": 0.358,
        "(2, 3, 1)": 0.532,
        "(2, 3, 2)": 0.347,
        "(2, 3, 3)": 0.502,
        "(3, 0, 0)": 0.779,
        "(3, 0, 1)": 0.383,
        "(3, 0, 2)": 0.732,
        "(3, 0, 3)": 1.71,
        "(3, 1, 0)": 0.432,
        "(3, 1, 1)": 0.107,
        "(3, 1, 2)": 0.414,
        "(3, 1, 3)": 0.59,
        "(3, 2, 0)": 0.179,
        "(3, 2, 1)": 0.444,
        "(3, 2, 2)": 0.173,
        "(3, 2, 3)": 0.418,
        "(3, 3, 0)": 1.35,
        "(3, 3, 1)": 0.224,
        "(3, 3, 2)": 0.44,
        "(3, 3, 3)": 0.644,
    },
    "Sb": {
        "(0, 0, 1)": 1.906,
        "(0, 0, 2)": 1.906,
        "(0, 0, 3)": 1.906,
        "(0, 1, 0)": 1.906,
        "(0, 1, 1)": 0.213,
        "(0, 1, 2)": 0.748,
        "(0, 1, 3)": 1.097,
        "(0, 2, 0)": 1.906,
        "(0, 2, 1)": 0.748,
        "(0, 2, 2)": 0.213,
        "(0, 2, 3)": 0.805,
        "(0, 3, 0)": 1.906,
        "(0, 3, 1)": 1.097,
        "(0, 3, 2)": 0.805,
        "(0, 3, 3)": 0.213,
        "(1, 0, 0)": 1.906,
        "(1, 0, 1)": 0.213,
        "(1, 0, 2)": 0.748,
        "(1, 0, 3)": 1.097,
        "(1, 1, 0)": 0.213,
        "(1, 1, 1)": 1.492,
        "(1, 1, 2)": 1.946,
        "(1, 1, 3)": 0.967,
        "(1, 2, 0)": 0.748,
        "(1, 2, 1)": 1.946,
        "(1, 2, 2)": 1.293,
        "(1, 2, 3)": 0.289,
        "(1, 3, 0)": 1.097,
        "(1, 3, 1)": 0.967,
        "(1, 3, 2)": 0.289,
        "(1, 3, 3)": 0.32,
        "(2, 0, 0)": 1.906,
        "(2, 0, 1)": 0.748,
        "(2, 0, 2)": 0.213,
        "(2, 0, 3)": 0.805,
        "(2, 1, 0)": 0.748,
        "(2, 1, 1)": 1.946,
        "(2, 1, 2)": 1.293,
        "(2, 1, 3)": 0.289,
        "(2, 2, 0)": 0.213,
        "(2, 2, 1)": 1.293,
        "(2, 2, 2)": 1.492,
        "(2, 2, 3)": 0.387,
        "(2, 3, 0)": 0.805,
        "(2, 3, 1)": 0.289,
        "(2, 3, 2)": 0.387,
        "(2, 3, 3)": 0.958,
        "(3, 0, 0)": 1.906,
        "(3, 0, 1)": 1.097,
        "(3, 0, 2)": 0.805,
        "(3, 0, 3)": 0.213,
        "(3, 1, 0)": 1.097,
        "(3, 1, 1)": 0.967,
        "(3, 1, 2)": 0.289,
        "(3, 1, 3)": 0.32,
        "(3, 2, 0)": 0.805,
        "(3, 2, 1)": 0.289,
        "(3, 2, 2)": 0.387,
        "(3, 2, 3)": 0.958,
        "(3, 3, 0)": 0.213,
        "(3, 3, 1)": 0.32,
        "(3, 3, 2)": 0.958,
        "(3, 3, 3)": 1.492,
    },
    "Sc": {
        "(0, 0, 1)": 2.638,
        "(0, 0, 2)": 2.638,
        "(0, 0, 3)": 2.638,
        "(0, 1, 0)": 1.911,
        "(0, 1, 1)": 0.42,
        "(0, 1, 2)": 1.294,
        "(0, 1, 3)": 0.25,
        "(0, 2, 0)": 1.911,
        "(0, 2, 1)": 1.153,
        "(0, 2, 2)": 0.42,
        "(0, 2, 3)": 0.926,
        "(0, 3, 0)": 1.911,
        "(0, 3, 1)": 0.47,
        "(0, 3, 2)": 0.898,
        "(0, 3, 3)": 0.42,
        "(1, 0, 0)": 0.956,
        "(1, 0, 1)": 2.099,
        "(1, 0, 2)": 0.647,
        "(1, 0, 3)": 1.249,
        "(1, 1, 0)": 1.655,
        "(1, 1, 1)": 0.79,
        "(1, 1, 2)": 1.402,
        "(1, 1, 3)": 0.603,
        "(1, 2, 0)": 0.722,
        "(1, 2, 1)": 0.177,
        "(1, 2, 2)": 0.668,
        "(1, 2, 3)": 0.154,
        "(1, 3, 0)": 0.265,
        "(1, 3, 1)": 0.655,
        "(1, 3, 2)": 0.254,
        "(1, 3, 3)": 0.604,
        "(2, 0, 0)": 0.956,
        "(2, 0, 1)": 0.231,
        "(2, 0, 2)": 2.099,
        "(2, 0, 3)": 0.185,
        "(2, 1, 0)": 0.361,
        "(2, 1, 1)": 0.884,
        "(2, 1, 2)": 0.334,
        "(2, 1, 3)": 0.769,
        "(2, 2, 0)": 1.655,
        "(2, 2, 1)": 0.409,
        "(2, 2, 2)": 0.79,
        "(2, 2, 3)": 0.374,
        "(2, 3, 0)": 0.438,
        "(2, 3, 1)": 0.109,
        "(2, 3, 2)": 0.425,
        "(2, 3, 3)": 0.103,
        "(3, 0, 0)": 0.956,
        "(3, 0, 1)": 0.47,
        "(3, 0, 2)": 0.898,
        "(3, 0, 3)": 2.099,
        "(3, 1, 0)": 0.53,
        "(3, 1, 1)": 0.131,
        "(3, 1, 2)": 0.507,
        "(3, 1, 3)": 0.121,
        "(3, 2, 0)": 0.219,
        "(3, 2, 1)": 0.544,
        "(3, 2, 2)": 0.213,
        "(3, 2, 3)": 0.513,
        "(3, 3, 0)": 1.655,
        "(3, 3, 1)": 0.274,
        "(3, 3, 2)": 0.54,
        "(3, 3, 3)": 0.79,
    },
    "Si": {
        "(0, 0, 1)": 1.358,
        "(0, 0, 2)": 1.358,
        "(0, 0, 3)": 1.358,
        "(0, 1, 0)": 1.358,
        "(0, 1, 1)": 1.92,
        "(0, 1, 2)": 0.607,
        "(0, 1, 3)": 0.859,
        "(0, 2, 0)": 1.358,
        "(0, 2, 1)": 0.607,
        "(0, 2, 2)": 1.92,
        "(0, 2, 3)": 0.377,
        "(0, 3, 0)": 1.358,
        "(0, 3, 1)": 0.859,
        "(0, 3, 2)": 0.377,
        "(0, 3, 3)": 1.92,
        "(1, 0, 0)": 1.358,
        "(1, 0, 1)": 1.92,
        "(1, 0, 2)": 0.607,
        "(1, 0, 3)": 0.859,
        "(1, 1, 0)": 1.92,
        "(1, 1, 1)": 2.351,
        "(1, 1, 2)": 1.108,
        "(1, 1, 3)": 0.409,
        "(1, 2, 0)": 0.607,
        "(1, 2, 1)": 1.108,
        "(1, 2, 2)": 0.452,
        "(1, 2, 3)": 0.726,
        "(1, 3, 0)": 0.859,
        "(1, 3, 1)": 0.409,
        "(1, 3, 2)": 0.726,
        "(1, 3, 3)": 0.934,
        "(2, 0, 0)": 1.358,
        "(2, 0, 1)": 0.607,
        "(2, 0, 2)": 1.92,
        "(2, 0, 3)": 0.377,
        "(2, 1, 0)": 0.607,
        "(2, 1, 1)": 1.108,
        "(2, 1, 2)": 0.452,
        "(2, 1, 3)": 0.726,
        "(2, 2, 0)": 1.92,
        "(2, 2, 1)": 0.452,
        "(2, 2, 2)": 2.351,
        "(2, 2, 3)": 0.329,
        "(2, 3, 0)": 0.377,
        "(2, 3, 1)": 0.726,
        "(2, 3, 2)": 0.329,
        "(2, 3, 3)": 0.579,
        "(3, 0, 0)": 1.358,
        "(3, 0, 1)": 0.859,
        "(3, 0, 2)": 0.377,
        "(3, 0, 3)": 1.92,
        "(3, 1, 0)": 0.859,
        "(3, 1, 1)": 0.409,
        "(3, 1, 2)": 0.726,
        "(3, 1, 3)": 0.934,
        "(3, 2, 0)": 0.377,
        "(3, 2, 1)": 0.726,
        "(3, 2, 2)": 0.329,
        "(3, 2, 3)": 0.579,
        "(3, 3, 0)": 1.92,
        "(3, 3, 1)": 0.934,
        "(3, 3, 2)": 0.579,
        "(3, 3, 3)": 2.351,
    },
    "Sm": {
        "(0, 0, 1)": 0.689,
        "(0, 0, 2)": 0.689,
        "(0, 0, 3)": 0.689,
        "(0, 1, 0)": 0.689,
        "(0, 1, 1)": 1.35,
        "(0, 1, 2)": 0.59,
        "(0, 1, 3)": 0.13,
        "(0, 2, 0)": 0.689,
        "(0, 2, 1)": 0.59,
        "(0, 2, 2)": 1.35,
        "(0, 2, 3)": 0.127,
        "(0, 3, 0)": 0.689,
        "(0, 3, 1)": 0.13,
        "(0, 3, 2)": 0.127,
        "(0, 3, 3)": 1.35,
        "(1, 0, 0)": 0.689,
        "(1, 0, 1)": 1.35,
        "(1, 0, 2)": 0.59,
        "(1, 0, 3)": 0.13,
        "(1, 1, 0)": 1.35,
        "(1, 1, 1)": 2.924,
        "(1, 1, 2)": 0.316,
        "(1, 1, 3)": 0.165,
        "(1, 2, 0)": 0.59,
        "(1, 2, 1)": 0.316,
        "(1, 2, 2)": 0.295,
        "(1, 2, 3)": 0.554,
        "(1, 3, 0)": 0.13,
        "(1, 3, 1)": 0.165,
        "(1, 3, 2)": 0.554,
        "(1, 3, 3)": 0.643,
        "(2, 0, 0)": 0.689,
        "(2, 0, 1)": 0.59,
        "(2, 0, 2)": 1.35,
        "(2, 0, 3)": 0.127,
        "(2, 1, 0)": 0.59,
        "(2, 1, 1)": 0.316,
        "(2, 1, 2)": 0.295,
        "(2, 1, 3)": 0.554,
        "(2, 2, 0)": 1.35,
        "(2, 2, 1)": 0.295,
        "(2, 2, 2)": 2.924,
        "(2, 2, 3)": 1.071,
        "(2, 3, 0)": 0.127,
        "(2, 3, 1)": 0.554,
        "(2, 3, 2)": 1.071,
        "(2, 3, 3)": 0.507,
        "(3, 0, 0)": 0.689,
        "(3, 0, 1)": 0.13,
        "(3, 0, 2)": 0.127,
        "(3, 0, 3)": 1.35,
        "(3, 1, 0)": 0.13,
        "(3, 1, 1)": 0.165,
        "(3, 1, 2)": 0.554,
        "(3, 1, 3)": 0.643,
        "(3, 2, 0)": 0.127,
        "(3, 2, 1)": 0.554,
        "(3, 2, 2)": 1.071,
        "(3, 2, 3)": 0.507,
        "(3, 3, 0)": 1.35,
        "(3, 3, 1)": 0.643,
        "(3, 3, 2)": 0.507,
        "(3, 3, 3)": 2.924,
    },
    "Sn": {
        "(0, 0, 1)": 2.058,
        "(0, 0, 2)": 2.058,
        "(0, 0, 3)": 2.058,
        "(0, 1, 0)": 2.092,
        "(0, 1, 1)": 0.503,
        "(0, 1, 2)": 1.079,
        "(0, 1, 3)": 0.273,
        "(0, 2, 0)": 2.092,
        "(0, 2, 1)": 1.203,
        "(0, 2, 2)": 0.503,
        "(0, 2, 3)": 0.845,
        "(0, 3, 0)": 2.092,
        "(0, 3, 1)": 0.635,
        "(0, 3, 2)": 0.189,
        "(0, 3, 3)": 0.503,
        "(1, 0, 0)": 0.697,
        "(1, 0, 1)": 1.51,
        "(1, 0, 2)": 0.36,
        "(1, 0, 3)": 0.82,
        "(1, 1, 0)": 1.482,
        "(1, 1, 1)": 0.629,
        "(1, 1, 2)": 1.038,
        "(1, 1, 3)": 0.432,
        "(1, 2, 0)": 0.736,
        "(1, 2, 1)": 0.221,
        "(1, 2, 2)": 0.588,
        "(1, 2, 3)": 0.172,
        "(1, 3, 0)": 0.365,
        "(1, 3, 1)": 0.678,
        "(1, 3, 2)": 0.311,
        "(1, 3, 3)": 0.566,
        "(2, 0, 0)": 0.697,
        "(2, 0, 1)": 1.203,
        "(2, 0, 2)": 1.51,
        "(2, 0, 3)": 0.845,
        "(2, 1, 0)": 0.245,
        "(2, 1, 1)": 0.664,
        "(2, 1, 2)": 0.196,
        "(2, 1, 3)": 0.517,
        "(2, 2, 0)": 1.482,
        "(2, 2, 1)": 0.344,
        "(2, 2, 2)": 0.629,
        "(2, 2, 3)": 0.286,
        "(2, 3, 0)": 0.444,
        "(2, 3, 1)": 0.139,
        "(2, 3, 2)": 0.391,
        "(2, 3, 3)": 0.121,
        "(3, 0, 0)": 0.697,
        "(3, 0, 1)": 0.212,
        "(3, 0, 2)": 0.567,
        "(3, 0, 3)": 1.51,
        "(3, 1, 0)": 0.365,
        "(3, 1, 1)": 0.678,
        "(3, 1, 2)": 0.311,
        "(3, 1, 3)": 0.566,
        "(3, 2, 0)": 0.148,
        "(3, 2, 1)": 0.418,
        "(3, 2, 2)": 0.13,
        "(3, 2, 3)": 0.363,
        "(3, 3, 0)": 1.482,
        "(3, 3, 1)": 0.235,
        "(3, 3, 2)": 0.445,
        "(3, 3, 3)": 0.629,
    },
    "Sr": {
        "(0, 0, 1)": 3.04,
        "(0, 0, 2)": 3.04,
        "(0, 0, 3)": 3.04,
        "(0, 1, 0)": 3.04,
        "(0, 1, 1)": 2.15,
        "(0, 1, 2)": 1.36,
        "(0, 1, 3)": 0.961,
        "(0, 2, 0)": 3.04,
        "(0, 2, 1)": 1.36,
        "(0, 2, 2)": 2.15,
        "(0, 2, 3)": 0.843,
        "(0, 3, 0)": 3.04,
        "(0, 3, 1)": 0.961,
        "(0, 3, 2)": 0.843,
        "(0, 3, 3)": 2.15,
        "(1, 0, 0)": 3.04,
        "(1, 0, 1)": 2.15,
        "(1, 0, 2)": 1.36,
        "(1, 0, 3)": 0.961,
        "(1, 1, 0)": 2.15,
        "(1, 1, 1)": 3.51,
        "(1, 1, 2)": 1.241,
        "(1, 1, 3)": 1.833,
        "(1, 2, 0)": 1.36,
        "(1, 2, 1)": 1.241,
        "(1, 2, 2)": 1.013,
        "(1, 2, 3)": 0.812,
        "(1, 3, 0)": 0.961,
        "(1, 3, 1)": 1.833,
        "(1, 3, 2)": 0.812,
        "(1, 3, 3)": 1.395,
        "(2, 0, 0)": 3.04,
        "(2, 0, 1)": 1.36,
        "(2, 0, 2)": 2.15,
        "(2, 0, 3)": 0.843,
        "(2, 1, 0)": 1.36,
        "(2, 1, 1)": 1.241,
        "(2, 1, 2)": 1.013,
        "(2, 1, 3)": 0.812,
        "(2, 2, 0)": 2.15,
        "(2, 2, 1)": 1.013,
        "(2, 2, 2)": 3.51,
        "(2, 2, 3)": 0.737,
        "(2, 3, 0)": 0.843,
        "(2, 3, 1)": 0.812,
        "(2, 3, 2)": 0.737,
        "(2, 3, 3)": 0.648,
        "(3, 0, 0)": 3.04,
        "(3, 0, 1)": 0.961,
        "(3, 0, 2)": 0.843,
        "(3, 0, 3)": 2.15,
        "(3, 1, 0)": 0.961,
        "(3, 1, 1)": 1.833,
        "(3, 1, 2)": 0.812,
        "(3, 1, 3)": 1.395,
        "(3, 2, 0)": 0.843,
        "(3, 2, 1)": 0.812,
        "(3, 2, 2)": 0.737,
        "(3, 2, 3)": 0.648,
        "(3, 3, 0)": 2.15,
        "(3, 3, 1)": 1.395,
        "(3, 3, 2)": 0.648,
        "(3, 3, 3)": 3.51,
    },
    "Ta": {
        "(0, 0, 1)": 1.655,
        "(0, 0, 2)": 1.655,
        "(0, 0, 3)": 1.655,
        "(0, 1, 0)": 1.655,
        "(0, 1, 1)": 2.341,
        "(0, 1, 2)": 0.74,
        "(0, 1, 3)": 1.047,
        "(0, 2, 0)": 1.655,
        "(0, 2, 1)": 0.74,
        "(0, 2, 2)": 2.341,
        "(0, 2, 3)": 0.459,
        "(0, 3, 0)": 1.655,
        "(0, 3, 1)": 1.047,
        "(0, 3, 2)": 0.459,
        "(0, 3, 3)": 2.341,
        "(1, 0, 0)": 1.655,
        "(1, 0, 1)": 2.341,
        "(1, 0, 2)": 0.74,
        "(1, 0, 3)": 1.047,
        "(1, 1, 0)": 2.341,
        "(1, 1, 1)": 0.956,
        "(1, 1, 2)": 1.351,
        "(1, 1, 3)": 0.499,
        "(1, 2, 0)": 0.74,
        "(1, 2, 1)": 1.351,
        "(1, 2, 2)": 0.552,
        "(1, 2, 3)": 0.885,
        "(1, 3, 0)": 1.047,
        "(1, 3, 1)": 0.499,
        "(1, 3, 2)": 0.885,
        "(1, 3, 3)": 0.38,
        "(2, 0, 0)": 1.655,
        "(2, 0, 1)": 0.74,
        "(2, 0, 2)": 2.341,
        "(2, 0, 3)": 0.459,
        "(2, 1, 0)": 0.74,
        "(2, 1, 1)": 1.351,
        "(2, 1, 2)": 0.552,
        "(2, 1, 3)": 0.885,
        "(2, 2, 0)": 2.341,
        "(2, 2, 1)": 0.552,
        "(2, 2, 2)": 0.956,
        "(2, 2, 3)": 0.401,
        "(2, 3, 0)": 0.459,
        "(2, 3, 1)": 0.885,
        "(2, 3, 2)": 0.401,
        "(2, 3, 3)": 0.706,
        "(3, 0, 0)": 1.655,
        "(3, 0, 1)": 1.047,
        "(3, 0, 2)": 0.459,
        "(3, 0, 3)": 2.341,
        "(3, 1, 0)": 1.047,
        "(3, 1, 1)": 0.499,
        "(3, 1, 2)": 0.885,
        "(3, 1, 3)": 0.38,
        "(3, 2, 0)": 0.459,
        "(3, 2, 1)": 0.885,
        "(3, 2, 2)": 0.401,
        "(3, 2, 3)": 0.706,
        "(3, 3, 0)": 2.341,
        "(3, 3, 1)": 0.38,
        "(3, 3, 2)": 0.706,
        "(3, 3, 3)": 0.956,
    },
    "Tb": {
        "(0, 0, 1)": 2.846,
        "(0, 0, 2)": 2.846,
        "(0, 0, 3)": 2.846,
        "(0, 1, 0)": 2.078,
        "(0, 1, 1)": 0.456,
        "(0, 1, 2)": 1.401,
        "(0, 1, 3)": 0.27,
        "(0, 2, 0)": 2.078,
        "(0, 2, 1)": 1.253,
        "(0, 2, 2)": 0.456,
        "(0, 2, 3)": 1.004,
        "(0, 3, 0)": 2.078,
        "(0, 3, 1)": 0.511,
        "(0, 3, 2)": 0.976,
        "(0, 3, 3)": 0.456,
        "(1, 0, 0)": 1.039,
        "(1, 0, 1)": 2.279,
        "(1, 0, 2)": 0.701,
        "(1, 0, 3)": 1.351,
        "(1, 1, 0)": 1.8,
        "(1, 1, 1)": 0.858,
        "(1, 1, 2)": 1.521,
        "(1, 1, 3)": 0.653,
        "(1, 2, 0)": 0.786,
        "(1, 2, 1)": 0.192,
        "(1, 2, 2)": 0.726,
        "(1, 2, 3)": 0.167,
        "(1, 3, 0)": 0.288,
        "(1, 3, 1)": 0.712,
        "(1, 3, 2)": 0.276,
        "(1, 3, 3)": 0.656,
        "(2, 0, 0)": 1.039,
        "(2, 0, 1)": 0.251,
        "(2, 0, 2)": 2.279,
        "(2, 0, 3)": 0.201,
        "(2, 1, 0)": 0.393,
        "(2, 1, 1)": 0.962,
        "(2, 1, 2)": 0.363,
        "(2, 1, 3)": 0.834,
        "(2, 2, 0)": 1.8,
        "(2, 2, 1)": 0.444,
        "(2, 2, 2)": 0.858,
        "(2, 2, 3)": 0.407,
        "(2, 3, 0)": 0.477,
        "(2, 3, 1)": 0.118,
        "(2, 3, 2)": 0.462,
        "(2, 3, 3)": 0.112,
        "(3, 0, 0)": 1.039,
        "(3, 0, 1)": 0.511,
        "(3, 0, 2)": 0.976,
        "(3, 0, 3)": 2.279,
        "(3, 1, 0)": 0.576,
        "(3, 1, 1)": 0.142,
        "(3, 1, 2)": 0.552,
        "(3, 1, 3)": 0.131,
        "(3, 2, 0)": 0.238,
        "(3, 2, 1)": 0.591,
        "(3, 2, 2)": 0.231,
        "(3, 2, 3)": 0.558,
        "(3, 3, 0)": 1.8,
        "(3, 3, 1)": 0.298,
        "(3, 3, 2)": 0.587,
        "(3, 3, 3)": 0.858,
    },
    "Tc": {
        "(0, 0, 1)": 2.197,
        "(0, 0, 2)": 2.197,
        "(0, 0, 3)": 2.197,
        "(0, 1, 0)": 1.582,
        "(0, 1, 1)": 0.348,
        "(0, 1, 2)": 1.075,
        "(0, 1, 3)": 0.208,
        "(0, 2, 0)": 1.582,
        "(0, 2, 1)": 0.955,
        "(0, 2, 2)": 0.348,
        "(0, 2, 3)": 0.768,
        "(0, 3, 0)": 1.582,
        "(0, 3, 1)": 0.389,
        "(0, 3, 2)": 0.744,
        "(0, 3, 3)": 0.348,
        "(1, 0, 0)": 0.791,
        "(1, 0, 1)": 1.74,
        "(1, 0, 2)": 0.537,
        "(1, 0, 3)": 1.039,
        "(1, 1, 0)": 1.37,
        "(1, 1, 1)": 0.654,
        "(1, 1, 2)": 1.163,
        "(1, 1, 3)": 0.5,
        "(1, 2, 0)": 0.598,
        "(1, 2, 1)": 0.146,
        "(1, 2, 2)": 0.554,
        "(1, 2, 3)": 0.127,
        "(1, 3, 0)": 0.219,
        "(1, 3, 1)": 0.542,
        "(1, 3, 2)": 0.21,
        "(1, 3, 3)": 0.5,
        "(2, 0, 0)": 0.791,
        "(2, 0, 1)": 0.191,
        "(2, 0, 2)": 1.74,
        "(2, 0, 3)": 0.154,
        "(2, 1, 0)": 0.299,
        "(2, 1, 1)": 0.732,
        "(2, 1, 2)": 0.277,
        "(2, 1, 3)": 0.637,
        "(2, 2, 0)": 1.37,
        "(2, 2, 1)": 0.338,
        "(2, 2, 2)": 0.654,
        "(2, 2, 3)": 0.31,
        "(2, 3, 0)": 0.363,
        "(2, 3, 1)": 0.54,
        "(2, 3, 2)": 0.352,
        "(2, 3, 3)": 0.51,
        "(3, 0, 0)": 0.791,
        "(3, 0, 1)": 0.389,
        "(3, 0, 2)": 0.744,
        "(3, 0, 3)": 1.74,
        "(3, 1, 0)": 0.439,
        "(3, 1, 1)": 0.108,
        "(3, 1, 2)": 0.42,
        "(3, 1, 3)": 0.1,
        "(3, 2, 0)": 0.181,
        "(3, 2, 1)": 0.45,
        "(3, 2, 2)": 0.176,
        "(3, 2, 3)": 0.425,
        "(3, 3, 0)": 1.37,
        "(3, 3, 1)": 0.227,
        "(3, 3, 2)": 0.447,
        "(3, 3, 3)": 0.654,
    },
    "Th": {
        "(0, 0, 1)": 2.54,
        "(0, 0, 2)": 2.54,
        "(0, 0, 3)": 2.54,
        "(0, 1, 0)": 2.54,
        "(0, 1, 1)": 1.796,
        "(0, 1, 2)": 1.136,
        "(0, 1, 3)": 0.803,
        "(0, 2, 0)": 2.54,
        "(0, 2, 1)": 1.136,
        "(0, 2, 2)": 1.796,
        "(0, 2, 3)": 0.704,
        "(0, 3, 0)": 2.54,
        "(0, 3, 1)": 0.803,
        "(0, 3, 2)": 0.704,
        "(0, 3, 3)": 1.796,
        "(1, 0, 0)": 2.54,
        "(1, 0, 1)": 1.796,
        "(1, 0, 2)": 1.136,
        "(1, 0, 3)": 0.803,
        "(1, 1, 0)": 1.796,
        "(1, 1, 1)": 2.933,
        "(1, 1, 2)": 1.037,
        "(1, 1, 3)": 1.532,
        "(1, 2, 0)": 1.136,
        "(1, 2, 1)": 1.037,
        "(1, 2, 2)": 0.847,
        "(1, 2, 3)": 0.679,
        "(1, 3, 0)": 0.803,
        "(1, 3, 1)": 1.532,
        "(1, 3, 2)": 0.679,
        "(1, 3, 3)": 1.165,
        "(2, 0, 0)": 2.54,
        "(2, 0, 1)": 1.136,
        "(2, 0, 2)": 1.796,
        "(2, 0, 3)": 0.704,
        "(2, 1, 0)": 1.136,
        "(2, 1, 1)": 1.037,
        "(2, 1, 2)": 0.847,
        "(2, 1, 3)": 0.679,
        "(2, 2, 0)": 1.796,
        "(2, 2, 1)": 0.847,
        "(2, 2, 2)": 2.933,
        "(2, 2, 3)": 0.616,
        "(2, 3, 0)": 0.704,
        "(2, 3, 1)": 0.679,
        "(2, 3, 2)": 0.616,
        "(2, 3, 3)": 0.542,
        "(3, 0, 0)": 2.54,
        "(3, 0, 1)": 0.803,
        "(3, 0, 2)": 0.704,
        "(3, 0, 3)": 1.796,
        "(3, 1, 0)": 0.803,
        "(3, 1, 1)": 1.532,
        "(3, 1, 2)": 0.679,
        "(3, 1, 3)": 1.165,
        "(3, 2, 0)": 0.704,
        "(3, 2, 1)": 0.679,
        "(3, 2, 2)": 0.616,
        "(3, 2, 3)": 0.542,
        "(3, 3, 0)": 1.796,
        "(3, 3, 1)": 1.165,
        "(3, 3, 2)": 0.542,
        "(3, 3, 3)": 2.933,
    },
    "Ti": {
        "(0, 0, 1)": 2.342,
        "(0, 0, 2)": 2.342,
        "(0, 0, 3)": 2.342,
        "(0, 1, 0)": 1.703,
        "(0, 1, 1)": 0.374,
        "(0, 1, 2)": 1.151,
        "(0, 1, 3)": 0.222,
        "(0, 2, 0)": 1.703,
        "(0, 2, 1)": 1.027,
        "(0, 2, 2)": 0.374,
        "(0, 2, 3)": 0.824,
        "(0, 3, 0)": 1.703,
        "(0, 3, 1)": 0.419,
        "(0, 3, 2)": 0.8,
        "(0, 3, 3)": 0.374,
        "(1, 0, 0)": 0.852,
        "(1, 0, 1)": 1.869,
        "(1, 0, 2)": 0.575,
        "(1, 0, 3)": 1.11,
        "(1, 1, 0)": 1.475,
        "(1, 1, 1)": 0.703,
        "(1, 1, 2)": 1.248,
        "(1, 1, 3)": 0.536,
        "(1, 2, 0)": 0.644,
        "(1, 2, 1)": 0.158,
        "(1, 2, 2)": 0.595,
        "(1, 2, 3)": 0.137,
        "(1, 3, 0)": 0.236,
        "(1, 3, 1)": 0.584,
        "(1, 3, 2)": 0.226,
        "(1, 3, 3)": 0.538,
        "(2, 0, 0)": 0.852,
        "(2, 0, 1)": 0.205,
        "(2, 0, 2)": 1.869,
        "(2, 0, 3)": 0.165,
        "(2, 1, 0)": 0.322,
        "(2, 1, 1)": 0.788,
        "(2, 1, 2)": 0.298,
        "(2, 1, 3)": 0.684,
        "(2, 2, 0)": 1.475,
        "(2, 2, 1)": 0.364,
        "(2, 2, 2)": 0.703,
        "(2, 2, 3)": 0.333,
        "(2, 3, 0)": 0.391,
        "(2, 3, 1)": 0.582,
        "(2, 3, 2)": 0.379,
        "(2, 3, 3)": 0.549,
        "(3, 0, 0)": 0.852,
        "(3, 0, 1)": 0.419,
        "(3, 0, 2)": 0.8,
        "(3, 0, 3)": 1.869,
        "(3, 1, 0)": 0.472,
        "(3, 1, 1)": 0.117,
        "(3, 1, 2)": 0.452,
        "(3, 1, 3)": 0.108,
        "(3, 2, 0)": 0.195,
        "(3, 2, 1)": 0.485,
        "(3, 2, 2)": 0.19,
        "(3, 2, 3)": 0.457,
        "(3, 3, 0)": 1.475,
        "(3, 3, 1)": 0.244,
        "(3, 3, 2)": 0.481,
        "(3, 3, 3)": 0.703,
    },
    "Tl": {
        "(0, 0, 1)": 2.766,
        "(0, 0, 2)": 2.766,
        "(0, 0, 3)": 2.766,
        "(0, 1, 0)": 1.998,
        "(0, 1, 1)": 0.439,
        "(0, 1, 2)": 1.355,
        "(0, 1, 3)": 0.262,
        "(0, 2, 0)": 1.998,
        "(0, 2, 1)": 1.205,
        "(0, 2, 2)": 0.439,
        "(0, 2, 3)": 0.969,
        "(0, 3, 0)": 1.998,
        "(0, 3, 1)": 0.491,
        "(0, 3, 2)": 0.939,
        "(0, 3, 3)": 0.439,
        "(1, 0, 0)": 0.999,
        "(1, 0, 1)": 2.196,
        "(1, 0, 2)": 0.678,
        "(1, 0, 3)": 1.309,
        "(1, 1, 0)": 1.73,
        "(1, 1, 1)": 0.826,
        "(1, 1, 2)": 1.467,
        "(1, 1, 3)": 0.631,
        "(1, 2, 0)": 0.755,
        "(1, 2, 1)": 0.185,
        "(1, 2, 2)": 0.699,
        "(1, 2, 3)": 0.161,
        "(1, 3, 0)": 0.277,
        "(1, 3, 1)": 0.685,
        "(1, 3, 2)": 0.265,
        "(1, 3, 3)": 0.631,
        "(2, 0, 0)": 0.999,
        "(2, 0, 1)": 0.241,
        "(2, 0, 2)": 2.196,
        "(2, 0, 3)": 0.194,
        "(2, 1, 0)": 0.378,
        "(2, 1, 1)": 0.925,
        "(2, 1, 2)": 0.349,
        "(2, 1, 3)": 0.804,
        "(2, 2, 0)": 1.73,
        "(2, 2, 1)": 0.427,
        "(2, 2, 2)": 0.826,
        "(2, 2, 3)": 0.392,
        "(2, 3, 0)": 0.458,
        "(2, 3, 1)": 0.114,
        "(2, 3, 2)": 0.445,
        "(2, 3, 3)": 0.107,
        "(3, 0, 0)": 0.999,
        "(3, 0, 1)": 0.491,
        "(3, 0, 2)": 0.939,
        "(3, 0, 3)": 2.196,
        "(3, 1, 0)": 0.554,
        "(3, 1, 1)": 0.137,
        "(3, 1, 2)": 0.531,
        "(3, 1, 3)": 0.126,
        "(3, 2, 0)": 0.229,
        "(3, 2, 1)": 0.568,
        "(3, 2, 2)": 0.222,
        "(3, 2, 3)": 0.537,
        "(3, 3, 0)": 1.73,
        "(3, 3, 1)": 0.287,
        "(3, 3, 2)": 0.565,
        "(3, 3, 3)": 0.826,
    },
    "Tm": {
        "(0, 0, 1)": 2.779,
        "(0, 0, 2)": 2.779,
        "(0, 0, 3)": 2.779,
        "(0, 1, 0)": 2.044,
        "(0, 1, 1)": 0.447,
        "(0, 1, 2)": 1.373,
        "(0, 1, 3)": 0.264,
        "(0, 2, 0)": 2.044,
        "(0, 2, 1)": 1.231,
        "(0, 2, 2)": 0.447,
        "(0, 2, 3)": 0.984,
        "(0, 3, 0)": 2.044,
        "(0, 3, 1)": 0.503,
        "(0, 3, 2)": 0.959,
        "(0, 3, 3)": 0.447,
        "(1, 0, 0)": 1.022,
        "(1, 0, 1)": 2.237,
        "(1, 0, 2)": 0.686,
        "(1, 0, 3)": 1.321,
        "(1, 1, 0)": 1.77,
        "(1, 1, 1)": 0.843,
        "(1, 1, 2)": 1.493,
        "(1, 1, 3)": 0.64,
        "(1, 2, 0)": 0.772,
        "(1, 2, 1)": 0.189,
        "(1, 2, 2)": 0.713,
        "(1, 2, 3)": 0.164,
        "(1, 3, 0)": 0.283,
        "(1, 3, 1)": 0.7,
        "(1, 3, 2)": 0.271,
        "(1, 3, 3)": 0.644,
        "(2, 0, 0)": 1.022,
        "(2, 0, 1)": 0.246,
        "(2, 0, 2)": 2.237,
        "(2, 0, 3)": 0.197,
        "(2, 1, 0)": 0.386,
        "(2, 1, 1)": 0.945,
        "(2, 1, 2)": 0.356,
        "(2, 1, 3)": 0.819,
        "(2, 2, 0)": 1.77,
        "(2, 2, 1)": 0.437,
        "(2, 2, 2)": 0.843,
        "(2, 2, 3)": 0.399,
        "(2, 3, 0)": 0.469,
        "(2, 3, 1)": 0.116,
        "(2, 3, 2)": 0.455,
        "(2, 3, 3)": 0.11,
        "(3, 0, 0)": 1.022,
        "(3, 0, 1)": 0.503,
        "(3, 0, 2)": 0.959,
        "(3, 0, 3)": 2.237,
        "(3, 1, 0)": 0.567,
        "(3, 1, 1)": 0.14,
        "(3, 1, 2)": 0.542,
        "(3, 1, 3)": 0.129,
        "(3, 2, 0)": 0.234,
        "(3, 2, 1)": 0.581,
        "(3, 2, 2)": 0.227,
        "(3, 2, 3)": 0.548,
        "(3, 3, 0)": 1.77,
        "(3, 3, 1)": 0.293,
        "(3, 3, 2)": 0.577,
        "(3, 3, 3)": 0.843,
    },
    "U": {
        "(0, 0, 1)": 5.86,
        "(0, 0, 2)": 5.86,
        "(0, 0, 3)": 5.86,
        "(0, 1, 0)": 4.948,
        "(0, 1, 1)": 3.78,
        "(0, 1, 2)": 2.521,
        "(0, 1, 3)": 1.817,
        "(0, 2, 0)": 4.948,
        "(0, 2, 1)": 2.279,
        "(0, 2, 2)": 3.78,
        "(0, 2, 3)": 1.533,
        "(0, 3, 0)": 4.948,
        "(0, 3, 1)": 1.588,
        "(0, 3, 2)": 1.437,
        "(0, 3, 3)": 3.78,
        "(1, 0, 0)": 2.85,
        "(1, 0, 1)": 2.563,
        "(1, 0, 2)": 2.043,
        "(1, 0, 3)": 1.611,
        "(1, 1, 0)": 2.47,
        "(1, 1, 1)": 2.276,
        "(1, 1, 2)": 1.888,
        "(1, 1, 3)": 1.532,
        "(1, 2, 0)": 1.868,
        "(1, 2, 1)": 1.78,
        "(1, 2, 2)": 1.575,
        "(1, 2, 3)": 1.35,
        "(1, 3, 0)": 1.427,
        "(1, 3, 1)": 1.387,
        "(1, 3, 2)": 1.283,
        "(1, 3, 3)": 1.152,
        "(2, 0, 0)": 2.85,
        "(2, 0, 1)": 1.385,
        "(2, 0, 2)": 2.563,
        "(2, 0, 3)": 1.151,
        "(2, 1, 0)": 1.369,
        "(2, 1, 1)": 1.333,
        "(2, 1, 2)": 1.241,
        "(2, 1, 3)": 1.121,
        "(2, 2, 0)": 2.47,
        "(2, 2, 1)": 1.208,
        "(2, 2, 2)": 2.276,
        "(2, 2, 3)": 1.044,
        "(2, 3, 0)": 1.078,
        "(2, 3, 1)": 1.06,
        "(2, 3, 2)": 1.012,
        "(2, 3, 3)": 0.944,
        "(3, 0, 0)": 2.85,
        "(3, 0, 1)": 0.938,
        "(3, 0, 2)": 0.904,
        "(3, 0, 3)": 2.563,
        "(3, 1, 0)": 0.933,
        "(3, 1, 1)": 0.921,
        "(3, 1, 2)": 0.889,
        "(3, 1, 3)": 0.842,
        "(3, 2, 0)": 0.887,
        "(3, 2, 1)": 0.877,
        "(3, 2, 2)": 0.849,
        "(3, 2, 3)": 0.808,
        "(3, 3, 0)": 2.47,
        "(3, 3, 1)": 0.815,
        "(3, 3, 2)": 0.793,
        "(3, 3, 3)": 2.276,
    },
    "V": {
        "(0, 0, 1)": 1.51,
        "(0, 0, 2)": 1.51,
        "(0, 0, 3)": 1.51,
        "(0, 1, 0)": 1.51,
        "(0, 1, 1)": 2.135,
        "(0, 1, 2)": 0.675,
        "(0, 1, 3)": 0.955,
        "(0, 2, 0)": 1.51,
        "(0, 2, 1)": 0.675,
        "(0, 2, 2)": 2.135,
        "(0, 2, 3)": 0.419,
        "(0, 3, 0)": 1.51,
        "(0, 3, 1)": 0.955,
        "(0, 3, 2)": 0.419,
        "(0, 3, 3)": 2.135,
        "(1, 0, 0)": 1.51,
        "(1, 0, 1)": 2.135,
        "(1, 0, 2)": 0.675,
        "(1, 0, 3)": 0.955,
        "(1, 1, 0)": 2.135,
        "(1, 1, 1)": 0.872,
        "(1, 1, 2)": 1.233,
        "(1, 1, 3)": 0.455,
        "(1, 2, 0)": 0.675,
        "(1, 2, 1)": 1.233,
        "(1, 2, 2)": 0.503,
        "(1, 2, 3)": 0.807,
        "(1, 3, 0)": 0.955,
        "(1, 3, 1)": 0.455,
        "(1, 3, 2)": 0.807,
        "(1, 3, 3)": 0.346,
        "(2, 0, 0)": 1.51,
        "(2, 0, 1)": 0.675,
        "(2, 0, 2)": 2.135,
        "(2, 0, 3)": 0.419,
        "(2, 1, 0)": 0.675,
        "(2, 1, 1)": 1.233,
        "(2, 1, 2)": 0.503,
        "(2, 1, 3)": 0.807,
        "(2, 2, 0)": 2.135,
        "(2, 2, 1)": 0.503,
        "(2, 2, 2)": 0.872,
        "(2, 2, 3)": 0.366,
        "(2, 3, 0)": 0.419,
        "(2, 3, 1)": 0.807,
        "(2, 3, 2)": 0.366,
        "(2, 3, 3)": 0.644,
        "(3, 0, 0)": 1.51,
        "(3, 0, 1)": 0.955,
        "(3, 0, 2)": 0.419,
        "(3, 0, 3)": 2.135,
        "(3, 1, 0)": 0.955,
        "(3, 1, 1)": 0.455,
        "(3, 1, 2)": 0.807,
        "(3, 1, 3)": 0.346,
        "(3, 2, 0)": 0.419,
        "(3, 2, 1)": 0.807,
        "(3, 2, 2)": 0.366,
        "(3, 2, 3)": 0.644,
        "(3, 3, 0)": 2.135,
        "(3, 3, 1)": 0.346,
        "(3, 3, 2)": 0.644,
        "(3, 3, 3)": 0.872,
    },
    "W": {
        "(0, 0, 1)": 1.58,
        "(0, 0, 2)": 1.58,
        "(0, 0, 3)": 1.58,
        "(0, 1, 0)": 1.58,
        "(0, 1, 1)": 2.234,
        "(0, 1, 2)": 0.707,
        "(0, 1, 3)": 0.999,
        "(0, 2, 0)": 1.58,
        "(0, 2, 1)": 0.707,
        "(0, 2, 2)": 2.234,
        "(0, 2, 3)": 0.438,
        "(0, 3, 0)": 1.58,
        "(0, 3, 1)": 0.999,
        "(0, 3, 2)": 0.438,
        "(0, 3, 3)": 2.234,
        "(1, 0, 0)": 1.58,
        "(1, 0, 1)": 2.234,
        "(1, 0, 2)": 0.707,
        "(1, 0, 3)": 0.999,
        "(1, 1, 0)": 2.234,
        "(1, 1, 1)": 0.912,
        "(1, 1, 2)": 1.29,
        "(1, 1, 3)": 0.476,
        "(1, 2, 0)": 0.707,
        "(1, 2, 1)": 1.29,
        "(1, 2, 2)": 0.527,
        "(1, 2, 3)": 0.845,
        "(1, 3, 0)": 0.999,
        "(1, 3, 1)": 0.476,
        "(1, 3, 2)": 0.845,
        "(1, 3, 3)": 0.362,
        "(2, 0, 0)": 1.58,
        "(2, 0, 1)": 0.707,
        "(2, 0, 2)": 2.234,
        "(2, 0, 3)": 0.438,
        "(2, 1, 0)": 0.707,
        "(2, 1, 1)": 1.29,
        "(2, 1, 2)": 0.527,
        "(2, 1, 3)": 0.845,
        "(2, 2, 0)": 2.234,
        "(2, 2, 1)": 0.527,
        "(2, 2, 2)": 0.912,
        "(2, 2, 3)": 0.383,
        "(2, 3, 0)": 0.438,
        "(2, 3, 1)": 0.845,
        "(2, 3, 2)": 0.383,
        "(2, 3, 3)": 0.674,
        "(3, 0, 0)": 1.58,
        "(3, 0, 1)": 0.999,
        "(3, 0, 2)": 0.438,
        "(3, 0, 3)": 2.234,
        "(3, 1, 0)": 0.999,
        "(3, 1, 1)": 0.476,
        "(3, 1, 2)": 0.845,
        "(3, 1, 3)": 0.362,
        "(3, 2, 0)": 0.438,
        "(3, 2, 1)": 0.845,
        "(3, 2, 2)": 0.383,
        "(3, 2, 3)": 0.674,
        "(3, 3, 0)": 2.234,
        "(3, 3, 1)": 0.362,
        "(3, 3, 2)": 0.674,
        "(3, 3, 3)": 0.912,
    },
    "Xe": {
        "(0, 0, 1)": 3.1,
        "(0, 0, 2)": 3.1,
        "(0, 0, 3)": 3.1,
        "(0, 1, 0)": 3.1,
        "(0, 1, 1)": 2.192,
        "(0, 1, 2)": 1.386,
        "(0, 1, 3)": 0.98,
        "(0, 2, 0)": 3.1,
        "(0, 2, 1)": 1.386,
        "(0, 2, 2)": 2.192,
        "(0, 2, 3)": 0.86,
        "(0, 3, 0)": 3.1,
        "(0, 3, 1)": 0.98,
        "(0, 3, 2)": 0.86,
        "(0, 3, 3)": 2.192,
        "(1, 0, 0)": 3.1,
        "(1, 0, 1)": 2.192,
        "(1, 0, 2)": 1.386,
        "(1, 0, 3)": 0.98,
        "(1, 1, 0)": 2.192,
        "(1, 1, 1)": 3.58,
        "(1, 1, 2)": 1.266,
        "(1, 1, 3)": 1.869,
        "(1, 2, 0)": 1.386,
        "(1, 2, 1)": 1.266,
        "(1, 2, 2)": 1.033,
        "(1, 2, 3)": 0.829,
        "(1, 3, 0)": 0.98,
        "(1, 3, 1)": 1.869,
        "(1, 3, 2)": 0.829,
        "(1, 3, 3)": 1.422,
        "(2, 0, 0)": 3.1,
        "(2, 0, 1)": 1.386,
        "(2, 0, 2)": 2.192,
        "(2, 0, 3)": 0.86,
        "(2, 1, 0)": 1.386,
        "(2, 1, 1)": 1.266,
        "(2, 1, 2)": 1.033,
        "(2, 1, 3)": 0.829,
        "(2, 2, 0)": 2.192,
        "(2, 2, 1)": 1.033,
        "(2, 2, 2)": 3.58,
        "(2, 2, 3)": 0.752,
        "(2, 3, 0)": 0.86,
        "(2, 3, 1)": 0.829,
        "(2, 3, 2)": 0.752,
        "(2, 3, 3)": 0.661,
        "(3, 0, 0)": 3.1,
        "(3, 0, 1)": 0.98,
        "(3, 0, 2)": 0.86,
        "(3, 0, 3)": 2.192,
        "(3, 1, 0)": 0.98,
        "(3, 1, 1)": 1.869,
        "(3, 1, 2)": 0.829,
        "(3, 1, 3)": 1.422,
        "(3, 2, 0)": 0.86,
        "(3, 2, 1)": 0.829,
        "(3, 2, 2)": 0.752,
        "(3, 2, 3)": 0.661,
        "(3, 3, 0)": 2.192,
        "(3, 3, 1)": 1.422,
        "(3, 3, 2)": 0.661,
        "(3, 3, 3)": 3.58,
    },
    "Y": {
        "(0, 0, 1)": 2.867,
        "(0, 0, 2)": 2.867,
        "(0, 0, 3)": 2.867,
        "(0, 1, 0)": 2.107,
        "(0, 1, 1)": 0.461,
        "(0, 1, 2)": 1.416,
        "(0, 1, 3)": 0.273,
        "(0, 2, 0)": 2.107,
        "(0, 2, 1)": 1.27,
        "(0, 2, 2)": 0.461,
        "(0, 2, 3)": 1.015,
        "(0, 3, 0)": 2.107,
        "(0, 3, 1)": 0.518,
        "(0, 3, 2)": 0.989,
        "(0, 3, 3)": 0.461,
        "(1, 0, 0)": 1.054,
        "(1, 0, 1)": 2.307,
        "(1, 0, 2)": 0.708,
        "(1, 0, 3)": 1.363,
        "(1, 1, 0)": 1.825,
        "(1, 1, 1)": 0.87,
        "(1, 1, 2)": 1.54,
        "(1, 1, 3)": 0.66,
        "(1, 2, 0)": 0.796,
        "(1, 2, 1)": 0.195,
        "(1, 2, 2)": 0.735,
        "(1, 2, 3)": 0.169,
        "(1, 3, 0)": 0.292,
        "(1, 3, 1)": 0.722,
        "(1, 3, 2)": 0.279,
        "(1, 3, 3)": 0.664,
        "(2, 0, 0)": 1.054,
        "(2, 0, 1)": 0.254,
        "(2, 0, 2)": 2.307,
        "(2, 0, 3)": 0.203,
        "(2, 1, 0)": 0.398,
        "(2, 1, 1)": 0.975,
        "(2, 1, 2)": 0.368,
        "(2, 1, 3)": 0.844,
        "(2, 2, 0)": 1.825,
        "(2, 2, 1)": 0.451,
        "(2, 2, 2)": 0.87,
        "(2, 2, 3)": 0.412,
        "(2, 3, 0)": 0.483,
        "(2, 3, 1)": 0.12,
        "(2, 3, 2)": 0.469,
        "(2, 3, 3)": 0.113,
        "(3, 0, 0)": 1.054,
        "(3, 0, 1)": 0.518,
        "(3, 0, 2)": 0.989,
        "(3, 0, 3)": 2.307,
        "(3, 1, 0)": 0.584,
        "(3, 1, 1)": 0.144,
        "(3, 1, 2)": 0.559,
        "(3, 1, 3)": 0.133,
        "(3, 2, 0)": 0.242,
        "(3, 2, 1)": 0.6,
        "(3, 2, 2)": 0.234,
        "(3, 2, 3)": 0.565,
        "(3, 3, 0)": 1.825,
        "(3, 3, 1)": 0.302,
        "(3, 3, 2)": 0.595,
        "(3, 3, 3)": 0.87,
    },
    "Yb": {
        "(0, 0, 1)": 2.745,
        "(0, 0, 2)": 2.745,
        "(0, 0, 3)": 2.745,
        "(0, 1, 0)": 2.745,
        "(0, 1, 1)": 1.941,
        "(0, 1, 2)": 1.228,
        "(0, 1, 3)": 0.868,
        "(0, 2, 0)": 2.745,
        "(0, 2, 1)": 1.228,
        "(0, 2, 2)": 1.941,
        "(0, 2, 3)": 0.761,
        "(0, 3, 0)": 2.745,
        "(0, 3, 1)": 0.868,
        "(0, 3, 2)": 0.761,
        "(0, 3, 3)": 1.941,
        "(1, 0, 0)": 2.745,
        "(1, 0, 1)": 1.941,
        "(1, 0, 2)": 1.228,
        "(1, 0, 3)": 0.868,
        "(1, 1, 0)": 1.941,
        "(1, 1, 1)": 3.17,
        "(1, 1, 2)": 1.121,
        "(1, 1, 3)": 1.655,
        "(1, 2, 0)": 1.228,
        "(1, 2, 1)": 1.121,
        "(1, 2, 2)": 0.915,
        "(1, 2, 3)": 0.734,
        "(1, 3, 0)": 0.868,
        "(1, 3, 1)": 1.655,
        "(1, 3, 2)": 0.734,
        "(1, 3, 3)": 1.259,
        "(2, 0, 0)": 2.745,
        "(2, 0, 1)": 1.228,
        "(2, 0, 2)": 1.941,
        "(2, 0, 3)": 0.761,
        "(2, 1, 0)": 1.228,
        "(2, 1, 1)": 1.121,
        "(2, 1, 2)": 0.915,
        "(2, 1, 3)": 0.734,
        "(2, 2, 0)": 1.941,
        "(2, 2, 1)": 0.915,
        "(2, 2, 2)": 3.17,
        "(2, 2, 3)": 0.666,
        "(2, 3, 0)": 0.761,
        "(2, 3, 1)": 0.734,
        "(2, 3, 2)": 0.666,
        "(2, 3, 3)": 0.585,
        "(3, 0, 0)": 2.745,
        "(3, 0, 1)": 0.868,
        "(3, 0, 2)": 0.761,
        "(3, 0, 3)": 1.941,
        "(3, 1, 0)": 0.868,
        "(3, 1, 1)": 1.655,
        "(3, 1, 2)": 0.734,
        "(3, 1, 3)": 1.259,
        "(3, 2, 0)": 0.761,
        "(3, 2, 1)": 0.734,
        "(3, 2, 2)": 0.666,
        "(3, 2, 3)": 0.585,
        "(3, 3, 0)": 1.941,
        "(3, 3, 1)": 1.259,
        "(3, 3, 2)": 0.585,
        "(3, 3, 3)": 3.17,
    },
    "Zn": {
        "(0, 0, 1)": 2.468,
        "(0, 0, 2)": 2.468,
        "(0, 0, 3)": 2.468,
        "(0, 1, 0)": 1.536,
        "(0, 1, 1)": 0.348,
        "(0, 1, 2)": 1.123,
        "(0, 1, 3)": 0.223,
        "(0, 2, 0)": 1.536,
        "(0, 2, 1)": 0.935,
        "(0, 2, 2)": 0.348,
        "(0, 2, 3)": 0.786,
        "(0, 3, 0)": 1.536,
        "(0, 3, 1)": 0.379,
        "(0, 3, 2)": 0.733,
        "(0, 3, 3)": 0.348,
        "(1, 0, 0)": 0.768,
        "(1, 0, 1)": 1.74,
        "(1, 0, 2)": 0.561,
        "(1, 0, 3)": 1.116,
        "(1, 1, 0)": 1.33,
        "(1, 1, 1)": 0.642,
        "(1, 1, 2)": 1.171,
        "(1, 1, 3)": 0.517,
        "(1, 2, 0)": 0.58,
        "(1, 2, 1)": 0.143,
        "(1, 2, 2)": 0.547,
        "(1, 2, 3)": 0.128,
        "(1, 3, 0)": 0.213,
        "(1, 3, 1)": 0.528,
        "(1, 3, 2)": 0.206,
        "(1, 3, 3)": 0.496,
        "(2, 0, 0)": 0.768,
        "(2, 0, 1)": 0.187,
        "(2, 0, 2)": 1.74,
        "(2, 0, 3)": 0.157,
        "(2, 1, 0)": 0.29,
        "(2, 1, 1)": 0.715,
        "(2, 1, 2)": 0.274,
        "(2, 1, 3)": 0.641,
        "(2, 2, 0)": 1.33,
        "(2, 2, 1)": 0.33,
        "(2, 2, 2)": 0.642,
        "(2, 2, 3)": 0.308,
        "(2, 3, 0)": 0.352,
        "(2, 3, 1)": 0.525,
        "(2, 3, 2)": 0.345,
        "(2, 3, 3)": 0.503,
        "(3, 0, 0)": 0.768,
        "(3, 0, 1)": 0.379,
        "(3, 0, 2)": 0.733,
        "(3, 0, 3)": 1.74,
        "(3, 1, 0)": 0.426,
        "(3, 1, 1)": 0.106,
        "(3, 1, 2)": 0.412,
        "(3, 1, 3)": 0.596,
        "(3, 2, 0)": 0.176,
        "(3, 2, 1)": 0.438,
        "(3, 2, 2)": 0.172,
        "(3, 2, 3)": 0.419,
        "(3, 3, 0)": 1.33,
        "(3, 3, 1)": 0.221,
        "(3, 3, 2)": 0.436,
        "(3, 3, 3)": 0.642,
    },
    "Zr": {
        "(0, 0, 1)": 2.573,
        "(0, 0, 2)": 2.573,
        "(0, 0, 3)": 2.573,
        "(0, 1, 0)": 1.865,
        "(0, 1, 1)": 0.41,
        "(0, 1, 2)": 1.262,
        "(0, 1, 3)": 0.244,
        "(0, 2, 0)": 1.865,
        "(0, 2, 1)": 1.125,
        "(0, 2, 2)": 0.41,
        "(0, 2, 3)": 0.903,
        "(0, 3, 0)": 1.865,
        "(0, 3, 1)": 0.459,
        "(0, 3, 2)": 0.877,
        "(0, 3, 3)": 0.41,
        "(1, 0, 0)": 0.932,
        "(1, 0, 1)": 2.048,
        "(1, 0, 2)": 0.631,
        "(1, 0, 3)": 1.218,
        "(1, 1, 0)": 1.615,
        "(1, 1, 1)": 0.77,
        "(1, 1, 2)": 1.368,
        "(1, 1, 3)": 0.588,
        "(1, 2, 0)": 0.705,
        "(1, 2, 1)": 0.173,
        "(1, 2, 2)": 0.652,
        "(1, 2, 3)": 0.15,
        "(1, 3, 0)": 0.259,
        "(1, 3, 1)": 0.639,
        "(1, 3, 2)": 0.248,
        "(1, 3, 3)": 0.589,
        "(2, 0, 0)": 0.932,
        "(2, 0, 1)": 0.225,
        "(2, 0, 2)": 2.048,
        "(2, 0, 3)": 0.181,
        "(2, 1, 0)": 0.352,
        "(2, 1, 1)": 0.863,
        "(2, 1, 2)": 0.326,
        "(2, 1, 3)": 0.75,
        "(2, 2, 0)": 1.615,
        "(2, 2, 1)": 0.399,
        "(2, 2, 2)": 0.77,
        "(2, 2, 3)": 0.365,
        "(2, 3, 0)": 0.428,
        "(2, 3, 1)": 0.106,
        "(2, 3, 2)": 0.415,
        "(2, 3, 3)": 0.1,
        "(3, 0, 0)": 0.932,
        "(3, 0, 1)": 0.459,
        "(3, 0, 2)": 0.877,
        "(3, 0, 3)": 2.048,
        "(3, 1, 0)": 0.517,
        "(3, 1, 1)": 0.128,
        "(3, 1, 2)": 0.495,
        "(3, 1, 3)": 0.118,
        "(3, 2, 0)": 0.214,
        "(3, 2, 1)": 0.531,
        "(3, 2, 2)": 0.208,
        "(3, 2, 3)": 0.501,
        "(3, 3, 0)": 1.615,
        "(3, 3, 1)": 0.268,
        "(3, 3, 2)": 0.527,
        "(3, 3, 3)": 0.77,
    },
}
