# -*- coding: utf-8 -*-
from distutils.core import setup

packages = \
['poetry',
 'poetry.console',
 'poetry.console.commands',
 'poetry.console.commands.debug',
 'poetry.console.styles',
 'poetry.installation',
 'poetry.io',
 'poetry.json',
 'poetry.layouts',
 'poetry.masonry',
 'poetry.masonry.builders',
 'poetry.masonry.publishing',
 'poetry.masonry.utils',
 'poetry.mixology',
 'poetry.mixology.contracts',
 'poetry.mixology.graph',
 'poetry.packages',
 'poetry.packages.constraints',
 'poetry.packages.utils',
 'poetry.puzzle',
 'poetry.puzzle.operations',
 'poetry.repositories',
 'poetry.semver',
 'poetry.semver.constraints',
 'poetry.toml',
 'poetry.toml.prettify',
 'poetry.toml.prettify.elements',
 'poetry.toml.prettify.elements.traversal',
 'poetry.toml.prettify.lexer',
 'poetry.toml.prettify.parser',
 'poetry.toml.prettify.tokens',
 'poetry.utils',
 'poetry.vcs',
 'poetry.version']

package_data = \
{'': ['*'], 'poetry.json': ['schemas/*']}

install_requires = \
['cleo (>=0.6.0.0,<0.7.0.0)',
 'requests (>=2.18.0.0,<3.0.0.0)',
 'toml (>=0.9.0.0,<0.10.0.0)',
 'cachy (>=0.1.0.0,<0.2.0.0)',
 'pip-tools (>=1.11.0.0,<2.0.0.0)',
 'requests-toolbelt (>=0.8.0.0,<0.9.0.0)',
 'jsonschema (>=2.6.0.0,<3.0.0.0)',
 'pyrsistent (>=0.14.2.0,<0.15.0.0)',
 'pyparsing (>=2.2.0.0,<3.0.0.0)']

entry_points = \
{'console_scripts': ['poetry = poetry:console.run']}

setup_kwargs = {
    'name': 'poetry',
    'version': '0.6.3b2',
    'description': 'Python dependency management and packaging made easy.',
    'long_description': '# Poetry: Dependency Management for Python\n\n![Poetry build status](https://travis-ci.org/sdispater/poetry.svg)\n\nPoetry helps you declare, manage and install dependencies of Python projects,\nensuring you have the right stack everywhere.\n\n![Poet Install](https://raw.githubusercontent.com/sdispater/poetry/master/assets/install.gif)\n\nThe package is **highly experimental** at the moment so expect things to change and break.\nHowever, if you feel adventurous feedback and pull requests are greatly appreciated.\n\nAlso, be aware that the features described here are the goal that this library is aiming\nfor and, as of now, not all of them are implemented (but most of them are).\nThe dependency management is done while the packaging and publishing are still at the beginning.\n\nAnd finally, Poetry\'s code is only compatible with Python 3.6+ but it can manage\nPython project\'s with previous versions without any problem.\n\n## Installation\n\n```bash\npip install poetry\n```\n\n### Enable tab completion for Bash, Fish, or Zsh\n\n`poetry` supports generating completion scripts for Bash, Fish, and Zsh.\nSee `poet help completions` for full details, but the gist is as simple as using one of the following:\n\n```bash\n# Bash\npoetry completions bash > /etc/bash_completion.d/pyproject.bash-completion\n\n# Bash (macOS/Homebrew)\npoetry completions bash > $(brew --prefix)/etc/bash_completion.d/pyproject.bash-completion\n\n# Fish\npoetry completions fish > ~/.config/fish/completions/pyproject.fish\n\n# Zsh\npoetry completions zsh > ~/.zfunc/_poetry\n```\n\n*Note:* you may need to restart your shell in order for the changes to take \neffect.\n\nFor `zsh`, you must then add the following line in your `~/.zshrc` before\n`compinit`:\n\n```zsh\nfpath+=~/.zfunc\n```\n\n\n## Introduction\n\n`poetry` is a tool to handle dependencies installation, building and packaging of Python packages.\nIt only needs one file to do all of that: the new, [standardized](https://www.python.org/dev/peps/pep-0518/) `pyproject.toml`.\n\n```toml\n[tool.poetry]\nname = "my-package"\nversion = "0.1.0"\ndescription = "The description of the package"\n\nlicense = "MIT"\n\nauthors = [\n    "Sébastien Eustace <sebastien@eustace.io>"\n]\n\nreadme = \'README.md\'  # Markdown files are supported\n\nrepository = "https://github.com/sdispater/poetry"\nhomepage = "https://github.com/sdispater/poetry"\n\nkeywords = [\'packaging\', \'poetry\']\n\n[tool.poetry.dependencies]\npython = "~2.7 || ^3.2"  # Compatible python versions must be declared here\ntoml = "^0.9"\n# Dependencies with extras\nrequests = { version = "^2.13", extras = [ "security" ] }\n# Python specific dependencies with prereleases allowed\npathlib2 = { version = "^2.2", python = "~2.7", allows_prereleases = true }\n# Git dependencies\ncleo = { git = "https://github.com/sdispater/cleo.git", branch = "master" }\n\n# Optional dependencies (extras)\npendulum = { version = "^1.4", optional = true }\n\n[tool.poetry.dev-dependencies]\npytest = "^3.0"\npytest-cov = "^2.4"\n\n[tool.poetry.scripts]\nmy-script = \'my_package:main\'\n```\n\nThere are some things we can notice here:\n\n* It will try to enforce [semantic versioning](<http://semver.org>) as the best practice in version naming.\n* You can specify the readme, included and excluded files: no more `MANIFEST.in`.\n`poetry` will also use VCS ignore files (like `.gitignore`) to populate the `exclude` section.\n* Keywords (up to 5) can be specified and will act as tags on the packaging site.\n* The dependencies sections support caret, tilde, wildcard, inequality and multiple requirements.\n* You must specify the python versions for which your package is compatible.\n\n`poetry` will also detect if you are inside a virtualenv and install the packages accordingly.\nSo, `poetry` can be installed globally and used everywhere.\n\n`poetry` also comes with a full fledged dependency resolution library, inspired by [Molinillo](https://github.com/CocoaPods/Molinillo).\n\n## Why?\n\nPackaging system and dependency management in Python is rather convoluted and hard to understand for newcomers.\nEven for seasoned developers it might be cumbersome at times to create all files needed in a Python project: `setup.py`,\n`requirements.txt`, `setup.cfg`, `MANIFEST.in` and the newly added `Pipfile`.\n\nSo I wanted a tool that would limit everything to a single configuration file to do:\ndependency management, packaging and publishing.\n\nIt takes inspiration in tools that exist in other languages, like `composer` (PHP) or `cargo` (Rust).\n\nAnd, finally, there is no reliable tool to properly resolves dependencies in Python, so I started `poetry`\nto bring an exhaustive depency resolver to the Python community.\n\n### What about Pipenv?\n\nIn short: I do not like the CLI it provides, or some of the decisions made,\nand I think we can do a better and more intuitive one. Here are a few things\nthat I don\'t like.\n\n#### Dependency resolution\n\nThe dependency resolution is erratic and will fail even is there is a solution. Let\'s take an example:\n\n```bash\npipenv install oslo.utils==1.4.0\n```\n\nwill fail with this error:\n\n```text\nCould not find a version that matches pbr!=0.7,!=2.1.0,<1.0,>=0.6,>=2.0.0\n```\n\nwhile Poetry will get you the right set of packages:\n\n```bash\npoetry add oslo.utils=1.4.0\n```\n\nresults in :\n\n```text\n  - Installing pytz (2018.3)\n  - Installing netifaces (0.10.6)\n  - Installing netaddr (0.7.19)\n  - Installing oslo.i18n (2.1.0)\n  - Installing iso8601 (0.1.12)\n  - Installing six (1.11.0)\n  - Installing babel (2.5.3)\n  - Installing pbr (0.11.1)\n  - Installing oslo.utils (1.4.0)\n```\n\n#### Install command\n\nWhen you specify a package to the `install` command it will add it as a wildcard\ndependency. This means that **any** version of this package can be installed which\ncan lead to compatibility issues.\n\nAlso, you have to explicitely tell it to not update the locked packages when you\ninstalled new ones. This should be the default.\n\n#### Remove command\n\nThe `remove` command will only remove the package specified but not its dependencies\nif they are no longer needed.\n\nYou either have to use `sync` or `clean` to fix that.\n\n#### Too limited in scope\n\nFinally, the `Pipfile` is just a replacement from `requirements.txt` and, in the end, you will still need to \npopulate your `setup.py` file (or `setup.cfg`) with the exact same dependencies you declared in your `Pipfile`.\nSo, in the end, you will still need to manage a few configuration files to properly setup your project.\n\n\n## Commands\n\n\n### new\n\nThis command will help you kickstart your new Python project by creating\na directory structure suitable for most projects.\n\n```bash\npoetry new my-package\n```\n\nwill create a folder as follows:\n\n```text\nmy-package\n├── pyproject.toml\n├── README.rst\n├── my_package\n    └── __init__.py\n└── tests\n    ├── __init__.py\n    └── test_my_package\n```\n\nIf you want to name your project differently than the folder, you can pass\nthe `--name` option:\n\n```bash\npoetry new my-folder --name my-package\n```\n\n\n### install\n\nThe `install` command reads the `pyproject.toml` file from the current directory, resolves the dependencies,\nand installs them.\n\n```bash\npoetry install\n```\n\nIf there is a `pyproject.lock` file in the current directory,\nit will use the exact versions from there instead of resolving them.\nThis ensures that everyone using the library will get the same versions of the dependencies.\n\nIf there is no `pyproject.lock` file, Poetry will create one after dependency resolution.\n\nYou can specify to the command that you do not want the development dependencies installed by passing\nthe `--no-dev` option.\n\n```bash\npoetry install --no-dev\n```\n\nYou can also specify the extras you want installed\nby passing the `--E|--extras` option (See [Extras](#extras) for more info)\n\n```bash\npoetry install --extras "mysql pgsql"\npoetry install -E mysql -E pgsql\n```\n\n#### Options\n\n* `--no-dev`: Do not install dev dependencies.\n* `-E|--extras`: Features to install (multiple values allowed).\n\n### update\n\nIn order to get the latest versions of the dependencies and to update the `pyproject.lock` file,\nyou should use the `update` command.\n\n```bash\npoetry update\n```\n\nThis will resolve all dependencies of the project and write the exact versions into `pyproject.lock`.\n\nIf you just want to update a few packages and not all, you can list them as such:\n\n```bash\npoetry update requests toml\n```\n\n#### Options\n\n* `--dry-run` : Outputs the operations but will not execute anything (implicitly enables --verbose).\n\n### add\n\nThe `add` command adds required packages to your `pyproject.toml` and installs them.\n\nIf you do not specify a version constraint,\npoetry will choose a suitable one based on the available package versions.\n\n```bash\npoetry add requests pendulum\n```\n\n#### Options\n\n* `--D|dev`: Add package as development dependency.\n* `--optional` : Add as an optional dependency.\n* `--dry-run` : Outputs the operations but will not execute anything (implicitly enables --verbose).\n\n\n### remove\n\nThe `remove` command removes a package from the current\nlist of installed packages\n\n```bash\npoetry remove pendulum\n```\n\n#### Options\n\n* `--D|dev`: Removes a package from the development dependencies.\n* `--dry-run` : Outputs the operations but will not execute anything (implicitly enables --verbose).\n\n\n### show\n\nTo list all of the available packages, you can use the `show` command.\n\n```bash\npoetry show\n```\n\nIf you want to see the details of a certain package, you can pass the package name.\n\n```bash\npoetry show pendulum\n\nname        : pendulum\nversion     : 1.4.2\ndescription : Python datetimes made easy\n\ndependencies:\n - python-dateutil >=2.6.1\n - tzlocal >=1.4\n - pytzdata >=2017.2.2\n```\n\n#### Options\n\n* `--tree`: List the dependencies as a tree.\n* `-l|--latest`: Show the latest version.\n*  `-o|--outdated`: Show the latest version but only for packages that are outdated.\n\n\n### build\n\nThe `build` command builds the source and wheels archives.\n\n```bash\npoetry build\n```\n\nNote that, at the moment, only pure python wheels are supported.\n\n#### Options\n\n* `-F|--format`: Limit the format to either wheel or sdist.\n\n### publish\n\nThis command builds (if not already built) and publishes the package to the remote repository.\n\nIt will automatically register the package before uploading if this is the first time it is submitted.\n\n```bash\npoetry publish\n```\n\n#### Options\n\n* `-r|--repository`: The repository to register the package to (default: `pypi`).\nShould match a repository name set by the [`config`](#config) command.\n\n\n### `config`\n\nThe `config` command allows you to edit poetry config settings and repositories.\n\n```bash\npoetry config --list\n```\n\n#### Usage\n\n````bash\npoetry config [options] [setting-key] [setting-value1] ... [setting-valueN]\n````\n\n`setting-key` is a configuration option name and `setting-value1` is a configuration value.\n\n#### Modifying repositories\n\nIn addition to modifying the config section,\nthe config command also supports making changes to the repositories section by using it the following way:\n\n```bash\npoetry config repositories.foo https://foo.bar/simple/\n```\n\nThis will set the url for repository `foo` to `https://foo.bar/simple/`.\n\nIf you want to store your credentials for a specific repository, you can do so easily:\n\n```bash\npoetry config http-basic.foo username password\n```\n\nIf you do not specify the password you will be prompted to write it.\n\n#### Options\n\n* `--unset`: Remove the configuration element named by `setting-key`.\n* `--list`: Show the list of current config variables.\n\n### search\n\nThis command searches for packages on a remote index.\n\n```bash\npoetry search requests pendulum\n```\n\n#### Options\n\n* `-N|--only-name`: Search only in name.\n\n### lock\n\nThis command locks (without installing) the dependencies specified in `pyproject.toml`.\n\n```bash\npoetry lock\n```\n\n\n## The `pyproject.toml` file\n\nThe `tool.poetry` section of the `pyproject.toml` file is composed of multiple sections.\n\n#### name\n\nThe name of the package. **Required**\n\n#### version\n\nThe version of the package. **Required**\n\nThis should follow [semantic versioning](http://semver.org/). However it will not be enforced and you remain\nfree to follow another specification.\n\n#### description\n\nA short description of the package. **Required**\n\n#### license\n\nThe license of the package.\n\nThe recommended notation for the most common licenses is (alphabetical):\n\n* Apache-2.0\n* BSD-2-Clause\n* BSD-3-Clause\n* BSD-4-Clause\n* GPL-2.0\n* GPL-2.0+\n* GPL-3.0\n* GPL-3.0+\n* LGPL-2.1\n* LGPL-2.1+\n* LGPL-3.0\n* LGPL-3.0+\n* MIT\n\nOptional, but it is highly recommended to supply this.\nMore identifiers are listed at the [SPDX Open Source License Registry](https://www.spdx.org/licenses/).\n\n#### authors\n\nThe authors of the package. This is a list of authors and should contain at least one author.\n\nAuthors must be in the form `name <email>`.\n\n#### readme\n\nThe readme file of the package. **Required**\n\nThe file can be either `README.rst` or `README.md`.\nIf it\'s a markdown file you have to install the [pandoc](https://github.com/jgm/pandoc) utility so that it can be automatically\nconverted to a RestructuredText file.\n\nYou also need to have the [pypandoc](https://pypi.python.org/pypi/pypandoc/) package installed. If you install `poet` via\n`pip` you can use the `markdown-readme` extra to do so.\n\n```bash\npip install pypoet[markdown-readme]\n```\n\n#### homepage\n\nAn URL to the website of the project. **Optional**\n\n#### repository\n\nAn URL to the repository of the project. **Optional**\n\n#### documentation\n\nAn URL to the documentation of the project. **Optional**\n\n#### keywords\n\nA list of keywords (max: 5) that the package is related to. **Optional**\n\n#### include and exclude\n\nA list of patterns that will be included in the final package.\n\nYou can explicitly specify to Poet that a set of globs should be ignored or included for the purposes of packaging.\nThe globs specified in the exclude field identify a set of files that are not included when a package is built.\n\nIf a VCS is being used for a package, the exclude field will be seeded with the VCS’ ignore settings (`.gitignore` for git for example).\n\n```toml\n[package]\n# ...\ninclude = ["package/**/*.py", "package/**/.c"]\n```\n\n```toml\nexclude = ["package/excluded.py"]\n```\n\n### `dependencies` and `dev-dependencies`\n\nPoetry is configured to look for dependencies on [PyPi](https://pypi.org) by default.\nOnly the name and a version string are required in this case.\n\n```toml\n[tool.poetry.dependencies]\nrequests = "^2.13.0"\n```\n\nIf you want to use a private repository, you can add it to your `pyproject.toml` file, like so:\n\n```toml\n[[tool.poetry.source]]\nname = \'private\'\nurl = \'http://example.com/simple\'\n```\n\nBe aware that declaring the python version for which your package\nis compatible is mandatory:\n\n```toml\n[tool.poetry.dependencies]\npython = "^3.6"\n```\n\n#### Caret requirement\n\n**Caret requirements** allow SemVer compatible updates to a specified version.\nAn update is allowed if the new version number does not modify the left-most non-zero digit in the major, minor, patch grouping.\nIn this case, if we ran `poetry update requests`, poetry would update us to version `2.14.0` if it was available,\nbut would not update us to `3.0.0`.\nIf instead we had specified the version string as `^0.1.13`, poetry would update to `0.1.14` but not `0.2.0`.\n`0.0.x` is not considered compatible with any other version.\n\nHere are some more examples of caret requirements and the versions that would be allowed with them:\n\n```text\n^1.2.3 := >=1.2.3 <2.0.0\n^1.2 := >=1.2.0 <2.0.0\n^1 := >=1.0.0 <2.0.0\n^0.2.3 := >=0.2.3 <0.3.0\n^0.0.3 := >=0.0.3 <0.0.4\n^0.0 := >=0.0.0 <0.1.0\n^0 := >=0.0.0 <1.0.0\n```\n\n#### Tilde requirements\n\n**Tilde requirements** specify a minimal version with some ability to update.\nIf you specify a major, minor, and patch version or only a major and minor version, only patch-level changes are allowed.\nIf you only specify a major version, then minor- and patch-level changes are allowed.\n\n`~1.2.3` is an example of a tilde requirement.\n\n```text\n~1.2.3 := >=1.2.3 <1.3.0\n~1.2 := >=1.2.0 <1.3.0\n~1 := >=1.0.0 <2.0.0\n```\n\n#### Wildcard requirements\n\n**Wildcard requirements** allow for any version where the wildcard is positioned.\n\n`*`, `1.*` and `1.2.*` are examples of wildcard requirements.\n\n```text\n* := >=0.0.0\n1.* := >=1.0.0 <2.0.0\n1.2.* := >=1.2.0 <1.3.0\n```\n\n#### Inequality requirements\n\n**Inequality requirements** allow manually specifying a version range or an exact version to depend on.\n\nHere are some examples of inequality requirements:\n\n```text\n>= 1.2.0\n> 1\n< 2\n!= 1.2.3\n```\n\n#### Multiple requirements\n\nMultiple version requirements can also be separated with a comma, e.g. `>= 1.2, < 1.5`.\n\n#### `git` dependencies\n\nTo depend on a library located in a `git` repository,\nthe minimum information you need to specify is the location of the repository with the git key:\n\n```toml\n[dependencies]\nrequests = { git = "https://github.com/requests/requests.git" }\n```\n\nSince we haven’t specified any other information,\nPoetry assumes that we intend to use the latest commit on the `master` branch to build our project.\nYou can combine the `git` key with the `rev`, `tag`, or `branch` keys to specify something else.\nHere\'s an example of specifying that you want to use the latest commit on a branch named `next`:\n\n```toml\n[dependencies]\nrequests = { git = "https://github.com/kennethreitz/requests.git", branch = "next" }\n```\n\n#### Python restricted dependencies\n\nYou can also specify that a dependency should be installed only for specific Python versions:\n\n```toml\n[dependencies]\npathlib2 = { version = "^2.2", python = "~2.7" }\n```\n\n```toml\n[dependencies]\npathlib2 = { version = "^2.2", python = ["~2.7", "^3.2"] }\n```\n\n### `scripts`\n\nThis section describe the scripts or executable that will be installed when installing the package\n\n```toml\n[scripts]\npoetry = \'poetry:console.run\'\n```\n\nHere, we will have the `poetry` script installed which will execute `console.run` in the `poetry` package.\n\n### `extras`\n\nPoetry supports extras to allow expression of:\n\n* optional dependencies, which enhance a package, but are not required; and\n* clusters of optional dependencies.\n\n```toml\n[tool.poetry]\nname = "awesome"\n\n[tool.poetry.dependencies]\n# These packages are mandatory and form the core of this package’s distribution.\nmandatory = "^1.0"\n\n# A list of all of the optional dependencies, some of which are included in the\n# below `extras`. They can be opted into by apps.\npsycopg2 = { version = "^2.7", optional = true }\nmysqlclient = { version = "^1.3", optional = true }\n\n[tool.poetry.extras]\nmysql = ["mysqlclient"]\npgsql = ["psycopg2"]\n```\n\nWhen installing packages, you can specify extras by using the `-E|--extras` option:\n\n```bash\npoet install --extras "mysql pgsql"\npoet install -E mysql -E pgsql\n```\n\n### `plugins`\n\nPoetry supports arbitrary plugins wich work similarly to\n[setuptools entry points](http://setuptools.readthedocs.io/en/latest/setuptools.html).\nTo match the example in the setuptools documentation, you would use the following:\n\n```toml\n[plugins] # Optional super table\n\n[plugins."blogtool.parsers"]\n".rst" = "some_module::SomeClass"\n```\n\n## Resources\n\n* [Official Website](https://poetry.eustace.io)\n* [Issue Tracker](https://github.com/sdispater/poetry/issues)\n',
    'author': 'Sébastien Eustace',
    'author_email': 'sebastien@eustace.io',
    'url': 'https://poetry.eustace.io/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>= 3.6.0.0, < 4.0.0.0',
}


setup(**setup_kwargs)
