# Modifications Copyright(C)[2025] Advanced Micro Devices, Inc. All rights reserved.
# https://github.com/thunlp/TritonBench - Apache License 2.0
import math
from typing import Union
import torch
import triton
from triton import language as tl

# try:
#     from triton.language.extra.cuda.libdevice import isfinited as _isfinited
# except ImportError:
#     try:
#         from triton.language.math import isfinited as _isfinited
#     except ImportError:
#         from triton.language.libdevice import isfinited as _isfinited

from triton.language.extra.hip.libdevice import isinf, isnan

def _isfinited(x):
    """Check if a tensor is finite."""
    return ~isinf(x) & ~isnan(x)
    # if x.dtype.is_fp64():
    # else:
    #     return ~isinf(x.to(tl.float32)) & ~isnan(x.to(tl.float32))

# try:
#     from triton.language.extra.cuda.libdevice import finitef as _finitef
# except ImportError:
#     try:
#         from triton.language.math import finitef as _finitef
#     except ImportError:
#         from triton.language.libdevice import finitef as _finitef

def heuristics_for_tile_size(max_tile_size, *sizes):
    ndim = len(sizes)
    tile_sizes = [0 for _ in range(ndim)]
    for i in range(ndim):
        size = sizes[ndim - 1 - i]
        tile_size = min(max_tile_size, triton.next_power_of_2(size))
        tile_sizes[ndim - 1 - i] = tile_size
        max_tile_size = max(1, max_tile_size // tile_size)
    return tuple(tile_sizes)

def heuristics_for_num_warps(tile_size):
    if tile_size < 2048:
        return 4
    elif tile_size < 4096:
        return 8
    else:
        return 16
    

class StridedBuffer:
    """A drop-in replacement of torch.Tensor that can be used in wrapper generated by
    PointwiseDynamicFunction. It allows us to use a different shape, stride, data
    pointer that that of the base tensor.

    It is a kind of reinterpretation of the base tensor. We make this class since we
    cannot get a Tensor view with negative strides via torch APIs, while we need this
    to implement flip op.

    Although generated code can accept torch.Tensor & StridedBuffer, but StridedBuffer
    may not have all the methods as torch.Tensors do. We add some attributes & methods
    with the same name as torch.Tensor, which are used in the generated code. But we
    may not cover all the methods, add one if what you need is missing here.

    And can also be used in triton kernels since it also has dtype & data_ptr().
    """

    def __init__(
        self, base: torch.Tensor, shape=None, strides=None, dtype=None, offset=0
    ):
        self._base = base
        self.dtype = dtype or base.dtype
        if offset == 0:
            self._data_ptr = self._base.data_ptr()
        else:
            offset = self.dtype.itemsize * offset
            self._data_ptr = self._base.data_ptr() + offset
        self.shape = tuple(shape if shape is not None else self._base.shape)
        self._strides = tuple(strides if strides is not None else self._base.stride())
        self.device = self._base.device
        self.ndim = len(self.shape)

    def stride(self):
        return self._strides

    def size(self):
        return self.shape

    def element_size(self):
        return self.dtype.itemsize

    def numel(self):
        return math.prod(self.shape)

    def dim(self):
        return self.ndim

    def unwrap(self):
        return self._base

    def data_ptr(self):
        return self._data_ptr


def isfinite_func_wrapper_rank_1(in0: Union[torch.Tensor, StridedBuffer], /, *, out0: Union[torch.Tensor, StridedBuffer]): 
    """Generated wrapper function with Pointwise: StridedBuffer, StridedBuffer(a1!) -> StridedBuffer(a1!)"""
    assert in0.shape == out0.shape, 'operand shapes mismatch'
    # task partitioning
    shape = out0.shape
    num_tasks = out0.numel()
    tile_sizes = heuristics_for_tile_size(512, *shape)
    tile_size = math.prod(tile_sizes)
    num_tiles = math.prod(triton.cdiv(size, tile_size) for size, tile_size in zip(shape, tile_sizes))
    num_ctas = min(65536, num_tiles)
    tiles_per_cta = triton.cdiv(num_tiles, num_ctas)
    num_warps = heuristics_for_num_warps(tile_size)
    one_tile_per_cta = tiles_per_cta==1
    grid = (num_ctas, 1, 1)
    # kernel launch
    in0_strides = in0.stride()
    in0_stride_order = (0,)
    out0_strides = out0.stride()
    out0_stride_order = (0,)
    with torch.cuda._DeviceGuard(in0.device.index):
        isfinite_func_kernel_rank_1[grid](
            in0, out0,
            in0_strides[0], # stride for in0
            in0_stride_order[0], # stride order for in0
            out0_strides[0], # stride for out0
            out0_stride_order[0], # stride orderfor out0
            shape[0], # task indexing space
            num_tasks, # num tasks
            tiles_per_cta=tiles_per_cta, # tiles_per_cta
            tile_size0=tile_sizes[0],
            one_tile_per_cta=one_tile_per_cta,
            num_warps=num_warps,
        )
    return out0

@triton.jit
def isfinite_func(x):
    return ~isinf(x) & ~isnan(x) #_isfinited(x) #if x.dtype.is_fp64() else _finitef(x.to(tl.float32))

@triton.jit
def isfinite_func_kernel_rank_1(
    in0_ptr: tl.tensor, # of tl.pointer_type
    out0_ptr: tl.tensor, # of tl.pointer_type
    in0_stride0: int, # strides for in0
    in0_stride_order0: tl.constexpr, # stride order for in0
    out0_stride0: int, # strides for out0
    out0_stride_order0: tl.constexpr, # stride order for out0
    s0: int, # task_space
    num_tasks: int,
    tiles_per_cta: int,
    tile_size0: tl.constexpr,
    one_tile_per_cta: tl.constexpr,
):
    pid = tl.program_id(0)
    num_tiles0 = tl.cdiv(s0, tile_size0)
    if one_tile_per_cta: # monolitic kernel style
        tile_id = pid
        # pid multi index recontruction: we use c ordering, right axes changes fastest
        tile_id0 = tile_id

        # tile offsets
        offset0 = tile_id0 * tile_size0
        # loads
        in0_bptr = tl.make_block_ptr(in0_ptr, (s0,), (in0_stride0,), (offset0,), (tile_size0,), order=(in0_stride_order0,))
        in0 = tl.load(in0_bptr, boundary_check=(in0_stride_order0,)).to(in0_ptr.type.element_ty) # workaround the bug on bool, we should use the original pointer's dtype(instead of block pointer's)

        # compute
        out0 = isfinite_func(in0)

        # stores, note that store to block pointer does not automatically cast the value to the pointer's dtype
        out0_bptr = tl.make_block_ptr(out0_ptr, (s0,), (out0_stride0,), (offset0,), (tile_size0,), order=(out0_stride_order0,))
        tl.store(out0_bptr, out0.to(out0_bptr.type.element_ty), boundary_check=(out0_stride_order0,))
    else: # grid-stride-loop style kernel
        num_ctas = tl.num_programs(0)
        for j in range(0, tiles_per_cta):
            tile_id = pid + j * num_ctas
            # pid multi index recontruction: we use c ordering, right axes changes fastest
            tile_id0 = tile_id

            # tile offsets
            offset0 = tile_id0 * tile_size0
            # loads
            in0_bptr = tl.make_block_ptr(in0_ptr, (s0,), (in0_stride0,), (offset0,), (tile_size0,), order=(in0_stride_order0,))
            in0 = tl.load(in0_bptr, boundary_check=(in0_stride_order0,)).to(in0_ptr.type.element_ty) # workaround the bug on bool, we should use the original pointer's dtype(instead of block pointer's)

            # compute
            out0 = isfinite_func(in0)

            # stores, note that store to block pointer does not automatically cast the value to the pointer's dtype
            out0_bptr = tl.make_block_ptr(out0_ptr, (s0,), (out0_stride0,), (offset0,), (tile_size0,), order=(out0_stride_order0,))
            tl.store(out0_bptr, out0.to(out0_bptr.type.element_ty), boundary_check=(out0_stride_order0,))




##################################################################################################################################################


def test_isfinite_func_wrapper_rank_1():
    results = {}

    # Test case 1: float32, single CTA
    in_tensor_1 = torch.randn(512, dtype=torch.float32, device='cuda')  # shape (512,)
    out_tensor_1 = torch.empty_like(in_tensor_1, dtype=torch.bool)
    isfinite_func_wrapper_rank_1(in_tensor_1, out0=out_tensor_1)
    results['test_case_1'] = out_tensor_1

    # Test case 2: float32, multiple CTAs
    in_tensor_2 = torch.randn(4096, dtype=torch.float32, device='cuda')  # shape (4096,)
    out_tensor_2 = torch.empty_like(in_tensor_2, dtype=torch.bool)
    isfinite_func_wrapper_rank_1(in_tensor_2, out0=out_tensor_2)
    results['test_case_2'] = out_tensor_2

    # Test case 3: float64, single CTA
    in_tensor_3 = torch.randn(512, dtype=torch.float64, device='cuda')  # shape (512,)
    out_tensor_3 = torch.empty_like(in_tensor_3, dtype=torch.bool)
    isfinite_func_wrapper_rank_1(in_tensor_3, out0=out_tensor_3)
    results['test_case_3'] = out_tensor_3

    # Test case 4: float64, multiple CTAs
    in_tensor_4 = torch.randn(4096, dtype=torch.float64, device='cuda')  # shape (4096,)
    out_tensor_4 = torch.empty_like(in_tensor_4, dtype=torch.bool)
    isfinite_func_wrapper_rank_1(in_tensor_4, out0=out_tensor_4)
    results['test_case_4'] = out_tensor_4

    # Test case 5: float32 tensor with infinities
    in_tensor_5 = torch.tensor([float('inf'), -float('inf'), 1.0, 2.0, float('nan')], dtype=torch.float32, device='cuda')
    out_tensor_5 = torch.empty_like(in_tensor_5, dtype=torch.bool)
    isfinite_func_wrapper_rank_1(in_tensor_5, out0=out_tensor_5)
    results['test_case_5'] = out_tensor_5

    # Test case 6: float64 tensor with infinities
    in_tensor_6 = torch.tensor([float('inf'), -float('inf'), 1.0, 2.0, float('nan')], dtype=torch.float64, device='cuda')
    out_tensor_6 = torch.empty_like(in_tensor_6, dtype=torch.bool)
    isfinite_func_wrapper_rank_1(in_tensor_6, out0=out_tensor_6)
    results['test_case_6'] = out_tensor_6

    # Test case 7: float32 tensor with mixed extreme and normal values
    in_tensor_7 = torch.cat([torch.tensor([float('inf'), -float('inf')], dtype=torch.float32, device='cuda'),
                            torch.randn(510, dtype=torch.float32, device='cuda')])
    out_tensor_7 = torch.empty_like(in_tensor_7, dtype=torch.bool)
    isfinite_func_wrapper_rank_1(in_tensor_7, out0=out_tensor_7)
    results['test_case_7'] = out_tensor_7

    # Test case 8: float64 tensor with mixed extreme and normal values
    in_tensor_8 = torch.cat([torch.tensor([float('inf'), -float('inf')], dtype=torch.float64, device='cuda'),
                            torch.randn(510, dtype=torch.float64, device='cuda')])
    out_tensor_8 = torch.empty_like(in_tensor_8, dtype=torch.bool)
    isfinite_func_wrapper_rank_1(in_tensor_8, out0=out_tensor_8)
    results['test_case_8'] = out_tensor_8

    # Test case 9: float32 tensor with NaN and finite values
    in_tensor_9 = torch.tensor([float('nan'), 0.0, -1.0, float('nan'), 3.14], dtype=torch.float32, device='cuda')
    out_tensor_9 = torch.empty_like(in_tensor_9, dtype=torch.bool)
    isfinite_func_wrapper_rank_1(in_tensor_9, out0=out_tensor_9)
    results['test_case_9'] = out_tensor_9

    # Test case 10: float64 tensor with NaN and finite values
    in_tensor_10 = torch.tensor([float('nan'), 0.0, -1.0, float('nan'), 3.14], dtype=torch.float64, device='cuda')
    out_tensor_10 = torch.empty_like(in_tensor_10, dtype=torch.bool)
    isfinite_func_wrapper_rank_1(in_tensor_10, out0=out_tensor_10)
    results['test_case_10'] = out_tensor_10

    return results

result_gold = test_isfinite_func_wrapper_rank_1()
