#
# Copyright 2023 by Delphix
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
Package "repository"
"""
from delphixpy.v1_11_24.web.repository import template
from urllib.parse import urlencode
from delphixpy.v1_11_24 import response_validator

def create(engine, source_repository=None):
    """
    Create a new SourceRepository object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param source_repository: Payload object.
    :type source_repository: :py:class:`v1_11_24.web.vo.SourceRepository`
    :rtype: ``str``
    """
    url = "/resources/json/delphix/repository"
    response = engine.post(url, source_repository.to_dict(dirty=True) if source_repository else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['str'], returns_list=False, raw_result=raw_result)

def get(engine, ref):
    """
    Retrieve the specified SourceRepository object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param ref: Reference to a :py:class:`delphixpy.v1_11_24.web.objects.Source
        Repository.SourceRepository` object
    :type ref: ``str``
    :rtype: :py:class:`v1_11_24.web.vo.SourceRepository`
    """
    url = "/resources/json/delphix/repository/%s" % ref
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['SourceRepository'], returns_list=False, raw_result=raw_result)

def get_all(engine, environment=None):
    """
    Returns a list of source repositories within the environment or the system.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param environment: Restrict source repositories belong to the specified
        environment.
    :type environment: ``str``
    :rtype: ``list`` of :py:class:`v1_11_24.web.vo.SourceRepository`
    """
    url = "/resources/json/delphix/repository"
    query_params = {"environment": environment}
    query_dct = {k: query_params[k] for k in query_params if query_params[k] is not None}
    if query_dct:
        url_params = urlencode(query_dct)
        url += "?%s" % url_params
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['SourceRepository'], returns_list=True, raw_result=raw_result)

def update(engine, ref, source_repository=None):
    """
    Update the specified SourceRepository object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param ref: Reference to a :py:class:`delphixpy.v1_11_24.web.objects.Source
        Repository.SourceRepository` object
    :type ref: ``str``
    :param source_repository: Payload object.
    :type source_repository: :py:class:`v1_11_24.web.vo.SourceRepository`
    """
    url = "/resources/json/delphix/repository/%s" % ref
    response = engine.post(url, source_repository.to_dict(dirty=True) if source_repository else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def delete(engine, ref):
    """
    Delete the specified SourceRepository object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param ref: Reference to a :py:class:`delphixpy.v1_11_24.web.objects.Source
        Repository.SourceRepository` object
    :type ref: ``str``
    """
    url = "/resources/json/delphix/repository/%s/delete" % ref
    response = engine.post(url, None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def compatible_repositories(engine, compatible_repositories_parameters):
    """
    Returns a list of repositories that match the specified parameters.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param compatible_repositories_parameters: Payload object.
    :type compatible_repositories_parameters:
        :py:class:`v1_11_24.web.vo.CompatibleRepositoriesParameters`
    :rtype: :py:class:`v1_11_24.web.vo.CompatibleRepositoriesResult`
    """
    url = "/resources/json/delphix/repository/compatibleRepositories"
    response = engine.post(url, compatible_repositories_parameters.to_dict(dirty=True) if compatible_repositories_parameters else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['CompatibleRepositoriesResult'], returns_list=False, raw_result=raw_result)

