#
# Copyright 2023 by Delphix
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
Package "selfservice.bookmark"
"""
from urllib.parse import urlencode
from delphixpy.v1_11_24 import response_validator

def create(engine, js_bookmark_create_parameters):
    """
    Create a new JSBookmark object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param js_bookmark_create_parameters: Payload object.
    :type js_bookmark_create_parameters:
        :py:class:`v1_11_24.web.vo.JSBookmarkCreateParameters`
    :rtype: ``str``
    """
    url = "/resources/json/delphix/selfservice/bookmark"
    response = engine.post(url, js_bookmark_create_parameters.to_dict(dirty=True) if js_bookmark_create_parameters else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['str'], returns_list=False, raw_result=raw_result)

def get(engine, ref):
    """
    Retrieve the specified JSBookmark object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_11_24.web.objects.JSBookmark.JSBookmark` object
    :type ref: ``str``
    :rtype: :py:class:`v1_11_24.web.vo.JSBookmark`
    """
    url = "/resources/json/delphix/selfservice/bookmark/%s" % ref
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['JSBookmark'], returns_list=False, raw_result=raw_result)

def get_all(engine, container=None, template=None):
    """
    Lists the Self-Service bookmarks in the system.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param container: List all usable bookmarks accessible to the specified
        data container. This option is mutually exclusive with all other
        options.
    :type container: ``str``
    :param template: List all usable bookmarks created in the specified data
        template. This option is mutually exclusive with all other options.
    :type template: ``str``
    :rtype: ``list`` of :py:class:`v1_11_24.web.vo.JSBookmark`
    """
    url = "/resources/json/delphix/selfservice/bookmark"
    query_params = {"container": container, "template": template}
    query_dct = {k: query_params[k] for k in query_params if query_params[k] is not None}
    if query_dct:
        url_params = urlencode(query_dct)
        url += "?%s" % url_params
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['JSBookmark'], returns_list=True, raw_result=raw_result)

def update(engine, ref, js_bookmark=None):
    """
    Update the specified JSBookmark object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_11_24.web.objects.JSBookmark.JSBookmark` object
    :type ref: ``str``
    :param js_bookmark: Payload object.
    :type js_bookmark: :py:class:`v1_11_24.web.vo.JSBookmark`
    """
    url = "/resources/json/delphix/selfservice/bookmark/%s" % ref
    response = engine.post(url, js_bookmark.to_dict(dirty=True) if js_bookmark else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def delete(engine, ref):
    """
    Delete the specified JSBookmark object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_11_24.web.objects.JSBookmark.JSBookmark` object
    :type ref: ``str``
    """
    url = "/resources/json/delphix/selfservice/bookmark/%s/delete" % ref
    response = engine.post(url, None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def share(engine, ref):
    """
    Shares the bookmark.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_11_24.web.objects.JSBookmark.JSBookmark` object
    :type ref: ``str``
    """
    url = "/resources/json/delphix/selfservice/bookmark/%s/share" % ref
    response = engine.post(url, None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def unshare(engine, ref):
    """
    Unshares the bookmark.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_11_24.web.objects.JSBookmark.JSBookmark` object
    :type ref: ``str``
    """
    url = "/resources/json/delphix/selfservice/bookmark/%s/unshare" % ref
    response = engine.post(url, None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def get_create_bookmark_operation(engine, ref):
    """
    Returns the CREATE_BOOKMARK operation associated with the bookmark.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_11_24.web.objects.JSBookmark.JSBookmark` object
    :type ref: ``str``
    :rtype: :py:class:`v1_11_24.web.vo.JSOperation`
    """
    url = "/resources/json/delphix/selfservice/bookmark/%s/getCreateBookmarkOperation" % ref
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['JSOperation'], returns_list=False, raw_result=raw_result)

def list_data_children(engine, ref):
    """
    Lists the data children of this bookmark.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_11_24.web.objects.JSBookmark.JSBookmark` object
    :type ref: ``str``
    :rtype: ``list`` of :py:class:`v1_11_24.web.vo.JSDataChild`
    """
    url = "/resources/json/delphix/selfservice/bookmark/%s/listDataChildren" % ref
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['JSDataChild'], returns_list=True, raw_result=raw_result)

def list_by_top_checkouts(engine, template=None, max_result_size=None):
    """
    Return bookmarks created in a given template and all of its child data
    containers sorted by checkoutCount. Can optionally limit to the top
    maxResultSize bookmarks.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_24.delphix_engine.DelphixEngine`
    :param template: The given data template.
    :type template: ``str``
    :param max_result_size: The maximum number of results to return. A value of
        0 means all are returned.
    :type max_result_size: ``int``
    :rtype: ``list`` of :py:class:`v1_11_24.web.vo.JSBookmark`
    """
    url = "/resources/json/delphix/selfservice/bookmark/listByTopCheckouts"
    query_params = {"template": template, "maxResultSize": max_result_size}
    query_dct = {k: query_params[k] for k in query_params if query_params[k] is not None}
    if query_dct:
        url_params = urlencode(query_dct)
        url += "?%s" % url_params
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['JSBookmark'], returns_list=True, raw_result=raw_result)

