#
# Copyright 2023 by Delphix
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
Package "service.userPaths"
"""
from urllib.parse import urlencode
from delphixpy.v1_11_6 import response_validator

def create(engine, user_path_storage=None):
    """
    Create a new UserPathStorage object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_6.delphix_engine.DelphixEngine`
    :param user_path_storage: Payload object.
    :type user_path_storage: :py:class:`v1_11_6.web.vo.UserPathStorage`
    :rtype: ``str``
    """
    url = "/resources/json/delphix/service/userPaths"
    response = engine.post(url, user_path_storage.to_dict(dirty=True) if user_path_storage else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['str'], returns_list=False, raw_result=raw_result)

def get(engine, ref):
    """
    Retrieve the specified UserPathStorage object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_6.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_11_6.web.objects.UserPathStorage.UserPathStorag
        e` object
    :type ref: ``str``
    :rtype: :py:class:`v1_11_6.web.vo.UserPathStorage`
    """
    url = "/resources/json/delphix/service/userPaths/%s" % ref
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['UserPathStorage'], returns_list=False, raw_result=raw_result)

def get_all(engine, pathtype=None):
    """
    Returns the list of all the check results that match the given criteria.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_6.delphix_engine.DelphixEngine`
    :param pathtype: Type of path stored. *(permitted values:
        UPGRADE_STAGING_LOC)*
    :type pathtype: ``str``
    :rtype: ``list`` of :py:class:`v1_11_6.web.vo.UserPathStorage`
    """
    url = "/resources/json/delphix/service/userPaths"
    query_params = {"pathtype": pathtype}
    query_dct = {k: query_params[k] for k in query_params if query_params[k] is not None}
    if query_dct:
        url_params = urlencode(query_dct)
        url += "?%s" % url_params
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['UserPathStorage'], returns_list=True, raw_result=raw_result)

def update(engine, ref, user_path_storage=None):
    """
    Update the specified UserPathStorage object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_6.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_11_6.web.objects.UserPathStorage.UserPathStorag
        e` object
    :type ref: ``str``
    :param user_path_storage: Payload object.
    :type user_path_storage: :py:class:`v1_11_6.web.vo.UserPathStorage`
    """
    url = "/resources/json/delphix/service/userPaths/%s" % ref
    response = engine.post(url, user_path_storage.to_dict(dirty=True) if user_path_storage else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def delete(engine, ref):
    """
    Delete the specified UserPathStorage object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_11_6.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_11_6.web.objects.UserPathStorage.UserPathStorag
        e` object
    :type ref: ``str``
    """
    url = "/resources/json/delphix/service/userPaths/%s/delete" % ref
    response = engine.post(url, None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

