#
# Copyright 2023 by Delphix
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
Package "snapshot"
"""
from urllib.parse import urlencode
from delphixpy.v1_4_3 import response_validator

def get(engine, ref):
    """
    Retrieve the specified TimeflowSnapshot object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_4_3.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_4_3.web.objects.TimeflowSnapshot.TimeflowSnapsh
        ot` object
    :type ref: ``str``
    :rtype: :py:class:`v1_4_3.web.vo.TimeflowSnapshot`
    """
    url = "/resources/json/delphix/snapshot/%s" % ref
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['TimeflowSnapshot'], returns_list=False, raw_result=raw_result)

def get_all(engine, database=None, from_date=None, page_size=None, timeflow=None, to_date=None, traverse_timeflows=None):
    """
    Returns a list of snapshots on the system or within a particular object. By
    default, all snapshots within the domain are listed.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_4_3.delphix_engine.DelphixEngine`
    :param database: Restrict snapshots to those within a timeflow of the
        specified database. This option is mutually exclusive with the
        "timeflow" option
    :type database: ``str``
    :param from_date: Start date to use for filtering out results.
    :type from_date: ``str``
    :param page_size: Limit the number of snapshots returned.
    :type page_size: ``int``
    :param timeflow: Restrict snapshots to those within the specified timeflow.
        This option is mutually exclusive with the "database" option.
    :type timeflow: ``str``
    :param to_date: End date to use for filtering out results.
    :type to_date: ``str``
    :param traverse_timeflows: Whether to restrict snapshots to those in the
        current timeflow and in parent timeflows older than the branch point.
        This option is only used with the "database" option. The default
        behavior is false, i.e. show all snapshots.
    :type traverse_timeflows: ``bool``
    :rtype: ``list`` of :py:class:`v1_4_3.web.vo.TimeflowSnapshot`
    """
    url = "/resources/json/delphix/snapshot"
    query_params = {"database": database, "fromDate": from_date, "pageSize": page_size, "timeflow": timeflow, "toDate": to_date, "traverseTimeflows": traverse_timeflows}
    query_dct = {k: query_params[k] for k in query_params if query_params[k] is not None}
    if query_dct:
        url_params = urlencode(query_dct)
        url += "?%s" % url_params
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['TimeflowSnapshot'], returns_list=True, raw_result=raw_result)

def update(engine, ref, timeflow_snapshot=None):
    """
    Update the specified TimeflowSnapshot object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_4_3.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_4_3.web.objects.TimeflowSnapshot.TimeflowSnapsh
        ot` object
    :type ref: ``str``
    :param timeflow_snapshot: Payload object.
    :type timeflow_snapshot: :py:class:`v1_4_3.web.vo.TimeflowSnapshot`
    """
    url = "/resources/json/delphix/snapshot/%s" % ref
    response = engine.post(url, timeflow_snapshot.to_dict(dirty=True) if timeflow_snapshot else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def delete(engine, ref):
    """
    Delete the specified TimeflowSnapshot object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_4_3.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_4_3.web.objects.TimeflowSnapshot.TimeflowSnapsh
        ot` object
    :type ref: ``str``
    """
    url = "/resources/json/delphix/snapshot/%s/delete" % ref
    response = engine.post(url, None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def timeflow_range(engine, ref, traverse_timeflows=None):
    """
    Return the provisionable timeflow range based on a specific snapshot.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_4_3.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_4_3.web.objects.TimeflowSnapshot.TimeflowSnapsh
        ot` object
    :type ref: ``str``
    :param traverse_timeflows: Whether to restrict the range to the branch
        point of the child timeflow. The default behavior is false.
    :type traverse_timeflows: ``str``
    """
    url = "/resources/json/delphix/snapshot/%s/timeflowRange" % ref
    query_params = {"traverseTimeflows": traverse_timeflows}
    query_dct = {k: query_params[k] for k in query_params if query_params[k] is not None}
    if query_dct:
        url_params = urlencode(query_dct)
        url += "?%s" % url_params
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def space(engine, snapshot_space_parameters):
    """
    Returns the space used by the specified snapshot space parameters.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_4_3.delphix_engine.DelphixEngine`
    :param snapshot_space_parameters: Payload object.
    :type snapshot_space_parameters:
        :py:class:`v1_4_3.web.vo.SnapshotSpaceParameters`
    :rtype: :py:class:`v1_4_3.web.vo.SnapshotSpaceResult`
    """
    url = "/resources/json/delphix/snapshot/space"
    response = engine.post(url, snapshot_space_parameters.to_dict(dirty=True) if snapshot_space_parameters else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['SnapshotSpaceResult'], returns_list=False, raw_result=raw_result)

