#
# Copyright 2023 by Delphix
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
Package "system.version"
"""
from delphixpy.v1_4_3 import response_validator

def get(engine, ref):
    """
    Retrieve the specified SystemVersion object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_4_3.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_4_3.web.objects.SystemVersion.SystemVersion`
        object
    :type ref: ``str``
    :rtype: :py:class:`v1_4_3.web.vo.SystemVersion`
    """
    url = "/resources/json/delphix/system/version/%s" % ref
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['SystemVersion'], returns_list=False, raw_result=raw_result)

def get_all(engine):
    """
    List SystemVersion objects on the system.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_4_3.delphix_engine.DelphixEngine`
    :rtype: ``list`` of :py:class:`v1_4_3.web.vo.SystemVersion`
    """
    url = "/resources/json/delphix/system/version"
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['SystemVersion'], returns_list=True, raw_result=raw_result)

def delete(engine, ref):
    """
    Delete the specified SystemVersion object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_4_3.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_4_3.web.objects.SystemVersion.SystemVersion`
        object
    :type ref: ``str``
    """
    url = "/resources/json/delphix/system/version/%s/delete" % ref
    response = engine.post(url, None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def apply(engine, ref, apply_version_parameters=None):
    """
    Applies the upgrade version to the current system

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_4_3.delphix_engine.DelphixEngine`
    :param ref: Reference to a
        :py:class:`delphixpy.v1_4_3.web.objects.SystemVersion.SystemVersion`
        object
    :type ref: ``str``
    :param apply_version_parameters: Payload object.
    :type apply_version_parameters:
        :py:class:`v1_4_3.web.vo.ApplyVersionParameters`
    """
    url = "/resources/json/delphix/system/version/%s/apply" % ref
    response = engine.post(url, apply_version_parameters.to_dict(dirty=True) if apply_version_parameters else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

