#
# Copyright 2023 by Delphix
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
Package "environment.oracle.clusternode"
"""
from urllib.parse import urlencode
from delphixpy.v1_8_0 import response_validator

def create(engine, oracle_cluster_node_create_parameters):
    """
    Create a new OracleClusterNode object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_8_0.delphix_engine.DelphixEngine`
    :param oracle_cluster_node_create_parameters: Payload object.
    :type oracle_cluster_node_create_parameters:
        :py:class:`v1_8_0.web.vo.OracleClusterNodeCreateParameters`
    :rtype: ``str``
    """
    url = "/resources/json/delphix/environment/oracle/clusternode"
    response = engine.post(url, oracle_cluster_node_create_parameters.to_dict(dirty=True) if oracle_cluster_node_create_parameters else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['str'], returns_list=False, raw_result=raw_result)

def get(engine, ref):
    """
    Retrieve the specified OracleClusterNode object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_8_0.delphix_engine.DelphixEngine`
    :param ref: Reference to a :py:class:`delphixpy.v1_8_0.web.objects.OracleCl
        usterNode.OracleClusterNode` object
    :type ref: ``str``
    :rtype: :py:class:`v1_8_0.web.vo.OracleClusterNode`
    """
    url = "/resources/json/delphix/environment/oracle/clusternode/%s" % ref
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['OracleClusterNode'], returns_list=False, raw_result=raw_result)

def get_all(engine, cluster=None):
    """
    Returns a list of host nodes filtered by cluster.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_8_0.delphix_engine.DelphixEngine`
    :param cluster: The cluster to filter by.
    :type cluster: ``str``
    :rtype: ``list`` of :py:class:`v1_8_0.web.vo.OracleClusterNode`
    """
    url = "/resources/json/delphix/environment/oracle/clusternode"
    query_params = {"cluster": cluster}
    query_dct = {k: query_params[k] for k in query_params if query_params[k] is not None}
    if query_dct:
        url_params = urlencode(query_dct)
        url += "?%s" % url_params
    response = engine.get(url)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=['OracleClusterNode'], returns_list=True, raw_result=raw_result)

def update(engine, ref, oracle_cluster_node=None):
    """
    Update the specified OracleClusterNode object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_8_0.delphix_engine.DelphixEngine`
    :param ref: Reference to a :py:class:`delphixpy.v1_8_0.web.objects.OracleCl
        usterNode.OracleClusterNode` object
    :type ref: ``str``
    :param oracle_cluster_node: Payload object.
    :type oracle_cluster_node: :py:class:`v1_8_0.web.vo.OracleClusterNode`
    """
    url = "/resources/json/delphix/environment/oracle/clusternode/%s" % ref
    response = engine.post(url, oracle_cluster_node.to_dict(dirty=True) if oracle_cluster_node else None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

def delete(engine, ref):
    """
    Delete the specified OracleClusterNode object.

    :param engine: The Delphix Engine
    :type engine: :py:class:`delphixpy.v1_8_0.delphix_engine.DelphixEngine`
    :param ref: Reference to a :py:class:`delphixpy.v1_8_0.web.objects.OracleCl
        usterNode.OracleClusterNode` object
    :type ref: ``str``
    """
    url = "/resources/json/delphix/environment/oracle/clusternode/%s/delete" % ref
    response = engine.post(url, None)
    result = response_validator.validate(response, engine)
    raw_result = getattr(engine, 'raw_result', False)
    return response_validator.parse_result(result, undef_enabled=True, return_types=None, returns_list=None, raw_result=raw_result)

