class Role:
    """Helper class to generate role strings for `Permission`."""
    @staticmethod
    def any():
        """Grants access to anyone.
        
        This includes authenticated and unauthenticated users.
        """
        return 'any'

    @staticmethod
    def user(id, status = ""):
        """Grants access to a specific user by user ID.

        You can optionally pass verified or unverified for
        `status` to target specific types of users.

        Parameters
        ----------
        id : str
        status : str, optional

        Returns
        -------
        str
        """
        if status:
            return f'user:{id}/{status}'
        return f'user:{id}'

    @staticmethod
    def users(status = ""):
        """Grants access to any authenticated or anonymous user.
        
        You can optionally pass verified or unverified for
        `status` to target specific types of users.

        Parameters
        ----------
        status : str, optional

        Returns
        -------
        str
        """
        if status:
            return f'users/{status}'
        return 'users'
    
    @staticmethod
    def guests():
        """Grants access to any guest user without a session.

        Authenticated users don't have access to this role.

        Returns
        -------
        str
        """
        return 'guests'

    @staticmethod
    def team(id, role = ""):
        """Grants access to a team by team ID.

        You can optionally pass a role for `role` to target
        team members with the specified role.

        Parameters
        ----------
        id : str
        role : str, optional

        Returns
        -------
        str
        """
        if role:
            return f'team:{id}/{role}'
        return f'team:{id}'

    @staticmethod
    def member(id):
        """Grants access to a specific member of a team.

        When the member is removed from the team, they will
        no longer have access.

        Parameters
        ----------
        id : str

        Returns
        -------
        str
        """
        return f'member:{id}'

    @staticmethod
    def label(name):
        """Grants access to a user with the specified label.

        Parameters
        ----------
        name : str

        Returns
        -------
        str
        """
        return f'label:{name}'